#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliParser
import CliCommand
import CliMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.BgpVpnDomainIdCli as BgpVpnDomainIdCli
from CliPlugin import IntfCli
from CliPlugin.BgpVpnCli import (
   RouteImportMatchFailureDiscardCmd,
   VpnClientIbgpAttributeTunneling,
)
from CliPlugin.IpRibLib import ResolutionRibsExprFactory
from CliPlugin.RoutingBgpCli import (
   afModeExtensionHook,
   BgpCmdBaseClass,
   PeerCliExpression,
   RouterBgpAfSharedModelet,
   RouterBgpAfVpnModelet,
   RouterBgpBaseMode,
   RouterBgpAfUnsupportedPrefixListSharedModelet,
)
from CliPlugin.RoutingBgpInstanceCli import (
   NextHopOriginateTunnelCmd,
   NexthopResolutionDisabledCmd,
   SetAddpathSendCmd,
)
from CliPlugin.RoutingBgpNeighborCli import (
   SetNeighborAddPathSendCmd,
   SetNeighborRcfCmd,
   locatorNameMatcher
)
from BgpLib import updateConfigAttrsAfMap
from CliMode.BgpCommon import RoutingBgpBaseAfMode
import CliToken.RoutingBgp as bgpTokens
import CliToken.IpRibLibCliTokens
from IpLibConsts import DEFAULT_VRF
from Toggles import RoutingLibToggleLib
from Toggles import TunnelToggleLib
# pylint: disable-msg=W0621

updateConfigAttrsAfMap( { 'vpn-ipv4' : [ 'VpnV4', 'AfVpnV4' ],
                          'vpn-ipv6' : [ 'VpnV6', 'AfVpnV6' ] } )

# Note: Defining this directly in the RouterBgpBaseMode,
# so this command is not available in Nd - VRF submode.
# Mpls in the MplsVpn below refers to the service label used for the VPN and not the
# EncapType( the transport label ).
# EncapType may be Mpls, GRE etc. but we only support Mpls at present.
class RouterBgpBaseAfMplsVpnV4Mode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family MPLS-VPN V4 configuration'

   def __init__( self, parent, session, addrFamily='vpn-ipv4' ):
      self.vrfName = DEFAULT_VRF
      self.addrFamily = addrFamily
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpBaseAfMplsVpnV6Mode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family MPLS-VPN V6 configuration'

   def __init__( self, parent, session, addrFamily='vpn-ipv6' ):
      self.vrfName = DEFAULT_VRF
      self.addrFamily = addrFamily
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpBaseAfMplsVpnSharedModelet( CliParser.Modelet ):
   """This modelete has all the commands which are shared between vpn-ipv4
   and vpn-ipv6 address-families"""

RouterBgpBaseAfMplsVpnV4Mode.addModelet( RouterBgpAfSharedModelet )
RouterBgpBaseAfMplsVpnV4Mode.addModelet( RouterBgpBaseAfMplsVpnSharedModelet )
RouterBgpBaseAfMplsVpnV4Mode.addModelet( RouterBgpAfVpnModelet )
RouterBgpBaseAfMplsVpnV4Mode.addModelet(
   RouterBgpAfUnsupportedPrefixListSharedModelet )
RouterBgpBaseAfMplsVpnV6Mode.addModelet( RouterBgpAfSharedModelet )
RouterBgpBaseAfMplsVpnV6Mode.addModelet( RouterBgpBaseAfMplsVpnSharedModelet )
RouterBgpBaseAfMplsVpnV6Mode.addModelet( RouterBgpAfVpnModelet )
RouterBgpBaseAfMplsVpnV6Mode.addModelet(
   RouterBgpAfUnsupportedPrefixListSharedModelet )

#------------------------------------------------------------------------------------
# "[ no | default ] neighbor PEER ( activate | deactivate )
#------------------------------------------------------------------------------------
neighborActivateCmdRegistration = \
      [ RouterBgpBaseAfMplsVpnSharedModelet ]

class MplsVpnNeighborActDeact( CliCommand.CliCommandClass ):
   syntax = 'neighbor PEER ( activate | deactivate )'
   noOrDefaultSyntax = 'neighbor PEER activate'
   data = {
         'neighbor' : bgpTokens.neighbor,
         'PEER' : PeerCliExpression,
         'activate' : bgpTokens.activate,
         'deactivate' : bgpTokens.deactivate,
   }
   handler = "MplsVpnCliHandler.MplsVpnNeighborActDeact_handler"
   noHandler = "MplsVpnCliHandler.MplsVpnNeighborActDeact_noHandler"
   defaultHandler = "MplsVpnCliHandler.MplsVpnNeighborActDeact_defaultHandler"

for mode in neighborActivateCmdRegistration:
   mode.addCommandClass( MplsVpnNeighborActDeact )


#------------------------------------------------------------------------------------
# "[ no | default ] address-family vpn-ipv4 | vpn-ipv6" config mode
#------------------------------------------------------------------------------------
tokenVpnV4 = CliMatcher.KeywordMatcher( 'vpn-ipv4',
                     helpdesc='MPLS L3 VPN IPv4 unicast address family' )
tokenVpnV6 = CliMatcher.KeywordMatcher( 'vpn-ipv6',
                     helpdesc='MPLS L3 VPN IPv6 unicast address family' )


class mplsVpnAfExpression( CliCommand.CliExpression ):
   expression = '( vpn-ipv4 | vpn-ipv6 )'
   data = {
         'vpn-ipv4' : tokenVpnV4,
         'vpn-ipv6' : tokenVpnV6
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'vpn-ipv4' in args:
         args[ 'AF' ] = 'vpn-ipv4'
      elif 'vpn-ipv6' in args:
         args[ 'AF' ] = 'vpn-ipv6'

class MplsVpnAfModeCmd( CliCommand.CliCommandClass ):
   syntax = 'address-family FAMILY'
   noOrDefaultSyntax = syntax
   data = {
         'address-family' : bgpTokens.addrFamily,
         'FAMILY' : mplsVpnAfExpression,
         }
   handler = "MplsVpnCliHandler.MplsVpnAfModeCmd_handler"
   noOrDefaultHandler = "MplsVpnCliHandler.MplsVpnAfModeCmd_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( MplsVpnAfModeCmd )

#------------------------------------------------------------------------------------
# [ no | default ] bgp additional-paths send any
#------------------------------------------------------------------------------------
RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass( SetAddpathSendCmd )

#--------------------------------------------------------------------------------
# ( no | default ) neighbor ( ADDR | V6ADDR | LLV6ADDR | PEERGROUPNAME )
# additional-paths send any
#--------------------------------------------------------------------------------
RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass( SetNeighborAddPathSendCmd )

# --------------------------------------------------------------------------------
# ( no | default ) neighbor ( ADDR | V6ADDR | LLV6ADDR | PEERGROUPNAME )
# rcf DIRECTION ( ( FUNCTION [ disabled ] ) | disabled )
# --------------------------------------------------------------------------------
RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass( SetNeighborRcfCmd )

#--------------------------------------------------------------------------------
# ( no | default ) neighbor default encapsulation mpls next-hop-self
# source-interface INTF
# "[no|default] neighbor default encapsulation segment-routing ipv6 locator LOC"
# in "address-family vpn-ipv4" and "address-family vpn-ipv6" modes.
#--------------------------------------------------------------------------------
class NeighborDefaultEncapMplsNhselfSourceInterfaceCmd( BgpCmdBaseClass ):
   syntax = 'neighbor default encapsulation mpls next-hop-self source-interface INTF'
   noOrDefaultSyntax = 'neighbor default encapsulation mpls next-hop-self \
         source-interface ...'
   data = {
      'neighbor' : bgpTokens.neighbor,
      'default' : bgpTokens.default,
      'encapsulation' : bgpTokens.encap,
      'mpls' : bgpTokens.mpls,
      'next-hop-self' : bgpTokens.nextHopSelf,
      'source-interface' : bgpTokens.sourceInterface,
      'INTF' : IntfCli.Intf.matcherWithIpSupport,
   }
   handler = \
      "MplsVpnCliHandler.NeighborDefaultEncapMplsNhselfSourceInterfaceCmd_handler"
   noOrDefaultHandler = \
      "MplsVpnCliHandler." + \
      "NeighborDefaultEncapMplsNhselfSourceInterfaceCmd_noOrDefaultHandler"

RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass(
      NeighborDefaultEncapMplsNhselfSourceInterfaceCmd )

class NeighborDefaultEncapSegmentRoutingIpv6Cmd( BgpCmdBaseClass ):
   syntax = 'neighbor default encapsulation segment-routing ipv6 locator LOCATOR'
   noOrDefaultSyntax = 'neighbor default encapsulation segment-routing ipv6'
   data = {
      'neighbor' : bgpTokens.neighbor,
      'default' : bgpTokens.default,
      'encapsulation' : bgpTokens.encap,
      'segment-routing' : bgpTokens.segmentRoutingForEvpn,
      'ipv6' : bgpTokens.ipv6Encaps,
      'locator' : bgpTokens.srv6Locator,
      'LOCATOR' : locatorNameMatcher,
   }
   handler = "MplsVpnCliHandler.NeighborDefaultEncapSegmentRoutingIpv6Cmd_handler"
   noOrDefaultHandler = \
      "MplsVpnCliHandler." + \
      "NeighborDefaultEncapSegmentRoutingIpv6Cmd_noOrDefaultHandler"

if RoutingLibToggleLib.toggleSRv6MplsVpnEnabled():
   RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass(
      NeighborDefaultEncapSegmentRoutingIpv6Cmd )

#--------------------------------------------------------------------------------
# ( no | default ) neighbor ( addr | peer group ) encapsulation mpls next-hop-self
# source-interface INTF
#--------------------------------------------------------------------------------
class NeighborPeerEncapMplsNhselfSourceInterfaceCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER encapsulation mpls next-hop-self source-interface INTF'
   noOrDefaultSyntax = syntax.replace( 'INTF', '...' )
   data = {
      'neighbor' : bgpTokens.neighbor,
      'PEER' : PeerCliExpression,
      'encapsulation' : bgpTokens.encap,
      'mpls' : bgpTokens.mpls,
      'next-hop-self' : bgpTokens.nextHopSelf,
      'source-interface' : bgpTokens.sourceInterface,
      'INTF' : IntfCli.Intf.matcherWithIpSupport,
   }
   handler = \
      "MplsVpnCliHandler.NeighborPeerEncapMplsNhselfSourceInterfaceCmd_handler"
   noOrDefaultHandler = \
      "MplsVpnCliHandler." + \
      "NeighborPeerEncapMplsNhselfSourceInterfaceCmd_noOrDefaultHandler"

RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass(
   NeighborPeerEncapMplsNhselfSourceInterfaceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mpls label allocation disabled
#--------------------------------------------------------------------------------
class MplsLabelAllocationDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls label allocation disabled'
   noOrDefaultSyntax = syntax
   data = {
      'mpls' : bgpTokens.mpls,
      'label' : 'VPN label',
      'allocation' : 'VPN label allocation scheme',
      'disabled' : 'Use explicit-null as local label',
   }
   handler = "MplsVpnCliHandler.MplsLabelAllocationDisabledCmd_handler"
   noOrDefaultHandler = \
      "MplsVpnCliHandler.MplsLabelAllocationDisabledCmd_noOrDefaultHandler"

RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass( MplsLabelAllocationDisabledCmd )

#-------------------------------------------------------------------------------
# "[no|default] next-hop resolution ribs ( vrf-unicast-rib | udp-tunnel | RIBS )"
#-------------------------------------------------------------------------------
class ResolutionVpnRibCmd( CliCommand.CliCommandClass ):
   if TunnelToggleLib.toggleDynamicGueTunnelsEnabled():
      syntax = \
            'next-hop resolution ribs ( vrf-unicast-rib | udp-tunnel | RIBS_HIDDEN )'
      noOrDefaultSyntax = 'next-hop resolution ribs [ vrf-unicast-rib | udp-tunnel ]'
   else:
      syntax = 'next-hop resolution ribs ( vrf-unicast-rib | RIBS_HIDDEN )'
      noOrDefaultSyntax = 'next-hop resolution ribs [ vrf-unicast-rib ]'
   data = {
      'next-hop' : bgpTokens.nextHopKw,
      'resolution' : CliToken.IpRibLibCliTokens.matcherResolution,
      'ribs' : CliToken.IpRibLibCliTokens.matcherRibs,
      'vrf-unicast-rib' :
      'Resolve the next-hop in the imported-vrf using unicast RIB',
   }
   if TunnelToggleLib.toggleDynamicGueTunnelsEnabled():
      data[ 'udp-tunnel' ] = 'Resolve the next-hop using UDP tunnels'
   data[ 'RIBS_HIDDEN' ] = ResolutionRibsExprFactory()
   handler = "MplsVpnCliHandler.ResolutionVpnRibCmd_handler"
   noOrDefaultHandler = "MplsVpnCliHandler.ResolutionVpnRibCmd_noOrDefaultHandler"

RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass( ResolutionVpnRibCmd )

#-------------------------------------------------------------------------------
# "[no|default] next-hop originate tunnel type udp policy POLICYNAME"
# in "address-family vpn-ipv4" and "address-family vpn-ipv6" modes
#-------------------------------------------------------------------------------
if TunnelToggleLib.toggleDynamicGueTunnelsEnabled():
   RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass( NextHopOriginateTunnelCmd )

#-------------------------------------------------------------------------------
# "[no|default] domain identifier global_admin:local_admin
# under 'address-family vpn-ipv4|vpn-ipv6' mode
#-------------------------------------------------------------------------------
# VPN domain id configuration under MPLS VPN address-family.
# "domain identifier ASN:LOCAL_ADMIN"
class SetMplsVpnDomainIdCmd( CliCommand.CliCommandClass ):
   syntax = 'domain identifier DOMAIN_ID'
   noOrDefaultSyntax = 'domain identifier ...'
   data = { 'domain' : BgpVpnDomainIdCli.tokenDomain,
            'identifier' : BgpVpnDomainIdCli.tokenIdentifier,
            'DOMAIN_ID' : BgpVpnDomainIdCli.DomainIdExpression }
   handler = "MplsVpnCliHandler.SetMplsVpnDomainIdCmd_handler"
   noOrDefaultHandler = "MplsVpnCliHandler.SetMplsVpnDomainIdCmd_noOrDefaultHandler"

RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass( SetMplsVpnDomainIdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] neighbor default encapsulation mpls next-hop-self
# received-vpnv4-routes
# in "address-family vpn-ipv4" mode
#--------------------------------------------------------------------------------
matcherReceivedVpnv4Routes = CliMatcher.KeywordMatcher( 'received-vpnv4-routes',
      helpdesc='Use local labels for advertising received VPN-IPv4 routes' )

class NeighborDefaultEncapMplsNhselfReceivedVpnv4RoutesCmd(
      CliCommand.CliCommandClass ):
   syntax = 'neighbor default encapsulation mpls next-hop-self received-vpnv4-routes'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor' : bgpTokens.neighbor,
      'default' : bgpTokens.default,
      'encapsulation' : bgpTokens.encap,
      'mpls' : bgpTokens.mpls,
      'next-hop-self' : bgpTokens.nextHopSelf,
      'received-vpnv4-routes' : matcherReceivedVpnv4Routes,
   }
   handler = \
      "MplsVpnCliHandler." + \
      "NeighborDefaultEncapMplsNhselfReceivedVpnv4RoutesCmd_handler"
   noOrDefaultHandler = \
      "MplsVpnCliHandler." + \
      "NeighborDefaultEncapMplsNhselfReceivedVpnv4RoutesCmd_noOrDefaultHandler"

RouterBgpBaseAfMplsVpnV4Mode.addCommandClass(
    NeighborDefaultEncapMplsNhselfReceivedVpnv4RoutesCmd )

#--------------------------------------------------------------------------------
# [ no | default ] neighbor default encapsulation mpls next-hop-self
# received-vpnv6-routes
# in "address-family vpn-ipv6" mode
#--------------------------------------------------------------------------------

matcherReceivedVpnv6Routes = CliMatcher.KeywordMatcher( 'received-vpnv6-routes',
      helpdesc='Use local labels for advertising received VPN-IPv6 routes' )

class NeighborDefaultEncapMplsNhselfReceivedVpnv6RoutesCmd(
      CliCommand.CliCommandClass ):
   syntax = 'neighbor default encapsulation mpls next-hop-self received-vpnv6-routes'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor' : bgpTokens.neighbor,
      'default' : bgpTokens.default,
      'encapsulation' : bgpTokens.encap,
      'mpls' : bgpTokens.mpls,
      'next-hop-self' : bgpTokens.nextHopSelf,
      'received-vpnv6-routes' : matcherReceivedVpnv6Routes,
   }
   handler = \
      "MplsVpnCliHandler." + \
      "NeighborDefaultEncapMplsNhselfReceivedVpnv6RoutesCmd_handler"
   noOrDefaultHandler = \
      "MplsVpnCliHandler." + \
      "NeighborDefaultEncapMplsNhselfReceivedVpnv6RoutesCmd_noOrDefaultHandler"

RouterBgpBaseAfMplsVpnV6Mode.addCommandClass(
    NeighborDefaultEncapMplsNhselfReceivedVpnv6RoutesCmd )

#---------------------------------------------------------------------------------
# "[no|default] next-hop resolution disabled" in "address-family vpn-ipv4" and
# "address-family vpn-ipv6" modes
#---------------------------------------------------------------------------------
RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass( NexthopResolutionDisabledCmd )

#---------------------------------------------------------------------------------
# "[no|default] route import match-failure action discard"
# in "address-family vpn-ipv4" and "address-family vpn-ipv6" modes.
#
# Enables the discarding of MPLSVPN paths that won't be imported into any VRF
# (AID6625).
#---------------------------------------------------------------------------------

RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass(
   RouteImportMatchFailureDiscardCmd )

# ---------------------------------------------------------------------------------
# "[no | default] vpn client ibgp attribute tunneling"
#
# Enables or disables RFC 6368 behavior when importing/exporting routes.

RouterBgpBaseAfMplsVpnSharedModelet.addCommandClass(
   VpnClientIbgpAttributeTunneling )

def Plugin( entityManager ):
   afModeExtensionHook.addAfModeExtension( 'vpn-ipv4', RouterBgpBaseAfMplsVpnV4Mode )
   afModeExtensionHook.addAfModeExtension( 'vpn-ipv6', RouterBgpBaseAfMplsVpnV6Mode )
