# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import BasicCli, LazyMount, ConfigMount
import CliParser
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliPlugin import RouterMulticastCliLib
from CliPlugin import TechSupportCli
from CliPlugin.McastCommonCli import mcastRoutingSupportedGuard
from CliPlugin.RouterMulticastCliLib import configGetters, lazyGetters, \
      AddressFamily
import CliToken.Ip
import Tac
from CliMode.Msdp import RoutingMsdpBaseMode, RoutingMsdpVrfMode, \
      RoutingMsdpPeerMode
from McastCommonCliLib import vrfFromMode
from IpLibConsts import DEFAULT_VRF, VRFNAMES_RESERVED
import CliCommand
import CliMatcher

msdpPeerAddrMatcher = IpAddrMatcher.IpAddrMatcher( 'MSDP peer address' )

vrfExprFactory = VrfCli.VrfExprFactory( helpdesc='VRF name' )

MAX_SA_CACHE_SIZE = 40000

MsdpConfigColl = "Routing::Msdp::ConfigColl"
( configColl,
  configCollFromMode,
  config,
  configFromMode ) = configGetters( MsdpConfigColl )

MsdpClearConfigColl = "Routing::Msdp::ClearConfigColl"
( clearConfigColl,
  clearConfigCollFromMode,
  clearConfig,
  clearConfigFromMode ) = lazyGetters( MsdpClearConfigColl,
                                       writeMount=True )

MfibVrfConfig = "Routing::Multicast::Fib::VrfConfig"
( mfibVrfConfigColl,
  mfibVrfConfigCollFromMode,
  mfibConfig,
  mfibConfigFromMode ) = lazyGetters( MfibVrfConfig, collectionName='config' )

# Globals
msdpConfig = None
msdpStatus = None
msdpPimStatus = None
rpfMapStatusDir = None
groupSourceMapColl = None
entityMgr = None
routingHwStatus = None

def msdpVrfDefinitionHook( vrfName ):
   for af in [ AddressFamily.ipv4 ]:
      clearConfig( af, vrfName )
      config( af, vrfName )

def _msdpVrfDeletionHook( vrfName ):
   for af in [ AddressFamily.ipv4 ]:
      if configColl( af ).vrfConfig.get( vrfName ):
         del configColl( af ).vrfConfig[ vrfName ]
         if clearConfigColl( af ).vrfConfig.get( vrfName ):
            del clearConfigColl( af ).vrfConfig[ vrfName ]


def msdpDeletionHook():
   for af in [ AddressFamily.ipv4 ]:
      _configColl = configColl( af )
      _msdpVrfDeletionHook( DEFAULT_VRF )

def vrfExists( af, vrfName ):
   if mfibConfig( af, vrfName ):
      return True
   return False

def printNonexistentPeerError( mode, peerIp ):
   mode.addError( "Peer is not configured in vrf %s: %s" % \
                     ( vrfFromMode( mode ), peerIp ) )

class RouterMsdpBaseMode( RoutingMsdpBaseMode, BasicCli.ConfigModeBase ):
   name = 'MSDP configuration'

   def __init__( self, parent, session ):
      self.vrfName = DEFAULT_VRF
      RoutingMsdpBaseMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterMsdpVrfMode( RoutingMsdpVrfMode, BasicCli.ConfigModeBase ):
   name = 'MSDP VRF configuration'

   def __init__( self, parent, session, vrfName ):
      assert vrfName not in VRFNAMES_RESERVED
      self.vrfName = vrfName
      RoutingMsdpVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterMsdpPeerMode( RoutingMsdpPeerMode, BasicCli.ConfigModeBase ):
   name = 'MSDP Peer configuration'

   def __init__( self, parent, session, vrfName=DEFAULT_VRF, peer=None ):
      RoutingMsdpPeerMode.__init__( self, vrfName, peer )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterMsdpSharedModelet( CliParser.SingletonModelet ):
   pass

RouterMsdpBaseMode.addModelet( RouterMsdpSharedModelet )
RouterMsdpVrfMode.addModelet( RouterMsdpSharedModelet )
BasicCli.GlobalConfigMode.addModelet( RouterMsdpSharedModelet )

#------------------------------------------------------------------------------------
# Clean-up VRF config
#------------------------------------------------------------------------------------

def cleanUpMsdpVrfConfig( vrfName ):
   if vrfName is None or vrfName == '':
      vrfName = DEFAULT_VRF
   if vrfName in msdpConfig.vrfConfig:
      msdpVrfConfig = msdpConfig.vrfConfig[ vrfName ]
      msdpVrfConfig.originatorId = ""
      msdpVrfConfig.defaultPeer.clear()
      msdpVrfConfig.sourceSALimit.clear()
      msdpVrfConfig.meshGroup.clear()
      msdpVrfConfig.peerConfig.clear()
      msdpVrfConfig.connRetryPeriod = \
            msdpVrfConfig.connRetryPeriodDefault

def deleteRouterMsdpVrfMode( mode, vrfName ):
   if vrfName in VRFNAMES_RESERVED:
      return
   if vrfName in msdpConfig.vrfConfig:
      cleanUpMsdpVrfConfig( vrfName )
      if vrfName != DEFAULT_VRF:
         _msdpVrfDeletionHook( vrfName )

#-------------------------------------------------------------------------------
# Common command utilities and 'msdp' node (note that it requires multicast to be
# supported).
#-------------------------------------------------------------------------------
msdpNode = CliCommand.Node(
      CliMatcher.KeywordMatcher(
         'msdp', helpdesc='MSDP protocol commands' ),
      guard=mcastRoutingSupportedGuard )

def peerConfig( peerIp, create=False, connSrc=None, vrfName=DEFAULT_VRF ):
   if vrfName not in msdpConfig.vrfConfig:
      msdpVrfDefinitionHook( vrfName )
   existingPeerConfig = msdpConfig.vrfConfig[ vrfName ].peerConfig.get( peerIp )
   if create:
      if existingPeerConfig:
         del msdpConfig.vrfConfig[ vrfName ].peerConfig[ peerIp ]
      if connSrc is not None:
         peer = \
             msdpConfig.vrfConfig[ vrfName ].peerConfig.newMember(
            peerIp, connSrc.name )
      else:
         peer = msdpConfig.vrfConfig[ vrfName ].peerConfig.newMember( peerIp, "" )
      return peer
   else:
      return existingPeerConfig

def safeDel( coll, key ):
   if key and key in coll:
      try:
         del coll[ key ]
      except KeyError:
         # this was a race to delete
         pass

#-------------------------------------------------------------------------------
# Deprecated cmd:
# [ no | default ] ip msdp cache-sa-state
#-------------------------------------------------------------------------------
class IpMsdpCacheSaStateCmd( CliCommand.CliCommandClass ):
   syntax = 'ip msdp cache-sa-state'
   data = {
      'ip': CliToken.Ip.ipMatcherForConfig,
      'msdp': msdpNode,
      'cache-sa-state':
         CliCommand.Node(
            CliMatcher.KeywordMatcher( 'cache-sa-state',
               helpdesc='Cache SA state (EOS always does this)' ),
            deprecatedByCmd='none (no longer supported)' )
   }

   @staticmethod
   def handler( mode, args ):
      # Do nothing.
      pass

RouterMsdpSharedModelet.addCommandClass( IpMsdpCacheSaStateCmd )

#-------------------------------------------------------------------------------
# Tech Support commands
#-------------------------------------------------------------------------------
TechSupportCli.registerShowTechSupportCmd(
   '2012-02-24 10:07:00',
   cmds=[ 'show ip msdp peer',
          'show ip msdp sa-cache rejected',
          'show ip msdp pim sa-cache',
          'show ip msdp summary' ],
   cmdsGuard=lambda: routingHwStatus.multicastRoutingSupported,
   summaryCmds=[ 'show ip msdp summary' ],
   summaryCmdsGuard=lambda: routingHwStatus.multicastRoutingSupported )

def Plugin( entityManager ):
   global msdpConfig
   global msdpStatus
   global msdpPimStatus
   global rpfMapStatusDir
   global groupSourceMapColl
   global routingHwStatus
   global entityMgr

   entityMgr = entityManager
   RouterMulticastCliLib.doConfigMounts( entityManager, [ MsdpConfigColl ],
                                       ipv4Only=True )
   RouterMulticastCliLib.doLazyMounts( entityManager, [ MsdpClearConfigColl ],
                                       useWriteMount=True, ipv4Only=True )
   RouterMulticastCliLib.doLazyMounts( entityManager, [ MfibVrfConfig ],
                                       useWriteMount=False, ipv4Only=True )
   msdpConfig = ConfigMount.mount( entityManager, 'routing/msdp/config',
                                   'Routing::Msdp::ConfigColl', 'w' )
   msdpStatus = LazyMount.mount( entityManager, 'routing/msdp/status',
                                 'Routing::Msdp::StatusColl', 'r' )
   msdpPimStatus = LazyMount.mount( entityManager, 'routing/msdp/pimstatus',
                                 'Routing::Msdp::PimStatusColl', 'r' )
   rpfMapStatusDir = LazyMount.mount( entityManager, 'routing/msdp/rpfMapStatus',
         'Tac::Dir', 'ri' )
   groupSourceMapColl = LazyMount.mount( entityManager,
                            "routing/pim/sparsemode/groupsourcemap",
                            "Routing::Pim::SparseMode::GroupSourceMapColl",
                                         "r" )
   routingHwStatus = LazyMount.mount( entityManager, 'routing/hardware/status',
         'Routing::Hardware::Status', 'r' )

