# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import BasicCli
import BasicCliModes
import ConfigMount
import LazyMount
import CliCommand
import CliMatcher
from CliMode.Mss import MssMode
from CliPlugin import ControllerCli
from CliPlugin import IpAddrMatcher
from CliPlugin.ControllerCli import CvxConfigMode, addNoCvxCallback
from CliToken.Refresh import refreshMatcherForExec
import MssL3Agent
import MssStaticCliLib
import Tac

defaults = Tac.Value( "Mss::CliDefaults" )
PolicyEnforcementConsistency = Tac.Type( 'Mss::PolicyEnforcementConsistency' )
# This CLI plugin defines the following modes and configuration commands:
#   cvx
#     service mss
#       [ no|default ] shutdown
#       [ no|default ] policy enforcement rules  { group verbatim|group|verbatim }
#       [ no|default ] policy enforcement consistency { best-effort|strict }
#       [ no|default ] vxlan traffic inspect [ add|remove ] { VTEPs }
#       [ no|default ] policy offload traffic logging    

#-----------------------------------------------------------------------------
# Mss Config CLI
#-----------------------------------------------------------------------------
class MssConfigMode( MssMode, BasicCli.ConfigModeBase ):
   # Attributes required of every Mode class.
   name = 'cvx-mss'

   def __init__( self, parent, session ):
      MssMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def setShutdown( self, args ):
      isEnabled = CliCommand.isNoCmd( args )
      if isEnabled and not controllerConfig.enabled:
         self.addError( "Macro-Segmentation Service cannot be enabled until the CVX "
               "server is enabled" )
         return
      config.enable = isEnabled
      serviceConfigDir.service[ MssL3Agent.name() ].enabled = isEnabled

   def setVxlanTrafficInspect( self, op=None, vtepList=None ):
      # vtepList should always be a list but added guard just in case
      if not isinstance( vtepList, list ):
         vtepList = []
      if op == 'default':
         # replace list, delete stale entries
         for vtep in config.vxlanTrafficInspect:
            ipAddr = vtep.stringValue
            if ipAddr not in vtepList:
               del config.vxlanTrafficInspect[ vtep ]
      if op in ( 'add', 'default' ):
         for vtep in vtepList:
            addr = Tac.Value( 'Arnet::IpGenAddr', vtep )
            config.vxlanTrafficInspect[ addr ] = True
      elif op == 'remove':
         if vtepList:
            for vtep in vtepList:
               addr = Tac.Value( 'Arnet::IpGenAddr', vtep )
               del config.vxlanTrafficInspect[ addr ]
         else:
            # remove all
            config.vxlanTrafficInspect.clear()

   @staticmethod
   def setPolicyEnforceRule( mode, args ):
      group = "group" in args
      verbatim = "verbatim" in args
      if group and not verbatim:
         mode.addErrorAndStop( "Group only policy enforcement is not supported" )

      config.policyEnforceRule = Tac.Value( 'Mss::PolicyEnforceRule',
                                            group=group, verbatim=verbatim )

   @staticmethod
   def defaultPolicyEnforceRule( mode, args ):
      config.policyEnforceRule = Tac.nonConst( defaults.policyEnforceRule )
   
   @staticmethod
   def _updateInternalCliModel( consistency ):
      if consistency == PolicyEnforcementConsistency.strict:
         # upgrade static CLI
         MssStaticCliLib.copyStaticV1ToV2(
               staticServiceDeviceV2Dir, staticPolicySetV2Dir,
               staticServiceDeviceDir, staticPolicySetDir )
      else:
         # downgrade static CLI
         MssStaticCliLib.copyStaticV2ToV1(
               staticServiceDeviceDir, staticPolicySetDir,
               staticServiceDeviceV2Dir, staticPolicySetV2Dir )

   @staticmethod
   def setPolicyEnforcementConsistency( mode, args ):
      consistency = args[ 'TYPE' ]
      if args[ 'TYPE' ] == 'strict':
         consistency = PolicyEnforcementConsistency.strict
      else:
         consistency = defaults.policyEnforcementConsistency

      try:
         MssConfigMode._updateInternalCliModel( consistency )
         config.policyEnforcementConsistency = consistency
      except MssStaticCliLib.DowngradeException as ex:
         mode.addError( ex.msg )

   @staticmethod
   def defaultPolicyEnforcementConsistency( mode, args ):
      try:
         MssConfigMode._updateInternalCliModel(
               defaults.policyEnforcementConsistency )
         config.policyEnforcementConsistency = defaults.policyEnforcementConsistency
      except MssStaticCliLib.DowngradeException as ex:
         mode.addError( ex.msg )

   def setPolicyLogOffload( self, args ):
      config.logOffloadPolicy = True

   def defaultPolicyLogOffload( self, args ):
      config.logOffloadPolicy = False

#------------------------------------------------------------------------------
# (config-cvx)# [ no | default ] service mss
#------------------------------------------------------------------------------
# All MSS submodes can register a cleanup function to be executed everytime the
# command 'no service mss' is executed. Each registered function will be called with
# the cli mode as the argument. Moreover, 'no cvx' will trigger the registered
# callbacks as well. So, the submodes don't need to register themselves for
# 'no cvx' call separately.
_noMssHook = []
def addNoMssCallback( fn ):
   assert callable( fn )
   _noMssHook.append( fn )

def noMss( mode, args=None ):
   config.reset()
   serviceConfigDir.service[ MssL3Agent.name() ].enabled = False

   # Execute all registered callbacks
   for fn in _noMssHook:
      fn( mode )

class MssModeCmd( CliCommand.CliCommandClass ):
   syntax = 'service mss'
   noOrDefaultSyntax = syntax
   data = {
      'service' : ControllerCli.serviceKwMatcher,
      'mss' : 'Macro-Segmentation Service'
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( MssConfigMode )
      mode.session_.gotoChildMode( childMode )
   
   noOrDefaultHandler = noMss

CvxConfigMode.addCommandClass( MssModeCmd )

# When the user issues "no cvx", we have to reset the Macro-Segmentation Service
addNoCvxCallback( noMss )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'Shutdown Macro-Segmentation Service',
   }

   handler = MssConfigMode.setShutdown
   noOrDefaultHandler = MssConfigMode.setShutdown

MssConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] policy offload traffic logging
#--------------------------------------------------------------------------------
class PolicyLogOffloadCmd( CliCommand.CliCommandClass ):
   syntax = 'policy offload traffic logging'
   noOrDefaultSyntax = syntax
   data = {
      'policy' : 'Firewall policy',
      'offload' : 'Offload policy',
      'traffic' : 'Traffic',
      'logging' : 'Logging traffic matched by a policy',
   }

   handler = MssConfigMode.setPolicyLogOffload
   noOrDefaultHandler = MssConfigMode.defaultPolicyLogOffload

MssConfigMode.addCommandClass( PolicyLogOffloadCmd )

#------------------------------------------------------------------------------
# (config-mss)# [ no|default ] policy enforcement rules
#                                         { group verbatim | group | verbatim }
#------------------------------------------------------------------------------

class PolicyEnforcementRuleCmd( CliCommand.CliCommandClass ):
   syntax = 'policy enforcement rules RULES'
   noOrDefaultSyntax = 'policy enforcement rules ...'
   data = {
      'policy' : 'Firewall policy',
      'enforcement': 'Firewall policy enforcement',
      'rules' : 'Policy enforcement rules',
      'RULES' : CliCommand.setCliExpression( {
         'group' : 'Enforce policies using group semantic',
         'verbatim' : 'Enforce policies using verbatim semantic' } )
   }

   handler = MssConfigMode.setPolicyEnforceRule
   noOrDefaultHandler = MssConfigMode.defaultPolicyEnforceRule

MssConfigMode.addCommandClass( PolicyEnforcementRuleCmd )

#------------------------------------------------------------------------------
# (config-mss)# [ no|default ] policy enforcement consistency
#               { best-effort | strict }
#------------------------------------------------------------------------------

class PolicyEnforcementConsistencyCmd( CliCommand.CliCommandClass ):
   syntax = 'policy enforcement consistency TYPE'
   noOrDefaultSyntax = 'policy enforcement consistency ...'
   data = {
      'policy' : 'Firewall policy',
      'enforcement': 'Firewall policy enforcement',
      'consistency' : 'Policy installation consistency model',
      'TYPE' : CliMatcher.EnumMatcher ( {
         'best-effort' : 'Install policies on a best-effort basis',
         'strict' : 'Install policies with strict consistency' } )
   }

   handler = MssConfigMode.setPolicyEnforcementConsistency
   noOrDefaultHandler = MssConfigMode.defaultPolicyEnforcementConsistency

MssConfigMode.addCommandClass( PolicyEnforcementConsistencyCmd )


#------------------------------------------------------------------------------
# (enable)# refresh service mss policy installation
#------------------------------------------------------------------------------

class RefreshPolicyInstallCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh service mss policy installation'
   data = {
      'refresh': refreshMatcherForExec,
      'service': ControllerCli.refreshServiceKwMatcher,
      'mss': 'Macro-Segmentation Service',
      'policy': 'MSS policy',
      'installation': 'Installation'
   }

   @staticmethod
   def handler( mode, args ):
      config.refreshCounter += 1

BasicCliModes.EnableMode.addCommandClass( RefreshPolicyInstallCmd )

#------------------------------------------------------------------------------
# (config-mss)# [no|default] vxlan traffic inspect [add|remove] {VTEPs}
# vxlan traffic inspect {VTEPs} - replace the entire default vtep list
# vxlan traffic inspect add {VTEPs} - add to default vtep list
# vxlan traffic inspect remove {VTEPs} - remove from default vtep list
# no vxlan traffic inspect - remove all vtep lists
#------------------------------------------------------------------------------

vtepAddrMatcher = IpAddrMatcher.IpAddrMatcher( 'IP address of VTEP' )

class VxlanTrafficInspectCmd( CliCommand.CliCommandClass ):
   syntax = ( 'vxlan traffic inspect [ ACTION ] { VTEPS }' )
   noOrDefaultSyntax = 'vxlan traffic inspect [ { VTEPS } ] ...'

   data = {
      'vxlan' : 'Configure MSS VXLAN parameters',
      'traffic' : 'Set tunneling traffic parameter to VTEPs',
      'inspect' : 'Inspect tunneled packets for DirectFlow matches',
      'ACTION': CliMatcher.EnumMatcher( {
         'add': 'Add VTEP(s)',
         'remove': 'Remove VTEP(s)',
      } ),
      'VTEPS' : vtepAddrMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      # define action as replace list/ add vtep/ remove vtep
      action = args.get( 'ACTION', 'default' )
      vteps = args[ 'VTEPS' ]

      badAddrs = []
      # Check for valid IP addr
      for vtep in vteps:
         addr = Tac.Value( 'Arnet::IpGenAddr', vtep )
         if ( addr.isLoopback or
              addr.isMulticast or
              addr.isUnspecified or
              ( addr.af == 'ipv4' and vtep == '255.255.255.255' ) ):
            badAddrs.append( vtep )
      if badAddrs:
         mode.addError( "Invalid VTEP address(es) - %s" % ','.join( badAddrs ) )
         return

      mode.setVxlanTrafficInspect( action, vteps )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      action = 'remove'
      vteps = args.get( 'VTEPS', [] )
      mode.setVxlanTrafficInspect( action, vteps )

MssConfigMode.addCommandClass( VxlanTrafficInspectCmd )

#------------------------------------------------------------------------------
# Plugin Setup
#------------------------------------------------------------------------------
config = None
preReleaseConfig = None
controllerConfig = None
serviceConfigDir = None
staticServiceDeviceDir = None
staticServiceDeviceV2Dir = None
staticPolicySetDir = None
staticPolicySetV2Dir = None

def Plugin( entityManager ):
   global config
   global preReleaseConfig
   global controllerConfig
   global serviceConfigDir
   global staticServiceDeviceDir
   global staticServiceDeviceV2Dir
   global staticPolicySetDir
   global staticPolicySetV2Dir

   config = ConfigMount.mount( entityManager, 'mss/config', 'Mss::Config', 'w' )
   preReleaseConfig = LazyMount.mount( entityManager, 'mss/preReleaseConfig',
         'Mss::PreReleaseConfig', 'r' )
   controllerConfig = LazyMount.mount( entityManager, 'controller/config',
         'Controllerdb::Config', 'r' )
   # To let the CVX infrastructure to know that the Mss service is enabled/disabled
   serviceConfigDir = ConfigMount.mount( entityManager,
         'controller/service/config', 'Controller::ServiceConfigDir',
         'w' )

   # Static CLI
   staticServiceDeviceDir = ConfigMount.mount( entityManager,
         'mssl3/serviceDeviceSourceConfig/cli',
         'MssL3::ServiceDeviceSourceConfig', 'w' )
   staticServiceDeviceV2Dir = ConfigMount.mount( entityManager,
         'mssl3/serviceDeviceSourceConfigV2/cli',
         'MssL3V2::ServiceDeviceSourceConfig', 'w' )
   staticPolicySetDir = ConfigMount.mount( entityManager,
         'mssl3/policySourceConfig/cli',
         'MssL3::MssPolicySourceConfig', 'w' )
   staticPolicySetV2Dir = ConfigMount.mount( entityManager,
         'mssl3/policySourceConfigV2/cli',
         'MssL3V2::MssPolicySourceConfig', 'w' )
