#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
# import CliModel

# pylint: disable=consider-using-f-string

import TableOutput
import operator
from CliModel import Str, List, Bool, Dict, Model, Float, Int, Submodel, Enum
from MssPolicyMonitor.Lib import ( PANORAMA_PLUGIN,
                                   PAN_FW_PLUGIN,
                                   FORTIMGR_PLUGIN,
                                   CHKP_MS_PLUGIN,
                                   timestampToStr,
                                   HA_ACTIVE_PASSIVE,
                                   HA_ACTIVE_ACTIVE,
                                   HA_ACTIVE,
                                   HA_PASSIVE,
                                   HA_ACTIVE_PRIMARY,
                                   HA_ACTIVE_SECONDARY,
                                   HA_DEVICE_SUSPENDED,
                                   HA_DISABLED_STATE,
                                   HA_ACTIVE_STATE,
                                   HA_STANDBY_STATE,
                                   LINK_STATE_UP,
                                   LINK_STATE_DOWN,
                                   LINK_STATE_UNKNOWN )

policySourceName = { PANORAMA_PLUGIN: 'palo-alto-panorama',
                     PAN_FW_PLUGIN: 'palo-alto-firewall',
                     FORTIMGR_PLUGIN: 'fortinet-fortimanager',
                     CHKP_MS_PLUGIN: 'check-point-mgmt-server',
                     'unassigned': '' }

# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
class MacroSegmentationGroupMembersInfoModel( Model ):
   __public__ = False

   groupName = Str( help='Policy tag in service device policies to act on' )
   groupMembers = List( help='Device group members for an Aggregation Manager',
                              valueType=str )
   aggregationMgrName = Str( help='Name of the Aggregation Manager device' )

class MacroSegmentationGroupMembersModel( Model ):
   __public__ = False

   groupMembers = Submodel( help='Device group member information for '
                            'Aggregation Manager device',
                            valueType=MacroSegmentationGroupMembersInfoModel )

   def render( self ):
      if not self.groupMembers:
         return

      table = TableOutput.TableFormatter( indent=4 )
      tableFormat = TableOutput.Format( justify='left', noBreak=True,
                                        terminateRow=True )
      tableFormat.noPadLeftIs( True )

      print( '%s is an Aggregation Manager. Current members of device-group %s:'
         % ( self.groupMembers.aggregationMgrName,
             self.groupMembers.groupName ) )
      print( 'ServiceDeviceID' )
      print( dashes() )

      for grpMem in self.groupMembers.groupMembers:
         createFormattedCell( table, grpMem, 1, tableFormat )
      createFormattedCell( table, '', 1, tableFormat )

      print( table.output().rstrip( '\n' ) )

# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
class MacroSegmentationServiceDevicesModel( Model ):
   __public__ = False

   groupName = Str( help='Name of the device group' )
   groupMembers = List( help='Device group members for an Aggregation Manager',
                          valueType=str )
   aggregationMgr = Bool( help='Is Aggregation Manager' )


class MacroSegmentationDeviceSetStatesModel( Model ):
   __public__ = False

   deviceType = Str( help='Type of the device set' )
   serviceDevices = Dict( help='A mapping between service devices and their status '
                          'information',
                          valueType=MacroSegmentationServiceDevicesModel )
   state = Str( help='State of the device set' )


class MacroSegmentationDynamicStatesModel( Model ):
   __public__ = False

   deviceSetStates = Dict( help='A mapping between device sets and their status '
                           'information',
                           valueType=MacroSegmentationDeviceSetStatesModel )
   numPoliciesProcessed = Int( help='Total policies processed', default=0 )


class MacroSegmentationDynamicModel( Model ):
   __public__ = False

   dynamicStates = Submodel( help='Device set status information',
                             valueType=MacroSegmentationDynamicStatesModel )

   def render( self ):
      # t0( 'entered render of dynamic' )
      if not self.dynamicStates:
         return

      table = TableOutput.createTable(
         ( 'Policy Source', 'Device Set', 'Service Device', 'State' ),
         tableWidth=140 )
      print( 'Total policies processed: %s' %
             self.dynamicStates.numPoliciesProcessed )
      formats = []
      for _ in range( 0, 4 ):
         f = TableOutput.Format( justify="left", maxWidth=50, minWidth=10,
                                 wrap=True )
         f.noPadLeftIs( False )
         f.padLimitIs( True )
         formats.append( f )
      table.formatColumns( *formats )

      dynStatesDict = {}
      for ds, dy in sorted( self.dynamicStates.deviceSetStates.items(),
                            key=operator.itemgetter( 0 ) ):
         dev = []
         for serviceMem in sorted( dy.serviceDevices ):
            dev.append( serviceMem )
         if not dy.deviceType in dynStatesDict:
            dynStatesDict[ dy.deviceType ] = []
         info = {}
         info[ 'deviceSet' ] = ds
         info[ 'state' ] = dy.state
         info[ 'serviceDevices' ] = dev
         dynStatesDict[ dy.deviceType ].append( info )

      for devType, devInfo in sorted( dynStatesDict.items() ):
         for devSet in devInfo:
            dev = devSet[ 'serviceDevices' ]
            if dev:
               table.newRow( policySourceName[ devType ], devSet[ 'deviceSet' ],
                             dev[ 0 ], devSet[ 'state' ] )
            else:
               table.newRow( policySourceName[ devType ], devSet[ 'deviceSet' ], '',
                             devSet[ 'state' ] )
            for each in dev[ 1: ]:
               table.newRow( '', '', each )
            table.newRow( '', '', '' )

      print( table.output().rstrip( '\n' ) )

# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
class MacroSegmentationStatusModel( Model ):
   __public__ = False

   state = Str( help='State of the device set' )
   deviceSetName = Str( help='Name of the device set' )
   deviceType = Str( help='Type of the device set' )
   mgmtIp = Str( help='Device management IP address' )
   haPeerMgmtIp = Str( help='HA peer device management IP address' )
   aggregationMgr = Bool( help='Is Aggregation Manager' )
   accessedVia = Str( help='Accessed via' )
   groupMembers = List( help='Members of the device group', valueType=str )
   timeLastSeen = Float( help='UTC timestamp when device was last seen' )
   numPoliciesProcessed = Int( help='Total number of policies processed since start',
                               default=0 )
   haEnabled = Bool( help='Is HA enabled' )
   haState = Enum( values=( HA_DISABLED_STATE, HA_ACTIVE_STATE, HA_STANDBY_STATE ),
                   help='HA state')

class MacroSegmentationServiceDeviceModel( Model ):
   __public__ = False
   deviceStatus = Dict( help='A mapping between service devices and their '
                        'status information',
                        keyType=str, valueType=MacroSegmentationStatusModel )

class MacroSegmentationServiceDeviceStatesModel( Model ):
   __public__ = False

   serviceDeviceStates = Submodel( help='Service device status information',
                                   valueType=MacroSegmentationServiceDeviceModel )

   def extractAggrMgrs( self, deviceStatusDict ):
      ''' Extract Aggregation managers from the device status dictionary
      '''
      aggrMgrDict = {}
      for servId, devStatus in list( deviceStatusDict.items() ):
         if devStatus.aggregationMgr:
            aggrMgrDict[ servId ] = devStatus
            del deviceStatusDict[ servId ]

      return aggrMgrDict

   def renderDevices( self, table, tableFormat, deviceStatusList ):
      for servId, devStatus in sorted( deviceStatusList.items() ):
         if devStatus.timeLastSeen:
            timeLastSeen = timestampToStr( devStatus.timeLastSeen )
         else:
            timeLastSeen = ''

         createFormattedCell( table, 'Device: %s' % ( servId ), 2, tableFormat )
         createFormattedCell( table, 'IP address: %s' % ( devStatus.mgmtIp ),
                              2, tableFormat, space=2 )
         createFormattedCell( table,
                              'HA state : %s' % ( devStatus.haState ),
                              2, tableFormat, space=2 )
         if devStatus.haPeerMgmtIp:
            createFormattedCell( table,
                                 'HA peer IP: %s' % ( devStatus.haPeerMgmtIp ),
                                 2, tableFormat, space=2 )
         createFormattedCell( table,
                              'Policy source type: %s' % ( devStatus.deviceType ),
                              2, tableFormat, space=2 )
         if devStatus.accessedVia:
            createFormattedCell( table,
                                 'Accessed via Aggregation Manager: %s' % (
                                    devStatus.accessedVia ), 2, tableFormat,
                                 space=2 )
         if devStatus.aggregationMgr:
            createFormattedCell( table, 'Aggregation Manager: %s'
                                 % ( devStatus.aggregationMgr ), 2, tableFormat,
                                 space=2 )
         if ( not devStatus.haEnabled or
            ( devStatus.haEnabled and devStatus.haState == HA_ACTIVE_STATE ) ):
            if devStatus.groupMembers:
               createFormattedCell( table, 'Device group member(s):', 2,
                                    tableFormat, space=2 )
               for servdevid in devStatus.groupMembers:
                  createFormattedCell( table, '%s' % ( servdevid ), 2, tableFormat,
                                       space=4 )
         createFormattedCell( table,
                              'Device set name: %s' % ( devStatus.deviceSetName ),
                              2, tableFormat, space=2 )
         createFormattedCell( table,
                              'Device set state: %s' % ( devStatus.state.title() ),
                              2, tableFormat, space=2 )
         if not devStatus.aggregationMgr:
            createFormattedCell( table, 'Total policies processed: %s'
                                 % ( devStatus.numPoliciesProcessed ),
                                 2, tableFormat, space=2 )
         createFormattedCell( table,
                              'Last seen at time: %s' % ( timeLastSeen ),
                              2, tableFormat, space=2 )
         createFormattedCell( table, '', 2, tableFormat )


   def render( self ):
      if not self.serviceDeviceStates: # self.deviceStatuses:
         return

      print( 'Service Device Policy Monitoring Status:\n' )
      table = TableOutput.TableFormatter( indent=0 )
      tableFormat = TableOutput.Format( justify='left', noBreak=True,
                              terminateRow=True )
      tableFormat.noPadLeftIs( True )

      # Extract Aggregation managers
      aggrMgrDict = self.extractAggrMgrs( self.serviceDeviceStates.deviceStatus )

      # Render Aggregation managers if present first
      self.renderDevices( table, tableFormat, aggrMgrDict )

      # Render member devices or standalone firewalls
      self.renderDevices( table, tableFormat,
          self.serviceDeviceStates.deviceStatus )

      if table.entries_:
         print( table.output().rstrip( '\n' ) )


# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

class MacroSegmentationHighAvailabilityInfoModel( Model ):
   __public__ = False

   haEnabled = Bool( help='Is High Availability enabled' )
   haMode = Enum( values=( HA_ACTIVE_PASSIVE, HA_ACTIVE_ACTIVE, '' ),
                  help='Mode of High Availability' )
   haState = Enum( values=( HA_ACTIVE, HA_PASSIVE, HA_ACTIVE_PRIMARY,
                            HA_ACTIVE_SECONDARY, HA_DEVICE_SUSPENDED, '' ),
                   help='State of High Availability' )

class MacroSegmentationHighAvailabilityModel( Model ):
   __public__ = False
   deviceHAState = Submodel( help='High-availability information for device-set',
                             valueType=MacroSegmentationHighAvailabilityInfoModel )

   def render( self ):
      if not self.deviceHAState:
         return

      table = TableOutput.TableFormatter( indent=0 )
      tableFormat = TableOutput.Format( justify='left', noBreak=True,
                                        terminateRow=True )
      tableFormat.noPadLeftIs( True )

      haInfo = self.deviceHAState
      createFormattedCell( table, 'HA Enabled: %s' %
                           ( 'True' if haInfo.haEnabled else 'False' ), 2,
                           tableFormat )
      if haInfo.haEnabled:
         createFormattedCell( table, 'HA Mode: %s' % ( haInfo.haMode ),
                              2, tableFormat )
         createFormattedCell( table, 'HA State: %s' % ( haInfo.haState ),
                              2, tableFormat )

      print( table.output().rstrip( '\n' ) )

# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
class MacroSegmentationPolicyZone( Model ):
   __public__ = False

   name = Str( help='Source zone of the policy' )
   intf = Str( help='Interfaces for the source zone' )
   ip = List( help='Ip addresses for the source zone', valueType=str )
   zoneClass = Str( help='Classification of source zone' )

class MacroSegmentationPoliciesInfoModel( Model ):
   __public__ = False

   action = Str( help='Action defined on the policy' )
   src = Submodel( help='Source zone details',
                   valueType=MacroSegmentationPolicyZone )
   dest = Submodel( help='Destination zone details',
                    valueType=MacroSegmentationPolicyZone )

   deviceSet = Str( help='Name of the device set' )
   tag = Str( help='Device policy tag(s)' )

class MacroSegmentationPoliciesModel( Model ):
   __public__ = False

   policies = Dict( help='A mapping between policy name and their policy '
                    'information', valueType=MacroSegmentationPoliciesInfoModel )
   serviceDevice = Str( help='Service device name' )

class MacroSegmentationDevicePoliciesModel( Model ):
   __public__ = False

   devicePolicies = Submodel( help='Policies read from service device that have the '
                              'MSS tag', valueType=MacroSegmentationPoliciesModel )

   def render( self ):
      if not self.devicePolicies.policies:
         return
      # only one service-device in the model
      servDev = self.devicePolicies.serviceDevice
      print( 'Policies for device: %s' % ( servDev ) )
      for policy, info in sorted( self.devicePolicies.policies.items(),
                                  key=operator.itemgetter( 0 ) ):
         print( 'Policy: {}   Tag: {}   Action: {}'.format( policy, info[ 'tag' ],
                                                     info[ 'action' ] ) )
         print( dashes( 80 ) )
         table = TableOutput.TableFormatter( indent=2 )
         tableFormat = TableOutput.Format( justify='left', noBreak=True,
                                           terminateRow=True )
         tableFormat.noPadLeftIs( True )
         tableFormat.padLimitIs( True )

         srcZone = info[ 'src' ]
         createFormattedCell( table, 'Source Zone: %s   Classification: %s'
                              % ( srcZone.name, srcZone.zoneClass ),
                              2, tableFormat )

         createFormattedCell( table, '   Interfaces: ' + srcZone.intf, 1,
                              tableFormat )

         if srcZone.zoneClass != 'nonIntercept':
            if info[ 'src' ].ip:
               sortedList = sorted( srcZone.ip )
               l = 0
               for idx, i in enumerate( sortedList ):
                  l += len( i )
                  if l > 50:
                     sortedList[ idx ] = '\n' + i
                     l = 0

               ipList = ', '.join( sortedList )
               ipList = 'IP addresses: ' + ipList
               ipNewLine = ipList.split( '\n' )
               createFormattedCell( table, ' ' *3 + ipNewLine.pop( 0 ), 1,
                                    tableFormat )
               for each in ipNewLine:
                  createFormattedCell( table, ' ' *5 + each, 1, tableFormat )

         destZone = info[ 'dest' ]
         createFormattedCell( table, '', 1, tableFormat )
         createFormattedCell( table,
                              'Destination Zone: %s   Classification: %s'
                              % ( destZone.name, destZone.zoneClass ),
                              2, tableFormat )
         createFormattedCell( table, '   Interfaces: ' + destZone.intf, 1,
                              tableFormat )

         if destZone.zoneClass != 'nonIntercept':
            if destZone.ip:
               sortedList = sorted( destZone.ip )
               l = 0
               for idx, i in enumerate( sortedList ):
                  l += len( i )
                  if l > 50:
                     sortedList[ idx ] = '\n' + i
                     l = 0

               ipList = ', '.join( sortedList )
               ipList = 'IP addresses: ' + ipList
               ipNewLine = ipList.split( '\n' )
               createFormattedCell( table, ' ' *3 + ipNewLine.pop( 0 ), 1,
                                    tableFormat )
               for each in ipNewLine:
                  createFormattedCell( table, ' ' *5 + each, 1, tableFormat )

         createFormattedCell( table, '', 1, tableFormat )
         print( table.output().rstrip( '\n' ) )

# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
class MacroSegmentationNetworkInfoModel( Model ):
   __public__ = False

   state = Enum( values=( LINK_STATE_UP, LINK_STATE_DOWN, LINK_STATE_UNKNOWN ),
                 help='State of the device set' )
   vlan = Str( help='Vlan for the interface' )
   mode = Str( help='Forwarding mode of the interface' )
   zone = Str( help='Zone infornamtion' )
   virtualWire = Str( help='Virtual wire information' )

class MacroSegmentationNetworkModel( Model ):
   __public__ = False

   interfaces = Dict( help='A mapping between interfaces and their network '
                      'information', valueType=MacroSegmentationNetworkInfoModel )
   serviceDevice = Str( help='Service device name' )

class MacroSegmentationDeviceNetworkModel( Model ):
   __public__ = False

   deviceNetworks = Submodel( help='Service device network interface information',
                              valueType=MacroSegmentationNetworkModel )

   def render( self ):
      if not self.deviceNetworks:
         return

      table = TableOutput.createTable( ( 'Interface', 'Link State', 'VLAN(s)',
                                         'Attributes' ) )

      formats = []

      fmtInterface = TableOutput.Format( justify="left", maxWidth=18, minWidth=18,
                                         wrap=True )  # Interface
      fmtInterface.noPadLeftIs( False )
      fmtInterface.padLimitIs( True )
      formats.append( fmtInterface )

      fmtState = TableOutput.Format( justify="left", maxWidth=11, minWidth=11,
                                     wrap=True )  # Link State
      fmtState.noPadLeftIs( False )
      fmtState.padLimitIs( True )
      formats.append( fmtState )

      fmtVlan = TableOutput.Format( justify="left", maxWidth=13, minWidth=13,
                                    wrap=True )  # VLAN(s)
      fmtVlan.noPadLeftIs( False )
      fmtVlan.padLimitIs( True )
      formats.append( fmtVlan )

      fmtAttributes = TableOutput.Format( justify="left", maxWidth=33, minWidth=33,
                                          wrap=True )  # Attributes
      fmtAttributes.noPadLeftIs( False )
      fmtAttributes.padLimitIs( True )
      formats.append( fmtAttributes )

      table.formatColumns( *formats )

      infoModel = list( self.deviceNetworks.interfaces.items() )

      for name, nw in sorted( infoModel ):
         table.newRow( name, nw.state, nw.vlan, 'Virtual Wire: %s' %
                       nw[ 'virtualWire' ] if nw[ 'virtualWire' ] else '' )
         if nw[ 'zone' ]:
            table.newRow( '', '', '', 'Zone: % s' % nw[ 'zone' ] )
         if nw[ 'mode' ]:
            table.newRow( '', '', '', 'Mode/Peer: %s' % nw[ 'mode' ] )
         table.newRow()  # for blank line between intfs

      print( table.output().rstrip( '\n' ) )

# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
class MacroSegmentationNeighborInfoModel( Model ):
   __public__ = False

   neighborIntf = Str( help='Neighbor name' )
   chassisId = Str( help='Chassis id' )
   systemName = Str( help='System name' )
   managementIp = Str( help='Management ip' )
   description = Str( help='Description of the neighbor' )

class MacroSegmentationNeighborModel( Model ):
   __public__ = False

   neighbors = Dict( help='A mapping between interfaces and their neighbor '
                     'information', valueType=MacroSegmentationNeighborInfoModel )
   serviceDevice = Str( help='Service device name' )

class MacroSegmentationDeviceNeigborModel( Model ):
   __public__ = False

   deviceIntfNeighbors = Submodel( help='Service device interface neighbors',
                                   valueType=MacroSegmentationNeighborModel )

   def render( self ):
      if not self.deviceIntfNeighbors:
         return

      table = TableOutput.createTable( ( 'Interface',
                                         'Neighbor Device Information' ) )
      formats = []
      for _ in range( 0, 2 ):
         fmt = TableOutput.Format( justify="left", maxWidth=65, minWidth=14,
                                 wrap=True )
         fmt.noPadLeftIs( False )
         fmt.padLimitIs( True )
         formats.append( fmt )
      table.formatColumns( *formats )

      servDev = self.deviceIntfNeighbors.serviceDevice
      infoModel = list( self.deviceIntfNeighbors.neighbors.items() )

      print( 'Neighbor information for device: %s' % ( servDev ) )
      for key, val in sorted( infoModel ):
         outputString = ''
         if val[ 'neighborIntf' ]:
            outputString += 'Interface:    %s\n' % val[ 'neighborIntf' ]
         if val[ 'chassisId']:
            outputString += 'System MAC:   %s\n' % val[ 'chassisId']
         if val[ 'systemName' ]:
            outputString += 'System Name:  %s\n' % val[ 'systemName' ]
         if val[ 'managementIp' ]:
            outputString += 'Mgmt Address: %s\n' % val[ 'managementIp' ]
         if val[ 'description' ]:
            outputString += 'Description:  %s\n' % val[ 'description' ]

         table.newRow( key, outputString )

      print( table.output().rstrip( '\n' ) )

# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
class MacroSegmentationResourcesInfoModel( Model ):
   __public__ = False

   resourceInfo = Str( help='Device resource information' )
   serviceDeviceName = Str( help='Service device name' )

class MacroSegmentationResourcesModel( Model ):
   __public__ = False

   deviceResources = Submodel( help='Service device system resource information',
                               valueType=MacroSegmentationResourcesInfoModel )

   def render( self ):
      if not self.deviceResources:
         return

      servDev = self.deviceResources.serviceDeviceName
      print( 'Resources status information for Device: %s\n' % ( servDev ) )
      print( dashes( 79 ) )
      print( self.deviceResources.resourceInfo )


# # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

def createFormattedCell( table, desc, cols, fmt, space=0 ):
   table.startRow()
   desc = ' ' * space + desc
   table.newFormattedCell( desc, nCols=cols, format=fmt )

def dashes( num=65 ):
   if num == 65:
      return '-' * 65
   else:
      return '-' * num
