# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import itertools
import BasicCli
import CliCommand
import CliMatcher
from CliPlugin import TechSupportCli
from CliPlugin.ControllerCli import serviceAfterShowKwMatcher
from CliPlugin.ControllerdbLib import registerNotifiee, switchIdCache
from CliPlugin.MssShowCliModel import ( SwitchIdModel,
                                        FlowInfoModel,
                                        FlowMatchModel,
                                        FlowActionModel,
                                        NextHopActionModel,
                                        MssStatusModel,
                                        MssPolicyEnforcementModel,
                                        MssPolicyEnforcementRulesModel,
                                        MssPolicyRootModel,
                                        MssPolicySourceModel,
                                        MssPolicyDeviceModel,
                                        MssPolicyVsysModel,
                                        MssPolicyNetworkVrfModel,
                                        MssPolicyL3Model,
                                        MssPolicyIpModel,
                                        MssPolicyModifierModel,
                                        MssVrfModel, MssVrfSourceModel,
                                        MssVrfVInstModel,
                                        MssVrfNetworkVRFModel, MssVrfFwVRFModel,
                                        MssVrfSubnetModel,
                                        MssVrfDeviceModel )
from CliPlugin.IpAddrMatcher import ipAddrMatcher, ipPrefixMatcher
import LazyMount
import Tac
from MssCliLib import getHostname, PolicyFilter, displayStringForState
from MssStaticCliLib import mssL3PolicyCliSource
import ShowCommand
from ArnetModel import IpGenericAddr
IpProtocolNumber = Tac.Type( 'Arnet::IpProtocolNumber' )
ipProtoUnknown = Tac.enumValue( IpProtocolNumber, 'ipProtoUnknown' )

# Globals
topologyStatus = None
config = None
status = None
mssL3Status = None
policyL3ConfigDir = None
policyL3StatusDir = None
policyL3ConfigV2Dir = None
policyL3StatusV2Dir = None
servicePolicyStatus = None
serviceDeviceL3V2Dir = None
MssL3MssPriority = Tac.Value( 'MssL3::MssPriority' )
MssL3V2MssPriority = Tac.Value( 'MssL3V2::MssPriority' )
MssL3MssAction = Tac.Type( 'MssL3::MssAction' )
MssL3V2MssAction = Tac.Type( 'MssL3V2::MssAction' )
MssL3MssPolicyModifier = Tac.Type( 'MssL3::MssPolicyModifier' )
MssL3V2MssPolicyModifier = Tac.Type( 'MssL3V2::MssPolicyModifier' )
FlowKey = Tac.Type( 'MssL3::V2::FlowKey' )
MssRuleState = Tac.Type( "MssL3V2::MssPolicyState" )
MssIpState = Tac.Type( "MssL3::MssIpState" )
LocalData = Tac.Type( 'MssL3V2::LocalData' )
PolicyEnforcementConsistency = Tac.Type( 'Mss::PolicyEnforcementConsistency' )
AddressFamily = Tac.Type( "Arnet::AddressFamily" )

matcherDetail = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='Show policy status in detail' )
matcherDevice = CliMatcher.KeywordMatcher( 'device',
      helpdesc='Service device' )
matcherName = CliMatcher.KeywordMatcher( 'name',
      helpdesc='Filter policy' )
matcherInternalRedirect = CliMatcher.KeywordMatcher( 'internal-redirect',
      helpdesc='internal-redirect policies' )
matcherPolicy = CliMatcher.KeywordMatcher( 'policy',
      helpdesc='Show policy' )
matcherPolicyName = CliMatcher.QuotedStringMatcher(
      helpdesc='Policy name', helpname='WORD' )
matcherSource = CliMatcher.KeywordMatcher( 'source',
      helpdesc='Source of a policy' )
matcherStatic = CliMatcher.KeywordMatcher( 'static',
      helpdesc='Policies configured via CLI' )
nodeMss = CliCommand.singleKeyword( 'mss',
      helpdesc='Macro-Segmentation Service' )
matcherFull = CliMatcher.KeywordMatcher( 'full',
      helpdesc='Display full list of IP addresses' )
sharedObj = object()

class PolicyFilterExpr( CliCommand.CliExpression ):
   expression = '( name POLICY )'
   data = {
      'name' : CliCommand.singleNode( matcherName, sharedMatchObj=sharedObj ),
      'POLICY' : matcherPolicyName,
   }

   @staticmethod
   def adapter( mode, args, argList ):
      val = args.pop( 'POLICY', None )
      if val:
         args[ 'POLICY' ] = val[ 0 ]

def vrfSourceNameFn( mode ):  
   return { deviceKey.getFwType() for sdConfig in serviceDeviceL3V2Dir.values()
            for deviceKey in sdConfig.serviceDevice }
   
def policySourceNameFn( mode ): 
   # use policy enforcement consistency to determin V1/V2        
   if config.policyEnforcementConsistency != PolicyEnforcementConsistency.bestEffort:
      return { devId.getFwType() for feeder in policyL3ConfigV2Dir.values()
               for devId in feeder.policySet }                    
   return { originConfig.source for feeder in policyL3ConfigDir.values() 
            for policySetConfig in feeder.policySet.values()  
            for policyConfig in policySetConfig.policy.values()   
            for ruleConfig in policyConfig.rule.values()   
            for originConfig in ruleConfig.origin.values() }
      
def generateSourceFilterExpr( nameFn, helpdesc='' ): 
   class SourceFilterExpr( CliCommand.CliExpression ):
      _helpdesc = 'Service device type'   
      expression = '( source SOURCE )' 
      data = { 
            'source': CliCommand.singleKeyword( 'source', 
                                                helpdesc=helpdesc ),
            'SOURCE': CliMatcher.DynamicNameMatcher( nameFn, helpdesc=_helpdesc )
      }

      @staticmethod 
      def adapter( mode, args, argsList ): 
         val = args.pop( 'SOURCE', None ) 
         if val:     
            args[ 'SOURCE' ] = val[ 0 ] 
   
   return SourceFilterExpr 

def resolveSwitchId( switchId ):
   switchId = switchId.stringValue.replace( ':', '-' )
   hostname = switchIdCache.getHost( switchId )
   if hostname:
      return hostname
   return ''

class DeviceFilterExpr( CliCommand.CliExpression ):
   expression = '( device DEVICE )'
   data = {
      'device': CliCommand.singleNode( matcherDevice ),
      'DEVICE': CliMatcher.PatternMatcher( pattern='.+',
                                           helpdesc='Device name',
                                           helpname='WORD' )
   }

   @staticmethod
   def adapter( mode, args, argList ):
      val = args.pop( 'DEVICE', None )
      if val:
         args[ 'DEVICE' ] = val[ 0 ]

class DeviceFilterExprV2( CliCommand.CliExpression ):
   expression = '( device DEVICE [ virtual instance VINST ] [vrf VRF] )'
   data = {
      'device': CliCommand.singleNode( matcherDevice ),
      'DEVICE': CliMatcher.PatternMatcher( pattern='.+',
                                           helpdesc='Device name',
                                           helpname='WORD' ),
      'virtual': CliCommand.singleKeyword( 'virtual',
                                           helpdesc='Virtual Instance Name' ),
      'instance': CliCommand.singleKeyword( 'instance',
                                            helpdesc='Virtual Instance Name' ),
      'VINST': CliMatcher.PatternMatcher( pattern='.+',
                                          helpdesc='Virtual Instance Name',
                                          helpname='WORD' ),
      'vrf': CliCommand.singleKeyword( 'vrf',
                                       helpdesc='VRF Name' ),
      'VRF': CliMatcher.PatternMatcher( pattern='.+',
                                        helpdesc='VRF Name',
                                        helpname='WORD' )
   }

   @staticmethod
   def adapter( mode, args, argList ):
      val = args.pop( 'DEVICE', None )
      if val:
         args[ 'DEVICE' ] = val[ 0 ]
      val = args.pop( 'VINST', None )
      if val:
         args[ 'VINST' ] = val[ 0 ]
      val = args.pop( 'VRF', None )
      if val:
         args[ 'VRF' ] = val[ 0 ]

class SwitchOrNameFilterExpr( CliCommand.CliExpression ):
   expression = '( switch SWITCHID )'
   data = {
      'switch' : CliCommand.singleKeyword( 'switch', helpdesc='Switch ID or Name' ),
      'SWITCHID': CliMatcher.PatternMatcher( pattern='.+',
                                             helpdesc='Switch ID or Name',
                                             helpname='WORD' )
   }

   @staticmethod
   def adapter( mode, args, argList ):
      val = args.pop( 'SWITCHID', None )
      if val:
         args[ 'SWITCHID' ] = val[ 0 ]

class ActiveFilterExpr( CliCommand.CliExpression ):
   expression = '( active )'
   data = {
      'active' : CliCommand.singleKeyword( 'active', helpdesc='Active policies' )
   }

   @staticmethod
   def adapter( mode, args, argList ):
      val = args.pop( 'active', None )
      if val:
         args[ 'ACTIVE' ] = True

class GroupRedirectFilterExpr( CliCommand.CliExpression ):
   expression = '( internal-redirect )'
   data = {
      'internal-redirect' : CliCommand.singleNode( matcherInternalRedirect,
                                                   sharedMatchObj=sharedObj ),
   }

   @staticmethod
   def adapter( mode, args, argList ):
      val = args.pop( 'internal-redirect', None )
      if val:
         args[ 'GROUP' ] = True

class Ipv4FilterExpr( CliCommand.CliExpression ):
   expression = '( ipv4 ( SUBNET | ( range START END ) ) )'
   data = {
      'ipv4' : CliCommand.singleKeyword( 'ipv4', helpdesc='IPv4 subnet' ),
      'SUBNET': ipPrefixMatcher,
      'range' : CliCommand.singleKeyword( 'range', helpdesc='IPv4 range' ),
      'START': ipAddrMatcher,
      'END': ipAddrMatcher
   }

   @staticmethod
   def adapter( mode, args, argList ):
      val = args.pop( 'SUBNET', None )
      if val:
         args[ 'SUBNET' ] = val[ 0 ]
      val = args.pop( 'START', None )
      if val:
         args[ 'START' ] = val[ 0 ]
      val = args.pop( 'END', None )
      if val:
         args[ 'END' ] = val[ 0 ]

filterExprV1 = CliCommand.setCliExpression( {
   'POLICY' : PolicyFilterExpr,
   'SOURCE' : generateSourceFilterExpr( policySourceNameFn, 
                                        helpdesc='Source of a policy' ), 
   'DEVICE' : DeviceFilterExpr
} )

filterExprV2 = CliCommand.setCliExpression( {
   'POLICY' : PolicyFilterExpr,
   'SOURCE' : generateSourceFilterExpr( policySourceNameFn,
                                        helpdesc='Source of a policy' ), 
   'DEVICE' : DeviceFilterExprV2,
   'SWITCH' : SwitchOrNameFilterExpr,
   'IPV4' : Ipv4FilterExpr,
   'ACTIVE' : ActiveFilterExpr,
   'GROUP' : GroupRedirectFilterExpr,
} )

filterExprInternal = filterExprV1

filterExpr = filterExprV2

# -----------------------------------------------------------------------------------
# This CLI plugin defines the following show commands:
#     show service mss status
#     show service mss policy [ name <policy-name> ]
#                             [ source (static|<dyn-src-name>) ]
#                             [ device <device-name>
#                                [ virtual instance < name > ] [ vrf < name > ] ]
#                             [ switch <switch-id> ]
#                             [ ipv4 <PREFIX> ]
#                             [ ipv4 range   <START>    <END>    ]
#                             [ active ]
#                             [ internal-redirect ]
#                             [ detail ]
#     show service mss vrf [ source <source-name> ]
#                          [ device <device-name>
#                               [ virtual instance <name> [ name <name> ] ] ]
# -----------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# show service mss status
#------------------------------------------------------------------------------
def showMssStatus( mode, args ):
   rules = MssPolicyEnforcementRulesModel(
                                       group=config.policyEnforceRule.group,
                                       verbatim=config.policyEnforceRule.verbatim )
   policyEnforcement = MssPolicyEnforcementModel(
         consistency=config.policyEnforcementConsistency, rules=rules )
   return MssStatusModel( enabled=config.enable, running=mssL3Status.running,
                          policyEnforcement=policyEnforcement )

class ServiceMssStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service mss status'
   data = {
      'service': serviceAfterShowKwMatcher,
      'mss': nodeMss,
      'status': 'Show status information',
   }
   handler = showMssStatus
   cliModel = MssStatusModel

BasicCli.addShowCommandClass( ServiceMssStatusCmd )

#------------------------------------------------------------------------------
# show service mss policy [ name <policy-name> ]
#                         [ source (static|<dyn-src-name>) ]
#                         [ device <device-name>
#                            [ virtual instance < name > ] [ vrf < name > ] ]
#                         [ switch <switch-id> ]
#                         [ ipv4 <PREFIX> ]
#                         [ ipv4 range   <START>    <END>    ]
#                         [ active ]
#                         [ internal-redirect ]
#                         [ detail [ full ] ]
#------------------------------------------------------------------------------
def showPolicy( mode, args ): 
   IpExpander = Tac.Type( 'MssL3Common::IpExpander' )

   def getMatchIps( match ):
      ips = set()
      if match:
         for ipPrefix in itertools.chain( match.srcIp, match.dstIp ):
            ipSet = IpExpander.expandIpPrefix( ipPrefix )
            ips |= set( ipSet.ip )
         for ipRange in itertools.chain( match.srcIpRange, match.dstIpRange ):
            ipSet = IpExpander.expandIpRange( ipRange.startIp, ipRange.endIp )
            ips |= set( ipSet.ip )
      return ips

   def getSrcDevVsysNames( deviceId ):
      sourceName = deviceId.getFwType()
      if not sourceName:
         sourceName = LocalData.zombieSourceName
      deviceName = deviceId.getPhyInstanceName()
      if not deviceName:
         deviceName = LocalData.zombieDeviceName
      vsysName = deviceId.getVirtInstanceName()
      return ( sourceName, deviceName, vsysName )

   def calculateState( state1, state2 ):
      # Returns the "worse" state
      if MssRuleState.error in ( state1, state2 ):
         return MssRuleState.error
      if MssRuleState.pending in ( state1, state2 ):
         return MssRuleState.pending
      if MssRuleState.dryrunError in ( state1, state2 ):
         return MssRuleState.dryrunError
      if MssRuleState.dryrunPending in ( state1, state2 ):
         return MssRuleState.dryrunPending
      if MssRuleState.dryrunActive in ( state1, state2 ):
         return MssRuleState.dryrunActive
      return MssRuleState.active

   def populateIpModelStatus( policyIpModel, ipStatus ):
      policyIpModel.state = displayStringForState( ipStatus.state, forCli=False )
      policyIpModel.deleting = not ipStatus.required
      policyIpModel.unconvergedSwitches = []
      for switch in ipStatus.pendingSwitch:
         switchIdModel = SwitchIdModel()
         switchIdModel.switchId = str( switch )
         switchIdModel.hostname = resolveSwitchId( switch )
         policyIpModel.unconvergedSwitches.append( switchIdModel )

   def populateIpModels( policyModel, origin, filterIpSet, ruleOriginStatus ):
      policyIps = getMatchIps( origin.match )
      for ip in policyIps:
         if filterIpSet and ip not in filterIpSet:
            # IP address is filtered out
            continue

         if ruleOriginStatus and ip in ruleOriginStatus.ip:
            ipStatus = ruleOriginStatus.ip[ ip ]
         else:
            ipStatus = Tac.newInstance( 'MssL3V2::MssIpStatus', ip )
            # Status is not available for an IP required by config
            ipStatus.required = True
         policyIpModel = MssPolicyIpModel()
         populateIpModelStatus( policyIpModel, ipStatus )
         policyModel.ips[ str( ip ) ] = policyIpModel

   def populateL3V2PolicyModel( policy, origin, priority, vrf, deviceId,
                                ruleConfig, ruleStatus ):
      sourceName, deviceName, vsysName = getSrcDevVsysNames( deviceId )
      if priority == MssL3V2MssPriority.ipRedirect:
         ruleOriginStatus = ruleStatus.origin.get(
            LocalData.internalRedirectPolicyName )
      else:
         ruleOriginStatus = ruleStatus.origin.get( policy )
      state = ruleOriginStatus.state if ruleOriginStatus else None
      if filterPolicy.areIpsNeeded():
         policyIps = getMatchIps( origin.match )
      else:
         policyIps = None
      policySwitches = set()
      if filterPolicy.areSwitchesNeeded() and ruleOriginStatus:
         for flowStatus in ruleOriginStatus.flow.values() :
            for switchId in flowStatus.switchStatus:
               policySwitches.add( str( switchId ) )
               switchHostname = resolveSwitchId( switchId )
               if switchHostname:
                  policySwitches.add( switchHostname )
         for ipStatus in ruleOriginStatus.ip.values() :
            for switchId in ipStatus.switchStatus:
               policySwitches.add( str( switchId ) )
               switchHostname = resolveSwitchId( switchId )
               if switchHostname:
                  policySwitches.add( switchHostname )
      if filterPolicy.isAllowed(
            policy, sourceName, deviceName,
            vsysName, vrf, policyIps,
            policySwitches, state ):
         src = model.sources.setdefault( sourceName, MssPolicySourceModel() )
         dev = src.devices.setdefault( deviceName, MssPolicyDeviceModel() )
         vsys = dev.virtualInstances.setdefault(
            vsysName, MssPolicyVsysModel() )
         vrfModel = vsys.networkVrfs.setdefault(
            vrf, MssPolicyNetworkVrfModel() )
         policyModel = vrfModel.policiesL3.get( policy )
         if policyModel is None:
            policyModel = MssPolicyL3Model( redirectStatus='N/A',
                                            offloadStatus='N/A' )
            vrfModel.policiesL3[ policy ] = policyModel
         policyModel.tags = origin.tags
         policyModel.internal = False
         for modifier in origin.policyModifierSet:
            modifierModel = MssPolicyModifierModel()
            modifierModel.policyModifier = modifier
            policyModel.policyModifierSet.append( modifierModel )
         if not ruleOriginStatus:
            if priority == MssL3V2MssPriority.ipRedirect:
               policyModel.dependencies.append(
                  LocalData.internalRedirectPolicyName )
               populateIpModels( policyModel, origin, filterPolicy.ipv4, None )
            return

         policyModel.dependencies = []
         offloadStatus = policyModel.offloadStatus
         redirectStatus = policyModel.redirectStatus
         if sourceName == LocalData.zombieSourceName:
            # Zombie Host Policy. No redirect or offload status
            pass
         elif LocalData.zombiePolicyName in policy:
            # Zombie policy
            for flowName, flowStatus in ruleOriginStatus.flow.items():
               flowKey = FlowKey( flowName )
               action = flowKey.getAction()
               if action.drop or action.bypass:
                  # Offload flow
                  offloadStatus = calculateState( offloadStatus, flowStatus.state )
               else:
                  # Redirect flow
                  redirectStatus = calculateState( redirectStatus, flowStatus.state )
         elif LocalData.internalRedirectPolicyName in policy:
            # Internal Group Redirect policy
            redirectStatus = ruleOriginStatus.state
            policyModel.internal = True
         elif priority == MssL3V2MssPriority.ipRedirect:
            # Redirect policy or the redirect origin of offload policy
            policyModel.dependencies.append( LocalData.internalRedirectPolicyName )
            redirectStatus = ruleOriginStatus.state
         elif origin.action == MssL3MssAction.redirect:
            # Redirect verbatim policy
            redirectStatus = ruleOriginStatus.state
         else:
            # Offload verbatim policy or the offload origin of offload policy
            offloadStatus = ruleOriginStatus.state

         if LocalData.internalRedirectPolicyName in policyModel.dependencies or \
            sourceName == LocalData.zombieSourceName:
            populateIpModels( policyModel, origin, filterPolicy.ipv4,
                              ruleOriginStatus )

         if priority != MssL3V2MssPriority.ipRedirect or \
            LocalData.internalRedirectPolicyName in policy or \
            LocalData.zombiePolicyName in policy:
            for flowName, flowStatus in ruleOriginStatus.flow.items():
               flowInfoModel = FlowInfoModel()
               flowInfoModel.state = displayStringForState( flowStatus.state,
                                                            forCli=False )
               flowInfoModel.deleting = not flowStatus.required
               flowInfoModel.unconvergedSwitches = []
               for switch in flowStatus.pendingSwitch:
                  switchIdModel = SwitchIdModel()
                  switchIdModel.switchId = str( switch )
                  switchIdModel.hostname = resolveSwitchId( switch )
                  flowInfoModel.unconvergedSwitches.append( switchIdModel )
               flowInfoModel.priority = flowStatus.priority
               flowInfoModel.match = FlowMatchModel()
               flowKey = FlowKey( flowName )
               match = flowKey.getMatch()
               flowInfoModel.match.ethertype = match.etherType
               flowInfoModel.match.srcIpv4 = match.srcIp
               flowInfoModel.match.dstIpv4 = match.dstIp
               flowInfoModel.match.srcGroup = \
                  match.srcGroup if match.srcGroup else None
               flowInfoModel.match.dstGroup = \
                  match.dstGroup if match.dstGroup else None
               flowInfoModel.match.ipProto = match.ipProto if \
                  match.ipProto != ipProtoUnknown else None
               flowInfoModel.match.srcL4 = match.srcL4 if match.srcL4 else None
               flowInfoModel.match.dstL4 = match.dstL4 if match.dstL4 else None
               flowInfoModel.action = FlowActionModel()
               action = flowKey.getAction()
               flowInfoModel.action.drop = action.drop
               flowInfoModel.action.bypass = action.bypass
               flowInfoModel.action.nextHop = NextHopActionModel()
               if action.nextHop:
                  nextHop = IpGenericAddr()
                  nextHop.ip = action.nextHop
                  flowInfoModel.action.nextHop.v4.append( nextHop )
               policyModel.flows[ str( flowName ) ] = flowInfoModel

         policyModel.offloadStatus = displayStringForState( offloadStatus,
                                                            forCli=False )
         policyModel.redirectStatus = displayStringForState( redirectStatus,
                                                             forCli=False )

   def populateL3PolicyModel( policy, origin, priority, vrf, deviceId,
                              ruleConfig, ruleStatus ):
      if filterPolicy.isAllowed( policy, origin.source, deviceId ):
         src = model.sources.setdefault( origin.source, MssPolicySourceModel() )
         dev = src.devices.setdefault( deviceId, MssPolicyDeviceModel() )
         # virtual instance are not supported in best-effort mode,
         # we use "default" as unique value here for every vendor.
         vsys = dev.virtualInstances.setdefault(
            'default', MssPolicyVsysModel() )
         vrfModel = vsys.networkVrfs.setdefault(
            'default', MssPolicyNetworkVrfModel() )
         policyModel = vrfModel.policiesL3.get( policy )
         if policyModel is None:
            policyModel = MssPolicyL3Model( redirectStatus='N/A',
                                            offloadStatus='N/A' )
            vrfModel.policiesL3[ policy ] = policyModel
         policyModel.tags = origin.tags
         for modifier in origin.policyModifierSet:
            modifierModel = MssPolicyModifierModel()
            modifierModel.policyModifier = modifier
            policyModel.policyModifierSet.append( modifierModel )
         if priority == MssL3MssPriority.ipRedirect or \
            ( MssL3MssPolicyModifier.verbatim in origin.policyModifierSet and \
              ruleConfig.action == MssL3MssAction.redirect ):
            policyModel.redirectStatus = ruleStatus.state
         else:
            policyModel.offloadStatus = ruleStatus.state

         policyModel.internal = False
         policyIps = set()
         if priority == MssL3MssPriority.ipRedirect and origin.match:
            for ipPrefix in itertools.chain( origin.match.srcIp,
                                             origin.match.dstIp ):
               ipSet = IpExpander.expandIpPrefix( ipPrefix )
               policyIps |= set( ipSet.ip )
            for ipRange in itertools.chain( origin.match.srcIpRange,
                                            origin.match.dstIpRange ):
               ipSet = IpExpander.expandIpRange( ipRange.startIp, ipRange.endIp )
               policyIps |= set( ipSet.ip )

         for ip in policyIps:
            if ruleStatus.match and ip in ruleStatus.match.srcIp:
               ipStatus = ruleStatus.match.srcIp[ ip ]
            else:
               ipStatus = Tac.newInstance( 'MssL3::MssIpStatus', ip )
            policyIpModel = MssPolicyIpModel()
            policyIpModel.state = ipStatus.state
            for switch in ipStatus.pendingSwitch:
               switchIdModel = SwitchIdModel()
               switchIdModel.switchId = str( switch )
               policyIpModel.unconvergedSwitches.append( switchIdModel )
            policyModel.ips[ str( ip ) ] = policyIpModel

   model = MssPolicyRootModel()
   model.setDetail( 'detail' in args )
   model.setFullIp( 'full' in args )
   model.setConsistency( config.policyEnforcementConsistency if config
                         else PolicyEnforcementConsistency.bestEffort )

   if not status.running and not mssL3Status.running:
      mode.addWarning( "Macro-Segmentation is not running" ) 
      model.setMssNotRunning()
      return model

   if args.get( 'SUBNET' ):
      ipSet = IpExpander.expandIpPrefix(
         Tac.Value( 'Arnet::IpGenPrefix', args[ 'SUBNET' ] ) )
      ips = set()
      ips |= set( ipSet.ip )
   elif args.get( 'range' ):
      ipSet = IpExpander.expandIpRange(
         Tac.Value( 'Arnet::IpGenAddr', args.get( 'START', '' ) ),
         Tac.Value( 'Arnet::IpGenAddr', args.get( 'END', '' ) ) )
      ips = set()
      ips |= set( ipSet.ip )
   else:
      ips = None
   filterPolicy = PolicyFilter( policy=args.get( 'POLICY' ),
                                source=args.get( 'SOURCE' ),
                                device=args.get( 'DEVICE' ),
                                vsys=args.get( 'VINST' ),
                                vrf=args.get( 'VRF' ),
                                switch=args.get( 'SWITCHID' ),
                                ipv4=ips,
                                active = args.get( 'ACTIVE' ),
                                group=args.get( 'GROUP' ) )
   l3V2Enabled = config and\
                 config.policyEnforcementConsistency == \
                 PolicyEnforcementConsistency.strict
   if l3V2Enabled:
      configDir = policyL3ConfigV2Dir
      statusDir = policyL3StatusV2Dir
   else:
      configDir = policyL3ConfigDir
      statusDir = policyL3StatusDir
   for feederName, feeder in configDir.items():
      for deviceId, policySetConfig in feeder.policySet.items():
         for vrf, policyConfig in policySetConfig.policy.items():
            for priority, ruleConfig in policyConfig.rule.items():
               # identify the rule state
               try:
                  policySourceStatus = statusDir[ feederName ]
                  policySetStatus = policySourceStatus.policySet[ deviceId ]
                  policyStatus = policySetStatus.policy[ vrf ]
                  ruleStatus = policyStatus.rule[ priority ]
               except KeyError:
                  # The rule state has not been populated yet
                  # Create a dummy one with the state: Pending
                  if l3V2Enabled:
                     ruleStatus = Tac.newInstance(
                        'MssL3V2::MssRuleStatus', priority )
                  else:
                     ruleStatus = Tac.newInstance( 'MssL3::MssRuleStatus', priority )
                     ruleState = Tac.Type( 'MssL3::MssRuleState' )
                     ruleStatus.state = ruleState.rulePending
               # Iterate over ruleConfig.origin to identify policies
               for policy, originConfig in ruleConfig.origin.items():
                  if l3V2Enabled:
                     populateL3V2PolicyModel( policy, originConfig, priority, vrf,
                                              deviceId, ruleConfig, ruleStatus )
                  else:
                     populateL3PolicyModel( policy, originConfig, priority, vrf,
                                            deviceId, ruleConfig, ruleStatus )

   if l3V2Enabled:    # pylint: disable-msg = R1702
      # Loop over the status to catch zombie policies
      for feederName, feeder in statusDir.items():
         for deviceId, policySetStatus in feeder.policySet.items():
            for vrf, policyStatus in policySetStatus.policy.items():
               for priority, ruleStatus in policyStatus.rule.items():
                  # identify the rule config
                  try:
                     policySourceConfig = configDir[ feederName ]
                     policySetConfig = policySourceConfig.policySet[ deviceId ]
                     policyConfig = policySetConfig.policy[ vrf ]
                     ruleConfig = policyConfig.rule[ priority ]
                  except KeyError:
                     # The rule config has not been populated yet
                     ruleConfig = None
                  # Iterate over ruleStatus.origin to identify policies
                  for policy, originStatus in ruleStatus.origin.items():
                     if ruleConfig and policy in ruleConfig.origin:
                        # This policy has an origin config. It is not a zombie
                        # Skip as it has been covered by the previous loop
                        continue
                     # Create a dummy ruleConfig
                     ruleConfig1 = Tac.newInstance(
                        'MssL3V2::MssRuleConfig', priority )
                     originConfig = ruleConfig1.newOrigin( policy )
                     originConfig.match = ()
                     # Fill the originConfig.match with matches from the flow names
                     # of MssRuleOriginStatus.flow or from the IPs of
                     # MssRuleOriginStatus.ip
                     for flowName in originStatus.flow:
                        flowKey = FlowKey( flowName )
                        match = flowKey.getMatch()
                        if match.srcIp.af == AddressFamily.ipv4:
                           originConfig.match.srcIp[ match.srcIp ] = True
                        if match.dstIp.af == AddressFamily.ipv4:
                           originConfig.match.dstIp[ match.dstIp ] = True
                     for ip in originStatus.ip:
                        if ip.af == AddressFamily.ipv4:
                           originConfig.match.srcIp[
                              Tac.Value( 'Arnet::IpGenPrefix', ip.stringValue )
                           ] = True

                     populateL3V2PolicyModel( policy, originConfig, priority, vrf,
                                              deviceId, ruleConfig, ruleStatus )

      # Move the internal redirect policy to the last position
      for sourceModel in model.sources.values():
         for deviceModel in sourceModel.devices.values():
            for vsysModel in deviceModel.virtualInstances.values():
               for vrf, vrfModel in vsysModel.networkVrfs.items():
                  # Move the internal redirect policy to the last position
                  internalRedirectPolicy = vrfModel.policiesL3.pop(
                     LocalData.internalRedirectPolicyName, None )
                  if internalRedirectPolicy:
                     vrfModel.policiesL3[ LocalData.internalRedirectPolicyName
                        ] = internalRedirectPolicy

   return model

class ServiceMssPolicyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service mss policy [ FILTER ] [ detail [ full ] ]'
   data = {
      'service': serviceAfterShowKwMatcher,
      'mss': nodeMss,
      'policy': matcherPolicy,
      'FILTER' : filterExpr,
      'detail': matcherDetail,
      'full' : matcherFull,
   }
   handler = showPolicy
   cliModel = MssPolicyRootModel

BasicCli.addShowCommandClass( ServiceMssPolicyCmd )

#------------------------------------------------------------------------------------
# show service mss vrf [source <source-name>]
#                      [device <device-name> [virtual instance <name> [name <name>]]]
#------------------------------------------------------------------------------------
def showMssVrf( mode, source=None, device=None, vinst=None, vrfName=None ):
   model = MssVrfModel()
   if config.policyEnforcementConsistency == PolicyEnforcementConsistency.bestEffort:
      mode.addWarning( "This command is only available in 'strict' policy "
                       "enforcement mode" )
      return model

   for sdConfig in serviceDeviceL3V2Dir.values():
      sourceModel = MssVrfSourceModel()
      for deviceKey, serviceDevice in sdConfig.serviceDevice.items():
         if ( ( device and deviceKey.getPhyInstanceName() != device ) or
              ( vinst and deviceKey.getVirtInstanceName() != vinst ) or
              ( source and deviceKey.getFwType() != source ) ):
            continue

         if vrfName and deviceKey.getFwType() == mssL3PolicyCliSource:
            # Static CLI doesn't use netVrf mapping
            vrfConfig = serviceDevice.netVrf.get( vrfName )
            netVrfs = { vrfName : vrfConfig } or {}
         elif vrfName:
            netVrfs = { k: v for k, v in serviceDevice.netVrf.items()
                        if vrfName == v.vrfName }
         else:
            netVrfs = serviceDevice.netVrf

         vinstModel = MssVrfVInstModel()
         for netVrfName, netVrfConfig in netVrfs.items():
            fwVrfName = ( netVrfName
                          if deviceKey.getFwType() == mssL3PolicyCliSource
                          else netVrfConfig.vrfName )
            fwVrfModel = vinstModel.firewallVRFs.setdefault( fwVrfName,
                                                             MssVrfFwVRFModel() )
            netVrfModel = MssVrfNetworkVRFModel()

            networkVrfStatus = mssL3Status.networkVrf.get( netVrfName )
            if networkVrfStatus:
               netVrfModel.vnis = list( networkVrfStatus.vni )
               for switchId in networkVrfStatus.switchVrf:
                  switchModel = SwitchIdModel()
                  switchModel.switchId = switchId.macAddr
                  switchModel.hostname = getHostname( topologyStatus,
                                                      switchId.macAddr )
                  netVrfModel.switches.append( switchModel )

            # Interface subnet
            for ip, l3Intf in netVrfConfig.l3Intf.items():
               subnetModel = MssVrfSubnetModel()
               for subnet in sorted( l3Intf.reachableSubnet ):
                  subnetModel.subnets.append( subnet )
               netVrfModel.nexthops[ ip.stringValue ] = subnetModel

            fwVrfModel.networkVRFs[ netVrfName ] = netVrfModel

         if vinstModel.firewallVRFs:
            sourceModel = model.sources.setdefault(
                           deviceKey.getFwType(), MssVrfSourceModel() )
            devModel = sourceModel.devices.setdefault(
                           deviceKey.getPhyInstanceName(), MssVrfDeviceModel() )
            devModel.virtualInstances[ deviceKey.getVirtInstanceName() ] = vinstModel

   return model

#---------------------------------------------------------------------------------
# show service mss vrf [ device <device-name>
#                          [ virtual instance <vinst-name> [ name <vrf-name> ] ] ]
#---------------------------------------------------------------------------------
class DeviceFilterVrfExprV2( CliCommand.CliExpression ):
   expression = '( device DEVICE [ virtual instance VINST [ name VRF ] ] )'
   data = {
      'device': CliCommand.singleNode( matcherDevice ),
      'DEVICE': CliMatcher.PatternMatcher( pattern='.+',
                                           helpdesc='Device name',
                                           helpname='WORD' ),
      'virtual': 'Virtual Instance Name',
      'instance': 'Virtual Instance Name',
      'VINST': CliMatcher.PatternMatcher( pattern='.+',
                                          helpdesc='Virtual Instance Name',
                                          helpname='WORD' ),
      'name': 'VRF Name',
      'VRF': CliMatcher.PatternMatcher( pattern='.+',
                                        helpdesc='VRF Name',
                                        helpname='WORD' )
   }

   @staticmethod
   def adapter( mode, args, argList ):
      val = args.pop( 'DEVICE', None )
      if val:
         args[ 'DEVICE' ] = val[ 0 ]
      val = args.pop( 'VINST', None )
      if val:
         args[ 'VINST' ] = val[ 0 ]
      val = args.pop( 'VRF', None )
      if val:
         args[ 'VRF' ] = val[ 0 ]

class ServiceMssVrfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show service mss vrf [ FILTER ]'''
   data = {
      'service' : serviceAfterShowKwMatcher,
      'mss' : nodeMss,
      'vrf' : 'Show Macro-Segmentation routing per VRF',
      'FILTER' : CliCommand.setCliExpression ( {
         'SOURCE' : generateSourceFilterExpr( vrfSourceNameFn, 
                                              helpdesc='Source of a routing table' ),
         'DEVICE' : DeviceFilterVrfExprV2,
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      return showMssVrf( mode, source=args.get( 'SOURCE' ),
                         device=args.get( 'DEVICE' ), vinst=args.get( 'VINST' ),
                         vrfName=args.get( 'VRF' ) )

   cliModel = MssVrfModel

BasicCli.addShowCommandClass( ServiceMssVrfCmd )

#----------------------------------------------------------------------
# register show tech-support callback
#----------------------------------------------------------------------
controllerdbConfig = None
showTechCmds = [
   'show service mss status',
   'show service mss dynamic status',
   'show service mss policy detail',
   'show service mss vrf'
]

TechSupportCli.registerShowTechSupportCmd(
   '2017-01-19 17:17:06',
   cmds=showTechCmds,
   cmdsGuard=lambda: controllerdbConfig and controllerdbConfig.enabled
   and config.enable )

#------------------------------------------------------------------------------
# Plugin Setup
#------------------------------------------------------------------------------
def doControllerMounts( controllerdbEm ):
   global topologyStatus
   if controllerdbEm is None:
      topologyStatus = None
   else:
      topologyStatus = LazyMount.mount(
         controllerdbEm,
         "topology/version3/global/status",
         "NetworkTopologyAggregatorV3::Status", "r" )

def Plugin( em ):
   global config
   global status
   global mssL3Status
   global policyL3ConfigDir
   global policyL3StatusDir
   global policyL3ConfigV2Dir
   global policyL3StatusV2Dir
   global controllerdbConfig
   global servicePolicyStatus
   global serviceDeviceL3V2Dir

   config = LazyMount.mount( em, 'mss/config', 'Mss::Config', 'r' )
   status = LazyMount.mount( em, 'mss/status', 'Mss::Status', 'r' )
   mssL3Status = LazyMount.mount( em, 'mssl3/status', 'MssL3::Status', 'r' )
   controllerdbConfig = LazyMount.mount( em, 'controller/config',
                                         'Controllerdb::Config', 'r' )
   servicePolicyStatus = LazyMount.mount( em, 'mss/servicePolicyStatus',
                                           'Mss::PolicyRunningInfoCol', 'r' )

   # Due to bug172287, we are mounting each entity under a Tac.Dir separately
   policyL3ConfigDir = {}
   policyL3StatusDir = {}
   policyL3ConfigV2Dir = {}
   policyL3StatusV2Dir = {}
   serviceDeviceL3V2Dir = {}
   feeders = [ 'cli', 'spm' ]
   for feeder in feeders:
      serviceDeviceL3V2Dir[ feeder ] = LazyMount.mount(
         em, 'mssl3/serviceDeviceSourceConfigV2/%s' % feeder,
         'MssL3V2::ServiceDeviceSourceConfig', 'r' )
      policyL3ConfigV2Dir[ feeder ] = LazyMount.mount(
         em, 'mssl3/policySourceConfigV2/%s' % feeder,
         'MssL3V2::MssPolicySourceConfig', 'r' )
      policyL3StatusV2Dir[ feeder ] = LazyMount.mount(
         em, 'mssl3/policySourceStatusV2/%s' % feeder,
         'MssL3V2::MssPolicySourceStatus', 'r' )
      policyL3ConfigDir[ feeder ] = LazyMount.mount(
         em, 'mssl3/policySourceConfig/%s' % feeder,
         'MssL3::MssPolicySourceConfig', 'r' )
      policyL3StatusDir[ feeder ] = LazyMount.mount(
         em, 'mssl3/policySourceStatus/%s' % feeder,
         'MssL3::MssPolicySourceStatus', 'r' )

   # Mount the unknown source
   policyL3StatusV2Dir[ LocalData.zombieSourceName ] = LazyMount.mount(
      em, 'mssl3/policySourceStatusV2/%s' % LocalData.zombieSourceName,
      'MssL3V2::MssPolicySourceStatus', 'r' )

   registerNotifiee( doControllerMounts )
