# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import BasicCli
import CliMatcher
import CliCommand
from CliMode.Mss import ( MssStaticDeviceMode, MssStaticVrfMode,
                          MssStaticL3NextHopMode, MssStaticL3PolicyMode,
                          MssStaticRuleMode )
from CliPlugin.IpAddrMatcher import ipAddrMatcher, ipPrefixMatcher
from CliPlugin.MssCli import MssConfigMode, addNoMssCallback
from MssStaticCliLib import MssL3PolicyActionCli
from MssStaticCliLib import defaults, InvalidIpAddress
from MssStaticCliLib import ServiceDeviceDirV1, ServiceDeviceDirV2
import ConfigMount
import LazyMount
from MssCliLib import l3Protocol, l4Protocol
from MssCliLib import TrafficInspectionCmd
from MssCliLib import DirectionCmd
import MssCliLib
import Plugins
import Tac

clusterConstants = Tac.Value( "Controller::Constants" )
l3ProtoHelp = { proto: proto for proto in l3Protocol.values() }

l4ProtoHelp = { proto: proto for proto in l4Protocol.values() }

def isStrictEnforcementMode():
   return mssConfig.policyEnforcementConsistency == "strict"

def getMssObject( objV1, objV2 ):
   return objV2 if isStrictEnforcementMode() else objV1

def getServiceDeviceDir():
   return getMssObject( serviceDeviceDirV1, serviceDeviceDirV2 )

# This CLI plugin defines the following modes and configuration commands:
#     service mss
#        [ no|default ] device device_name
#           [ no|default ] traffic inspection local edge
#           [ no ] vrf <vrf-name>
#              [ no ] next-hop address ipv4 <A.B.C.D>
#                 [ no ] route <A.B.C.D/E>
#                 [ no ] route ...
#
#              traffic-policy
#                 [ no ] direction forward | forward reverse
#                 [ no ] match <rule-name> ipv4 [ <after | before> <rule-name> ]
#                    source address       { <A.B.C.D/E> }
#                    no source address       [ { <A.B.C.D/E> } ]
#                    destination address  { <A.B.C.D/E> }
#                    no destination address  [ { <A.B.C.D/E> } ]
#                    source protocol <protocol> port <port-list>
#                    no source protocol <protocol> [ port <port-list> ]
#                    destination protocol <protocol> port <port-list>
#                    no destination protocol <protocol> [ port <port-list> ]
#                    [ no ] protocol <protocol-list>
#                    action drop | forward | redirect | ip-redirect
#                    [ no ] direction forward | forward reverse
#                    validate
#                    abort
#                 move match <rule-name> <after | before> <rule-name>
#                 validate
#                 abort

#------------------------------------------------------------------------------
# (config-mss)# [ no|default ] device device_name
#------------------------------------------------------------------------------
class MssStaticDeviceConfigMode( MssStaticDeviceMode, BasicCli.ConfigModeBase ):
   # Attributes required of every Mode class.
   name = 'cvx-mss-static-device'

   #----------------------------------------------------------------------------
   # Constructs a new MssStaticDeviceConfigMode instance for a particular Service
   # Device object. This is called every time the user enters
   # "static device device_name" in the MSS config mode.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session, serviceDevice ):
      self.serviceDevice = serviceDevice
      MssStaticDeviceMode.__init__( self, serviceDevice.deviceName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def trafficInspectionIs( self, local, outbound ):
      trafficInspection = Tac.Value( "Mss::TrafficInspection",
                                     local=local, outbound=outbound )
      self.serviceDevice.setTrafficInspection( trafficInspection )

   def noTrafficInspection( self ):
      self.serviceDevice.setTrafficInspection( defaults.trafficInspection )

#--------------------------------------------------------------------------------
# [ no | default ] static device DEVICE_NAME
#--------------------------------------------------------------------------------
class StaticDeviceDevicenameCmd( CliCommand.CliCommandClass ):
   syntax = 'static device DEVICE_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'static' : MssCliLib.matcherMssStaticPolicySource,
      'device' : MssCliLib.matcherMssDevice,
      'DEVICE_NAME' : CliMatcher.DynamicNameMatcher(
         lambda mode: getServiceDeviceDir().serviceDevice(),
         helpdesc='Service device name' )

   }

   @staticmethod
   def handler( mode, args ):
      # We need to create the Service Device before entering the child mode, so that
      # if for some reason the Service Device can't be created, we remain in the
      # parent mode.
      deviceName = args[ 'DEVICE_NAME' ]
      serviceDevice = getServiceDeviceDir().createDevice( deviceName )
      childMode = mode.childMode( MssStaticDeviceConfigMode,
                                  serviceDevice=serviceDevice )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      getServiceDeviceDir().destroyDevice( args[ 'DEVICE_NAME' ] )

MssConfigMode.addCommandClass( StaticDeviceDevicenameCmd )
addNoMssCallback( lambda mode: getServiceDeviceDir().clearAllDevices() )

#--------------------------------------------------------------------------------
# [ no | default ] traffic inspection {local|local outbound}
#--------------------------------------------------------------------------------
MssStaticDeviceConfigMode.addCommandClass( TrafficInspectionCmd )

#----------------------------------------------------------------------
# Static device VRF config commands
#
# static device <device-name>
#    [ no ] vrf <vrf-name>
#       [ no ] next-hop address ipv4 <A.B.C.D>
#          [ no ] route <A.B.C.D/E>
#          [ no ] route ...
#----------------------------------------------------------------------
defaultVrfName = 'default'

class MssStaticVrfConfigMode( MssStaticVrfMode, BasicCli.ConfigModeBase ):
   name = 'cvx-mss-static-device-vrf'

   def __init__( self, parent, session, vrfConfig ):
      self.vrfConfig = vrfConfig
      MssStaticVrfMode.__init__( self, ( parent.deviceName, vrfConfig.vrfName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      self.vrfConfig.commitRoutes()
      BasicCli.ConfigModeBase.onExit( self )

class MssStaticL3NextHopConfigMode( MssStaticL3NextHopMode,
                                    BasicCli.ConfigModeBase ):
   name = 'cvx-mss-static-device-vrf-nexthop'

   def __init__( self, parent, session, l3Intf ):
      self.l3Intf = l3Intf
      MssStaticL3NextHopMode.__init__( self, ( parent.deviceName, parent.vrfName,
                                               l3Intf.ip ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#----------------------------------------------------------------------
# Static device VRF config mode
#
# (config-cvx-mss-sd-<device-name>)#
#    [ no ] vrf <vrf-name>
#----------------------------------------------------------------------
class VrfConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF-NAME'
   noOrDefaultSyntax = syntax
   data = {
      'vrf': 'Static device VRF config',
      'VRF-NAME': CliMatcher.DynamicNameMatcher(
                  lambda mode: mode.serviceDevice.getVrfNames(),
                  helpdesc='VRF name',
                  helpname='WORD' )
   }

   @staticmethod
   def handler( mode, args ) :
      vrfName = args[ 'VRF-NAME' ]
      if vrfName == defaultVrfName or isStrictEnforcementMode():
         vrfConfig = mode.serviceDevice.addVrf( vrfName )
         childMode = mode.childMode( MssStaticVrfConfigMode, vrfConfig=vrfConfig )
         mode.session_.gotoChildMode( childMode )
      else:
         mode.addError( '\'%s\' is the only supported VRF name.' % defaultVrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = args[ 'VRF-NAME' ]
      mode.serviceDevice.removeVrf( vrfName )

MssStaticDeviceConfigMode.addCommandClass( VrfConfigCmd )

#----------------------------------------------------------------------
# Static device VRF layer 3 interface config mode
#
# (config-cvx-mss-sd-<device-name>-<vrf-name>)#
#    [no] next-hop address ipv4 <A.B.C.D>
#----------------------------------------------------------------------
class NextHopConfigCmd ( CliCommand.CliCommandClass ):
   syntax = 'next-hop address ipv4 ADDRESS'
   noOrDefaultSyntax = syntax
   data = {
      'next-hop': 'Next-hop address',
      'address': 'Next-hop address',
      'ipv4': 'IPv4 address',
      'ADDRESS': ipAddrMatcher
   }

   @staticmethod
   def handler( mode, args ):
      ipv4 = Tac.Value( 'Arnet::IpGenAddr', args[ 'ADDRESS' ] )
      l3Intf = mode.vrfConfig.getOrCreateL3Intf( ipv4 )

      childMode = mode.childMode( MssStaticL3NextHopConfigMode,
                                  l3Intf=l3Intf )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ipv4 = Tac.Value( 'Arnet::IpGenAddr', args[ 'ADDRESS' ] )
      mode.vrfConfig.removeL3Intf( ipv4 )

MssStaticVrfConfigMode.addCommandClass( NextHopConfigCmd )

#----------------------------------------------------------------------
# Static Device VRF Route config
#
# (config-cvx-mss-sd-<device_name>-<vrf-name>-<A.B.C.D/E>)#
#    [no] route <A.B.C.D/E>>
#----------------------------------------------------------------------
class RouteConfigCmd ( CliCommand.CliCommandClass ):
   syntax = 'route ADDRESS'
   noOrDefaultSyntax = syntax
   data = {
      'route': 'Address reachable by next hop',
      'ADDRESS': ipPrefixMatcher
   }

   @staticmethod
   def handler( mode, args ) :
      # Catch invalid IP prefix that can not be filtered out by ipPrefixMatcher,
      # e.g. 10.10.10.0/16
      try:
         ipPrefix = Tac.Value( 'Arnet::IpGenPrefix', args[ 'ADDRESS' ] )
         mode.l3Intf.addReachableSubnet( ipPrefix )
      except IndexError:
         mode.addError( "Invalid IP prefix: " + str( args[ 'ADDRESS' ] ) )


   @staticmethod
   def noOrDefaultHandler( mode, args ):
      try:
         ipPrefix = Tac.Value( 'Arnet::IpGenPrefix', args[ 'ADDRESS' ] )
         mode.l3Intf.removeReachableSubnet( ipPrefix )
      except IndexError:
         mode.addError( "Invalid IP prefix: " + str( args[ 'ADDRESS' ] ) )

MssStaticL3NextHopConfigMode.addCommandClass( RouteConfigCmd )

#----------------------------------------------------------------------
# L3 policy config commands
#
# static device <device-name>
#    vrf <vrf-name>
#       traffic-policy
#          [ no ]  direction forward | forward reverse
#          [ no ] match <rule-name> ipv4 [ <after | before> <rule-name> ]
#             source address       { <A.B.C.D/E> }
#             no source address       [ { <A.B.C.D/E> } ]
#             destination address  { <A.B.C.D/E> }
#             no destination address  [ { <A.B.C.D/E> } ]
#             source protocol <protocol> port <port-list>
#             no source protocol <protocol> [ port <port-list> ]
#             destination protocol <protocol> port <port-list>
#             no destination protocol <protocol> [ port <port-list> ]
#             [ no ] protocol <protocol-list>
#             action drop | forward | redirect | ip-redirect
#             [ no ] direction forward | forward reverse
#             validate
#             abort
#          move match <rule-name> <after | before> <rule-name>
#          validate
#          abort
#----------------------------------------------------------------------
class MssStaticL3PolicyConfigMode( MssStaticL3PolicyMode, BasicCli.ConfigModeBase ):
   name = 'cvx-mss-static-device-vrf-policy'

   def __init__( self, parent, session, vrfConfig ):
      self.vrfConfig = vrfConfig
      self.vrfConfig.backup()
      self.isAbort = False

      MssStaticL3PolicyMode.__init__( self, ( parent.deviceName,
                                              vrfConfig.vrfName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def validate( self ):
      return self.vrfConfig.validate()

   def onExit( self ):
      if not self.isAbort:
         errorMsg = self.validate()
         if errorMsg:
            self.vrfConfig.restore()
            self.addError( errorMsg )
         else:
            self.vrfConfig.commitRules()
      else:
         self.vrfConfig.restore()

      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.isAbort = True
      self.vrfConfig.restore()
      self.session_.gotoParentMode()

   def directionIs( self, forward, reverse ):
      self.vrfConfig.setDirection( forward=forward, reverse=reverse )

   def noDirection( self ):
      self.vrfConfig.setDirection()

class MssStaticRuleConfigMode( MssStaticRuleMode, BasicCli.ConfigModeBase ):
   name = 'cvx-mss-static-device-vrf-policy-rule'

   def __init__( self, parent, session, candidateRule ):
      self.candidateRule = candidateRule
      self.candidateRule.backup()
      self.isAbort = False

      MssStaticRuleMode.__init__( self, ( parent.deviceName, parent.vrfName,
                                          candidateRule.ruleName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def validate( self ):
      return self.candidateRule.validate()

   def onExit( self ):
      if not self.isAbort:
         errorMsg = self.validate()
         if errorMsg:
            self.candidateRule.restore()
            self.addError( errorMsg )

      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.candidateRule.restore()
      self.isAbort = True
      self.session_.gotoParentMode()

   def directionIs( self, forward, reverse ):
      self.candidateRule.setDirection( forward, reverse )

   def noDirection( self ):
      self.candidateRule.noDirection()

#----------------------------------------------------------------------
# L3 policy config mode
#
# (config-cvx-mss-sd-<device_name>-<vrf_name>)#
#    [no] traffic-policy
#----------------------------------------------------------------------
class PolicyConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-policy'
   noOrDefaultSyntax = syntax
   data = {
      'traffic-policy': 'L3 policy config'
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( MssStaticL3PolicyConfigMode,
                                  vrfConfig=mode.vrfConfig )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.vrfConfig.clearAllRules()

MssStaticVrfConfigMode.addCommandClass( PolicyConfigCmd )

#----------------------------------------------------------------------
# L3 policy rule config mode
#
# (config-cvx-mss-sd-<device_name>-<vrf_name>-policy)#
#    [no] match <rule-name> ipv4 [ <after | before> <rule-name> ]
#----------------------------------------------------------------------
class RuleConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'match rule-name ipv4 [ ( after | before ) ref-rule-name ]'
   noOrDefaultSyntax = 'match rule-name'
   data = {
      'match': 'Match for a rule',
      'rule-name': CliMatcher.DynamicNameMatcher( lambda mode:
                                             mode.vrfConfig.ruleNameList,
                                             helpdesc='Rule name',
                                             helpname='WORD' ),
      'ipv4': 'Match for an IPv4 rule',
      'after': 'Lower priority',
      'before': 'Higher priority',
      'ref-rule-name': CliMatcher.DynamicNameMatcher( lambda mode:
                                             mode.vrfConfig.ruleNameList,
                                             helpdesc='Rule name',
                                             helpname='WORD' )
   }

   @staticmethod
   def handler( mode, args ) :
      ruleName = args[ 'rule-name' ]

      order = None
      if 'after' in args:
         order = 'after'
      elif 'before' in args:
         order = 'before'

      refRuleName = None
      if 'ref-rule-name' in args:
         if ( mode.vrfConfig.hasRule( args[ 'ref-rule-name' ] ) and
              ruleName != refRuleName ):
            refRuleName = args[ 'ref-rule-name' ]

      candidateRule = mode.vrfConfig.addRule( ruleName, order, refRuleName )
      childMode = mode.childMode( MssStaticRuleConfigMode,
                                  candidateRule=candidateRule )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ruleName = args[ 'rule-name' ]
      mode.vrfConfig.removeRule( ruleName )

MssStaticL3PolicyConfigMode.addCommandClass( RuleConfigCmd )

#------------------------------------------------------------------------------------
# Direction command that overrides the default policy enforcement direction for a
# device VRF.
# (config-cvx-mss-sd-<device_name>-<vrf_name>-policy)# direction
#                                                           forward | forward reverse
#------------------------------------------------------------------------------------
MssStaticL3PolicyConfigMode.addCommandClass( DirectionCmd )

#------------------------------------------------------------------------------------
# Direction command that overrides the enforcement direction per rule. This
# configuration is not supported for ip-redirect rule.
# (config-cvx-mss-sd-<device_name>-<vrf_name>-<rule-name>)# direction
#                                                           forward | forward reverse
#------------------------------------------------------------------------------------
MssStaticRuleConfigMode.addCommandClass( DirectionCmd )

#----------------------------------------------------------------------
# L3 policy rule source address config
#
# (config-cvx-mss-sd-<device_name>-<vrf_name>-<rule-name>)#
#    source address { <A.B.C.D/E> }
#    no source address [ { <A.B.C.D/E> } ]
#----------------------------------------------------------------------
class RuleSrcAddressConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'source address { ADDRESSES }'
   noOrDefaultSyntax = 'source address [ { ADDRESSES } ]'
   data = {
      'source': 'Source',
      'address': 'Address',
      'ADDRESSES': ipPrefixMatcher
   }

   @staticmethod
   def handler( mode, args ) :
      try:
         mode.candidateRule.currentRuleCfg.addSrcAddress( args.get( 'ADDRESSES' ) )
      except InvalidIpAddress as err:
         mode.addError( "Invalid IP prefix: " + str( err.ipAddr ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      addrs = args.get( 'ADDRESSES' )
      if addrs:
         try:
            mode.candidateRule.currentRuleCfg.removeSrcAddress( addrs )
         except InvalidIpAddress as err:
            mode.addError( "Invalid IP prefix: " + str( err.ipAddr ) )
      else:
         mode.candidateRule.currentRuleCfg.clearSrcAddress()

MssStaticRuleConfigMode.addCommandClass( RuleSrcAddressConfigCmd )

#----------------------------------------------------------------------
# L3 policy rule destination address config
#
# (config-cvx-mss-sd-<device_name>-<vrf_name>-<rule-name>)#
#    destination address { <A.B.C.D/E> }
#    no destination address [ { <A.B.C.D/E> } ]
#----------------------------------------------------------------------
class RuleDstAddressConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'destination address { ADDRESSES }'
   noOrDefaultSyntax = 'destination address [ { ADDRESSES } ]'
   data = {
      'destination': 'Destination',
      'address': 'Address',
      'ADDRESSES': ipPrefixMatcher
   }

   @staticmethod
   def handler( mode, args ) :
      try:
         mode.candidateRule.currentRuleCfg.addDstAddress( args.get( 'ADDRESSES' ) )
      except InvalidIpAddress as err:
         mode.addError( "Invalid IP prefix: " + str( err.ipAddr ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      addrs = args.get( 'ADDRESSES' )
      if addrs:
         try:
            mode.candidateRule.currentRuleCfg.removeDstAddress( addrs )
         except InvalidIpAddress as err:
            mode.addError( "Invalid IP prefix: " + str( err.ipAddr ) )
      else:
         mode.candidateRule.currentRuleCfg.clearDstAddress()

MssStaticRuleConfigMode.addCommandClass( RuleDstAddressConfigCmd )

#----------------------------------------------------------------------
# L3 policy rule source port config
#
# (config-cvx-mss-sd-<device_name>-<vrf_name>-<rule-name>)#
#    source protocol <protocol> port {<port>}
#    no source protocol <protocol> [ port {<port>} ]
#----------------------------------------------------------------------
class RuleSrcPortConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'source protocol PROTOCOL port { PORTS }'
   noOrDefaultSyntax = 'source protocol PROTOCOL [ port { PORTS } ]'
   data = {
      'source': 'Source',
      'protocol': 'L4 protocol',
      'PROTOCOL': CliMatcher.DynamicKeywordMatcher( lambda mode: l4ProtoHelp ),
      'port': 'L4 Port',
      'PORTS': CliMatcher.IntegerMatcher( 0, 65535,
                                          helpdesc='Port list' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.candidateRule.currentRuleCfg.addSrcL4App(
                        args.get( 'PROTOCOL' ), args.get( 'PORTS' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.candidateRule.currentRuleCfg.removeSrcL4App(
                        args.get( 'PROTOCOL' ), args.get( 'PORTS' ) )

MssStaticRuleConfigMode.addCommandClass( RuleSrcPortConfigCmd )

#----------------------------------------------------------------------
# L3 policy rule destination port config
#
# (config-cvx-mss-sd-<device_name>-<vrf_name>-<rule-name>)#
#    destination protocol <protocol> port {<port>}
#    [no] destination protocol <protocol> [ port {<port>} ]
#----------------------------------------------------------------------
class RuleDstPortConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'destination protocol PROTOCOL port { PORTS }'
   noOrDefaultSyntax = 'destination protocol PROTOCOL [ port { PORTS } ]'
   data = {
      'destination': 'Destination',
      'protocol': 'L4 protocol',
      'PROTOCOL': CliMatcher.DynamicKeywordMatcher( lambda mode: l4ProtoHelp ),
      'port': 'L4 Port',
      'PORTS': CliMatcher.IntegerMatcher( 0, 65535,
                                          helpdesc='Port list' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.candidateRule.currentRuleCfg.addDstL4App(
                        args.get( 'PROTOCOL' ), args.get( 'PORTS' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.candidateRule.currentRuleCfg.removeDstL4App(
                        args.get( 'PROTOCOL' ), args.get( 'PORTS' ) )

MssStaticRuleConfigMode.addCommandClass( RuleDstPortConfigCmd )

#----------------------------------------------------------------------
# L3 policy rule protocol config
#
# (config-cvx-mss-sd-<device_name>-<vrf_name>-<rule-name>)#
#    [no] protocol <protocol-list>
#----------------------------------------------------------------------
class RuleProtocolConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'protocol { PROTOCOLS }'
   noOrDefaultSyntax = syntax
   data = {
      'protocol': 'IP protocol config',
      'PROTOCOLS': CliMatcher.DynamicKeywordMatcher( lambda mode: l3ProtoHelp ),
   }

   @staticmethod
   def handler( mode, args ):
      for proto in args.get( 'PROTOCOLS' ):
         mode.candidateRule.currentRuleCfg.addProtocol( proto )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for proto in args.get( 'PROTOCOLS' ):
         mode.candidateRule.currentRuleCfg.removeProtocol( proto )

MssStaticRuleConfigMode.addCommandClass( RuleProtocolConfigCmd )

#----------------------------------------------------------------------
# L3 policy rule action config
#
# (config-cvx-mss-sd-<device_name>-<vrf_name>-<rule-name>)#
#    action <drop | forward | redirect | ip-redirect>
#----------------------------------------------------------------------
class RuleActionConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'action ( drop | forward | redirect | ip-redirect)'
   noOrDefaultSyntax = 'action'
   data = {
      'action': 'Action config',
      'drop': 'Drop matching flow (default)',
      'forward': 'Bypass any redirect and forward the matching flow normally',
      'redirect': 'Redirect the matching flow to service device',
      'ip-redirect': 'Redirect the IP address packet to service device'
   }

   @staticmethod
   def handler( mode, args ):
      rule = mode.candidateRule
      if 'ip-redirect' in args:
         rule.currentRuleCfg.action = MssL3PolicyActionCli.ipRedirect
      elif 'drop' in args:
         rule.currentRuleCfg.action = MssL3PolicyActionCli.drop
      elif 'forward' in args:
         rule.currentRuleCfg.action = MssL3PolicyActionCli.forward
      elif 'redirect' in args:
         rule.currentRuleCfg.action = MssL3PolicyActionCli.redirect

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      rule = mode.candidateRule
      rule.currentRuleCfg.action = MssL3PolicyActionCli.drop

MssStaticRuleConfigMode.addCommandClass( RuleActionConfigCmd )

#----------------------------------------------------------------------
# L3 policy rule config validate
#
# (config-cvx-mss-sd-<device_name>-<vrf_name>-<rule-name>)#
#    validate
#----------------------------------------------------------------------
class RuleValidateCmd( CliCommand.CliCommandClass ):
   syntax = 'validate'
   data = {
      'validate': 'Validate the rule configuration',
   }

   @staticmethod
   def handler( mode, args ):
      errorMsg = mode.validate()
      if errorMsg:
         mode.addError( errorMsg )
      else:
         mode.addMessage( 'All configurations are valid.' )

MssStaticRuleConfigMode.addCommandClass( RuleValidateCmd )

#----------------------------------------------------------------------
# L3 rule abort all changes
#
# (config-cvx-mss-sd-<device_name>-<vrf_name>-<rule-name>)#
#    abort
#----------------------------------------------------------------------
class RuleAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': 'Discard all changes',
   }

   @staticmethod
   def handler( mode, args ) :
      mode.abort()

MssStaticRuleConfigMode.addCommandClass( RuleAbortCmd )

#----------------------------------------------------------------------
# L3 policy config validate
#
# (config-cvx-mss-sd-<device_name>-<vrf_name>-policy)#
#    validate
#----------------------------------------------------------------------
class PolicyValidateCmd( CliCommand.CliCommandClass ):
   syntax = 'validate'
   data = {
      'validate': 'Validate the policy configuration',
   }

   @staticmethod
   def handler( mode, args ):
      errorMsg = mode.validate()
      if errorMsg:
         mode.addError( errorMsg )
      else:
         mode.addMessage( 'All configurations are valid.' )

MssStaticL3PolicyConfigMode.addCommandClass( PolicyValidateCmd )

#----------------------------------------------------------------------
# L3 policy abort all changes
#
# (config-cvx-mss-sd-<device_name>-<vrf_name>-policy)#
#    abort
#----------------------------------------------------------------------
class PolicyAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': 'Discard all changes',
   }

   @staticmethod
   def handler( mode, args ) :
      mode.abort()

MssStaticL3PolicyConfigMode.addCommandClass( PolicyAbortCmd )

#----------------------------------------------------------------------
# L3 policy change rule priority
#
# (config-cvx-mss-sd-<device_name>-<vrf_name>-policy)#
#    move match <rule-name> <after | before> <rule-name>
#----------------------------------------------------------------------
class RulePriorityConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'move match rule-name ( after | before ) ref-rule-name'
   data = {
      'move': 'Change rule priority',
      'match': 'change rule priority',
      'rule-name': CliMatcher.DynamicNameMatcher( lambda mode:
                                             mode.candidateL3Policy.ruleNameList,
                                             helpdesc='Rule name',
                                             helpname='RuleName' ),
      'after': 'Lower priority',
      'before': 'Higher priority',
      'ref-rule-name': CliMatcher.DynamicNameMatcher( lambda mode:
                                             mode.candidateL3Policy.ruleNameList,
                                             helpdesc='Rule name',
                                             helpname='RuleName' )
   }

   @staticmethod
   def handler( mode, args ):
      ruleName = args[ 'rule-name' ]
      refRuleName = args[ 'ref-rule-name' ]
      if 'after' in args:
         order = 'after'
      elif 'before' in args:
         order = 'before'

      if ( mode.vrfConfig.hasRule( ruleName ) and
           mode.vrfConfig.hasRule( refRuleName ) and
           ruleName != refRuleName ):
         mode.vrfConfig.setRulePriority( ruleName, order, refRuleName )

MssStaticL3PolicyConfigMode.addCommandClass( RulePriorityConfigCmd )

#------------------------------------------------------------------------------
# Plugin Setup
#------------------------------------------------------------------------------
serviceDeviceDirV1 = None
serviceDeviceDirV2 = None
mssConfig = None

@Plugins.plugin( requires=( "ControllerdbMgr", ) )
def Plugin( entityManager ):
   global serviceDeviceDirV1
   global serviceDeviceDirV2
   global mssConfig

   # V1 inputs
   serviceDeviceDirV1 = ServiceDeviceDirV1(
            ConfigMount.mount( entityManager,
                               "mssl3/serviceDeviceSourceConfig/cli",
                               "MssL3::ServiceDeviceSourceConfig", "w" ),
            ConfigMount.mount( entityManager,
                               "mssl3/policySourceConfig/cli",
                               "MssL3::MssPolicySourceConfig", "w" ) )

   # V2 inputs
   serviceDeviceDirV2 = ServiceDeviceDirV2(
            ConfigMount.mount( entityManager,
                               "mssl3/serviceDeviceSourceConfigV2/cli",
                               "MssL3V2::ServiceDeviceSourceConfig", "w" ),
            ConfigMount.mount( entityManager,
                               "mssl3/policySourceConfigV2/cli",
                               "MssL3V2::MssPolicySourceConfig", "w" ) )

   # mss config
   mssConfig = LazyMount.mount( entityManager, "mss/config", "Mss::Config", "r" )
