# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BgpLib
import Cell
import CliCommand
import CliToken.RoutingBgp as bgpTokens
import LazyMount
from CliMode.Mvpn import (
    MvpnIpv4VrfMode,
    MvpnIpv4VrfSpmsiMode,
    MvpnIpv4VrfIntraAsIPmsiMode
)
from CliMatcher import KeywordMatcher, EnumMatcher, DynamicNameMatcher
from CliPlugin.RoutingBgpCli import (
   afModeExtensionHook,
   configForVrf,
   RouterBgpBaseMode,
   RoutingBgpBaseAfMode,
   RouterBgpVrfMode,
   V4OrPeerGroupCliExpression,
   RouterBgpAfUnsupportedPrefixListSharedModelet,
)
from CliPlugin import IntfCli
from IpLibConsts import DEFAULT_VRF
import SmashLazyMount
import Tac
from Toggles import (
   MvpnLibToggleLib,
   TunnelToggleLib
)
from TypeFuture import TacLazyType

RsvpSessionType = TacLazyType( 'Rsvp::RsvpSessionType' )
p2pLspTunnel = RsvpSessionType.p2pLspTunnel
p2mpLspTunnel = RsvpSessionType.p2mpLspTunnel

mvpnIpv4Kw = KeywordMatcher( "mvpn-ipv4",
                             helpdesc="Multicast VPN IPv4 address-family" )

selectivePmsiKw = KeywordMatcher( "selective-pmsi",
                             helpdesc="Multicast VPN Selective PMSI A-D Route" )

intraAsIPmsiKw = KeywordMatcher( "intra-as-ipmsi",
                             helpdesc="Multicast VPN Intra AS PMSI A-D Route" )

rsvpLerClient = None
vrfIdStatus = None
vrfNameStatus = None
mvpnRsvpLeafStatus = None
mvpnIntfStatus = None
mldpTunnelDecapStatus = None
rsvpTunnelDecapStatus = None
mvpnTunnelEncapStatus = None
AddressFamily = TacLazyType( 'Arnet::AddressFamily' )
MvpnRsvpLeafStatus = TacLazyType( 'Routing::Multicast::MvpnRsvpLeafStatus' )
MvpnTunnelEncapStatus = TacLazyType( 'Routing::Multicast::MvpnTunnelEncapStatus' )
MvpnClient = TacLazyType( "Routing::Multicast::MvpnClient" )

class RouterBgpAfMvpnV4Mode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family MVPN V4 configuration'

   def __init__( self, parent, session, addrFamily='mvpn-ipv4' ):
      # This address-family mode is only available in default vrf
      self.vrfName = DEFAULT_VRF
      self.addrFamily = addrFamily
      RoutingBgpBaseAfMode.__init__( self, addrFamily )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

RouterBgpAfMvpnV4Mode.addModelet( RouterBgpAfUnsupportedPrefixListSharedModelet )

#------------------------------------------------------------------------------------
# '[ no | default ] address-family mvpn-ipv4' config mode
#------------------------------------------------------------------------------------
class MvpnIpv4ModeCmd( CliCommand.CliCommandClass ):
   syntax = 'address-family mvpn-ipv4'
   noOrDefaultSyntax = syntax
   data = {
         'address-family': bgpTokens.addrFamily,
         'mvpn-ipv4': mvpnIpv4Kw,
   }
   handler = "MvpnCliHandler.MvpnIpv4ModeCmd_handler"
   noOrDefaultHandler = "MvpnCliHandler.MvpnIpv4ModeCmd_noOrDefaultHandler"

# -----------------------------------------------------------------------------------
# '[ no | default ] neighbor PEER ( activate | deactivate )' for
# mvpn-ipv4 address-family
# -----------------------------------------------------------------------------------
# NOTE: Unlike 'neighbor ... activate' commands in other address families (ipv4, ipv6
# etc.), the cmd in mvpn-ipv4 address-family mode ALWAYS honors no-equals-default
# semantics (irrespective of "service routing configuration bgp no-equals-default"
# config) thereby ALWAYS provides the "deactivate" token.
class NeighborActivateCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor PEER ( activate | deactivate )'
   noOrDefaultSyntax = syntax

   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': V4OrPeerGroupCliExpression,  # IPv6 transport peer is not supported
      'activate': bgpTokens.activate,
      'deactivate': bgpTokens.deactivate,
   }
   handler = "MvpnCliHandler.NeighborActivateCmd_handler"
   noOrDefaultHandler = "MvpnCliHandler.NeighborActivateCmd_noOrDefaultHandler"

BgpLib.updateConfigAttrsAfMap( { 'mvpn-ipv4': [ 'MvpnV4' ] } )
# NOTE: Defining this directly in the RouterBgpBaseMode, so this command is not
# available in non default VRF submode.
RouterBgpBaseMode.addCommandClass( MvpnIpv4ModeCmd )
RouterBgpAfMvpnV4Mode.addCommandClass( NeighborActivateCmd )

# -------------------------------------------------------------------------------
# '[ no | default ] mvpn-ipv4'
# for the vrf sub-mode under 'router bgp'
# -------------------------------------------------------------------------------
class MvpnIpv4VrfCmd( CliCommand.CliCommandClass ):
   syntax = 'mvpn-ipv4'
   noOrDefaultSyntax = syntax
   data = {
      'mvpn-ipv4': mvpnIpv4Kw
   }
   handler = "MvpnCliHandler.MvpnIpv4VrfCmd_handler"
   noOrDefaultHandler = "MvpnCliHandler.MvpnIpv4VrfCmd_noOrDefaultHandler"

class MvpnIpv4VrfConfigMode( MvpnIpv4VrfMode, BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, vrfName ):
      self.session_ = session
      config = configForVrf( vrfName )
      config.afMvpnV4Vrf = True
      MvpnIpv4VrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# ----------------------------------------------------------------------------------
# '[ no | default ] rsvp p2mp tunnel profile PROFILE' for mvpn-ipv4 address-family
# ----------------------------------------------------------------------------------
def getRsvpTunnelProfileNames( mode ):
   return [ p.profileName for p in rsvpLerClient.tunnelProfile.values()
            if p.sessionType == p2mpLspTunnel ]

class MvpnIpv4RsvpP2mpTunnelProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'rsvp p2mp tunnel profile PROFILE_NAME'
   noOrDefaultSyntax = 'rsvp p2mp tunnel profile [ PROFILE_NAME ]'
   data = {
      'rsvp': 'RSVP-TE settings',
      'p2mp': 'point-to-multipoint',
      'tunnel': 'P-tunnel',
      'profile': 'Associate a tunnel profile with the RSVP-TE P2MP P-tunnel',
      'PROFILE_NAME': DynamicNameMatcher( getRsvpTunnelProfileNames,
                                          'Tunnel profile name', helpname='WORD' )
   }
   handler = 'MvpnCliHandler.handlerMvpnIpv4RsvpP2mpTunnelProfileCmd'
   noOrDefaultHandler = \
         'MvpnCliHandler.noOrDefaultHandlerIpv4RsvpP2mpTunnelProfileCmd'

class NeighborSourceInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor default encapsulation mpls next-hop-self source-interface INTF'
   noOrDefaultSyntax = 'neighbor default encapsulation mpls next-hop-self \
         source-interface ...'
   data = {
      'neighbor': bgpTokens.neighbor,
      'default': bgpTokens.default,
      'encapsulation': bgpTokens.encap,
      'mpls': bgpTokens.mpls,
      'next-hop-self': bgpTokens.nextHopSelf,
      'source-interface': ( "Source interface to update BGP next hop address and "
                            "Originator address for type1,type3 paths" ),
      'INTF': IntfCli.Intf.matcherWithIpSupport,
   }
   handler = "MvpnCliHandler.NeighborSourceInterfaceCmd_handler"
   noOrDefaultHandler = \
      "MvpnCliHandler.NeighborSourceInterfaceCmd_noOrDefaultHandler"

#------------------------------------------------------------------------------------
# '[ no | default ] neighbor default encapsulation mpls protocol mldp p2mp' for
# mvpn-ipv4 address-family
#------------------------------------------------------------------------------------
protocolValues = {
   'mldp': 'Multipoint LDP',
}

if TunnelToggleLib.toggleMvpnRsvpP2mpEnabled():
   RouterBgpAfMvpnV4Mode.addCommandClass( MvpnIpv4RsvpP2mpTunnelProfileCmd )
   MvpnIpv4VrfConfigMode.addCommandClass( MvpnIpv4RsvpP2mpTunnelProfileCmd )
   RouterBgpAfMvpnV4Mode.addCommandClass( NeighborSourceInterfaceCmd )
   protocolValues[ 'rsvp' ] = ( 'Resource Reservation Protocol - Traffic Engineering'
                                ' (RSVP-TE)' )
class NeighborDefaultEncapProtocolCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor default encapsulation mpls protocol PROTOCOL p2mp'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor': bgpTokens.neighbor,
      'default': bgpTokens.default,
      'encapsulation': bgpTokens.encap,
      'mpls': bgpTokens.mpls,
      'protocol': 'PMSI tunnel protocol',
      'PROTOCOL': CliCommand.Node( matcher=EnumMatcher( protocolValues ),
                                   maxMatches=1 ),
      'p2mp': 'point-to-multipoint'
   }
   handler = "MvpnCliHandler.NeighborDefaultEncapProtocolCmd_handler"
   noOrDefaultHandler = \
      "MvpnCliHandler.NeighborDefaultEncapProtocolCmd_noOrDefaultHandler"

RouterBgpAfMvpnV4Mode.addCommandClass( NeighborDefaultEncapProtocolCmd )

# --------------------------------------------------------------------------------
# [ no | default ] mvpn-ipv4
# mode command for the vrf sub-mode under 'router bgp'
# --------------------------------------------------------------------------------
class MvpnIpv4VrfModeCmd( CliCommand.CliCommandClass ):
   syntax = 'mvpn-ipv4'
   noOrDefaultSyntax = syntax
   data = {
      'mvpn-ipv4': mvpnIpv4Kw
   }
   handler = "MvpnCliHandler.handlerMvpnIpv4VrfModeCmd"
   noOrDefaultHandler = "MvpnCliHandler.noOrDefaultHandlerMvpnIpv4VrfModeCmd"

if MvpnLibToggleLib.toggleMvpnSelectiveTunnelsEnabled():
   RouterBgpVrfMode.addCommandClass( MvpnIpv4VrfModeCmd )
else:
   RouterBgpVrfMode.addCommandClass( MvpnIpv4VrfCmd )


class MvpnIpv4VrfSpmsiConfigMode( MvpnIpv4VrfSpmsiMode, BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, vrfName ):
      self.session_ = session
      self.vrfName_ = vrfName
      MvpnIpv4VrfSpmsiMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# --------------------------------------------------------------------------------
# [ no | default ] selective-pmsi
# for the mvpn-ipv4 submode under router bgp vrf
# --------------------------------------------------------------------------------
class MvpnIpv4VrfSpmsiModeCmd( CliCommand.CliCommandClass ):
   syntax = 'selective-pmsi'
   noOrDefaultSyntax = syntax
   data = {
      'selective-pmsi': selectivePmsiKw
   }
   handler = "MvpnCliHandler.handlerMvpnIpv4VrfSpmsiModeCmd"
   noOrDefaultHandler = "MvpnCliHandler.noOrDefaultHandlerMvpnIpv4VrfSpmsiModeCmd"

class MvpnIpv4VrfIntraAsIPmsiConfigMode( MvpnIpv4VrfIntraAsIPmsiMode,
                                         BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, vrfName ):
      self.session_ = session
      self.vrfName_ = vrfName
      MvpnIpv4VrfIntraAsIPmsiMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# --------------------------------------------------------------------------------
# [ no | default ] intra-as-ipmsi
# for the mvpn-ipv4 submode under router bgp vrf
# --------------------------------------------------------------------------------
class MvpnIpv4VrfIntraAsIPmsiModeCmd( CliCommand.CliCommandClass ):
   syntax = 'intra-as-ipmsi'
   noOrDefaultSyntax = syntax
   data = {
      'intra-as-ipmsi': intraAsIPmsiKw
   }
   handler = "MvpnCliHandler.handlerMvpnIpv4VrfIntraAsIPmsiModeCmd"
   noOrDefaultHandler = \
      "MvpnCliHandler.noOrDefaultHandlerMvpnIpv4VrfIntraAsIPmsiModeCmd"

# --------------------------------------------------------------------------------
# [ no | default ] pmsi-tunnel protocol [mldp|rsvp] p2mp
# for the mvpn-ipv4 submode under router bgp vrf
# --------------------------------------------------------------------------------
protocolKw = KeywordMatcher( "protocol",
                             helpdesc="Set Intra AS I-PMSI and Selective PMSI A-D "
                                      "routes PMSI tunnel attribute value" )

class MvpnIpv4VrfPmsiTunnelProtocolCmd( CliCommand.CliCommandClass ):
   syntax = 'pmsi-tunnel protocol PROTOCOL p2mp'
   noOrDefaultSyntax = syntax
   data = {
      'pmsi-tunnel': "PMSI tunnel settings",
      'protocol': protocolKw,
      'PROTOCOL': CliCommand.Node( matcher=EnumMatcher( protocolValues ),
                                   maxMatches=1 ),
      'p2mp': 'point-to-multipoint'
   }
   handler = "MvpnCliHandler.NeighborDefaultEncapProtocolCmd_handler"
   noOrDefaultHandler = \
      "MvpnCliHandler.NeighborDefaultEncapProtocolCmd_noOrDefaultHandler"

if TunnelToggleLib.toggleMvpnRsvpP2mpEnabled():
   MvpnIpv4VrfConfigMode.addCommandClass( MvpnIpv4VrfPmsiTunnelProtocolCmd )
if MvpnLibToggleLib.toggleMvpnSelectiveTunnelsEnabled():
   MvpnIpv4VrfConfigMode.addCommandClass( MvpnIpv4VrfSpmsiModeCmd )
   MvpnIpv4VrfConfigMode.addCommandClass( MvpnIpv4VrfIntraAsIPmsiModeCmd )

pTunnelKw = KeywordMatcher( "p-tunnel",
               helpdesc="Provider tunnel" )

sourceGroupKw = KeywordMatcher( "source-group",
               helpdesc="A tunnel for each multicast overlay source group" )

starStarKw = KeywordMatcher( "star-star",
                       helpdesc="A tunnel for all overlay multicast traffic" )

# --------------------------------------------------------------------------------
# [ no | default ] source-group
# for the seletive-pmsi submode under router bgp vrf mvpn-ipv4
# --------------------------------------------------------------------------------
class MvpnIpv4VrfSpmsiSourceGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'p-tunnel source-group'
   noOrDefaultSyntax = syntax
   data = {
      'p-tunnel': pTunnelKw,
      'source-group': sourceGroupKw,
   }
   handler = "MvpnCliHandler.handlerMvpnIpv4VrfSpmsiSourceGroupCmd"
   noOrDefaultHandler = \
      "MvpnCliHandler.noOrDefaultHandlerMvpnIpv4VrfSpmsiSourceGroupCmd"

# --------------------------------------------------------------------------------
# [ no | default ] p-tunnel star-star
# for the seletive-pmsi submode under router bgp vrf mvpn-ipv4
# --------------------------------------------------------------------------------
class MvpnIpv4VrfSpmsiStarStarCmd( CliCommand.CliCommandClass ):
   syntax = 'p-tunnel star-star'
   noOrDefaultSyntax = syntax
   data = {
      'p-tunnel': pTunnelKw,
      'star-star': starStarKw,
   }
   handler = "MvpnCliHandler.handlerMvpnIpv4VrfSpmsiStarStarCmd"
   noOrDefaultHandler = \
      "MvpnCliHandler.noOrDefaultHandlerMvpnIpv4VrfSpmsiStarStarCmd"

routeKw = KeywordMatcher( "route",
               helpdesc="Intra AS I-PMSI A-D route" )

attributeKw = KeywordMatcher( "attribute",
               helpdesc="Intra AS I-PMSI A-D route attribute" )

pmsiTunnelKw = KeywordMatcher( "pmsi-tunnel",
               helpdesc="Intra AS I-PMSI A-D route PMSI tunnel attribute" )

noAdvertiseKw = KeywordMatcher( "no-advertise",
         helpdesc="Send Intra AS I-PMSI A-D routes without PMSI tunnel attribute" )

advertiseKw = KeywordMatcher( "advertise",
            helpdesc="Send Intra AS I-PMSI A-D routes with PMSI tunnel attribute" )

# --------------------------------------------------------------------------------
# [ no | default ] route attribute pmsi-tunnel no-advertise
# for the  intra-as-pmsi submode under router bgp vrf mvpn-ipv4
# --------------------------------------------------------------------------------
class MvpnIpv4VrfIPmsiTunnelNoAdvertiseCmd( CliCommand.CliCommandClass ):
   syntax = 'route attribute pmsi-tunnel no-advertise'
   noOrDefaultSyntax = syntax
   data = {
      'route': routeKw,
      'attribute': attributeKw,
      'pmsi-tunnel': pmsiTunnelKw,
      'no-advertise': noAdvertiseKw,
   }
   handler = "MvpnCliHandler.handlerMvpnIpv4VrfIPmsiTunnelNoAdvertiseCmd"
   noOrDefaultHandler = \
      "MvpnCliHandler.noOrDefaultHandlerMvpnIpv4VrfIPmsiTunnelNoAdvertiseCmd"

#--------------------------------------------------------------------------------
# [ no | default ] route attribute pmsi-tunnel advertise
# for the  intra-as-pmsi submode under router bgp vrf mvpn-ipv4
#--------------------------------------------------------------------------------
class MvpnIpv4VrfIPmsiTunnelAdvertiseCmd( CliCommand.CliCommandClass ):
   syntax = 'route attribute pmsi-tunnel advertise'
   noOrDefaultSyntax = syntax
   data = {
      'route': routeKw,
      'attribute': attributeKw,
      'pmsi-tunnel': pmsiTunnelKw,
      'advertise': advertiseKw,
   }
   handler = "MvpnCliHandler.handlerMvpnIpv4VrfIPmsiTunnelAdvertiseCmd"
   noOrDefaultHandler = \
      "MvpnCliHandler.noOrDefaultHandlerMvpnIpv4VrfIPmsiTunnelAdvertiseCmd"

if MvpnLibToggleLib.toggleMvpnSelectiveTunnelsEnabled():
   MvpnIpv4VrfSpmsiConfigMode.addCommandClass( MvpnIpv4VrfSpmsiSourceGroupCmd )
   MvpnIpv4VrfSpmsiConfigMode.addCommandClass( MvpnIpv4VrfSpmsiStarStarCmd )
   MvpnIpv4VrfIntraAsIPmsiConfigMode.addCommandClass(
                                 MvpnIpv4VrfIPmsiTunnelNoAdvertiseCmd )
   MvpnIpv4VrfIntraAsIPmsiConfigMode.addCommandClass(
                                 MvpnIpv4VrfIPmsiTunnelAdvertiseCmd )

def Plugin( entityManager ):
   global rsvpLerClient
   global vrfIdStatus
   global vrfNameStatus
   global mvpnRsvpLeafStatus
   global mvpnIntfStatus
   global mldpTunnelDecapStatus
   global rsvpTunnelDecapStatus
   global mvpnTunnelEncapStatus

   em = entityManager
   afModeExtensionHook.addAfModeExtension( 'mvpn-ipv4', RouterBgpAfMvpnV4Mode )
   rsvpLerClient = LazyMount.mount( entityManager, "mpls/rsvp/lerClient/cli",
                                    "Rsvp::RsvpLerClient", "r" )
   vrfIdStatus = SmashLazyMount.mount( em, 'vrf/vrfIdMapStatus',
                                       'Vrf::VrfIdMap::Status',
                                       SmashLazyMount.mountInfo( 'reader' ),
                                       autoUnmount=True )
   vrfNameStatus = LazyMount.mount( em, Cell.path( 'vrf/vrfNameStatus' ),
                                    'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )
   path = MvpnRsvpLeafStatus.mountPath( AddressFamily.ipv4 )
   leafStatusType = 'Routing::Multicast::MvpnRsvpLeafStatus'
   mvpnRsvpLeafStatus = LazyMount.mount( em, path, leafStatusType, "r" )

   mvpnIntfStatusTypeStr = 'Routing::Multicast::MvpnIntfStatus'
   mvpnIntfStatus = SmashLazyMount.mount(
                       em, Tac.Type( mvpnIntfStatusTypeStr ).mountPath(),
                       mvpnIntfStatusTypeStr, SmashLazyMount.mountInfo( 'reader' ) )

   mvpnTunnelDecapStatusTypeStr = 'Routing::Multicast::MvpnTunnelDecapStatus'
   mvpnTunnelDecapStatusType = Tac.Type( mvpnTunnelDecapStatusTypeStr )
   mldpTunnelDecapStatus = LazyMount.mount( em,
      mvpnTunnelDecapStatusType.mountPath( 'ipv4', "mLdpP2mpLsp" ),
      mvpnTunnelDecapStatusTypeStr, 'r' )
   rsvpTunnelDecapStatus = LazyMount.mount( em,
      mvpnTunnelDecapStatusType.mountPath( 'ipv4', "rsvpTeP2mpLsp" ),
      mvpnTunnelDecapStatusTypeStr, 'r' )

   path = MvpnTunnelEncapStatus.mountPath( AddressFamily.ipv4, MvpnClient.bgp )
   encapStatusType = "Routing::Multicast::MvpnTunnelEncapStatus"
   mvpnTunnelEncapStatus = LazyMount.mount( em, path, encapStatusType, "r" )
