# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from CliCommand import CliExpression, Node
import CliMatcher
from CliPlugin import MvpnLibCliModels
from CliPlugin.BgpVpnCli import RdCliExpression
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.MvpnCli import (
   mvpnIpv4Kw,
   protocolValues
)
from CliPlugin.RoutingBgpShowCli import summaryVrfModel
from CliPlugin.VrfCli import VrfExprFactory
from CliToken.Ipv4 import ipv4MatcherForShow
import CliToken.RoutingBgp as bgpTokens
from CliToken.RoutingBgpShowCliTokens import (
      bgpAfterShow,
      detail as detailKw,
      summary as summaryKw
)
import ShowCommand
import Toggles.MvpnLibToggleLib
from Toggles.TunnelToggleLib import toggleMvpnRsvpP2mpEnabled

#-------------------------------------------------------------------------------
# Helper Methods
#-------------------------------------------------------------------------------
matcherRouteType = CliMatcher.KeywordMatcher( 'route-type',
      helpdesc='Filter by NLRI route type' )
matcherIntraAsIpmsi = CliMatcher.KeywordMatcher( 'intra-as-ipmsi',
      helpdesc='Filter by intra AS ipmsi routes' )
matcherSpmsi = CliMatcher.KeywordMatcher( 'selective-pmsi',
      helpdesc='Filter by selective pmsi routes' )
matcherSourceActive = CliMatcher.KeywordMatcher( 'source-active',
      helpdesc='Filter by source active routes' )
matcherSharedTree = CliMatcher.KeywordMatcher( 'shared-tree',
      helpdesc='Filter by shared tree routes' )
matcherSourceTree = CliMatcher.KeywordMatcher( 'source-tree',
      helpdesc='Filter by source tree routes' )
vrfNameExpr = VrfExprFactory( helpdesc='VRF name' )
matcherOrigIpv4 = IpAddrMatcher( helpdesc='Originating router IPv4 address' )

#-------------------------------------------------------------------------------
# 'show bgp mvpn-ipv4 route-type intra-as-ipmsi
# [ rd <rd-val> ] [ orig-ip ] [ detail ]'
#-------------------------------------------------------------------------------
class ShowBgpMvpnV4Type1Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show bgp mvpn-ipv4 route-type intra-as-ipmsi
               [ RD ] [ ORIG_IP ] [ detail ]'''
   data = {
         'bgp': bgpAfterShow,
         'mvpn-ipv4': mvpnIpv4Kw,
         'route-type': matcherRouteType,
         'intra-as-ipmsi': matcherIntraAsIpmsi,
         'RD': RdCliExpression,
         'ORIG_IP': matcherOrigIpv4,
         'detail': detailKw,
   }
   cliModel = "MvpnCliModels.MvpnRoutesVrfModel"
   handler = "MvpnShowCliHandler.handlerMvpnV4IntraAsIpmsi"

BasicCli.addShowCommandClass( ShowBgpMvpnV4Type1Cmd )

class SrcGrpExpression( CliExpression ):
   expression = 'IP1 [ IP2 ]'
   data = {
      'IP1': IpAddrMatcher(
         helpdesc='Multicast group or source address' ),
      'IP2': IpAddrMatcher(
         helpdesc='Multicast group or source address' )
   }

   @staticmethod
   def adapter( mode, args, argList ):
      # Update args with IP1, IP2 as keys of a nested dict
      firstVal = args.pop( 'IP1', None )
      if firstVal:
         args[ 'SRC_GRP' ] = { 'IP1': firstVal }

      secondVal = args.pop( 'IP2', None )
      if secondVal:
         args[ 'SRC_GRP' ].update( { 'IP2': secondVal } )

originatorKw = CliMatcher.KeywordMatcher( 'originator',
                                          'Originating router address' )

# -------------------------------------------------------------------------------
# 'show bgp mvpn-ipv4 route-type selective-pmsi [ RD ] [ <source-addr> ]
# [ <group-addr> ] [ originator <orig-ip> ] [ detail ]'
# -------------------------------------------------------------------------------
class ShowBgpMvpnV4Type3Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show bgp mvpn-ipv4 route-type selective-pmsi [ RD ] [ SRC_GRP ] '
              '[ originator ORIG_IP ] [ detail ]' )
   data = {
         'bgp': bgpAfterShow,
         'mvpn-ipv4': mvpnIpv4Kw,
         'route-type': matcherRouteType,
         'selective-pmsi': matcherSpmsi,
         'RD': RdCliExpression,
         'SRC_GRP': SrcGrpExpression,
         'originator': originatorKw,
         'ORIG_IP': matcherOrigIpv4,
         'detail': detailKw,
   }
   cliModel = "MvpnCliModels.MvpnRoutesVrfModel"
   handler = "MvpnShowCliHandler.handlerMvpnV4Spmsi"

if Toggles.MvpnLibToggleLib.toggleMvpnSelectiveTunnelsEnabled():
   BasicCli.addShowCommandClass( ShowBgpMvpnV4Type3Cmd )

# -------------------------------------------------------------------------------
# 'show bgp mvpn-ipv4 route-type source-active [ rd <rd-val> ] [ <source-addr> ]
# [ <group-addr> ] [ detail ]'
# -------------------------------------------------------------------------------
class ShowBgpMvpnV4Type5Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp mvpn-ipv4 route-type source-active [RD] [SRC_GRP] [detail]'
   data = {
         'bgp': bgpAfterShow,
         'mvpn-ipv4': mvpnIpv4Kw,
         'route-type': matcherRouteType,
         'source-active': matcherSourceActive,
         'RD': RdCliExpression,
         'SRC_GRP': SrcGrpExpression,
         'detail': detailKw,
   }
   cliModel = "MvpnCliModels.MvpnRoutesVrfModel"
   handler = "MvpnShowCliHandler.handlerMvpnV4SourceActive"

BasicCli.addShowCommandClass( ShowBgpMvpnV4Type5Cmd )

sourceAsKw = CliMatcher.KeywordMatcher( 'source-as',
                                      'Source autonomous-system' )

#-------------------------------------------------------------------------------
# 'show bgp mvpn-ipv4 route-type shared-tree [ rd <rd-val> ] [ <source-addr> ]
# [ <group-addr> ] [ source-as <as-val> ] [ detail ]'
#-------------------------------------------------------------------------------
class ShowBgpMvpnV4Type6Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show bgp mvpn-ipv4 route-type shared-tree [RD] [source-as ASN] '
              '[SRC_GRP] [detail]' )
   data = {
         'bgp': bgpAfterShow,
         'mvpn-ipv4': mvpnIpv4Kw,
         'route-type': matcherRouteType,
         'shared-tree': matcherSharedTree,
         'RD': RdCliExpression,
         'SRC_GRP': SrcGrpExpression,
         'source-as': sourceAsKw,
         'ASN': bgpTokens.AsNumCliExpr,
         'detail': detailKw,
   }
   cliModel = "MvpnCliModels.MvpnRoutesVrfModel"
   handler = "MvpnShowCliHandler.handlerMvpnV4SharedTree"

BasicCli.addShowCommandClass( ShowBgpMvpnV4Type6Cmd )

#-------------------------------------------------------------------------------
# 'show bgp mvpn-ipv4 route-type source-tree [ detail ]'
#-------------------------------------------------------------------------------
class ShowBgpMvpnV4Type7Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show bgp mvpn-ipv4 route-type source-tree [RD] [source-as ASN] '
              '[SRC_GRP] [detail]' )
   data = {
         'bgp': bgpAfterShow,
         'mvpn-ipv4': mvpnIpv4Kw,
         'route-type': matcherRouteType,
         'source-tree': matcherSourceTree,
         'RD': RdCliExpression,
         'SRC_GRP': SrcGrpExpression,
         'source-as': sourceAsKw,
         'ASN': bgpTokens.AsNumCliExpr,
         'detail': detailKw,
   }
   cliModel = "MvpnCliModels.MvpnRoutesVrfModel"
   handler = "MvpnShowCliHandler.handlerMvpnV4SourceTree"

BasicCli.addShowCommandClass( ShowBgpMvpnV4Type7Cmd )

# -------------------------------------------------------------------------------
# 'show bgp mvpn-ipv4 vrf <vrf-name> route-type intra-as-ipmsi
# [ orig-ip ] [ detail ]'
# -------------------------------------------------------------------------------
class ShowBgpMvpnV4VrfType1Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show bgp mvpn-ipv4 VRF route-type intra-as-ipmsi
               [ ORIG_IP ] [ detail ]'''
   data = {
         'bgp': bgpAfterShow,
         'mvpn-ipv4': mvpnIpv4Kw,
         'VRF': vrfNameExpr,
         'route-type': matcherRouteType,
         'intra-as-ipmsi': matcherIntraAsIpmsi,
         'ORIG_IP': matcherOrigIpv4,
         'detail': detailKw,
   }
   cliModel = "MvpnCliModels.MvpnRoutesVrfModel"
   handler = "MvpnShowCliHandler.handlerMvpnV4VrfIntraAsIpmsi"

BasicCli.addShowCommandClass( ShowBgpMvpnV4VrfType1Cmd )

# -------------------------------------------------------------------------------
# 'show bgp mvpn-ipv4 vrf <vrf-name> route-type selective-pmsi [ <source-addr> ]
# [ < group-addr > ] [ originator <orig-ip> ] [ detail ]'
# -------------------------------------------------------------------------------
class ShowBgpMvpnV4VrfType3Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show bgp mvpn-ipv4 VRF route-type selective-pmsi [ SRC_GRP ] '
              '[ originator ORIG_IP ] [ detail ]' )
   data = {
         'bgp': bgpAfterShow,
         'mvpn-ipv4': mvpnIpv4Kw,
         'VRF': vrfNameExpr,
         'route-type': matcherRouteType,
         'selective-pmsi': matcherSpmsi,
         'SRC_GRP': SrcGrpExpression,
         'originator': originatorKw,
         'ORIG_IP': matcherOrigIpv4,
         'detail': detailKw,
   }
   cliModel = "MvpnCliModels.MvpnRoutesVrfModel"
   handler = "MvpnShowCliHandler.handlerMvpnV4VrfSpmsi"

if Toggles.MvpnLibToggleLib.toggleMvpnSelectiveTunnelsEnabled():
   BasicCli.addShowCommandClass( ShowBgpMvpnV4VrfType3Cmd )

# -------------------------------------------------------------------------------
# 'show bgp mvpn-ipv4 vrf <vrf-name> route-type source-active [ <source-addr> ]
# [ <group-addr> ] [ detail ]'
# -------------------------------------------------------------------------------
class ShowBgpMvpnV4VrfType5Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp mvpn-ipv4 VRF route-type source-active [SRC_GRP] [detail]'
   data = {
         'bgp': bgpAfterShow,
         'mvpn-ipv4': mvpnIpv4Kw,
         'VRF': vrfNameExpr,
         'route-type': matcherRouteType,
         'source-active': matcherSourceActive,
         'SRC_GRP': SrcGrpExpression,
         'detail': detailKw,
   }
   cliModel = "MvpnCliModels.MvpnRoutesVrfModel"
   handler = "MvpnShowCliHandler.handlerMvpnV4VrfSourceActive"

BasicCli.addShowCommandClass( ShowBgpMvpnV4VrfType5Cmd )

# -------------------------------------------------------------------------------
# 'show bgp mvpn-ipv4 vrf route-type shared-tree [ umh-rd <umh-rd-val> ]
# [ <source-addr> ] [ <group-addr> ] [ source-as <as-val> ] [ detail ]'
# -------------------------------------------------------------------------------
class ShowBgpMvpnV4VrfType6Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show bgp mvpn-ipv4 VRF route-type shared-tree [UMH_RD] '
              '[source-as ASN] [SRC_GRP] [detail]' )
   data = {
         'bgp': bgpAfterShow,
         'mvpn-ipv4': mvpnIpv4Kw,
         'VRF': vrfNameExpr,
         'route-type': matcherRouteType,
         'shared-tree': matcherSharedTree,
         'UMH_RD': RdCliExpression,
         'SRC_GRP': SrcGrpExpression,
         'source-as': sourceAsKw,
         'ASN': bgpTokens.AsNumCliExpr,
         'detail': detailKw,
   }
   cliModel = "MvpnCliModels.MvpnRoutesVrfModel"
   handler = "MvpnShowCliHandler.handlerMvpnV4VrfSharedTree"

BasicCli.addShowCommandClass( ShowBgpMvpnV4VrfType6Cmd )

# -------------------------------------------------------------------------------
# 'show bgp mvpn-ipv4 route-type source-tree [ detail ]'
# -------------------------------------------------------------------------------
class ShowBgpMvpnV4VrfType7Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show bgp mvpn-ipv4 VRF route-type source-tree [UMH_RD] '
              '[source-as ASN] [SRC_GRP] [detail]' )
   data = {
         'bgp': bgpAfterShow,
         'mvpn-ipv4': mvpnIpv4Kw,
         'VRF': vrfNameExpr,
         'route-type': matcherRouteType,
         'source-tree': matcherSourceTree,
         'UMH_RD': RdCliExpression,
         'SRC_GRP': SrcGrpExpression,
         'source-as': sourceAsKw,
         'ASN': bgpTokens.AsNumCliExpr,
         'detail': detailKw,
   }
   cliModel = "MvpnCliModels.MvpnRoutesVrfModel"
   handler = "MvpnShowCliHandler.handlerMvpnV4VrfSourceTree"

BasicCli.addShowCommandClass( ShowBgpMvpnV4VrfType7Cmd )

# ------------------------------------------------------
# "show bgp mvpn-ipv4 summary"
# ------------------------------------------------------
class ShowBgpMvpnV4SummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp mvpn-ipv4 summary'

   data = {
         'bgp': bgpAfterShow,
         'mvpn-ipv4': mvpnIpv4Kw,
         'summary': summaryKw,
   }
   cliModel = summaryVrfModel
   handler = "MvpnShowCliHandler.handlerMvpnV4Summary"

BasicCli.addShowCommandClass( ShowBgpMvpnV4SummaryCmd )

# -------------------------------------------------------------------------------
# 'show bgp mvpn-ipv4 [ vrf <vrf-name> ] rsvp leaf'
# -------------------------------------------------------------------------------
class ShowBgpMvpnVrfRsvpLeafCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show bgp mvpn-ipv4 [ VRF ] rsvp leaf' )
   data = {
         'bgp': bgpAfterShow,
         'mvpn-ipv4': mvpnIpv4Kw,
         'VRF': vrfNameExpr,
         'rsvp': "RSVP-TE information",
         'leaf': "Show leaf information",
   }
   cliModel = "MvpnCliModels.MvpnRsvpLeafStatus"
   handler = "MvpnShowCliHandler.handleBgpMvpnRsvpLeaf"

# --------------------------------------------------------------------------------
# show mvpn ipv4 [ vrf <vrf-name> ] [ ( mldp | rsvp ) p2mp ] decapsulation
# --------------------------------------------------------------------------------
class ShowMvpnIpv4DecapCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mvpn ipv4 [ VRF ] [ PROTOCOL p2mp ] decapsulation'
   data = {
      'mvpn': 'Multicast VPN',
      'ipv4': ipv4MatcherForShow,
      'VRF': VrfExprFactory( helpdesc='VRF name' ),
      'PROTOCOL': Node( matcher=CliMatcher.EnumMatcher( protocolValues ),
                        maxMatches=1 ),
      'p2mp': 'point-to-multipoint',
      'decapsulation': 'Decapsulation Information',
   }
   cliModel = MvpnLibCliModels.PmsiTunnelDecapStatus
   handler = "MvpnShowCliHandler.handleMvpnIpv4DecapCmd"

# -------------------------------------------------------------------------------
# 'show mvpn ipv4 [ vrf <vrfName> ] [ ( mldp|rsvp ) p2mp ] encapsulation'
# -------------------------------------------------------------------------------
class ShowMvpnEncapStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show mvpn ipv4 [ VRF ] [ PROTOCOL p2mp ] encapsulation' )
   data = {
         'mvpn': "Multicast VPN",
         'ipv4': ipv4MatcherForShow,
         'VRF': vrfNameExpr,
         'PROTOCOL': Node( matcher=CliMatcher.EnumMatcher( protocolValues ),
                            maxMatches=1 ),
         'p2mp': 'point-to-multipoint',
         'encapsulation': 'encapsulation status entries'
   }
   cliModel = "BgpVpnModels.MvpnEncapStatusModel"
   handler = "MvpnShowCliHandler.handleMvpnEncapStatus"

if toggleMvpnRsvpP2mpEnabled():
   BasicCli.addShowCommandClass( ShowBgpMvpnVrfRsvpLeafCmd )
   BasicCli.addShowCommandClass( ShowMvpnIpv4DecapCmd )
   BasicCli.addShowCommandClass( ShowMvpnEncapStatusCmd )
