# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
import CliPlugin.TechSupportCli
from CliPlugin.MvrpModels import MvrpIntfInfo, MvrpVlanInfo, MrpDatabase
import ConfigMount
import Intf
import LazyMount
import MultiRangeRule
import MvrpCliLib
import ShowCommand
import Tac
import Tracing
from Toggles.MvrpToggleLib import toggleDot1xMvrpVsaEnabled

__defaultTraceHandle__ = Tracing.Handle( 'MvrpCli' )
t0 = Tracing.trace0
t1 = Tracing.trace1

#------------------------------------------------------------------------------------
# The Mvrp globals
#------------------------------------------------------------------------------------
mvrpCliConfig = None
mvrpDot1xConfig = None
mvrpStatus = None
mrpStatus = None
mvrpHwStatus = None
mvrpHwConfig = None

mvrpApp = Tac.Type( 'Mrp::MrpApplication' ).MVRP

class MvrpModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( 'Ethernet' ) and not mode.intf.isSubIntf()

IntfCli.IntfConfigMode.addModelet( MvrpModelet )

#------------------------------------------------------------------------------------
# Mvrp tokens
#------------------------------------------------------------------------------------
def guardMvrp( mode, token ):
   if mvrpHwStatus.mvrpSupported:
      return None
   return CliParser.guardNotThisPlatform

matcherInterfaces = CliMatcher.KeywordMatcher( 'interfaces',
      helpdesc='Show MVRP information for specific interfaces' )
nodeMvrp = CliCommand.guardedKeyword( 'mvrp',
      helpdesc='Show MVRP information', guard=guardMvrp )
matcherIntfRange = Intf.IntfRange.IntfRangeMatcher(
      explicitIntfTypes=( EthIntfCli.EthPhyAutoIntfType, ) )

#--------------------------------------------------------------------------------
# [ no | default ] mvrp
#--------------------------------------------------------------------------------
class MvrpCmd( CliCommand.CliCommandClass ):
   syntax = 'mvrp'
   noOrDefaultSyntax = syntax
   data = {
      'mvrp' : CliCommand.guardedKeyword( 'mvrp',
         helpdesc='Configure MVRP parameters', guard=guardMvrp ),
   }

   @staticmethod
   def handler( mode, args ):
      intf = mode.intf.name
      mvrpCliConfig.intfEnable[ intf ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = mode.intf.name
      del mvrpCliConfig.intfEnable[ intf ]

MvrpModelet.addCommandClass( MvrpCmd )

#--------------------------------------------------------------------------------
# show mvrp [ interfaces INTF ]
#--------------------------------------------------------------------------------
class MvrpInterfacesIntfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mvrp [ interfaces INTF ]'
   data = {
      'mvrp' : nodeMvrp,
      'interfaces' : matcherInterfaces,
      'INTF' : matcherIntfRange,
   }
   cliModel = MvrpIntfInfo

   @staticmethod
   def handler( mode, args ):
      # Global information
      intf = args.get( 'INTF' )
      mvrpModel = MvrpIntfInfo()
      mvrpModel.mvrpEnabled = mvrpHwConfig.mvrpGlobalEnable

      t0( 'showMvrp', intf )

      # Interface information
      intfs = IntfCli.Intf.getAll( mode, intf, None, EthIntfCli.EthIntf )

      t0( 'intfs', intfs )

      for intf in intfs:
         if not intf.name.startswith( 'Ethernet' ):
            #and not intf.name.startswith( 'Port-Channel' ):
            continue
         mvrpIntfModel = mvrpModel.Intf()
         mvrpModel.interfaces[ intf.name ] = mvrpIntfModel

         if intf.name not in mvrpStatus.intfStatus:
            mvrpIntfModel.operState = 'Disabled'
         else:
            intfStatus = mvrpStatus.intfStatus[ intf.name ]
            if ( intfStatus.linkStatus == 'linkUp' and
                 intfStatus.portState == 'portForwarding' ):
               mvrpIntfModel.operState = 'Active'
            elif intfStatus.linkStatus != 'linkUp':
               # Check link status before port state, because if link is down,
               # port state will also be blocked.
               mvrpIntfModel.operState = 'LinkDown'
            elif intfStatus.portState != 'portForwarding':
               mvrpIntfModel.operState = 'Blocked'

            if mvrpIntfModel.operState != 'Active':
               continue

            mvrpIntfModel.registeredVlans = list( intfStatus.registeredVlans )
            mvrpIntfModel.declaredVlans = list( intfStatus.declaredVlans )

            if ( toggleDot1xMvrpVsaEnabled() and
                 intf.name in mvrpDot1xConfig.intfEnable ):
               mvrpIntfModel.requester = 'Dot1x'

      return mvrpModel

BasicCli.addShowCommandClass( MvrpInterfacesIntfCmd )

#--------------------------------------------------------------------------------
# show mvrp vlans [ VLAN_SET ]
#--------------------------------------------------------------------------------
class MvrpVlansCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mvrp vlans [ VLAN_SET ]'
   data = {
      'mvrp' : nodeMvrp,
      'vlans' : 'MVRP Vlans',
      'VLAN_SET' : MultiRangeRule.MultiRangeMatcher(
         rangeFn=lambda: ( 1, 4094 ),
         noSingletons=False,
          helpdesc='VLAN ID or range(s) of VLAN IDs' ),
   }
   cliModel = MvrpVlanInfo

   @staticmethod
   def handler( mode, args ):
      vlanIds = list( args[ 'VLAN_SET' ].values() ) if 'VLAN_SET' in args else None
      mvrpModel = MvrpVlanInfo()

      if not vlanIds:
         vlans = list( range( 1, 4094 ) )
      else:
         vlans = vlanIds

      for vlanId in vlans:
         vlanStatus = mvrpStatus.vlanStatus.get( vlanId )
         if vlanStatus:
            mvrpVlanModel = MvrpVlanInfo.Vlan()
            mvrpVlanModel.registeredIntfs = list( vlanStatus.registeredIntfs )
            mvrpVlanModel.declaredIntfs = list( vlanStatus.declaredIntfs )
            mvrpModel.vlans[ vlanId ] = mvrpVlanModel

      return mvrpModel

BasicCli.addShowCommandClass( MvrpVlansCmd )

#--------------------------------------------------------------------------------
# show mvrp mrp [ interfaces INTF ] [ detail ]
#--------------------------------------------------------------------------------
class MvrpMrpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mvrp mrp [ interfaces INTF ] [ detail ]'
   data = {
      'mvrp' : nodeMvrp,
      'mrp' : CliCommand.guardedKeyword( 'mrp', helpdesc='Show MRP information',
         guard=guardMvrp, hidden=True ),
      'interfaces' : matcherInterfaces,
      'INTF' : matcherIntfRange,
      'detail' : 'More comprehensive output',
   }
   cliModel = MrpDatabase

   @staticmethod
   def handler( mode, args ):
      intf = args.get( 'INTF' )
      detail = 'detail' in args
      intfs = IntfCli.Intf.getAll( mode, intf, None, EthIntfCli.EthIntf )

      mrpModel = MrpDatabase()
      for intf in intfs:
         if intf.name not in mrpStatus.application[ mvrpApp ].attrDatabase:
            continue
         intfDatabase = mrpStatus.application[ mvrpApp ].attrDatabase[ intf.name ]
         mrpIntfModel = mrpModel.IntfDatabase()
         mrpModel.interfaces[ intf.name ] = mrpIntfModel

         for attrType in intfDatabase.intfTypeApplicant:
            intfAttrDatabase = intfDatabase.intfTypeApplicant[ attrType ]
            mrpAttrTypeModel = mrpIntfModel.AttrTypeDatabase()
            mrpIntfModel.attributeTypes[ attrType ] = mrpAttrTypeModel

            for mrpAttributeKey in intfAttrDatabase.mrpAttribute:
               # Filter on the basis of attrType
               intfMrpAttrDatabase = intfAttrDatabase.mrpAttribute[ mrpAttributeKey ]
               mrpAttrModel = mrpAttrTypeModel.Attribute()
               mrpAttrModel.applicantState = intfMrpAttrDatabase.applicantState
               mrpAttrModel.registrarState = intfMrpAttrDatabase.registrarState

               if detail:
                  firstValue = intfMrpAttrDatabase.attributeValue
                  values = MvrpCliLib.getAttrs( firstValue, attrType )
                  mrpAttrModel.value = str( values )

               mrpAttrTypeModel.attributes[ mrpAttributeKey ] = mrpAttrModel

      return mrpModel

BasicCli.addShowCommandClass( MvrpMrpCmd )

class MvrpIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      t0( 'MvrpIntfJanitor: interface', self.intf_.name, 'going away' )
      del mvrpCliConfig.intfEnable[ self.intf_.name ]

#--------------------------------------------------------------------------------
# Register to show tech-support
#--------------------------------------------------------------------------------
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2014-04-07 00:00:00',
   cmds=[ 'show mvrp' ],
   cmdsGuard=lambda: mvrpHwStatus.mvrpSupported )

#--------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#--------------------------------------------------------------------------------
def Plugin( entityManager ):
   global mvrpCliConfig, mvrpStatus, mrpStatus, mvrpHwStatus, mvrpHwConfig
   global mvrpDot1xConfig

   mvrpCliConfig = ConfigMount.mount( entityManager, 'mvrp/input/config/cli',
                                 'Mvrp::Config', 'w' )
   if toggleDot1xMvrpVsaEnabled():
      mvrpDot1xConfig = LazyMount.mount( entityManager, 'mvrp/input/config/dot1x',
                                         'Mvrp::Config', 'r' )
   mvrpStatus = LazyMount.mount( entityManager, 'mvrp/status',
                                 'Mvrp::Status', 'r' )
   mrpStatus = LazyMount.mount( entityManager, 'mrp/status',
                                 'Mrp::MrpStatus', 'r' )
   mvrpHwStatus = LazyMount.mount( entityManager, 'mvrp/hwStatus',
                                      'Mvrp::HwStatus', 'r' )
   mvrpHwConfig = LazyMount.mount( entityManager, 'mvrp/hwConfig',
                                     'Mvrp::HwConfig', 'r' )

   IntfCli.Intf.registerDependentClass( MvrpIntfJanitor )
