#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
from CliCommand import (
      CliCommandClass,
      guardedKeyword,
      singleNode,
      Node,
)
import CliMatcher
from CliPlugin.IntfCli import Intf
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.NatCli import (
      cmdClearIpNatCounters,
      cmdClearIpNatTranslations,
      matcherAddress,
      matcherInterface,
      matcherIpAddr,
      matcherProfile,
      matcherProfileName,
      matcherProto,
      matcherSrcPort,
      matcherTranslation,
      natInVrfSupported,
      natIntfDynamicSupported,
      nodeCounters,
)
from CliPlugin.VrfCli import VrfExprFactory
from CliToken.Clear import clearKwNode
from CliToken.Ip import ipMatcherForClear

matcherTranslation = CliMatcher.KeywordMatcher( 'translation',
                                                helpdesc='Nat translation rule' )
nodeNat = guardedKeyword( 'nat', 'Reset NAT configuration', natIntfDynamicSupported )

#--------------------------------------------------------------------------------
# clear ip nat counters [ VRF ]
#--------------------------------------------------------------------------------
class ClearIpNatCountersCmd( CliCommandClass ):
   syntax = 'clear ip nat counters [ VRF ]'
   data = {
      'clear' : clearKwNode,
      'ip' : ipMatcherForClear,
      'nat' : nodeNat,
      'counters' : nodeCounters,
      'VRF' : VrfExprFactory( helpdesc='Clear NAT counters in a VRF',
                              inclAllVrf=True,
                              guard=natInVrfSupported ),
   }

   handler = cmdClearIpNatCounters

BasicCliModes.EnableMode.addCommandClass( ClearIpNatCountersCmd )

#--------------------------------------------------------------------------------
# clear ip nat flow translation [ { ( interface INTF ) | ( profile PROFILENAME ) |
#                                   VRF | PROTO | 
#                                   ( address SRCIPADDR [ SRCPORT ] [ DESTIPADDR ]
#                                     [ DESTPORT ] ) } ]
#--------------------------------------------------------------------------------
sharedScope = object()
class ClearIpNatFlowTranslationCmd( CliCommandClass ):
   syntax = ( 'clear ip nat ( ( flow translation ) | deprecatedTranslation ) '
              '[ { ( interface INTF ) | ( profile PROFILENAME ) | VRF | PROTO | '
                  '( address SRCIPADDR [ SRCPORT ] [ DESTIPADDR [ DESTPORT ] ] ) '
              '} ]' )
   data = {
      'clear' : clearKwNode,
      'ip' : ipMatcherForClear,
      'nat' : nodeNat,
      'flow' :'NAT flow',
      'translation' : matcherTranslation,
      'deprecatedTranslation' : Node( matcher=matcherTranslation,
                                   deprecatedByCmd="clear ip nat flow translation" ),
      'interface' : singleNode( matcherInterface, sharedMatchObj=sharedScope ),
      'INTF' : Intf.matcher,
      'profile' : singleNode( matcherProfile, sharedMatchObj=sharedScope ),
      'PROFILENAME' : matcherProfileName,
      'VRF' : VrfExprFactory( helpdesc='VRF',
                              sharedMatchObj=sharedScope,
                              inclAllVrf=True,
                              guard=natInVrfSupported ),
      'PROTO' : singleNode( matcherProto ),
      'address' : singleNode( matcherAddress ),
      'SRCIPADDR' : matcherIpAddr,
      'SRCPORT' : matcherSrcPort,
      'DESTIPADDR' : IpAddrMatcher( helpdesc='Destination host IP address' ),
      'DESTPORT' : CliMatcher.IntegerMatcher( 1, 65535,
                                              helpdesc='Destination host port' ),
   }

   handler = cmdClearIpNatTranslations

BasicCliModes.EnableMode.addCommandClass( ClearIpNatFlowTranslationCmd )

