#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
from CliCommand import (
      Node,
      guardedKeyword,
      CliCommandClass,
)
from CliMatcher import (
      KeywordMatcher,
      IntegerMatcher,
      StringMatcher,
)
from CliToken.Ip import ipMatcherForConfigIf, ipMatcherForConfig
from CliPlugin.IntfCli import Intf
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.AclCli import userIpAclNameMatcher
import CliPlugin.NatCli as NatCli # pylint: disable=consider-using-from-import
from CliPlugin.NatCli import (
      matcherNat,
      matcherSource,
      matcherDestination,
      matcherStatic,
      matcherIpLocal,
      matcherIpGlobal,
      matcherPoolName,
      matcherServiceListName,
      matcherProto,
      matcherProfileName,
      matcherPrefixValue,
      matcherFlowPolicyName,
      nodeNat,
      nodePool,
      nodeServiceList,
      NatAclExpression,
)
from CliPlugin.NatCli import (
      natIpIntfConfigSupported,
      natL4PortSupported,
      natStaticSupported,
      natAddrOnlySupported,
      natIpProfileConfigSupported,
      natIpServiceProfileConfigSupported,
      natIntfSnatIngressStaticSupported,
      natIntfSnatEgressStaticSupported,
      natIntfDnatEgressStaticSupported,
      natGroupSupported,
      natDynamicTwiceSupported,
      natProtocolSupported,
      natIntfDynamicSupported,
      natInVrfSupported,
      natFlowSupported,
      natSyncFwdTrappedPktSupported,
      natSwPrefixTranslationSupported,
)
from CliPlugin.IraIpIntfCli import RoutingProtocolIntfConfigModelet
from CliPlugin.VrfCli import VrfExprFactory

matcherPriorityValue = IntegerMatcher( 0, 2**32-1, helpdesc='Priority' )

nodeDynamic = guardedKeyword( 'dynamic', 'Dynamic NAT', natIntfDynamicSupported )
nodeStatic = Node( matcher=matcherStatic, guard=natStaticSupported )
nodeSnatIngress = guardedKeyword( 'ingress', 'Ingress NAT',
                                  natIntfSnatIngressStaticSupported )
nodeSnatEgress = guardedKeyword( 'egress', 'Egress NAT',
                                  natIntfSnatEgressStaticSupported )
# Enable the ingress DNAT keyword when egress DNAT is supported
# to allow specifying the default direction
nodeDnatIngress = guardedKeyword( 'ingress', 'Ingress NAT',
                                  natIntfDnatEgressStaticSupported )
nodeDnatEgress = guardedKeyword( 'egress', 'Egress NAT',
                                 natIntfDnatEgressStaticSupported )
nodeGroup = guardedKeyword( 'group', 'Twice NAT group', natGroupSupported )
nodeDynGroup = guardedKeyword( 'group', 'Twice NAT group',
                               natDynamicTwiceSupported )
nodePriority = guardedKeyword( 'priority', 'Dynamic NAT Priority',
                               natIntfDynamicSupported )
nodePort = Node( matcher=IntegerMatcher( 1, 65535, helpdesc='Port' ),
                 guard=natL4PortSupported )
nodeProtocol = guardedKeyword( 'protocol', 'TCP or UDP port', natProtocolSupported )
nodePrefix = guardedKeyword( 'prefix-length', 'Subnet prefix length. ' \
                             'Creates NAT entries for all hosts',
                             natSwPrefixTranslationSupported )
matcherPolicy = KeywordMatcher( 'policy', helpdesc='NAT flow policy' )
nodeComment = KeywordMatcher( 'comment', helpdesc='Comment on this NAT rule' )
matcherCommentStr = StringMatcher( helpname='COMMENT',
                                   helpdesc='Comment on this NAT rule' )
matcherMiss = KeywordMatcher( 'miss', helpdesc='miss' )

for ( mode, guard ) in (
      ( RoutingProtocolIntfConfigModelet, natIpIntfConfigSupported ),
      ( NatCli.NatProfileConfigMode, natIpProfileConfigSupported ) ):
   #------------------------------
   # dynamic nat configurations
   #------------------------------
   #--------------------------------------------------------------------------------
   # ip nat source dynamic acl ACLNAME pool POOLNAME
   #          [ full-cone | address-only | group GROUP ]
   #          [ service-list SERVICELISTNAME ] [ priority PRIORITY ]
   #          [ comment COMMENT ]
   #--------------------------------------------------------------------------------
   class IpNatDynamicSourceWithPoolCmd( CliCommandClass ):
      syntax = (
         'ip nat source dynamic acl ACLNAME pool POOLNAME '
         '[ full-cone | address-only | ( group GROUP ) ] '
         '[ service-list SERVICELISTNAME ] '
         '[ priority PRIORITY ]'
         '[ comment COMMENT ]' )
      noOrDefaultSyntax = (
         'ip nat source dynamic acl ACLNAME pool POOLNAME '
         '[ full-cone | address-only | ( group GROUP ) ] ...' )
      data = {
         'ip' : ipMatcherForConfigIf,
         'nat' : matcherNat,
         'source' : Node( matcher=matcherSource, guard=guard ),
         'dynamic' : nodeDynamic,
         'acl' : NatAclExpression,
         'ACLNAME' : userIpAclNameMatcher,
         'pool' : nodePool,
         'POOLNAME' : matcherPoolName,
         'full-cone' : guardedKeyword( 'full-cone', 'Enable full-cone NAT',
                                       natIntfDynamicSupported ),
         'address-only' : guardedKeyword( 'address-only', 'Enable address-only NAT',
                                          natAddrOnlySupported ),
         'group' : nodeDynGroup,
         'GROUP' : IntegerMatcher( 1, 65535, helpdesc='Group ID' ),
         'service-list' : nodeServiceList,
         'SERVICELISTNAME' : matcherServiceListName,
         'priority' : nodePriority,
         'PRIORITY' : matcherPriorityValue,
         'comment' : nodeComment,
         'COMMENT' : matcherCommentStr,
      }

      handler = NatCli.configDynamicNat
      noOrDefaultHandler = NatCli.configNoDynamicNat

   mode.addCommandClass( IpNatDynamicSourceWithPoolCmd )

   #--------------------------------------------------------------------------------
   # ip nat destination dynamic acl ACLNAME pool POOLNAME [ group GROUP ]
   # [ service-list SERVICELISTNAME ] [ priority PRIORITY ] [ comment COMMENT ]
   #--------------------------------------------------------------------------------
   class IpNatDynamicDestinationWithPoolCmd( CliCommandClass ):
      syntax = (
         'ip nat destination dynamic acl ACLNAME pool POOLNAME '
         '[ group GROUP ]'
         '[ service-list SERVICELISTNAME ] [ priority PRIORITY ]'
         '[ comment COMMENT ]' )
      noOrDefaultSyntax = (
         'ip nat destination dynamic acl ACLNAME pool POOLNAME '
         '[ group GROUP ] ...' )
      data = {
         'ip' : ipMatcherForConfigIf,
         'nat' : matcherNat,
         'destination' : Node( matcher=matcherDestination, guard=guard ),
         'dynamic' : nodeDynamic,
         'acl' : NatAclExpression,
         'ACLNAME' : userIpAclNameMatcher,
         'pool' : nodePool,
         'POOLNAME' : matcherPoolName,
         'group' : nodeDynGroup,
         'GROUP' : IntegerMatcher( 1, 65535, helpdesc='Group ID' ),
         'service-list' : nodeServiceList,
         'SERVICELISTNAME' : matcherServiceListName,
         'priority' : nodePriority,
         'PRIORITY' : matcherPriorityValue,
         'comment' : nodeComment,
         'COMMENT' : matcherCommentStr,
      }

      handler = NatCli.configDynamicNat
      noOrDefaultHandler = NatCli.configNoDynamicNat

   mode.addCommandClass( IpNatDynamicDestinationWithPoolCmd )

   #--------------------------------------------------------------------------------
   # ip nat source dynamic acl ACLNAME overload [ priority PRIORITY ]
   #                                            [ comment COMMENT ]
   #--------------------------------------------------------------------------------
   class IpNatDynamicWithOverloadCmd( CliCommandClass ):
      syntax = (
         'ip nat source dynamic acl ACLNAME overload [ priority PRIORITY ]'
         '[ comment COMMENT ]' )
      noOrDefaultSyntax = (
         'ip nat source dynamic acl ACLNAME overload ...' )
      data = {
         'ip' : ipMatcherForConfigIf,
         'nat' : matcherNat,
         'source' : Node( matcher=matcherSource, guard=guard ),
         'dynamic' : nodeDynamic,
         'acl' : NatAclExpression,
         'ACLNAME' : userIpAclNameMatcher,
         'overload' : guardedKeyword( 'overload',
                                      'NAT overload many-to-one translation',
                                      natIntfDynamicSupported ),
         'priority' : nodePriority,
         'PRIORITY' : matcherPriorityValue,
         'comment' : nodeComment,
         'COMMENT' : matcherCommentStr,
      }

      handler = NatCli.configDynamicNat
      noOrDefaultHandler = NatCli.configNoDynamicNat

   mode.addCommandClass( IpNatDynamicWithOverloadCmd )

   #------------------------------
   # static nat configurations
   #------------------------------
   #--------------------------------------------------------------------------------
   # ip nat TARGET [ ingress | egress ] static LOCALIP [ LOCALPORT ] [ acl ACLNAME ]
   # GLOBALIP [ GLOBALPORT ] [ protocol PROTO ]
   # [ group GROUP | prefix-length PREFIXLEN ] [ comment COMMENT ]
   #--------------------------------------------------------------------------------
   class IpNatStaticCmd( CliCommandClass ):
      # syntax and noOrDefaultSyntax are almost the same except GLOBALIP is optional
      syntax = (
         'ip nat '
         '( source [ SNAT_INGRESS | SNAT_EGRESS ] ) | '
         '( destination [ DNAT_INGRESS | DNAT_EGRESS ] ) '
         'static LOCALIP [ LOCALPORT ] [ acl ACLNAME ] GLOBALIP [ GLOBALPORT ] '
         '[ protocol PROTO ] [ ( group GROUP ) | ( prefix-length PREFIXLEN ) ] '
         '[ comment COMMENT ]' )
      noOrDefaultSyntax = (
         'ip nat '
         '( source [ SNAT_INGRESS | SNAT_EGRESS ] ) | '
         '( destination [ DNAT_INGRESS | DNAT_EGRESS ] ) '
         'static LOCALIP [ LOCALPORT ] [ acl ACLNAME ] [ GLOBALIP [ GLOBALPORT ] ] '
         '[ protocol PROTO ] [ ( group GROUP ) | ( prefix-length PREFIXLEN ) ] ...' )

      data = {
         'ip' : ipMatcherForConfigIf,
         'nat' : matcherNat,
         'source' : Node( matcher=matcherSource, guard=guard ),
         'SNAT_INGRESS' : nodeSnatIngress,
         'SNAT_EGRESS' : nodeSnatEgress,
         'destination' : Node( matcher=matcherDestination, guard=guard ),
         'DNAT_INGRESS' : nodeDnatIngress,
         'DNAT_EGRESS' : nodeDnatEgress,
         'static' : nodeStatic,
         'LOCALIP' : matcherIpLocal,
         'LOCALPORT' : nodePort,
         'acl' : NatAclExpression,
         'ACLNAME' : userIpAclNameMatcher,
         'GLOBALIP' : matcherIpGlobal,
         'GLOBALPORT' : nodePort,
         'protocol' : nodeProtocol,
         'PROTO' : matcherProto,
         'group' : nodeGroup,
         'GROUP' : IntegerMatcher( 1, 65535, helpdesc='Group ID' ),
         'prefix-length' : nodePrefix,
         'PREFIXLEN' : matcherPrefixValue,
         'comment' : nodeComment,
         'COMMENT' : matcherCommentStr,
      }

      handler = NatCli.cmdStaticNat
      noOrDefaultHandler = NatCli.cmdNoStaticNat

   mode.addCommandClass( IpNatStaticCmd )

   #--------------------------------------------------------------------------------
   # ip nat flow policy <name>
   #--------------------------------------------------------------------------------
   class IntfNatFlowPolicyCmd( CliCommandClass ):
      syntax = ( 'ip nat flow policy POLICYNAME' )
      noOrDefaultSyntax = ( 'ip nat flow policy POLICYNAME ...' )
      data = {
         'ip' : ipMatcherForConfigIf,
         'nat' : matcherNat,
         'flow' : guardedKeyword( 'flow', 'NAT flow', natFlowSupported ),
         'policy' : Node( matcher=matcherPolicy, guard=guard ),
         'POLICYNAME' : matcherFlowPolicyName,
      }
      handler = NatCli.configNatFlowPolicyIntf
      noOrDefaultHandler = NatCli.clearNatFlowPolicyIntf

   if mode == NatCli.NatProfileConfigMode:
      mode.addCommandClass( IntfNatFlowPolicyCmd )

   #--------------------------------------------------------------------------------
   # translation miss match peer-criteria action forward interface INTF
   #   peer-address IP
   #--------------------------------------------------------------------------------
   class TranslationMissForwardIntfCmd( CliCommandClass ):
      syntax = (
         'translation miss match peer-criteria action forward'
         ' interface INTF peer-address IP' )
      noOrDefaultSyntax = (
         'translation miss match peer-criteria ...' )
      data = {
         'translation' : guardedKeyword( 'translation',
                                         'NAT translation rule',
                                         natSyncFwdTrappedPktSupported ),
         'miss' : 'Missed NAT translation',
         'match' : 'Match',
         'peer-criteria' : 'Match the criteria classified as peer-criteria',
         'action' : 'Match action',
         'forward' : 'Forward packet',
         'interface' : 'Interface to send the packet on',
         'INTF' :  Intf.matcher,
         'peer-address' : 'Peer IP address',
         'IP' : IpAddrMatcher( helpdesc='Peer IP address' ),
      }

      handler = NatCli.configTranslationMissForwardIntf
      noOrDefaultHandler = NatCli.clearTranslationMissForwardIntf

   if mode == NatCli.NatProfileConfigMode:
      mode.addCommandClass( TranslationMissForwardIntfCmd )

#------------------------------
# miscellaneous nat configurations
#------------------------------
#--------------------------------------------------------------------------------
# [ no | default ] ip nat service-profile PROFILENAME 
#--------------------------------------------------------------------------------
class IntfNatProfileCmd( CliCommandClass ):
   syntax = 'ip nat service-profile PROFILENAME'
   noOrDefaultSyntax = 'ip nat service-profile ...'
   data = {
      'ip' : ipMatcherForConfigIf,
      'nat' : matcherNat,
      'service-profile' : guardedKeyword( 'service-profile', 'NAT interface profile',
                                          natIpServiceProfileConfigSupported ),
      'PROFILENAME' : matcherProfileName,
   }

   handler = NatCli.setIntfNatProfile
   noOrDefaultHandler = NatCli.clearIntfNatProfile

RoutingProtocolIntfConfigModelet.addCommandClass( IntfNatProfileCmd ) 

#--------------------------------------------------------------------------------
# [ no | default ] ip nat profile PROFILENAME [ vrf VRF ]
#--------------------------------------------------------------------------------
class IpNatProfileProfilenameCmd( CliCommandClass ):
   syntax = 'ip nat profile PROFILENAME [ VRF ]'
   noOrDefaultSyntax = 'ip nat profile PROFILENAME ...'
   data = {
      'ip' : ipMatcherForConfig,
      'nat' : nodeNat,
      'profile' : guardedKeyword( 'profile', 'NAT profile definition',
                                  natIpProfileConfigSupported ),
      'PROFILENAME' : matcherProfileName,
      'VRF' : VrfExprFactory( helpdesc='Specify VRF for NAT profile',
                              guard=natInVrfSupported ),
   }

   handler = NatCli.gotoIpNatProfileConfigMode
   noOrDefaultHandler = NatCli.deleteIpNatProfileConfig

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatProfileProfilenameCmd )

