#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from CliModel import Float
from CliModel import Bool
from CliModel import Enum
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from CliModel import Dict
from IntfModels import Interface
from ArnetModel import IpAddr, Ip4Address, MacAddress
from ArnetModel import IpAddrAndPort, IpAddrAndMask
from Arnet import sortIntf
import Tac
from CliModel import GeneratorList # pylint: disable=ungrouped-imports
from CliModel import DeferredModel
from datetime import datetime
from datetime import timedelta
from IpLibConsts import DEFAULT_VRF
from socket import IPPROTO_TCP, IPPROTO_UDP, IPPROTO_ICMP, IPPROTO_IGMP
import TableOutput

MOUNT_STATE = Tac.Type( "Ip::Nat::Sync::MountState" )
STATUS_STATE = Tac.Type( "Ip::Nat::Sync::Status::State" )
PORT_RANGE_MODE = Tac.Type( "Ip::Nat::Sync::PortRangeMode" )

ipProtoMap = { IPPROTO_TCP: 'TCP',
               IPPROTO_UDP: 'UDP',
               IPPROTO_IGMP: 'IGMP',
               IPPROTO_ICMP: 'ICMP' }

#-------------------------------------------------------------------
# 'show ip nat translation'
#-------------------------------------------------------------------

showIpNatFmtStr = "%-21.21s %-21.21s %-21.21s %-21.21s %-5s %-3s %-4s %-3s %-20s"
showIpNatFmtStrDetail = showIpNatFmtStr + " %-15s %3s %6s %15s %15s"
showIpNatFmtStrComment = showIpNatFmtStr + " %-30s"
showIpNatFmtStrDetailComment = showIpNatFmtStrDetail + " %-30s"

class NatIpTranslations( Model ):
   __revision__ = 3
   class Translation( Model ):
      class Details( Model ):
         vrfName = Str( help="VRF name" )
         established = Bool( help="Network connection established", optional=True )
         sourceVrfName = Str( help="Source VRF name", optional=True )
         packetCount = Int( help="Number of packets translated from the "
                            "original IP address" )
         packetReplyCount = Int( help="Number of packets translated to the "
                                 "original IP address" )
         origin = Str( help="Discovered locally or advertised from sync peer" )

      interface = Str( help="Interface or NAT profile name", optional=True )

      srcAddress = Submodel( valueType=IpAddr, optional=True,
                             help="The original source IP address "
                             "and port/mask of the packet." )
      dstAddress = Submodel( valueType=IpAddr, optional=True,
                             help="The destination IP address "
                             "and port/mask of the packet." )
      newSrcAddress = Submodel( valueType=IpAddr, optional=True,
                                   help="The translated source IP address "
                                   "and port/mask of the packet." )
      newDstAddress = Submodel( valueType=IpAddr, optional=True,
                                help="The translated destination IP address "
                                "and port/mask of the packet." )
      protocol = Int( help="IP Protocol Number, if specified", optional=True )
      target = Enum( values=( "SRC", "DST" ),
                     help="Source NAT or destination NAT", optional=True )
      natType = Enum( values=( "ST", "E-ST", "TW", "D-TW",
                               "DYN", "FC", "AO" ),
                      help="Static NAT or Dynamic NAT" )
      direction = Enum( values=( "ING", "EGR" ), help="Ingress or Egress interface",
                        optional=True )
      details = Submodel( valueType=Details, optional=True,
                          help="Details, if requested" )
      comment = Str( help="comment on translation rule", optional=True )

   translations = GeneratorList( valueType=Translation,
         help="List of NAT translations active on this interface." )

   detailedView = Bool( help="includes additional details if True" )

   hasComments = Bool( help="True if any translations shown have comments",
                       default=False )

   def degrade( self, dictRepr, revision ):
      if revision < 3:
         newDict = {}
         newDict[ 'detailedView' ] = dictRepr[ 'detailedView' ]
         newDict[ 'pendingView' ] = False
         newDict[ 'translations' ] = []
         for t in dictRepr[ 'translations' ]:
            natType = t[ 'natType' ]
            if natType in [ 'TW', 'D-TW' ]:
               continue
            t2 = {}
            if natType in [ 'ST', 'E-ST' ]:
               t2[ 'natType' ] = 'STAT'
            else:
               t2[ 'natType' ] = 'DYN'
            t2[ 'table' ] = 'egress' if natType == 'E-ST' else 'defaultImpl'
            if 'target' in t:
               t2[ 'target' ] = t[ 'target' ]
               if t[ 'target' ] == 'SRC':
                  if 'newSrcAddress' in t:
                     t2[ 'globalAddress' ] = t[ 'newSrcAddress' ]
               else:
                  if 'newDstAddress' in t:
                     t2[ 'globalAddress' ] = t[ 'newDstAddress' ]
            if 'srcAddress' in t:
               t2[ 'sourceAddress' ] = t[ 'srcAddress' ]
            if 'dstAddress' in t:
               t2[ 'destinationAddress' ] = t[ 'dstAddress' ]
            if 'interface' in t:
               t2[ 'interface' ] = t[ 'interface' ]
            details = t.get( 'details' )
            if details:
               d2 = {}
               d2[ 'vrfName' ] = details[ 'vrfName' ]
               d2[ 'packetCount' ] = details[ 'packetCount' ]
               d2[ 'packetReplyCount' ] = details[ 'packetReplyCount' ]
               protocol = t.get( 'protocol' )
               if protocol:
                  d2[ 'protocolNumber' ] = protocol
                  d2[ 'protocol' ] = ipProtoMap.get( protocol, str( protocol ) ) \
                                     if protocol else '-'
               d2[ 'origin' ] = details[ 'origin' ]
               t2[ 'details' ] = d2
            newDict[ 'translations' ].append( t2 )
         return newDict
      return dictRepr

   def render( self ):
      if self.detailedView and self.hasComments:
         fmt = showIpNatFmtStrDetailComment
         print( fmt %
            ( "Source IP", "Destination IP", "Translated Src IP",
              "Translated Dst IP", "Proto", "Tgt", "Type", "Dir",
              "Interface/Profile", "VRF", "Est", "Origin", "Packets",
              "Packets Reply", "Comment" ) )
         print( "-" * 217 )
      elif self.detailedView:
         fmt = showIpNatFmtStrDetail
         print( fmt %
            ( "Source IP", "Destination IP", "Translated Src IP",
              "Translated Dst IP", "Proto", "Tgt", "Type", "Dir",
              "Interface/Profile", "VRF", "Est", "Origin", "Packets",
              "Packets Reply" ) )
         print( "-" * 186 )
      elif self.hasComments:
         fmt = showIpNatFmtStrComment
         print( fmt %
            ( "Source IP", "Destination IP", "Translated Src IP",
              "Translated Dst IP", "Proto", "Tgt", "Type", "Dir",
              "Interface/Profile", "Comment" ) )
         print( "-" * 157 )
      else:
         fmt = showIpNatFmtStr
         print( fmt %
            ( "Source IP", "Destination IP", "Translated Src IP",
              "Translated Dst IP", "Proto", "Tgt", "Type", "Dir",
              "Interface/Profile" ) )
         print( "-" * 126 )

      for t in self.translations:
         srcAddress = t.srcAddress.formatStr() if t.srcAddress else '-'
         dstAddress = t.dstAddress.formatStr() if t.dstAddress else '-'
         newSrcAddress = t.newSrcAddress.formatStr() if t.newSrcAddress else '-'
         newDstAddress = t.newDstAddress.formatStr() if t.newDstAddress else '-'
         target = t.target if t.target else '-'
         direction = t.direction if t.direction else '-'
         intfName = t.interface if t.interface else '-'
         protocol = ipProtoMap.get( t.protocol, str( t.protocol ) ) if t.protocol \
                    else '-'
         args = ( srcAddress, dstAddress, newSrcAddress, newDstAddress,
                  protocol, target, t.natType, direction, intfName )
         if t.details:
            tcp = str( int( t.details.established ) ) if t.details.established \
                  is not None else '-'
            vrfName = t.details.vrfName
            if t.details.sourceVrfName:
               vrfName += ( ', source %s' % t.details.sourceVrfName )
            origin = t.details.origin if t.details.origin else 'Local'
            args = args + ( vrfName, tcp, origin, str( t.details.packetCount ),
                            str( t.details.packetReplyCount ) )
         if self.hasComments:
            args += ( t.comment, )
         print( fmt % args )

#-------------------------------------------------------------------
# 'show ip nat translation twice'
#-------------------------------------------------------------------

showIpTwiceNatFmtStr = "%-21.21s %-21.21s %-21.21s %-21.21s"
showIpTwiceNatFmtStrPendingDetail = showIpTwiceNatFmtStr + \
      " %-20s %-10s %-5s %-5s %6s"
showIpTwiceNatFmtStrDetail = showIpTwiceNatFmtStrPendingDetail + " %20s %20s"

showIpTwiceNatFmtStrComment = showIpTwiceNatFmtStr + " %-30s"
showIpTwiceNatFmtStrPendingDetailComment = showIpTwiceNatFmtStrPendingDetail + \
                                           " %-30s"
showIpTwiceNatFmtStrDetailComment = showIpTwiceNatFmtStrDetail + " %-30s"

class TwiceNatIpTranslations( Model ):
   __revision__ = 2
   class Translation( Model ):
      class Details( Model ):
         interface = Str( help="Interface/Profile", optional=True )
         vrfName = Str( help="VRF name" )
         group = Int( help="Twice NAT Group" )
         protocolNumber = Int( help="IP Protocol Number, if specified",
                               optional=True )
         protocol = Enum( values=( 'TCP', 'UDP', 'IGMP', 'ICMP' ), optional=True,
                          help="IP Protocol, if specified" )
         packetCount = Int( help="Number of packets translated from the "
                            "original IP address" )
         packetReplyCount = Int( help="Number of packets translated to the "
                                 "original IP address" )
         origin = Str( help="Discovered locally or advertised from sync peer" )


      srcIpAndPortOld = Submodel( valueType=IpAddrAndPort,
                                  help="The original source IP address "
                                       "and port of the packet." )
      dstIpAndPortOld = Submodel( valueType=IpAddrAndPort,
                                  help="The original destination IP address "
                                       "and port of the packet." )
      srcIpAndPortNew = Submodel( valueType=IpAddrAndPort,
                                  help="The translated source IP address "
                                       "and port of the packet." )
      dstIpAndPortNew = Submodel( valueType=IpAddrAndPort,
                                  help="The translated destination IP address "
                                       "and port of the packet." )
      details = Submodel( valueType=Details, optional=True,
                          help="Details, if requested" )
      comment = Str( help="Comment on translation rule", optional=True )

   translations = GeneratorList( valueType=Translation,
                                 help="List of Twice NAT translations active "
                                      "on this interface." )

   detailedView = Bool( help="includes additional details if True" )

   hasComments = Bool( help="True if any translations shown have comments",
                       default=False )

   pendingView = Bool( help="Excludes unnecessary columns if True" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # Modified type of optional field 'interface' in revision 2
         for translation in dictRepr[ 'translations' ]:
            details = translation.get( 'details' )
            if details:
               details.pop( 'interface', None )
      return dictRepr

   def render( self ):
      if self.detailedView:
         if self.pendingView:
            if self.hasComments:
               fmt = showIpTwiceNatFmtStrPendingDetailComment
               print( fmt  % 
                      ( "Source IP", "Destination IP", "Translated Src IP",
                        "Translated Dst IP", "Interface/Profile", "VRF",
                        "Group", "Proto", "Origin", "Comment" ) )
               print( "-" * 167 )
            else:
               fmt = showIpTwiceNatFmtStrPendingDetail
               print( showIpTwiceNatFmtStrPendingDetail %
                      ( "Source IP", "Destination IP",
                        "Translated Src IP", "Translated Dst IP",
                        "Interface/Profile", "VRF", "Group", "Proto", "Origin" ) )
               print( "-" * 137 )
         else:
            if self.hasComments:
               fmt = showIpTwiceNatFmtStrDetailComment
               print( fmt %
                      ( "Source IP", "Destination IP",
                        "Translated Src IP", "Translated Dst IP",
                        "Interface/Profile", "VRF", "Group", "Proto",
                        "Origin", "Packets", "Packets Reply", "Comment" ) )
               print( "-" * 209 )
            else:
               fmt = showIpTwiceNatFmtStrDetail
               print( showIpTwiceNatFmtStrDetail %
                      ( "Source IP", "Destination IP",
                        "Translated Src IP", "Translated Dst IP",
                        "Interface/Profile", "VRF", "Group", "Proto",
                        "Origin", "Packets", "Packets Reply" ) )
               print( "-" * 179 )
      else:
         if self.hasComments:
            fmt = showIpTwiceNatFmtStrComment
            print( fmt %
                   ( "Source IP", "Destination IP",
                     "Translated Src IP", "Translated Dst IP", "Comment" ) )
            print( "-" * 114 )
         else:
            fmt = showIpTwiceNatFmtStr
            print( showIpTwiceNatFmtStr %
                   ( "Source IP", "Destination IP",
                     "Translated Src IP", "Translated Dst IP" ) )
            print( "-" * 83 )

      for t in self.translations:

         srcIpAndPortOld = t.srcIpAndPortOld.formatStr()
         dstIpAndPortOld = t.dstIpAndPortOld.formatStr()
         srcIpAndPortNew = t.srcIpAndPortNew.formatStr()
         dstIpAndPortNew = t.dstIpAndPortNew.formatStr()
         args = ( srcIpAndPortOld, dstIpAndPortOld,
                  srcIpAndPortNew, dstIpAndPortNew )

         if t.details:
            intfName = t.details.interface if t.details.interface else '-'
            vrfName = t.details.vrfName
            group = t.details.group
            if t.details.protocol:
               proto = t.details.protocol
            elif t.details.protocolNumber:
               proto = str( t.details.protocolNumber )
            else:
               proto = '-'
            origin = t.details.origin if t.details.origin else 'Local'
            args = args + ( intfName, vrfName, str( group ), proto, origin )
            if not self.pendingView:
               args = args + ( str( t.details.packetCount ),
                               str( t.details.packetReplyCount ) )

         if self.hasComments:
            args += ( t.comment, )
         print( fmt % args )

#-------------------------------------------------------------------
# 'show ip nat translation address-only mapping'
#-------------------------------------------------------------------

class NatAddrOnlyTranslations( DeferredModel ):
   __revision__ = 2

   class SrcIpTranslations( Model ):
      class TranslationList( Model ):
         class Translation( Model ):
            sourceAddress = Submodel( valueType=IpAddrAndPort, optional=True,
                                      help="The original source IP address "
                                      "and port/mask of the packet." )
            destinationAddress = Submodel( valueType=IpAddrAndPort, optional=True,
                                           help="The destination IP address "
                                           "and port/mask of the packet." )
            globalAddress = Submodel( valueType=IpAddrAndPort,
                                      help="The translated IP address "
                                      "and port/mask of the packet." )
            natType = Enum( values=( "STAT", "DYN" ),
                            help="Static NAT or Dynamic NAT" )

            target = Enum( values=( "SRC", "DST" ),
                           help="Source NAT or destination NAT" )

            protocol = Enum( values=( 'TCP', 'UDP', 'IGMP', 'ICMP', "-" ),
                             help="IP Protocol" )

            vrfName = Str( help="VRF name" )
         translations = List( valueType=Translation,
                              help="List of active NAT translations" )

      srcIpTranslations = Dict( keyType=Ip4Address, valueType=TranslationList,
                                help="Translations for a source Ip" )

   addrOnlyTranslations = Dict( keyType=str, valueType=SrcIpTranslations,
                           help="List of active Nat translations per interface" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # Modified keyType of field 'addrOnlyTranslations' in revision 2
         dictRepr[ 'addrOnlyTranslations' ] = {}
      return dictRepr

#--------------------------------------------------------------------------------
# show ip nat translation <address-only | full-cone> flows
#--------------------------------------------------------------------------------

showIpNatFlowFmtStr = "%-21.21s %-21.21s %-21.21s %-5s %-3s %-3s %-20s %-15s"

class NatTranslationFlows( Model ):
   class IntfTranslationFlows( Model ):
      class TranslationFlows( Model ):
         class TranslationFlow( Model ):
            source = Submodel(
               valueType=IpAddrAndPort,
               help="Packet flow source IP address and L4 port" )
            destination = Submodel(
               valueType=IpAddrAndPort,
               help="Packet flow destination IP address and L4 port" )
            protocol = Enum( values=( 'TCP', 'UDP', 'IGMP', 'ICMP', '-' ),
                             help="IP Protocol" )
            target = Enum( values=( "SRC", "DST" ),
                           help="Source NAT or destination NAT" )
            established = Bool( help="Network connection established" )
            sourceVrfName = Str( help="Source VRF name", optional=True )

         globalAddr = Submodel( valueType=IpAddrAndPort, help="NAT address" )
         flows = GeneratorList( valueType=TranslationFlow,
                                help="List of active NAT translations" )

      vrfName = Str( help="VRF name" )
      translationFlows = List(
         valueType=TranslationFlows,
         help="Flows using an address-only or full-cone NAT Translation" )

   intfTranslationFlows = Dict(
      keyType=str, valueType=IntfTranslationFlows,
      help="List of active NAT translation flows per interface" )

   def render( self ):
      fmt = showIpNatFlowFmtStr
      print( fmt %
         ( "Source IP", "Destination IP", "Translated IP",
           "Proto", "Tgt", "Est", "Interface/Profile", "VRF" ) )
      print( "-" * 116 )

      for intf, intfTranslations in self.intfTranslationFlows.items():
         for translation in intfTranslations.translationFlows:
            for flow in translation.flows:
               estStr = " Y " if flow.established else " N "
               vrfName = intfTranslations.vrfName
               if flow.sourceVrfName:
                  vrfName += ( ', source %s' % flow.sourceVrfName )
               if translation.globalAddr.port == 0:
                  # If there is no port specified in the global address, take it
                  # from the source of the flow. However, do not modify the original
                  # translation object as that would affect every flow of the
                  # mapping, which may have a different source port. This is
                  # especially important for ICMP full cone entries
                  globalAddr = IpAddrAndPort( ip=translation.globalAddr.ip,
                                              port=flow.source.port )
               else:
                  globalAddr = translation.globalAddr
               if flow.target == 'SRC':
                  args = ( flow.source.formatStr(), flow.destination.formatStr(),
                           globalAddr.formatStr(), flow.protocol,
                           flow.target, estStr, intf, vrfName )
               else:
                  args = ( flow.destination.formatStr(), globalAddr.formatStr(),
                           flow.source.formatStr(), flow.protocol,
                           flow.target, estStr, intf, vrfName )
               print( fmt % args )

#-------------------------------------------------------------------
# 'show ip nat translation vrf' (VRF hopping mode only)
#-------------------------------------------------------------------

showIpVrfNatFmtStrDetail = \
                  "%-15.15s %-15.15s %-15.15s %-6s %-15s %-15s %15s %15s"
showIpVrfNatFmtStrPendingDetail = \
                  "%-15.15s %-15.15s %-15.15s %-6s %-15s %-15s"
showIpVrfNatFmtStr = "%-15.15s  %-15.15s  %-15.15s  %-10s  %-17s"

class VrfNatIpTranslations( Model ):
   class Translation( Model ):
      class Details( Model ):
         outsideVrf = Str( help="Outside vrf name", optional=True )
         packetCount = Int( help="Number of packets translated from the "
                            "original IP address" )
         packetReplyCount = Int( help="Number of packets translated to the "
                                 "original IP address" )

      insideVrf = Str( help="Inside vrf name." )

      sourceAddress = Ip4Address( optional=True,
                                  help="The original source IP address "
                                  "and port/mask of the packet." )

      destinationAddress = Ip4Address( optional=True,
                                       help="The destination IP address "
                                       "and port/mask of the packet." )

      globalAddress = Ip4Address( help="The translated IP address "
                                  "and port/mask of the packet." )

      target = Enum( values=( "SRC", "DST" ),
                     help="Source NAT or destination NAT" )

      details = Submodel( valueType=Details, optional=True,
                          help="Details, if requested" )

   translations = List( valueType=Translation,
                        help="List of Vrf NAT translations active "
                             "on this interface." )

   detailedView = Bool( help="includes additional details if True" )

   pendingView = Bool( help="Excludes unnecessary columns if True" )

   def render( self ):
      if self.detailedView:
         if self.pendingView:
            fmt = showIpVrfNatFmtStrPendingDetail
            print( showIpVrfNatFmtStrPendingDetail %
                  ( "Source IP", "Destination IP",
                    "Translated IP", "Target", "Inside VRF",
                    "Outside VRF" ) )
            print( "-" * 86 )
         else:
            fmt = showIpVrfNatFmtStrDetail
            print( showIpVrfNatFmtStrDetail %
                  ( "Source IP", "Destination IP",
                    "Translated IP", "Target", "Inside VRF",
                    "Outside VRF", "Packets", "Packets Reply" ) )
            print( "-" * 118 )
      else:
         fmt = showIpVrfNatFmtStr
         print( showIpVrfNatFmtStr %
               ( "Source IP", "Destination IP",
                 "Translated IP", "Target", "Inside VRF" ) )
         print( "-" * 80 )

      for t in self.translations:
         srcAddress = t.sourceAddress if t.sourceAddress else '-'
         dstAddress = t.destinationAddress if t.destinationAddress else '-'
         args = ( srcAddress, dstAddress, t.globalAddress, t.target, t.insideVrf )
         if self.detailedView:
            args += ( t.details.outsideVrf, )
            if not self.pendingView:
               args += ( str( t.details.packetCount ),
                         str( t.details.packetReplyCount ) )
         print( fmt % args )

#-------------------------------------------------------------------
# 'show ip nat translation summary'
#-------------------------------------------------------------------

showIpNatTransSummaryHdr1 = ( "Interface/Profile", "Static", "Static",
                              "Dynamic", "Dynamic", "Dynamic",
                              "Dyn.twice", "Dyn.twice", "Dyn.twice",
                              "St.twice", "St.twice" )
showIpNatTransSummaryHdr2 = ( "", "Software", "Hardware",
                              "Software", "Hardware", "Kernel",
                              "Software", "Hardware", "Kernel",
                              "Software", "Hardware" )
showIpNatTransSummaryHdr3 = "-" * 119
showIpNatSummaryFormat = "%-20s" + " %-9s" * ( len( showIpNatTransSummaryHdr1 ) - 1 )

class NatTranslationSummary( Model ):
   ''' CAPI model for show ip nat translation summary [ address <ip> ] '''
   __revision__ = 3

   class TranslationSummary( Model ):
      ''' CAPI model for NAT summary '''
      numStaticSwConnections = Int( help="Number of Static NAT connections "
                                    "in Software", default=0 )
      numStaticHwConnections = Int( help="Number of Static NAT connections "
                                    "in Hardware", default=0 )
      numDynSwConnections = Int( help="Number of Dynamic NAT connections "
                                 "in Software", default=0 )
      numDynHwConnections = Int( help="Number of Dynamic NAT connections "
                                 "in Hardware", default=0 )
      numKernelConnections = Int( help="Number of Dynamic NAT connections "
                                  "in Kernel", default=0 )
      numDynTwiceSwConnections = Int( help="Number of Dynamic twice NAT connections "
                                 "in Software", default=0 )
      numDynTwiceHwConnections = Int( help="Number of Dynamic twice NAT connections "
                                 "in Hardware", default=0 )
      numDynTwiceKernelConnections = Int( help="Number of Dynamic twice NAT "
                                          "connections in Kernel", default=0 )
      numTwiceSwConnections = Int( help="Number of Twice NAT connections "
                                   "in Software", default=0 )
      numTwiceHwConnections = Int( help="Number of Twice NAT connections "
                                   "in Hardware", default=0 )

   translations = Dict( keyType=str, valueType=TranslationSummary,
                        help="Map of NAT translation summary per interface." )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # Modified keyType of field 'translations' in revision 2
         dictRepr[ 'translations' ] = {}
      return dictRepr

   def render( self ):
      print( showIpNatSummaryFormat % showIpNatTransSummaryHdr1 )
      print( showIpNatSummaryFormat % showIpNatTransSummaryHdr2 )
      print( showIpNatTransSummaryHdr3 )
      for intfName in sorted( self.translations ):
         t = self.translations[ intfName ]
         args = ( intfName,
                  t.numStaticSwConnections,
                  t.numStaticHwConnections,
                  t.numDynSwConnections,
                  t.numDynHwConnections,
                  t.numKernelConnections,
                  t.numDynTwiceSwConnections,
                  t.numDynTwiceHwConnections,
                  t.numDynTwiceKernelConnections,
                  t.numTwiceSwConnections,
                  t.numTwiceHwConnections )
         print( showIpNatSummaryFormat % args )

class NatTranslationDynamicSummary( Model ):
   ''' CAPI model for show ip nat translation dynamic summary'''

   class DynamicTranslationSummary( Model ):
      ''' CAPI model for dynamic NAT summary '''
      numDynSwConnections = Int( help="Number of Dynamic NAT connections "
                                 "in Software", default=0 )
      numDynHwConnections = Int( help="Number of Dynamic NAT connections "
                                 "in Hardware", default=0 )
      numFcSwConnections = Int( help="Number of Fullcone Dynamic NAT connections "
                                  "in Software", default=0 )
      numFcHwConnections = Int( help="Number of Fullcone Dynamic NAT connections "
                                 "in Hardware", default=0 )
      numSymSwConnections = Int( help="Number of Symmetric Dynamic NAT connections "
                                 "in Software", default=0 )
      numSymHwConnections = Int( help="Number of Symmetric Dynamic NAT connections "
                                 "in Hardware", default=0 )

   interfaces = Dict( keyType=str, valueType=DynamicTranslationSummary,
                  help="Map of Dynamic NAT translation summary per interface." )

   def render( self ):
      headings = ( 'Profile/Interface' , 
            ( 'Full Cone', ( 'Software', 'Hardware' ) ),
            ( 'Symmetric', ( 'Software', 'Hardware' ) ),
            ( 'Total', ( 'Software', 'Hardware' ) ) )

      table = TableOutput.createTable( headings )
      fmtStr = TableOutput.Format( justify='left', minWidth=14, 
            maxWidth=18, wrap=True )
      fmtStr.noPadLeftIs( True )
      fmtStr.padLimitIs( True )
      fmtNum = TableOutput.Format( justify='right', minWidth=8, 
            maxWidth=8, wrap=True )
      fmtNum.noPadLeftIs( True )
      fmtNum.padLimitIs( True )
      table.formatColumns( fmtStr, fmtNum, fmtNum, fmtNum, fmtNum, fmtNum, fmtNum )

      for intfName in sortIntf( self.interfaces ):
         t = self.interfaces[ intfName ]

         table.newRow( intfName,
                  t.numFcSwConnections,
                  t.numFcHwConnections,
                  t.numSymSwConnections,
                  t.numSymHwConnections,
                  t.numDynSwConnections,
                  t.numDynHwConnections )
      print( table.output() )

#-------------------------------------------------------------------
# 'show ip nat acl'
#-------------------------------------------------------------------

class NatAclList( Model ):
   class NatAcl( Model ):
      class AclRule( Model ):
         seqNo = Int( help="Sequence number of the rule" )
         valid = Bool( help="True if NAT can use this rule to filter translations" )
         description = Str( help="Textual description of the ACL rule if valid, "
                            "or explanation of why the rule doesn't apply if it "
                            "isn't" )

      class NatAclInfo( Model ):
         fullCone = Bool( help='Access list used in full cone NAT' )
         addrOnly = Bool( help='Access list used in address only NAT' )
         poolId = Int( help='NAT pool ID associated with the access list' )

      aclName = Str( help="The name of the specified ACL" )
      aclRules = List( valueType=AclRule,
                       help="Rules that belongs to this ACL, "
                       "ordered by the sequence number of the rule" )
      interfaces = List( valueType=Interface,
                      help="List of interfaces using this ACL for NAT" )
      profiles = List( valueType=str,
                      help='List of profiles using this ACL for NAT' )
      aclInfo = Dict( keyType=str, valueType=NatAclInfo,
                      help='NAT configuration for the ACL' )

      def render( self ):
         print( "acl %s" % self.aclName )
         for rule in self.aclRules:
            if not rule.valid:
               print( "\tSeq no %d: %s" % ( rule.seqNo, rule.description ) )
            else:
               print( '\t' + rule.description )

         if self.interfaces:
            print( '    Interfaces using the access list for NAT:' )
            for intf in self.interfaces:
               info = ""
               aclInfo = self.aclInfo.get( intf )
               if aclInfo:
                  info = " (pool: " + str( aclInfo.poolId )
                  if not aclInfo.poolId:
                     info += ", overload"
                  if aclInfo.fullCone:
                     info += ", full cone"
                  if aclInfo.addrOnly:
                     info += ", address only"
                  info += ")"
               print( '\t%s' % ( intf + info ) )

         if self.profiles:
            print( '    Profiles using the access list for NAT:' )
            for profile in self.profiles:
               info = ""
               aclInfo = self.aclInfo.get( profile )
               if aclInfo:
                  info = " (pool: " + str( aclInfo.poolId )
                  if not aclInfo.poolId:
                     info += ", overload"
                  if aclInfo.fullCone:
                     info += ", full cone"
                  if aclInfo.addrOnly:
                     info += ", address only"
                  info += ")"
               print( '\t%s' % ( profile + info ) )

   aclList = List( valueType=NatAcl,
                   help="List of the ACLs to display" )

   def render( self ):
      for acl in self.aclList:
         acl.render()

#-------------------------------------------------------------------
# 'show ip nat counters'
#-------------------------------------------------------------------

class NatVrfCounters( Model ):
   __revision__ = 2

   class NatCounters( Model ):

      newConnectionEvents = Int(
                     help="Number of Conntrack new connection events" )
      updateConnectionEvents = Int(
                     help="Number of Conntrack update connection events" )
      deleteConnectionEvents = Int(
                     help="Number of Conntrack delete connection events" )
      updateConnectionErrors = Int(
                     help="Conntrack update connection errors" )
      deleteConnectionErrors = Int(
                     help="Conntrack delete connection errors" )
      kernelSyncRequests = Int(
                     help="Conntrack table sync requests" )
      kernelSyncRequestErrors = Int(
                     help="Conntrack table sync request errors" )
      kernelSyncInProgress = Int(
                     help="Conntrack table sync already in progress count" )
      iptablesWriteErrors = Int(
                     help="Kernel iptables write errors" )
      netlinkSyncEagainErrors = Int(
                     help="Number of Netlink Sync EAGAIN errors" )
      netlinkSyncNoBufferErrors = Int(
                     help="Number of Netlink Sync ENOBUFS errors" )
      netlinkSyncReceiveErrors = Int(
                     help="Number of Netlink Sync receive errors" )
      netlinkSyncMsgErrors = Int(
                     help="Number of Netlink Sync messages with errors" )
      netlinkSyncCreateTxErrors = Int(
                     help="Number of Netlink Sync creation event errors" )
      netlinkSyncUpdateTxErrors = Int(
                     help="Number of Netlink Sync update event errors" )
      netlinkSyncDeleteTxErrors = Int(
                     help="Number of Netlink Sync deletion event errors" )
      netlinkSyncFCConflictErrors = Int(
                     help="Number of Netlink Sync full cone conflict event errors" )
      netlinkReceiveErrors = Int(
                     help="Number of Netlink receive errors" )
      netlinkNoBufferErrors = Int(
                     help="Number of Netlink no space errors" )
      netlinkTruncatedMessages = Int(
                     help="Number of Netlink truncated messages" )
      netlinkSeqNumMismatches = Int(
                     help="Number of Netlink sequence number mismatches" )
      netlinkSeqNumRangeMismatches = Int(
                     help="Number of Netlink sequence number out of range errors" )
      netlinkBadMessageLengthErrors = Int(
                     help="Number of Netlink bad message length errors" )
      netlinkNoSpaceErrors = Int(
                     help="Number of Netlink no space in message errors" )
      netlinkDumpDoneErrors = Int(
                     help="Number of Netlink unexpected dump done messages" )
      netlinkDumpIgnoreTcpConnections = Int(
                     help="Number of Netlink TCP connections ignored in dump" )
      netlinkMsgErrors = Int( help="Number of Netlink msg errors", optional=True )
      enobufDumpCnt = Int(
                     help="Number of Netlink no space errors during kernel sync",
                     optional=True )
      enobufMsgCnt = Int(
                     help="Number of Netlink no space errors during kernel update",
                     optional=True )
      eagainDumpCnt = Int(
                     help="Number of Netlink no data errors during kernel sync",
                     optional=True )
      eagainRecvCnt = Int(
                     help="Number of Netlink no data errors during kernel update",
                     optional=True )
      maxMsgCntInRecvBuf = Int(
                     help="Max number of messages processed in one buffer",
                     optional=True )

      nlAdd5s = Int( help="5s NAT entries learning rate", optional=True )
      nlAdd30s = Int( help="30s NAT entries learning rate", optional=True )
      nlAdd60s = Int( help="60s NAT entries learning rate", optional=True )
      nlAdd300s = Int( help="300s NAT entries learning rate", optional=True )
      nlDel5s = Int( help="5s NAT entries deletion rate", optional=True )
      nlDel30s = Int( help="30s NAT entries deletion rate", optional=True )
      nlDel60s = Int( help="60s NAT entries deletion rate", optional=True )
      nlDel300s = Int( help="300s NAT entries deletion rate", optional=True )

   vrfs = Dict( keyType=str, valueType=NatCounters, help='NAT counters of each VRF' )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         if DEFAULT_VRF in dictRepr[ 'vrfs' ]:
            return dictRepr[ 'vrfs' ][ DEFAULT_VRF ]
         else:
            # set every counter to 0 as default
            return { attr: 0 for attr in self.NatCounters.__attributes__ }
      return dictRepr

   def render( self ):
      for vrf, counters in self.vrfs.items():
         output = (
            ( "Description", "Value" ),
            ( "Conntrack new connection events",
              counters.newConnectionEvents ),
            ( "Conntrack update connection events",
              counters.updateConnectionEvents ),
            ( "Conntrack delete connection events",
              counters.deleteConnectionEvents ),
            ( "Conntrack update connection errors",
              counters.updateConnectionErrors ),
            ( "Conntrack delete connection errors",
              counters.deleteConnectionErrors ),
            ( "Conntrack table sync requests", counters.kernelSyncRequests ),
            ( "Conntrack table sync request errors",
              counters.kernelSyncRequestErrors ),
            ( "Conntrack table sync already in progress count",
              counters.kernelSyncInProgress ),
            ( "Kernel iptables write errors", counters.iptablesWriteErrors ),
            ( "Netlink EAGAIN during connection dump loop", counters.eagainDumpCnt ),
            ( "Netlink EAGAIN during regular message recv loop",
              counters.eagainRecvCnt ),
            ( "Netlink ENOBUFS errors", counters.netlinkNoBufferErrors ),
            ( "Netlink ENOBUFS during dump recv loop", counters.enobufDumpCnt ),
            ( "Netlink ENOBUFS during regular msg recv loop",
              counters.enobufMsgCnt ),
            ( "Netlink receive errors", counters.netlinkReceiveErrors ),
            ( "Netlink truncated messages", counters.netlinkTruncatedMessages ),
            ( "Netlink sequence number mismatches",
              counters.netlinkSeqNumMismatches ),
            ( "Netlink sequence number out of range errors",
              counters.netlinkSeqNumRangeMismatches ),
            ( "Netlink bad message length errors",
              counters.netlinkBadMessageLengthErrors ),
            ( "Netlink no space in message errors",
              counters.netlinkNoSpaceErrors ),
            ( "Netlink unexpected dump done messages",
              counters.netlinkDumpDoneErrors ),
            ( "Netlink TCP connections ignored in dump",
              counters.netlinkDumpIgnoreTcpConnections ),
            ( "Netlink message errors", counters.netlinkMsgErrors ),
            ( "Netlink sync EAGAIN errors",
              counters.netlinkSyncEagainErrors ),
            ( "Netlink sync ENOBUFS errors",
              counters.netlinkSyncNoBufferErrors ),
            ( "Netlink sync receive errors",
              counters.netlinkSyncReceiveErrors ),
            ( "Netlink sync message errors",
              counters.netlinkSyncMsgErrors ),
            ( "Netlink sync create event errors",
              counters.netlinkSyncCreateTxErrors ),
            ( "Netlink sync update event errors",
              counters.netlinkSyncUpdateTxErrors ),
            ( "Netlink sync delete event errors",
              counters.netlinkSyncDeleteTxErrors ),

            ( "  5s Netlink learning rate",
              counters.nlAdd5s if counters.nlAdd5s is not None else "N/A" ),
            ( " 30s Netlink learning rate",
              counters.nlAdd30s if counters.nlAdd30s is not None else "N/A" ),
            ( " 60s Netlink learning rate",
              counters.nlAdd60s if counters.nlAdd60s is not None else "N/A" ),
            ( "300s Netlink learning rate",
              counters.nlAdd300s if counters.nlAdd300s is not None else "N/A" ),
            ( "  5s Netlink deletion rate",
              counters.nlDel5s if counters.nlDel5s is not None else "N/A" ),
            ( " 30s Netlink deletion rate",
              counters.nlDel30s if counters.nlDel30s is not None else "N/A" ),
            ( " 60s Netlink deletion rate",
              counters.nlDel60s if counters.nlDel60s is not None else "N/A" ),
            ( "300s Netlink deletion rate",
              counters.nlDel300s if counters.nlDel300s is not None else "N/A" ),

            ( "Max number of messages processed in one buffer",
              counters.maxMsgCntInRecvBuf ),

            ( "Netlink sync full cone conflict event errors",
              counters.netlinkSyncFCConflictErrors ),
         )
         print( "VRF: %s" % vrf )
         for out in output:
            print( "%-55s %-s" % ( out[ 0 ], out[ 1 ] ) )
         print( '\n' )


#-------------------------------------------------------------------
# 'show ip nat dynamic counters'
#-------------------------------------------------------------------
class NatDynamicCountersProtoModel( Model ):
   source = Int( help="Number of source NAT packets" )
   destination = Int( help="Number of destination NAT packets" )
   hairpin = Int( help="Number of hairpin NAT packets" )

class NatDynamicCountersProfileModel( Model ):
   tcp = Submodel( NatDynamicCountersProtoModel, help="TCP NAT counters" )
   udp = Submodel( NatDynamicCountersProtoModel, help="UDP NAT counters" )
   other = Submodel( NatDynamicCountersProtoModel,
                     help="Non-TCP, non-UDP NAT counters" )

class NatDynamicCountersModel( Model ):
   profiles = Dict( keyType=str, valueType=NatDynamicCountersProfileModel,
                    help='NAT profiles, keyed by profile name' )
   
   def render( self ):
      headers = ( "Protocol", "Source NAT", "Destination NAT", "Hairpin NAT" )

      for profileName, profile in self.profiles.items():
         print( "Profile: ", profileName )
         
         table = TableOutput.createTable( headers )
         fmtProto = TableOutput.Format( justify='left' )
         fmtProto.padLimitIs( True )
         fmtCount = TableOutput.Format( justify='right' )
         fmtCount.padLimitIs( True )

         table.formatColumns( fmtProto, fmtCount, fmtCount, fmtCount )
         table.newRow( 'TCP', profile.tcp.source, profile.tcp.destination,
               profile.tcp.hairpin )
         table.newRow( 'UDP', profile.udp.source, profile.udp.destination,
               profile.udp.hairpin )
         table.newRow( 'Other', profile.other.source, profile.other.destination,
               profile.other.hairpin )

         print( table.output() )

#-------------------------------------------------------------------
# 'show ip nat max-entries'
#-------------------------------------------------------------------

class NatHostConnectionLimit( DeferredModel ):
   hostIp = Ip4Address( help="Host IP address" )
   connectionLimit = Int( help="Connection limit per host" )
   numberOfConnections = Int( help="Number of connections per host",
                              optional=True )
   numberOfLocalConnections = Int( help=
         "Number of connections per host in Nat Sync agent",
         optional=True )
   lowMarkPerc = Int( help="Below this percentage, connection drops are"
                      " disabled" )

class NatPoolConnectionLimit( DeferredModel ):
   poolName = Str( help="Pool name" )
   connectionLimit = Int( help="Connection limit per pool" )
   numberOfConnections = Int( help="Number of connections per pool",
                              optional=True )
   numberOfLocalConnections = Int( help=
         "Number of connections per pool in Nat Sync agent",
         optional=True )
   lowMarkPerc = Int( help="Below this percentage, connection drops are"
                      " disabled" )


class NatConnectionLimit( DeferredModel ):

   connectionLimit = Int( help="Global connection limit" )
   connectionLimitLowMarkPerc = Int( help="Below this percentage, connection"
                                          " drops are disabled" )
   connectionLimitLowMark = Int( help="Below this count, connection"
                                      " drops are disabled" )
   totalNumberOfConnections = Int( help="Active connections in Nat agent" )
   totalNumberOfLocalConnections = Int( help="Local connections in Nat Sync agent" )

   symmetricConnLimit = Int( help="Symmetric connection limit" )
   symmetricConnLimitLowMarkPerc = Int( help="Below this percentage, symmetric"
                                             " connection drops are disabled" )
   symmetricConnLimitLowMark = Int( help="Below this count, symmetric connection"
                                         " drops are disabled" )
   numberOfSymmetricConnections = Int( help="Active symmetric connections"
                                            " in Nat agent" )
   
   fullConeConnLimit = Int( help="Full-cone connection limit" )
   fullConeConnLimitLowMarkPerc = Int( help="Below this percentage, full-cone"
                                             " connection drops are disabled" )
   fullConeConnLimitLowMark = Int( help="Below this count, full-cone connection"
                                         " drops are disabled" )
   numberOfFullConeConnections = Int( help="Active full-cone connections"
                                           " in Nat agent" )

   allHostsConnLimit = Int( help="Hosts' connection limit", optional=True )
   allHostsLowMarkPerc = Int( help="Below this percentage, connection drops are"
                              " disabled", optional=True )
   allPoolsLowMarkPerc = Int( help="Below this percentage, connection drops are"
                              " disabled", optional=True )
   hostConnLimits = List( valueType=NatHostConnectionLimit,
                          optional=True,
                          help="List of host's connection limit" )

   poolConnLimits = List( valueType=NatPoolConnectionLimit,
                          optional=True,
                          help="List of pool's connection limit" )
#-------------------------------------------------------------------
# show ip nat translations rates
#-------------------------------------------------------------------

class NatTranslationRate( Model ):
   timeLen = Int( help="Length of rates measure" )
   numAdd = Int( help="Number of NAT entries added" )
   numDel = Int( help="Number of NAT entries deleted" )

class NatTranslationRates( Model ):
   rate = List( valueType=NatTranslationRate,
                help="NAT translations measured in a specific interval" )

class NatTranslationRatesAll( Model ):
   rates = Dict( keyType=str, valueType=NatTranslationRates,
                 help='NAT translation learning rates' )

   def render( self ):
      for rateType, rates in self.rates.items():
         print( "Learning rates for %s:" % rateType )
         print( "-" * 50 )
         for rate in rates.rate:
            print( "Time: %3ds, add rate: %7.1f, del rate: %7.1f" % ( rate.timeLen,
                  rate.numAdd / rate.timeLen, rate.numDel / rate.timeLen ) )
         print( '' )

#-------------------------------------------------------------------
# 'show ip nat synchronization'
#-------------------------------------------------------------------

class Version( Model ):
   minVersion = Int( help='Minimum version number' )
   maxVersion = Int( help='Maximum version number' )

def renderUtc( value ):
   if not value:
      return 'n/a'
   return '%s' % datetime.fromtimestamp( int( value ) )

class NatSyncFwdToPeer( Model ):
   peerIpAddr = Ip4Address( help="Peer's IP address" )
   peerIntf = Interface( help="Local interface connected to peer" )
   peerResolved = Bool( help="Peer's reachability status" )

class NatSynchronizationModel( Model ):

   peerIpAddr = Ip4Address( help='Peer\'s IP address' )
   connPort = Int( help='Port used for the connection' )
   connSrcIp = Ip4Address( help='Connection source Ip' )
   connState = Enum( values=STATUS_STATE.attributes,
         help='State of the connection with peer' )
   kernelDevName = Str( help='Kernel device name' )
   localIntf = Interface( help='Local interface' )
   connEstablishedTime = Float( help='Peer Connection Establishment time' )
   connAttempts = Int( help='Number of attempts at synchronizing with peer' )
   natVersion = Submodel( valueType=Version, help='Nat version' )
   natVerCompatible = Bool( help='Peer\'s version is Compatible' )
   shutdown = Bool( help='Shutdown State'
         'False: peer connection is administratively down, '
         'True: peer connection is administratively up' )
   statusMountState = Enum( values=MOUNT_STATE.attributes,
         help='Mount state of peer\'s status' )
   versionMountState = Enum( values=MOUNT_STATE.attributes,
         help='Mount state of peer\'s version')
   portRangeMode = Enum( values=PORT_RANGE_MODE.attributes,
         help='Port range mode used in the system' )
   fwdToPeer = Dict( keyType=str, valueType=NatSyncFwdToPeer,
               help='NAT profiles\' fowardTopeer status' )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         oldRepr = dictRepr
         oldRepr[ 'recoverMountState' ] = oldRepr[ 'statusMountState' ]
         oldRepr[ 'rebootMountState' ] = oldRepr[ 'statusMountState' ]
         return oldRepr
      return dictRepr

   def render( self ):
      print( '%s: %-s' % ( 'Peer',
                            self.peerIpAddr if self.peerIpAddr else 'n/a' ) )
      # pylint: disable-next=bad-string-format-type
      print( '%s: %-d' % ( 'Connection port',
                           self.connPort if self.connPort else 'n/a' ) )
      print( '%s: %-s' % ( 'Connection source',
                           self.connSrcIp if self.connSrcIp else 'n/a' ) )
      print( '%s: %-s' % ( 'Connection state', self.connState ) )
      print( '%s: %-s' % ( 'Kernel interface',
                            self.kernelDevName.capitalize() if self.kernelDevName
                            else 'n/a' ) )
      print( '%s: %-s' % ( 'Local interface',
                            self.localIntf.stringValue if self.localIntf
                            else 'n/a' ) )
      print( '%s: %-s' % ( 'Established time',
                            renderUtc( self.connEstablishedTime ) ) )
      # pylint: disable-next=bad-string-format-type
      print( '%s: %-d' % ( 'Connection attempts', self.connAttempts ) )
      print( '%s: %-s' % ( 'Oldest supported version',
                            self.natVersion.minVersion ) )
      print( '%s: %-s' % ( 'Newest supported version',
                            self.natVersion.maxVersion ) )
      print( '%s: %-s' % ( 'Version compatible',
                           str( self.natVerCompatible ).lower() ) )
      print( '%s: %-s' % ( 'Shutdown state', str( self.shutdown ).lower() ) )
      print( '%s: %-s' % ( 'Status state',
                            self.statusMountState.lstrip( 'mount' ).lower() ) )
      print( '%s: %-s' % ( 'Version state',
                            self.versionMountState.lstrip( 'mount' ).lower() ) )
      print( '%s: %-s' % ( 'Port range mode',
                            self.portRangeMode.lstrip( 'portRange' ).lower() ) )
      
      print( '' )
      for profile, peerInfo  in sorted( self.fwdToPeer.items() ):
         print( 'Profile: %s' % profile )
         print( 'Translation miss peer: %s' % peerInfo.peerIpAddr )
         print( 'Translation miss interface: %s' %
                peerInfo.peerIntf.stringValue )
         print( 'Translation miss peer status: %s\n' % ( 'reachable' if
                peerInfo.peerResolved else 'unreachable' ) )


class NatSynchronizationPeer( Model ):
   __revision__ = 2

   peerIpAddr = Ip4Address( help='Peer\'s IP address')
   connPort = Int( help='Port used for the connection' )
   connSrcIp = Ip4Address( help='Connection source Ip' )
   kernelDevName = Str( help='Kernel device name' )
   localIntf = Interface( help='Local interface' )
   connEstablishedTime = Float( help='Peer Connection Establishment time' )
   connAttempts = Int( help='Number of attempts at synchronizing with peer' )
   natVersion = Submodel( valueType=Version, help='Nat version' )
   natVerCompatible = Bool( help='Peer\'s version is Compatible' )
   connState = Enum( values=STATUS_STATE.attributes,
         help='State of the connection with peer' )
   shutdown = Bool( help='Shutdown State'
         'False: peer connection is administratively down'
         'True: peer connection is administratively up' )
   statusMountState = Enum( values=MOUNT_STATE.attributes,
         help='Mount state of peer\'s status' )
   versionMountState = Enum( values=MOUNT_STATE.attributes,
         help='Mount state of peer\'s version')
   portRangeMode = Enum( values=PORT_RANGE_MODE.attributes,
         help='Port range mode used in the system' )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         oldRepr = dictRepr
         oldRepr[ 'recoverMountState' ] = oldRepr[ 'statusMountState' ]
         oldRepr[ 'rebootMountState' ] = oldRepr[ 'statusMountState' ]
         return oldRepr
      return dictRepr

   def render( self ):
      print( '%-55s : %-s' % ( 'Description', 'Value' ) )
      print( '%-55s : %-s' % ( 'Peer', self.peerIpAddr ) )
      # pylint: disable-next=bad-string-format-type
      print( '%-55s : %-d' % ( 'Connection Port', self.connPort ) )
      print( '%-55s : %-s' % ( 'Connection Source', self.connSrcIp ) )
      print( '%-55s : %-s' % ( 'Kernel Interface', self.kernelDevName ) )
      print( '%-55s : %-s' % ( 'Local Interface', self.localIntf.stringValue ) )
      print( '%-55s : %-s' % ( 'Established Time',
            renderUtc( self.connEstablishedTime ) ) )
      # pylint: disable-next=bad-string-format-type
      print( '%-55s : %-d' % ( 'Connection Attempts', self.connAttempts ) )
      print( '%-55s : %-s' % ( 'Oldest Supported Version',
            self.natVersion.minVersion ) )
      print( '%-55s : %-s' % ( 'Newest Supported Version',
            self.natVersion.maxVersion ) )
      print( '%-55s : %-s' % ( 'Version Compatible', self.natVerCompatible ) )
      print( '%-55s : %-s' % ( 'Connection State', self.connState ) )
      print( '%-55s : %-s' % ( 'Shutdown State', self.shutdown ) )
      print( '%-55s : %-s' % ( 'Status Mount State', self.statusMountState ) )
      print( '%-55s : %-s' % ( 'Version Mount State', self.versionMountState ) )
      print( '%-55s : %-s' % ( 'Port range mode', self.portRangeMode.lstrip(
         'portRange' ) ) )

def timeElapsedString( utcTime ):
   """Returns a human readable string that represents the time elapsed since the
   (UTC) time passed in"""

   currentTime = Tac.utcNow()
   delta = currentTime - utcTime
   td = timedelta( seconds=int( delta ) )

   if td > timedelta( days=1 ) :
      # If time elapsed is > 1 day display the time in XdXXh format
      return str( td.days ) + " days " + str( td.seconds // 3600 ) + " hours"
   else:
      # Display time elapsed in HH:MM:SS format
      return str( td )

showIpNatAdverTransFmt = '%-21.21s %-21.21s %-21.21s %-3s %-4s %-20s'
showIpNatAdverTransDetailFmt = showIpNatAdverTransFmt + ' %-5s %-11s %-20s'

class NatSynchronizationAdvertisedTranslations( Model ):
   __revision__ = 3
   class Translation( Model ):
      class Details( Model ):
         protocolNumber = Int( help="IP Protocol Number, if specified",
                               optional=True )
         protocol = Enum( values=( 'TCP', 'UDP', 'IGMP', 'ICMP' ), optional=True,
                          help="IP Protocol, if specified" )
         packetCount = Int( help="Number of packets translated from the "
                            "original IP address" )
         packetReplyCount = Int( help="Number of packets translated to the "
                                 "original IP address" )
         tcpState = Enum( values=( "NONE", "SYN_SENT", "SYN_RECV", "ESTABLISHED", \
               "FIN_WAIT", "CLOSE_WAIT", "LAST_ACK", "TIME_WAIT", "CLOSE", \
               "LISTEN", "MAX", "IGNORE", "RETRANS", "UNACK", "TIMEOUT_MAX" ),
               help="State of the TCP connection", optional=True )
         lastActivityTime = Float( help='Last time data packets matching this \
               translation were seen' )
         vrfName = Str( help="VRF name" )
         sourceVrfName = Str( help="Source VRF name", optional=True )

      interface = Str( help="Interface or NAT profile name", optional=True )

      sourceAddress = Submodel( valueType=IpAddr, optional=True,
                                help="The original source IP address "
                                "and port/mask of the packet." )
      destinationAddress = Submodel( valueType=IpAddr, optional=True,
                                     help="The destination IP address "
                                     "and port/mask of the packet." )
      globalAddress = Submodel( valueType=IpAddr,
                                help="The translated IP address "
                                "and port/mask of the packet." )
      natType = Enum( values=( "STAT", "DYN", "FC", "AO", "D-TW" ),
                      help="Static NAT or Dynamic NAT" )

      target = Enum( values=( "SRC", "DST" ),
                     help="Source NAT or destination NAT" )

      details = Submodel( valueType=Details, optional=True,
                          help="Details, if requested" )

   translations = GeneratorList( valueType=Translation,
                        help="List of NAT translations active on this interface." )

   detailedView = Bool( help="includes additional details if True" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # Modified type of optional field 'interface' in revision 2
         for translation in dictRepr[ 'translations' ]:
            translation.pop( 'interface', None )
      if revision <= 2:
         # Added fields 'vrfName' and 'sourceVrfName' in revision 3
         for translation in dictRepr[ 'translations' ]:
            if 'details' in translation:
               translation[ 'details' ].pop( 'vrfName', None )
               translation[ 'details' ].pop( 'sourceVrfName', None )
      return dictRepr

   def render( self ):
      if self.detailedView:
         fmt = showIpNatAdverTransDetailFmt
         print( showIpNatAdverTransDetailFmt %
               ( "Source IP", "Destination IP", "Translated IP", "TGT",
                 "Type", "Interface/Profile", "Proto", "TCP State", "VRF" ) )
         print( "-" * 133 )
      else:
         fmt = showIpNatAdverTransFmt
         print( showIpNatAdverTransFmt %
               ( "Source IP", "Destination IP", "Translated IP",
                 "TGT", "Type", "Interface/Profile" ) )
         print( "-" * 92 )

      for t in self.translations:
         srcAddress = t.sourceAddress.formatStr() if t.sourceAddress else '-'
         dstAddress = t.destinationAddress.formatStr() \
                      if t.destinationAddress else '-'
         intfName = t.interface if t.interface else '-'
         args = ( srcAddress,
                  dstAddress,
                  t.globalAddress.formatStr(),
                  t.target, t.natType, intfName )

         if t.details:
            if t.details.protocol:
               proto = t.details.protocol
            elif t.details.protocolNumber:
               proto = str( t.details.protocolNumber )
            else:
               proto = '-'
            vrfName = t.details.vrfName
            if t.details.sourceVrfName:
               vrfName += ', source %s' % t.details.sourceVrfName
            args = args + ( proto, str( t.details.tcpState ), vrfName )
         print( fmt % args )

#-------------------------------------------------------------------
# 'show ip nat service-list'
#-------------------------------------------------------------------

class NatServiceListCommandModel( Model ):
   protocol = Str( help="protocol for ALG" )

class NatServiceListModel( Model ):
   class NatServiceList( Model ):
      serviceListName = Str( help="The name of the specified service-list" )
      serviceCmdList = List( valueType=NatServiceListCommandModel,
                             help="List of service commands" )

   serviceLists = List( valueType=NatServiceList, help="List of service-list" )

   def render( self ):
      for sl in self.serviceLists:
         print( "service-list %s" % sl.serviceListName )
         for cmd in sl.serviceCmdList:
            print( '    service %s' % cmd.protocol )

#-------------------------------------------------------------------
# 'show ip nat flow match'
#-------------------------------------------------------------------

class NatFlowAddress( Model ):
   srcAddr = Submodel( valueType=IpAddrAndMask,
                       help='The source IP address to match on' )
   dstAddr = Submodel( valueType=IpAddrAndMask,
                       help='The destination IP address to match on' )
   protocol = Int( help='The protocol of the match' )
   srcPortStart = Int( help='The start of the source port range of the match' )
   srcPortEnd = Int( help='The end of the source port range of the match' )
   dstPortStart = Int( help='The start of the destination port range of the match' )
   dstPortEnd = Int( help='The end of the destination port range of the match' )
   priority = Int( help='The priority of the match' )

class NatFlowMatch( Model ):
   flow = Submodel( valueType=NatFlowAddress,
                    help='NAT flow match fields' )
   ucGroup = Int( help="Hardware unicast replication group", optional=True )
   mcGroup = Int( help="Hardware multicast replication group", optional=True )

class NatFlowMatchPolicy( Model ):
   matches = Dict( keyType=str, valueType=NatFlowMatch,
                   help='Matches for this policy, keyed by match name' )

showFlowMatchHdr = ( "Policy", "Match", "Source", "Destination", "SrcPort",
                     "DstPort", "Proto", "Prio" )
showFlowMatchHdrLine = "-" * 110
showFlowMatchFmt = "%-18s %-18s %-18s %-18s %-11s %-11s %-5s %-4s"

showFlowMatchHwHdr = showFlowMatchHdr + ( "UcGroup", "McGroup" )
showFlowMatchHwHdrLine = "-" * 130
showFlowMatchHwFmt = showFlowMatchFmt + " %-7s %-7s"

class NatFlowMatchCmdModel( Model ):
   hardware = Bool( help="NAT flow hardware state" )
   policies = Dict( keyType=str, valueType=NatFlowMatchPolicy,
                    help='NAT flow policies, keyed by name' )

   def render( self ):
      hardware = self.hardware
      hdr = False
      for policyName, policy in sorted( self.policies.items() ):
         for matchName, match in sorted( policy.matches.items() ):
            flow = match.flow
            src = flow.srcAddr.formatStr() if flow.srcAddr.ip else  "-"
            dst = flow.dstAddr.formatStr() if flow.dstAddr.ip else  "-"
            if flow.srcPortStart:
               srcPort = "-" + str( flow.srcPortEnd ) \
                         if flow.srcPortEnd != flow.srcPortStart else ""
               srcPort = str( flow.srcPortStart ) + srcPort
            else:
               srcPort = "-"
            if flow.dstPortStart:
               dstPort = "-" + str( flow.dstPortEnd ) \
                         if flow.dstPortEnd != flow.dstPortStart else ""
               dstPort = str( flow.dstPortStart ) + dstPort
            else:
               dstPort = "-"
            proto = ipProtoMap.get( flow.protocol, str( flow.protocol ) ) \
                    if flow.protocol else "-"
            args = ( policyName, matchName, src, dst, srcPort, dstPort,
                     proto, flow.priority )
            if not hdr:
               if hardware:
                  print( showFlowMatchHwHdrLine )
                  print( showFlowMatchHwFmt % showFlowMatchHwHdr )
                  print( showFlowMatchHwHdrLine )
               else:
                  print( showFlowMatchHdrLine )
                  print( showFlowMatchFmt % showFlowMatchHdr )
                  print( showFlowMatchHdrLine )
               hdr = True
            if hardware:
               args += ( match.ucGroup, match.mcGroup, )
               print( showFlowMatchHwFmt % args )
            else:
               print( showFlowMatchFmt % args )
      print()

#-------------------------------------------------------------------
# 'show ip nat flow action'
#-------------------------------------------------------------------

class NatFlowIntf( Model ):
   mac = MacAddress( help='The destination MAC address for the output' )
   vlanId = Int( help='The VLAN ID for the output' )

class NatFlowAction( Model ):
   natSrc = Submodel( valueType=IpAddrAndPort,
                      help='The source IP address and port for the NAT '
                      'translation' )
   natDst = Submodel( valueType=IpAddrAndPort,
                      help='The destination IP address and port for the NAT '
                      'translation' )
   intfs = Dict( keyType=Interface, valueType=NatFlowIntf,
                 help='Flow outputs, keyed by interface' )
   flowType = Str( help="NAT flow type", optional=True )

class NatFlowMatchActions( Model ):
   actions = Dict( keyType=str, valueType=NatFlowAction,
                   help='Actions for this match, keyed by action name' )

class NatFlowActionPolicy( Model ):
   matches = Dict( keyType=str, valueType=NatFlowMatchActions,
                   help='Matches for this policy, keyed by match name' )
   profiles = List( valueType=str,
                    help='List of profiles using this NAT flow policy',
                    optional=True )

showFlowActionHdr = ( "Policy", "Match", "Action", "Source", "Destination",
                      "SPort", "DPort", "Intf", "Vlan", "Mac" )
showFlowActionHdrLine = "-" * 125
showFlowActionFmt = "%-18s %-18s %-18s %-14s %-14s %-5s %-5s %-7s %-4s %-17s"

class NatFlowActionCmdModel( Model ):
   policies = Dict( keyType=str, valueType=NatFlowActionPolicy,
                    help='NAT flow policies, keyed by name' )

   def render( self ):
      hdr = False
      for policyName, policy in sorted( self.policies.items() ):
         for matchName, match in sorted( policy.matches.items() ):
            for actionName, action in sorted( match.actions.items() ):
               if action.natSrc.ip:
                  src = action.natSrc.ip
                  sport = str( action.natSrc.port ) if action.natSrc.port else "-"
               else:
                  src = "-"
                  sport = "-"
               if action.natDst.ip:
                  dst = action.natDst.ip
                  dport = str( action.natDst.port ) if action.natDst.port else "-"
               else:
                  dst = "-"
                  dport = "-"
               if not hdr:
                  print( showFlowActionHdrLine )
                  print( showFlowActionFmt % showFlowActionHdr )
                  print( showFlowActionHdrLine )
                  hdr = True
               for intfName, intf in sorted( action.intfs.items() ):
                  shortName = Tac.Value( "Arnet::IntfId", intfName ).shortName
                  vlan = intf.vlanId if intf.vlanId else "-"
                  mac = intf.mac if intf.mac else "-"
                  args = ( policyName, matchName, actionName, src, dst, sport, dport,
                           shortName, vlan, mac )
                  print( showFlowActionFmt % args )
               if not action.intfs:
                  shortName = "-"
                  vlan = "-"
                  mac = "-"
                  args = ( policyName, matchName, actionName, src, dst, sport, dport,
                           shortName, vlan, mac )
                  print( showFlowActionFmt % args )
      if hdr:
         print()

#-------------------------------------------------------------------
# 'show ip nat flow profile'
#-------------------------------------------------------------------

class NatFlowMatchInfo( Model ):
   groupUc = Int( help="Multicast replication group used for unicast traffic" )
   groupMc = Int( help="Multicast replication group used for multicast traffic" )

class NatFlowPolicyInfo( Model ):
   match = Dict( keyType=str, valueType=NatFlowMatchInfo,
                 help='NAT flow match details, keyed by match name',
                 optional=True )

class NatFlowProfilePolicy( Model ):
   policies = Dict( keyType=str, valueType=NatFlowPolicyInfo,
                    help='NAT flow policy details, keyed by policy name' )

showFlowProfileHdr = ( "Profile", "Policy" )
showFlowProfileFmt = "%-20s %-20s"
showFlowProfileHdrLine = "-" * 90
showFlowProfileHdrHw = ( "Profile", "Policy", "Match", "Groups" )
showFlowProfileFmtHw = "%-20s %-20s %-20s %-6s"

class NatFlowProfileCmdModel( Model ):
   hardware = Bool( help="NAT flow hardware state" )
   profiles = Dict( keyType=str, valueType=NatFlowProfilePolicy,
                    help='NAT flow profiles, keyed by name' )

   def render( self ):
      hdr = False
      for profile, policyModel in sorted( self.profiles.items() ):
         for policy, policyInfo in sorted( policyModel.policies.items() ):
            if not hdr:
               print( showFlowProfileHdrLine )
               if self.hardware:
                  print( showFlowProfileFmtHw % showFlowProfileHdrHw )
               else:
                  print( showFlowProfileFmt % showFlowProfileHdr )
               print( showFlowProfileHdrLine )
               hdr = True

            if self.hardware:
               for match, matchInfo in sorted( policyInfo.match.items() ):
                  groupUc = matchInfo.groupUc
                  groupMc = matchInfo.groupMc
                  args = ( profile, policy, match, str( groupUc ) )
                  fmt = showFlowProfileFmtHw
                  if groupUc != groupMc:
                     fmt += " %-6s"
                     args += ( str( groupMc ), )
                  print( fmt % args )
            else:
               args = ( profile, policy )
               print( showFlowProfileFmt % args )
      print()

#-------------------------------------------------------------------
# 'show ip nat profile'
#-------------------------------------------------------------------

class NatProfileModel( Model ):
   class NatProfile( Model ):
      profileName = Str( help="The name of the specified NAT profile" )
      profileId = Int( help="The numeric ID assigned to each NAT profile" )
      vrfName = Str( help="The VRF associated with each NAT profile" )
      intfList = Dict( keyType=Interface, valueType=int,
                       help="List of interfaces configured with the NAT profile",
                       optional=True )

   profiles = List( valueType=NatProfile, help="List of profiles" )

   def render( self ):
      print( "%3s %-20s %-20s %-20s %4s" % (
         "Id", "Profile", "Vrf", "Interface", "Vlan" ) )
      profileIds = {}
      for p in self.profiles:
         profileIds[ p.profileId ] = p
      for pId in sorted( profileIds ):
         p = profileIds[ pId ]
         hdr = False
         for key, value in p.intfList.items():
            vlanStr = str( value ) if value != 0 else "-"
            if hdr:
               print( "                                              %-20s %4s" % (
                  key, vlanStr ) )
            else:
               print( "%3s %-20s %-20s %-20s %4s" % (
                  pId, p.profileName, p.vrfName, key, vlanStr ) )
               hdr = True
         if not hdr:
            print( "%3s %-20s %-20s" % ( pId, p.profileName, p.vrfName ) )

#-------------------------------------------------------------------
# 'show ip nat logging'
#-------------------------------------------------------------------

class NatLoggingModel( Model ):
   enabled = Bool( help="NAT event logging is enabled" )

   def render( self ):
      state = "enabled" if self.enabled else "disabled"
      print( "NAT event logging: %s" % ( state ) )

#-------------------------------------------------------------------
# 'show ip nat pool [ synchronization ]'
#-------------------------------------------------------------------

showIpNatPoolFormat = "%-20s %7s %-15s %-15s %6s %10s %8s"
showIpNatPoolSplitFormat = "%-20s %7s %-15s %-15s %6s %10s %8s %16s %14s"
showIpNatPoolHdr = "-" * 90

class NatPoolModel( Model ):
   class NatPool( Model ):
      class NatRange( Model ):
         startIp = Ip4Address( help="NAT start IP address" )
         endIp = Ip4Address( help="NAT end IP address" )
         startPort = Int( help="NAT start port number" )
         endPort = Int( help="NAT end port number" )
         localStartPort = Int( help="NAT sync port split start", optional=True )
         localEndPort = Int( help="NAT sync port split end", optional=True )

      natRanges = List( valueType=NatRange, help="List of NAT pool ranges" )
      prefixLen = Int( help="Length of the prefix" )
      poolId = Int( help="NAT pool ID" )
      portOnly = Bool( help="NAT Port only pool" )

   poolName = Str( help="Name of the NAT pool to show", optional=True )
   natPools = Dict( keyType=str, valueType=NatPool,
                    help="Map of NAT pools keyed by their name",
                    optional=True )
   showSplit = Bool( help="Show local NAT sync port range split", optional=True )

   def render( self ):
      if self.showSplit:
         print( showIpNatPoolSplitFormat % (
            "Pool", "Pool ID", "Start IP", "End IP", "Prefix", "Start Port",
            "End Port", "Port Split Start", "Port Split End" ) )
         print( showIpNatPoolSplitFormat % ( "-" * 20, "-" * 7, "-" * 15, "-" * 15,
               "-" * 6, "-" * 10, "-" * 8, "-" * 16, "-" * 14 ) )
      else:
         print( showIpNatPoolFormat % ( "Pool", "Pool ID", "Start IP", "End IP",
               "Prefix", "Start Port", "End Port" ) )
         print( showIpNatPoolFormat % ( "-" * 20, "-" * 7, "-" * 15, "-" * 15,
               "-" * 6, "-" * 10, "-" * 8 ) )
      for poolName, pool in self.natPools.items():
         if not pool.natRanges:
            prefixLenStr = pool.prefixLen if not pool.portOnly else '-'
            print( showIpNatPoolFormat % ( poolName, pool.poolId, '-', '-',
                   prefixLenStr, '-', '-' ) )

         for pr in pool.natRanges:
            if self.showSplit:
               print( showIpNatPoolSplitFormat % ( poolName,
                        pool.poolId,
                        pr.startIp if not pool.portOnly else '-',
                        pr.endIp if not pool.portOnly else '-',
                        pool.prefixLen if not pool.portOnly else '-',
                        pr.startPort if pr.startPort else '-',
                        pr.endPort if pr.endPort else '-',
                        pr.localStartPort, pr.localEndPort ) )
            else:
               print( showIpNatPoolFormat % ( poolName,
                        pool.poolId,
                        pr.startIp if not pool.portOnly else '-',
                        pr.endIp if not pool.portOnly else '-',
                        pool.prefixLen if not pool.portOnly else '-',
                        pr.startPort if pr.startPort else '-',
                        pr.endPort if pr.endPort else '-' ) )

#-------------------------------------------------------------------
# 'show ip nat translation [vrf <VRF>] connection limit drop
#-------------------------------------------------------------------
class NatTranslationConnectionDropped( Model ):

   class DropTable( Model ):
      class DropCounter( Model ):
         counter = Int( help="Counter" )
         srcIp = Submodel( valueType=IpAddr, help="Source IP" )
         dstIp = Submodel( valueType=IpAddr, help="Destination IP" )

      vrfName = Str( help="VRF name" )
      counters = List( valueType=DropCounter, help="Counters" )

      def renderTable( self ):
         print( "VRF:", self.vrfName )

         # create and format table
         headings = ( 'Source IP', 'Destination IP', 'Packets' )
         table = TableOutput.createTable( headings )
         fmtIp = TableOutput.Format( justify='left' )
         fmtIp.padLimitIs( True )
         fmtCounter = TableOutput.Format( justify='right' )
         table.formatColumns( fmtIp, fmtIp, fmtCounter )

         for cnt in self.counters:
            table.newRow( cnt.srcIp.formatStr(), cnt.dstIp.formatStr(),
                  cnt.counter )
         print( table.output() )

   vrfs = Dict( keyType=str, valueType=DropTable,
             help="Connection drop counters, keyed by VRF name")

   class PoolDropCounter( Model ):
      poolName = Str( help="Pool name" )
      vrfName = Str( help="VRF name" )
      counter = Int( help="Counter" )

   pools = List( valueType=PoolDropCounter,
         help="Connection drop counters per pool" )

   class DropRuleCounter( Model ):
      counter = Int( help="Counter" )

   rules = Dict( keyType=str, valueType=DropRuleCounter,
             help="Connection drop counters, keyed by rule")

   def printPoolCounters( self ):
      # create and format table
      headings = ( 'Pool', 'Vrf', 'Packets' )
      table = TableOutput.createTable( headings )
      fmtL = TableOutput.Format( justify='left' )
      fmtL.padLimitIs( True )
      fmtCounter = TableOutput.Format( justify='right' )
      table.formatColumns( fmtL, fmtL, fmtCounter )
      for cnt in sorted( self.pools ):
         table.newRow( cnt.poolName, cnt.vrfName, cnt.counter )
      print( table.output() )

   def printGlobalCounters( self ):
      # create and format table
      headings = ( 'Drop Rule', 'Packets' )
      table = TableOutput.createTable( headings )
      fmtL = TableOutput.Format( justify='left' )
      fmtL.padLimitIs( True )
      fmtCounter = TableOutput.Format( justify='right' )
      table.formatColumns( fmtL, fmtCounter )
      for rule, cnt in self.rules.items():
         table.newRow( rule, cnt.counter )
      print( table.output() )


   def render( self ):
      for table in self.vrfs.values():
         table.renderTable()
      if self.pools:
         self.printPoolCounters()
      if self.rules:
         self.printGlobalCounters()
