#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
import CliCommand
from CliMatcher import KeywordMatcher, IntegerMatcher
from CliPlugin import NatCli
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.NatCli import (
   nodeNat,
   nodePool,
   matcherPoolName,
   matcherPrefixValue,
   natConnectionLimitSupported,
   natPortOnlySupported
)
from CliToken.Ip import ipMatcherForConfig

matcherStartIp = IpAddrMatcher( helpdesc='First address' )
matcherEndIp = IpAddrMatcher( helpdesc='Last IP address in range' )
matcherPrefixLength = KeywordMatcher( 'prefix-length',
                                      helpdesc='Length of the pool prefixes' )
matcherNetmask = KeywordMatcher( 'netmask', helpdesc='Mask' )
matcherNetmaskValue = IpAddrMatcher( helpdesc="Subnet's mask value" )
matcherPortOnly = CliCommand.guardedKeyword( 'port-only', 'Port only pool',
                                             natPortOnlySupported )
matcherStartPort = IntegerMatcher( 1, 2 ** 16 - 1, helpdesc='Start Port' )
matcherEndPort = IntegerMatcher( 1, 2 ** 16 - 1, helpdesc='End Port' )
matcherUtilizationThreshold = KeywordMatcher( 'threshold',
                        helpdesc='Set NAT pool utilization threshold value' )
matcherUtilizationThresholdValue = IntegerMatcher( 1, 100,
                        helpdesc='NAT Pool utilization threshold percentage value' )
matcherUtilizationAction = KeywordMatcher( 'action',
                        helpdesc='Set NAT pool utilization action' )
matcherUtilizationActionLog = KeywordMatcher( 'log',
                        helpdesc='Set NAT pool utilization log action' )

nodeUtilization = CliCommand.guardedKeyword( 'utilization',
                                             'NAT pool utilization configuration',
                                             natConnectionLimitSupported )

#--------------------------------------------------------------------------------
# ip nat pool POOLNAME STARTIP ENDIP
# ( ( prefix-length PREFIXLEN ) | ( netmask MASK ) ) [ STARTPORT ENDPORT ]
#--------------------------------------------------------------------------------
class IpNatPoolIpRangeCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip nat pool POOLNAME STARTIP ENDIP '
              '( ( prefix-length PREFIXLEN ) | ( netmask MASK ) ) '
              '[ STARTPORT ENDPORT ]' )
   data = {
      'ip' : ipMatcherForConfig,
      'nat' : nodeNat,
      'pool' : nodePool,
      'POOLNAME' : matcherPoolName,
      'STARTIP' : matcherStartIp,
      'ENDIP' : matcherEndIp,
      'prefix-length' : matcherPrefixLength,
      'PREFIXLEN' : matcherPrefixValue,
      'netmask' : matcherNetmask,
      'MASK' : matcherNetmaskValue,
      'STARTPORT' : matcherStartPort,
      'ENDPORT' : matcherEndPort,
   }

   handler = NatCli.createIpNatPoolConfig

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatPoolIpRangeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip nat pool POOLNAME
# ( ( port-only ) | ( prefix-length PREFIXLEN ) | ( netmask MASK ) )
#--------------------------------------------------------------------------------
class IpNatPoolCmd( CliCommand.CliCommandClass ):
   syntax = '''ip nat pool POOLNAME 
               ( ( port-only )
               | ( prefix-length PREFIXLEN )
               | ( netmask MASK ) )'''
   noOrDefaultSyntax = 'ip nat pool POOLNAME ...'
   data = {
      'ip' : ipMatcherForConfig,
      'nat' : nodeNat,
      'pool' : nodePool,
      'POOLNAME' : matcherPoolName,
      'prefix-length' : matcherPrefixLength,
      'PREFIXLEN' : matcherPrefixValue,
      'netmask' : matcherNetmask,
      'MASK' : matcherNetmaskValue,
      'port-only' : matcherPortOnly,
   }

   handler = NatCli.gotoIpNatPoolConfigMode
   noOrDefaultHandler = NatCli.deleteIpNatPoolMode

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatPoolCmd )

#--------------------------------------------------------------------------------
# [ no | default ] range STARTIP ENDIP [ STARTPORT ENDPORT ]
#--------------------------------------------------------------------------------
class IpNatPoolAppendRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'range STARTIP ENDIP [ STARTPORT ENDPORT ]'
   noOrDefaultSyntax = syntax
   data = {
      'range' : 'IP address range',
      'STARTIP' : matcherStartIp,
      'ENDIP' : matcherEndIp,
      'STARTPORT' : matcherStartPort,
      'ENDPORT' : matcherEndPort,
   }

   handler = NatCli.NatPoolConfigMode.addPoolRange
   noOrDefaultHandler = NatCli.NatPoolConfigMode.noPoolRange

NatCli.NatPoolConfigMode.addCommandClass( IpNatPoolAppendRangeCmd )

#----------------------------------------------------------------------------------
# [no|default] utilization threshold <value> action log
#----------------------------------------------------------------------------------

class IpNatPoolUtilizationThresholdCmd( CliCommand.CliCommandClass ):
   syntax = ( 'utilization threshold VALUE action log' )
   noOrDefaultSyntax = ( 'utilization threshold ...' )
   data = {
      'utilization' : nodeUtilization,
      'threshold' : matcherUtilizationThreshold,
      'VALUE' : matcherUtilizationThresholdValue,
      'action' : matcherUtilizationAction,
      'log' : matcherUtilizationActionLog
   }

   handler = NatCli.NatPoolConfigMode.doSetIpNatPoolUtilizationThreshold
   noOrDefaultHandler = NatCli.NatPoolConfigMode.doSetIpNatPoolUtilizationThreshold

NatCli.NatPoolConfigMode.addCommandClass( IpNatPoolUtilizationThresholdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] port range STARTPORT ENDPORT
#--------------------------------------------------------------------------------
class IpNatPortOnlyPoolAppendRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'port range STARTPORT ENDPORT'
   noOrDefaultSyntax = syntax
   data = {
      'port' : 'Port',
      'range' : 'Port range',
      'STARTPORT' : matcherStartPort,
      'ENDPORT' : matcherEndPort,
   }

   handler = NatCli.NatPortOnlyPoolConfigMode.addPoolRange
   noOrDefaultHandler = NatCli.NatPortOnlyPoolConfigMode.noPoolRange

NatCli.NatPortOnlyPoolConfigMode.addCommandClass( IpNatPortOnlyPoolAppendRangeCmd )
