#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=ungrouped-imports

import BasicCli
import CliCommand
import CliMatcher
import ShowCommand
from CliToken.Ip import ipMatcherForShow
from CliPlugin.AclCli import userIpAclNameMatcher
from CliPlugin.VrfCli import VrfExprFactory, VrfNameExprFactory
from CliPlugin.IntfCli import Intf
from CliPlugin.NatModels import (
   NatProfileModel, NatAddrOnlyTranslations, NatTranslationSummary,
   NatConnectionLimit, TwiceNatIpTranslations, NatServiceListModel,
   NatSynchronizationModel, NatSynchronizationPeer, NatVrfCounters,
   NatSynchronizationAdvertisedTranslations,
   NatIpTranslations, VrfNatIpTranslations, NatHostConnectionLimit, NatAclList,
   NatPoolModel, NatLoggingModel, NatFlowMatchCmdModel, NatFlowActionCmdModel,
   NatFlowProfileCmdModel, NatTranslationFlows, NatTranslationConnectionDropped,
   NatTranslationRatesAll, NatDynamicCountersModel, NatTranslationDynamicSummary
)

import CliPlugin.NatCli as NatCli # pylint: disable=consider-using-from-import
from CliPlugin.NatCli import (
      NatAclExpression,
      matcherAddress,
      matcherInterface,
      matcherIpAddr,
      matcherPoolName,
      matcherProfile,
      matcherProfileName,
      matcherServiceListName,
      matcherSrcPort,
      matcherTarget,
      matcherFlowPolicyName,
      natAddrOnlyFlowSupported,
      natAddrOnlyKernelSupported,
      natAddrOnlySupported,
      natConnectionLimitSupported,
      natIntfStaticSupported,
      natIntfDynamicSupported,
      natInVrfSupported,
      natIpIntfConfigSupported,
      natIpProfileConfigSupported,
      natOutsideVrfSupported,
      natSupported,
      natVrfHopStaticSupported,
      natDynamicTwiceSupported,
      nodeCounters,
      nodeLogging,
      nodeFlow,
      nodePool,
      nodeServiceList,
      natUpnpPortMapSupported,
      dynamicNatCountersSupported,
)

matcherNat = CliMatcher.KeywordMatcher( 'nat', helpdesc='Show NAT information' )
matcherConfig = CliMatcher.KeywordMatcher( 'config', helpdesc='Configured state' )
matcherHardware = CliMatcher.KeywordMatcher( 'hardware', helpdesc='Hardware state' )
matcherPending = CliMatcher.KeywordMatcher( 'pending', helpdesc='Pending state' )
matcherDetail = CliMatcher.KeywordMatcher( 'detail', helpdesc='Verbose state' )
matcherMaxEntries = CliMatcher.KeywordMatcher( 'max-entries',
                                              helpdesc='Configure connection limit' )
matcherSynchronization = CliMatcher.KeywordMatcher( 'synchronization',
                                                    helpdesc='NAT synchronization' )
matcherTranslation = CliMatcher.KeywordMatcher( 'translation',
                                                helpdesc='NAT translation rule' )
matcherDynamicTwiceNat = CliMatcher.KeywordMatcher( 'dynamic-twice',
                                       helpdesc='Dynamic twice NAT translation' )
matcherFlowProfile = CliMatcher.KeywordMatcher( 'profile',
                                       helpdesc='NAT flow profiles configuration' )
matcherConnection = CliMatcher.KeywordMatcher( 'connection',
      helpdesc='NAT connection info' )
matcherLimit = CliMatcher.KeywordMatcher( 'limit',
      helpdesc='Limits' )
matcherDrop = CliMatcher.KeywordMatcher( 'drop',
      helpdesc='Drop counters' )
matcherUpnp = CliMatcher.KeywordMatcher( 'upnp', helpdesc='UPNP translation' )

nodeSynchronization = CliCommand.Node( matcher=matcherSynchronization,
                                       guard=natIntfDynamicSupported )
nodeTranslation = CliCommand.Node( matcher=matcherTranslation,
                                   guard=natSupported )
nodeTranslationDyn = CliCommand.Node( matcher=matcherTranslation,
                                      guard=natIntfDynamicSupported )
nodeFlowProfile = CliCommand.Node( matcher=matcherFlowProfile,
                                   guard=natIpProfileConfigSupported )
nodeMaxEntries = CliCommand.Node( matcher=matcherMaxEntries,
                                  guard=natConnectionLimitSupported )

tableOptions = {
   'hardware': ( 'Hardware state' ),
   'pending': ( 'Pending state' ),
}
tableEnumMatcher = CliCommand.SetEnumMatcher( tableOptions, maxMatches=1 )

def getKernelTableTypeOptions( mode, context ):
   options = {
      'hardware': ( 'Hardware state' ),
      'pending': ( 'Pending state' ),
   }
   if( NatCli.natHwCapabilities and
       NatCli.natHwCapabilities.natSupported and
       not NatCli.natHwCapabilities.natDpdkProcessingSupported ):
      options[ 'kernel' ] = 'Kernel state'
   return options
kernelTableEnumMatcher = CliCommand.SetEnumMatcher( getKernelTableTypeOptions,
                                                    maxMatches=1 )

#--------------------------------------------------------------------------------
# show ip nat acl [ interface INTF ] [ ACL_NAME ]
#--------------------------------------------------------------------------------
class IpNatAclCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat acl [ interface INTF ] [ ACL_NAME ]'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'acl': NatAclExpression,
      'interface': matcherInterface,
      'INTF': Intf.matcher,
      'ACL_NAME': userIpAclNameMatcher,
   }
   handler = NatCli.cmdShowIpNatAcl
   cliModel = NatAclList

BasicCli.addShowCommandClass( IpNatAclCmd )

#--------------------------------------------------------------------------------
# show ip nat counters [ VRF ]
#--------------------------------------------------------------------------------
class IpNatCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat counters [ VRF ]'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'counters': nodeCounters,
      'VRF': VrfExprFactory( helpdesc='Show NAT counters in a VRF',
                             guard=natInVrfSupported,
                             inclAllVrf=True ),
   }
   handler = NatCli.cmdShowIpNatCounters
   cliModel = NatVrfCounters

BasicCli.addShowCommandClass( IpNatCountersCmd )

#--------------------------------------------------------------------------------
# show ip nat logging
#--------------------------------------------------------------------------------
class IpNatLoggingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat logging'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'logging': nodeLogging,
   }
   handler = NatCli.cmdShowIpNatLogging
   cliModel = NatLoggingModel

BasicCli.addShowCommandClass( IpNatLoggingCmd )

#--------------------------------------------------------------------------------
# show ip nat flow match [policy POLICY] [config|hardware]
#--------------------------------------------------------------------------------
class IpNatFlowMatchCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip nat flow match [ policy POLICY ] [ config | hardware ]' )
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'flow': nodeFlow,
      'match' : 'Display only flow match fields',
      'policy': 'Display a specific policy',
      'POLICY': matcherFlowPolicyName,
      'config' : matcherConfig,
      'hardware' : matcherHardware,
   }
   handler = NatCli.cmdShowIpNatFlowMatch
   cliModel = NatFlowMatchCmdModel

BasicCli.addShowCommandClass( IpNatFlowMatchCmd )

#--------------------------------------------------------------------------------
# show ip nat flow action [policy POLICY] [config|hardware]
#--------------------------------------------------------------------------------
class IpNatFlowActionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip nat flow action [ policy POLICY ] [ config | hardware ]' )
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'flow': nodeFlow,
      'action': 'Display only flow action fields',
      'policy': 'Display a specific policy',
      'POLICY': matcherFlowPolicyName,
      'config' : matcherConfig,
      'hardware' : matcherHardware,
   }
   handler = NatCli.cmdShowIpNatFlowAction
   cliModel = NatFlowActionCmdModel

BasicCli.addShowCommandClass( IpNatFlowActionCmd )

#--------------------------------------------------------------------------------
# show ip nat flow profile [policy POLICY] [config|hardware]
#--------------------------------------------------------------------------------
class IpNatFlowProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip nat flow profile [ policy POLICY ] [ config | hardware ]' )
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'flow': nodeFlow,
      'profile': nodeFlowProfile,
      'policy': 'Display a specific policy',
      'POLICY': matcherFlowPolicyName,
      'config' : matcherConfig,
      'hardware' : matcherHardware,
   }
   handler = NatCli.cmdShowIpNatFlowProfile
   cliModel = NatFlowProfileCmdModel

BasicCli.addShowCommandClass( IpNatFlowProfileCmd )

#--------------------------------------------------------------------------------
# show ip nat pool [ POOL ]
#--------------------------------------------------------------------------------
class IpNatPoolCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat pool [ POOL ] [ synchronization ]'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'pool': nodePool,
      'POOL': matcherPoolName,
      'synchronization': CliMatcher.KeywordMatcher( 'synchronization',
         helpdesc='Show local NAT sync port range split' )
   }
   handler = NatCli.cmdShowIpNatPool
   cliModel = NatPoolModel

BasicCli.addShowCommandClass( IpNatPoolCmd )

#--------------------------------------------------------------------------------
# show ip nat profile [ PROFILE ]
#--------------------------------------------------------------------------------
class IpNatProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat profile [ PROFILE ]'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'profile': CliCommand.Node( matcherProfile,
                                  guard=natIpProfileConfigSupported ),
      'PROFILE': matcherProfileName,
   }
   handler = NatCli.cmdShowIpNatProfiles
   cliModel = NatProfileModel

BasicCli.addShowCommandClass( IpNatProfileCmd )

#--------------------------------------------------------------------------------
# show ip nat service-list [ SERVICE_LIST ]
#--------------------------------------------------------------------------------
class IpNatServiceListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat service-list [ SERVICE_LIST ]'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'service-list': nodeServiceList,
      'SERVICE_LIST': matcherServiceListName,
   }
   handler = NatCli.cmdShowIpNatServiceList
   cliModel = NatServiceListModel

BasicCli.addShowCommandClass( IpNatServiceListCmd )

#--------------------------------------------------------------------------------
# show ip nat synchronization
#--------------------------------------------------------------------------------
class IpNatSynchronizationCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat synchronization'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'synchronization': nodeSynchronization,
   }
   handler = NatCli.cmdShowIpNatSynchronization
   cliModel = NatSynchronizationModel

BasicCli.addShowCommandClass( IpNatSynchronizationCmd )

#--------------------------------------------------------------------------------
# show ip nat synchronization advertised-translations
#      [ { interface INTF } ] [ detail ]
#--------------------------------------------------------------------------------
class IpNatSynchronizationAdvertisedTranslationsCmd(
         ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip nat synchronization advertised-translations '
              '[ { interface INTF } ] [ detail ]' )
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'synchronization': nodeSynchronization,
      'advertised-translations': CliCommand.guardedKeyword(
            'advertised-translations', 'Advertised translations status',
            natIntfDynamicSupported ),
      'interface': CliCommand.singleNode( matcherInterface ),
      'INTF': Intf.matcher,
      'detail': matcherDetail,
   }
   handler = NatCli.cmdShowIpNatSynchronizationAdvertisedTranslations
   cliModel = NatSynchronizationAdvertisedTranslations

BasicCli.addShowCommandClass( IpNatSynchronizationAdvertisedTranslationsCmd )

#--------------------------------------------------------------------------------
# show ip nat synchronization discovered-translations
#      [ { interface INTF } ] [ detail ]
#--------------------------------------------------------------------------------
class IpNatSynchronizationDiscoveredTranslationsCmd(
         ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip nat synchronization discovered-translations '
              '[ { interface INTF } ] [ detail ]' )
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'synchronization': nodeSynchronization,
      'discovered-translations': CliCommand.guardedKeyword(
            'discovered-translations', 'Discovered translations status',
            natIntfDynamicSupported ),
      'interface': matcherInterface,
      'INTF': Intf.matcher,
      'detail': matcherDetail,
   }
   handler = NatCli.cmdShowIpNatSynchronizationDiscoveredTranslations
   cliModel = NatSynchronizationAdvertisedTranslations

BasicCli.addShowCommandClass( IpNatSynchronizationDiscoveredTranslationsCmd )

#--------------------------------------------------------------------------------
# show ip nat synchronization peer
#--------------------------------------------------------------------------------
class IpNatSynchronizationPeerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat synchronization peer'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'synchronization': nodeSynchronization,
      'peer': CliCommand.guardedKeyword( 'peer', 'Peer status',
                                         natIntfDynamicSupported ),
   }
   handler = NatCli.cmdShowIpNatSynchronizationPeer
   cliModel = NatSynchronizationPeer

BasicCli.addShowCommandClass( IpNatSynchronizationPeerCmd )

#--------------------------------------------------------------------------------
# show ip nat translation static
#  [ interface <INTF> | profile <PROFILE> | vrf <VRF> ] |
#  [ hardware | pending ] |
#  [ source | destination ] |
#  [ address <ADDRESS> [ <PORT> ] |
#  [ detail ]
#--------------------------------------------------------------------------------

intfOrVrfOrProfile = object()
vrfExprFactory = VrfExprFactory( helpdesc='VRF',
      guard=natInVrfSupported,
      maxMatches=1,
      sharedMatchObj=intfOrVrfOrProfile,
      inclAllVrf=True )

class IpNatTranslationStaticCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip nat translation STATIC_TYPE
               [ { ( ( interface INTF ) | ( profile PROFILE ) | ( VRF ) )
                 | TABLE
                 | TARGET
                 | ( address ADDRESS [ PORT ] )
                 | detail } ]'''
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslation,
      'STATIC_TYPE': CliCommand.singleNode(
         CliMatcher.KeywordMatcher( 'static', helpdesc='Static NAT translation' ),
         guard=natIntfStaticSupported ),
      'interface': CliCommand.singleNode( matcherInterface,
                                          sharedMatchObj=intfOrVrfOrProfile,
                                          guard=natIpIntfConfigSupported ),
      'INTF': CliCommand.singleNode( Intf.matcher ),
      'profile': CliCommand.singleNode( matcherProfile,
                                        sharedMatchObj=intfOrVrfOrProfile,
                                        guard=natIpProfileConfigSupported ),
      'PROFILE': CliCommand.singleNode( matcherProfileName ),
      'VRF': vrfExprFactory,
      'TABLE': tableEnumMatcher,
      'TARGET': CliCommand.singleNode( matcherTarget ),
      'address': CliCommand.singleNode( matcherAddress ),
      'ADDRESS': CliCommand.singleNode( matcherIpAddr ),
      'PORT': CliCommand.singleNode( matcherSrcPort ),
      'detail': CliCommand.singleNode( matcherDetail ),
   }
   handler = NatCli.cmdShowIpNatTranslations
   cliModel = NatIpTranslations

BasicCli.addShowCommandClass( IpNatTranslationStaticCmd )

#--------------------------------------------------------------------------------
# show ip nat translation dynamic
#  [ interface <INTF> | profile <PROFILE> | vrf <VRF> ] |
#  [ hardware | kernel | pending ] |
#  [ source | destination ] |
#  [ address <ADDRESS> [ <PORT> ] |
#  [ detail ]
#--------------------------------------------------------------------------------

class IpNatTranslationDynamicCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip nat translation DYNAMIC_TYPE
               [ { ( ( interface INTF ) | ( profile PROFILE ) | ( VRF ) )
                 | TABLE
                 | TARGET
                 | ( address ADDRESS [ PORT ] )
                 | detail } ]'''
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslation,
      'DYNAMIC_TYPE': CliCommand.singleNode(
         CliMatcher.KeywordMatcher( 'dynamic', helpdesc='Dynamic NAT translation' ),
         guard=natIntfDynamicSupported ),

      'interface': CliCommand.singleNode( matcherInterface,
                                          sharedMatchObj=intfOrVrfOrProfile,
                                          guard=natIpIntfConfigSupported ),
      'INTF': CliCommand.singleNode( Intf.matcher ),
      'profile': CliCommand.singleNode( matcherProfile,
                                        sharedMatchObj=intfOrVrfOrProfile,
                                        guard=natIpProfileConfigSupported ),
      'PROFILE': CliCommand.singleNode( matcherProfileName ),
      'VRF': vrfExprFactory,
      'TABLE': kernelTableEnumMatcher,
      'TARGET': CliCommand.singleNode( matcherTarget ),
      'address': CliCommand.singleNode( matcherAddress ),
      'ADDRESS': CliCommand.singleNode( matcherIpAddr ),
      'PORT': CliCommand.singleNode( matcherSrcPort ),
      'detail': CliCommand.singleNode( matcherDetail ),
   }
   handler = NatCli.cmdShowIpNatTranslations
   cliModel = NatIpTranslations

BasicCli.addShowCommandClass( IpNatTranslationDynamicCmd )

#--------------------------------------------------------------------------------
# show ip nat translation [ full-cone | dynamic-twice ]
#  [ interface <INTF> | profile <PROFILE> | vrf <VRF> ] |
#  [ hardware | kernel | pending ] |
#  [ address <ADDRESS> [ <PORT> ] |
#  [ detail ] | 
#  [ upnp ]
#--------------------------------------------------------------------------------
# NOTE: If you try to combine the `vrf` and `VRF` using `VrfExprFactory`,
# you'll get an ambiguous command around the token `vrf` for the command
# 'show ip nat translation vrf hardware detail' due to a collision with
# `IpNatTranslationVrfCmd`. It somehow works when they are separate.
class IpNatTranslationCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip nat translation [ DYNAMIC_TYPE | DYNAMIC_TWICE ]
               [ { ( ( interface INTF ) | ( profile PROFILE ) | ( vrf VRF ) )
                 | TABLE
                 | ( address ADDRESS [ PORT ] )
                 | detail | upnp} ]'''
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslation,
      'DYNAMIC_TYPE': CliCommand.singleNode(
         CliMatcher.KeywordMatcher( 'full-cone',
                                    helpdesc='Full-cone dynamic NAT translation' ),
         guard=natIntfDynamicSupported ),
      'DYNAMIC_TWICE': CliCommand.singleNode(
         matcher=matcherDynamicTwiceNat,
         guard=natDynamicTwiceSupported ),
      'interface': CliCommand.singleNode( matcherInterface,
                                          sharedMatchObj=intfOrVrfOrProfile,
                                          guard=natIpIntfConfigSupported ),
      'INTF': CliCommand.singleNode( Intf.matcher ),
      'profile': CliCommand.singleNode( matcherProfile,
                                        sharedMatchObj=intfOrVrfOrProfile,
                                        guard=natIpProfileConfigSupported ),
      'PROFILE': matcherProfileName,
      'vrf': CliCommand.singleKeyword( 'vrf',
                                       helpdesc='VRF',
                                       sharedMatchObj=intfOrVrfOrProfile,
                                       guard=natInVrfSupported ),
      'VRF': VrfNameExprFactory( maxMatches=1, inclAllVrf=True ),
      'TABLE': kernelTableEnumMatcher,
      'address': CliCommand.singleNode( matcherAddress ),
      'ADDRESS': CliCommand.singleNode( matcherIpAddr ),
      'PORT': CliCommand.singleNode( matcherSrcPort ),
      'detail': CliCommand.singleNode( matcherDetail ),
      'upnp': CliCommand.singleNode( matcherUpnp, guard=natUpnpPortMapSupported ),
   }
   handler = NatCli.cmdShowIpNatTranslations
   cliModel = NatIpTranslations

BasicCli.addShowCommandClass( IpNatTranslationCmd )

#--------------------------------------------------------------------------------
# show ip nat translation twice
#  [ interface <INTF> | profile <PROFILE> | vrf <VRF> ] |
#  [ hardware | pending ] |
#  [ address <ADDRESS> [ <PORT> ] |
#  [ detail ]
#--------------------------------------------------------------------------------

class IpNatTranslationTwiceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip nat translation twice [ { '
              '( ( interface INTF ) | ( profile PROFILE ) | ( VRF ) ) | '
              'TABLE | '
              '( address ADDRESS [ PORT ] ) | '
              'detail'
              '} ]' )
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslation,
      'twice': 'Twice NAT translation',
      'interface': CliCommand.singleNode( matcherInterface,
                                          sharedMatchObj=intfOrVrfOrProfile,
                                          guard=natIpIntfConfigSupported ),
      'INTF': CliCommand.singleNode( Intf.matcher ),
      'profile': CliCommand.singleNode( matcherProfile,
                                        sharedMatchObj=intfOrVrfOrProfile,
                                        guard=natIpProfileConfigSupported ),
      'PROFILE': matcherProfileName,
      'VRF': vrfExprFactory,
      'TABLE': tableEnumMatcher,
      'address': CliCommand.singleNode( matcherAddress ),
      'ADDRESS': CliCommand.singleNode( matcherIpAddr ),
      'PORT': CliCommand.singleNode( matcherSrcPort ),
      'detail': matcherDetail,
   }
   handler = NatCli.cmdShowIpNatTranslationsTwice
   cliModel = TwiceNatIpTranslations

BasicCli.addShowCommandClass( IpNatTranslationTwiceCmd )

#--------------------------------------------------------------------------------
# show ip nat translation address-only
#  [ interface <INTF> | profile <PROFILE> | vrf <VRF> ] |
#  [ hardware | kernel | pending ] |
#  [ source | destination ] |
#  [ address <ADDRESS> |
#  [ detail ]
#
# Note: the address-only cli has no PORT option
#--------------------------------------------------------------------------------

class IpNatTranslationAddressOnlyCmd( ShowCommand.ShowCliCommandClass ):
   syntax =  '''show ip nat translation DYNAMIC_TYPE
                [ { ( ( interface INTF ) | ( profile PROFILE ) | ( VRF ) )
                    | TABLE
                    | TARGET
                    | ( address ADDRESS )
                    | detail } ]'''
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslation,
      'DYNAMIC_TYPE': CliCommand.guardedKeyword(
         'address-only', 'Address-only dynamic NAT translation',
         natAddrOnlySupported ),
      'interface': CliCommand.singleNode( matcherInterface,
                                          sharedMatchObj=intfOrVrfOrProfile,
                                          guard=natIpIntfConfigSupported ),
      'INTF': CliCommand.singleNode( Intf.matcher ),
      'profile': CliCommand.singleNode( matcherProfile,
                                        sharedMatchObj=intfOrVrfOrProfile,
                                        guard=natIpProfileConfigSupported ),
      'PROFILE': matcherProfileName,
      'VRF': vrfExprFactory,
      'TABLE': kernelTableEnumMatcher,
      'TARGET': CliCommand.singleNode( matcherTarget ),
      'address': CliCommand.singleNode( matcherAddress ),
      'ADDRESS': CliCommand.singleNode( matcherIpAddr ),
      'detail': CliCommand.singleNode( matcherDetail ),
   }
   handler = NatCli.cmdShowIpNatTranslations
   cliModel = NatIpTranslations

BasicCli.addShowCommandClass( IpNatTranslationAddressOnlyCmd )

#--------------------------------------------------------------------------------
# show ip nat translation address-only mapping
#   [ ADDRESS ] [ interface <INTF> | profile <PROFILE> ) | VRF ]
#--------------------------------------------------------------------------------
class IpNatTranslationAddressOnlyMappingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip nat translation address-only mapping'
              '[ ADDRESS ] '
              '[ ( interface INTF ) | ( profile PROFILE ) | ( VRF ) ]'
   )
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslation,
      'address-only': CliCommand.guardedKeyword(
         'address-only', 'Address-only dynamic NAT translation',
         natAddrOnlySupported ),
      'mapping': CliCommand.guardedKeyword(
         'mapping', 'Address-only dynamic NAT mappings',
         natAddrOnlyKernelSupported ),
      'ADDRESS': matcherIpAddr,
      'interface': matcherInterface,
      'INTF': Intf.matcher,
      'profile': CliCommand.Node( matcherProfile,
                                  guard=natIpProfileConfigSupported ),
      'PROFILE': matcherProfileName,
      'VRF': VrfExprFactory( helpdesc='VRF',
                             guard=natInVrfSupported,
                             inclAllVrf=True ),
   }
   handler = NatCli.cmdShowIpNatTranslationsAddrOnly
   cliModel = NatAddrOnlyTranslations

BasicCli.addShowCommandClass( IpNatTranslationAddressOnlyMappingCmd )

#--------------------------------------------------------------------------------
# show ip nat translation <address-only | full-cone> flows
#  [ interface <INTF> | profile <PROFILE> | vrf <VRF> ] |
#  [ address <ADDRESS> |
#--------------------------------------------------------------------------------

class IpNatTranslationFlowsCmd( ShowCommand.ShowCliCommandClass ):
   syntax =  '''show ip nat translation (ADDR_ONLY | FULL_CONE) flows
                [ { ( ( interface INTF ) | ( profile PROFILE ) | ( VRF ) )
                    | ( address ADDRESS ) } ]'''
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslation,
      'ADDR_ONLY': CliCommand.guardedKeyword(
         'address-only', 'Address-only dynamic NAT translations',
         natAddrOnlySupported ),
      'FULL_CONE': CliCommand.guardedKeyword(
         'full-cone', 'Full-cone dynamic NAT translations',
         natIntfDynamicSupported ),
      'flows': CliCommand.guardedKeyword(
         'flows', 'Display dynamic NAT entries mappings',
         natAddrOnlyFlowSupported ),
      'interface': CliCommand.singleNode( matcherInterface,
                                          sharedMatchObj=intfOrVrfOrProfile,
                                          guard=natIpIntfConfigSupported ),
      'INTF': CliCommand.singleNode( Intf.matcher ),
      'profile': CliCommand.singleNode( matcherProfile,
                                        sharedMatchObj=intfOrVrfOrProfile,
                                        guard=natIpProfileConfigSupported ),
      'PROFILE': matcherProfileName,
      'VRF': vrfExprFactory,
      'address': CliCommand.singleNode( matcherAddress ),
      'ADDRESS': CliCommand.singleNode( matcherIpAddr ),
   }
   handler = NatCli.cmdShowIpNatTranslationsFlows
   cliModel = NatTranslationFlows

BasicCli.addShowCommandClass( IpNatTranslationFlowsCmd )

#--------------------------------------------------------------------------------
# show ip nat translation vrf [ { INSIDE_VRF |
#                                 OUTSIDE_VRF |
#                                 TARGET |
#                                 ( address ADDRESS ) } ]
#                             [ TABLE ]
#                             [ detail ]
#--------------------------------------------------------------------------------
class IpNatTranslationVrfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip nat translation vrf '
              '[ { INSIDE_VRF | OUTSIDE_VRF | TARGET | ( address ADDRESS ) } ] '
              '[ TABLE ] [ detail ]' )
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslation,
      'vrf': CliCommand.guardedKeyword( 'vrf',
                                        helpdesc='VRFs NAT translation',
                                        guard=natVrfHopStaticSupported ),
      'INSIDE_VRF': VrfExprFactory( keyword='inside-vrf',
                                    helpdesc='Inside VRF',
                                    guard=natVrfHopStaticSupported,
                                    maxMatches=1 ),
      'OUTSIDE_VRF': VrfExprFactory( keyword='outside-vrf',
                                     helpdesc='Outside VRF',
                                     guard=natOutsideVrfSupported,
                                     maxMatches=1 ),
      'TARGET' : CliCommand.singleNode( matcherTarget ),
      'address': CliCommand.singleNode( matcherAddress ),
      'ADDRESS': CliCommand.singleNode( matcherIpAddr ),
      'TABLE': tableEnumMatcher,
      'detail': matcherDetail,
   }
   handler = NatCli.cmdShowIpVrfNatTranslations
   cliModel = VrfNatIpTranslations

BasicCli.addShowCommandClass( IpNatTranslationVrfCmd )

#--------------------------------------------------------------------------------
# show ip nat translation max-entries
#--------------------------------------------------------------------------------
class IpNatTranslationMaxEntriesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat translation max-entries'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslationDyn,
      'max-entries': nodeMaxEntries,
   }
   handler = NatCli.cmdShowIpNatMaxEntries
   cliModel = NatConnectionLimit

BasicCli.addShowCommandClass( IpNatTranslationMaxEntriesCmd )

#--------------------------------------------------------------------------------
# show ip nat translation max-entries IPADDR
#--------------------------------------------------------------------------------
class IpNatTranslationMaxEntriesHostCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat translation max-entries IPADDR'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslationDyn,
      'max-entries': nodeMaxEntries,
      'IPADDR': matcherIpAddr,
   }
   handler = NatCli.cmdShowIpNatMaxEntriesHost
   cliModel = NatHostConnectionLimit

BasicCli.addShowCommandClass( IpNatTranslationMaxEntriesHostCmd )

#--------------------------------------------------------------------------------
# show ip nat translation max-entries pool POOLNAME
#--------------------------------------------------------------------------------
class IpNatTranslationMaxEntriesPoolCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat translation max-entries pool POOL'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslationDyn,
      'max-entries': nodeMaxEntries,
      'pool': nodePool,
      'POOL': matcherPoolName,
   }
   handler = NatCli.cmdShowIpNatMaxEntriesPool
   cliModel = NatHostConnectionLimit

BasicCli.addShowCommandClass( IpNatTranslationMaxEntriesPoolCmd )

#--------------------------------------------------------------------------------
# show ip nat translation rates
#--------------------------------------------------------------------------------
class IpNatTranslationRatesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat translation rates'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslationDyn,
      'rates' : 'Learning and programming rates for NAT translations',
   }
   handler = NatCli.cmdShowIpNatTranslationRates
   cliModel = NatTranslationRatesAll

BasicCli.addShowCommandClass( IpNatTranslationRatesCmd )

#--------------------------------------------------------------------------------
# show ip nat translation summary [ address ADDRESS ]
#--------------------------------------------------------------------------------
class IpNatTranslationSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat translation summary [ address ADDRESS ]'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslation,
      'summary': 'Get translation summary info',
      'address': matcherAddress,
      'ADDRESS': matcherIpAddr,
   }
   handler = NatCli.cmdShowIpNatTranslationSummary
   cliModel = NatTranslationSummary

BasicCli.addShowCommandClass( IpNatTranslationSummaryCmd )

#--------------------------------------------------------------------------------
# show ip nat translation dynamic summary
#--------------------------------------------------------------------------------
class IpNatTranslationDynamicSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat translation dynamic summary'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslation,
      'dynamic': CliCommand.guardedKeyword( 'dynamic', 
            helpdesc='Dynamic NAT translation',
            guard=natIntfDynamicSupported ),
      'summary': 'Dynamic translation summary',
   }

   handler = NatCli.cmdShowIpNatTranslationDynamicSummary
   cliModel = NatTranslationDynamicSummary

BasicCli.addShowCommandClass( IpNatTranslationDynamicSummaryCmd )

#--------------------------------------------------------------------------------
# show ip nat translation  [ vrf VRF ] connection limit drop
#--------------------------------------------------------------------------------
class IpNatTranslationDroppedCmd( ShowCommand.ShowCliCommandClass ):
   #syntax = 'show ip nat translation [ vrf VRF ] connection limit drop'
   syntax = 'show ip nat translation [ VRF ] connection limit drop'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'translation': nodeTranslation,
      'connection': matcherConnection,
      'limit': matcherLimit,
      'drop': matcherDrop,
      'VRF': VrfExprFactory( helpdesc='Show counters in a VRF',
                             guard=natInVrfSupported,
                             inclAllVrf=True ),
   }
   handler = NatCli.cmdShowIpNatTranslationConnectionDropped
   cliModel = NatTranslationConnectionDropped

BasicCli.addShowCommandClass( IpNatTranslationDroppedCmd )

#--------------------------------------------------------------------------------
# show ip nat dynamic counters [ profile <PROFILE> ]
#--------------------------------------------------------------------------------
class IpNatDynamicCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip nat dynamic counters [ profile PROFILE ]'
   data = {
      'ip': ipMatcherForShow,
      'nat': matcherNat,
      'dynamic': CliCommand.guardedKeyword( 'dynamic', 'Dynamic NAT info',
                                            dynamicNatCountersSupported ),
      'counters': CliMatcher.KeywordMatcher( 'counters',
                                             'Dynamic NAT summary counters' ),
      'profile': CliCommand.Node( matcherProfile,
                                  guard=natIpProfileConfigSupported ),
      'PROFILE': matcherProfileName,
   }
   handler = NatCli.cmdShowIpNatDynamicCounters
   cliModel = NatDynamicCountersModel

BasicCli.addShowCommandClass( IpNatDynamicCountersCmd )
