# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
from CliCommand import CliCommandClass, guardedKeyword
from CliMatcher import StringMatcher, IntegerMatcher
from CliPlugin.IntfCli import Intf
from CliPlugin.IpAddrMatcher import IpAddrMatcher
import CliPlugin.NatCli as NatCli # pylint: disable=consider-using-from-import
from CliPlugin.NatCli import (
      nodeNat,
      natIntfDynamicSupported,
)
from CliToken.Ip import ipMatcherForConfig

matcherPortRangeValue = IntegerMatcher( 1024, 65535, helpdesc='Port range value' )

#--------------------------------------------------------------------------------
# [ no | default ] description DESCRIPTION
#--------------------------------------------------------------------------------
class IpNatPeerDescriptionCmd( CliCommandClass ):
   syntax = 'description DESCRIPTION'
   noOrDefaultSyntax = 'description ...'
   data = {
      'description' : 'NAT peer description',
      'DESCRIPTION' : StringMatcher( helpdesc='Description of the Peer',
                                     helpname='DESCRIPTION' ),
   }

   handler = NatCli.configNatPeerDesc
   noOrDefaultHandler = NatCli.removeNatPeerDesc

NatCli.NatSyncConfigMode.addCommandClass( IpNatPeerDescriptionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] expiry-interval INTERVAL
#--------------------------------------------------------------------------------
class IpNatPeerExpiryIntervalCmd( CliCommandClass ):
   syntax = 'expiry-interval INTERVAL'
   noOrDefaultSyntax = 'expiry-interval ...'
   data = {
      'expiry-interval' : 'NAT peer expiry interval',
      'INTERVAL' : IntegerMatcher(
         60, 3600, helpdesc='Num of seconds to wait before expiring peer entries.' ),
   }

   handler = NatCli.configNatPeerExpiryInterval
   noOrDefaultHandler = NatCli.removeNatPeerExpiryInterval

NatCli.NatSyncConfigMode.addCommandClass( IpNatPeerExpiryIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] local-interface INTF
#--------------------------------------------------------------------------------
class IpNatPeerLocalInterfaceCmd( CliCommandClass ):
   syntax = 'local-interface INTF'
   noOrDefaultSyntax = 'local-interface ...'
   data = {
      'local-interface' : 'NAT peer local interface to use',
      'INTF' : Intf.matcher,
   }

   handler = NatCli.configNatPeerLocalIntf
   noOrDefaultHandler = NatCli.removeNatPeerLocalIntf

NatCli.NatSyncConfigMode.addCommandClass( IpNatPeerLocalInterfaceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] peer-address PEERIP
#--------------------------------------------------------------------------------
class IpNatPeerAddressCmd( CliCommandClass ):
   syntax = 'peer-address PEERIP'
   noOrDefaultSyntax = 'peer-address ...'
   data = {
      'peer-address' : 'NAT peer IP address',
      'PEERIP' : IpAddrMatcher( helpdesc='NAT peer IP address' ),
   }

   handler = NatCli.configNatPeerAddress
   noOrDefaultHandler = NatCli.removeNatPeerAddress

NatCli.NatSyncConfigMode.addCommandClass( IpNatPeerAddressCmd )

#--------------------------------------------------------------------------------
# [ no | default ] port-range LOWER [ UPPER ]
#--------------------------------------------------------------------------------
class IpNatPeerPortRangeCmd( CliCommandClass ):
   syntax = 'port-range LOWER [ UPPER ]'
   noOrDefaultSyntax = 'port-range ...'
   data = {
      'port-range' : 'NAT port range',
      'LOWER' : matcherPortRangeValue,
      'UPPER' : matcherPortRangeValue,
   }

   handler = NatCli.configNatPortRange
   noOrDefaultHandler = NatCli.removeNatPortRange

NatCli.NatSyncConfigMode.addCommandClass( IpNatPeerPortRangeCmd )

#--------------------------------------------------------------------------------
# [no | default] port-range split [disabled]
#--------------------------------------------------------------------------------
class IpNatPortRangeSplitCmd( CliCommandClass ):
   syntax = 'port-range split [disabled]'
   noOrDefaultSyntax = 'port-range split ...'
   data = {
      'port-range' : 'NAT port range',
      'split' : 'NAT port-range split mode',
      'disabled' : 'Disable NAT port-range split mode'
   }

   handler = NatCli.configNatPortRangeSplit
   noOrDefaultHandler = NatCli.configNatPortRangeSplit

NatCli.NatSyncConfigMode.addCommandClass( IpNatPortRangeSplitCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class IpNatPeerShutdownCmd( CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'NAT peer shutdown',
   }

   handler = NatCli.configNatPeerShutdown
   noOrDefaultHandler = NatCli.removeNatPeerShutdown

NatCli.NatSyncConfigMode.addCommandClass( IpNatPeerShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] sysname SYSNAME
#--------------------------------------------------------------------------------
class IpNatPeerSysnameCmd( CliCommandClass ):
   syntax = 'sysname SYSNAME'
   noOrDefaultSyntax = 'sysname ...'
   data = {
      'sysname' : 'NAT peer sysname',
      'SYSNAME' : StringMatcher( helpdesc='Sysname of the Peer',
                                 helpname='SYSNAME' ),
   }
   hidden = True

   handler = NatCli.configNatPeerSysname
   noOrDefaultHandler = NatCli.removeNatPeerSysname

NatCli.NatSyncConfigMode.addCommandClass( IpNatPeerSysnameCmd )

#--------------------------------------------------------------------------------
# [ no | default] ip nat synchronization
#--------------------------------------------------------------------------------
class IpNatSynchronizationCmd( CliCommandClass ):
   syntax = 'ip nat synchronization'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : ipMatcherForConfig,
      'nat' : nodeNat,
      'synchronization' : guardedKeyword( 'synchronization',
                                          'NAT peer synchronization',
                                          natIntfDynamicSupported ),
   }

   handler = NatCli.gotoIpNatSyncConfigMode
   noOrDefaultHandler = NatCli.removeIpNatSyncConfigMode

BasicCliModes.GlobalConfigMode.addCommandClass( IpNatSynchronizationCmd )

