#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ConfigMount
from TypeFuture import TacLazyType

import BasicCli
from CliParser import InvalidInputError
import CliCommand
import CliMatcher
from CliPlugin.MacAddr import macAddrMatcher
from CliPlugin.NbrClassificationCliModel import allClassifications
from CliPlugin import VlanCli
from CliToken.Neighbor import neighborMatcherForConfig

EthAddr = TacLazyType( "Arnet::EthAddr" )
IntfId = TacLazyType( "Arnet::IntfId" )
Neighbor = TacLazyType( "Identity::NbrClassification::Neighbor" )
Source = TacLazyType( "Identity::NbrClassification::Source" )
Classification = TacLazyType( "Identity::NbrClassification::Classification" )

identityCliConfig = None

classifications = dict( zip( allClassifications(),
                             [ 'Set classification as a phone' ] ) )

#------------------------------------------------------------------------------------
# [ no | default ] neighbor classification mac address MACADDRESS ( CLASSIFICATION |
#                                                                   none )
#------------------------------------------------------------------------------------
# This CLI is hidden becasue of BUG548893

classificationKw = CliCommand.guardedKeyword( 'classification',
   helpdesc='Set neighbor classification',
   guard=VlanCli.macBasedVlanSupportedGuard,
   hidden=False )

class SetNeighborClassificationCmd( CliCommand.CliCommandClass ):
   syntax = '''neighbor classification mac address MACADDRESS ( CLASSIFICATION |
                                                                none )'''
   noOrDefaultSyntax = 'neighbor classification mac address MACADDRESS ...'
   data = {
      'neighbor' : neighborMatcherForConfig,
      'classification' : classificationKw,
      'mac' : 'Set neighbor classification for given MAC address',
      'address' : 'Set neighbor classification for given MAC address',
      'MACADDRESS' : macAddrMatcher,

      'CLASSIFICATION' : CliMatcher.EnumMatcher( classifications ),
      'none' : 'Set as not classified',
   }

   @staticmethod
   def handler( mode, args ):
      phoneKw = [ 'phone', 'telephone' ]
      classificationArg = args.get( 'CLASSIFICATION' )
      mac = args[ 'MACADDRESS' ]

      if classificationArg and classificationArg not in phoneKw:
         raise InvalidInputError( "unknown classification %s" % classificationArg )

      classification = Classification()
      if 'none' not in args:
         classification.telephone = classificationArg in phoneKw

      identityCliConfig.neighbor.addMember( Neighbor( mac,
                                                      IntfId(),
                                                      classification,
                                                      Source.sourceCli ) )

   @staticmethod
   def noHandler( mode, args ):
      mac = args[ 'MACADDRESS' ]
      del identityCliConfig.neighbor[ mac ]

   @staticmethod
   def defaultHandler( mode, args ):
      SetNeighborClassificationCmd.noHandler( mode, args )

BasicCli.GlobalConfigMode.addCommandClass( SetNeighborClassificationCmd )

def Plugin( entityManager ):
   global identityCliConfig
   identityCliConfig = ConfigMount.mount( entityManager,
                                        "identity/nbrClassification/cliConfig",
                                        "Identity::NbrClassification::CliConfig",
                                        "w" )
