#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This file has the show command of the Identity package.
# It's in LLDP to prevent a dependency cycle from forming, as LLDP already depends
# on Identity but this show commands uses LLDP information in the description column.

import BasicCli
import CliMatcher
import Intf.IntfRange as IntfRange # pylint: disable=consider-using-from-import
import ShowCommand
import LazyMount
import Tac

import CliPlugin.TechSupportCli
from CliPlugin.MacAddr import MacAddrMatcher
from CliPlugin.NbrClassificationCliModel import ( allClassifications,
                                                  NbrClassificationsModel,
                                                  NbrClassificationsSummaryModel )

nbrClassificationStatus = None
lldpStatus = None

classifications = dict( zip( allClassifications(),
                                        [ 'Show neighbor classified as phones' ] ) )

class ShowNbrClassification( ShowCommand.ShowCliCommandClass ):
   syntax = '''show neighbor classification [ ( interface INTFS ) |
                                              ( mac address MACADDRESS ) |
                                              CLASSIFICATION ]'''

   data = {
      'neighbor': 'Show neighbor information',
      'classification': 'Show neighbor classification information',

      'interface': 'Show neighbor classification information for given interface',
      'INTFS': IntfRange.intfRangeMatcher,

      'mac': 'Show neighbor classification information for given MAC address',
      'address': 'Show neighbor classification information for given MAC address',
      'MACADDRESS': MacAddrMatcher(),

      'CLASSIFICATION': CliMatcher.EnumMatcher( classifications ),
   }

   cliModel = NbrClassificationsModel

   @staticmethod
   def handler( mode, args ):
      res = NbrClassificationsModel()
      res.populate( mode, nbrClassificationStatus, lldpStatus,
                    args.get( 'CLASSIFICATION' ), args.get( 'INTFS' ),
                    args.get( 'MACADDRESS' ) )
      return res

BasicCli.addShowCommandClass( ShowNbrClassification )

# ShowNbrClassification with summary as suffix
class ShowNbrClassificationSummary( ShowCommand.ShowCliCommandClass ):
   syntax = ShowNbrClassification.syntax + ' summary'

   data = dict( ShowNbrClassification.data )
   data[ 'summary' ] = 'Show summary output instead of full table output'

   cliModel = NbrClassificationsSummaryModel

   @staticmethod
   def handler( mode, args ):
      res = NbrClassificationsSummaryModel()
      res.populate( mode, nbrClassificationStatus, lldpStatus,
                    args.get( 'CLASSIFICATION' ), args.get( 'INTFS' ),
                    args.get( 'MACADDRESS' ) )
      return res

BasicCli.addShowCommandClass( ShowNbrClassificationSummary )

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2019-07-18 05:38:20',
   cmds=[ 'show neighbor classification' ],
   summaryCmds=[ 'show neighbor classification summary' ] )

def Plugin( entityManager ):
   global nbrClassificationStatus
   nbrClassificationStatus = LazyMount.mount( entityManager,
                                 'identity/nbrClassification/status',
                                 'Identity::NbrClassification::Status',
                                 'r' )
   global lldpStatus
   lldpStatus = LazyMount.mount( entityManager,
                                 'l2discovery/lldp/status/all',
                                 'Lldp::AllStatus',
                                 'r' )
