# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from ArnetModel import Ip4Address, Ip6Address, IpGenericAddress
from CliModel import Dict, Int, List, Model, Submodel, Str
import TableOutput

MAX_NAME_LEN = 29

class DomainName( Model ):
   domainName = Str( help="The domain name of the switch", optional=True )

   def render( self ):
      if self.domainName:
         print( self.domainName )

class NameServerConfig( Model ):
   ipAddr = IpGenericAddress( help="Name server IP address" )
   vrf = Str( help="Name server VRF" )
   priority = Int( help="Name server priority" )

class NameServerTable( Model ):
   nameServerConfigs = List( valueType=NameServerConfig,
                             help="List of configured name servers" )

   def render( self ):
      table = TableOutput.createTable( [ 'IP Address', 'VRF', 'Priority' ] )
      f1 = TableOutput.Format( justify='left' )
      f1.padLimitIs( True )
      f2 = TableOutput.Format( justify='left' )
      f2.padLimitIs( True )
      f3 = TableOutput.Format( justify='right' )
      f3.padLimitIs( True )
      f3.noPadLeftIs( True )
      table.formatColumns( f1, f2, f3 )
      for nameServer in self.nameServerConfigs:
         table.newRow( nameServer.ipAddr, nameServer.vrf,
                       nameServer.priority )
      print( table.output() )

class NameServerExcessPriorities( Model ):
   largestValidPriorityValue = Int( help="Last valid priority value" )
   firstRedundantPriorityValue = Int( help="First redundant priority value" )

class SysMgrNameServerInfo( NameServerTable ):
   v4NameServers = List( valueType=str,
                         help="List of IPV4 name servers" )
   v6NameServers = List( valueType=str,
                         help="List of IPV6 name servers" )
   _excessPriorities = Submodel(
         valueType=NameServerExcessPriorities,
         help="No more than five distinct priority values",
         optional=True )

   def render( self ):
      NameServerTable.render( self )
      if self._excessPriorities:
         largestValid = self._excessPriorities.largestValidPriorityValue
         firstRedundant = self._excessPriorities.firstRedundantPriorityValue
         print( f"Name servers with priority {firstRedundant} or greater will be " +
                f"treated as having priority {largestValid} and queried " +
                "simultaneously." )

class Hostname( Model ):
   hostname = Str( help="The current hostname of the switch" )
   fqdn = Str( help="The fully qualified domain name of the switch" )

   def render( self ):
      if not self.hostname:
         print( "No hostname set" )
      else:
         print( "Hostname:", self.hostname )
         print( "FQDN:    ", self.fqdn )

class SysMgrHostInfo( NameServerTable ):
   class IpAddresses( Model ):
      ipv4Addresses = List( valueType=Ip4Address, help="List of IPV4 addresses" )
      ipv6Addresses = List( valueType=Ip6Address, help="List of IPV6 addresses" )

   hosts = Dict( keyType=str, valueType=IpAddresses,
                 help="A mapping between host names and their IP addresses" )
   domainName = Str( help="The domain name that the listed hosts are "
                     "associated with",
                     optional=True )
   domainList = List( valueType=str,
                      help="List of additional domains to lookup domains with" )
   nameServers = List( valueType=str,
                       help="List of domain name servers used to resolve names" )

   def render( self ):
      if self.domainName:
         print( '\nDefault domain is: %s' % self.domainName )
      else:
         print( '\nDefault domain is not configured' )
      if self.domainList:
         print( 'Additional domains are: ' + ', '.join( self.domainList ) )
      print( 'Name/address lookup uses domain service' )

      print( 'Name servers are:' )
      NameServerTable.render( self )

      print( 'Static Mappings:' )
      table = TableOutput.createTable( [ 'Hostname', 'IP', 'Addresses' ] )
      f1 = TableOutput.Format( justify='left' )
      f1.padLimitIs( True )
      f2 = TableOutput.Format( justify='left' )
      f2.padLimitIs( True )
      f3 = TableOutput.Format( justify='left' )
      f3.padLimitIs( True )
      table.formatColumns( f1, f2, f3 )

      def addRows( host, af, addrs ):
         firstLineFlag = True
         for addr in addrs:
            if firstLineFlag:
               table.newRow( host, af, addr )
               firstLineFlag = False
            else:
               table.newRow( '', '', addr )

      for host, addresses in sorted( self.hosts.items() ):
         addRows( host, 'IPV4', addresses.ipv4Addresses )
         v6AddrStrs = [ addr.stringValue for addr in addresses.ipv6Addresses ]
         addRows( host, 'IPV6', v6AddrStrs )

      print( table.output() )

   def _printAddresses( self, addresses, host, ipType ):
      firstLineFlag = True
      for addr in addresses:
         if ipType == 'IPV6':
            addr = addr.stringValue
         if firstLineFlag:
            print ( self._formatHostMapping( addr, ipType, host ) )
            firstLineFlag = False
         else:
            print ( self._formatHostMapping( addr ) )

   def _formatHostMapping( self, address, ipType='', name='' ):
      return '%-*s %4s    %s' % ( MAX_NAME_LEN, name, ipType, address )

class DnsCacheCountersModel( Model ):
   size = Int( help="Cache size" )
   insertions = Int( help="Insertions" )
   evictions = Int( help="Evictions" )
   misses = Int( help="Queries forwarded" )
   hits = Int( help="Queries answered locally" )

   def render( self ):
      print( "Cache size: %s" % self.size )
      print( "Cache insertions: %s" % self.insertions )
      print( "Unexpired cache evictions: %s" % self.evictions )
      print( "Queries forwarded: %s" % self.misses )
      print( "Queries replied: %s" % self.hits )
