# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

#------------------------------------------------------------------------------------
# This module provides the 'ping', 'traceroute', 'telnet' CLI
# commands.
#
# XXX Note that the industry-standard version of each of these commands has many more
# options.  We just implement the basics.
#
# This module also provides support for invoking scp or sftp-server in response to 
# remote scp or sftp requests.
#------------------------------------------------------------------------------------
import glob
from netaddr import IPAddress
import os
import pwd
import shlex
import socket
import subprocess
import sys
import Tac

import Arnet
from Arnet.NsLib import DEFAULT_NS
import BasicCliModes
import CliCommand
import CliMatcher
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
import CliParser
import CmdExtension
import ConfigMount
import DscpCliLib
import HostnameCli
from IpLibConsts import DEFAULT_VRF
import LazyMount
import Logging
from PyWrappers.Iputils import pingName
import PyWrappers.Telnet as telnet
import PyWrappers.Traceroute as traceroute
import SysMgrLib
from SshAlgorithms import FIPS_CIPHERS, FIPS_KEXS, FIPS_MACS
from IcmpErrorCliHooks import maybeAddNodeIdExtensionParsing

intfStatusAll = None
ipStatus = None
sshConfig = None
ip6Status = None
networkUrlConfig = None
tracerouteConfig = None
routeStatus = None
bridgingConfig = None
tracerouteCapabilities = None
vtiStatusDir = None

# TODO: these are used by MplsUtil package, need to convert that package
# and then remove these
repeatKwNode = CliCommand.singleKeyword( 'repeat',
      helpdesc='specify repeat count' )
countRangeMatcher = CliMatcher.IntegerMatcher( 1, 0x7fffffff,
                                               helpdesc='Count' )

class MissingIntfError( Exception ):
   pass

SECURITY_SSH_CLIENT_CONNECTING = Logging.LogHandle(
              'SECURITY_SSH_CLIENT_CONNECTING',
              severity=Logging.logInfo,
              fmt='%s is connecting a SSH session from the switch to %s@%s:%s',
              explanation='A person logged into the switch began a SSH session'
              ' to an external server.',
              recommendedAction=Logging.NO_ACTION_REQUIRED )
SECURITY_SSH_CLIENT_DISCONNECTED = Logging.LogHandle(
              'SECURITY_SSH_CLIENT_DISCONNECTED',
              severity=Logging.logInfo,
              fmt='%s disconnected the SSH session from the switch to %s@%s:%d',
              explanation='The SSH session from the switch to the remote server'
              ' is now over',
              recommendedAction=Logging.NO_ACTION_REQUIRED )


def _resolveDestAddrFam( destination ):
   # if the destination is configured with one address
   # family but not the other, use that address family
   try:
      addrList = socket.getaddrinfo( destination, None, 0, socket.SOCK_DGRAM )
      ipv4 = any( addr[0] == socket.AF_INET for addr in addrList )
      ipv6 = any( addr[0] == socket.AF_INET6 for addr in addrList )
      if ipv4 and not ipv6:
         return 'ipv4'
      elif ipv6 and not ipv4:
         return 'ipv6'
   except UnicodeError:
      pass
   except socket.gaierror:
      pass
   return None

def _cliIntfToKernelIntf( cliIntf ):
   interfaceName = cliIntf.name
   try:
      return intfStatusAll.intfStatus[ interfaceName ].deviceName
   except KeyError:
      return None

def _ipFromInterface( interface, vrf, ipProto ):
   result4 = None
   result6 = None
   if not interface:
      # if no interface specified pick one
      vrfIpIntfStatus = ipStatus.vrfIpIntfStatus.get( vrf )
      if not vrfIpIntfStatus:
         return None

      for intf in vrfIpIntfStatus.ipIntfStatus:
         if intf.startswith( 'Ethernet' ) or \
            ( intf.startswith( 'Vlan' ) and
              not vrfIpIntfStatus.ipIntfStatus[ intf ].dpRoutingEnabled ):
            interface = intf
            break
      if not interface:
         return None
   ipIntf4 = ipStatus.ipIntfStatus.get( interface )
   if ipIntf4 and ipIntf4.vrf == vrf and ipIntf4.activeAddrWithMask:
      result4 = ipIntf4.activeAddrWithMask.address

   if ipProto == 'ipv4':
      if not ipIntf4:
         raise MissingIntfError( "No source interface " + interface )
      return Arnet.IpGenAddr( result4 ) if result4 else None

   ipIntf6 = ip6Status.intf.get( interface )
   if ipIntf6 and ipIntf6.vrf == vrf:
      maybe6 = ipIntf6.highestSourceAddress()
      if not maybe6.isZero:
         result6 = maybe6.stringValue

   if ipProto == 'ipv6':
      if not ipIntf6:
         raise MissingIntfError( "No source interface " + interface )
      return Arnet.IpGenAddr( result6 ) if result6 else None

   result = result4 or result6

   if not result:
      if not ipIntf4 and not ipIntf6:
         raise MissingIntfError( "No source interface " + interface )
      return None

   return Arnet.IpGenAddr( result )

def _resolveSrcAndAddrFamily( mode, sourceIp, intfWithSrcIp, ipProto,
                              vrfName, networkTool ):
   '''
   Given the two different ways of specifying the source ip to choose:
    - sourceIp: the source ip itself
    - intfWithSrcIp: choose an ip from the interface (if v4, active if v6, highest)
   choose/resolve the correct sourceIp.

   If no sourceIp or intfWithSrcIp are specified, we check the protocolConfig to see
   if the user has specified an interface and set intfWithSrcIp.
   '''
   srcIntfForVrf = None
   if not sourceIp and not intfWithSrcIp:
      protoConf = networkUrlConfig.protocolConfig.get( networkTool )
      if protoConf:
         srcIntfForVrf = protoConf.srcIntf.get( vrfName )
         if srcIntfForVrf:
            intfWithSrcIp = srcIntfForVrf.intfName

   if intfWithSrcIp:
      sourceIp = _ipFromInterface( str( intfWithSrcIp ), vrfName, ipProto )

   sourceIpProto = None
   if sourceIp:
      sourceIpProto = sourceIp.af

   sourceIpProto = sourceIpProto or ipProto

   # pylint: disable-next=singleton-comparison,consider-using-in
   if ipProto != None and sourceIpProto != ipProto:
      mode.addError(
         "Destination IP address family for desintation does not match source" )
      return None, None

   return sourceIp.stringValue if sourceIp else None, sourceIpProto

#------------------------------------------------------------------------------------
# The 'ping' command, in enable mode.
#
# The full syntax of this command is:
#
#  ping [vrf <vrf name>] [ip|ipv6] <destination> [df-bit] [repeat <count>]
#                            [size <size>] [timeout <timeout>]
#                            [interval <interval>] [source <ip-address>]
#                            [pattern <pattern>]
#------------------------------------------------------------------------------------

# According to the ping man page:
# You may specify up to 16 ``pad'' bytes to fill out the packet you send.
pingDataPattern = '(0x)?([0-9a-fA-F][0-9a-fA-F]){1,16}'

class CommonOptionsExpr( CliCommand.CliExpression ):
   expression = ( 'df-bit | '
                  '( repeat COUNT ) | '
                  '( size SIZE ) | '
                  '( timeout TIMEOUT ) | '
                  '( interval INTERVAL ) | '
                  '( source ( IP_GEN_ADDR | INTF_WITH_IP ) ) | '
                  '( interface INTF ) | '
                  '( pattern PATTERN )' )
   data = {
            'df-bit': CliCommand.singleKeyword( 'df-bit',
               helpdesc='enable do not fragment bit in IP header' ),
            'repeat': repeatKwNode,
            'COUNT': CliMatcher.IntegerMatcher( 1, 0x7fffffff,
               helpdesc='Count' ),
            'size': CliCommand.singleKeyword( 'size',
               helpdesc='specify datagram size' ),
            'SIZE': CliMatcher.IntegerMatcher( 36, 18024,
               helpdesc='Ping packet size' ),
            'timeout': CliCommand.singleKeyword( 'timeout',
               helpdesc='specify time to wait for a response' ),
            'TIMEOUT': CliMatcher.IntegerMatcher( 0, 3600,
               helpdesc='Number of seconds' ),
            'interval': CliCommand.singleKeyword( 'interval',
               helpdesc='specify time to wait between sending each packet' ),
            'INTERVAL': CliMatcher.FloatMatcher( 0, 60,
               helpdesc='Interval in units of seconds', precisionString='%.3f' ),
            'source': CliCommand.singleKeyword(
               'source',
               helpdesc='specify source address or choose an address from specified '
               'interface' ),
            'IP_GEN_ADDR': IpGenAddrMatcher.ipGenAddrMatcher,
            'INTF': IntfCli.Intf.matcher,
            'interface': CliCommand.singleKeyword( 'interface',
               helpdesc='specify egress interface' ),
            'INTF_WITH_IP': CliCommand.Node( IntfCli.Intf.matcher ),
            'pattern': CliCommand.singleKeyword( 'pattern',
               helpdesc='specify up to 16 bytes of packet data' ),
            'PATTERN': CliMatcher.PatternMatcher( pingDataPattern,
               helpname='PATTERN',
               helpdesc='Up to 16 bytes of hex data' )
          }

class IpOptionsExpr( CliCommand.CliExpression ):
   expression = 'record-route | COMMON_OPTIONS'
   data = {
            'record-route': CliCommand.singleKeyword( 'record-route',
               helpdesc='enable record route option' ),
            'COMMON_OPTIONS': CommonOptionsExpr
          }

def doResolveSrcProtoVrf( mode, args, destination, networkTool='ping' ):
   proto = None
   if 'ipv6' in args:
      proto = 'ipv6'
   elif 'ip' in args:
      proto = 'ipv4'
   vrf = args.get( 'VRF' )

   if vrf is None:
      # check if running in a vrf context
      vinf = mode.session.sessionData( 'vrf',
                                       defaultValue=( DEFAULT_VRF, DEFAULT_NS ) )
      vrf = vinf[ 0 ]

   sourceIp = args.get( 'IP_GEN_ADDR', [ '' ] )[ 0 ]
   intfWithSrcIp = args.get( 'INTF_WITH_IP', [ None ] )[ 0 ]

   if not proto:
      proto = _resolveDestAddrFam( destination )
   try:
      src, proto = _resolveSrcAndAddrFamily(
         mode, sourceIp, intfWithSrcIp, proto, vrf, networkTool )
   except MissingIntfError as mie:
      mode.addError( str( mie ) )
      return None

   return src, proto, vrf

def doPing( mode, args ):
   cliCmdExt = CmdExtension.getCmdExtender()
   destination = args[ 'DESTINATION' ]

   srcProtoVrf = doResolveSrcProtoVrf( mode, args, destination )
   if srcProtoVrf is None:
      # there was an error, just return
      return
   src, proto, vrf = srcProtoVrf

   egressIntf = None
   egressIntfList = args.get( 'INTF' )
   if egressIntfList:
      egressIntf = _cliIntfToKernelIntf( egressIntfList[ 0 ] )
      if not egressIntf:
         mode.addError( "No such interface: %s" % egressIntfList[ 0 ] )
         return

   cmd = pingName()
   pArgs = [ 'sudo', cmd ]
   sizeModifier = 8 # ICMP header size
   # bcast pkts are permitted with -M option when ping run with root privileges
   # '-b' option is valid for ipv4 only
   if proto == 'ipv6':
      pArgs.extend( [ '-6' ] )
      sizeModifier += 40 # size of ipv6 header
   else:
      pArgs.extend( [ '-4', '-b' ] )
      sizeModifier += 20 # size of ipv4 header

   interval = args.get( 'INTERVAL', [ 0 ] )[ 0 ]
   if not interval:
      # Flag for adaptive ping is set when interval is not specified or set to 0
      pArgs.extend( [ '-A' ] )


   pArgs.extend( [ '-c', str( args.get( 'COUNT', [ 5 ] )[ 0 ] ),
                  '-s', str( args.get( 'SIZE', [ 100 ] )[ 0 ] - sizeModifier ),
                  '-W', str( args.get( 'TIMEOUT', [ 2 ] )[ 0 ] ),
                  '-i', str( interval ) ] )

   if args.get( 'df-bit', False ) :
      pArgs.extend( [ '-M', 'do' ] )
   else:
      pArgs.extend( [ '-M', 'dont' ] )

   pattern = args.get( 'PATTERN', [ None ] )[ 0 ]
   if pattern:
      patternOpt = pattern.replace( '0x', '' )
      pArgs.extend( [ '-p', patternOpt ] )
   if args.get( 'record-route', False ):
      pArgs.extend( [ '-R' ] )

   if src:
      pArgs.extend( [ '-I', src ] )

   if egressIntf:
      # When pinging a link-local IPv6 address using ICMP sockets, -I doesn't work
      # for some kernel versions. The workaround is to use a link-specification
      # encoded in the destination.
      # This can be removed after we move to kernel 5.10.96 or 4.19.228
      # github.com/iputils/iputils/commit/ed7c27d32ec6498569b6715008643d03517a804f
      isLinkLocalIpv6Dst = False
      try:
         ip6Addr = Arnet.Ip6Addr( destination )
         isLinkLocalIpv6Dst = ip6Addr.isLinkLocal
      except IndexError:
         # destination is not an IPv6 address,
         # it could be an IPv4 address or a DNS name
         pass

      if isLinkLocalIpv6Dst:
         linkSpec =  '%' + egressIntf
         destination += linkSpec
      else:
         pArgs.extend( [ '-I', egressIntf ] )

   pArgs.extend( [ '--', destination ] )

   try:
      # mashing stderr into stdout because under capi stderr is devnulled to prevent
      # traces interference during cohab btests.
      p = cliCmdExt.subprocessPopen( pArgs, mode.session, vrfName=vrf,
                                     stdout=sys.stdout, stderr=sys.stdout )
      p.communicate()
   except OSError as e:
      mode.addError( e.strerror )

pingMatcher = CliMatcher.KeywordMatcher( 'ping', helpdesc='Ping remote systems' )
dnsNameOrIpv4OrIpv6AddressPattern = \
   r'(?!(^((ip)|(ipv6)|(mpls)|' \
   r'(adaptive\-virtual\-topology)|(overlay)|(underlay)|(resolve)' \
   r')$))[0-9a-zA-Z_.\-:]+'
dstMatcher = CliMatcher.PatternMatcher( dnsNameOrIpv4OrIpv6AddressPattern,
                                        helpname='WORD',
                                        helpdesc='Destination address ' \
                                                 'or hostname' )
pingVrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='Ping in a VRF',
      inclDefaultVrf=True )

class DoPing( CliCommand.CliCommandClass ):
   syntax = 'ping [ VRF ] [ ip ] DESTINATION [ { OPTIONS } ]'
   data = {
            'ping': pingMatcher,
            'VRF': pingVrfExprFactory,
            'ip': 'IPv4 ping',
            'DESTINATION': dstMatcher,
            'OPTIONS': IpOptionsExpr
          }
   handler = doPing

class DoPingV6( CliCommand.CliCommandClass ):
   syntax = 'ping [ VRF ] ipv6 DESTINATION [ { OPTIONS } ]'
   data = {
            'ping': pingMatcher,
            'VRF': pingVrfExprFactory,
            'ipv6': 'IPv6 ping',
            'DESTINATION': dstMatcher,
            'OPTIONS': CommonOptionsExpr
          }
   handler = doPing

BasicCliModes.EnableMode.addCommandClass( DoPing )
BasicCliModes.EnableMode.addCommandClass( DoPingV6 )

#------------------------------------------------------------------------------------
# Add a restricted version of the 'ping' command to unprivileged mode.
#
# The full syntax of this command is:
#
#  ping [ ip | ipv6 ] DESTINATION
#------------------------------------------------------------------------------------
class DoPingUnprivCmd( CliCommand.CliCommandClass ):
   syntax = 'ping [ ip | ipv6 ] DESTINATION'
   data = {
            'ping': pingMatcher,
            'ip': 'IP echo',
            'ipv6': 'IPv6 ping',
            'DESTINATION': dstMatcher,
          }
   handler = doPing

BasicCliModes.UnprivMode.addCommandClass( DoPingUnprivCmd )

def getSourceIpForRoute( output ):
   outputList = output.split()
   srcIndex = outputList.index( 'src' )
   if srcIndex:
      # return the keyword after the src keyword which will be the
      # source ip address
      return outputList[ srcIndex + 1 ]
   return None

def getCmdNonDefaultVrf( cmd, vrf ):
   nsVrfName = 'ns-%s' % vrf
   cmd = "nsenter --net=/var/run/netns/%s -- %s" % ( nsVrfName, cmd )
   return cmd

def runCmdAndGetSrcIntfIp( cmd ):
   src = None
   cmdList = cmd.split( " " )
   output = Tac.run( cmdList, stdout=Tac.CAPTURE, stderr=Tac.DISCARD,
                     ignoreReturnCode=True, asRoot=True )
   # search for unreachable route or fwd in route,
   # if found return source ip as None else return
   # source ip associated with the route
   if ( output and 'unreachable' not in output
        and 'fwd' not in output ):
      src = getSourceIpForRoute( output )
   return src

def getKernelRouteSourceIntfIp( vrf, destination, proto ):
   try:
      socket.inet_aton( destination )
      if proto and proto != 'ipv4':
         return None
      cmd = "ip route get %s" % ( destination )
      if vrf != DEFAULT_VRF:
         if not VrfCli.vrfExists( vrf ):
            return None
         cmd = getCmdNonDefaultVrf( cmd, vrf )
      return runCmdAndGetSrcIntfIp( cmd )
   except OSError:
      try:
         socket.inet_pton( socket.AF_INET6, destination )
         if proto and proto != 'ipv6':
            return None
         cmd = "ip -f inet6 route get %s" % ( destination )
         if vrf != DEFAULT_VRF:
            if not VrfCli.vrfExists( vrf ):
               return None
            cmd = getCmdNonDefaultVrf( cmd, vrf )
         return runCmdAndGetSrcIntfIp( cmd )
      except OSError:
         return None

def getLowestIntfIndexIpAddr( vrf, destination, proto ):
   src, intfs = None, []
   try:
      socket.inet_aton( destination )
      if proto and proto != 'ipv4':
         return None
      ipIntfVrfStatus = ipStatus.vrfIpIntfStatus.get( vrf )
      if ipIntfVrfStatus:
         for intf in ipIntfVrfStatus.ipIntfStatus.values():
            intfs.append( intf.intfId )
         if intfs:
            intfs.sort()
            for intf in intfs:
               intf = ipIntfVrfStatus.ipIntfStatus.get( intf )
               # if we remove ip address after configuring to an interface
               # and do not change to switchport, ip address shows up as 0.0.0.0.
               # For ipv6 case, removing ip address also removes from
               # ipIntfStatus collection.
               if intf.activeAddrWithMask.address != "0.0.0.0":
                  return intf.activeAddrWithMask.address
      return src
   except OSError:
      try: # pylint: disable=too-many-nested-blocks
         socket.inet_pton( socket.AF_INET6, destination )
         if proto and proto != 'ipv6':
            return None
         ipIntfVrfStatus = ip6Status.vrfIp6IntfStatus.get( vrf )
         if ipIntfVrfStatus:
            for intf in ipIntfVrfStatus.ip6IntfStatus.values():
               intfs.append( intf.intfId )
            if intfs:
               intfs.sort()
               for intf in intfs:
                  intf = ipIntfVrfStatus.ip6IntfStatus.get( intf )
                  address = intf.selectDefaultSourceAddress()
                  if address:
                     return address.stringValue
         return src
      except OSError:
         return src

#------------------------------------------------------------------------------------
# The 'traceroute' command, in enable mode.
#
# The full syntax of this command is:
#
#  traceroute [vrf <vrf name>]  [ip|ipv6] 
#       ( <destination> | overlay [ resolve ] <srcIp> <dstIp> <sport>
#               <dport> <proto> [ maxhops <maxhops> ] )
#               [source <source interface>]
#------------------------------------------------------------------------------------
def doOverlayTraceroute( mode, args, vrf, src, ipVer ):
   dstIp = args[ 'DST_IP' ].stringValue

   trArgs = [ 'sudo', '-E', '/usr/bin/trout' ]
   trArgs += [ args[ 'SRC_IP' ].stringValue ]
   trArgs += [ dstIp ]
   trArgs += [ str( args[ 'SPORT' ] ) ]
   trArgs += [ str( args[ 'DPORT' ] ) ]
   if args[ 'PROTO' ] == 'udp':
      trArgs += [ str( socket.IPPROTO_UDP ) ]
   else:
      trArgs += [ str( socket.IPPROTO_TCP ) ]
   trArgs += [ '--bridgeMac', bridgingConfig.bridgeMacAddr ]

   if args.get( 'underlay' ):
      trArgs += [ '-u' ]

   # get source kernel interface ( fwd )
   vrfStatus = routeStatus.vrfStatus.get( vrf )

   if vrfStatus is None:
      mode.addError( f'Route status for VRF {vrf} not found' )
      return

   if vrfStatus.fwd0EthTypeEntry.get( ipVer ) is None:
      mode.addError( f'{ipVer} routing is not enabled' )
      return

   fwdInterface = vrfStatus.fwd0EthTypeEntry[ ipVer ].devName
   trArgs += [ '-i', fwdInterface ]

   trArgs += [ '--ip-version', ipVer.strip( 'ipv' ) ]

   vtiStatus = vtiStatusDir.vtiStatus.get( 'Vxlan1' )
   if vtiStatus is None:
      mode.addError( 'VTI status for interface Vxlan1 not found' )
      return

   if vtiStatus.udpPort != 4789:
      trArgs += [ '-p', str( vtiStatus.udpPort ) ]

   maxHops = args.get( 'MAX_HOPS', 30 )
   trArgs += [ '-m', str( maxHops ) ]

   if args.get( 'resolve' ):
      trArgs += [ '-n' ]

   # get source IP
   if not src:
      src = _ipFromInterface( None, vrf, ipVer )
      if not src:
         mode.addError( 'No interface with valid IP address' )
         return
      trArgs += [ '-s', src.stringValue ]
   else:
      trArgs += [ '-s', src ]

   if vrf != DEFAULT_VRF:
      trArgs += [ '--vrf', vrf ]

   cliCmdExt = CmdExtension.getCmdExtender()
   try:
      p = cliCmdExt.subprocessPopen( trArgs, mode.session, vrfName=DEFAULT_VRF,
                                     stdout=sys.stdout, stderr=subprocess.PIPE )
      p.communicate()
   except OSError as e:
      mode.addError( e.strerror )

def doTraceroute( mode, args ):
   destination = args.get( 'DESTINATION' )

   proto = 'ipv4'
   if 'ipv6' in args:
      proto = 'ipv6'
   elif 'ip' in args:
      proto = 'ipv4'
   elif destination:
      proto = _resolveDestAddrFam( destination )

   vrf = args.get( 'VRF' )
   if vrf is None:
      # check if running in a vrf context
      vrf = mode.session.sessionData( 'vrf',
                     defaultValue=( DEFAULT_VRF, DEFAULT_NS ) )[ 0 ]

   sourceIp = None
   intfWithSrcIp = None
   if 'IP_GEN_ADDR' in args:
      sourceIp = args[ 'IP_GEN_ADDR' ]
   elif 'INTF_WITH_IP' in args:
      intfWithSrcIp = args[ 'INTF_WITH_IP' ]

   try:
      src, proto = _resolveSrcAndAddrFamily(
         mode, sourceIp, intfWithSrcIp, proto, vrf, 'traceroute' )
   except MissingIntfError as mie:
      mode.addError( str( mie ) )
      return

   if destination is None:
      doOverlayTraceroute( mode, args, vrf, src, proto )
      return

   egressIntf = args.get( 'INTF' )
   if egressIntf:
      egressIntf = _cliIntfToKernelIntf( egressIntf )
      if not egressIntf:
         mode.addError( "No such interface: %s" % egressIntf )
         return

   # Choose the outgoing interface ip from kernel routing table given by
   # ip route get <dest> for the vrf as source ip.
   # If unreachable route choose the lowest interface index ip address
   # out of all the interfaces in the given vrf as source ip.
   if src is None:
      src = getKernelRouteSourceIntfIp( vrf, destination, proto )
   if src is None:
      src = getLowestIntfIndexIpAddr( vrf, destination, proto )

   args = [ traceroute.name() ]
   args.extend( [ '-e' ] ) #show Icmp extensions, if present
   
   maybeAddNodeIdExtensionParsing( args )
   
   if tracerouteConfig.dscpValue:
      tos = tracerouteConfig.dscpValue << 2
      args += [ '-t', str( tos ) ]

   if proto == 'ipv4':
      args.extend( [ '-4' ] )
   elif proto == 'ipv6':
      args.extend( [ '-6' ] )

   # The interface option to the source keyword is hidden. If the user specifies it
   # though, override the interface keyword. A user can specify both source <ip> and
   # interface <intf>.
   # Note: Fedora traceroute requires root privilages if specifying source interface
   if src and egressIntf:
      args = [ 'sudo', '-E' ] + args + [ '-i', egressIntf, '-s', src ]
   elif egressIntf:
      args = [ 'sudo', '-E' ] + args + [ '-i', egressIntf ]
   elif src:
      args += [ '-s', src ]

   args.extend( [ '--', destination ] )

   cliCmdExt = CmdExtension.getCmdExtender()
   try:
      p = cliCmdExt.subprocessPopen( args, mode.session, vrfName=vrf,
                                     stdout=sys.stdout, stderr=subprocess.PIPE )
      # Filtering error output by removing unwanted lines.
      # mashing stderr into stdout because under capi stderr is devnulled to prevent
      # traces interference during cohab btests.
      subprocess.call( [ 'grep', '-v', '--line-buffered', 'Cannot handle.*arg' ],
                       stdin=p.stderr, stdout=sys.stdout )
      p.wait()
   except OSError as e:
      mode.addError( e.strerror )

tracerouteKwMatcher = CliMatcher.KeywordMatcher( 'traceroute',
                                         helpdesc='Traceroute command' )
overlayKwMatcher = CliMatcher.KeywordMatcher( 'overlay',
      helpdesc='Specify overlay network five tuple and trace'
      ' both underlay and overlay paths' )
underlayKwMatcher = CliMatcher.KeywordMatcher( 'underlay',
      helpdesc='Specify overlay network five tuple but only trace underlay path' )

def overlayTracerouteGuard( mode, token ):
   if tracerouteCapabilities.overlaySupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

class TracerouteCmd( CliCommand.CliCommandClass ):
   syntax = ( 'traceroute [ VRF ] [ ip | ipv6 ] '
              '( DESTINATION '
                  '[ source ( IP_GEN_ADDR | INTF_WITH_IP ) ]'
                     ' [ interface INTF ] ) |'
            '( ( overlay | underlay ) [ resolve ] SRC_IP DST_IP SPORT DPORT PROTO '
               '[ max-hops MAX_HOPS ] [ source ( IP_GEN_ADDR | INTF_WITH_IP ) ] )' )
   data = {
            'traceroute': tracerouteKwMatcher,
            'VRF': VrfCli.VrfExprFactory( helpdesc='Trace route in a VRF',
                                          inclDefaultVrf=True ),
            'ip': 'IPv4 traceroute',
            'ipv6': 'IPv6 traceroute',
            'DESTINATION': dstMatcher,
            'overlay': CliCommand.Node( overlayKwMatcher,
                                        guard=overlayTracerouteGuard ),
            'underlay': CliCommand.Node( underlayKwMatcher,
                                        guard=overlayTracerouteGuard ),
            'resolve' : CliMatcher.KeywordMatcher( 'resolve',
               helpdesc='Resolve host names' ),
            'SRC_IP': IpGenAddrMatcher.IpGenAddrMatcher(
               'IP address of the source' ),
            'DST_IP': IpGenAddrMatcher.IpGenAddrMatcher(
               'IP address of the destination' ),
            'SPORT': CliMatcher.IntegerMatcher( 0, 65535,
               helpdesc='L4 source port' ),
            'DPORT': CliMatcher.IntegerMatcher( 0, 65535,
               helpdesc='L4 destination port' ),
            'PROTO': CliMatcher.EnumMatcher( {
               'udp' : 'UDP',
               'tcp' : 'TCP' } ),
            'max-hops': 'Maximum number of nexthop routes to trace',
            'MAX_HOPS': CliMatcher.IntegerMatcher( 0, 255,
               helpdesc='Maximum number of hops' ),
            'source': CliCommand.singleKeyword( 'source',
               helpdesc='specify source address or choose an address from specified '
               'interface' ),
            'IP_GEN_ADDR': IpGenAddrMatcher.ipGenAddrMatcher,
            'INTF': IntfCli.Intf.matcher,
            'interface': CliCommand.singleKeyword( 'interface',
                helpdesc='specify egress interface' ),
            'INTF_WITH_IP': CliCommand.Node( IntfCli.Intf.matcher )
          }
   handler = doTraceroute

BasicCliModes.ExecMode.addCommandClass( TracerouteCmd )

#------------------------------------------------------------------------------------
# The traceroute qos dscp command in router config mode.
#
# The full syntax of this command is:
#
#  traceroute qos dscp <dscpVal>
#------------------------------------------------------------------------------------
def setDscp( mode, args ):
   tracerouteConfig.dscpValue = args[ 'DSCP' ]

def noDscp( mode, args ):
   tracerouteConfig.dscpValue = tracerouteConfig.dscpValueDefault

DscpCliLib.addQosDscpCommandClass( BasicCliModes.GlobalConfigMode, setDscp, noDscp,
                                   tracerouteKwMatcher )

#------------------------------------------------------------------------------------
# The 'telnet' command, in enable and unprivileged modes.
#
# The full syntax of this command is:
#
#  telnet <destination> [<port>] [/source-interface <source>]
#  connect <destination> [<port>] [/source-interface <source>]
#------------------------------------------------------------------------------------
commonPorts = [
   (   7, 'echo',        'Echo' ),
   (   9, 'discard',     'Discard' ),
   (  13, 'daytime',     'Daytime' ),
   (  19, 'chargen',     'Character generator' ),
   (  20, 'ftp-data',    'FTP data connections' ),
   (  21, 'ftp',         'File Transfer Protocol' ),
   (  23, 'telnet',      'Telnet' ),
   (  25, 'smtp',        'Simple Mail Transport Protocol' ),
   (  37, 'time',        'Time' ),
   (  43, 'whois',       'Nicname' ),
   (  49, 'tacacs',      'TAC Access Control System' ),
   (  53, 'domain',      'Domain Name Service' ),
   (  70, 'gopher',      'Gopher' ),
   (  79, 'finger',      'Finger' ),
   (  80, 'www',         'World Wide Web', 'HTTP' ),
   ( 101, 'hostname',    'NIC hostname server' ),
   ( 109, 'pop2',        'Post Office Protocol v2' ),
   ( 110, 'pop3',        'Post Office Protocol v3' ),
   ( 111, 'sunrpc',      'Sun Remote Procedure Call' ),
   ( 113, 'ident',       'Ident Protocol' ),
   ( 119, 'nntp',        'Network News Transport Protocol' ),
   ( 179, 'bgp',         'Border Gateway Protocol' ),
   ( 194, 'irc',         'Internet Relay Chat' ),
   ( 496, 'pim-auto-rp', 'PIM Auto-RP' ),
   ( 512, 'exec',        'Exec', 'rsh' ),
   ( 513, 'login',       'Login', 'rlogin' ),
   ( 514, 'cmd',         'Remote commands', 'rcmd' ),
   ( 514, 'syslog',      'Syslog' ),
   ( 515, 'lpd',         'Printer service' ),
   ( 517, 'talk',        'Talk' ),
   ( 540, 'uucp',        'Unix-to-Unix Copy Program' ),
   ( 543, 'klogin',      'Kerberos login' ),
   ( 544, 'kshell',      'Kerberos shell' ) ]

portNameToDesc = {}
portNameToPortNum = {}

for portRecord in commonPorts:
   try:
      # pylint: disable-next=unbalanced-tuple-unpacking
      ( portNo, name, description, alias ) = portRecord
      helpdesc = '%s (%s, %d)' % ( description, alias, portNo )
   except ValueError:
      ( portNo, name, description ) = portRecord
      helpdesc = '%s (%d)' % ( description, portNo )

   portNameToDesc[ name ] = helpdesc
   portNameToPortNum[ name ] = portNo

def _srcArgs( mode, srcIntfName, dst ):
   def warn( message ):
      mode.addWarning( message )
      mode.addWarning( 'Telnet will use the default interface instead.' )

   if not srcIntfName:
      return []
   
   if srcIntfName not in intfStatusAll.intfStatus:
      warn( 'Interface %s has not been configured.' % srcIntfName )
      return []
   if intfStatusAll.intfStatus[ srcIntfName ].operStatus != 'intfOperUp':
      warn( 'Interface %s is not enabled.' % srcIntfName )
      return []

   # Work out address family of destination, and find an address in that family
   # on source interface to bind to
   proto = _resolveDestAddrFam( dst )
   srcIpAddr = None

   if ( proto == 'ipv4' or proto is None ) and srcIntfName in ipStatus.ipIntfStatus:
      # IPv4 address is acceptable. Do we have one?
      srcIpAddr = ipStatus.ipIntfStatus[ srcIntfName ].activeAddrWithMask.address

   if srcIpAddr is None and ( proto == 'ipv6' or proto is None ):
      # No acceptable IPv4 address. Do we have an IPv6 one?
      ip6Intf = ip6Status.intf.get( srcIntfName )
      if ip6Intf is not None:
         arnetIp6Addr = ip6Intf.highestSourceAddress()
         if arnetIp6Addr:
            srcIpAddr = arnetIp6Addr.stringValue

   if srcIpAddr and srcIpAddr != '0.0.0.0':
      # -b binds the local socket to a different interface than the one that 
      # connect() chooses naturally.
      return [ '-b', srcIpAddr ]
   else:
      def fmtProto( proto ):
         if proto == "ipv4":
            return "IP"
         if proto == "ipv6":
            return "IPv6"
         return "IP or IPv6"

      warn( 'Interface %s has no %s address.' % ( srcIntfName, fmtProto( proto ) ) )
      return []
     
def doTelnet( mode, args ):
   destination = args[ 'DESTINATION' ]
   if 'PORT_NUM' in args:
      port = args[ 'PORT_NUM' ]
   elif 'PORT_NAME' in args:
      port = portNameToPortNum[ args[ 'PORT_NAME' ] ]
   else:
      port = None
   srcIntf = args.get( 'INTF' )

   args = [ telnet.name(), '-E' ]
   # -E disables the escape character, so that users can't break into command mode.

   srcIntfName = None
   if srcIntf:
      srcIntfName = srcIntf.name
   else:
      protConf = networkUrlConfig.protocolConfig.get( 'telnet' )
      if protConf:
         srcIntf = protConf.srcIntf.get( DEFAULT_VRF )
         if srcIntf:
            srcIntfName = srcIntf.intfName

   if srcIntfName:
      args.extend( _srcArgs( mode, srcIntfName, destination ) )
   args.extend( [ '--', destination ] )
   if port:
      # If this resolves to localhost AND the port is blacklisted, do not
      # allow connecting
      addrInfo = []
      try:
         addrInfo = socket.getaddrinfo( destination, None )
      except socket.gaierror:
         mode.addError( 'Could not resolve host: ' + destination )
         return
      _, _, _, _, sockaddr = addrInfo[ 0 ]
      address = sockaddr[ 0 ]
      if IPAddress( address ).is_loopback():
         mode.addError( 'Loopback addresses are not supported' )
         return
      args.append( str( port ) )


   cliCmdExt = CmdExtension.getCmdExtender()

   try:
      p = cliCmdExt.subprocessPopen( args, mode.session, stdin=sys.stdin )
      p.communicate()
   except OSError as e:
      mode.addError( e.strerror )

class TelnetCmd( CliCommand.CliCommandClass ):
   syntax = ( '( telnet | connect ) DESTINATION [ PORT_NUM | PORT_NAME ] '
                                               '[ /source-interface INTF ]' )
   data = {
            'telnet': 'TELNET client',
            'connect': 'TELNET client',
            'DESTINATION': dstMatcher,
            'PORT_NUM': CliMatcher.IntegerMatcher( 0, 65535,
               helpdesc='Number of the port to use' ),
            'PORT_NAME': CliMatcher.EnumMatcher( portNameToDesc ),
            '/source-interface': CliMatcher.KeywordMatcher( '/source-interface',
               helpdesc='Telnet source interface',
               alternates=[ 'source-interface' ] ),
            'INTF': IntfCli.Intf.matcherWithIpSupport

          }
   handler = doTelnet

BasicCliModes.ExecMode.addCommandClass( TelnetCmd )

#------------------------------------------------------------------------------------
# The 'scp [-v] {-t | -f} <path>' command in exec mode.
# 
# This command is expected to only be run in response to a remote scp request which
# is why it is hidden.  If we implement a command to initiate an scp from EOS
# that command will not be hidden.  But since 'bash scp ...' works fine, why bother
# to implement an scp client command?
#
# scp is in EnableMode, so only users who authorize at a privilege level high
# enough to drop straight into EnableMode (i.e. level >= 2) will be able to
# use remote scp.
#------------------------------------------------------------------------------------
def printRemoteRequestError( msg ):
   # For CLI commands that run in response to remote requests (like in scp and sftp).
   # See comments in BUG401347 for why we need to do it like this.
   # In summary, mode.addWarning/Error generates extra newlines that make scp/sftp 
   # client unhappy.
   print( "%% %s" % msg )

def doScpServer( mode, args ):
   if not mode.privileged:
      printRemoteRequestError( 'The \'scp\' command is only'
                               ' available in privileged mode.' )
      return

   cmdline = args[ 'CMD_LINE' ]
   # Set cwd so that paths relative to /mnt/flash/ will work.
   fsroot = os.path.abspath( os.environ.get( 'FILESYSTEM_ROOT', '/mnt' ) )
   workingDir = os.path.join( fsroot, 'flash' )
   if not os.path.exists( workingDir ):
      workingDir = None
   splitArgs = shlex.split( cmdline )
   filePath = splitArgs[ -1 ]
   if '-t' not in splitArgs:
      # -t is used by scp to put files onto a switch, so we do not
      # attempt to check if the filepath exists in that case
      globPath = filePath
      if not filePath.startswith( '/' ) and workingDir:
         globPath = workingDir + '/' + globPath
      filePathList = glob.glob( globPath )
      if not filePathList:
         printRemoteRequestError( '%s: No such file or directory' % filePath )
         return
   else:
      filePathList = [ filePath, ]
   args = [ '/usr/bin/scp' ] + splitArgs[ :-1 ] + filePathList
   cliCmdExt = CmdExtension.getCmdExtender()
   try:
      p = cliCmdExt.subprocessPopen( args, mode.session, cwd=workingDir,
                                     stdin=sys.stdin, stdout=sys.stdout,
                                     stderr=sys.stderr )
      p.wait()
   except OSError as e:
      printRemoteRequestError( e.strerror )

class ScpServerCmd( CliCommand.CliCommandClass ):
   syntax = 'scp CMD_LINE'
   data = {
            'scp': 'Initiate a secure copy',
            'CMD_LINE': CliMatcher.StringMatcher( helpname='COMMAND LINE',
               helpdesc='Scp command line' )
          }
   hidden = True
   handler = doScpServer

BasicCliModes.ExecMode.addCommandClass( ScpServerCmd )

#------------------------------------------------------------------------------------
# Invokes the external sftp server in response to a remote sftp request.
#
# The way sftp works is that the client runs 'ssh2 <server host> -s sftp' which
# looks up the sftp subsystem on the remote (in /etc/ssh/sshd_config) and attempts to
# invoke it. Thus, the command that the Cli expects is the path to the sftp-server 
# specified in the sshd_config file, rather than just 'sftp'.
# 
# This command is expected to only be run in response to a remote sftp request which
# is why it is hidden.  
#
# sftp is in EnableMode, so only users who authorize at a privilege level high
# enough to drop straight into EnableMode (i.e. level >= 2) will be able to
# use remote sftp. Unprivileged users will receive "Connection closed" error
#------------------------------------------------------------------------------------
def doSftpServer( mode, args ):
   if not mode.privileged:
      # We need this CLI command because it will be called in response to a
      # remote sftp request by an unprivileged user. But we don't print anything
      # because it will make sftp client confused. See BUG415021.
      return
   args = [ '/usr/libexec/openssh/sftp-server' ] 
   cliCmdExt = CmdExtension.getCmdExtender()
   try:
      p = cliCmdExt.subprocessPopen( args, mode.session, stdin=sys.stdin )
      p.wait()
   except OSError as e:
      printRemoteRequestError( e.strerror )

class SftpServerCmd( CliCommand.CliCommandClass ):
   syntax = '/usr/libexec/openssh/sftp-server'
   data = {
            '/usr/libexec/openssh/sftp-server': 'Invoke the external sftp server'
          }
   hidden = True
   handler = doSftpServer

BasicCliModes.ExecMode.addCommandClass( SftpServerCmd )

#------------------------------------------------------------------------------------
# The 'ssh' client command (available in unprivileged mode)
#
# The full syntax of this command is:
#
#  ssh [-l <login>] [-y] [-v <1-2>] [-c <cipher list>] [-p <port>] <host/ip> [cmds]
#------------------------------------------------------------------------------------
# A CliSetExpression is used to implement the cipher options.
# -c (des, 3des, blowfish, ...)
# After the -c option is passed, the user is expected to input at least
# one of the cipher spec options (see minMembers=1).
# After one option is passed, user has the choice to pass additional
# cipher spec options or the other ssh options like -l, -p, etc
# Note that only one cipher option can be used with protocol version 1.
# User is allowed to pass multiple cipher specs but the cmd handler
# will dump an error message when it detects this and will run the ssh
# command with the default cipher.
cipherDict = {
   'blowfish'     : 1,
   '3des'         : 1,
   'des'          : 1,
   'aes128-ctr'   : 2,
   'aes192-ctr'   : 2,
   'aes256-ctr'   : 2,
   'arcfour256'   : 2,
   'arcfour128'   : 2,
   'aes128-cbc'   : 2,
   '3des-cbc'     : 2,
   'blowfish-cbc' : 2,
   'cast128-cbc'  : 2,
   'aes192-cbc'   : 2,
   'aes256-cbc'   : 2,
   'arcfour'      : 2, 
}
cipherSetDict = { k: '%s (v%d)' % ( k, v ) for k, v in cipherDict.items() }

kexAlgos = [
   'diffie-hellman-group-exchange-sha256',
   'diffie-hellman-group-exchange-sha1',
   'diffie-hellman-group14-sha1',
   'diffie-hellman-group1-sha1',
]

macs = [
   'hmac-md5',
   'hmac-sha1',
   'hmac-ripemd160',
   'hmac-sha1-96',
   'hmac-md5-96',
]

# Default version 1 cipher is 3des. For version 2, the default is an 
# ordered list of cipher options (aes128-ctr, aes192-ctr, aes256-ctr, ...).
defaultCipherV1 = '3des'

class SshOptions( CliCommand.CliExpression ):
   expression = ( '( -l LOGIN_NAME ) | '
                  '( -c CIPHERS ) | '
                  '( -m MAC ) | '
                  '( -v VERSION ) | '
                  '-y | ' # syslog
                  '( -o KexAlgorithms KEX ) | '
                  '( -o StrictHostKeyChecking ( yes | no ) ) | '
                  '( -p PORT )' )
   data = {
            '-l': CliCommand.singleKeyword( '-l', helpdesc='Login name' ),
            'LOGIN_NAME': CliMatcher.PatternMatcher( '.+', helpname='WORD', 
               helpdesc='Login name' ),
            '-c': CliCommand.singleKeyword( '-c',
               helpdesc='Cipher specification for encryption' ),
            'CIPHERS': CliCommand.setCliExpression( cipherSetDict, name='CIPHERS' ),
            '-m': CliCommand.singleKeyword( '-m',
               helpdesc='MAC specification for encryption' ),
            'MAC': CliCommand.setCliExpression( { k: k for k in macs }, name='MAC' ),
            '-v': CliCommand.singleKeyword( '-v',
               helpdesc='Protocol version to force' ),
            'VERSION': CliMatcher.IntegerMatcher( 1, 2, helpdesc='Version' ),
            '-y': CliCommand.singleKeyword( '-y',
               helpdesc='Have the ssh client redirect messages to syslog' ),
            '-o': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( '-o',
               helpdesc='Additional SSH option specification' ),
               maxMatches=expression.count( ' -o ' ) ),
            'KexAlgorithms': CliCommand.singleKeyword( 'KexAlgorithms',
               helpdesc='Key Exchange specification for encryption' ),
            'KEX': CliCommand.setCliExpression( { k: k for k in kexAlgos },
               name='KEX' ),
            'StrictHostKeyChecking': 
               CliCommand.singleKeyword( 'StrictHostKeyChecking', 
                  helpdesc='Enforce strictly checking remote hostkeys' ),
            'yes': 'Check remote server\'s hostkey before connecting',
            'no': 'Ignore remote server\'s hostkey',
            '-p': CliCommand.singleKeyword( '-p', helpdesc='Port' ),
            'PORT': CliMatcher.IntegerMatcher( 1, 65535,
               helpdesc='Number of the port to use' )
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'OPTIONS' in args:
         return
      options = []
      args[ 'OPTIONS' ] = options
      
      for keyword, value in argsList:
         if keyword == 'LOGIN_NAME':
            options.append( ( 'loginName', value ) )
         elif keyword == '-c' and args[ 'CIPHERS' ]:
            options.append( ( 'cipherSpec', args[ 'CIPHERS' ] ) )
         elif keyword == '-m' and args[ 'MAC' ]:
            options.append( ( 'macSpec', args[ 'MAC' ] ) )
         elif keyword == 'VERSION':
            options.append( ( 'version', value ) )
         elif keyword == 'KexAlgorithms' in args and args[ 'KEX' ]:
            options.append( ( 'kexSpec', args[ 'KEX' ] ) )
         elif keyword == '-y':
            options.append( ( 'syslog', True ) )
         elif keyword == 'StrictHostKeyChecking':
            if 'yes' in args:
               options.append( ( 'hostKeyChecking', 'yes' ) )
            elif 'no' in args:
               options.append( ( 'hostKeyChecking', 'no' ) )
         elif keyword == 'PORT':
            options.append( ( 'port', value ) )

def getKexStr( mode, kexSpec ):
   '''
   Parses the kex spec options passed in by the user, checking
   if the version is correct.

   Returns-
   - an empty string if default kex options have to be used
   - a comma separated list of kex options if user has passed
     multiple kex algorithms
   '''
   kexArgs = []
   for kex in kexSpec:
      if sshConfig.fipsRestrictions and kex not in FIPS_KEXS:
         mode.addWarning(
               '%s is disabled when using FIPS algorithms.' % ( kex, ) )
      else:
         kexArgs.append( kex )

   return ','.join( kexArgs )

def getMacStr( mode, macSpec, version ):
   '''
   Parses the mac spec options passed in by the user, checking
   if the version is correct. Since it only works for version 2,
   it will return a blank string if version is not 2.

   Returns-
   - an empty string if default mac options have to be used
   - a comma separated list of mac options if user has passed
     multiple macs
   '''

   if version == 1:
      mode.addWarning( 'MACs can only be used for'
                       ' protocol version 2' )
      return ''
   macArgs = []
   for _mac in macSpec:
      if sshConfig.fipsRestrictions and _mac not in FIPS_MACS:
         mode.addWarning(
               '%s is disabled when using FIPS algorithms.' % ( _mac, ) )
      else:
         macArgs.append( _mac )

   return ','.join( macArgs )

def getCipherStr( mode, cipherSpec, version ) :
   '''
   Parses the cipher spec options passed by the user, checking
   whether they are compatible with the version in question.
   cipherDict contains the compatibility information.
   In addition, it builds a string (of cipher options) that will be
   passed to the 'stock' ssh command.
  
   Returns -
   - an empty string if default cipher options have to be used
   - a single cipher for protocol version 1
     For example: 'blowfish', '3des' or 'des'
   - a comma separated list of cipher options if user has
     passed multiple ciphers (for protocol version 2). 
     For example: 'aes128-ctr,aes192-ctr,aes256-ctr'
   '''

   # For version 1, verify that only one cipher type has been passed
   # and that the cipher type is supported by protocol version 1.
   if ( version == 1 ) and ( len( cipherSpec ) > 1 ):
      mode.addWarning(
             'Only one cipher type can be specified '
             'for protocol version 1' )
      mode.addWarning(
             'Using default (%s)' % defaultCipherV1 )
      return ''
   # Verify that the cipher options passed are supported for
   # the version in question.
   # Also, build a string of comma separated cipher options.
   # For example: 'aes128-ctr,aes192-ctr,aes256-ctr'
   # In the case of version 1, a single cipher will be returned.
   cipherArgs = []
   for cipher in cipherSpec:
      # Get the protocol that this cipher is compatible with and
      # verify that this is same as that is being used.
      proto = cipherDict[ cipher ]
      if proto != version:
         mode.addWarning(
                'Invalid cipher. %s is not supported for protocol version %d'
                         % ( cipher, version ) )
         if version == 1:
            mode.addWarning(
                'Using default cipher (%s) for version %d'
                         % ( defaultCipherV1, version ) )
         else:
            mode.addWarning( 'Using default cipher list for version %d'
                             % ( version ) )
         return ''
      if sshConfig.fipsRestrictions and cipher not in FIPS_CIPHERS:
         mode.addWarning(
               '%s is disabled when using FIPS algorithms.' % ( cipher, ) )
         continue

      cipherArgs.append( cipher )

   # Return a string of comma separated cipher options.
   return ','.join( cipherArgs )

def strictHostKeyCheckOptions( vrfName ):
   '''
   Append the arguments for StrictHostKeyChecking and the
   related host key files.
   '''
   args = []
   args.append( '-oStrictHostKeyChecking=yes' )
   args.append( '-oUserKnownHostsFile=/etc/ssh/ssh_known_hosts' + \
               ('-' + vrfName if vrfName else '' ))
   args.append( '-oGlobalKnownHostsFile=/etc/ssh/ssh_known_hosts' + \
               ('-' + vrfName if vrfName else '' ))
   return args

def doSsh( mode, args ):
   # for the syslog variables
   vrfName = args.get( 'VRF' )
   host = args[ 'HOSTNAME' ]
   options = args.get( 'OPTIONS', [] )
   args = [ 'ssh' ]
   if not vrfName:
      vrfName = VrfCli.vrfMap.getCliSessVrf( mode.session )
      assert vrfName, 'name should never be None'
      if VrfCli.vrfMap.isDefaultVrf( vrfName ):
         vrfName = None

   # The version option needs to be processed outside the
   # for loop where we iterate over the options list.
   # The user can pass options in any order.
   # We need to know the version value before processing the
   # cipher option.
   # Now the version option could be specified after the
   # cipher options.
   # So we need to figure out the version value prior to
   # processing the cipher option while iterating the option list below.

   version = 2
   loginName = ''
   port = '22'
   for ( optName, optVal ) in options:
      if optName == 'version':
         version = optVal
         break
   if ( sshConfig.fipsRestrictions ) and ( version == 1 ):
      mode.addWarning( 'Protocol version 1 is disabled'
                       ' when using FIPS algorithms.' )
      version = 2
   # Now process the remaining options.
   if vrfName:
      args.extend( [ '-F', '/etc/ssh/ssh_config-' + vrfName ] )
   strictHostKeysSet = False
   for optName, optVal in options:
      if optName == 'version':
         args.append( '-' + str( version ) )

      elif optName == 'loginName':
         args.extend( [ '-l', optVal ] )
         loginName = optVal

      elif optName == 'syslog':
         args.extend( [ '-y', ] )

      elif optName == 'macSpec':
         macStr = getMacStr( mode, optVal, version )
         if macStr != '':
            args.append( '-oMACs=' + macStr )

      elif optName == 'kexSpec':
         kexStr = getKexStr( mode, optVal )
         if kexStr != '':
            args.append( '-oKexAlgorithms=' + kexStr )
      
      elif optName == 'cipherSpec':
         cipherStr = getCipherStr( mode, optVal, version )
         if cipherStr != '':
            if version == 1:
               args.append( '-oCipher=' + cipherStr )
            else:
               args.append( '-oCiphers=' + cipherStr )
      
      elif optName == 'port':
         args.extend( [ '-p', str( optVal ) ] )
         port = str( optVal )
      elif optName == 'hostKeyChecking':
         if optVal == 'no' and sshConfig.enforceCheckHostKeys:
            mode.addWarning( 'StrictHostKeyChecking is enforced' )
         elif optVal == 'yes':
            strictHostKeysSet = True
            args.extend( strictHostKeyCheckOptions( vrfName ) )
   if sshConfig.enforceCheckHostKeys and not strictHostKeysSet:
      args.extend( strictHostKeyCheckOptions( vrfName ) )

   # To know which bind address we ( may ) need to use, we resolve
   # the host address first
   try:
      addrList = socket.getaddrinfo( host, None, 0, socket.SOCK_STREAM )
   except socket.gaierror:
      mode.addError( 'Could not resolve host: ' + host )
      return

   _, _, _, _, sockaddr = addrList[ 0 ]
   # Loopback addresses are disabled for security reason
   address = sockaddr[ 0 ]
   if IPAddress( address ).is_loopback():
      mode.addError( 'Loopback addresses are not supported' )
      return

   # Pick the first address returned
   bindAddr = None
   if addrList[ 0 ][ 0 ] == socket.AF_INET6:
      # if the destination is an ipv6 address
      bindAddr = SysMgrLib.getIp6SrcIntfAddr( networkUrlConfig, ip6Status, 'ssh',
                                              vrfName=vrfName )
   else:
      bindAddr = SysMgrLib.getIp4SrcIntfAddr( networkUrlConfig, ipStatus, 'ssh',
                                              vrfName=vrfName )
   if bindAddr:
      args.extend( [ '-oBindAddress=%s' % bindAddr ] )

   # Flush out the error messages before running the ssh client.
   sys.stdout.flush()

   args.append( host )

   cliCmdExt = CmdExtension.getCmdExtender()

   if mode.session.aaaUser():
      uid = mode.session.aaaUser().uid
   else:
      # test
      uid = os.geteuid()

   userName = pwd.getpwuid( uid )[ 0 ]

   if loginName == '':
      loginName = userName
   try:
      Logging.log( SECURITY_SSH_CLIENT_CONNECTING,
                   userName, loginName, host, port )
      p = cliCmdExt.subprocessPopen( args, mode.session,
                                     stdin=sys.stdin, vrfName=vrfName )
      p.communicate()
   except OSError as e:
      mode.addError( e.strerror )
   finally:
      Logging.log( SECURITY_SSH_CLIENT_DISCONNECTED,
                   userName, loginName, host, port )

class SshCmd( CliCommand.CliCommandClass ):
   syntax = 'ssh [ VRF ] [ { OPTIONS } ] HOSTNAME'
   data = {
            'ssh': 'Open ssh connection',
            'VRF': VrfCli.VrfExprFactory( helpdesc='SSH in a VRF',
                                          inclDefaultVrf=True ),
            'OPTIONS': SshOptions,
            'HOSTNAME': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True,
                       helpname='Hostname or IPv4/IPv6 address',
                       helpdesc='Remote hostname or IP address' )
          }
   handler = doSsh

BasicCliModes.ExecMode.addCommandClass( SshCmd )

#-------------------------------------------------------------------------------
# Mount interface/status and ip/status
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global intfStatusAll, ipStatus, sshConfig, tracerouteConfig
   global networkUrlConfig, ip6Status, routeStatus, bridgingConfig
   global tracerouteCapabilities, vtiStatusDir

   intfStatusAll = LazyMount.mount( entityManager,
                                    'interface/status/all',
                                    'Interface::AllIntfStatusDir', 'r' )
   ipStatus = LazyMount.mount( entityManager, 'ip/status', 'Ip::Status', 'r' )
   ip6Status = LazyMount.mount( entityManager, 'ip6/status', 'Ip6::Status', 'r' )
   routeStatus= LazyMount.mount( entityManager, 'routing/hardware/route/status',
                                 'Routing::Hardware::RouteStatus', 'r' )
   bridgingConfig = LazyMount.mount( entityManager, 'bridging/config',
                                     'Bridging::Config', 'r' )
   sshConfig = LazyMount.mount( entityManager,
                                'mgmt/ssh/config',
                                'Mgmt::Ssh::Config', 'r' )
   networkUrlConfig = LazyMount.mount( entityManager,
                                       'mgmt/networkUrl/config',
                                       'Mgmt::NetworkUrl::Config',
                                       'r' )
   tracerouteConfig = ConfigMount.mount( entityManager,
                                         'mgmt/traceroute/config',
                                         'Mgmt::Traceroute::Config', 'w' )
   tracerouteCapabilities = LazyMount.mount( entityManager,
                                'mgmt/traceroute/capabilities',
                                'Mgmt::Traceroute::Capabilities', 'r' )
   vtiStatusDir = LazyMount.mount( entityManager,
                                'interface/status/eth/vxlan/',
                                'Vxlan::VtiStatusDir', 'r' )
