#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=ungrouped-imports

import Plugins
import LazyMount
import ConfigMount
import CliPlugin.ControllerdbLib
import BasicCli
import CliCommand
import CliMatcher
import ShowCommand
from CliPlugin import NetworkTopologyModels
from CliPlugin import TechSupportCli
from CliPlugin.ControllerdbLib import controllerGuard
from CliPlugin.ControllerCli import CvxConfigMode
from CliPlugin.ControllerCli import serviceKwMatcher

controllerConfig = None
status = None
config = None

matcherNetwork = CliMatcher.KeywordMatcher( 'network', 
      helpdesc='Network topology' )

nodeNetwork = CliCommand.Node( 
      matcher=matcherNetwork, 
      guard=controllerGuard )

matcherPhysicalTopology = CliMatcher.KeywordMatcher( 'physical-topology', 
      helpdesc='Physical network topology' )

matcherHostDetails = CliMatcher.KeywordMatcher( 'details', 
      helpdesc='Additional details' )
nodeHostDetailsHidden = CliCommand.Node(
      matcher=matcherHostDetails,
      hidden=True )

matcherHost = CliMatcher.KeywordMatcher( 'host', 
      helpdesc='Network host' )

matcherTopology = CliMatcher.KeywordMatcher( 'topology',
      helpdesc='Topology aggregation service' )

#--------------------------------------------------------------------------------
# service topology
#--------------------------------------------------------------------------------
class ServiceTopologyCmd( CliCommand.CliCommandClass ):
   syntax = 'service topology'
   noOrDefaultSyntax = syntax
   data = {
      'service': serviceKwMatcher,
      'topology': matcherTopology,
   }

   handler = "NetworkTopologyServiceHandler.gotoServiceTopology"
   noOrDefaultHandler = "NetworkTopologyServiceHandler.clearServiceTopology"

CvxConfigMode.addCommandClass( ServiceTopologyCmd )

class NetworkPhysicalTopologyNeighborsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show network physical-topology neighbors [ host HOSTNAME ]'
   data = {
      'network': nodeNetwork,
      'physical-topology': matcherPhysicalTopology,
      'neighbors': 'Show physical topology neighbors',
      'host': matcherHost,
      'HOSTNAME': CliMatcher.PatternMatcher( helpdesc='Name of the host', 
         helpname='HOST', pattern=r'.+' ),
   }
   handler = "NetworkTopologyServiceHandler.showNeighbors"
   cliModel = NetworkTopologyModels.TopologyNeighbors

BasicCli.addShowCommandClass( NetworkPhysicalTopologyNeighborsCmd )

class NetworkPhysicalTopologyHostsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show network physical-topology hosts [ host HOSTNAME ] [ details ]'
   data = {
      'network': nodeNetwork,
      'physical-topology': matcherPhysicalTopology,
      'host': matcherHost,
      'hosts': 'Show physical topology hosts',
      'HOSTNAME': CliMatcher.PatternMatcher( helpdesc='Name of the host', 
         helpname='HOST', pattern=r'.+' ),
      'details': nodeHostDetailsHidden,
   }
   handler = "NetworkTopologyServiceHandler.showHosts"
   cliModel = NetworkTopologyModels.TopologyHosts

BasicCli.addShowCommandClass( NetworkPhysicalTopologyHostsCmd )

# ----------------------------------------------------------------------
# register show tech-support callback
# ----------------------------------------------------------------------
showTechCmds = [
   'show network physical-topology hosts',
   'show network physical-topology neighbors',
]

TechSupportCli.registerShowTechSupportCmd(
   '2023-12-19 17:17:06',
   cmds=showTechCmds,
   cmdsGuard=lambda: controllerConfig and controllerConfig.enabled )

#--------------------------------------------------------------------------------
# Mount global topology status from Controllerdb
#--------------------------------------------------------------------------------

def doControllerMounts( controllerdbEm ):
   global status
   if controllerdbEm:
      status = LazyMount.mount( controllerdbEm,
                                "topology/version3/global/status",
                                "NetworkTopologyAggregatorV3::Status",
                                "r" )
   else:
      status = None

@Plugins.plugin( requires=( "ControllerdbMgr", ) )
def Plugin( entityManager ):
   global config
   global controllerConfig

   controllerConfig = LazyMount.mount( entityManager,
                             'controller/config',
                             'Controllerdb::Config',
                             'r' )
   config = ConfigMount.mount( entityManager, "topology/aggregator/config",
         "NetworkTopologyAggregatorV3::Config", "w" )
   
   CliPlugin.ControllerdbLib.registerNotifiee( doControllerMounts )
