# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import time
from ArnetModel import IpGenericAddress
from CliModel import Enum, Dict, Float, Int, List, Model, Str

NTP_PEER_CONDITIONS = { 'reject': ' ', 'falsetick': 'x', 'excess': '.',
                        'outlier': '-', 'candidate': '+', 'backup': '#',
                        'sys.peer': '*', 'pps.peer': 'o' }

class NtpPeer( Model ):
   condition = Enum( values=list( NTP_PEER_CONDITIONS ),
                     help='Select field of the peer status word' )
   peerIpAddr = IpGenericAddress( help='IP of the peer' )
   refid = Str( help='Association ID or KISS code' )
   stratumLevel = Int( help='Accuracy of the clock source (1 is best)' )
   peerType = Enum ( values=( 'unicast', 'broadcast', 'local' ),
                 help='Type of peer' )
   lastReceived = Float( help='Timestamp of last received packet' )
   pollInterval = Int( help='Poll interval (seconds)' )
   reachabilityHistory = List( valueType=bool,
                               help='A list of poll history success' )
   delay = Float( help='Roundtrip delay (milliseconds)' )
   offset = Float( help='offset of server relative to this host (milliseconds)' )
   jitter = Float( help='jitter (milliseconds)' )

class NtpPeers( Model ):
   peers = Dict( valueType=NtpPeer,
                 help="A mapping between peer host name and peer host informaion" )

   def render( self ):
      as01 = [ '0', '1' ]
      now = time.time()
      # pylint: disable-next=consider-using-f-string
      print( ' {:^15} {:^14} {:^2} {:1} {:>4} {:>5} {:^5} {:>7} {:>8} {:>7}'.
             format( 'remote', 'refid', 'st', 't', 'when', 'poll', 'reach',
                     'delay', 'offset', 'jitter' ) )
      print( '=' * 78 )
      for hostname, peer in self.peers.items():
         # Reachability history is a list of `True/False`. Make binary, then integer.
         reach = int( ''.join( as01[ x ] for x in peer.reachabilityHistory ), 2 )
         # pylint: disable-next=consider-using-f-string
         print( '{tally}{remote:15.15} {refid:14.14} {stratumLevel:>2} '
                '{peerType[0]:1} {lastReceivedDelta:>4.0f} {pollInterval:>5} '
                '{reach:^5o} {delay:>7} {offset:>8} {jitter:>7}'.
                format( tally=NTP_PEER_CONDITIONS[ peer.condition ],
                        remote=hostname, reach=reach,
                        lastReceivedDelta=( now - peer.lastReceived ),
                        **peer.toDict() ) )

