# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import datetime
import BasicCli
import CliGlobal
import LazyMount
import ShowCommand
import CliModel
import CliToken.Hardware
from Plugins import SkipPluginModule
from Toggles import OliveToggleLib

if not OliveToggleLib.toggleOliveShowCmdEnabled():
   raise SkipPluginModule( 'Olive Show Cmd Toggle Not Enabled' )

# The gloval var dict to hold any Sysdb objects that we need
globalVar = CliGlobal.CliGlobal( dict( status=None ) )

class TpmVersion( CliModel.Model ):
   family = CliModel.Str( help="The main type/version of the TPM" )
   vendorStr = CliModel.Str(
         help="The vendor id string of the TPM manufacturer" )
   vendorId = CliModel.Str(
         help="A vendor identifier that is unique to each TPM manufacturer" )
   specLevel = CliModel.Int( help="The specification level" )
   specRevision = CliModel.Float( help="The specification revision" )
   specDate = CliModel.Str( help="The month, day, and year in which the "
                            "TPM specification was published" )

   def render( self ):
      result = ( "Family: %s\n"
                 "Vendor: %s\n"
                 "Vendor ID: %s\n"
                 "Revision: %1.2f\n"
                 "Date: %s" %
                    ( self.family,
                      self.vendorStr,
                      self.vendorId,
                      self.specRevision,
                      self.specDate ) )
      print( result )

def doShowTpmVersion( mode, args ):
   tpmVersion = globalVar.status.tpmVersion
   tpmYear = tpmVersion.tpmSpecYear
   tpmDate = []
   # If the version data is not populated, inform the user
   # we can tell if it's not initialized by looking at whether
   # the tpmSpecfamily is an empty string or not
   if not globalVar.status.tpmVersion.initialized():
      mode.addError( "TPM version data not yet populated" )
      return None
   # The datetime module will throw an error if the year is 0
   if tpmYear != 0:
      tpmDay = tpmVersion.tpmSpecDay - 1 # The first day is already accounted for
      beginningDate = datetime.date( tpmYear, 1, 1 )
      wholeDate = beginningDate + datetime.timedelta( days=tpmDay )
      tpmDate = "%s-%s-%s" % ( str( wholeDate.day ),
                               str( wholeDate.month ),
                               str( wholeDate.year ) )
   return TpmVersion( family=tpmVersion.tpmSpecFamily,
                      vendorStr=tpmVersion.tpmVendorStr,
                      vendorId=tpmVersion.tpmVendorId,
                      specLevel=tpmVersion.tpmSpecLevel,
                      specRevision=tpmVersion.tpmSpecRevision,
                      specDate=tpmDate )

# -----------------------------------------------------
# The "show hardware license tpm" command
# -----------------------------------------------------
class ShowTpmVersion( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hardware license tpm'
   data = {
      'hardware': CliToken.Hardware.hardwareForShowMatcher,
      'license': 'Used to verify Arista hardware',
      'tpm': 'Trusted Platform Module chip'
   }
   handler = doShowTpmVersion
   cliModel = TpmVersion

BasicCli.addShowCommandClass( ShowTpmVersion )

def Plugin( em ):
   # Lazy mount the Olive::status
   globalVar.status = LazyMount.mount( em, 'hardware/olive/status',
                                       'Olive::Status', 'r' )
