#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
from CliPlugin import OpenConfigCliLib
from CliPlugin import Gnmi
from CliPlugin import IntfCli
from CliPlugin import VrfCli
import CliToken.OpenConfigTokens
import ConfigMount
import HostnameCli
from IpLibConsts import DEFAULT_VRF
import Tac

gnmiConfig = None

matcherProfile = CliMatcher.KeywordMatcher( 'profile',
      helpdesc='Configure SSL profile' )
matcherSsl = CliMatcher.KeywordMatcher( 'ssl',
      helpdesc='Configure SSL related options' )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable protocol',
   }

   @staticmethod
   def handler( mode, args ):
      mode.config_.grpctunnelClients[ mode.name ].enabled = False
      OpenConfigCliLib.updateLevelEnabledFlag( gnmiConfig )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.config_.grpctunnelClients[ mode.name ].enabled = True
      OpenConfigCliLib.updateLevelEnabledFlag( gnmiConfig )

Gnmi.GrpctunnelTransportConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] vrf VRF
#--------------------------------------------------------------------------------
class VrfCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF'
   noOrDefaultSyntax = 'vrf ...'
   data = {
      'vrf': 'Configure VRF',
      'VRF': CliMatcher.DynamicNameMatcher( VrfCli.getVrfNames, 'VRF name' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.config_.grpctunnelClients[ mode.name ].vrf = args[ 'VRF' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.config_.grpctunnelClients[ mode.name ].vrf = DEFAULT_VRF

Gnmi.GrpctunnelTransportConfigMode.addCommandClass( VrfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel ssl profile PROFILE
#--------------------------------------------------------------------------------
class TunnelSslProfileProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel ssl profile PROFILE'
   noOrDefaultSyntax = 'tunnel ssl profile ...'
   data = {
      'tunnel': 'Configure gRPC-tunnel SSL related options',
      'ssl': matcherSsl,
      'profile': matcherProfile,
      'PROFILE': Gnmi.profileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE' ]
      mode.config_.grpctunnelClients[ mode.name ].tunnelSslProfile = profileName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.config_.grpctunnelClients[ mode.name ].tunnelSslProfile = ''

Gnmi.GrpctunnelTransportConfigMode.addCommandClass( TunnelSslProfileProfilenameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] gnmi ssl profile PROFILE
#--------------------------------------------------------------------------------
class GnmiSslProfileProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'gnmi ssl profile PROFILE'
   noOrDefaultSyntax = 'gnmi ssl profile ...'
   data = {
      'gnmi': 'Configure gNMI SSL related options',
      'ssl': matcherSsl,
      'profile': matcherProfile,
      'PROFILE': Gnmi.profileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE' ]
      mode.config_.grpctunnelClients[ mode.name ].gnmiSslProfile = profileName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.config_.grpctunnelClients[ mode.name ].gnmiSslProfile = ''

Gnmi.GrpctunnelTransportConfigMode.addCommandClass( GnmiSslProfileProfilenameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] destination HOST port PORT
#--------------------------------------------------------------------------------
class DestinationCmd( CliCommand.CliCommandClass ):
   syntax = 'destination HOST port PORT'
   noOrDefaultSyntax = 'destination ...'
   data = {
      'destination': 'Configure dial-out destination',
      'HOST': HostnameCli.IpAddrOrHostnameMatcher(
                     ipv6=True, helpdesc='Destination IP address or hostname' ),
      'port': 'Select destination port',
      'PORT': CliToken.OpenConfigTokens.matcherPortValue,
   }

   @staticmethod
   def handler( mode, args ):
      hostPort = Tac.Value( 'Arnet::HostAndPort',
            hostname=args[ 'HOST' ], port=args[ 'PORT' ] )
      mode.config_.grpctunnelClients[ mode.name ].destinationHostAndPort = hostPort

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      hostPort = Tac.Value( 'Arnet::HostAndPort', hostname='', port=0 )
      mode.config_.grpctunnelClients[ mode.name ].destinationHostAndPort = hostPort

Gnmi.GrpctunnelTransportConfigMode.addCommandClass( DestinationCmd )

#--------------------------------------------------------------------------------
# [ no | default ] local interface INTF [ port PORT ]
#--------------------------------------------------------------------------------
class LocalInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'local interface INTF [ port PORT ]'
   noOrDefaultSyntax = 'local ...'
   data = {
      'local': 'Configure local interface to dial out from',
      'interface': 'Configure local interface',
      'INTF': IntfCli.Intf.matcherWithIpSupport,
      'port': 'Select local port',
      'PORT': CliToken.OpenConfigTokens.matcherPortValue,
   }

   @staticmethod
   def handler( mode, args ):
      intf = args[ 'INTF' ].name
      port = args.get( 'PORT', 0 )
      hostPort = Tac.Value( 'Arnet::HostAndPort', hostname=intf, port=port )
      mode.config_.grpctunnelClients[ mode.name ].localInterfaceAndPort = hostPort

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      hostPort = Tac.Value( 'Arnet::HostAndPort', hostname='', port=0 )
      mode.config_.grpctunnelClients[ mode.name ].localInterfaceAndPort = hostPort

Gnmi.GrpctunnelTransportConfigMode.addCommandClass( LocalInterfaceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] target ( serial-number | TARGET_ID )
#--------------------------------------------------------------------------------
class TargetIdCmd( CliCommand.CliCommandClass ):
   syntax = 'target ( serial-number | TARGET_ID )'
   noOrDefaultSyntax = 'target ...'
   data = {
      'target': 'Configure target ID',
      'serial-number': 'Use serial number as target ID',
      'TARGET_ID': CliMatcher.StringMatcher( helpdesc='Target ID', helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      targetId = args.get( 'TARGET_ID', 'serial-number' )
      # We may support these target IDs in the future.
      reservedKeywords = { 'system-mac-address', 'hostname' }
      if targetId in reservedKeywords:
         mode.addError( f"target ID cannot be '{targetId}'" )
         return
      mode.config_.grpctunnelClients[ mode.name ].targetId = targetId

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.config_.grpctunnelClients[ mode.name ].targetId = ''

Gnmi.GrpctunnelTransportConfigMode.addCommandClass( TargetIdCmd )

def Plugin( entityManager ):
   global gnmiConfig

   gnmiConfig = ConfigMount.mount( entityManager, "mgmt/gnmi/config",
                                   "Gnmi::Config", "w" )

