# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
from CliMatcher import DynamicNameMatcher, PatternMatcher, IntegerMatcher
import CliParserCommon
from CliPlugin import ModelsCli
from CliPlugin import VrfCli
import HostnameCli

import CliToken.OpenConfigTokens
import Toggles.OpenConfigToggleLib as OCToggle
import Toggles.RoutingLibToggleLib as RoutingLibToggle

#--------------------------------------------------------------------------------
# AFT
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# [ no | default ] ipv4-unicast
#--------------------------------------------------------------------------------
class AFTIPv4UnicastCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv4-unicast'
   noOrDefaultSyntax = syntax
   data = {
      'ipv4-unicast': 'Enable ipv4-unicast',
   }
   handler = ModelsCli.aftIpv4Unicast
   noOrDefaultHandler = ModelsCli.noAftIpv4Unicast

ModelsCli.ProviderAFTConfigMode.addCommandClass( AFTIPv4UnicastCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6-unicast
#--------------------------------------------------------------------------------
class AFTIPv6UnicastCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6-unicast'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6-unicast': 'Enable ipv6-unicast',
   }
   handler = ModelsCli.aftIpv6Unicast
   noOrDefaultHandler = ModelsCli.noAftIpv6Unicast

ModelsCli.ProviderAFTConfigMode.addCommandClass( AFTIPv6UnicastCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mpls
#--------------------------------------------------------------------------------
class AFTMplsCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': 'Enable MPLS',
   }
   handler = ModelsCli.aftMpls
   noOrDefaultHandler = ModelsCli.noAftMpls

if OCToggle.toggleOCLfibAftEnabled():
   ModelsCli.ProviderAFTConfigMode.addCommandClass( AFTMplsCmd )

# --------------------------------------------------------------------------------
# [ no | default ] route-summary
# --------------------------------------------------------------------------------
class AFTRouteSummaryCmd( CliCommand.CliCommandClass ):
   syntax = 'route-summary'
   noOrDefaultSyntax = syntax
   data = {
      'route-summary': 'Enable OpenConfig route summary',
   }
   handler = ModelsCli.aftRouteSummary
   noOrDefaultHandler = ModelsCli.noAftRouteSummary

if RoutingLibToggle.toggleRouteSummaryPublishEnabled():
   ModelsCli.ProviderAFTConfigMode.addCommandClass( AFTRouteSummaryCmd )

#--------------------------------------------------------------------------------
# Smash
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# [ no | default ] path SMASHPATH [ disabled ]
#--------------------------------------------------------------------------------
class PathSmashCmd( CliCommand.CliCommandClass ):
   syntax = 'path SMASHPATH [ disabled ]'
   noOrDefaultSyntax = syntax
   data = {
      'path': 'Configure a path',
      'SMASHPATH': PatternMatcher( pattern=r'.+',
                                   helpdesc='Smash Path',
                                   helpname='PATH'),
      'disabled': 'Disabled Smash path for Octa',
   }
   handler = ModelsCli.setSmashPath
   noOrDefaultHandler = ModelsCli.noSmashPath

ModelsCli.ProviderSmashConfigMode.addCommandClass( PathSmashCmd )

#--------------------------------------------------------------------------------
# Sysdb
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# [ no | default ] path SYSDBPATH disabled
#--------------------------------------------------------------------------------
class PathSysdbCmd( CliCommand.CliCommandClass ):
   syntax = 'path SYSDBPATH disabled'
   noOrDefaultSyntax = syntax
   data = {
      'path': 'Configure a path',
      'SYSDBPATH': PatternMatcher( pattern=r'.+',
                                   helpdesc='Sysdb Path',
                                   helpname='PATH'),
      'disabled': 'Disabled Sysdb path for Octa',
   }
   handler = ModelsCli.setSysdbPath
   noOrDefaultHandler = ModelsCli.noSysdbPath

ModelsCli.ProviderSysdbConfigMode.addCommandClass( PathSysdbCmd )

#--------------------------------------------------------------------------------
# IPFIX
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# [ no | default ] listener ADDR port PORT
#--------------------------------------------------------------------------------
class IPFIXListenerCmd( CliCommand.CliCommandClass ):
   syntax = 'listener ADDRESS [ port PORT ]'
   noOrDefaultSyntax = 'listener ...'
   data = {
      'listener': 'Configure a listener for IPFIX messages',
      'ADDRESS': HostnameCli.IpAddrOrHostnameMatcher( ipv6=False, 
                     helpdesc='Listening IP address or hostname' ),
      'port': 'Select the listening port for the collector',
      'PORT': CliToken.OpenConfigTokens.matcherPortValue,
   }
   handler = ModelsCli.setIPFIXListener
   noOrDefaultHandler = ModelsCli.defaultIPFIXListenerHandler

class IPFIXDomainCmd( CliCommand.CliCommandClass ):
   syntax = 'domain NAME'
   noOrDefaultSyntax = 'domain ...'
   data = {
      'domain': 'Configure a domain for IPFIX messages',
      'NAME': PatternMatcher( pattern=r'.+',
                              helpdesc='Collector domain name',
                              helpname='NAME' ),
   }
   handler = ModelsCli.setIPFIXDomain
   noOrDefaultHandler = ModelsCli.defaultIPFIXDomainHandler

class IPFIXFlowCmd( CliCommand.CliCommandClass ):
   # This syntax means we must have at least one of table size or lifetime
   syntax = 'flow ( table size SIZE lifetime LIFETIME seconds ) |'\
                 '( table size SIZE ) |'\
                 '( lifetime LIFETIME seconds )'
   noOrDefaultSyntax = 'flow ...'
   data = {
      'flow': 'Configure internal collection parameters for IPFIX messages',
      'table': 'Configure flow table',
      'size': 'Maximum size of flow table',
      'SIZE': IntegerMatcher( lbound=1,
                              ubound=0xFFFFFFFF,
                              helpdesc='Maximum size of flow table',
                              helpname='SIZE' ),
      'lifetime': 'Configure maximum size of flow table',
      'LIFETIME': IntegerMatcher( lbound=1,
                                  ubound=0xFFFFFFFF,
                                  helpdesc='Number of seconds to keep flow data',
                                  helpname='LIFETIME' ),
      'seconds': 'Seconds',
   }
   handler = ModelsCli.setIPFIXFlow
   noOrDefaultHandler = ModelsCli.defaultIPFIXFlowHandler

ModelsCli.ProviderIPFIXConfigMode.addCommandClass( IPFIXListenerCmd )
ModelsCli.ProviderIPFIXConfigMode.addCommandClass( IPFIXDomainCmd )
ModelsCli.ProviderIPFIXConfigMode.addCommandClass( IPFIXFlowCmd )

#--------------------------------------------------------------------------------
# sFlow
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# [ default ] listener ADDR port PORT
#--------------------------------------------------------------------------------
class SflowListenerCmd( CliCommand.CliCommandClass ):
   syntax = 'listener ADDRESS [ port PORT ]'
   noOrDefaultSyntax = 'listener ...'
   data = {
      'listener': 'Configure a listener for sFlow messages',
      'ADDRESS': HostnameCli.IpAddrOrHostnameMatcher( ipv6=False, 
                     helpdesc='Listening IP address or hostname' ),
      'port': 'Select the listening port for the collector',
      'PORT': CliToken.OpenConfigTokens.matcherPortValue,
   }
   handler = ModelsCli.setSflowListener
   noOrDefaultHandler = ModelsCli.defaultSflowListenerHandler

class SflowDomainCmd( CliCommand.CliCommandClass ):
   syntax = 'domain NAME'
   noOrDefaultSyntax = 'domain ...'
   data = {
      'domain': 'Configure a domain for sFlow messages',
      'NAME': PatternMatcher( pattern=r'.+',
                              helpdesc='Collector domain name',
                              helpname='NAME' ),
   }
   handler = ModelsCli.setSflowDomain
   noOrDefaultHandler = ModelsCli.defaultSflowDomainHandler

class SflowSamplesCmd( CliCommand.CliCommandClass ):
   # This syntax means we must have at least one of table size or lifetime
   syntax = 'samples ( table size SIZE [ lifetime LIFETIME seconds ] ) |'\
                    '( lifetime LIFETIME seconds )'
   noOrDefaultSyntax = 'samples ...'
   data = {
      'samples': 'Configure internal collection parameters for sFlow messages',
      'table': 'Configure flow table',
      'size': 'Maximum size of flow table',
      'SIZE': IntegerMatcher( lbound=1,
                              ubound=0xFFFFFFFF,
                              helpdesc='Maximum size of flow table',
                              helpname='SIZE' ),
      'lifetime': 'Configure maximum size of flow table',
      'LIFETIME': IntegerMatcher( lbound=1,
                                  ubound=0xFFFFFFFF,
                                  helpdesc='Number of seconds to keep flow data',
                                  helpname='LIFETIME' ),
      'seconds': 'Seconds',
   }
   handler = ModelsCli.setSflowSamples
   noOrDefaultHandler = ModelsCli.defaultSflowSamplesHandler

ModelsCli.ProviderSflowConfigMode.addCommandClass( SflowListenerCmd )
ModelsCli.ProviderSflowConfigMode.addCommandClass( SflowDomainCmd )
ModelsCli.ProviderSflowConfigMode.addCommandClass( SflowSamplesCmd )

#--------------------------------------------------------------------------------
# IS-IS
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# [ no | default ] link-state-database
#--------------------------------------------------------------------------------
class IsisLinkStateDatabaseCmd( CliCommand.CliCommandClass ):
   syntax = 'link-state-database'
   noOrDefaultSyntax = syntax
   data = {
      'link-state-database': 'Enable link-state-database',
   }
   handler = ModelsCli.isisLinkStateDatabase
   noOrDefaultHandler = ModelsCli.noIsisLinkStateDatabase

ModelsCli.ProviderIsisConfigMode.addCommandClass( IsisLinkStateDatabaseCmd )

#--------------------------------------------------------------------------------
# OSPF
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# [ no | default ] link-state-database [ VRF ]
#--------------------------------------------------------------------------------
class OspfLinkStateDatabaseCmd( CliCommand.CliCommandClass ):
   syntax = 'link-state-database [ VRF ]'
   noOrDefaultSyntax = syntax
   data = {
      'link-state-database': 'Enable link-state-database',
      'VRF': VrfCli.VrfExprFactory( helpdesc='VRF name',
                                    inclDefaultVrf=True,
                                    inclAllVrf=True ),
   }
   handler = ModelsCli.ospfLinkStateDatabase
   noOrDefaultHandler = ModelsCli.noOspfLinkStateDatabase

ModelsCli.ProviderOspfConfigMode.addCommandClass( OspfLinkStateDatabaseCmd )

#--------------------------------------------------------------------------------
# BGP
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# [ no | default ] bgp-rib
#--------------------------------------------------------------------------------
class BgpRibCmd( CliCommand.CliCommandClass ):
   syntax = 'bgp-rib'
   noOrDefaultSyntax = syntax
   data = {
      'bgp-rib': 'Enable streaming of BGP locRib',
   }
   handler = ModelsCli.gotoProviderBgpRibConfigMode
   noOrDefaultHandler = ModelsCli.noProviderBgpRibConfigMode

ModelsCli.ProviderBgpConfigMode.addCommandClass( BgpRibCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv4-unicast
#--------------------------------------------------------------------------------
class BgpRibIPv4UnicastCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv4-unicast'
   noOrDefaultSyntax = syntax
   data = {
      'ipv4-unicast': 'Enable ipv4-unicast',
   }
   handler = ModelsCli.bgpRibIpv4Unicast
   noOrDefaultHandler = ModelsCli.noBgpRibIpv4Unicast

ModelsCli.ProviderBgpRibConfigMode.addCommandClass( BgpRibIPv4UnicastCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6-unicast
#--------------------------------------------------------------------------------
class BgpRibIPv6UnicastCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6-unicast'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6-unicast': 'Enable ipv6-unicast',
   }
   handler = ModelsCli.bgpRibIpv6Unicast
   noOrDefaultHandler = ModelsCli.noBgpRibIpv6Unicast

ModelsCli.ProviderBgpRibConfigMode.addCommandClass( BgpRibIPv6UnicastCmd )

#--------------------------------------------------------------------------------
# Macsec
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# [ no | default ] mka
#--------------------------------------------------------------------------------
class MacsecMkaCmd( CliCommand.CliCommandClass ):
   syntax = 'mka'
   noOrDefaultSyntax = syntax
   data = {
      'mka': 'Enable mka',
   }
   handler = ModelsCli.setMacsecOptions
   noOrDefaultHandler = ModelsCli.resetMacsecOptions

ModelsCli.ProviderMacsecConfigMode.addCommandClass( MacsecMkaCmd )

#--------------------------------------------------------------------------------
# [ no | default ] interfaces
#--------------------------------------------------------------------------------
class MacsecInterfacesCmd( CliCommand.CliCommandClass ):
   syntax = 'interfaces'
   noOrDefaultSyntax = syntax
   data = {
      'interfaces': 'Enable interfaces',
   }
   handler = ModelsCli.setMacsecOptions
   noOrDefaultHandler = ModelsCli.resetMacsecOptions

ModelsCli.ProviderMacsecConfigMode.addCommandClass( MacsecInterfacesCmd )

#--------------------------------------------------------------------------------
# Configuration
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# [ no | default ] updates
#--------------------------------------------------------------------------------
class ConfigurationUpdatesCmd( CliCommand.CliCommandClass ):
   syntax = 'updates'
   noOrDefaultSyntax = syntax
   data = { 
      'updates': 'Enable configuration update telemetry',
   }   
   handler = ModelsCli.configurationUpdates
   noOrDefaultHandler = ModelsCli.noConfigurationUpdates

ModelsCli.ProviderConfigurationConfigMode.addCommandClass( 
                                                   ConfigurationUpdatesCmd )

#--------------------------------------------------------------------------------
# [ no | default ] history limit
#--------------------------------------------------------------------------------
class HistoryLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'history limit HISTORY_LIMIT'
   noOrDefaultSyntax = 'history limit ...'
   data = {
      'history': 'Configure history of configuration diffs',
      'limit': 'Set the size of configuration diff history. Default is 20',
      'HISTORY_LIMIT': IntegerMatcher( 1, 100, helpdesc='Value of history limit' ),
   }
   handler = ModelsCli.setHistoryLimit
   noOrDefaultHandler = ModelsCli.noSetHistoryLimit

ModelsCli.ProviderConfigurationConfigMode.addCommandClass( HistoryLimitCmd )

#--------------------------------------------------------------------------------
# models
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# [ no | default ] path PATH disabled
#--------------------------------------------------------------------------------

class ModelPathCmd( CliCommand.CliCommandClass ):
   syntax = 'path PATH disabled'
   noOrDefaultSyntax = 'path PATH ...'
   data = {
      'path': 'Configure a YANG path',
      'PATH': PatternMatcher( pattern=r'.+',
                              helpdesc='YANG path',
                              helpname='WORD'),
      'disabled': 'Disable path for the YANG tree',
   }
   handler = ModelsCli.addModelPath
   noOrDefaultHandler = ModelsCli.noModelPath

ModelsCli.ModelsConfigMode.addCommandClass( ModelPathCmd )

#--------------------------------------------------------------------------------
# modules
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# [ no | default ] module group MODULE_GROUP exclusive
#--------------------------------------------------------------------------------

class ModuleGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'module group MODULE_GROUP exclusive'
   noOrDefaultSyntax = 'module group MODULE_GROUP ...'
   data = {
      'module': 'Configure YANG module',
      'group': 'Configure YANG module group',
      'MODULE_GROUP': DynamicNameMatcher(
         lambda mode: ModelsCli.supportedModuleGroups,
         'YANG module group',
         helpname='WORD',
         extraEmptyTokenCompletionFn=lambda mode, context:
            [ CliParserCommon.Completion(
                 moduleGroup, f'{moduleGroup} module group' )
              for moduleGroup in ModelsCli.supportedModuleGroups ] ),
      'exclusive': 'Exclusively load YANG module group',
   }
   handler = ModelsCli.addModuleGroup
   noOrDefaultHandler = ModelsCli.noModuleGroup

if OCToggle.toggleOCExclusiveModuleGroupSegmentationToggleEnabled():
   ModelsCli.ModulesConfigMode.addCommandClass( ModuleGroupCmd )
