# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import glob
import os
from natsort import natsorted

import ArnetModel
import CliModel
from CliPlugin import AclCli
from CliPlugin.AclCliModel import AllAclList
import IntfModels
import LazyMount
import AuthnUserPriorityCli
import Toggles.OpenConfigToggleLib as OCToggle

aclCpConfig = None
aclStatus = None
aclCheckpoint = None
GNMIStatus = None
GNMIConfig = None
RestconfStatus = None
RestconfConfig = None
NetconfStatus = None
NetconfConfig = None
OctaConfig = None
GNSIConfig = None

def printLineItem( label, content ):
   print( f'{label}: {content}' )

class CommonEndpointStatus( CliModel.Model ):
   enabled = CliModel.Bool( help="Endpoint is enabled" )
   port = CliModel.Int( help="The port this protocol's server is listening on" )
   vrfName = CliModel.Str( help="The VRF this protocol's server is listening in" )
   error = CliModel.Str( help="error which occurred while starting the endpoint" )

   def render( self ):
      printLineItem( "Enabled", "yes" if self.enabled else "no" )
      printLineItem( "Server", f"running on port {self.port}, in {self.vrfName} VRF"
                               if self.port != 0 else "not yet running" )
      if self.error:
         printLineItem( "Error", self.error )

class GnmiEndpointStatus( CommonEndpointStatus ):
   __revision__ = 3

   sslProfile = CliModel.Str( help="SSL profile name" )
   qosDscp = CliModel.Int( help="The QoS DSCP value" )
   authorization = CliModel.Bool( help="per-RPC authorization" )
   accounting = CliModel.Bool( help="RPC requests accounting" )
   useNotifSendTime = CliModel.Bool( help="Subscription notification timestamp" )
   listenAddrs = CliModel.List( valueType=ArnetModel.IpGenericAddr,
                                help="Addresses for gRPC to listen on" )
   authnUsernamePriority = CliModel.List( valueType=str,
                        help="Authentication username extraction priority" )

   def degrade( self, dictRepr, revision ):
      if revision == 2:
         dictRepr[ 'certUsernameAuthn' ] = False
      return dictRepr

   def render( self ):
      super().render()
      printLineItem( "SSL profile", self.sslProfile if self.sslProfile else "none" )
      printLineItem( "QoS DSCP", self.qosDscp if self.qosDscp != 0 else "none" )
      printLineItem( "Authorization required",
         "yes" if self.authorization else "no" )
      printLineItem( "Accounting requests",
         "yes" if self.accounting else "no" )
      printLineItem( "Notification timestamp",
         "send time" if self.useNotifSendTime else "last change time" )
      printLineItem( "Listen addresses",
            " ".join( natsorted(str(ip.ip) for ip in self.listenAddrs ) ) )
      printLineItem( "Authentication username priority",
            ", ".join( self.authnUsernamePriority ) )

class GrpctunnelClientStatus( CliModel.Model ):
   status = CliModel.Str( help="Status of the gRPC-tunnel dial-out client" )
   error = CliModel.Str(
         help="Error which occurred while starting the gRPC-tunnel dial-out client" )
   enabled = CliModel.Bool( help="gRPC-tunnel dial-out client is enabled" )
   vrf = CliModel.Str( help="VRF of the gRPC-tunnel dial-out client" )
   tunnelSslProfile = CliModel.Str( help="SSL profile used by the tunnel" )
   gnmiSslProfile = CliModel.Str( help="SSL profile used by the gNMI server" )
   destinationHost = CliModel.Str(
         help="Destination address the client is dialing to" )
   destinationPort = CliModel.Int(
         help="Destination port the client is dialing to" )
   localInterface = IntfModels.Interface(
         help="Local interface the client is dialing from" )
   localAddressAndPort = CliModel.Submodel(
         valueType=ArnetModel.IpGenericAddrAndPort,
         help="Local address/port the client is dialing from" )
   targetId = CliModel.Str( help="Target ID used by the gRPC-tunnel connection" )

   def render( self ):
      if self.status:
         printLineItem( "Status", self.status )
      if self.error:
         printLineItem( "Error", self.error )
      printLineItem( "Enabled", "yes" if self.enabled else "no" )
      if self.vrf:
         printLineItem( "VRF", self.vrf )
      if self.tunnelSslProfile:
         printLineItem( "Tunnel SSL profile", self.tunnelSslProfile )
      if self.gnmiSslProfile:
         printLineItem( "gNMI SSL profile", self.gnmiSslProfile )
      if self.destinationHost and self.destinationPort:
         printLineItem( "Destination address", "{}, port {}".format(
            self.destinationHost, self.destinationPort ) )
      if self.localInterface:
         printLineItem( "Local interface", self.localInterface.stringValue )
      if self.localAddressAndPort.ip:
         localPort = ""
         if self.localAddressAndPort.port:
            localPort = f", port {self.localAddressAndPort.port}"
         printLineItem( "Local address", "{}{}".format(
            self.localAddressAndPort.ip, localPort ) )
      if self.targetId:
         printLineItem( "Target ID", self.targetId )

class GnsiEndpointStatus( CommonEndpointStatus ):
   def render( self ):
      printLineItem( "Transport enabled", "yes" if self.enabled else "no" )
      printLineItem( "Server", f"running on port {self.port}, in {self.vrfName} VRF"
                               if self.port != 0 else "not yet running" )
      if self.error:
         printLineItem( "Error", self.error )

class GnsiStatus( CliModel.Model ):
   transports = CliModel.Dict( valueType=GnsiEndpointStatus,
                               help="Transports indexed by name" )
   authzEnabled = CliModel.Bool( help="Authz service is enabled" )
   authzPolicy = CliModel.Bool( help="Authz policy exists/is configured" )
   certzEnabled = CliModel.Bool( help="Certz service is enabled" )
   credentialzEnabled = CliModel.Bool( help="Credentialz service is enabled" )
   acctzEnabled = CliModel.Bool( help="Acctz service is enabled" )
   acctzHistoryLimit = CliModel.Int( help="Acctz records history limit" )
   pathzEnabled = CliModel.Bool( help="Pathz service is enabled" )


   def render( self ):
      for ( name, transport ) in self.transports.items():
         printLineItem( "Transport", name )
         transport.render()
         # Print an empty line between transports
         print()
      printLineItem( "Acctz enabled", "yes" if self.acctzEnabled else "no" )
      if self.acctzEnabled:
         printLineItem( "Acctz records history limit", self.acctzHistoryLimit )
      printLineItem( "Authz enabled", "yes" if self.authzEnabled else "no" )
      printLineItem( "Authz policy", ( "present" if self.authzPolicy else
                                    "not present" ) +
                                    " at /persist/sys/gnsi/authz/policy.json" )
      printLineItem( "Certz enabled", "yes" if self.certzEnabled else "no" )
      printLineItem( "Credentialz enabled", "yes" if self.credentialzEnabled else
                                             "no" )
      printLineItem( "Pathz enabled", "yes" if self.pathzEnabled else "no" )

class GnmiAggregateStatus( CliModel.Model ):
   __revision__ = 3
   enabled = CliModel.Bool( help="Aggregate transports enabled" )
   setPersistence = CliModel.Bool( help="Set is persistent" )
   octa = CliModel.Bool( help="Octa enabled" )
   transports = CliModel.Dict( valueType=GnmiEndpointStatus,
                               help="Transports indexed by name" )
   grpctunnelClients = CliModel.Dict( valueType=GrpctunnelClientStatus,
                               help="gRPC-tunnel clients indexed by name" )

   def render( self ):
      printedHeader = False
      if not self.transports and not self.grpctunnelClients:
         printLineItem( "Enabled", "no transports enabled" )
         printedHeader = True
      if self.octa:
         printLineItem( "Octa", "enabled" )
         printedHeader = True
      if self.setPersistence:
         printLineItem( "Set persistence", "enabled" )
         printedHeader = True
      # Print a blank line before printing transports.
      if printedHeader and ( self.transports or self.grpctunnelClients ):
         print()
      for i, ( name, transport ) in enumerate( sorted(
                                   self.transports.items() ) ):
         printLineItem( "Transport", name )
         transport.render()
         # Print a blank line only in between transports.
         if i != len( self.transports ) - 1:
            print()
      if self.transports and self.grpctunnelClients:
         print()
      for i, ( name, client ) in enumerate( sorted(
                                   self.grpctunnelClients.items() ) ):
         printLineItem( "Transport gRPC-tunnel", name )
         client.render()
         # Print a blank line only in between transports.
         if i != len( self.grpctunnelClients ) - 1:
            print()

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # Revision 1 holds a single transport status.
         # Return the first transport status in transports.
         for _, transport in sorted( dictRepr[ 'transports' ].items() ):
            return transport
         dictRepr = {
            'enabled': False,
            'port': 0,
            'vrfName': '',
            'sslProfile': '',
            'qosDscp': 0,
            'authorization': False,
            'error': '',
            'listenAddrs': [],
         }
      return dictRepr

class GnmiAclStatus( CliModel.Model ):
   __revision__ = 2
   transports = CliModel.Dict( valueType=AllAclList,
                               help="Transport ACLs indexed by transport name" )

   def render( self ):
      for name, acl in sorted( self.transports.items() ):
         print( f'Transport: {name}' )
         acl.render()

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         # Revision 1 holds a single transport ACL.
         # Return the first ACL in the transport ACLs.
         for _, acl in sorted( dictRepr[ 'transports' ].items() ):
            return acl
         dictRepr = {
            'ipAclList': {
               'aclList': [],
            }
         }
      return dictRepr

class RestconfEndpointStatus( CommonEndpointStatus ):
   sslProfile = CliModel.Str( help="SSL profile name" )
   qosDscp = CliModel.Int( help="The QoS DSCP value" )
   def render( self ):
      super().render()
      printLineItem( "SSL profile", self.sslProfile if self.sslProfile else "none" )
      printLineItem( "QoS DSCP", self.qosDscp if self.qosDscp != 0 else "none" )

class NetconfEndpointStatus( CommonEndpointStatus ):
   pass

class SysdbExcludesStatus( CliModel.Model ):
   status = CliModel.Enum( values=( 'ok',
                                    'unknown',
                                    'error' ),
                                    help="Status of the excluded Sysdb path" )

class IpfixOptions( CliModel.Model ):
   enableCollector = CliModel.Bool( help='The collector is enabled' )
   address = CliModel.Str( help='Listening address for the collector',
                           optional=True)
   port = CliModel.Int( help='Listening port for the collector',
                           optional=True)
   domain = CliModel.Str( help='Domain value used by the collector',
                           optional=True)
   flowTableSize = CliModel.Int( help='Maximum number of entries in '
                                      'internal flow table',
                           optional=True)
   flowLifetime = CliModel.Int( help='Lifetime (in seconds) of entries in '
                                     'internal flow table',
                           optional=True)

class SflowOptions( CliModel.Model ):
   enableCollector = CliModel.Bool( help='The collector is enabled' )
   address = CliModel.Str( help='Listening address for the collector',
                           optional=True)
   port = CliModel.Int( help='Listening port for the collector',
                           optional=True)
   domain = CliModel.Str( help='Domain value used by the collector',
                           optional=True)
   flowTableSize = CliModel.Int( help='Maximum number of entries in '
                                      'internal sample table',
                           optional=True)
   flowLifetime = CliModel.Int( help='Lifetime (in seconds) of entries in '
                                     'internal sample table',
                           optional=True)

class IsisOptions( CliModel.Model ):
   linkStateDatabase = CliModel.Bool( help="Set if Link State Database is enabled" )

class ModelsStatus( CliModel.Model ):
   __revision__ = 2

   enabledSmashPaths = CliModel.List( valueType=str,
                                 help="List of enabled Smash paths" )
   disabledSmashPaths = CliModel.List( valueType=str,
                                  help="List of disabled Smash paths" )
   disabledSysdbPaths = CliModel.Dict( keyType=str,
                                       valueType=SysdbExcludesStatus,
                                       help="A mapping of a disabled Sysdb path to"
                                            " its status" )
   aftIPv4Unicast = CliModel.Bool( help="IPv4 Unicast is enabled for AFT" )
   aftIPv6Unicast = CliModel.Bool( help="IPv6 Unicast is enabled for AFT" )
   aftMpls = CliModel.Bool( help="MPLS is enabled for AFT" )
   aftRouteSummary = CliModel.Bool(
      help="OpenConfig route summary is enabled for AFT" )

   ipfixOptions = CliModel.Submodel( valueType = IpfixOptions,
                                     help="Set of IPFIX options" )

   sflowOptions = CliModel.Submodel( valueType = SflowOptions,
                                     help="Set of sFlow options" )

   isisOptions = CliModel.Submodel( valueType=IsisOptions,
                                    help="Set of IS-IS options" )
   cpuCountersEapiIntervalMinutes = CliModel.Int(
      help="Interval used to summarise CPU counters queue in minutes" )
   yangPaths = CliModel.Dict(
      keyType=str,
      valueType=bool,
      help="A mapping of a YANG path to whether it is explicitly enabled" )

   exclusiveModuleGroups = CliModel.List( valueType=str,
      help="List of exclusive YANG module groups" )

   if OCToggle.toggleOCDefaultLeafsResponseToggleEnabled():
      defaultLeafVisible = CliModel.Bool( help="Visibility of default leafs" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         key = 'disabledSysdbPaths'
         # Rev 1 has a list of the paths.
         dictRepr[ key ] = list( dictRepr[ key ] )
      return dictRepr

   def render( self ):
      if self.aftIPv4Unicast or self.aftIPv6Unicast:
         print( 'provider aft' )
         if self.aftIPv4Unicast:
            print( '  ipv4-unicast' )
         if self.aftIPv6Unicast:
            print( '  ipv6-unicast' )
         if self.aftMpls:
            print( '  mpls' )
         if self.aftRouteSummary:
            print( '  route-summary' )
      if self.isisOptions and self.isisOptions.linkStateDatabase:
         print( 'provider isis' )
         print( '  link-state-database' )
      print( 'provider smash' )
      for path in self.enabledSmashPaths:
         print( f'  path /Smash/{path}' )
      for path in self.disabledSmashPaths:
         print( f'  path /Smash/{path} disabled' )
      print( 'provider sysdb' )
      for path, sysdbExcludesStatus in self.disabledSysdbPaths.items():
         print( f'  path /Sysdb/{path} {sysdbExcludesStatus.status}' )
      print( 'provider ipfix' )
      print( '  Enabled:                           {}'.format(
         self.ipfixOptions.enableCollector ) )
      if self.ipfixOptions.enableCollector:
         print( '  Collector listening address:       {}'.format(
            self.ipfixOptions.address ) )
         print( '  Collector listening port:          {}'.format(
            self.ipfixOptions.port ) )
         print( '  Domain:                            {}'.format(
            self.ipfixOptions.domain ) )
         print( '  Maximum flow table entry count:    {}'.format(
            self.ipfixOptions.flowTableSize ) )
         print( '  Flow table entry life (seconds):   {}'.format(
            self.ipfixOptions.flowLifetime ) )
      print( 'provider sFlow' )
      print( '  Enabled:                           {}'.format(
         self.sflowOptions.enableCollector ) )
      if self.sflowOptions.enableCollector:
         print( '  Collector listening address:       {}'.format(
            self.sflowOptions.address ) )
         print( '  Collector listening port:          {}'.format(
            self.sflowOptions.port ) )
         print( '  Domain:                            {}'.format(
            self.sflowOptions.domain ) )
         print( '  Maximum sample table entry count:  {}'.format(
            self.sflowOptions.flowTableSize ) )
         print( '  Sample table entry life (seconds): {}'.format(
            self.sflowOptions.flowLifetime ) )
      if self.cpuCountersEapiIntervalMinutes != 0:
         print( 'provider http-commands' )
         print( '   command show cpu counters queue summary'
                f' {self.cpuCountersEapiIntervalMinutes} minutes' )
      print( 'models' )
      for path, available in sorted( self.yangPaths.items() ):
         print( '  path {}{}'.format(
            path, ' disabled' if not available else '' ) )
      if OCToggle.toggleOCExclusiveModuleGroupSegmentationToggleEnabled():
         print( 'modules' )
         for moduleGroup in sorted( self.exclusiveModuleGroups ):
            print( f'  module group {moduleGroup} exclusive' )
      if OCToggle.toggleOCDefaultLeafsResponseToggleEnabled():
         print( 'leaf default-value' )
         print( '  visible:                           {}'.format(
            self.defaultLeafVisible ) )

#-------------------------------------------------------------------------------
# The "show management api gnmi" command
#-------------------------------------------------------------------------------
def showStatusGnmi( mode, args ):
   model = GnmiAggregateStatus()
   model.octa = OctaConfig.enabled
   model.enabled = False
   model.setPersistence = GNMIConfig.setPersistence
   for name in GNMIConfig.endpoints:
      transport = GnmiEndpointStatus()
      # Transport status entity has not yet been created by the OpenConfig agent.
      if name not in GNMIStatus.endpoints:
         transport.enabled = False
         transport.port = 0
         transport.vrfName = ''
         transport.error = ''
         transport.sslProfile = ''
         transport.qosDscp = 0
         transport.authorization = False
         transport.accounting = False
         transport.useNotifSendTime = False
         transport.listenAddrs = []
         transport.authnUsernamePriority = []
         model.transports[ name ] = transport
         continue
      e = GNMIStatus.endpoints[ name ]
      if e.enabled:
         model.enabled = True
      transport.enabled = e.enabled
      transport.port = e.port
      transport.vrfName = e.vrfName
      transport.error = e.error
      transport.sslProfile = e.sslProfile
      transport.qosDscp = e.qosDscp
      transport.authorization = e.authorization
      transport.accounting = e.accounting
      transport.useNotifSendTime = e.useNotifSendTime

      priority = [ elm[ 1 ].source if elm[ 1 ].source !=
            AuthnUserPriorityCli.usernameSourceType.x509Oid else elm[ 1 ].x509Oid
            for elm in e.authnUsernamePriority.items() ]
      priority = list( map( lambda v:
         AuthnUserPriorityCli.authnUsernamePriorityToCLI[ v ] if v in
         AuthnUserPriorityCli.authnUsernamePriorityToCLI else
         f"x509-{v}", priority ) )
      transport.authnUsernamePriority = priority
      addrs = e.listenAddrs or [ "::" ]
      transport.listenAddrs = \
            [ ArnetModel.IpGenericAddr( ip=ip ) for ip in addrs ]
      model.transports[ name ] = transport
   for name in GNMIConfig.grpctunnelClients:
      client = GrpctunnelClientStatus()
      # Transport status entity has not yet been created by the OpenConfig agent.
      # As a result, the status is 'starting'.
      if name not in GNMIStatus.grpctunnelClients:
         client.status = 'starting'
         client.error = ''
         client.enabled = False
         client.vrf = ''
         client.tunnelSslProfile = ''
         client.gnmiSslProfile = ''
         client.destinationHost = ''
         client.destinationPort = 0
         client.localInterface = ''
         client.localAddressAndPort = ArnetModel.IpGenericAddrAndPort(
               ip='', port=0 )
         client.targetId = ''
         model.grpctunnelClients[ name ] = client
         continue
      c = GNMIStatus.grpctunnelClients[ name ]
      if c.enabled:
         model.enabled = True
      client.status = c.status
      client.error = c.error
      client.enabled = c.enabled
      client.vrf = c.vrf
      client.tunnelSslProfile = c.tunnelSslProfile
      client.gnmiSslProfile = c.gnmiSslProfile
      client.destinationHost = c.destinationHost
      client.destinationPort = c.destinationPort
      client.localInterface = c.localInterface
      client.localAddressAndPort = ArnetModel.IpGenericAddrAndPort(
            ip=c.localIp, port=c.localPort )
      client.targetId = c.targetId
      model.grpctunnelClients[ name ] = client
   return model

#-------------------------------------------------------------------------------
# The "show management api gnmi access-list" command
#-------------------------------------------------------------------------------
def showAclStatusGnmi( mode, args ):
   aclName = args.get( 'ACL' )
   model = GnmiAclStatus()
   for name, endpoint in GNMIConfig.endpoints.items():
      if ( not endpoint.enabled
           or endpoint.serviceAcl == ''
           or ( aclName and aclName != endpoint.serviceAcl ) ):
         # Skip if an ACL is not configured for the enabled endpoint or when the
         # ACL name argument (if provided) does not correspond with the endpoint
         # ACL name.
         continue
      params = [ endpoint.serviceAcl, 'summary' in args ]
      model.transports[ name ] = AclCli.showServiceAcl( mode, aclCpConfig,
            aclStatus, aclCheckpoint, 'ip', params, serviceName='gnmi' )
   return model

#-------------------------------------------------------------------------------
# The "show management api restconf" command
#-------------------------------------------------------------------------------
def showStatusRestconf( mode, args ):
   model = RestconfEndpointStatus()
   endpoint = None
   # Select the endpoint status in order:
   #  1. is enabled,
   #  2. the error is not empty.
   for _, e in sorted( RestconfStatus.endpoints.items() ):
      if e.enabled:
         endpoint = e
         break
      if e.error != '':
         endpoint = e # Continue because a subsequent endpoint may be enabled.
   if endpoint is None:
      model.enabled = False
      model.vrfName = ''
      model.port = 0
      model.sslProfile = ''
      model.qosDscp = 0
      model.error = ''
      return model
   model.enabled = endpoint.enabled
   model.vrfName = endpoint.vrfName
   model.port = endpoint.port
   model.sslProfile = endpoint.sslProfile
   model.qosDscp = endpoint.qosDscp
   model.error = endpoint.error
   return model

#-------------------------------------------------------------------------------
# The "show management api restconf access-list" command
#-------------------------------------------------------------------------------
def showAclStatusRestconf( mode, args ):
   params = [ args.get( 'ACL' ), 'summary' in args ]
   for endpoint in RestconfConfig.endpoints.values():
      if endpoint.enabled:
         return AclCli.showServiceAcl( mode, aclCpConfig, aclStatus,
               aclCheckpoint, 'ip', params, serviceName='restconf' )
   return AllAclList()

#-------------------------------------------------------------------------------
# The "show management api netconf" command
#-------------------------------------------------------------------------------
def showStatusNetconf( mode, args ):
   model = NetconfEndpointStatus()
   model.enabled = False
   model.port = NetconfStatus.port
   model.vrfName = NetconfStatus.vrfName
   model.error = NetconfStatus.error
   for endpoint in NetconfConfig.endpoints.values():
      if not endpoint.enabled:
         continue
      vrfName = endpoint.vrfName
      port = endpoint.port
      sshdConfig = "/etc/ssh/sshd_config"
      if vrfName != "default":
         sshdConfig += "-" + vrfName
      if not os.path.exists( sshdConfig ):
         continue
      with open( sshdConfig ) as f:
         for line in f:
            if "Subsystem netconf netconf start-client" in line:
               configs = glob.glob( '/etc/xinetd.d/sshnetconf-' + endpoint.name )
               if configs:
                  model.vrfName = vrfName
                  model.port = port
                  model.enabled = True
                  return model
   return model

#-------------------------------------------------------------------------------
# The "show management api models" command
#-------------------------------------------------------------------------------
def showStatusModels( mode, args ):
   model = ModelsStatus()
   model.aftIPv4Unicast = OctaConfig.aftOptions.ipv4Unicast
   model.aftIPv6Unicast = OctaConfig.aftOptions.ipv6Unicast
   model.aftMpls = OctaConfig.aftOptions.mpls
   model.aftRouteSummary = OctaConfig.aftOptions.routeSummary
   model.enabledSmashPaths = [ path for path in OctaConfig.smashIncludes.values()
         if path ]
   model.disabledSmashPaths = [ path for path in OctaConfig.smashExcludes.values()
         if path ]

   for key, value in OctaConfig.sysdbExcludes.items():
      sysdbExcludesStatus = SysdbExcludesStatus()
      sysdbExcludesStatus.status = value
      model.disabledSysdbPaths[ key ] = sysdbExcludesStatus

   if OctaConfig.ecoOptions is None:
      ipfixEnabled = False
      sflowEnabled = False
   else:
      ipfixEnabled = OctaConfig.ecoOptions.ipfixEnableCollector
      sflowEnabled = OctaConfig.ecoOptions.sflowEnableCollector

   model.ipfixOptions = IpfixOptions()
   model.ipfixOptions.enableCollector = ipfixEnabled
   if ipfixEnabled:
      model.ipfixOptions.address = OctaConfig.ecoOptions.ipfixAddress
      model.ipfixOptions.port = OctaConfig.ecoOptions.ipfixPort
      model.ipfixOptions.domain = OctaConfig.ecoOptions.ipfixDomain
      model.ipfixOptions.flowTableSize = OctaConfig.ecoOptions.ipfixFlowTableSize
      model.ipfixOptions.flowLifetime = OctaConfig.ecoOptions.ipfixFlowLifetime

   model.sflowOptions = SflowOptions()
   model.sflowOptions.enableCollector = sflowEnabled
   if sflowEnabled:
      model.sflowOptions.address = OctaConfig.ecoOptions.sflowAddress
      model.sflowOptions.port = OctaConfig.ecoOptions.sflowPort
      model.sflowOptions.domain = OctaConfig.ecoOptions.sflowDomain
      model.sflowOptions.flowTableSize = OctaConfig.ecoOptions.sflowFlowTableSize
      model.sflowOptions.flowLifetime = OctaConfig.ecoOptions.sflowFlowLifetime

   model.cpuCountersEapiIntervalMinutes = \
           OctaConfig.httpCommands.sandCounters.cpuCountersEapiInterval

   model.isisOptions = IsisOptions()
   model.isisOptions.linkStateDatabase = OctaConfig.isisOptions.linkStateDatabase

   model.yangPaths = dict( OctaConfig.yangPaths )

   model.exclusiveModuleGroups = list( OctaConfig.exclusiveModuleGroups )

   if OCToggle.toggleOCDefaultLeafsResponseToggleEnabled():
      model.defaultLeafVisible = OctaConfig.defaultLeafVisible

   return model

# -------------------------------------------------------------------------------
# The "show management api gnsi" command
# -------------------------------------------------------------------------------
def showStatusGnsi( mode, args ):
   model = GnsiStatus()
   model.authzEnabled = GNSIConfig.service.authz
   model.certzEnabled = GNSIConfig.service.certz
   model.credentialzEnabled = GNSIConfig.service.credentialz
   model.pathzEnabled = GNSIConfig.service.pathz
   model.authzPolicy = os.path.exists( '/persist/sys/gnsi/authz/policy.json' )
   model.acctzEnabled = GNSIConfig.service.acctz
   if model.acctzEnabled:
      model.acctzHistoryLimit = GNSIConfig.acctzConfig.historyLimit
   else:
      # This won't be rendered if acctz is not configured
      model.acctzHistoryLimit = 0
   for name in sorted( GNSIConfig.endpoints ):
      transport = GnsiEndpointStatus()
      if name in GNMIStatus.endpoints:
         endpointStatus = GNMIStatus.endpoints[ name ]
         transport.enabled = endpointStatus.enabled
         transport.port = endpointStatus.port
         transport.vrfName = endpointStatus.vrfName
         transport.error = endpointStatus.error
      else:
         transport.enabled = False
         transport.port = 0
         transport.vrfName = ''
         transport.error = ''
      model.transports[ name ] = transport

   return model

# Plug-in definition:
def Plugin( entityManager ):
   global GNMIStatus, GNMIConfig
   global RestconfStatus, RestconfConfig
   global NetconfStatus, NetconfConfig, OctaConfig
   global aclCpConfig, aclStatus, aclCheckpoint
   global GNSIConfig

   GNMIStatus = LazyMount.mount( entityManager, "mgmt/gnmi/status",
                                 "Gnmi::Status", "r" )
   GNMIConfig = LazyMount.mount( entityManager, "mgmt/gnmi/config",
                                 "Gnmi::Config", "r" )
   RestconfStatus = LazyMount.mount( entityManager, "mgmt/restconf/status",
                                 "Restconf::Status", "r" )
   RestconfConfig = LazyMount.mount( entityManager, "mgmt/restconf/config",
                                 "Restconf::Config", "r" )
   NetconfStatus = LazyMount.mount( entityManager, "mgmt/netconf/status",
                                 "Netconf::Status", "r" )
   NetconfConfig = LazyMount.mount( entityManager, "mgmt/netconf/config",
                                 "Netconf::Config", "r" )
   OctaConfig = LazyMount.mount( entityManager, "mgmt/octa/config",
                                 "Octa::Config", "r" )
   aclCpConfig = LazyMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "r" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                    "Acl::CheckpointStatus", "r" )
   GNSIConfig = LazyMount.mount( entityManager, "mgmt/gnsi/config",
                                 "Gnsi::Config", "r" )
