#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=too-many-nested-blocks
# pylint: disable=consider-using-f-string
from Ark import utcTimeRelativeToNowStr
from CliModel import Dict
from CliModel import Enum
from CliModel import Int
from CliModel import Float
from CliModel import Model
from CliModel import Str
from CliModel import Bool
from CliModel import List
from CliPlugin.NetworkTopologyModels import DirectedEdge, Host
from TableOutput import Format, createTable, TableFormatter, FormattedCell
import Arnet
import Tac
from Ethernet import convertMacAddrToCanonical

import sys

MAP_CHARS = ' -> '

# Helper function that return a list of ( switch, switchport ) tuples that
# connect to this host and on which this instancePort is provisioned
def getPortSwitchports( allHosts, allNeighbors, host, instancePort ):
   portSwitchports = list() # pylint: disable=use-list-literal
   for port in host.port.values():
      edge = allNeighbors.get( '%s-%s' %
                               ( host.hostname, port.name ) )
      if edge:
         # Assuming that hosts are attached to switches only
         for connectedPort in edge.toPort:
            switch = allHosts.get( connectedPort.hostname )
            if switch == None: # pylint: disable=singleton-comparison
               continue

            # If switchports is not empty, this logical port may only be logically
            # connected to a subset of a hosts ports. If that is the case, only add
            # the physical port if this logical port contains a reference to the
            # physical port
            switchId = convertMacAddrToCanonical( switch.name )
            if instancePort.switches:
               if switchId in instancePort.switches and connectedPort.name \
                  in instancePort.switches[ switchId ].switchports:
                  portSwitchports.append( ( switch.hostname,
                                            switch.name,
                                            connectedPort.name ) )
            else:
               portSwitchports.append( ( switch.hostname,
                                         switch.name,
                                         connectedPort.name ) )
   return portSwitchports

# Tenant Config Classes
# Rendering prints out the commands that would have been used to configure
# openstack with the passed in model data. This is used for testing. A dict with
# the openstack configuration is returned on the CAPI call which is used by the
# ML2 agent during a sync to download all the data associated with the specified
# region.
# NOTE: Use a hierarchy of classes just for config because the ML2 driver is
# sensitive to any empty dicts that may show up in the output. Example is the
# networkVmInstancesIdName dict where it is defined as optional but the
# default generator for TypeDict will generate an empty dict.

class SegmentConfig( Model ):
   segmentId = Str( help='Segment id' )
   segmentationType = Enum( help='Segmentation Type', values=( 'vlan', 'vxlan' ) )
   segmentationTypeId = Int( help='VLAN/VxLAN ID' )
   networkId = Str( help='Network id associated with the segment.' )

# Tenant Network Config Information
class NetworkConfig( Model ):
   networkId = Str( help='Tenant Network Id' )
   networkName = Str( help='Tenant network name' )
   segmentationType = Enum( help='Segmentation type of the first static segment',
                            values=( 'vlan', 'vxlan' ) )
   segmentationTypeId = Int( help='VLAN/VxLAN ID of the first static segment' )
   shared = Bool( help='Indicates whether the network is a shared network or not.' )

   staticSegment = List( valueType=SegmentConfig,
                         help='Collection of static segments in the network' )

   dynamicSegment = List( valueType=SegmentConfig,
                          help='Collection of dynamic segments in the network',
                          optional=True )

# Physical Interface Config Information
class SwitchportList( Model ):
   switchports = List( valueType=str, help='Connected physical interfaces' )

# Port Config Information
class PortConfig( Model ):
   portId = Str( help='Port Id' )
   networkId = Str( help='Tenant Network Id' )
   portVlanType = Enum( help='Port Vlan Type',
                        values=( 'allowed', 'access', 'native' ),
                        optional=True )
   hosts = List( valueType=str, help='Hosts on which the port exists (used to '
                 'provision vms and routers)', optional=True )
   switches = Dict( valueType=SwitchportList,
                    help='Collection of physical switchports for this logical '
                    'port keyed by switch id (used to provision baremetal nodes)',
                    optional=True )
   segments = List( valueType=SegmentConfig,
                    help='Collection of segments in the network' )

# VM Instance Config Information
class VmInstanceConfig( Model ):
   vmInstanceId = Str( help='VM Instance Id' )
   vmHostId = Str( help='Host on which VM Instance executes' )
   vmPorts = Dict( valueType=PortConfig, help='Collection of ports for a VM' )

   def setInstanceId( self, instanceId ):
      self.vmInstanceId = instanceId

   def setInstanceHostId( self, hostId ):
      self.vmHostId = hostId

   def getInstanceId( self ):
      return self.vmInstanceId

   def getInstanceHost( self ):
      return self.vmHostId

   def getInstancePorts( self ):
      return self.vmPorts

# BM Instance Config Information
class BaremetalInstanceConfig( Model ):
   baremetalInstanceId = Str( help='Baremetal Instance Id' )
   baremetalHostId = Str(
      help='Ironic node id on which Baremetal Instance executes' )
   baremetalPorts = Dict( valueType=PortConfig, help='Collection of ports for a '
                          'baremetal instance' )

   def setInstanceId( self, instanceId ):
      self.baremetalInstanceId = instanceId

   def setInstanceHostId( self, hostId ):
      self.baremetalHostId = hostId

   def getInstanceId( self ):
      return self.baremetalInstanceId

   def getInstanceHost( self ):
      return self.baremetalHostId

   def getInstancePorts( self ):
      return self.baremetalPorts

# Router Instance Config Information
class RouterInstanceConfig( Model ):
   routerInstanceId = Str( help='Router Instance Id' )
   routerHostId = Str( help='Host on which Router Instance executes' )
   routerPorts = Dict( valueType=PortConfig,
                       help='Collection of ports for a Router' )

   def setInstanceId( self, instanceId ):
      self.routerInstanceId = instanceId

   def setInstanceHostId( self, hostId ):
      self.routerHostId = hostId

   def getInstanceId( self ):
      return self.routerInstanceId

   def getInstanceHost( self ):
      return self.routerHostId

   def getInstancePorts( self ):
      return self.routerPorts

# Tenant Config Information
class TenantConfig( Model ):
   tenantId = Str( help='Tenant Id' )
   tenantNetworks = Dict( valueType=NetworkConfig,
                          help='Collection of tenant networks for a tenant' )
   tenantVmInstances = Dict( valueType=VmInstanceConfig,
                             help='Collection of VMs for a tenant' )
   tenantBaremetalInstances = Dict( valueType=BaremetalInstanceConfig,
                                    help='Collection of baremetal instances for a '
                                    'tenant', optional=True )
   tenantRouterInstances = Dict( valueType=RouterInstanceConfig,
                                 help='Collection of Routers for a tenant',
                                 optional=True )

   def getInstancesOfType( self, instanceType ):
      typeMap = { 'virtual' : self.tenantVmInstances,
                  'baremetal' : self.tenantBaremetalInstances,
                  'router' : self.tenantRouterInstances }
      return typeMap[ instanceType ]

   def render( self ):
      sys.stdout.write( 'tenant %s\n' % self.tenantId )
      for network in self.tenantNetworks.values():
         sys.stdout.write( 'network id %s\n' %
                           network.networkId )
         for segment in network.staticSegment:
            sys.stdout.write( 'segment %s type %s id %s static\n' %
                              ( segment.segmentId,
                                segment.segmentationType,
                                segment.segmentationTypeId ) )
         for segment in network.dynamicSegment:
            sys.stdout.write( 'segment %s type %s id %s dynamic\n' %
                              ( segment.segmentId,
                                segment.segmentationType,
                                segment.segmentationTypeId ) )

      for instanceType in [ 'virtual', 'baremetal', 'router' ]:
         instances = self.getInstancesOfType( instanceType )
         for instance in instances.values():
            instanceId = instance.getInstanceId()
            instanceHost = instance.getInstanceHost()
            if instanceHost:
               sys.stdout.write( 'instance id %s hostid %s type %s\n' %
                                 ( instanceId, instanceHost, instanceType ) )
            else:
               sys.stdout.write( 'instance id %s type %s\n' %
                                 ( instanceId, instanceType ) )
            instancePorts = instance.getInstancePorts()
            for port in instancePorts.values():
               portFormatString = 'port id %s network-id %s type %s'
               hostPortFormatString = portFormatString + ' hostid %s\n'
               switchportPortFormatString = portFormatString + \
                                            ' switch-id %s switchport %s\n'
               portFormatString += '\n'
               for host in port.hosts:
                  if host != instanceHost:
                     sys.stdout.write( hostPortFormatString % ( port.portId,
                                                                port.networkId,
                                                                port.portVlanType,
                                                                host ) )
                     for level, s in enumerate( port.segments ):
                        sys.stdout.write( "segment level %d id %s\n" % ( level,
                           s.segmentId ) )
               if port.switches:
                  for switch in port.switches:
                     for switchport in port.switches[ switch ].switchports:
                        sys.stdout.write( switchportPortFormatString %
                                          ( port.portId, port.networkId,
                                            port.portVlanType, switch, switchport ) )
                        for level, s in enumerate( port.segments ):
                           sys.stdout.write( "segment level %d id %s\n" % ( level,
                              s.segmentId ) )
               else:
                  sys.stdout.write( portFormatString %
                     ( port.portId, port.networkId, port.portVlanType ) )
                  for level, s in enumerate( port.segments ):
                     sys.stdout.write( "segment level %d id %s\n" % ( level,
                        s.segmentId ) )

# Class used by the "show openstack config"
class TenantsConfig( Model ):
   tenants = Dict( valueType=TenantConfig,
                   help='Collection of tenants within the specified region' )

   def render( self ):
      for tenant in self.tenants.values():
         tenant.render()

# VM Instance Information - ID and name only
class VmInstanceIdName( Model ):
   vmInstanceId = Str( help='VM Instance Id' )
   vmInstanceName = Str( help='VM Instance Name', optional=True )
   vmHostId = Str( help='Host on which VM Instance executes' )
   switches = Dict( valueType=SwitchportList,
                    help='Collection of physical switchports for this vm '
                    'on a single network keyed by switch id', optional=True )

   def setInstanceId( self, instanceId ):
      self.vmInstanceId = instanceId

   def setInstanceName( self, instanceName ):
      self.vmInstanceName = instanceName

   def setInstanceHostId( self, hostId ):
      self.vmHostId = hostId

   def getInstanceId( self ):
      return self.vmInstanceId

   def getInstanceHost( self ):
      return self.vmHostId

# Baremetal Instance Information - ID and name only
class BaremetalInstanceIdName( Model ):
   baremetalInstanceId = Str( help='Baremetal Instance Id' )
   baremetalInstanceName = Str( help='Baremetal Instance Name', optional=True )
   baremetalHostId = Str(
      help='Ironic node id on which Baremetal Instance executes' )
   switches = Dict( valueType=SwitchportList,
                    help='Collection of physical switchports for this baremetal '
                    'on a single network keyed by switch id', optional=True )

   def setInstanceId( self, instanceId ):
      self.baremetalInstanceId = instanceId

   def setInstanceName( self, instanceName ):
      self.baremetalInstanceName = instanceName

   def setInstanceHostId( self, hostId ):
      self.baremetalHostId = hostId

   def getInstanceId( self ):
      return self.baremetalInstanceId

   def getInstanceHost( self ):
      return self.baremetalHostId

# Port Information
class Port( Model ):
   portId = Str( help='Port Id' )
   portName = Str( help='Port Name', optional=True )
   networkId = Str( help='Tenant Network Id' )
   portVlanType = Enum( help='Port Vlan Type',
                        values=( 'allowed', 'access', 'native' ),
                        optional=True )
   hosts = List( valueType=str, help='Hosts on which the port exists (used to '
                 'provision vms and routers)', optional=True )
   switches = Dict( valueType=SwitchportList,
                    help='Collection of physical switchports for this logical '
                    'port keyed by switch id (used to provision baremetal nodes)',
                    optional=True )
   segments = List( valueType=SegmentConfig,
                    help='List of segments the port is bound to' )

# VM Instance Information
class VmInstance( Model ):
   vmInstanceId = Str( help='VM Instance Id' )
   vmInstanceName = Str( help='VM Instance Name', optional=True )
   vmHostId = Str( help='Host on which VM Instance executes' )
   vmPorts = Dict( valueType=Port, help='Collection of ports for a VM' )

   def setInstanceId( self, instanceId ):
      self.vmInstanceId = instanceId

   def setInstanceName( self, instanceName ):
      self.vmInstanceName = instanceName

   def setInstanceHostId( self, hostId ):
      self.vmHostId = hostId

   def getInstanceId( self ):
      return self.vmInstanceId

   def getInstanceName( self ):
      return self.vmInstanceName

   def getInstanceHost( self ):
      return self.vmHostId

   def getInstancePorts( self ):
      return self.vmPorts

   def hasMultipleHosts( self ):
      for port in self.vmPorts.values():
         for host in port.hosts:
            if host != self.vmHostId:
               return True
      return False

# Baremetal Instance Information
class BaremetalInstance( Model ):
   baremetalInstanceId = Str( help='Baremetal Instance Id' )
   baremetalInstanceName = Str( help='Baremetal Instance Name', optional=True )
   baremetalHostId = Str(
      help='Ironic node id on which Baremetal Instance executes' )
   baremetalPorts = Dict( valueType=Port, help='Collection of ports for a Baremetal '
                          'instance' )

   def setInstanceId( self, instanceId ):
      self.baremetalInstanceId = instanceId

   def setInstanceName( self, instanceName ):
      self.baremetalInstanceName = instanceName

   def setInstanceHostId( self, hostId ):
      self.baremetalHostId = hostId

   def getInstanceId( self ):
      return self.baremetalInstanceId

   def getInstanceName( self ):
      return self.baremetalInstanceName

   def getInstanceHost( self ):
      return self.baremetalHostId

   def getInstancePorts( self ):
      return self.baremetalPorts

   def hasMultipleHosts( self ):
      for port in self.baremetalPorts.values():
         for host in port.hosts:
            if host != self.baremetalHostId:
               return True
      return False

# DHCP Instance Information
class DhcpInstance( Model ):
   dhcpInstanceId = Str( help='DHCP Instance Id' )
   dhcpHostId = Str( help='Host on which DHCP Instance executes' )
   dhcpPorts = Dict( valueType=Port, help='Collection of ports for a DHCP instance' )

   def setInstanceId( self, instanceId ):
      self.dhcpInstanceId = instanceId

   def setInstanceName( self, instanceName ):
      # DHCP instances have no name
      pass

   def setInstanceHostId( self, hostId ):
      self.dhcpHostId = hostId

   def getInstancePorts( self ):
      return self.dhcpPorts

# Router Instance Information
class RouterInstance( Model ):
   routerInstanceId = Str( help='Router Instance Id' )
   routerInstanceName = Str( help='Router Instance Name' )
   routerHostId = Str( help='Host on which the router in placed on' )
   routerPorts = Dict( valueType=Port,
                       help='Collection of ports for a router instance' )

   def setInstanceId( self, instanceId ):
      self.routerInstanceId = instanceId

   def setInstanceName( self, instanceName ):
      self.routerInstanceName = instanceName

   def setInstanceHostId( self, hostId ):
      self.routerHostId = hostId

   def getInstanceId( self ):
      return self.routerInstanceId

   def getInstanceName( self ):
      return self.routerInstanceName

   def getInstanceHost( self ):
      return self.routerHostId

   def getInstancePorts( self ):
      return self.routerPorts

   def hasMultipleHosts( self ):
      for port in self.routerPorts.values():
         for host in port.hosts:
            if host != self.routerHostId:
               return True
      return False

# Tenant Network Information
class Network( Model ):
   networkId = Str( help='Tenant Network Id' )
   networkName = Str( help='Tenant Network Name', optional=True )
   segmentationType = Enum( help='Segmentation Type', values=( 'vlan', 'vxlan' ) )
   segmentationTypeId = Int( help='VLAN/VxLAN ID' )
   networkVmInstancesIdName = Dict( valueType=VmInstanceIdName,
                                    help='Collection of VMs for a tenant',
                                    optional=True )
   networkBaremetalInstancesIdName = Dict( valueType=BaremetalInstanceIdName,
                                           help='Collection of baremetal instances '
                                           'for a tenant', optional=True )
   dhcpInstances = Dict( valueType=DhcpInstance,
                         help='Collection of DHCP instances',
                         optional=True )
   mappedVni = Int( help='VNI to which the VLAN maps to' )
   shared = Bool( help='Indicates whether the network is a shared network or not.' )

   def getInstanceIdNames( self ):
      allInstancesIdNames = list( self.networkVmInstancesIdName.values() )[ : ]
      allInstancesIdNames.extend( self.networkBaremetalInstancesIdName.values() )
      return allInstancesIdNames

# Tenant Information
class Tenant( Model ):
   tenantId = Str( help='Tenant Id' )
   tenantName = Str( help='Tenant Name', optional=True )
   tenantNetworks = Dict( valueType=Network,
                          help='Collection of tenant networks for a tenant' )
   tenantVmInstances = Dict( valueType=VmInstance,
                             help='Collection of VMs for a tenant' )
   tenantBaremetalInstances = Dict( valueType=BaremetalInstance,
                                    help='Collection of baremetal instances for a '
                                    'tenant', optional=True )
   tenantRouterInstances = Dict( valueType=RouterInstance,
                                 help='Collection of routers for a tenant' )

   def getInstancesOfType( self, instanceType ):
      instanceTypeMap = { 'virtual' : self.tenantVmInstances,
                          'baremetal' : self.tenantBaremetalInstances,
                          'router' : self.tenantRouterInstances }
      return instanceTypeMap[ instanceType ]

   def hasInstanceId( self, instanceId ):
      if instanceId in self.tenantVmInstances:
         return True
      if instanceId in self.tenantBaremetalInstances:
         return True
      if instanceId in self.tenantRouterInstances:
         return True
      return False

# Service EndPoint Information
class ServiceEndPoint( Model ):
   name = Str( help='Service Name' )
   authUrl = Str( help='Service Auth URL' )
   username = Str( help='Keystone admin user name', optional=True )
   tenantname = Str( help='Keystone admin tenant name', optional=True )

# Region Information
class Region( Model ):
   regionName = Str( help='Region Name' )
   tenants = Dict( valueType=Tenant, help='Collection of tenants within the region' )
   serviceEndPoints = Dict( valueType=ServiceEndPoint, help='Collection of service'
                            'endpoints within the region', optional=True )
   SyncStatusEnum = Tac.Type( 'OpenStack::SyncStatus' )
   syncStatus = Enum( help='Sync Status',
                      values=( SyncStatusEnum.unknown,
                               SyncStatusEnum.syncInProgress,
                               SyncStatusEnum.syncTimedout,
                               SyncStatusEnum.syncComplete ),
                      optional=True )
   syncInterval = Float( help='ML2 Sync Interval', optional=True )
   lastSyncTime = Float( help='Last successful sync', optional=True )

   def render( self ):
      syncStatus = Tac.Type( 'OpenStack::SyncStatus' )
      syncStatusDict = { syncStatus.unknown : 'Unknown',
                         syncStatus.syncInProgress : 'In Progress',
                         syncStatus.syncTimedout : 'Timed Out',
                         syncStatus.syncComplete : 'Completed' }
      print( "Region: %s" % self.regionName )
      print( "Sync Status: %s" % syncStatusDict.get( self.syncStatus ) )
      print( "Sync Interval: %s s" % str( self.syncInterval ) )
      print( "Last Sync Time: %s\n" %
             utcTimeRelativeToNowStr( self.lastSyncTime ) )

      # If there is not keystone service endpoint, don't print any service endpoints
      if 'keystone' in self.serviceEndPoints:
         serviceTable = createTable( ( ( createHdr( "Service" ), "l" ),
                                       ( createHdr( "Authentication URL" ), "l" ),
                                       ( createHdr( "User" ), "l" ),
                                       ( createHdr( "Tenant" ), "l" ) ),
                                     indent=2 )
         keystoneService = self.serviceEndPoints[ 'keystone' ]
         authUrl = keystoneService.authUrl
         serviceTable.newRow( 'keystone',
                              authUrl,
                              keystoneService.username,
                              keystoneService.tenantname )
         for _, service in sorted( self.serviceEndPoints.items() ):
            if service.name != 'keystone':
               serviceTable.newRow( service.name,
                                    service.authUrl,
                                    '',
                                    '' )
         print( serviceTable.output() )
      else:
         print( "  Keystone service endpoint is not configured" )

def dash( count ):
   # Return a string with the specified number of dashes
   return '-' * count

def createHdr( title ):
   hdrFormat = Format( justify="left" )
   return FormattedCell( nCols=1, content=title, format=hdrFormat )

# Regions Class
class RegionServiceEndPoints( Model ):
   regions = Dict( valueType=Region, help='Collection of regions' )

   def render( self ):
      for _, region in sorted( self.regions.items() ):
         region.render()

# Region timestamps
class RegionTimestamp( Model ):
   regionName = Str( help='Region name' )
   regionTimestamp = Str( help='OpenStack agent UUID' )

   def render( self ):
      sys.stdout.write( "Region: %s\n" % self.regionName )
      sys.stdout.write( "OpenStack agent UUID: %s\n" % self.regionTimestamp )

# Tenants Class
class RegionTenants( Model ):
   regions = Dict( valueType=Region, help='Collection of regions' )

   def render( self ):
      genOutput = False
      tenantTable = createTable( ( ( createHdr( "Region" ), "l" ),
                                   ( createHdr( "Tenant Name" ), "l" ),
                                   ( createHdr( "Tenant Id" ), "l" ) ) )
      for _, region in sorted( self.regions.items() ):
         newline = False
         for _, tenant in sorted( region.tenants.items(),
                                  key=lambda x: x[ 1 ].tenantName ):
            if newline:
               tenantTable.newRow( '', tenant.tenantName, tenant.tenantId )
            else:
               tenantTable.newRow( region.regionName,
                                   tenant.tenantName,
                                   tenant.tenantId )
               newline = True
            genOutput = True

      if genOutput:
         sys.stdout.write( tenantTable.output() )

class RegionConfig( Model ):
   regionName = Str( help='Region Name' )
   tenants = Dict( valueType=TenantConfig,
                   help='Collection of tenants within the region' )

   def render( self ):
      sys.stdout.write( 'Region %s\n' % self.regionName )

      for tenant in self.tenants.values():
         tenant.render()

      sys.stdout.write( '\n' )

class RegionsConfig( Model ):
   regions = Dict( valueType=RegionConfig, help='Collection of regions' )

   def render( self ):
      for region in self.regions.values():
         region.render()

class NetworkSegment( Model ):
   networkId = Str( help='Tenant Network Id' )
   networkName = Str( help='Tenant network name' )
   segments = List( valueType=SegmentConfig,
                    help='List of segments on the network' )

class PortBinding( Model ):
   portId = Str( help='Port Id' )
   hosts = Dict( valueType=NetworkSegment,
                 help='Hosts on which the port exists keyed by host name',
                 optional=True )
   switches = Dict( valueType=NetworkSegment,
                    help='Physical switches on which the port is connected to '
                         'keyed by switch name and interface',
                    optional=True )

class InstancePortBinding( Model ):
   ports = Dict( valueType=PortBinding,
                 help='Collection of ports in instance keyed by portId' )
   instanceId = Str( help='Instance Id' )
   instanceName = Str( help='Instance Name' )
   tenantId = Str( help='Tenant Id' )
   tenantName = Str( help='Tenant name' )

class RegionInstanceBinding( Model ):
   instances = Dict( valueType=InstancePortBinding,
                     help='Collection of ports keyed by instance Id' )

class RegionOpenStackSegments( Model ):
   regions = Dict( valueType=RegionInstanceBinding,
                   help='Collection of instance in region' )

   def orderData( self ):
      """Order regions data based on region, tenants, instances

      This method uses data in regions from RegionOpenStackSegments and returns
      a dictionary with the following format which is suitable for displaying 
      openstack segment information:
         { <regions name> : {
               'tenants' : {
                  <tenant id> : {
                     'instances' : {
                        <instance id> : {
                           'switches'|'hosts' : {
                              <switchname - interface>|<hostname> : {
                                 'networks' : {
                                    <network id> : {
                                       'ports' : {
                                          <port id> : {
                                             'segments' : [{...}, ...],
                                             ...
                                          } } } } } } } } } } } }
      """
      regionData = {}
      for regionName, instances in self.regions.items():
         tenantInfo = {}
         for instanceId, instance in instances.instances.items():
            instanceInfo = {}
            tn = instance.tenantId
            if tn not in tenantInfo:
               tenantInfo[ tn ] = {}
               tenantInfo[ tn ][ 'name' ] = instance.tenantName
               tenantInfo[ tn ][ 'instances' ] = {}

            if instanceId not in tenantInfo[ tn ][ 'instances' ]:
               instanceInfo[ instanceId ] = {}
               instanceInfo[ instanceId ][ 'hosts' ] = {}
               instanceInfo[ instanceId ][ 'switches' ] = {}
               instanceInfo[ instanceId ][ 'name' ] = instance.instanceName

            hostInfo = {}
            for portId, port in instance.ports.items():
               for hostName, host in port.hosts.items():
                  portInfo = {}
                  if hostName not in hostInfo:
                     hostInfo[ hostName ] = {}
                     hostInfo[ hostName ][ 'networks' ] = {}
                  networkInfo = hostInfo[ hostName ][ 'networks' ]
                  networkId = host.networkId
                  networkName = host.networkName
                  if networkId not in networkInfo:
                     networkInfo[ networkId ] = { 'name' : networkName,
                                                'ports' : {} }
                  portInfo[ portId ] = {}
                  portInfo[ portId ] = { 'segments' : [] }
                  portInfo[ portId ][ 'segments' ].extend( host.segments )
                  networkInfo[ networkId ][ 'ports' ].update( portInfo )
                  hostInfo[ hostName ][ 'networks' ].update( networkInfo )
               instanceInfo[ instanceId ][ 'hosts' ].update( hostInfo )

               switchInfo = {}
               for switchId, switch in port.switches.items():
                  portInfo = {}
                  if switchId not in switchInfo:
                     switchInfo[ switchId ] = {}
                     switchInfo[ switchId ][ 'networks' ] = {}
                  networkInfo = switchInfo[ switchId ][ 'networks' ]
                  networkId = switch.networkId
                  networkName = switch.networkName
                  networkInfo[ networkId ] = { 'name' : networkName,
                                               'ports' : {} }
                  portInfo[ portId ] = {}
                  portInfo[ portId ] = { 'segments' : [] }
                  portInfo[ portId ][ 'segments' ].extend( switch.segments )
                  networkInfo[ networkId ][ 'ports' ].update( portInfo )
                  switchInfo[ switchId ][ 'networks' ].update( networkInfo )
               instanceInfo[ instanceId ][ 'switches' ].update( switchInfo )
            tenantInfo[ tn ][ 'instances' ].update( instanceInfo )

         if tenantInfo:
            if regionName not in regionData:
               regionData[ regionName ] = {}
               regionData[ regionName ][ 'tenants' ] = {}
            regionData[ regionName ][ 'tenants' ].update( tenantInfo )
      return regionData

   def render( self ):
      regionData = self.orderData()

      for regionName, tenants in sorted( regionData.items(),
                                         key=lambda x: x[ 0 ] ):
         print( 'Region: %s' % regionName )
         for tenantId, instances in sorted( tenants[ 'tenants' ].items(),
               key=lambda x: x[ 1 ][ 'name' ] ):
            indent = ' ' * 2
            print( '{}Tenant: {} ( {} ) '.format(
               indent, tenants[ 'tenants' ][ tenantId ][ 'name' ], tenantId ) )
            for instanceId, instance in sorted( instances[ 'instances' ].items(),
                                          key=lambda x: x[ 1 ][ 'name' ] ):
               indent = ' ' * 4
               print( '{}Instance: {} ( {} ) '.format( indent, instance[ 'name' ],
                                                   instanceId ) )
               for elem in [ 'hosts', 'switches' ]:
                  for elemName, ne in sorted( instance[ elem ].items(),
                                          key=lambda x: x[ 0 ] ):
                     indent = ' ' * 6
                     print( '{}{}: {}'.format( indent,
                                           'Host' if elem == 'hosts' else 'Switch',
                                           elemName ) )
                     for networkId, network in sorted( ne[ 'networks' ].items(),
                                          key=lambda x: x[ 1 ][ 'name' ] ):
                        indent = ' ' * 8
                        print( '{}Network: {} ( {} )'.format( indent,
                                                          network[ 'name' ],
                                                          networkId ) )
                        firstPort = True
                        segmentTable = None
                        for portId, port in sorted( network[ 'ports' ].items(),
                              key=lambda x: x[ 0 ] ):
                           if firstPort:
                              segmentTableHeaders = [
                                         ( createHdr( "Port Id" ), "l" ),
                                         ( createHdr( "Segment Id" ), "l" ),
                                         ( createHdr( "Segment Type" ), "l" ),
                                         ( createHdr( "Segmentation Id" ), "l" ) ]
                              segmentTable = createTable(
                                 tuple( segmentTableHeaders ), indent=8 )
                              firstPort = False
                           segList = []
                           for segment in port[ 'segments' ]:
                              row = [ portId,
                                     segment.segmentId,
                                     segment.segmentationType,
                                     segment.segmentationTypeId ]
                              segList.append( tuple( row ) )
                           firstRow = True
                           for seg in segList:
                              if firstRow:
                                 segmentTable.newRow( *seg )
                                 firstRow = False
                              else:
                                 segmentTable.newRow( '',
                                                     seg[ 1 ], seg[ 2 ], seg[ 3 ] )
                        if segmentTable:
                           print( segmentTable.output() )

# Networks Class
class TenantNetworks( Model ):
   regions = Dict( valueType=Region, help='Collection of regions' )
   neighbors = Dict( valueType=DirectedEdge,
                     help='Collection of neighbors in topology', optional=True )
   hosts = Dict( valueType=Host, help='Collection of hosts in topology',
                 optional=True )
   detail = Bool( help='Display detailed information', optional=True )

   def renderSimple( self, regionTable ):
      for _, region in sorted( self.regions.items() ):
         writeRegion = True
         for _, tenant in sorted( region.tenants.items(),
                                  key=lambda x: x[ 1 ].tenantName ):
            tenantTable = TableFormatter( indent=2 )
            tenantTable.formatColumns( Format( justify="left" ) )
            tenantTable.newRow( 'Tenant Name: %s' % tenant.tenantName )
            tenantTable.newRow( 'Tenant Id: %s' % tenant.tenantId )
            tenantTable.newRow( '' )

            tableHeaders = [ ( createHdr( "Network Name" ), "l" ),
                             ( createHdr( "Network Id" ), "l" ),
                             ( createHdr( "Network Type" ), "l" ),
                             ( createHdr( "Seg Id" ), "l" ),
                           ]

            networkTable = createTable( tuple( tableHeaders ), indent=1 )

            for _, network in sorted( tenant.tenantNetworks.items(),
                                      key=lambda x: x[ 1 ].networkName ):

               row = [ network.networkName,
                       network.networkId,
                       network.segmentationType,
                       network.segmentationTypeId,
                     ]

               networkTable.newRow( *row )

            if len( tenant.tenantNetworks ):
               if writeRegion:
                  regionTable.newRow( 'Region: %s\n' % region.regionName )
                  writeRegion = False
               tenantTable.newRow( networkTable.output() )
               regionTable.newRow( tenantTable.output() )

   def renderDetail( self, regionTable ):
      for _, region in sorted( self.regions.items() ):
         writeRegion = True
         for _, tenant in sorted( region.tenants.items(),
                                  key=lambda x: x[ 1 ].tenantName ):
            tenantTable = TableFormatter( indent=2 )
            tenantTable.formatColumns( Format( justify="left" ) )
            tenantTable.newRow( 'Tenant Name: %s' % tenant.tenantName )
            tenantTable.newRow( 'Tenant Id: %s' % tenant.tenantId )
            tenantTable.newRow( '' )
            networkTable = TableFormatter( indent=2 )
            networkTable.formatColumns( Format( justify="left" ) )

            for _, network in sorted( tenant.tenantNetworks.items(),
                                      key=lambda x: x[ 1 ].networkName ):
               networkTable.newRow( 'Network Name: %s' % network.networkName )
               networkTable.newRow( 'Network Id: %s' % network.networkId )
               if network.shared:
                  networkTable.newRow(
                     'This network is shared across all tenants in %s.' % (
                        region.regionName ) )
               if network.dhcpInstances:
                  for _, dhcpInstance in sorted( network.dhcpInstances.items(),
                                              key=lambda x: x[ 1 ].dhcpInstanceId ):
                     segmentList = []
                     networkTable.newRow( 'DHCP VM Id: %s' %
                           dhcpInstance.dhcpInstanceId )
                     networkTable.newRow( '      Host: %s' %
                           dhcpInstance.dhcpHostId )
                     # Only one port is associated with a dhcpInstance
                     if not dhcpInstance.dhcpPorts:
                        continue
                     port = next( iter( dhcpInstance.dhcpPorts.values() ) )
                     if not port:
                        continue
                     segmentList += port.segments
                     networkTable.newRow( '      Port Id: %s' % port.portId )
                     networkTable.newRow( '      Port Name: %s' % port.portName )
                     networkTable.newRow( '' )

                     topoTable = createTable( (
                        ( createHdr( "Switch Name" ), "l" ),
                        ( createHdr( "Switch Id" ), "l" ),
                        ( createHdr( "Switch Interface" ), "l" ),
                        ( createHdr( "Segment(s)" ), "l" ) ), indent=1 )

                     hostList = set()
                     h = self.hosts.get( dhcpInstance.dhcpHostId )
                     if not h:
                        continue
                     # TODO: Get all hosts for the instance
                     instanceSwitchports = getPortSwitchports( self.hosts,
                                                               self.neighbors,
                                                               h, port )
                     for ( switchHostname, switchName,
                           switchport ) in instanceSwitchports:
                        if segmentList:
                           segmentInfo = MAP_CHARS.join( [ '{}:{}'.format(
                                 segment.segmentationType,
                                 segment.segmentationTypeId )
                                 for segment in segmentList[ : : -1 ] ] )
                           if network.mappedVni:
                              # VLAN to VNI map is configured
                              segmentInfo += (
                                 MAP_CHARS + "vxlan:%s" ) % network.mappedVni
                           listItems = [ switchHostname,
                                         switchName,
                                         switchport,
                                         segmentInfo ]
                        else:
                           segmentInfo = "{}:{}".format( network.segmentationType,
                                                         network.segmentationTypeId )
                           if network.mappedVni:
                              segmentInfo = (
                                 MAP_CHARS + "vxlan:%s" ) % network.mappedVni
                           listItems = [ switchHostname,
                                         switchName,
                                         switchport,
                                         segmentInfo ]
                        hostList.add( tuple( listItems ) )

                     if hostList:
                        hostList = sorted( hostList, key=lambda x: (
                           x[ 0 ], x[ 1 ], Arnet.intfNameKey( x[ 2 ] ) ) )
                        for item in hostList:
                           topoTable.newRow( *item )
                        networkTable.newRow( topoTable.output() )

            if len( tenant.tenantNetworks ):
               if writeRegion:
                  regionTable.newRow( 'Region: %s\n' % region.regionName )
                  writeRegion = False
               tenantTable.newRow( networkTable.output() )
               regionTable.newRow( tenantTable.output() )

   def render( self ):
      regionTable = TableFormatter()
      regionTable.formatColumns( Format( justify="left" ) )

      if self.detail:
         self.renderDetail( regionTable )
      else:
         self.renderSimple( regionTable )

      sys.stdout.write( regionTable.output() )

# Instances Class
class Instances( Model ):
   regions = Dict( valueType=Region, help='Collection of regions' )
   neighbors = Dict( valueType=DirectedEdge,
                     help='Collection of neighbors in topology', optional=True )
   hosts = Dict( valueType=Host, help='Collection of hosts in topology',
                 optional=True )
   detail = Bool( help='Display detailed information', optional=True )

   def renderSimple( self, regionTable ):
      outputGenerated = False
      for _, region in sorted( self.regions.items() ):
         writeRegion = True
         for _, tenant in sorted( region.tenants.items(),
                                  key=lambda x: x[ 1 ].tenantName ):
            tenantTable = TableFormatter( indent=2 )
            tenantTable.formatColumns( Format( justify="left" ) )
            tenantTable.newRow( 'Tenant Name: %s' % tenant.tenantName )
            tenantTable.newRow( 'Tenant Id: %s' % tenant.tenantId )
            tenantTable.newRow( '' )
            instanceTable = createTable( ( ( createHdr( "Instance Name" ), "l" ),
                                           ( createHdr( "Instance Id" ), "l" ),
                                           ( createHdr( "Host" ), "l" ),
                                           ( createHdr( "Network Name" ), "l" ),
                                           ( createHdr( "Type" ), "l" ) ),
                                   indent=1 )
            genOutput = False
            for instanceType in [ 'virtual', 'baremetal', 'router' ]:
               instances = tenant.getInstancesOfType( instanceType )
               for instance in sorted( instances.values(),
                                       key=lambda x: x.getInstanceName() ):
                  instanceList = []
                  for port in instance.getInstancePorts().values():
                     network = tenant.tenantNetworks.get( port.networkId )
                     networkName = network.networkName
                     if network.shared:
                        networkName += '*'
                     # TODO: Print multiple hosts if port has multiple hosts
                     instanceList.append( ( instance.getInstanceName(),
                                            instance.getInstanceId(),
                                            instance.getInstanceHost(),
                                            networkName,
                                            instanceType ) )
                     genOutput = True
                  if len( instanceList ) > 0:
                     instanceList.sort( key=lambda x: x[ 3 ] )
                     firstPort = True
                     for item in instanceList:
                        if firstPort:
                           instanceTable.newRow( *item )
                           firstPort = False
                        else:
                           instanceTable.newRow( '', '', '', item[ 3 ] )

            if genOutput:
               if writeRegion:
                  regionTable.newRow( 'Region: %s\n' % region.regionName )
                  writeRegion = False
               tenantTable.newRow( instanceTable.output() )
               regionTable.newRow( tenantTable.output() )
               outputGenerated = True

      return outputGenerated

   def renderDetail( self, regionTable ):
      outputGenerated = False
      for _, region in sorted( self.regions.items() ):
         writeRegion = True
         for _, tenant in sorted( region.tenants.items(),
                                  key=lambda x: x[ 1 ].tenantName ):
            writeTenant = False
            tenantTable = TableFormatter( indent=2 )
            tenantTable.formatColumns( Format( justify="left" ) )
            tenantTable.newRow( 'Tenant Name: %s' % tenant.tenantName )
            tenantTable.newRow( 'Tenant Id: %s' % tenant.tenantId )
            tenantTable.newRow( '' )
            instanceTable = TableFormatter( indent=2 )
            instanceTable.formatColumns( Format( justify="left" ) )
            for instanceType in [ 'virtual', 'baremetal', 'router' ]:
               instances = tenant.getInstancesOfType( instanceType )
               for instance in sorted( instances.values(),
                                       key=lambda x: x.getInstanceName() ):
                  if len( instance.getInstancePorts() ) == 0:
                     continue
                  instanceHost = instance.getInstanceHost()
                  writeTenant = True
                  instanceTable.newRow( 'Instance Name: %s' %
                                        instance.getInstanceName() )
                  instanceTable.newRow( 'Instance Id: %s' %
                                         instance.getInstanceId() )
                  instanceTable.newRow( 'Instance Type: %s' % instanceType )
                  if instanceHost:
                     instanceTable.newRow( 'Host: %s' % instanceHost )
                  instanceTable.newRow( '' )
                  topoTableHeaders = [ ( createHdr( "Switch Name" ), "l" ),
                                       ( createHdr( "Switch Id" ), "l" ),
                                       ( createHdr( "Switch Interface" ), "l" ),
                                       ( createHdr( "Segment(s)" ), "l" ),
                                       ( createHdr( "Instance Port Name" ), "l" ),
                                     ]
                  portTableHeaders = [ ( createHdr( "Instance Port Id" ), "l" ),
                                       ( createHdr( "Instance Port Name" ), "l" ),
                                       ( createHdr( "Segment(s)" ), "l" ),
                                     ]

                  # We only print out a hosts column for instances that exist on
                  # multiple hosts
                  instanceHasMultipleHosts = instance.hasMultipleHosts()
                  if instanceHasMultipleHosts:
                     portTableHeaders += [ ( createHdr( "Host" ), "l" ) ]
                     topoTableHeaders += [ ( createHdr( "Host" ), "l" ) ]
                  portTable = createTable( tuple( portTableHeaders ), indent=1 )
                  topoTable = createTable( tuple( topoTableHeaders ), indent=1 )

                  someHostsNotFound = False
                  portDict = dict() # pylint: disable=use-dict-literal
                  for instancePort in sorted( instance.getInstancePorts().values(),
                                              key=lambda x: x.portName ):
                     network = tenant.tenantNetworks.get( instancePort.networkId )
                     if not network:
                        continue

                     segId = network.segmentationTypeId
                     # The host list is a union of the union of the instance's host
                     # and the port's hosts
                     portHosts = list() # pylint: disable=use-list-literal
                     if instanceHost and instanceHost not in [
                           'distributed', 'HA Router', '(see router ports)' ]:
                        portHosts.append( instanceHost )
                     if instancePort.hosts:
                        for h in instancePort.hosts:
                           if h != instanceHost:
                              portHosts.append( h )

                     firstHost = True
                     for portHost in portHosts:
                        sharedSegId = None
                        if network.shared:
                           sharedSegId = segId
                        h = self.hosts.get( portHost )
                        if h == None: # pylint: disable=singleton-comparison
                           someHostsNotFound = True
                           if firstHost:
                              firstHost = False
                              if instancePort.segments:
                                 segmentInfo = MAP_CHARS.join(
                                    [ '{}:{}'.format( segment.segmentationType,
                                      str( sharedSegId ) + '*'
                                      if segment.segmentationTypeId == sharedSegId
                                      else segment.segmentationTypeId )
                                       for segment in instancePort.segments[ : : -1 ]
                                    ] )
                                 if network.mappedVni:
                                    segmentInfo += (
                                       "%s vxlan:%s" %
                                       ( MAP_CHARS, network.mappedVni ) )

                                 row = [ instancePort.portId,
                                         instancePort.portName,
                                         segmentInfo
                                       ]
                                 if instanceHasMultipleHosts:
                                    row += [ portHost ]
                                 portTable.newRow( *row )
                              else:
                                 segmentInfo = "{}:{}".format(
                                    network.segmentationType,
                                    network.segmentationTypeId )
                                 row = [ instancePort.portId,
                                         instancePort.portName,
                                         segmentInfo
                                       ]
                                 if instanceHasMultipleHosts:
                                    row += [ portHost ]
                                 portTable.newRow( *row )
                           else:
                              portTable.newRow( '', '', '', '', portHost )
                        else:
                           portSwitchports = getPortSwitchports( self.hosts,
                                                                 self.neighbors,
                                                                 h, instancePort )
                           for ( switchHostname, switchName,
                                 switchport ) in portSwitchports:
                              if instancePort.segments:
                                 segmentInfo = MAP_CHARS.join(
                                    [ '{}:{}'.format(
                                      segment.segmentationType,
                                      segment.segmentationTypeId )
                                      for segment in instancePort.segments[ : : -1 ]
                                    ]
                                 )
                                 if network.mappedVni:
                                    segmentInfo += ( MAP_CHARS + "vxlan:%s" ) % (
                                       network.mappedVni )
                                 connection = [ switchport,
                                                segmentInfo,
                                                instancePort.portName,
                                              ]
                                 if instanceHasMultipleHosts:
                                    connection += [ portHost ]
                              else:
                                 segmentInfo = "{}:{}".format(
                                    network.segmentationType,
                                    network.segmentationTypeId )
                                 connection = [ switchport,
                                                segmentInfo,
                                                instancePort.portName,
                                              ]
                                 if instanceHasMultipleHosts:
                                    connection += [ portHost ]
                              if ( switchHostname,
                                   switchName ) not in portDict:
                                 portDict[ ( switchHostname, switchName ) ] = set()
                              portDict[ ( switchHostname, switchName ) ].add(
                                 tuple( connection ) )

                  if len( portDict ) == 0:
                     instanceTable.newRow(
                        " !!! Host Topology Information Unavailable !!!" )
                  elif someHostsNotFound:
                     instanceTable.newRow(
                        " !!! Some Host Topology Information Unavailable !!!" )
                  if len( portDict ) > 0:
                     for ( switchHostname, switchName ) in sorted( portDict ):
                        portDict[ ( switchHostname, switchName ) ] = list(
                           portDict[ ( switchHostname, switchName ) ] )
                        portDict[ ( switchHostname, switchName ) ].sort(
                           key=lambda x: ( Arnet.intfNameKey( x[ 0 ] ), x[ 2 ] ) )
                        firstLine = True
                        for item in portDict[ ( switchHostname, switchName ) ]:
                           if firstLine:
                              topoTable.newRow( switchHostname, switchName, *item )
                              firstLine = False
                           else:
                              topoTable.newRow( '', '', *item )
                     instanceTable.newRow( '' )
                     instanceTable.newRow( topoTable.output() )
                  if someHostsNotFound:
                     instanceTable.newRow( '' )
                     instanceTable.newRow( portTable.output() )

            if ( len( tenant.tenantVmInstances ) or
                 len( tenant.tenantBaremetalInstances ) or
                 len( tenant.tenantRouterInstances ) ) and writeTenant:
               if writeRegion:
                  regionTable.newRow( 'Region: %s\n' % region.regionName )
                  writeRegion = False
               tenantTable.newRow( instanceTable.output() )
               regionTable.newRow( tenantTable.output() )
               outputGenerated = True
      return outputGenerated

   def render( self ):
      # Don't display VMs with no ports
      regionTable = TableFormatter()
      regionTable.formatColumns( Format( justify="left" ) )

      outputGenerated = False
      if self.detail:
         outputGenerated = self.renderDetail( regionTable )
      else:
         outputGenerated = self.renderSimple( regionTable )

      sys.stdout.write( regionTable.output() )
      if outputGenerated:
         sys.stdout.write( "\t* Indicates that the network is shared across tenants"
                          " in a region." )
         sys.stdout.write( "\n" )

class Vms( Instances ):
   # Keep this model around (as a copy of Instances) to allow show openstack vms
   # to continue returning this model
   pass

class Interface( Model ):
   name = Str( help='Interface name (Ethernet1, Management1 etc)' )
   allowedVlans = Str( help='Allowed list of trunk vlans on the interface' )
   nativeVlan = Int( help='Native vlan on the interface', optional=True )

class Switch( Model ):
   switchId = Str( help='Unique identifier (typically MAC-addr) for the switch' )
   hostname = Str( help='Hostname configured by user' )
   interfaces = Dict( valueType=Interface, help='Collection of interfaces on '
                      'this switch' )

class PhysicalNetwork( Model ):
   regions = Dict( valueType=Region, help='Collection of regions' )
   neighbors = Dict( valueType=DirectedEdge,
                     help='Collection of neighbors in topology', optional=True )
   hosts = Dict( valueType=Host, help='Collection of hosts in topology',
                 optional=True )
   detail = Bool( help='Display detailed information', optional=True )
   switches = Dict( valueType=Switch, help='Collection of switches' )

   def orderDataByHost( self ):
      ''' Order tenants, networks and VMs by the host '''
      tenantInfo = {}
      networkInfo = {}
      instanceInfo = {}
      for region in self.regions:
         for tenantId in self.regions[ region ].tenants:
            tenant = self.regions[ region ].tenants[ tenantId ]
            for instanceType in [ 'virtual', 'baremetal', 'router' ]:
               instances = tenant.getInstancesOfType( instanceType )
               for instance in instances.values():
                  # TODO: Get all hosts for DVR port
                  instanceHost = instance.getInstanceHost()
                  if instance.getInstanceHost() in instanceInfo:
                     if region in instanceInfo[ instanceHost ]:
                        instanceInfo[ instanceHost ][ region ].append( instance )
                     else:
                        instanceInfo[ instanceHost ][ region ] = [ instance ]
                  else:
                     instanceInfo[ instanceHost ] = {}
                     instanceInfo[ instanceHost ][ region ] = [ instance ]
                  for port in instance.getInstancePorts().values():
                     if instanceHost not in tenantInfo:
                        tenantInfo[ instanceHost ] = {}
                     if region not in tenantInfo[ instanceHost ]:
                        tenantInfo[ instanceHost ][ region ] = set()

                     if instanceHost not in networkInfo:
                        networkInfo[ instanceHost ] = {}
                     if region not in networkInfo[ instanceHost ]:
                        networkInfo[ instanceHost ][ region ] = set()

                     tenantInfo[ instanceHost ][ region ].add( tenant )
                     networkInfo[ instanceHost ][ region ].add(
                        tenant.tenantNetworks[ port.networkId ] )

      return tenantInfo, networkInfo, instanceInfo

   def buildSwitchData( self, switch, tenantInfo, networkInfo, instanceInfo ):
      ''' Return a dictionary on switch interfaces. Each interface has a
          dictionary of tenants which in turn contains a dictionary of instances.
          Each instance has a dictionary of networks it belongs to.
          A tenant is present in the interface only if the host connected to
          the interface has a instance belonging to the tenant '''

      # Sort the interfaces based on the interface name
      sortedInterfaces = sorted( switch.interfaces.values(),
                                 key=lambda x: Arnet.intfNameKey( x.name ) )

      interfaces = {}
      for intf in sortedInterfaces:

         allowedVlans = "None"
         if intf.allowedVlans != "":
            allowedVlans = intf.allowedVlans

         switchIfName = f"{switch.hostname}-{intf.name}"
         if ( switchIfName not in self.neighbors or
              # pylint: disable-next=use-implicit-booleaness-not-len
              not len( self.neighbors[ switchIfName ].toPort ) ):
            continue

         connectedHost = self.neighbors[ switchIfName ].toPort[ 0 ].hostname

         if not connectedHost:
            # There is nothing connected to the interface
            continue

         intfData = {
            'name' : intf.name,
            'allowedVlans' : allowedVlans,
            'connectedHost' : connectedHost,
            'tenants' : {}
         }

         interfaceTenants = intfData[ 'tenants' ]

         if connectedHost in instanceInfo and \
                len( instanceInfo[ connectedHost ] ):
            for region in instanceInfo[ connectedHost ]:

               # These conditions can be true because of the filters
               if connectedHost not in tenantInfo: # pylint: disable=no-else-continue
                  continue
               elif region not in tenantInfo[ connectedHost ]:
                  continue

               for tenant in tenantInfo[ connectedHost ][ region ]:

                  tenantData = {
                     'name' : tenant.tenantName,
                     'id' : tenant.tenantId,
                     'region' : region,
                     'instances' : {}
                  }

                  tenantInstances = tenantData[ 'instances' ]

                  for instance in instanceInfo[ connectedHost ][ region ]:
                     instanceId = instance.getInstanceId()
                     if not tenant.hasInstanceId( instanceId ):
                        continue
                     instanceName = instance.getInstanceName()
                     instancePorts = instance.getInstancePorts()

                     instanceData = {
                        'name' : instanceName,
                        'id' : instanceId,
                        'networks' : {},
                        'ports' : {}
                     }
                     tenantInstances[ instanceId ] = instanceData
                     instanceNetworks = instanceData[ 'networks' ]
                     instPorts = instanceData[ 'ports' ]
                     for port in instancePorts.values():
                        networkId = port.networkId
                        network = tenant.tenantNetworks[ networkId ]
                        instPorts[ port.portId ] = port
                        for hostNetwork in networkInfo[ connectedHost ][ region ]:
                           if hostNetwork.networkId == networkId:
                              h = self.hosts[ connectedHost ]
                              portSwitchports = getPortSwitchports( self.hosts,
                                                                    self.neighbors,
                                                                    h,
                                                                    port )
                              # pylint: disable-next=use-implicit-booleaness-not-len
                              if len( portSwitchports ):
                                 networkData = {
                                    'name' : network.networkName,
                                    'id' : network.networkId,
                                    'type' : network.segmentationType,
                                    'sId' : network.segmentationTypeId
                                 }
                                 instanceNetworks[ network.networkId ] = networkData

                     if len( instanceNetworks ):
                        interfaceTenants[ tenant.tenantId ] = tenantData
                        # Add the intfData to the interface only if there is some
                        # data to display
                        if intf.name not in interfaces:
                           interfaces[ intf.name ] = intfData

      return interfaces

   def renderSwitch( self, switch, tenantInfo, networkInfo, instanceInfo ):
      switchData = self.buildSwitchData( switch, tenantInfo, networkInfo,
                                         instanceInfo )

      # If the switch does not contain any data, then do not print anything.
      if not len( switchData ): # pylint: disable=use-implicit-booleaness-not-len
         return

      # Begin switch section
      sys.stdout.write( "Switch: {} ( {} )\n".format( switch.switchId,
                                                  switch.hostname ) )
      for intfName in sorted( switchData.keys(),
                              # pylint: disable-next=unnecessary-lambda
                              key=lambda x: Arnet.intfNameKey( x ) ):
         # Begin interface section
         indent = ' ' * 2
         intf = switchData[ intfName ]
         sys.stdout.write( "{}{}:\n".format( indent, intf[ 'name' ] ) )
         indent = ' ' * 4
         sys.stdout.write( "{}Allowed vlans: {}\n".format( indent,
                                                       intf[ 'allowedVlans' ] ) )
         sys.stdout.write( "{}Connected host: {}\n".format( indent,
                                                        intf[ 'connectedHost' ] ) )

         for tenantId in intf[ 'tenants' ]:
            tenant = intf[ 'tenants' ][ tenantId ]
            indent = ' ' * 6
            # Begin tenant section
            sys.stdout.write( "{}Region: {}\n".format( indent, tenant[ 'region' ] ) )
            indent = ' ' * 8
            sys.stdout.write( "{}Tenant: {} ( {} )\n".format( indent,
                                tenant[ 'name' ],
                                tenant[ 'id' ] ) )
            tenantInstances = intf[ 'tenants' ][ tenantId ][ 'instances' ]
            indent = ' ' * 10
            for instance in tenantInstances.values():
               # Begin VM section
               sys.stdout.write( "{}Instance: {} ( {} )\n".format( indent,
                                                               instance[ 'name' ],
                                                               instance[ 'id' ] ) )
               # Begin network table
               netTable = createTable( ( ( createHdr( "Network Name" ), "l" ),
                                         ( createHdr( "Network Id" ), "l" ),
                                         ( createHdr( "Segment(s)" ), "l" )
                                        ), indent=10
                                     )
               netInfo = set()
               for port in instance[ 'ports' ].values():
                  segmentInfo = MAP_CHARS.join(
                     [ '{}:{}'.format(
                        segment.segmentationType,
                        segment.segmentationTypeId )
                        for segment in port.segments[ : : - 1 ] ] )
                  netInfo.add( tuple( [
                                 instance[ 'networks' ][ port.networkId ][ 'name' ],
                                 port.networkId, segmentInfo ] ) )
               for ent in netInfo:
                  netTable.newRow( *ent )

               sys.stdout.write( netTable.output() )
               sys.stdout.write( '\n' )
      return

   def render( self ):
      tenantInfo, networkInfo, instanceInfo = self.orderDataByHost()
      switchIds = {}

      for switchId in self.switches:
         switchIds[ self.switches[ switchId ].hostname ] = switchId

      sys.stdout.write( '\n' )

      for hostName in sorted( switchIds.keys() ):
         switchId = switchIds[ hostName ]
         self.renderSwitch( self.switches[ switchId ], tenantInfo, networkInfo,
                            instanceInfo )

class VlanInfo( Model ):
   switches = Dict( valueType=Switch,
                    help='Collection of switches and their vlan information' )

   def render( self ):
      vlanTable = createTable( ( ( createHdr( "Switch Name" ), "l" ),
                                 ( createHdr( "Switch Id" ), "l" ),
                                 ( createHdr( "Interface" ), "l" ),
                                 ( createHdr( "Vlans" ), "l" ) ) )

      for switch in sorted( self.switches.values(), key=lambda x: x.hostname ):
         if len( switch.interfaces ):
            newline = False
            for intf in sorted( switch.interfaces.values(),
                                key=lambda x: Arnet.intfNameKey( x.name ) ):
               if newline:
                  vlanTable.newRow( "", "", intf.name, intf.allowedVlans )
               else:
                  vlanTable.newRow( switch.hostname, switch.switchId, intf.name,
                                intf.allowedVlans )
               newline = True

      if len( self.switches ):
         sys.stdout.write( vlanTable.output() )

class Range( Model ):
   start = Int( help='Start of range' )
   end = Int( help='End of range' )

def sortRange( unsortedRange ):
   ''' This method sorts the range based on the start. The unsortedRange
   contains a list of strings of the form <start>-<end>. '''
   return sorted( unsortedRange, key=lambda x: int( x.split( '-' )[ 0 ] ) )

class VlanToVniMap( Model ):
   mapping = Dict( valueType=str,
                   help='Mapping from vlan range to vni range' )

   def render( self ):
      mappingTable = createTable( ( ( createHdr( "VLAN range" ), "l" ),
                                    ( createHdr( "VNI range" ), "l" ) ) )
      sortedVlanRange = sortRange( list( self.mapping ) )
      for vlanRange in sortedVlanRange:
         vniRange = self.mapping[ vlanRange ]
         mappingTable.newRow( vlanRange, vniRange )

      if self.mapping:
         sys.stdout.write( mappingTable.output() )

class RegionVlanToVniMap( Model ):
   vlanToVniMap = Dict( valueType=VlanToVniMap,
                        help='VLAN to VNI mapping keyed by OpenStack region names' )

   def render( self ):
      for region in sorted( self.vlanToVniMap ):
         sys.stdout.write( "Region: %s\n\n" % region )
         vniMap = self.vlanToVniMap[ region ]
         vniMap.render()
         sys.stdout.write( "\n" )

class VlanPool( Model ):
   assignedVlans = Str( help='VLANs assigned to the OpenStack region' )
   allocatedVlans = Str( help='VLANs that are being used by the OpenStack region' )
   availableVlans = Str( help='VLANs that are unused in the OpenStack region\'s'
                         'VLAN assignment' )

   def render( self ):
      print( "Assigned VLANs: %s" % self.assignedVlans )
      print( "Allocated VLANs: %s" % self.allocatedVlans )
      print( "Available VLANs: %s" % self.availableVlans )

class PhysicalNetworkVlanPool( Model ):
   vlanPool = Dict( valueType=VlanPool,
                    help='VLAN pools keyed by physical network names' )

   def render( self ):
      for physNet in sorted( self.vlanPool.keys() ):
         vlanPool = self.vlanPool[ physNet ]
         vlanPool.render()

class RegionPhysicalNetwork( Model ):
   physicalNetwork = Dict( valueType=PhysicalNetworkVlanPool,
                           help='Physical networks keyed by OpenStack region names' )

   def render( self ):
      for region in sorted( self.physicalNetwork.keys() ):
         print( "Region: %s\n" % region )
         physicalNet = self.physicalNetwork[ region ]
         physicalNet.render()
         print( "" )

class GracePeriod( Model ):
   gracePeriod = Float( help='Grace period in seconds for which the OpenStack agent'
                             ' waits for OpenStack region data' )

   def render( self ):
      sys.stdout.write( 'Grace period: %0.2f seconds' % self.gracePeriod )
      sys.stdout.write( '\n' )

class RegionSyncTimeout( Model ):
   regionSyncTimeout = Float( help='Timeout value in seconds for region'
                                   ' syncronization' )

   def render( self ):
      sys.stdout.write( 'Region sync timeout deprecated '
                        '( see sync interval in \'show openstack regions\' )' )
      sys.stdout.write( '\n' )

class AgentUuid( Model ):
   uuid = Str( help='OpenStack agent UUID' )

   def render( self ):
      sys.stdout.write( 'OpenStack agent UUID: %s' % self.uuid )
      sys.stdout.write( '\n' )

class VlanAssignmentUuid( Model ):
   uuid = Str( help='OpenStack VLAN assignment UUID' )

   def render( self ):
      if self.uuid:
         print( 'OpenStack VLAN assignment UUID: %s' % self.uuid )

class RegionLockOwner( Model ):
   owner = Str( help='client id of the lock owner' )
   requestId = Str( help='Unique request id' )

   def render( self ):
      if self.owner:
         sys.stdout.write( 'Region lock is owned by %s\n' % self.owner )
         sys.stdout.write( 'Lock request id: %s\n' % self.requestId )
      else:
         sys.stdout.write( 'Region is not locked\n' )

class OpenStackFeatures( Model ):
   features = Dict( keyType=str, valueType=int, help='OpenStack Features' )

   def render( self ):
      for name, version in sorted( self.features.items() ):
         print( 'Name: %s' % name )
         print( 'Version: %d\n' % version )
