# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
from CliModel import (
      Bool,
      Enum,
      Int,
      List,
      Str,
      Submodel,
      Dict,
      Model
)
from CliPlugin.SegmentRoutingCliShowHelper import (
   SrAdjacencySegmentFlagsModel,
   SrAdjacencySegmentModel,
   npHelpStr,
   reachabilityAlgorithmEnumValues,
   renderSRGB,
   renderCommonSummary,
   renderPrefixSegmentSummary,
   SrCommonHeader,
   SelfOriginatedSegmentStatisticsModel,
   SrPrefixSegmentFlagsModel,
   SrPrefixSegmentModel,
   SrGlobalBlockModel,
)
from CliPlugin.TunnelModels import TunnelTableEntry, Via
from TableOutput import terminalWidth
import TableOutput
import Toggles.gatedToggleLib
import TunnelLib

class OspfSrCommonModel( SrCommonHeader ):
   instanceId = Int( help='OSPF instance ID' )

   def render( self ):
      print( "OSPF Instance ID: %s" % self.instanceId )
      SrCommonHeader.render( self )

class OspfShowSrModel( OspfSrCommonModel ):
   """Model for show ip ospf segment-routing
   """
   srgbBase = Int( help="Base of SRGB" )
   srgbSize = Int( help="Size of SRGB" )
   srPeerCount = Int( help="Number of OSPF SR capable peers" )
   reachabilityAlgorithm = Enum( values=reachabilityAlgorithmEnumValues,
                                 help="OSPF Reachability Algorithm" )
   mappingServer = Bool( help="This is a Segment Routing Mapping Server (SRMS)" )
   selfOriginatedSegmentStatistics = Submodel( 
         valueType=SelfOriginatedSegmentStatisticsModel,
         help="Self-originated segment statistics" )

   def render( self ):
      OspfSrCommonModel.render( self )
      renderSRGB( self )
      renderCommonSummary( self, "OSPF" )      
      self.selfOriginatedSegmentStatistics.render()

class OspfPrefixSegmentFlagsModel( SrPrefixSegmentFlagsModel ):
   """Model for flags used in prefix segment
   """
   np = Bool( help=npHelpStr, default=False )
   m = Bool( help="Set if the Prefix-SID was advertised by a "
                  "Segment Routing Mapping Server",
             default=False )

   def toStr( self ):
      # pylint: disable-next=bad-string-format-type
      flags = "NP:%d M:%d E:%d V:%d L:%d" % ( self.np, self.m, self.e,
                                              self.v, self.l )
      return flags

class OspfPrefixSegmentModel( SrPrefixSegmentModel ):
   """Model to store individual OSPF prefix segment record
   """
   flags = Submodel( valueType=OspfPrefixSegmentFlagsModel, help="Flags" )
   if Toggles.gatedToggleLib.toggleOspfTilfaEnabled():
      protection = Enum( values=[ "node", "link", "unprotected" ],
                         help="Protection mode of the prefix segment" )
   routerIds = List( valueType=str, help="Router IDs" )

class OspfShowSrPrefixSegmentSummaryModel( OspfSrCommonModel ):
   """Model for show ip ospf segment-routing prefix-segments
   """
   nodeSidCount = Int( help="Number of SR node segments" )
   proxyNodeSidCount = Int(help="Number of SR proxy node segments" )
   prefixSidCount = Int( help="Number of SR prefix segments" )
   prefixSegments = List( valueType=OspfPrefixSegmentModel,
                          help="Prefix segments" )

   def render( self ):
      OspfSrCommonModel.render( self )
      
      leftJustify = TableOutput.Format( justify='left' )
      leftJustify.padLimitIs( 1 )
      rightJustify = TableOutput.Format( justify='right' )
      rightJustify.padLimitIs( 1 )

      legend = \
         "Flag Descriptions: NP: No-PHP, M: Mapping Server, E: Explicit-NULL,\n" \
         "                   V: Value, L: Local\n" \
         "Segment status codes: * - Self originated Prefix"

      headings = ( "", "Prefix", "SID", "Type", "Flags", "Router ID", )
      if Toggles.gatedToggleLib.toggleOspfTilfaEnabled():
         headings += ( "Protection", )
         
      table = TableOutput.createTable( headings, tableWidth=terminalWidth() )
      if Toggles.gatedToggleLib.toggleOspfTilfaEnabled():
         table.formatColumns( leftJustify, leftJustify, rightJustify, leftJustify,
                              leftJustify, leftJustify, leftJustify )
      else:
         table.formatColumns( leftJustify, leftJustify, rightJustify, leftJustify,
                              leftJustify, leftJustify )
      
      for item in self.prefixSegments:
         for rtrId in item.routerIds:
            if Toggles.gatedToggleLib.toggleOspfTilfaEnabled():
               table.newRow( item.statusCode(), item.prefix, item.segmentId,
                             item.segmentType, item.flags.toStr(), rtrId,
                             item.protection )
            else:
               table.newRow( item.statusCode(), item.prefix, item.segmentId,
                             item.segmentType, item.flags.toStr(), rtrId )

      renderPrefixSegmentSummary( legend, table, self.nodeSidCount,
                                  self.prefixSidCount, self.proxyNodeSidCount )

class OspfSrTunnelTableEntry( TunnelTableEntry ):
   vias = List( valueType=Via, help="List of nexthops" )

   def renderOspfTunnelTableEntry( self, table, tunnelIndex ):
      nhStr = intfStr = labelsStr = '-'
      if not self.vias:
         table.newRow( tunnelIndex, str( self.endpoint ), nhStr, intfStr,
                       labelsStr )
         return

      # Vias are stored in a sorted order in the model itself.
      for i, via in enumerate( self.vias ):
         nhStr, intfStr = TunnelLib.getNhAndIntfStrs( via )
         labelsStr = '[ ' + ' '.join( via.labels ) + ' ]'
         if i == 0:
            table.newRow( tunnelIndex, str( self.endpoint ), nhStr, intfStr,
                          labelsStr )
         else:
            table.newRow( '', '', nhStr, intfStr, labelsStr )

class OspfShowSrTunnelModel( Model ):
   """Model for show ip ospf segment-routing tunnel
   """
   entries = Dict( keyType=int, valueType=OspfSrTunnelTableEntry,
                   help="Tunnel table entries keyed by tunnel index" )

   def render( self ):
      headings = ( "Index", "Endpoint", "Next Hop/Tunnel Index", "Interface",
                   "Labels" )
      fl = TableOutput.Format( justify='left' )
      fr = TableOutput.Format( justify='right' )
      table = TableOutput.createTable( headings )
      table.formatColumns( fr, fl, fl, fl, fl )

      for tunnelIndex, entry in sorted( self.entries.items() ):
         entry.renderOspfTunnelTableEntry( table, tunnelIndex )

      print( table.output() )

class OspfGlobalBlockModel( SrGlobalBlockModel ):
   """Model to store individual global block record.
   """
   routerId = Str( help="Router ID" )

class OspfShowSrGlobalBlocksSummaryModel( OspfSrCommonModel ):
   """Model for show ip ospf segment-routing global-blocks
   """
   srPeerCount = Int( help="Number of OSPF SR peers capable segments" )
   globalBlocks = List( valueType=OspfGlobalBlockModel, help="Global blocks" )

   def render(self):
      OspfSrCommonModel.render( self )
      
      headings = ( "Router ID", "Base", "Size" )
      table = TableOutput.createTable( headings )
      for item in self.globalBlocks:
         table.newRow( item.routerId, item.base, item.size )

      print( "Number of OSPF segment routing capable nodes excluding self: "\
         "%d" % self.srPeerCount )
      print()
      print( table.output() )

class OspfAdjacencySegmentFlagsModel( SrAdjacencySegmentFlagsModel ):
   """Model for flags used in adjacency segment
   """
   g = Bool( help="Adj-SID refers to a group of adjacencies", default=False )
   p = Bool( help="Adj-SID is persistent across router restart and/or"
             " interface flap", default=False )

   def toStr( self ):
      flags = ""
      flags += "B:%d " % ( self.b )
      flags += "V:%d " % ( self.v )
      flags += "L:%d " % ( self.l )
      flags += "G:%d " % ( self.g )
      flags += "P:%d" % ( self.p )
      return flags

class OspfAdjacencySegmentModel( SrAdjacencySegmentModel ):
   """Model to store individual adjacency segments record
   """
   flags = Submodel( valueType=OspfAdjacencySegmentFlagsModel, help="Flags" )

class OspfShowSrAdjacencySegmentSummaryModel( OspfSrCommonModel ):
   """Model for show ip ospf segment-routing adjacency-segments
   """
   adjSidAllocationMode = Enum( values=[ "none", "srOnly", "all" ],
                                help="Adj-SID allocation mode" )
   adjSidPoolBase = Int( help="Base of Adj-SID pool", optional=True )
   adjSidPoolSize = Int( help="Size of Adj-SID pool", optional=True )
   adjacencySegments = List( valueType=OspfAdjacencySegmentModel,
                             help="Adjacency Segments" )

   def render( self ):
      OspfSrCommonModel.render( self )

      srAdjSidAllocationModeEnumMapping = dict( [
         ( "none", "None" ),
         ( "srOnly", "SR-adjacencies" ),
         ( "all", "all-adjacencies" ) ] )
      print( "Adj-SID allocation mode: %s" %
             srAdjSidAllocationModeEnumMapping[ self.adjSidAllocationMode ] )

      if self.adjSidAllocationMode != "None":
         print( "Adj-SID allocation pool: Base: %s Size: %s" %
                ( str( self.adjSidPoolBase ).ljust( 10 ),
                  str( self.adjSidPoolSize ) ) )

      adjCnt = len( self.adjacencySegments )
      print( "Adjacency Segment Count: %d" % adjCnt )
      print( "Flag Descriptions: B: Backup, V: Value, L: Local,\n"
             "                   G: Group, P: Persistent\n" )
      print( "Segment Status codes: P2P - Point-to-Point adjacency, "
             "LAN - Broadcast adjacency\n" )

      headings = ( "Adj IP Address", "Local Intf", "SID", "SID Source",
                   "Flags", "Type", "Protection" )
      table = TableOutput.createTable( headings, tableWidth=terminalWidth() )
      sidOrigin = "Dynamic"
      for item in self.adjacencySegments:
         adjType = "LAN" if item.lan else "P2P"
         table.newRow( item.ipAddress, item.localIntf.shortName, item.sid, sidOrigin,
                       item.flags.toStr(), adjType, item.protection )
      print( table.output() )
