#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import AuthnUserPriorityCli as aupCli
import Tac
import AgentDirectory
import CliParser
from IpLibConsts import DEFAULT_VRF
import LazyMount
import ConfigMount
from Toggles.P4RuntimeConfigToggleLib import (
   toggleP4RuntimeAccountingRequestsEnabled,
   toggleP4RuntimeAUPEnabled,
   toggleP4RuntimeDscpEnabled )

p4RtPortMappingConfig = None
p4RtPortMappingStatus = None
p4RtDeviceConfig = None
p4RtSystemConfig = None
p4RtHwStatus = None
p4RtSystemStatus = None
sslStatus = None

# for the type OpenFlowTable::FlowMode
# pkgdeps: library OpenFlow
flowMode = Tac.Type( "OpenFlowTable::FlowMode" )
hwResourceNames = Tac.Value( "Hardware::Bfd::HwResourceNames" )

def p4RuntimeSystemConfig():
   return p4RtSystemConfig


def p4RuntimeSupportedGuard( mode, token ):
   if p4RtHwStatus.p4RuntimeSupported:
      return None
   return CliParser.guardNotThisPlatform

def getSslStatusProfile( mode ):
   return sslStatus.profileStatus

def setEnabled( mode, args ):
   p4RtSystemConfig.enabled = True

def clearEnabled( mode, args ):
   p4RtSystemConfig.enabled = False

def setChassisId( mode, args ):
   chassisId = args[ 'CHASSISID' ]
   p4RtSystemConfig.chassisId = chassisId

def clearChassisId( mode, args ):
   p4RtSystemConfig.chassisId = p4RtSystemConfig.chassisIdDefault

def setDeviceId( mode, args ):
   fapName = args[ 'FAPNAME' ]
   if hwResourceNames.fapYangName not in fapName:
      mode.addError( 'Invalid forwarding chip name: {}. '
         'Name should start with {}'.format( fapName,
            hwResourceNames.fapYangName ) )
      return
   deviceId = args[ 'DEVICEID' ]
   currFapName = p4RtDeviceConfig.deviceIdToFapName.get( deviceId )
   if currFapName and currFapName != fapName:
      errMsg = "Cannot configure forwarding chip {} with id {} since it is"\
            " configured for {}".format( fapName, deviceId, currFapName )
      # This device id is assigned for another fap already. reject it.
      mode.addError( errMsg )
      return

   # Delete the old reverse mapping if it exists
   oldDeviceId = p4RtDeviceConfig.fapNameToDeviceId.get( fapName )
   if oldDeviceId is not None:
      del p4RtDeviceConfig.deviceIdToFapName[ oldDeviceId ]
   p4RtDeviceConfig.deviceIdToFapName[ deviceId ] = fapName
   p4RtDeviceConfig.fapNameToDeviceId[ fapName ] = deviceId

def clearDeviceId( mode, args ):
   fapName = args[ 'FAPNAME' ]
   deviceId = p4RtDeviceConfig.fapNameToDeviceId.get( fapName )
   del p4RtDeviceConfig.fapNameToDeviceId[ fapName ]
   if deviceId is not None:
      del p4RtDeviceConfig.deviceIdToFapName[ deviceId ]

def setGrpcServerPort( mode, args ):
   portNum = args[ 'PORTNUM' ]
   p4RtSystemConfig.grpcServerPort = portNum

def setDefaultGrpcServerPort( mode=None, args=None ):
   P4RtSystemConfig = Tac.Value( "P4RuntimeConfig::ServerParams" )
   p4RtSystemConfig.grpcServerPort = P4RtSystemConfig.defaultPiServerPort

def setGrpcServerVrf( mode, args ):
   p4RtSystemConfig.grpcServerVrfName = args.get( 'VRFNAME', DEFAULT_VRF )

def clearGrpcServerVrf():
   p4RtSystemConfig.grpcServerVrfName = DEFAULT_VRF

def setTransportDscp( mode,args ):
   if not toggleP4RuntimeDscpEnabled():
      return
   p4RtSystemConfig.dscp = args.get( 'DSCP', 0 )

def setDefaultTransportDscp( mode=None,args=None ):
   if not toggleP4RuntimeDscpEnabled():
      return
   p4RtSystemConfig.dscp = 0

def setAccountingRequests( mode,args ):
   p4RtSystemConfig.accountingRequests = True

def clearAccountingRequests( mode=None,args=None ):
   p4RtSystemConfig.accountingRequests = False

# This makes sure that the port mapping configuration has been processed
# by the P4Runtime agent. This is our interlock to make sure that P4 table
# entries have the configured port mapping available by the time the gRPC
# is processed
def portMappingConfigProcessed():
   for p4PortId in p4RtPortMappingConfig.p4PortIdToIntfId:
      if p4RtPortMappingStatus.p4PortIdToIntfId.get( p4PortId ) != \
         p4RtPortMappingConfig.p4PortIdToIntfId[ p4PortId ]:
         return False
   return True

def maybeDeleteOldMapping( portMappingConfig, p4PortId, intfName ):
   prevP4PortId = portMappingConfig.intfIdToP4PortId.get( intfName )
   if prevP4PortId and p4PortId != prevP4PortId:
      del portMappingConfig.p4PortIdToIntfId[ prevP4PortId ]
      del portMappingConfig.intfIdToP4PortId[ intfName ]
   prevIntfName = portMappingConfig.p4PortIdToIntfId.get( p4PortId )
   if prevIntfName and intfName != prevIntfName:
      del portMappingConfig.p4PortIdToIntfId[ p4PortId ]
      del portMappingConfig.intfIdToP4PortId[ prevIntfName ]

def setPortMapping( mode, args ):
   portIdSet = list( args[ 'P4_PORT_ID_SET' ].values() )
   intfList = args[ 'INTFLIST' ]

   for intfs in intfList:
      for index, intfName in enumerate( intfs ):
         if index >= len( portIdSet ):
            break
         p4PortId = portIdSet[ index ]
         maybeDeleteOldMapping( p4RtPortMappingConfig, p4PortId, intfName )
         p4RtPortMappingConfig.p4PortIdToIntfId[ p4PortId ] = intfName
         p4RtPortMappingConfig.intfIdToP4PortId[ intfName ] = p4PortId

def clearPortMapping( mode, args ):
   portIdSet = list(
      args[ 'P4_PORT_ID_SET' ].values() ) if 'P4_PORT_ID_SET' in args else None
   if not portIdSet:
      portIdSet = p4RtPortMappingConfig.p4PortIdToIntfId
   for p4PortId in portIdSet:
      intfId = p4RtPortMappingConfig.p4PortIdToIntfId.get( p4PortId )
      del p4RtPortMappingConfig.p4PortIdToIntfId[ p4PortId ]
      if intfId is not None:
         del p4RtPortMappingConfig.intfIdToP4PortId[ intfId ]

P4RtParsingResultType = Tac.Type( 'P4Runtime::ParsingResult' )

def pipelineProcessed():
   return p4RtSystemStatus.parsingStatus.parsingResult != \
      P4RtParsingResultType.parsingNotStarted

def isP4RuntimeRunning( mode ):
   return bool( AgentDirectory.agent( mode.sysname, "P4Runtime" ) )

def setSslProfile( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   p4RtSystemConfig.sslProfileName = profileName

def clearSslProfile( mode=None, args=None ):
   p4RtSystemConfig.sslProfileName = ''

def setCertificateUsernameAuthn( mode, args ):
   if not toggleP4RuntimeAUPEnabled():
      p4RtSystemConfig.certUsernameAuthn = True
   else:
      p4RtSystemConfig.authnUsernamePriority.clear()
      for elm in aupCli.defaultAuthnUserPriorityWithCUA:
         p4RtSystemConfig.authnUsernamePriority.push( elm )

def clearCertificateUsernameAuthn( mode=None, args=None ):
   if not toggleP4RuntimeAUPEnabled():
      p4RtSystemConfig.certUsernameAuthn = False
   else:
      setDefaultAup()

def setDefaultAup( mode=None, args=None ):
   p4RtSystemConfig.authnUsernamePriority.clear()
   for elm in aupCli.defaultAuthnUserPriority:
      p4RtSystemConfig.authnUsernamePriority.push( elm )

def clearP4RtTransport():
   clearSslProfile()
   clearCertificateUsernameAuthn()
   setDefaultGrpcServerPort()
   setDefaultTransportDscp()
   clearGrpcServerVrf()
   if toggleP4RuntimeAccountingRequestsEnabled():
      clearAccountingRequests()
   p4RtSystemConfig.transportName = ""
   setDefaultAup()

def resetP4Runtime( mode, args ):
   clearEnabled( mode, args )
   p4RtDeviceConfig.fapNameToDeviceId.clear()
   p4RtDeviceConfig.deviceIdToFapName.clear()
   clearPortMapping( mode, args )
   clearChassisId( mode, args )
   clearP4RtTransport()

def Plugin( entityManager ):
   global p4RtHwStatus, p4RtSystemConfig
   global p4RtPortMappingConfig, p4RtPortMappingStatus
   global p4RtSystemStatus
   global sslStatus
   global p4RtDeviceConfig

   p4RtDeviceConfig = ConfigMount.mount( entityManager,
                                         "p4runtime/config/deviceConfig",
                                         "P4RuntimeConfig::P4RtDeviceConfig",
                                         "w" )
   p4RtSystemConfig = ConfigMount.mount( entityManager, "p4runtime/config/system",
                                         "P4RuntimeConfig::P4RtSystemConfig", "w" )
   p4RtPortMappingConfig = ConfigMount.mount(
      entityManager, "p4runtime/config/port-mapping/cli",
      "P4RuntimeConfig::PortMappingConfig", "w" )
   p4RtPortMappingStatus = LazyMount.mount(
      entityManager, "p4runtime/status/port-mapping",
      "P4RuntimeConfig::PortMappingStatus", "r" )
   p4RtHwStatus = LazyMount.mount( entityManager, "p4runtime/status/hardware/status",
                                   "P4Runtime::HwStatus", "r" )
   p4RtSystemStatus = LazyMount.mount( entityManager,
                                         "p4runtime/status/system",
                                         "P4RuntimeConfig::P4RtSystemStatus", "r" )
   sslStatus = LazyMount.mount( entityManager, "mgmt/security/ssl/status",
                             "Mgmt::Security::Ssl::Status", "r" )
