# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import ( Bool, # pylint: disable=unused-import
                       Dict,
                       Float,
                       GeneratorDict,
                       Int,
                       List,
                       Model,
                       Str,
                       Submodel )
from TableOutput import (
   Format,
   createTable,
)

from Toggles.P4RuntimeConfigToggleLib import (
   toggleP4RuntimeAccountingRequestsEnabled,
   toggleP4RuntimeAUPEnabled )

class P4RuntimeElectionId( Model ):
   high = Int( help="High 64 bits of the 128-bit election ID" )
   low = Int( help="Low 64 bits of the 128-bit election ID" )

   def render( self ):
      print( f"Primary Election ID: High: {self.high}, Low: {self.low}" )

class P4RtDeviceStatusEntry( Model ):
   deviceName = Str( help="Device name" )
   deviceId = Int( help='P4Runtime device ID' )
   primaryElectionId = Submodel( optional=True, valueType=P4RuntimeElectionId,
                                 help="Primary election ID" )

class P4RtTransport( Model ):
   __revision__ = 2
   port = Int( help="The port this server is listening on" )
   vrfName = Str( help="The VRF this server is listening in", default="default" )
   sslProfile = Str( help='SSL Profile', optional=True )
   if toggleP4RuntimeAUPEnabled():
      authnUsernamePriority = List( valueType=str,
            help="Authentication username extraction priority" )
   else:
      certUsernameAuthn = Bool( help="Use the username from the certificate"
            " for AAA authentication" )
   if toggleP4RuntimeAccountingRequestsEnabled():
      accountingRequests = Bool( help="P4Runtime RPC accounting requests is enabled")

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         dictRepr[ 'certUsernameAuthn' ] = False
      return dictRepr

class P4RtStatus( Model ):
   __revision__ = 2
   enabled = Bool( help="P4Runtime server is enabled" )
   transport = Submodel( valueType=P4RtTransport, optional=True,
                         help='P4Runtime server transport status' )
   numClients = Int( help="Number of P4Runtime client connections" )
   devices = GeneratorDict( keyType=str, valueType=P4RtDeviceStatusEntry,
                        optional=True,
                        help="P4Runtime device status entries keyed by device name" )
   authzEnabled = Bool( help="P4Runtime Authz is enabled" )
   lastServiceStartTimeStamp = Float( help="Timestamp of last gRPC service start" )

   def render( self ):
      print( "Enabled:", ( "yes" if self.enabled else "no" ) )
      if not self.enabled:
         return
      if self.transport:
         # pylint: disable-next=consider-using-f-string
         print( "Server: running on port {}, in {} VRF".format( self.transport.port,
                self.transport.vrfName ) )
      else:
         print( "Server: initializing" )
         return
      profile = self.transport.sslProfile if self.transport.sslProfile else 'none'
      # pylint: disable-next=consider-using-f-string
      print( "SSL profile: {}".format( profile ) )
      if toggleP4RuntimeAUPEnabled():
         print( "Authentication username priority:",
               f"{', '.join( self.transport.authnUsernamePriority )}" )
      else:
         print( "Certificate username authentication:",
               'yes' if self.transport.certUsernameAuthn else 'no' )
      print( "Total clients:", self.numClients )
      if toggleP4RuntimeAccountingRequestsEnabled():
         print( "Accounting requests:",
                'yes' if self.transport.accountingRequests else 'no' )
      leftCol = Format( justify='left' )
      leftCol.noPadLeftIs( True )
      rightCol = Format( justify='right' )
      rightCol.noTrailingSpaceIs( True )
      t = createTable( ( 'Device', 'Device ID',
                         'Primary Election ID (High, Low)' ) )
      t.formatColumns( leftCol, rightCol, rightCol )
      for deviceName, entry in self.devices:
         if entry.primaryElectionId:
            # pylint: disable-next=consider-using-f-string
            electionIdStr = "({}, {})".format( entry.primaryElectionId.high,
                                                  entry.primaryElectionId.low )
         else:
            electionIdStr = "None"
         t.newRow( deviceName, entry.deviceId, electionIdStr )
      print( t.output() )

class P4RtControllerPktMetadataEntry( Model ):
   messageType = Str( help="Type of Message containing the metadata" )
   name = Str( help="Name of the metadata" )
   metadataId = Int( help="Metadata ID" )
   bitWidth = Int( help="Bit width of the metadata" )

class P4RtDeviceStatusEntryDetail( Model ):
   deviceName = Str( help="Device name" )
   deviceId = Int( help="P4Runtime device ID" )
   primaryElectionId = Submodel( valueType=P4RuntimeElectionId,
                                 help="Primary election ID" )
   writeRpcReceived = Bool( help="Write RPC request received" )
   packetMetadata = List( valueType=P4RtControllerPktMetadataEntry,
                          help="Packet metatadatas received" )

   def render( self ):
      print( "Device", self.deviceName )
      print( "Device ID", self.deviceId )
      if not self.primaryElectionId:
         print( 'Primary Election ID: None' )
      else:
         self.primaryElectionId.render()
      print( "Write RPC request received:",
         'yes' if self.writeRpcReceived else 'no' )
      print( "Received controller metadata:" )
      leftCol = Format( justify='left' )
      leftCol.noPadLeftIs( True )
      rightCol = Format( justify='right' )
      rightCol.noTrailingSpaceIs( True )
      t = createTable( ( 'Message Type', 'Metadata Name', 'ID', 'Bit Width' ) )
      t.formatColumns( leftCol, leftCol, rightCol, rightCol )
      for entry in self.packetMetadata:
         t.newRow( entry.messageType, entry.name, entry.metadataId, entry.bitWidth )
      print( t.output() )

class P4RtDeviceStatus( Model ):
   devices = GeneratorDict( keyType=str, valueType=P4RtDeviceStatusEntryDetail,
                            optional=True,
                            help="P4Runtime device status entries keyed by name" )

   def render( self ):
      for _, entry in self.devices:
         entry.render()

class P4RtDeviceCounterEntry( Model ):
   packetIn = Int( help="Number of PacketIn responses sent" )
   packetOut = Int( help="Number of PacketOut requests received" )

class P4RtDeviceCounters( Model ):
   devicesCounter = Dict( keyType=str, valueType=P4RtDeviceCounterEntry,
                   help="A mapping of P4Runtime devices to its packet IO counters" )

   def render( self ):
      leftCol = Format( justify='left' )
      leftCol.noPadLeftIs( True )
      rightCol = Format( justify='right' )
      rightCol.noTrailingSpaceIs( True )
      t = createTable( ( 'Device', 'Packets In', 'Packets Out' ) )
      t.formatColumns( leftCol, rightCol, rightCol )
      for deviceName, counterEntry in self.devicesCounter.items():
         t.newRow( deviceName, counterEntry.packetIn, counterEntry.packetOut )
      print( t.output() )
