#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

# pylint: disable=ungrouped-imports
import AuthnUserPriorityCli as aupCli
import BasicCli
import CliCommand
import CliMatcher
import CliPlugin.P4RuntimeCli as P4RuntimeCli
from CliPlugin.P4RuntimeCli import p4RuntimeSystemConfig
import CliPlugin.P4RuntimeGlobalConfigMode
from CliMode.P4RuntimeMode import P4RuntimeTransportMode
from Intf.IntfRange import IntfRangeMatcher
from MultiRangeRule import MultiRangeMatcher
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.LagIntfCli as LagIntfCli
from CliPlugin.SwitchIntfCli import SwitchAutoIntfType
import CliPlugin.VrfCli as VrfCli
from TypeFuture import TacLazyType
from Toggles.P4RuntimeConfigToggleLib import (
   toggleP4RuntimeAccountingRequestsEnabled,
   toggleP4RuntimeAUPEnabled,
   toggleP4RuntimeDscpEnabled )

P4RuntimeConfigMode = CliPlugin.P4RuntimeGlobalConfigMode.P4RuntimeConfigMode
ChassisId = TacLazyType( 'P4Runtime::ChassisId' )

matcherConfiguration = CliMatcher.KeywordMatcher(
   'configuration', helpdesc='P4Runtime Forwarding Pipeline Configuration' )
matcherDevice = CliMatcher.KeywordMatcher(
   'device', helpdesc='Device configuration' )
matcherPortId = CliMatcher.KeywordMatcher(
   'id', helpdesc='P4 PortId assigned to interface' )
matcherDeviceId = CliMatcher.KeywordMatcher(
   'id', helpdesc='Device Identifier' )
matcherPipeline = CliMatcher.KeywordMatcher(
   'pipeline', helpdesc='Switch Forwarding Pipeline' )
matcherPort = CliMatcher.KeywordMatcher(
   'port', helpdesc='P4/SDN Ports related Configuration' )
matcherCert = CliMatcher.KeywordMatcher( 'certificate',
   helpdesc='Configure certificate options' )
matcherUsername = CliMatcher.KeywordMatcher( 'username',
   helpdesc='Configure certificate username options' )
matcherAuthentication = CliMatcher.KeywordMatcher( 'authentication',
   helpdesc='Configure certificate username authentication' )
authenticationMatcher = matcherAuthentication
if toggleP4RuntimeAUPEnabled():
   authenticationMatcher = CliCommand.Node( matcher=matcherAuthentication,
         deprecatedByCmd=f"{aupCli.cliCommandBase}" )

class P4RuntimeTransportConfigMode( P4RuntimeTransportMode,
                                    BasicCli.ConfigModeBase ):
   """CLI configuration submode 'transport grpc <name>'."""

   name = 'Transport for P4Runtime'

   def __init__( self, parent, session, name ):
      self.name = name
      self.config_ = p4RuntimeSystemConfig()
      P4RuntimeTransportMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoP4RuntimeTransportConfigMode( mode, args ):
   transportName = args.get( 'TRANSPORT_NAME' )
   if not p4RuntimeSystemConfig().transportName:
      p4RuntimeSystemConfig().transportName = transportName
      P4RuntimeCli.setDefaultAup()
   else:
      if p4RuntimeSystemConfig().transportName != transportName:
         mode.addError( "transport '%s' already enabled, "
                  "cannot enable another" % p4RuntimeSystemConfig().transportName )
         return

   childMode = mode.childMode( P4RuntimeTransportConfigMode, name=transportName )
   mode.session_.gotoChildMode( childMode )

def noP4RuntimeTransportConfigMode( mode, args ):
   transportName = args.get( 'TRANSPORT_NAME' )
   if p4RuntimeSystemConfig().transportName == transportName:
      P4RuntimeCli.clearP4RtTransport()

# ------------------------------------------------------
# [ no | default ] chassis id <chassis-id>
# ------------------------------------------------------
class ChassisIdCmd( CliCommand.CliCommandClass ):
   syntax = 'chassis id CHASSISID'
   noOrDefaultSyntax = 'chassis id ...'
   data = {
      'chassis': 'Chassis configuration',
      'id': 'Chassis Identifier',
      'CHASSISID': CliMatcher.IntegerMatcher( ChassisId.min, ChassisId.max,
                                              helpdesc='Chassis ID value' ),
   }
   handler = P4RuntimeCli.setChassisId
   noOrDefaultHandler = P4RuntimeCli.clearChassisId

P4RuntimeConfigMode.addCommandClass( ChassisIdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] forwarding chip <fap-name> device id <device-id>
#--------------------------------------------------------------------------------
class DeviceIdCmd( CliCommand.CliCommandClass ):
   syntax = 'forwarding chip FAPNAME device id DEVICEID'
   noOrDefaultSyntax = syntax
   data = {
      'forwarding': CliMatcher.KeywordMatcher( 'forwarding',
                                         helpdesc='Forwarding chip configuration' ),
      'chip': CliMatcher.KeywordMatcher( 'chip',
                                         helpdesc='Forwarding chip configuration' ),
      'FAPNAME': CliMatcher.PatternMatcher( pattern=r'[A-Za-z0-9\/]+',
                                           helpname='FAPNAME',
                                           helpdesc='Forwarding chip names' ),
      'device': matcherDevice,
      'id': matcherDeviceId,
      'DEVICEID': CliMatcher.IntegerMatcher( 1, 18446744073709551615,
                                             helpdesc='Device ID value' ),
   }
   handler = P4RuntimeCli.setDeviceId
   noOrDefaultHandler = P4RuntimeCli.clearDeviceId

P4RuntimeConfigMode.addCommandClass( DeviceIdCmd )

def getTransportName( mode ):
   names = []
   if p4RuntimeSystemConfig().transportName:
      names.append( p4RuntimeSystemConfig().transportName )
   return names

# ------------------------------------------------------
# [ no | default ] transport grpc <name>
# ------------------------------------------------------
class P4RuntimeTransportConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'transport grpc TRANSPORT_NAME'
   noOrDefaultSyntax = syntax
   data = {
         'transport': 'Configure a transport',
         'grpc': 'Configure gRPC transport for P4Runtime',
         'TRANSPORT_NAME': CliMatcher.DynamicNameMatcher( getTransportName,
                                                          helpdesc='Transport name',
                                                          helpname='WORD' ),
   }

   handler = gotoP4RuntimeTransportConfigMode
   noOrDefaultHandler = noP4RuntimeTransportConfigMode

P4RuntimeConfigMode.addCommandClass( P4RuntimeTransportConfigCmd )

#--------------------------------------------------------------------------------
# [ no | default ] port [ PORTNUM ]
#--------------------------------------------------------------------------------
class GrpcServerPortCmd( CliCommand.CliCommandClass ):
   syntax = 'port PORTNUM'
   noOrDefaultSyntax = 'port ...'
   data = {
      'port': 'The port number to listen on',
      'PORTNUM': CliMatcher.IntegerMatcher(
         1, 65535, helpdesc='gRPC Server port to listen to' ),
   }
   handler = P4RuntimeCli.setGrpcServerPort
   noOrDefaultHandler = P4RuntimeCli.setDefaultGrpcServerPort

P4RuntimeTransportConfigMode.addCommandClass( GrpcServerPortCmd )

#--------------------------------------------------------------------------------
# [ no | default ] qos dscp [ dscp ]
#--------------------------------------------------------------------------------
class GrpcTransportDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'qos dscp DSCP'
   noOrDefaultSyntax = 'qos dscp ...'
   data = {
     'qos': CliMatcher.KeywordMatcher( 'qos',
                                        'Set QoS parameters for gRPC traffic' ),
     'dscp': CliMatcher.KeywordMatcher( 'dscp', 'Set DSCP value for gRPC traffic' ),
     'DSCP': CliMatcher.IntegerMatcher( 0, 63, helpdesc='DSCP value' )
  }
   handler = P4RuntimeCli.setTransportDscp
   noOrDefaultHandler = P4RuntimeCli.setDefaultTransportDscp

if toggleP4RuntimeDscpEnabled():
   P4RuntimeTransportConfigMode.addCommandClass( GrpcTransportDscpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] accounting requests
#--------------------------------------------------------------------------------
class P4RuntimeTransportAccountingRequestsCmd( CliCommand.CliCommandClass ):
   syntax = 'accounting requests'
   noOrDefaultSyntax = syntax
   data = {
     'accounting': 'Configure accounting',
     'requests': 'Configure RPC requests accounting'
  }
   handler = P4RuntimeCli.setAccountingRequests
   noOrDefaultHandler = P4RuntimeCli.clearAccountingRequests

if toggleP4RuntimeAccountingRequestsEnabled():
   P4RuntimeTransportConfigMode.addCommandClass(
         P4RuntimeTransportAccountingRequestsCmd )

# --------------------------------------------------------------------------------
# [ no | default ] vrf <vrfName>
# --------------------------------------------------------------------------------
class GrpcServerTransportVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRFNAME'
   noOrDefaultSyntax = 'vrf ...'
   data = {
      'vrf': 'Configure VRF',
      'VRFNAME': CliMatcher.DynamicNameMatcher( VrfCli.getVrfNames, 'VRF name' ),
   }

   handler = P4RuntimeCli.setGrpcServerVrf
   noOrDefaultHandler = handler

P4RuntimeTransportConfigMode.addCommandClass( GrpcServerTransportVrfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] port id [ P4_PORT_ID_SET ] ...
#--------------------------------------------------------------------------------
class PortIdCmd( CliCommand.CliCommandClass ):
   syntax = 'port id P4_PORT_ID_SET interface { INTFLIST }'
   noOrDefaultSyntax = 'port id [ P4_PORT_ID_SET ... ]'
   data = {
      'port': matcherPort,
      'id': matcherPortId,
      'P4_PORT_ID_SET': MultiRangeMatcher(
         rangeFn=lambda: ( 1, ( 2 ** 32 - 1 ) ), noSingletons=False,
         helpdesc='P4 Port ID or range(s) of P4 Port IDs' ),
      'interface': 'Interfaces that are mapped to P4 Port Ids',
      'INTFLIST': IntfRangeMatcher(
         explicitIntfTypes=( EthIntfCli.EthPhyAutoIntfType,
                             LagIntfCli.LagAutoIntfType,
                             SwitchAutoIntfType ) )
   }
   handler = P4RuntimeCli.setPortMapping
   noOrDefaultHandler = P4RuntimeCli.clearPortMapping

P4RuntimeConfigMode.addCommandClass( PortIdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable the P4Runtime feature',
   }
   handler = P4RuntimeCli.clearEnabled
   noOrDefaultHandler = P4RuntimeCli.setEnabled

P4RuntimeConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ssl profile [ PROFILE_NAME ]
#--------------------------------------------------------------------------------
class SslProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile PROFILENAME'
   noOrDefaultSyntax = 'ssl profile ...'
   data = {
      'ssl': 'Configure SSL related options',
      'profile': 'Configure SSL profile',
      'PROFILENAME': CliMatcher.DynamicNameMatcher(
         P4RuntimeCli.getSslStatusProfile, 'Profile name' ),
   }
   handler = P4RuntimeCli.setSslProfile
   noOrDefaultHandler = P4RuntimeCli.clearSslProfile

P4RuntimeTransportConfigMode.addCommandClass( SslProfileCmd )

# --------------------------------------------------------------------------------
# [ no | default ] certificate username authentication
# --------------------------------------------------------------------------------
class CertUsernameAuthenticationCmd( CliCommand.CliCommandClass ):
   syntax = 'certificate username authentication'
   noOrDefaultSyntax = syntax
   data = {
      'certificate': matcherCert,
      'username': matcherUsername,
      'authentication': authenticationMatcher,
   }
   handler = P4RuntimeCli.setCertificateUsernameAuthn
   noOrDefaultHandler = P4RuntimeCli.clearCertificateUsernameAuthn

P4RuntimeTransportConfigMode.addCommandClass( CertUsernameAuthenticationCmd )

if toggleP4RuntimeAUPEnabled():
   P4RuntimeTransportConfigMode.addCommandClass(
         aupCli.authnUsernamePriorityNoOidCmd )
