# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import AgentCommandRequest
import BasicCli
from CliModel import Str, Model, List, cliPrinted, Int
from ArnetModel import IpGenericAddress
import CliToken.Platform
from CliPlugin.BridgingEtbaCli import tfaNode
from CliPlugin import TechSupportCli
import ShowCommand
from IntfModels import Interface

class ArfaPbrNexthopModel( Model ):
   vrf = Str( help="vrf in which the nexthop is used" )
   nexthop = IpGenericAddress( help="nexthop in use" )

class ArfaPbrNexthopsModel( Model ):
   nexthopsInUse = List( ArfaPbrNexthopModel,
                         help="list of nexthops in use" )

class ArfaPbrNexthopGroupRewrite( Model ):
   nexthopGroup = Str( help="Nexthop-group before rewrite is applied" )
   ingressIntfId = Interface( help="Interface which may apply this rewrite" )
   dscp = Int( help="DSCP of packet needed to apply this rewrite" )
   rewrittenNexthopGroup = Str( help="Nexthop-group after this rewrite is applied" )
   __public__ = False

class ArfaPbrNexthopGroupRewritesModel( Model ):
   nexthopGroupRewrites = List( ArfaPbrNexthopGroupRewrite,
                                help="List of PBR nexthop-group rewrites" )
   __public__ = False

def showArfaPbrConfig( mode, args ):
   AgentCommandRequest.runCliPrintSocketCommand( mode.entityManager,
                                                 "Etba", "debug",
                                                 "SHOW_PBR PolicyMapConfig",
                                                 mode )

def showArfaPbrStatus( mode, args ):
   AgentCommandRequest.runCliPrintSocketCommand( mode.entityManager,
                                                 "Etba", "debug",
                                                 "SHOW_PBR PolicyMapStatus",
                                                 mode )

def showArfaPbrNexthops( mode, args ):
   ret = AgentCommandRequest.runCliPrintSocketCommand( mode.entityManager,
                                                       "Etba", "debug",
                                                       "SHOW_PBR NexthopsInUse",
                                                       mode,
                                                       model=ArfaPbrNexthopsModel )
   return cliPrinted( ret ) if ret else None

def showArfaPbrNexthopGroupRewrites( mode, args, acrArg="NexthopGroupRewrites" ):
   ret = AgentCommandRequest.runCliPrintSocketCommand(
      mode.entityManager, "Etba", "debug", f"SHOW_PBR {acrArg}", mode,
      model=ArfaPbrNexthopGroupRewritesModel )
   return cliPrinted( ret ) if ret else None

def showArfaPbrNexthopGroupRewritesNoIntf( mode, args ):
   return showArfaPbrNexthopGroupRewrites(
      mode, args, acrArg="NexthopGroupRewritesNoIntf" )

platformMatcherForShow = CliToken.Platform.platformMatcherForShow

class ShowArfaShowPbrConfigCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa pbr config'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNode,
      'pbr': 'PBR for TFA',
      'config': 'PBR internal config for TFA',
   }

   handler = showArfaPbrConfig
   hidden = True

BasicCli.addShowCommandClass( ShowArfaShowPbrConfigCmd )

class ShowArfaShowPbrStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa pbr status'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNode,
      'pbr': 'PBR for TFA',
      'status': 'PBR internal status for TFA',
   }

   handler = showArfaPbrStatus
   hidden = True

BasicCli.addShowCommandClass( ShowArfaShowPbrStatusCmd )

class ShowArfaShowPbrNexthopCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa pbr nexthops'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNode,
      'pbr': 'PBR for TFA',
      'nexthops': 'PBR nexthops in use for TFA',
   }

   handler = showArfaPbrNexthops
   hidden = True
   cliModel = ArfaPbrNexthopsModel

BasicCli.addShowCommandClass( ShowArfaShowPbrNexthopCmd )

class ShowArfaShowPbrNexthopGroupRewritesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa pbr nexthop-group rewrites'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNode,
      'pbr': 'PBR for TFA',
      'nexthop-group': 'PBR nexthop-groups for TFA',
      'rewrites': 'PBR nexthop-groups rewrites for TFA',
   }

   handler = showArfaPbrNexthopGroupRewrites
   hidden = True
   cliModel = ArfaPbrNexthopGroupRewritesModel

BasicCli.addShowCommandClass( ShowArfaShowPbrNexthopGroupRewritesCmd )

class ShowArfaShowPbrNexthopGroupRewritesNoIntfCmd(
      ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa pbr nexthop-group rewrites no-interface'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNode,
      'pbr': 'PBR for TFA',
      'nexthop-group': 'PBR nexthop-groups for TFA',
      'rewrites': 'PBR nexthop-groups rewrites for TFA',
      'no-interface': 'PBR nexthop-groups rewrites with no interface for TFA',
   }

   handler = showArfaPbrNexthopGroupRewritesNoIntf
   hidden = True
   cliModel = ArfaPbrNexthopGroupRewritesModel

BasicCli.addShowCommandClass( ShowArfaShowPbrNexthopGroupRewritesNoIntfCmd )

# Add all of these commands to "show tech-support extended tfa"
TechSupportCli.registerShowTechSupportCmd(
      '2021-09-30 14:56:56', cmds=[
         "show platform tfa pbr config",
         "show platform tfa pbr status",
         "show platform tfa pbr nexthops",
         "show platform tfa pbr nexthop-group rewrites",
         #  "show platform tfa pbr nexthop-group rewrites no-interface", # BUG972325
         ],
      extended="tfa" )
