#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-from-import
# pylint: disable=consider-using-f-string

from SysConstants.in_h import IPPROTO_IP
import os

from Arnet import IpGenAddr
import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.AclCli as AclCli
import CliPlugin.AclCliRules as AclCliRules
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.PolicyMapModel as PolicyMapModel
from CliPlugin.PolicyMapModelImpl import ClassMapModelContainer
from CliPlugin.PolicyMapModelImpl import PolicyMapModelContainer
from CliPlugin.PolicyMapModelImpl import clearPolicyMapCounters
from CliPlugin.PolicyMapModelImpl import clearPolicyMapCheckpoint
import CliPlugin.PolicyMapCliLib as PolicyMapCliLib
import CliPlugin.VrfCli as VrfCli
import CliToken.Ip
import CliToken.Platform
import ConfigMount
import Intf.Log
import LazyMount
import SharkLazyMount
import PbrLib
import PolicyMap
import ShowCommand
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'PbrCli' )
t0 = __defaultTraceHandle__.trace0
t1 = __defaultTraceHandle__.trace1
t7 = __defaultTraceHandle__.trace7
t8 = __defaultTraceHandle__.trace8
t9 = __defaultTraceHandle__.trace9

UniqueId = Tac.Type( "Ark::UniqueId" )
UnresolvedNexthopAction = Tac.Type( "PolicyMap::PbrUnresolvedNexthopAction" )
PolicyMapCountersKey = Tac.Type( "PolicyMap::Counters::PolicyMapKey" )
NoAction = UnresolvedNexthopAction.na
DropDelayed = UnresolvedNexthopAction.dropDelayed
DropImmediate = UnresolvedNexthopAction.dropImmediate

pbrCheckpoint = None

pbrCliIntfConfig = None
pbrCliConfig = None
pbrStatusRequestDir = None
pbrStatus = None
aclStatus = None
rtgHwStatus = None
ethLagIntfStatusDir = None
allIntfStatusDir = None
policyOpChkr = None
aclStatus = None
statusDp = None
pbrMergedIntfConfig = None
pbrMergedConfig = None
pbrConfigMergeSm = None
pbrCliIntfConfigReadOnly = None
pbrCliConfigReadOnly = None
pbrIntfConfigDir = None
pbrConfigDir = None
entityManager = None
cliNexthopGroupConfig = None
vtiStatusDir = None
dynVlanIntfDir = None
counterStatus = None

def _pbrSessionCheckpoint( mode ):
   pbrSessionCheckpoint = mode.session.sessionData( 'PbrCli.sessionCheckpoint',
                                                     None )
   if pbrSessionCheckpoint is None:
      pbrSessionCheckpoint = Tac.newInstance( 'Pbr::CheckpointStatus',
                                              'pbr-counter-checkpoint' )
      mode.session.sessionDataIs( 'PbrCli.sessionCheckpoint', pbrSessionCheckpoint )
   return pbrSessionCheckpoint

#-------------------------------------------------------------------------------
# Tokens for Policy maps and class maps.
#-------------------------------------------------------------------------------
def guardPolicyMap( mode, token ):
   if rtgHwStatus.pbrSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardClassMap( mode, token ):
   if rtgHwStatus.pbrSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPMapPbr( mode, token ):
   if rtgHwStatus.pbrSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPbrNexthopVrf( mode, token ):
   if rtgHwStatus.pbrNexthopVrfSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardIp6AclsPbr( mode, token ):
   if rtgHwStatus.pbrSupported and rtgHwStatus.ipv6AclPbrSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardIp6NexthopPbr( mode, token ):
   if rtgHwStatus.pbrSupported and rtgHwStatus.ipv6NexthopPbrSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardNexthopGroupPbr( mode, token ):
   if rtgHwStatus.pbrSupported and rtgHwStatus.nexthopGroupPbrSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardMplsRawMatchPbr( mode, token ):
   if rtgHwStatus.pbrSupported and rtgHwStatus.mplsMatchRulePbrSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardBgpRedirectPbr( mode, token ):
   if rtgHwStatus.pbrSupported and rtgHwStatus.bgpRedirectPbrSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardNonAgingMacPbr( mode, token ):
   if rtgHwStatus.nonAgingMacPbrSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardServicePolicyPbr( mode, token ):
   if rtgHwStatus.pbrSupported:
      if mode.intf.hardware() == 'tunnel':
         if rtgHwStatus.pbrOnTunnelIntfsSupported:
            return None
         return CliParser.guardNotThisPlatform
      if ( rtgHwStatus.pbrOnSubintfsSupported or not mode.intf.isSubIntf() ):
         return None

   return CliParser.guardNotThisPlatform

def guardTtlAction( mode, token ):
   if rtgHwStatus.pbrTtlActionSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardHardwarePersistent( mode, token ):
   if rtgHwStatus.pbrHardwarePersistentSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardHardwareUpdateHitfulForced( mode, token ):
   if rtgHwStatus.pbrInPlaceTcamUpdateSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardCountersPollCommands( mode, token ):
   if rtgHwStatus.pbrSharkCounterSupported:
      return None
   return CliParser.guardNotThisPlatform

def getTtlRange( mode, context ):
   return ( rtgHwStatus.pbrTtlActionMinValue, rtgHwStatus.pbrTtlActionMaxValue )

nodePbr = CliCommand.guardedKeyword( 'pbr', helpdesc='Pbr type',
      guard=guardPMapPbr )
matcherNexthop = CliMatcher.KeywordMatcher( 'nexthop',
      helpdesc='Next hop IP address for forwarding' )
matcherNexthopGroup = CliCommand.guardedKeyword( 'nexthop-group',
                                           helpdesc='next hop group name',
                                           guard=guardNexthopGroupPbr )
def maxNexthopForEcmp():
   maxEcmp = rtgHwStatus.maxLogicalProtocolEcmp if rtgHwStatus else 0
   return maxEcmp
matcherRecursive = CliMatcher.KeywordMatcher( 'recursive',
                              helpdesc='Enable Recursive Next hop resolution' )

def getCounterStatusDir():
   counterInShark = rtgHwStatus.pbrSharkCounterSupported
   return counterStatus if counterInShark else None

class NexthopExprFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      class _NexthopExpr( CliCommand.CliExpression ):
         expression = '{ IPV4_ADDRS } | { IPV6_ADDRS }'
         data = {
            'IPV4_ADDRS' : IpAddrMatcher.IpAddrMatcher(
               helpdesc='next hop IP address' ),
            'IPV6_ADDRS' : CliCommand.Node(
               matcher=Ip6AddrMatcher.Ip6AddrMatcher(
                  helpdesc='next hop IPv6 address' ),
                guard=guardIp6NexthopPbr )
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            ipAddrList = args.get( 'IPV4_ADDRS', args.get( 'IPV6_ADDRS' ) )
            if not ipAddrList:
               return
            args.pop( 'IPV4_ADDRS', None )
            args.pop( 'IPV6_ADDRS', None )

            maxNexthop = maxNexthopForEcmp()
            if maxNexthop and len( ipAddrList ) > maxNexthop:
               mode.addErrorAndStop(
                     'Only %s nexthop addresses allowed' % maxNexthop )
            args[ name ] = ipAddrList

      return _NexthopExpr

matcherNoAction = CliMatcher.KeywordMatcher( 'no-action',
                                      helpdesc='stop policy-map lookup' )
nodeIpv6AccessGroupType = CliCommand.guardedKeyword( 'ipv6',
      helpdesc='Specify Ipv6 Access-groups',
      guard=guardIp6AclsPbr )
matcherDscp = CliMatcher.KeywordMatcher( 'dscp', helpdesc='Packet DSCP value' )
matcherSingleDscpValue = CliMatcher.IntegerMatcher(
      PbrLib.tacDscp.min, PbrLib.tacDscp.max,
        helpdesc='DSCP value between %d and %d' % ( PbrLib.tacDscp.min,
           PbrLib.tacDscp.max ) )
nodeTtl = CliCommand.guardedKeyword( 'ttl',
      helpdesc='TTL effective with nexthop/nexthop-group',
      guard=guardTtlAction )
matcherSingleTtlValue = CliMatcher.DynamicIntegerMatcher(
      getTtlRange, helpdesc='TTL value' )

def getNexthopGroupNames( mode ):
   return cliNexthopGroupConfig.nexthopGroup.members()

matcherNexthopGroupName = CliMatcher.DynamicNameMatcher(
   getNexthopGroupNames, 'nexthop group Name' )

class NoActionExpr( CliCommand.CliExpression ):
   expression = 'no-action'
   data = {
      'no-action' : CliCommand.Node( matcher=matcherNoAction, hidden=True ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'no-action' not in args:
         return
      args.pop( 'no-action' )
      args[ 'rawRuleActType' ] = ( PbrLib.actPermit,
            { 'type' : PbrLib.actPermit } )

class NexthopExpr( CliCommand.CliExpression ):
   expression = 'nexthop [ recursive ] NEXTHOP [ VRF ]'
   data = {
      'nexthop' : matcherNexthop,
      'recursive' : matcherRecursive,
      'NEXTHOP' : NexthopExprFactory(),
      'VRF' : VrfCli.VrfExprFactory(
         helpdesc='VRF for next hop action',
         guard=guardPbrNexthopVrf ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'nexthop' not in args:
         return
      args.pop( 'nexthop' )
      if 'recursive' in args:
         args.pop( 'recursive' )
         recursive = True
      else:
         recursive = False
      args[ 'rawRuleActType' ] = ( PbrLib.actSetNexthop,
            { 'type' : PbrLib.actSetNexthop,
              'nexthops' : args.pop( 'NEXTHOP' ),
              'recursive' : recursive,
              'vrf' : args.get( 'VRF' ) } )

class NexthopGroupExpr( CliCommand.CliExpression ):
   expression = 'NEXTHOP_GROUP_KW RAW_RULE_NEXTHOP_GROUP'
   data = {
      'NEXTHOP_GROUP_KW' : matcherNexthopGroup,
      'RAW_RULE_NEXTHOP_GROUP' : matcherNexthopGroupName
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'NEXTHOP_GROUP_KW' not in args:
         return
      args.pop( 'NEXTHOP_GROUP_KW' )
      args[ 'rawRuleActType' ] = ( PbrLib.actSetNexthopGroup,
            { 'type' : PbrLib.actSetNexthopGroup,
              'nexthopGroup' : args.pop( 'RAW_RULE_NEXTHOP_GROUP' ) } )

class TtlExpr( CliCommand.CliExpression ):
   expression = 'ttl TTL'
   data = {
      'ttl' : nodeTtl,
      'TTL' : matcherSingleTtlValue
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ttl' not in args:
         return
      args.pop( 'ttl' )
      args[ 'rawRuleActType' ] = ( PbrLib.actSetTtl, args.pop( 'TTL' ) )

class DscpExpr( CliCommand.CliExpression ):
   expression = 'DSCP_KW RAW_RULE_DSCP'
   data = {
      'DSCP_KW' : CliCommand.Node( matcher=matcherDscp, hidden=True ),
      'RAW_RULE_DSCP' : matcherSingleDscpValue
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'DSCP_KW' not in args:
         return
      args.pop( 'DSCP_KW' )
      args[ 'rawRuleActType' ] = ( PbrLib.actSetDscp, args.pop( 'RAW_RULE_DSCP' ) )

setActions = {
   'SET_NO_ACTION' : NoActionExpr,
   'SET_NEXTHOP' : NexthopExpr,
   'SET_NEXTHOP_GROUP' : NexthopGroupExpr,
   'SET_TTL' : TtlExpr
}

if 'PBR_FUTURE' in os.environ:
   # These action types are not yet supported.
   setActions[ 'SET_DSCP' ] = DscpExpr

class SetActionsExpr( CliCommand.CliExpression ):
   expression = ' | '.join( setActions.keys() )
   data = setActions

class ActionSetWithNhopActTypeExpr( CliCommand.CliExpression ):
   expression = 'drop | ( set SET_ACTIONS )'
   data = {
      'drop' : PolicyMapCliLib.matcherDrop,
      'set' : PolicyMapCliLib.matcherActionSet,
      'SET_ACTIONS' : SetActionsExpr,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args.pop( 'set', None )
      if 'drop' not in args:
         return

      args.pop( 'drop' )
      args[ 'rawRuleActType' ] = ( PbrLib.actDeny,
            { 'type' : PbrLib.actDeny } )

# Functions for dynamic name rules
def getStandardAccessListNames( ipVer ):
   names = []
   ipConfig = AclCli.config.config[ ipVer ].acl
   for name in ipConfig:
      if not ipConfig[ name ].standard:
         names.append( name )
   return sorted( names )

# [ no ] match ip access-group <access-list>
matcherIpAccessGroup = CliMatcher.DynamicNameMatcher(
       lambda mode: getStandardAccessListNames( 'ip' ),
        'access-group' )

matcherIpv6AccessGroup = CliMatcher.DynamicNameMatcher(
       lambda mode: getStandardAccessListNames( 'ipv6' ),
    'access-group' )

#
# Modelet under Intf mode for service-policy commands
#
class IntfPbrModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      # In the dut reload case, or any scenario that acts as a dut reload (note this
      # is not the same as a config replace), the rtgHwStatus will not be populated
      # before reloading of startup-config happens. Thus we need to bypass this by
      # ignoring the check when startupConfig() is true. Invalid configs cannot get
      # the running-config to be saved, and should the user introduce an invalid
      # config into the startup-config, the platform agents will handle them
      # gracefully.
      return mode.intf.name.startswith(
            ( 'Ethernet', 'Switch', 'Port-Channel', 'Vlan', 'Tunnel' ) )

IntfCli.IntfConfigMode.addModelet( IntfPbrModelet )

#
# Acl change status callback that returns the impact of an Acl change on
# routing policy.
#
def isPolicyHwStatusOk( mode, aclName, aclType, aclUpdateType ):
   rc = True
   err = {}
   err[ 'error' ] = ''

   # Just return success if there are no statusDp as we accept the configuration if
   # there are no linecards inserted
   if not statusDp or not rtgHwStatus.pbrSupported:
      return rc, err

   if not mode.session_.startupConfig() and AclCli.allHwStatusEnabled():
      rc, result = policyOpChkr.verify( 'acl', aclName )
      if not rc:
         err[ 'error' ] = result.error if result else 'unknown PBR policy status'
      else:
         if result and result.warning:
            err[ 'warning' ] = "Warning: ACL %s %s" % ( aclName, result.warning )
   return rc, err

def getPmaps():
   return pbrCliConfig.pmapType.pmap

class PolicyMapClassModePbr( PolicyMapCliLib.PolicyMapClassMode ):
   name = 'Policy Map Pbr Class Configuration'

   def __init__( self, parent, session, context ):
      PolicyMapCliLib.PolicyMapClassMode.__init__( self, parent, session,
                                                   context, 'pbr' )

   def _delAction( self, actionType ):
      t8( "delAction", actionType )
      policyRuleAction = self.pmapClassContext.policyRuleAction
      pbrAct = policyRuleAction.delAction( actionType )

      return pbrAct

   def _addAction( self, actionType, action ):
      t8( "addAction", actionType )
      policyRuleAction = self.pmapClassContext.policyRuleAction
      policyRuleAction.addAction( actionType, action )

   def delFwdAction( self ):
      self._delAction( PbrLib.actPermit )
      self._delAction( PbrLib.actDeny )
      self._delAction( PbrLib.actSetNexthop )
      self._delAction( PbrLib.actSetNexthopGroup )

   def setNoAction( self, no ):
      self.delFwdAction()
      if not no:
         pbrAct = pbrCliConfig.pbrActions.noAction.newMember(
            self.cmapName, UniqueId() )
         self._addAction( PbrLib.actPermit, pbrAct )

   def setActionDrop( self, no ):
      self.delFwdAction()
      if not no:
         pbrAct = pbrCliConfig.pbrActions.dropAction.newMember(
            self.cmapName, UniqueId() )
         self._addAction( PbrLib.actDeny, pbrAct )

   def setNexthop( self, ipAddrList, recursive, vrfName ):
      self.delFwdAction()
      pbrAct = pbrCliConfig.pbrActions.nexthopAction.newMember(
         self.cmapName, UniqueId() )
      for ipAddr in ipAddrList:
         pbrAct.nexthop[ IpGenAddr( str( ipAddr ) ) ] = True
      if recursive:
         pbrAct.recursive = True
      if vrfName:
         pbrAct.vrfName = vrfName
      self._addAction( PbrLib.actSetNexthop, pbrAct )

   def setNexthopGroupName( self, nexthopGroupName ):
      self.delFwdAction()
      pbrAct = pbrCliConfig.pbrActions.nexthopGroupAction.newMember(
         self.cmapName, UniqueId() )
      pbrAct.nexthopGroup = nexthopGroupName
      self._addAction( PbrLib.actSetNexthopGroup, pbrAct )

   def setTtl( self, ttlVal ):
      self.delTtl()
      pbrAct = pbrCliConfig.pbrActions.ttlAction.newMember( self.cmapName,
                                                            UniqueId(), ttlVal )
      self._addAction( PbrLib.actSetTtl, pbrAct )

   def delTtl( self ):
      self._delAction( PbrLib.actSetTtl )

   def setDscp( self, dscpVal ):
      self.delDscp()
      pbrAct = pbrCliConfig.pbrActions.dscpAction.newMember( self.cmapName,
                               UniqueId(), dscpVal )
      self._addAction( PbrLib.actSetDscp, pbrAct )

   def delDscp( self ):
      self._delAction( PbrLib.actSetDscp )

class PolicyMapModePbr( PolicyMapCliLib.PolicyMapMode ):
   name = 'Policy Map Pbr Configuration'

   def __init__( self, parent, session, context ):
      PolicyMapCliLib.PolicyMapMode.__init__( self, parent, session, context,
                        'pbr' )

   def switchToPmapClassMode( self, context ):
      childMode = self.childMode( PolicyMapClassModePbr, context=context )
      self.session_.gotoChildMode( childMode )

   def defaultActionTypeAndData( self ):
      actType = PbrLib.actPermit
      actData = { 'type' : PbrLib.actPermit }
      return actType, actData

   def actionDataToClassAction( self, cmapName, actType, actData ):
      if actType == PbrLib.actSetDscp:
         return pbrCliConfig.pbrActions.dscpAction.newMember( cmapName,
                               UniqueId(), actData )
      if actType == PbrLib.actSetTtl:
         return pbrCliConfig.pbrActions.ttlAction.newMember( cmapName,
                               UniqueId(), actData )
      if actType not in [ PbrLib.actPermit, PbrLib.actDeny, PbrLib.actSetNexthop,
                          PbrLib.actSetNexthopGroup ]:
         return None

      if actData[ 'type' ] == PbrLib.actPermit:
         pbrAct = pbrCliConfig.pbrActions.noAction.newMember( cmapName, UniqueId() )
      elif actData[ 'type' ] == PbrLib.actDeny:
         pbrAct = pbrCliConfig.pbrActions.dropAction.newMember( cmapName,
                                                                UniqueId() )
      elif actData[ 'type' ] == PbrLib.actSetNexthopGroup:
         pbrAct = pbrCliConfig.pbrActions.nexthopGroupAction.newMember( cmapName,
                                                                        UniqueId() )
         pbrAct.nexthopGroup = actData[ 'nexthopGroup' ]
      elif actData[ 'type' ] == PbrLib.actSetNexthop:
         pbrAct = pbrCliConfig.pbrActions.nexthopAction.newMember( cmapName,
                                                                   UniqueId() )
         for ipAddr in actData['nexthops']:
            pbrAct.nexthop[ IpGenAddr( str( ipAddr ) ) ] = True
         pbrAct.recursive = actData['recursive']
         if actData['vrf']:
            pbrAct.vrfName = actData['vrf']
      return pbrAct

class ClassMapModePbr( PolicyMapCliLib.ClassMapMode ):
   name = "Class Map Pbr Configuration"

   def __init__( self, parent, session, context ):
      PolicyMapCliLib.ClassMapMode.__init__( self, parent, session, context, 'pbr' )

matcherPmapNamePbr = CliMatcher.DynamicNameMatcher(
   lambda mode: pbrCliConfig.pmapType.pmap, 'Pbr Policy Map Name' )

matcherCmapNamePbr = CliMatcher.DynamicNameMatcher(
   lambda mode: pbrCliConfig.cmapType.cmap, 'Class Map Name' )

#--------------------------------------------------------------------------------
# [ no | default ] class-map type pbr match-any CMAP_NAME
#--------------------------------------------------------------------------------
class ClassMapModeCmd( CliCommand.CliCommandClass ):
   syntax = 'class-map type pbr match-any CMAP_NAME'
   noOrDefaultSyntax = 'class-map type pbr [ match-any ] CMAP_NAME'
   data = {
      'class-map' : CliCommand.guardedKeyword( 'class-map',
         helpdesc='Configure Class Map', guard=guardClassMap ),
      'type' : PolicyMapCliLib.matcherType,
      'pbr' : nodePbr,
      'match-any' : 'Logical-OR all match statements under this class-map',
      'CMAP_NAME' : matcherCmapNamePbr,
   }

   @staticmethod
   def handler( mode, args ):
      cmapName = args[ 'CMAP_NAME' ]
      mapType = 'pbr'
      matchType = 'match-any'
      context = None
      t0( 'gotoClassMapMode %s' % ( cmapName ) )
      emapType = PolicyMap.mapTypeToEnum( mapType )
      if context is None:
         context = PolicyMapCliLib.ClassMapContext( pbrCliConfig,
               pbrStatusRequestDir, pbrStatus, emapType, cmapName, matchType )
      if cmapName in pbrCliConfig.cmapType.cmap:
         entry = pbrCliConfig.cmapType.cmap[ cmapName ]
         context.copyEditCmap( entry )
      else:
         context.newEditCmap()

      childMode = mode.childMode( ClassMapModePbr, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cmapName = args[ 'CMAP_NAME' ]
      lastMode = mode.session_.modeOfLastPrompt()
      if isinstance( lastMode, PolicyMapCliLib.ClassMapMode ) and \
             lastMode.cmapContext and \
             lastMode.cmapContext.cmapName() == cmapName:
         # deleting self inside the config mode, so discard the config
         # since we'll succeed
         lastMode.cmapContext = None

      if not cmapName in pbrCliConfig.cmapType.cmap:
         return

      PolicyMapCliLib.deleteClassMap( mode, cmapName, pbrCliConfig,
                                       pbrStatusRequestDir, pbrStatus )

BasicCli.GlobalConfigMode.addCommandClass( ClassMapModeCmd )

#-------------------------------------------------------------------------------
# class-map mode abort
#-------------------------------------------------------------------------------
class ClassMapAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort' : PolicyMapCliLib.matcherAbort
   }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

ClassMapModePbr.addCommandClass( ClassMapAbortCmd )

#--------------------------------------------------------------------------------
# [ no | default ] [ SEQ_NUM ] match ip access-group ACCESS_GROUP
#--------------------------------------------------------------------------------
class MatchValueIpv4Cmd( CliCommand.CliCommandClass ):
   syntax = '[ SEQ_NUM ] match ip access-group ACCESS_GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'SEQ_NUM' : PolicyMapCliLib.matcherSeqNum,
      'match' : PolicyMapCliLib.matcherMatch,
      'ip' : PolicyMapCliLib.matcherIpAccessGroupType,
      'access-group' : PolicyMapCliLib.matcherAccessGroup,
      'ACCESS_GROUP' : matcherIpAccessGroup
   }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( no, args.get( 'SEQ_NUM' ), 'ip', args[ 'ACCESS_GROUP' ] )

   noOrDefaultHandler = handler

ClassMapModePbr.addCommandClass( MatchValueIpv4Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] [ SEQ_NUM ] match ipv6 access-group ACCESS_GROUP
#--------------------------------------------------------------------------------
class MatchValueIpv6Cmd( CliCommand.CliCommandClass ):
   syntax = '[ SEQ_NUM ] match ipv6 access-group ACCESS_GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'SEQ_NUM' : PolicyMapCliLib.matcherSeqNum,
      'match' : PolicyMapCliLib.matcherMatch,
      'ipv6' : nodeIpv6AccessGroupType,
      'access-group' : PolicyMapCliLib.matcherAccessGroup,
      'ACCESS_GROUP' : matcherIpv6AccessGroup
   }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( no, args.get( 'SEQ_NUM' ), 'ipv6', args[ 'ACCESS_GROUP' ] )

   noOrDefaultHandler = handler

ClassMapModePbr.addCommandClass( MatchValueIpv6Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] SEQUENCES
#--------------------------------------------------------------------------------
class PbrNoSequenceCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'SEQUENCES'
   data = {
      'SEQUENCES': AclCliRules.seqRangeMatcher
   }
   noOrDefaultHandler = ClassMapModePbr.handleNoSeq

ClassMapModePbr.addCommandClass( PbrNoSequenceCmd )

#-----------------------------------------------------------------
# resequence [ start [ incr ] ]
#-----------------------------------------------------------------
class PbrResequenceCmd( CliCommand.CliCommandClass ):
   syntax = 'resequence [ START [ INC ] ]'
   data = {
      'resequence' : PolicyMapCliLib.matcherResequence,
      'START' : PolicyMapCliLib.matcherStartSequence,
      'INC' : PolicyMapCliLib.matcherIncSequence,
   }

   @staticmethod
   def handler( mode, args ):
      mode.resequence( args.get( 'START', PolicyMapCliLib.defaultSeqStart ),
            inc=args.get( 'INC', PolicyMapCliLib.defaultSeqInc ) )

ClassMapModePbr.addCommandClass( PbrResequenceCmd )

class PbrPolicyMapContext( PolicyMapCliLib.PolicyMapContext ):
   def __init__( self, config, statusReqDir, status, pmapName, pmapType,
                 rollbackEnabled=True ):
      PolicyMapCliLib.PolicyMapContext.__init__( self, config, statusReqDir, \
                        status, pmapName, pmapType, rollbackEnabled=rollbackEnabled )

   def defaultSequenceInc( self ):
      return PbrLib.pbrDefaultSeqInc

   def maxRules( self ):
      return PbrLib.pbrMaxRules

   def removeAction( self, actionObj ): # pylint: disable=arguments-renamed
      removeActionFromPbrActions( pbrCliConfig, actionObj )

   def copyAction( self, src ):
      actionType = src.actionType
      pbrActions = self.config().pbrActions
      if actionType == PbrLib.actSetDscp:
         return pbrActions.dscpAction.newMember( src.className, UniqueId(), \
                  src.dscp )
      if actionType == PbrLib.actSetTtl:
         return pbrActions.ttlAction.newMember( src.className, UniqueId(), \
                                                src.ttl )



      assert actionType in [ PbrLib.actPermit, PbrLib.actDeny, PbrLib.actSetNexthop,
                             PbrLib.actSetNexthopGroup ]
      if actionType == PbrLib.actPermit:
         pbrAct = pbrActions.noAction.newMember( src.className, UniqueId() )
      elif actionType == PbrLib.actDeny:
         pbrAct = pbrActions.dropAction.newMember( src.className, UniqueId() )
      elif actionType == PbrLib.actSetNexthop:
         pbrAct = pbrActions.nexthopAction.newMember( src.className, UniqueId() )
         for n in src.nexthop:
            pbrAct.nexthop[ n ] = True
         pbrAct.recursive = src.recursive
         pbrAct.vrfName = src.vrfName
      elif actionType == PbrLib.actSetNexthopGroup:
         pbrAct = pbrActions.nexthopGroupAction.newMember( src.className,
                                                           UniqueId() )
         pbrAct.nexthopGroup = src.nexthopGroup

      return pbrAct

   def identicalClassActions( self, c1Action, c2Action ):
      if set( c1Action.policyAction ) != set( c2Action.policyAction ):
         return PolicyMapCliLib.CHANGED
      for actType in c1Action.policyAction:
         if actType == PbrLib.actSetDscp:
            dscp1 = c1Action.policyAction[ actType ].dscp
            dscp2 = c2Action.policyAction[ actType ].dscp
            if dscp1 != dscp2:
               return PolicyMapCliLib.CHANGED
         elif actType == PbrLib.actSetTtl:
            ttl1 = c1Action.policyAction[ actType ].ttl
            ttl2 = c2Action.policyAction[ actType ].ttl
            if ttl1 != ttl2:
               return PolicyMapCliLib.CHANGED
         elif actType in [ PbrLib.actPermit, PbrLib.actDeny ] :
            if not c2Action.policyAction[ actType ]:
               return PolicyMapCliLib.CHANGED
         elif actType == PbrLib.actSetNexthopGroup:
            fwdAct1 = c1Action.policyAction[ actType ]
            fwdAct2 = c2Action.policyAction[ actType ]
            if fwdAct1.nexthopGroup != fwdAct2.nexthopGroup:
               return PolicyMapCliLib.CHANGED
         elif actType == PbrLib.actSetNexthop:
            fwdAct1 = c1Action.policyAction[ actType ]
            fwdAct2 = c2Action.policyAction[ actType ]
            if fwdAct1.recursive != fwdAct2.recursive:
               return PolicyMapCliLib.CHANGED
            if set( fwdAct1.nexthop ) != set( fwdAct2.nexthop ):
               return PolicyMapCliLib.CHANGED
            if fwdAct1.vrfName != fwdAct2.vrfName:
               return PolicyMapCliLib.CHANGED

      return PolicyMapCliLib.IDENTICAL

def removeActionFromPbrActions( config, action ):
   actType = action.actionType
   pbrActions = config.pbrActions
   if actType == PbrLib.actPermit:
      del pbrActions.noAction[ action.id ]
   elif actType == PbrLib.actDeny:
      del pbrActions.dropAction[ action.id ]
   elif actType == PbrLib.actSetNexthop:
      del pbrActions.nexthopAction[ action.id ]
   elif actType == PbrLib.actSetNexthopGroup:
      del pbrActions.nexthopGroupAction[ action.id ]
   elif actType == PbrLib.actSetDscp:
      del pbrActions.dscpAction[ action.id ]
   elif actType == PbrLib.actSetTtl:
      del pbrActions.ttlAction[ action.id ]
   else:
      assert False, 'unknown action ' + action

#--------------------------------------------------------------------------------
# [ no | default ] policy-map type pbr PMAP_NAME
#--------------------------------------------------------------------------------
def clearPolicyMapCheckpointStatus( mode, pmapName ):
   clearPolicyMapCheckpoint( pmapName, _pbrSessionCheckpoint( mode ) )
   clearPolicyMapCheckpoint( pmapName, pbrCheckpoint )

class PolicyMapModeCmd( CliCommand.CliCommandClass ):
   syntax = 'policy-map type pbr PMAP_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'policy-map' : CliCommand.guardedKeyword( 'policy-map',
         helpdesc='Configure Policy Map', guard=guardPolicyMap ),
      'type' : PolicyMapCliLib.matcherType,
      'pbr' : nodePbr,
      'PMAP_NAME' : matcherPmapNamePbr,
   }

   @staticmethod
   def handler( mode, args ):
      pmapName = args[ 'PMAP_NAME' ]
      mapType = 'pbr'
      t0( 'gotoPolicyMapMode of policy', pmapName )
      emapType = PolicyMap.mapTypeToEnum( mapType )
      inConfigSession = mode.session_.inConfigSession()
      context = PbrPolicyMapContext( pbrCliConfig,
                                     pbrStatusRequestDir,
                                     pbrStatus,
                                     pmapName,
                                     emapType,
                                     rollbackEnabled=not inConfigSession )

      if pmapName in pbrCliConfig.pmapType.pmap:
         context.copyEditPmap()
      else:
         context.newEditPmap()

      childMode = mode.childMode( PolicyMapModePbr, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pmapName = args[ 'PMAP_NAME' ]
      mapType = 'pbr'
      lastMode = mode.session_.modeOfLastPrompt()
      if isinstance( lastMode, PolicyMapModePbr ):
         if pmapName == lastMode.pmapName:
            lastMode.pmapContext = None
      if pmapName not in pbrCliConfig.pmapType.pmap:
         return

      clearPolicyMapCheckpointStatus( mode, pmapName )

      emapType = PolicyMap.mapTypeToEnum( mapType )
      dummyContext = PbrPolicyMapContext( pbrCliConfig, pbrStatusRequestDir,
                                          pbrStatus, pmapName, emapType )
      dummyContext.delPmap( mode, pmapName )

BasicCli.GlobalConfigMode.addCommandClass( PolicyMapModeCmd )

#-------------------------------------------------------------------------------
# abort
#-------------------------------------------------------------------------------
PolicyMapModePbr.addCommandClass( ClassMapAbortCmd )
PolicyMapClassModePbr.addCommandClass( ClassMapAbortCmd )

#--------------------------------------------------------------------------------
# [ no | default ] [ SEQ_NUM ] class CMAP_NAME
#--------------------------------------------------------------------------------
class ClassCmapnameCmd( CliCommand.CliCommandClass ):
   syntax = '[ SEQ_NUM ] class CMAP_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'SEQ_NUM' : PolicyMapCliLib.matcherSeqNum,
      'class' :  PolicyMapCliLib.matcherClass,
      'CMAP_NAME' : matcherCmapNamePbr
   }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setClass( no, args.get( 'SEQ_NUM' ), args[ 'CMAP_NAME' ] )

   noOrDefaultHandler = handler

PolicyMapModePbr.addCommandClass( ClassCmapnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] set nexthop [ recursive ] NEXTHOP [ VRF ]
#--------------------------------------------------------------------------------
class SetNexthopCmd( CliCommand.CliCommandClass ):
   syntax = ( 'set nexthop [ recursive ] NEXTHOP [ VRF ]' )
   noOrDefaultSyntax = 'set nexthop [ recursive ] ...'
   data = {
      'set' : PolicyMapCliLib.matcherActionSet,
      'nexthop' : matcherNexthop,
      'recursive' : matcherRecursive,
      'NEXTHOP' : NexthopExprFactory(),
      'VRF' : VrfCli.VrfExprFactory(
         helpdesc='Configure counter in a VRF',
         guard=guardPbrNexthopVrf ),
   }

   @staticmethod
   def handler( mode, args ):
      recursive = 'recursive' in args
      vrfName = args.get( 'VRF' )
      mode.setNexthop( args[ 'NEXTHOP' ], recursive, vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.delFwdAction()

PolicyMapClassModePbr.addCommandClass( SetNexthopCmd )

#-------------------------------------------------------------------------------
# [ no | default ] set nexthop-group GROUP
#-------------------------------------------------------------------------------
class SetNexthopGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'set nexthop-group GROUP'
   noOrDefaultSyntax = 'set nexthop-group ...'
   data = {
      'set' : PolicyMapCliLib.matcherActionSet,
      'nexthop-group' : matcherNexthopGroup,
      'GROUP' : matcherNexthopGroupName
   }

   @staticmethod
   def handler( mode, args ):
      mode.setNexthopGroupName( args[ 'GROUP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.delFwdAction()

PolicyMapClassModePbr.addCommandClass( SetNexthopGroupCmd )

#-------------------------------------------------------------------------------
# [ no | default ] set no-action
#-------------------------------------------------------------------------------
class NoActionCmd( CliCommand.CliCommandClass ):
   syntax = 'set no-action'
   noOrDefaultSyntax = syntax
   data = {
      'set' : PolicyMapCliLib.matcherActionSet,
      'no-action' : matcherNoAction,
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      mode.setNoAction( CliCommand.isNoOrDefaultCmd( args ) )

   noOrDefaultHandler = handler

PolicyMapClassModePbr.addCommandClass( NoActionCmd )

#-------------------------------------------------------------------------------
# [ no | default ] drop
#-------------------------------------------------------------------------------
class DropCmd( CliCommand.CliCommandClass ):
   syntax = 'drop'
   noOrDefaultSyntax = syntax
   data = {
      'drop' : PolicyMapCliLib.matcherDrop,
   }

   @staticmethod
   def handler( mode, args ):
      mode.setActionDrop( CliCommand.isNoOrDefaultCmd( args ) )

   noOrDefaultHandler = handler

PolicyMapClassModePbr.addCommandClass( DropCmd )

#-------------------------------------------------------------------------------
# [ no | default ] set dscp DSCP
#-------------------------------------------------------------------------------
class SetDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'set dscp DSCP'
   noOrDefaultSyntax = 'set dscp ...'
   data = {
      'set' : PolicyMapCliLib.matcherActionSet,
      'dscp' : matcherDscp,
      'DSCP' : matcherSingleDscpValue
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      mode.setDscp( args[ 'DSCP' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.delDscp()

PolicyMapClassModePbr.addCommandClass( SetDscpCmd )

#-------------------------------------------------------------------------------
# [ no | default ] set ttl TTL
#-------------------------------------------------------------------------------
class SetTtlCmd( CliCommand.CliCommandClass ):
   syntax = 'set ttl TTL'
   noOrDefaultSyntax = 'set ttl ...'
   data = {
      'set' : PolicyMapCliLib.matcherActionSet,
      'ttl' : nodeTtl,
      'TTL' : matcherSingleTtlValue
   }

   @staticmethod
   def handler( mode, args ):
      mode.setTtl( args[ 'TTL' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.delTtl()

PolicyMapClassModePbr.addCommandClass( SetTtlCmd )

#-------------------------------------------------------------------------------
# Raw match statements in policy map mode
#-------------------------------------------------------------------------------

class PbrCfgCmdBinder( PolicyMapCliLib.PolicyCfgCmdBinder ):

   def __init__( self ):
      PolicyMapCliLib.PolicyCfgCmdBinder.__init__( self )
      cls = self.__class__

      ########################################################
      #                    IP filters
      ########################################################
      class IpAclCommand( CliCommand.CliCommandClass ):
         syntax = ( '[ SEQ_NUM ] ACTION [ VLAN_EXPR ] ip [ ETHERTYPE ] '
                    'IP_FILTER [ log ] [ RAW_RULE ]' )
         noOrDefaultSyntax = ( 'ACTION [ VLAN_EXPR ] ip [ ETHERTYPE ] '
                               'IP_FILTER [ log ] [ RAW_RULE ]' )
         data = {
            'SEQ_NUM': AclCli.seqnumMatcher,
            'ACTION': self.actionMatcher,
            'VLAN_EXPR': self.vlanExpression,
            'ip' : PolicyMapCliLib.matcherIpAccessGroupType,
            'ETHERTYPE':  AclCliRules.EtherTypeMatcher( 'ip' ),
            'IP_FILTER': self.ipFilterExpression,
            'log': AclCliRules.logMatcher,
            'RAW_RULE' : ActionSetWithNhopActTypeExpr,
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            cls.adaptRuleCommonArgs( mode, args )
            args[ 'mirrorSession' ] = None
            args[ 'rawRuleActType' ] = args.get( 'rawRuleActType' )
            args[ 'protocol' ] = args.get( 'protocol', IPPROTO_IP )
            if args[ 'protocol' ] != IPPROTO_IP:
               args[ 'protoMask' ] = 0xff
            args.pop( 'ip' )
            cls._handleIpArgs( mode, args ) # pylint: disable-msg=protected-access

         @staticmethod
         def handler( mode, args ):
            if CliCommand.isNoOrDefaultCmd( args ):
               cls.adaptNoOrDefaultRuleCommonArgs( mode, args )

            cls.handleIpRule( mode, **args )

         noOrDefaultHandler = handler

      PolicyMapModePbr.addCommandClass( IpAclCommand )

      ########################################################
      #                       mpls filters
      ########################################################
      class MplsFilterCmd( CliCommand.CliCommandClass ):
         syntax = '[ SEQ_NUM ] ACTION mpls any [ RAW_RULE ]'
         noOrDefaultSyntax = 'ACTION mpls any [ RAW_RULE ]'
         data = {
            'SEQ_NUM' : AclCli.seqnumMatcher,
            'ACTION': self.actionMatcher,
            'mpls' : CliCommand.guardedKeyword( 'mpls',
               helpdesc= 'Specify mpls packets to match',
               guard=guardMplsRawMatchPbr ),
            'any' : CliCommand.guardedKeyword( 'any',
               helpdesc= 'Specify all mpls packets to match',
               guard=guardMplsRawMatchPbr ),
            'RAW_RULE' : ActionSetWithNhopActTypeExpr,
         }

         @staticmethod
         def handler( mode, args ):
            no = CliCommand.isNoOrDefaultCmd( args )
            seqNum = args.pop( 'SEQ_NUM', 0 )
            mode.handleRawMatchStmt( no, seqNum, 'permit', None,
                  PolicyMapCliLib.PolicyMplsRuleAction.keyTag(), None, None,
                  rawRuleActType=args.get( 'rawRuleActType' ) )

         noOrDefaultHandler = handler

      PolicyMapModePbr.addCommandClass( MplsFilterCmd )

      ########################################################
      #                    IPv6 filters
      ########################################################
      class Ip6AclCommand( CliCommand.CliCommandClass ):
         syntax = ( '[ SEQ_NUM ] ACTION [ VLAN_EXPR ] ipv6 [ ETHERTYPE ] '
                    'IP_FILTER [ log ] [ RAW_RULE ]' )
         noOrDefaultSyntax = ( 'ACTION [ VLAN_EXPR ] ipv6 IP_FILTER [ ETHERTYPE ] '
                               '[ log ] [ RAW_RULE ]' )
         data = {
            'SEQ_NUM': AclCli.seqnumMatcher,
            'ACTION': self.actionMatcher,
            'VLAN_EXPR': self.vlanExpression,
            'ipv6' : nodeIpv6AccessGroupType,
            'ETHERTYPE':  AclCliRules.EtherTypeMatcher( 'ipv6' ),
            'IP_FILTER': self.ip6FilterExpression,
            'log': AclCliRules.logMatcher,
            'RAW_RULE' : ActionSetWithNhopActTypeExpr,
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            cls.adaptRuleCommonArgs( mode, args )
            args[ 'mirrorSession' ] = None
            args[ 'rawRuleActType' ] = args.get( 'rawRuleActType' )
            args[ 'protocol' ] = args.get( 'protocol', IPPROTO_IP )
            if args[ 'protocol' ] != IPPROTO_IP:
               args[ 'protoMask' ] = 0xff
            args.pop( 'ipv6' )
            self._handleIp6Args( mode, args ) # pylint: disable-msg=protected-access

         @staticmethod
         def handler( mode, args ):
            if CliCommand.isNoOrDefaultCmd( args ):
               cls.adaptNoOrDefaultRuleCommonArgs( mode, args )

            cls.handleIp6Rule( mode, **args )

         noOrDefaultHandler = handler

      PolicyMapModePbr.addCommandClass( Ip6AclCommand )

      #-----------------------------------------------------------------
      # [ no | default ] SEQUENCES
      #-----------------------------------------------------------------
      class NoSequenceCmd( CliCommand.CliCommandClass ):
         noOrDefaultSyntax = 'SEQUENCES'
         data = {
            'SEQUENCES': AclCliRules.seqRangeMatcher
         }
         noOrDefaultHandler = cls.handleNoSeq

      PolicyMapModePbr.addCommandClass( NoSequenceCmd )

      #-----------------------------------------------------------------
      # resequence [ start [ incr ] ]
      #-----------------------------------------------------------------
      class ResequenceCmd( CliCommand.CliCommandClass ):
         syntax = 'resequence [ START [ INC ] ]'
         data = {
            'resequence' : PolicyMapCliLib.matcherResequence,
            'START' : PolicyMapCliLib.matcherStartSequence,
            'INC' : PolicyMapCliLib.matcherIncSequence,
         }

         @staticmethod
         def handler( mode, args ):
            start = args.get( 'START', PolicyMapCliLib.defaultSeqStart )
            inc = args.get( 'INC', PolicyMapCliLib.defaultSeqInc )
            cls.handleResequence( mode, start, inc )

      PolicyMapModePbr.addCommandClass( ResequenceCmd )

PbrCfgCmdBinder()

#--------------------------------------------------------------------------------
# [ no | default ] service-policy type pbr input PMAP_NAME [ fallback ]
#--------------------------------------------------------------------------------
def pbrFallbackPolicySupportedGuard( mode, token ):
   if rtgHwStatus.pbrFallbackPolicySupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

class HandleServicePolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'service-policy type pbr input PMAP_NAME [ fallback ]'
   noOrDefaultSyntax = syntax
   data = {
      'service-policy' : CliCommand.guardedKeyword( 'service-policy',
         helpdesc='Service policy configuration', guard=guardServicePolicyPbr ),
      'type' : PolicyMapCliLib.matcherType,
      'pbr' : nodePbr,
      'input' : 'Apply the policy map to ingress packets',
      'PMAP_NAME' :  matcherPmapNamePbr,
      'fallback' : CliCommand.guardedKeyword( 'fallback',
         helpdesc=( 'Fallback policy to be in effect when the PBR policy applied on '
                    'an interface is being modified or no PBR policy is applied.' ),
         guard=pbrFallbackPolicySupportedGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      fallback = 'fallback' in args
      pmapName = args[ 'PMAP_NAME' ]
      no = CliCommand.isNoOrDefaultCmd( args )
      def maybeAddWarning( mode, allIntfStatus, intfName ):
         intfStatus = allIntfStatus.intfStatus.get( intfName )
         if intfStatus and intfStatus.forwardingModel != 'intfForwardingModelRouted':
            mode.addWarning( 'Configuration will be ignored while interface %s '
                             'is not a routed port.' % intfName )

      if no is False:
         intfName = mode.intf.name
         if pmapName != pbrCliIntfConfig.intf.get( intfName, None ):
            maybeAddWarning( mode, allIntfStatusDir, intfName )

      PolicyMapCliLib.handleServicePolicy( mode, no, pmapName, pbrCliConfig, \
                  pbrStatusRequestDir, pbrStatus, fallback,
                  intfConfig=pbrCliIntfConfig,
                  configSessionRollbackSupported=True )

   noOrDefaultHandler = handler

IntfPbrModelet.addCommandClass( HandleServicePolicyCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip policy hardware persistent
#--------------------------------------------------------------------------------
matcherPolicy = CliMatcher.KeywordMatcher( 'policy', helpdesc='Configure IP Policy' )
matcherHardware = CliMatcher.KeywordMatcher(
      'hardware',
      helpdesc='Hardware update configuration' )

class HardwarePersistentCmd( CliCommand.CliCommandClass ):
   syntax = "ip policy hardware persistent"
   noOrDefaultSyntax = syntax
   data = {
      "ip" : CliToken.Ip.ipMatcherForConfig,
      "policy" : matcherPolicy,
      "hardware" : matcherHardware,
      "persistent" : CliCommand.guardedKeyword(
                        "persistent",
                        helpdesc="Policy to remain in hardware for shutdown "
                                 "interfaces",
                        guard=guardHardwarePersistent )

   }
   @staticmethod
   def handler( mode, args ):
      pbrCliConfig.isConfigDriven = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pbrCliConfig.isConfigDriven = False

BasicCli.GlobalConfigMode.addCommandClass( HardwarePersistentCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip policy hardware update hitful forced
#--------------------------------------------------------------------------------
class HardwareUpdateHitfulForcedCmd( CliCommand.CliCommandClass ):
   syntax = 'ip policy hardware update hitful forced'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'policy' : matcherPolicy,
      'hardware' : matcherHardware,
      'update' : 'Policy update configuration',
      'hitful' : CliCommand.guardedKeyword(
                     'hitful', helpdesc='Hitful policy updates',
                     guard=guardHardwareUpdateHitfulForced ),
      'forced' : 'Only allow hitful policy updates',
   }

   @staticmethod
   def handler( mode, args ):
      pbrCliConfig.inPlaceUpdateDisabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pbrCliConfig.inPlaceUpdateDisabled = False

BasicCli.GlobalConfigMode.addCommandClass( HardwareUpdateHitfulForcedCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip policy unresolved-nexthop action drop [ immediate ]
#--------------------------------------------------------------------------------
class NexthopUnresolvedDropCmd( CliCommand.CliCommandClass ):
   syntax = 'ip policy unresolved-nexthop action drop [ immediate ]'
   noOrDefaultSyntax = 'ip policy unresolved-nexthop action [ drop [ immediate ] ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'policy' : matcherPolicy,
      'unresolved-nexthop' : ( 'Configure the action to take when nexthops in '
                               'IP policy are unresolved' ),
      'action' : ( 'Configure the action to take when nexthops in IP policy are '
                   'unresolved' ),
      'drop' : 'Drop packets',
      'immediate' : 'Drop packets immediately',
   }

   @staticmethod
   def handler( mode, args ):
      if 'immediate' in args:
         pbrCliConfig.pbrUnresolvedNexthopAction = DropImmediate
      else:
         pbrCliConfig.pbrUnresolvedNexthopAction = DropDelayed

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      hasDrop = 'drop' in args
      hasDropImmediate = 'immediate' in args
      currAction = pbrCliConfig.pbrUnresolvedNexthopAction
      # Remove dropAction only when current unresolved nexthop action matches the
      # configured drop type
      if hasDropImmediate:
         if currAction != DropImmediate:
            return
      elif hasDrop:
         if currAction != DropDelayed:
            return
      pbrCliConfig.pbrUnresolvedNexthopAction = NoAction

BasicCli.GlobalConfigMode.addCommandClass( NexthopUnresolvedDropCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip policy counters poll interval SECONDS seconds
#--------------------------------------------------------------------------------
class CountersPollIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ip policy counters poll interval SECONDS seconds'
   noOrDefaultSyntax = 'ip policy counters poll interval ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'policy' : matcherPolicy,
      'counters' : 'Counters update configuration',
      'poll' : CliCommand.guardedKeyword( 'poll',
                                          helpdesc='Counters polling configuration',
                                          guard=guardCountersPollCommands ),
      'interval' : 'Counters update interval',
      'SECONDS' : CliMatcher.IntegerMatcher( 2, 60,
                                             helpdesc='Seconds between consecutive '
                                                      'polls' ),
      'seconds' : 'Time unit in seconds',
   }

   @staticmethod
   def handler( mode, args ):
      pbrCliConfig.counterPollInterval = args[ 'SECONDS' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pbrCliConfig.counterPollInterval = 0.0

BasicCli.GlobalConfigMode.addCommandClass( CountersPollIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip policy match protocol bgp
#--------------------------------------------------------------------------------
class BgpPbrRedirectCmd( CliCommand.CliCommandClass ):
   syntax = 'ip policy match protocol bgp'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'policy' : matcherPolicy,
      'match' : CliCommand.guardedKeyword( 'match',
         helpdesc='Match the access rule specified',
         guard=guardBgpRedirectPbr ),
      'protocol' : 'Specify protocol of matched packets',
      'bgp' : 'Match BGP packets',
   }

   @staticmethod
   def handler( mode, args ):
      pbrCliConfig.bgpPbrRedirect = not CliCommand.isNoOrDefaultCmd( args )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( BgpPbrRedirectCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip policy mac address aging
#--------------------------------------------------------------------------------
class NexthopEnableNonAgingCmd( CliCommand.CliCommandClass ):
   syntax = 'ip policy mac address aging'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'policy' : matcherPolicy,
      'mac' : CliToken.Mac.macMatcherForConfig,
      'address' : CliCommand.guardedKeyword( 'address',
         helpdesc='Configure the mac-addresses', guard=guardNonAgingMacPbr ),
      'aging' : 'Set to allow the nexthop mac-addresses to age',
   }

   @staticmethod
   def handler( mode, args ):
      pbrCliConfig.pbrEnableNonAgingMacAddr = CliCommand.isNoOrDefaultCmd( args )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( NexthopEnableNonAgingCmd )

#-------------------------------------------------------------------------------
# Support for default interface command.
#-------------------------------------------------------------------------------
class PbrIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      t0( "PbrIntfJanitor: interface", self.intf_.name, "going away" )
      if self.intf_.name in pbrCliIntfConfig.intfFallback:
         del pbrCliIntfConfig.intfFallback[ self.intf_.name ]
      if self.intf_.name in pbrCliIntfConfig.intf:
         del pbrCliIntfConfig.intf[ self.intf_.name ]

#--------------------------------------------------------------------------------
# show class-map type pbr [ CMAP_NAME ]
#--------------------------------------------------------------------------------
class ClassMapTypePbrCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show class-map type pbr [ CMAP_NAME ]'
   data = {
      'class-map' : CliCommand.guardedKeyword( 'class-map',
         helpdesc='Show Class Map', guard=guardClassMap ),
      'type' : PolicyMapCliLib.matcherType,
      'pbr' : nodePbr,
      'CMAP_NAME' : matcherCmapNamePbr
   }
   cliModel = PolicyMapModel.ClassMapAllModel

   @staticmethod
   def handler( mode, args ):
      mapType = 'pbr'
      cmapName = args.get( 'CMAP_NAME' )
      cmapAllModel = PolicyMapModel.ClassMapAllModel()
      emapType = PolicyMap.mapTypeToEnum( mapType )
      cmapsContainer = ClassMapModelContainer( pbrMergedConfig, pbrStatus, emapType,
                                               cmapAllModel )
      if cmapName is None:
         cmapsContainer.populateAll()
      else:
         cmapsContainer.populateClassMap( cmapName )
      return cmapAllModel

BasicCli.addShowCommandClass( ClassMapTypePbrCmd )

#--------------------------------------------------------------------------------
# show policy-map type pbr [ PMAP_NAME ] [ summary ]
#--------------------------------------------------------------------------------
class PolicyMapTypePbrCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policy-map type pbr [ PMAP_NAME ] [ summary ]'
   data = {
      'policy-map' : CliCommand.guardedKeyword( 'policy-map',
         helpdesc='Show Policy Map', guard=guardPolicyMap ),
      'type' : PolicyMapCliLib.matcherType,
      'pbr' : nodePbr,
      'summary' : 'Policy map summary',
      'PMAP_NAME' :  matcherPmapNamePbr
   }
   cliModel = PolicyMapModel.PolicyMapAllModel

   @staticmethod
   def handler( mode, args ):
      mapType = 'pbr'
      summary = 'summary' in args
      pmapName = args.get( 'PMAP_NAME' )
      pmapAllModel = PolicyMapModel.PolicyMapAllModel()
      pmapAllModel.mapType = 'mapPbr'
      emapType = PolicyMap.mapTypeToEnum( mapType )
      counterStatusDir = getCounterStatusDir()
      pmapsContainer = PolicyMapModelContainer( pbrCliConfig, pbrCliIntfConfig,
                                                pbrMergedIntfConfig,
                                                pbrMergedConfig, pbrStatus,
                                                AclCli.config, None, None,
                                                vtiStatusDir, dynVlanIntfDir,
                                                emapType, None, 
                                                policyMapAllModel=pmapAllModel,
                                                counterStatusDir=counterStatusDir )
      checkpoints = ( pbrCheckpoint, _pbrSessionCheckpoint( mode ) )
      if pmapName is None:
         pmapsContainer.populateAll( mode, summary, checkpoints )
      else:
         pmapsContainer.populatePolicyMap( mode, pmapName, summary, checkpoints )
      return pmapAllModel

BasicCli.addShowCommandClass( PolicyMapTypePbrCmd )

#--------------------------------------------------------------------------------
# clear ip policy counters [ PMAP_NAME ] [ session ]
#--------------------------------------------------------------------------------
def clearPolicyMapCountersAll( mode, ckptStatus ):
   for policyName in pbrCliConfig.pmapType.pmap:
      clearOnePolicyMapCounters( mode, policyName, ckptStatus )

def clearOnePolicyMapCounters( mode, policyName, ckptStatus ):
   # Clear counters for one policymap
   pmapConfig = PolicyMapCliLib.getPmapSubConfig( pbrCliConfig, policyName )
   pmapStatuses = [ status.status.get( policyName ) for status in
                    pbrStatus.values() ]
   pmapStatuses = [ _f for _f in pmapStatuses if _f ]
   if pmapStatuses:
      pmapStatus = pmapStatuses[ 0 ]
   else:
      pmapStatus = None
   if not pmapConfig or not pmapStatus:
      return

   counterStatusDir = getCounterStatusDir()
   counterInShark =  counterStatusDir is not None
   if counterInShark :
      agentName = pmapStatus.parent.name # get agent name from sysdb
      pmapKey = PolicyMapCountersKey( policyName, "" )
      pmapCounters = counterStatus.policyMapCounters.get( pmapKey )
      if not pmapCounters:
         return
      pbrCounterStatus = { agentName: pmapCounters }
      clearPolicyMapCounters( mode, policyName, pbrCliConfig, pbrMergedConfig,
                              pbrCounterStatus, ckptStatus,
                              counterInShark=counterInShark )
      return
   clearPolicyMapCounters( mode, policyName, pbrCliConfig, pbrMergedConfig,
                           pbrStatus, ckptStatus )
   
class ClearIpPolicyCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip policy counters [ PMAP_NAME ] [ session ]'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'ip' : CliToken.Ip.ipMatcherForClear,
      'policy' : CliCommand.guardedKeyword( 'policy',
         helpdesc='Clear IP policy(PBR) counters', guard=guardPolicyMap ),
      'counters' : 'Clear IP policy(PBR) counters',
      'session' : 'Clear IP Policy (PBR) counters in current session',
      'PMAP_NAME' :  matcherPmapNamePbr
   }

   @staticmethod
   def handler( mode, args ):
      if 'session' in args:
         ckptStatus = _pbrSessionCheckpoint( mode )
         logAction = False
      else:
         ckptStatus = pbrCheckpoint
         logAction = True

      policyName = args.get( 'PMAP_NAME' )
      if policyName:
         if logAction:
            Intf.Log.logClearCounters( "ip policy", "policy map %s" % policyName )
         clearOnePolicyMapCounters( mode, policyName, ckptStatus )
      else:
         if logAction:
            Intf.Log.logClearCounters( "ip policy", "policy map all" )
         clearPolicyMapCountersAll( mode, ckptStatus )

BasicCli.EnableMode.addCommandClass( ClearIpPolicyCountersCmd )

def startPbrConfigMergeSm():
   global pbrMergedIntfConfig
   global pbrMergedConfig
   global pbrConfigMergeSm
   global pbrMergedIntfConfig
   pbrMergedConfig = Tac.newInstance( 'Pbr::PbrConfig', 'pbrConfig' )
   pbrMergedIntfConfig = Tac.newInstance( 'PolicyMap::IntfConfig', 'pbrIntfConfig' )
   pbrMergedIntfConfig.type = 'mapPbr'
   pbrConfigMergeSm = Tac.newInstance( "Pbr::PbrConfigMergeSm",
                                       entityManager.cEntityManager(), pbrConfigDir,
                                       pbrIntfConfigDir, pbrCliIntfConfigReadOnly,
                                       pbrCliConfigReadOnly, vtiStatusDir,
                                       dynVlanIntfDir, pbrMergedIntfConfig,
                                       pbrMergedConfig )
   pbrConfigMergeSm.initialized = True

#--------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#--------------------------------------------------------------------------------
def Plugin( em ):
   global pbrCliIntfConfig, pbrCliConfig, pbrStatusRequestDir, pbrStatus
   global pbrCliIntfConfigReadOnly, pbrCliConfigReadOnly, pbrConfigDir
   global pbrIntfConfigDir
   global rtgHwStatus, ethLagIntfStatusDir, allIntfStatusDir
   global policyOpChkr
   global aclStatus
   global pbrCheckpoint
   global statusDp
   global entityManager
   global cliNexthopGroupConfig
   global vtiStatusDir
   global dynVlanIntfDir
   global counterStatus

   pbrRootNode = 'pbr'
   pbrCellRootNode = 'cell/%d/pbr' % Cell.cellId()
   aclRootNode = 'acl'
   pbrCliConfigNode = 'input/pmap/cli'
   pbrConfigDirNode = 'input/pmap/config'
   statusNode = 'status'
   pbrCliConfigPath = pbrRootNode + '/' + pbrCliConfigNode
   pbrConfigDirPath = pbrRootNode + '/' + pbrConfigDirNode
   pbrCliConfigType = 'Pbr::PbrConfig'
   pbrStatusPath = pbrCellRootNode + '/' + statusNode
   pbrStatusType = 'Tac::Dir'
   aclStatusPath = aclRootNode + '/' + statusNode + '/' + 'all'
   aclStatusType = 'Acl::Status'
   lagStatusPath = 'interface/status/eth/lag'
   lagStatusType = 'Interface::EthLagIntfStatusDir'
   routingHwStatusPath = 'routing/hardware/status'
   routingHwStatusType = 'Routing::Hardware::Status'
   allIntfStatusPath = 'interface/status/all'
   allIntfStatusType = 'Interface::AllIntfStatusDir'
   entityManager = em

   mountGroup = entityManager.mountGroup()
   pbrStatus = mountGroup.mount( pbrStatusPath, pbrStatusType, 'ri' )
   mountGroup.close( callback=None )

   pbrCliIntfConfig = ConfigMount.mount( entityManager, "pbr/input/intf/cli",
                                         "PolicyMap::IntfConfig", 'wi' )
   pbrCliConfig = ConfigMount.mount( entityManager, pbrCliConfigPath,
                                     pbrCliConfigType, 'wi' )
   pbrStatusRequestDir = LazyMount.mount( entityManager,
                                          PbrLib.pbrStatusRequestDirPath,
                                          PbrLib.pbrStatusRequestDirType,
                                          'wc' )
   ethLagIntfStatusDir = LazyMount.mount( entityManager, lagStatusPath,
                                          lagStatusType, 'r' )
   cliNexthopGroupConfig = LazyMount.mount( entityManager,
                                            'routing/nexthopgroup/input/cli',
                                            'Routing::NexthopGroup::ConfigInput',
                                            'r' )

   mergeConfigMg = entityManager.mountGroup()
   pbrCliConfigReadOnly = mergeConfigMg.mount( pbrCliConfigPath,
                                               pbrCliConfigType, 'wi' )
   pbrCliIntfConfigReadOnly = mergeConfigMg.mount( 'pbr/input/intf/cli',
                                                   'PolicyMap::IntfConfig', 'wi' )
   pbrConfigDir = mergeConfigMg.mount( pbrConfigDirPath, 'Tac::Dir', 'ri' )
   pbrIntfConfigDir = mergeConfigMg.mount( 'pbr/input/intf/config', 'Tac::Dir',
                                           'ri' )

   vtiStatusDir = mergeConfigMg.mount('interface/status/eth/vxlan',
                                      'Vxlan::VtiStatusDir', 'r')
   dynVlanIntfDir = mergeConfigMg.mount( 'interface/input/dynvlanintf',
                                         'Tac::Dir', 'ri')
   mergeConfigMg.close( startPbrConfigMergeSm )

   aclStatus = LazyMount.mount( entityManager, aclStatusPath, aclStatusType, 'r' )
   rtgHwStatus = LazyMount.mount( entityManager, routingHwStatusPath,
                                  routingHwStatusType, 'r' )
   allIntfStatusDir = LazyMount.mount( entityManager, allIntfStatusPath,
                                       allIntfStatusType, "r" )

   pbrCheckpoint = LazyMount.mount( entityManager, "pbr/checkpoint",
                                    "Pbr::CheckpointStatus", "w" )

   statusDp = LazyMount.mount( entityManager,
                               "cell/%d/acl/status/dp" % Cell.cellId(),
                               "Tac::Dir", "ri" )
   sharkMountInfo = SharkLazyMount.mountInfo( 'shadow' )
   counterStatus = SharkLazyMount.mount(
      entityManager,
      "policyMap/counters/pbr",
      "PolicyMap::Counters::PolicyMapTypeCounters",
      sharkMountInfo,
      autoUnmount = True )
   # Install a callback to verify impact of acl changes on policy
   policyOpChkr = PolicyMapCliLib.PolicyOpChkr( pbrStatusRequestDir, pbrStatus )
   AclCli.registerAclConfigValidCallback( isPolicyHwStatusOk )
   IntfCli.Intf.registerDependentClass( PbrIntfJanitor )
