# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os

import BasicCli
import CliMatcher
import ShowCommand
import Tracing

from CliPlugin import PcrLogCliModel, TpmCliLib
from CliPlugin import ConfigMgmtMode
import TpmPcrLib
from TpmPcrLib import PcrEventTypeCamelCase
from TpmGeneric.Defs import PcrEventRevision, PcrEventType

th = Tracing.Handle( 'PcrLogCli' )

pcrIndexMatcher = CliMatcher.IntegerMatcher( 0, 23,
                                             helpdesc='PCR index' )

def populatePcrLog( model, pcrLog ):
   for l in pcrLog:
      pcrLogModel = PcrLogCliModel.PcrLog()
      logVer = int( l[ TpmPcrLib.verIdx ] )
      idx = TpmPcrLib.logIdxList[ logVer ]
      pcrLogModel.timestamp = float( l[ idx.timestamp ] )
      pcrLogModel.register = int( l[ idx.pcr ] )
      if logVer >= 1:
         pcrLogModel.event = PcrEventTypeCamelCase[ 0x0 ]
         pcrLogModel.eventRevision = 0
         pcrLogModel.extend = l[ idx.hex ]
         pcrLogModel.context = l[ idx.extra : ]
      if logVer >= 2:
         # Event type added in PCR log version 2
         pcrLogModel.eventRevision = 0
         eventTypeCode = int( l[ idx.event ], 16 )
         event = PcrEventTypeCamelCase.get( eventTypeCode,
                                            PcrEventTypeCamelCase[ 0x0 ] )
         pcrLogModel.event = event
      if logVer >= 3:
         # Event revision number added in PCR log version 3 (last item in the log)
         pcrLogModel.eventRevision = int( l[ idx.eventRev ] )
         # Log metadata will not include the last item in the ver3 log
         pcrLogModel.context = l[ idx.extra : idx.eventRev ]

      model.logs.append( pcrLogModel )

def showPcrLog( mode, args ):
   result = PcrLogCliModel.PcrLogList()
   path = os.environ.get( 'TEST_PCRLOGCLI', TpmPcrLib.PCR_LOG_PATH )

   try:
      pcrLog = TpmPcrLib.pcrLogRead( args.get( 'PCR_INDEX' ), path=path )
   except FileNotFoundError:
      mode.addErrorAndStop( "Cannot find PCR extension log file." )
   except ValueError as error:
      mode.addErrorAndStop( str( error ) )

   populatePcrLog( result, pcrLog )
   return result

def populatePcrEventRevision( mode, model, abootRevDict ):
   def _snakeToCamel( name ):
      return PcrEventTypeCamelCase.get( PcrEventType[ name ].value,
                                        PcrEventTypeCamelCase[ 0x0 ] )

   if abootRevDict:
      foundUnknownAbootEvent = False

      for eventName, revNum in abootRevDict.items():
         revModel = model.abootEventRevision
         try:
            if PcrEventType[ eventName ].value > TpmPcrLib.TCG_EVENT_FLAG:
               revModel = model.biosEventRevision
            revModel[ _snakeToCamel( eventName ) ] = revNum
         except KeyError:
            revModel[ eventName ] = revNum
            foundUnknownAbootEvent = True

      if foundUnknownAbootEvent:
         mode.addWarning( "Unknown event found in Aboot/BIOS "
                          "PCR extension revision file" )

   for eventName, revNum in PcrEventRevision.items():
      model.eosEventRevision[ _snakeToCamel( eventName ) ] = revNum

def showPcrEventRevision( mode, args ):
   result = PcrLogCliModel.PcrEventRevisionList()
   abootRevDict = None
   path = os.environ.get( 'TEST_ABOOTPCRREVCLI',
                          TpmPcrLib.ABOOT_PCR_LOG_REVISION_PATH )

   try:
      abootRevDict = TpmPcrLib.parseAbootPcrEventRevision( path=path )
   except FileNotFoundError:
      mode.addWarning( "Cannot find Aboot/BIOS PCR extension revision file" )
   except ( ValueError, OSError ) as error:
      mode.addErrorAndStop( str( error ) )

   populatePcrEventRevision( mode, result, abootRevDict )
   return result

class ShowPcrLogCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management security tpm pcr log [ PCR_INDEX ]'
   data = {
      'management': ConfigMgmtMode.managementShowKwMatcher,
      'security': 'Show security status',
      'tpm': TpmCliLib.tpmMatcher,
      'pcr': 'Show PCR attestation information',
      'log': 'Show PCR extension logs',
      'PCR_INDEX': pcrIndexMatcher,
   }
   cliModel = PcrLogCliModel.PcrLogList
   handler = showPcrLog

BasicCli.addShowCommandClass( ShowPcrLogCmd )

class ShowPcrEventRevisionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management security tpm pcr event revision'
   data = {
      'management': ConfigMgmtMode.managementShowKwMatcher,
      'security': 'Show security status',
      'tpm': TpmCliLib.tpmMatcher,
      'pcr': 'Show PCR attestation information',
      'event': 'Show PCR extension event info',
      'revision': 'Show PCR log revisions',
   }
   cliModel = PcrLogCliModel.PcrEventRevisionList
   handler = showPcrEventRevision

BasicCli.addShowCommandClass( ShowPcrEventRevisionCmd )
