# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from datetime import datetime

from CliModel import Dict, Enum, Float, Int, List, Model, Str
from TableOutput import createTable, Format
from TpmPcrLib import PcrEventTypeCamelCase

class PcrLog( Model ):
   timestamp = Float( help="Timestamp when PCR extended" )
   register = Int( help="TPM PCR index" )
   event = Enum( values=PcrEventTypeCamelCase.values(),
                 help="Event type of this PCR log" )
   eventRevision = Int( help="Event revision number of this PCR log" )
   extend = Str( help="Hexdecimal extended to PCR" )
   context = List( valueType=str,
                   help=( "Additional data/context about this PCR log" ) )

class PcrLogList( Model ):
   logs = List( valueType=PcrLog,
                help=( "TPM Platform Configuration Register extension logs" ) )

   def render( self ):
      t = createTable( ( "Timestamp", "PCR", "Event Type", "Event Revision",
                         "Hexdecimal Extended",
                         "Additional Data" ), indent=0 )
      f = Format( justify="left", maxWidth=25, wrap=True )
      f.noPadLeftIs( True )
      f.padLimitIs( True )
      rightFormat = Format( justify="right" )
      rightFormat.noPadLeftIs( True )
      rightFormat.padLimitIs( True )
      t.formatColumns( f, rightFormat, f, f, f )
      for l in self.logs:
         t.newRow( datetime.fromtimestamp( l.timestamp ),
                   l.register,
                   l.event,
                   l.eventRevision,
                   l.extend,
                   ' | '.join( l.context ) )
      print( t.output() )

class PcrEventRevisionList( Model ):
   biosEventRevision = Dict( keyType=str, valueType=int,
                             help=( "A mapping between BIOS TPM Platform "
                                    "Configuration Register extension event "
                                    "name and its revision number" ) )
   abootEventRevision = Dict( keyType=str, valueType=int,
                              help=( "A mapping between Aboot TPM Platform "
                                     "Configuration Register extension event "
                                     "name and its revision number" ) )
   eosEventRevision = Dict( keyType=str, valueType=int,
                            help=( "A mapping between EOS TPM Platform "
                                   "Configuration Register extension event "
                                   "name and its revision number" ) )

   def render( self ):
      t = createTable( ( "Event Domain", "Event Type", "Event Revision" ), indent=0 )
      f = Format( justify="left", maxWidth=30, wrap=True )
      f.noPadLeftIs( True )
      f.padLimitIs( True )
      t.formatColumns( f, f, f )

      for k, v in self.biosEventRevision.items():
         t.newRow( "BIOS", k, v )
      for k, v in self.abootEventRevision.items():
         t.newRow( "Aboot", k, v )
      for k, v in self.eosEventRevision.items():
         t.newRow( "EOS", k, v )

      print( t.output() )
