#!/usr/bin/env python3
# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=anomalous-backslash-in-string

from __future__ import absolute_import, division, print_function
import re
import Tac
from CliPlugin import PhysicalIntfRule
from CliPlugin import StpCliLib
from CliPlugin import VirtualIntfRule

PortChannelNum = Tac.Type( "Lag::PortChannelNum" )

peerEthRegex = '([Pp](?:eer)?[Ee]t(?:hernet)?)\s*' \
               '(\d+(?:/\d{1,3}){0,2})'

peerPortChannelRegex = '([Pp](?:eer)?[Pp]o(?:rt-[Cc]hannel)?)\s*' \
                       '([1-9]\d*)'

peerIntfRegex = peerEthRegex + '|' + peerPortChannelRegex

class PeerIntf( object ): # pylint: disable=useless-object-inheritance
   """PeerIntf is meant to be a stand-in for an IntfCli.Intf in places that
   only access the name and shortname members."""
   compiledRegex = re.compile( peerIntfRegex )
   def __init__( self, inputName ):
      raiseError = False
      # inputName might be a full name or a short name, and it might not be
      # capitalized correctly.  Decode it and set name and shortname.
      m = re.match( self.compiledRegex, inputName )

      try:
         if ( ( m.start() == 0 ) and ( m.end() == len( inputName ) ) ):
         # peerIntfRegex has matched the entire inputName 
            # pylint: disable-next=singleton-comparison,superfluous-parens
            prefix, suffix = [ g.lower() for g in m.groups() if (g != None) ]
            if 'et' in prefix:
               self.name = 'PeerEthernet' + suffix
               self.shortname = 'PEt' + suffix
            elif 'po' in prefix:
               self.name = 'PeerPort-Channel' + suffix
               self.shortname = 'PPo' + suffix
            else:
               raiseError = True
         else:
         # We have extra chars either at the beginning or end of
         # inputName that do not match peerIntfRegex
            raiseError = True

      except AttributeError:
         # inputName does not match peerIntfRegex
         raiseError = True

      if raiseError:
         # pylint: disable-next=consider-using-f-string
         raise Exception( "Unrecognized interface name: %s" % inputName )

# for new parser in STP package, used in the EBNF syntaxes 
peerPortChannelIntfMatcher = VirtualIntfRule.VirtualIntfMatcher(
   'PeerPort-Channel',
   PortChannelNum.min,
   PortChannelNum.max,
   alternates=[ 'PPort-Channel' ],
   value=lambda mode, intf: PeerIntf( intf ) )


peerEthIntfMatcher = PhysicalIntfRule.PhysicalIntfMatcher(
   'PeerEthernet',
   helpdesc='Peer ethernet interface',
   alternates=[ 'PEt' ],
   value=lambda mode, intf: PeerIntf( intf ),
   peer=True )

StpCliLib.stpIntfMatcher |= peerPortChannelIntfMatcher
StpCliLib.stpIntfMatcher |= peerEthIntfMatcher
