# Copyright (c) 2008-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
Implementation of the CLI command(s):

   show interface [ <interface> ] phy [ detail ]
"""

# pylint: disable=too-many-nested-blocks
# pylint: disable=consider-using-enumerate
# pylint: disable=consider-using-f-string
# pylint: disable=import-outside-toplevel
# pylint: disable=consider-using-from-import

# pkgdeps: library L1IntfResourceTypes

from collections import defaultdict
from collections import namedtuple
from collections import OrderedDict
import os
import sys

import Ark
import Arnet
import ArPyUtils
import AgentDirectory
import BasicCli
import BasicCliUtil
import Cell
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.TechSupportCli
import CliPlugin.XcvrAllStatusDir as XcvrAllStatusDir
import CliPlugin.XcvrEthIntfDir as XcvrEthIntfDir
from CliPlugin import PhyFeatureCli
from CliPlugin import PhyFecCli
from CliPlugin import PhyLineRateModel
from CliPlugin import PhyModel
from CliPlugin import PhyModulationModel
from CliPlugin import PhyPrecodingModel
from CliPlugin import PhyStatusModel
from CliPlugin.PhyStatusModel import escapeNonPrintable
import CliSavePlugin.PhyCliSave as PhyCliSave
import CliToken.Clear
import CliToken.Platform
import CliToken.Reset
import CommonGuards
import ConfigMount
import EthIntfLib
import LazyMount
import PhyStatusLib
import Intf.IntfRange as IntfRange
import Tac
from TypeFuture import TacLazyType
import CliGlobal

PhyPosition = TacLazyType( 'PhyEee::PhyPosition' )
intfType = namedtuple( 'interfaceType', 'coherent coherentCapable' )
IntfState = namedtuple( 'IntfState', [ 'state', 'changes', 'lastChange' ] )
FecHistogramState = namedtuple( 'FecHistogramState', [ 'model',
                                                       'phyPos',
                                                       'histogram',
                                                       'phyName' ] )
PhyFeatureConfigType = TacLazyType( "Hardware::Phy::PhyFeatureConfig" )
LineRateEnum = TacLazyType( 'Phy::Coherent::LineRate' )
ModEnum = TacLazyType( 'Phy::Coherent::Modulation' )
OpticalDetect = TacLazyType( 'Xcvr::OpticalDetect' )
PhyState = TacLazyType( 'Hardware::Phy::PhyStatus::PhyState' )
IdOps = TacLazyType( 'Hardware::L1Topology::IdentifierOperators' )
ComponentType = TacLazyType( 'Hardware::L1Topology::ComponentType' )

em = None
xcvrStatusDir = None
intfXcvrStatusDir = None
phyDebugStateDir = None
phyFeatureConfigSliceDir = None
coherentStatusDir = None
archerXcvrStatusDir = None
allPhyModels = None
precodingSliceDir = None
fecStatusDir = None
fecConfigDir = None
fecCounterCheckpointDir = None
pcspmaCounterCheckpointDir = None
phyFeatureConfigPreCreatorStartUpSm = None
xcvrConfigDir = None
fecHistogramDir = None
hwSliceDir = None
chipSliceResetDir = None
ethPhyIntfDefaultConfigDir = None
gv = CliGlobal.CliGlobal( l1PhyStatusDir=None,
                          aggregatedTable=None,
                          serdesMappingAggrSm=None,
                          l1MappingDir=None,
                          l1TopoDir=None,
                          topoHelper=None )


# Search and retrieve the data for the requested intfs alone
# under available sub-slices.
testPatternStatusDirGlobal = None
def getAllTestPatternStatus( intfs ):
   global testPatternStatusDirGlobal
   if testPatternStatusDirGlobal is None:
      mg = em.mountGroup()
      testPatternStatusDir = mg.mount(
         'hardware/phy/status/data/testPattern/slice', 'Tac::Dir', 'ri' )
      mg.close( blocking=True )
      testPatternStatusDirGlobal = testPatternStatusDir
   else:
      testPatternStatusDir = testPatternStatusDirGlobal

   # Get all the Status info from Fixed/Modular Systems
   allTestPatternStatus = {}
   allTestPatternCaps = {}
   for linecard in testPatternStatusDir.values():
      for intfList in linecard.values():
         for intf in intfs:
            if intf.name in intfList.capabilities:
               allTestPatternCaps[ intf.name ] = \
                     intfList.capabilities[ intf.name ]
            if intf.name in intfList.testPatternStatus:
               allTestPatternStatus[ intf.name ] = \
                     intfList.testPatternStatus[ intf.name ]
   return allTestPatternStatus, allTestPatternCaps

def coherentStatusGuard( mode, token ):
   # pylint: disable=len-as-condition
   if coherentStatusDir.get( 'slice' ) and len( coherentStatusDir[ 'slice' ] ):
      return None
   else:
      return CliParser.guardNotThisPlatform

phyShowKwNode = CliCommand.guardedKeyword( 'phy',
                                       'Display low-level PHY status',
                                       CommonGuards.standbyGuard )
phyNode = CliCommand.guardedKeyword( 'phy',
      helpdesc='Configure phy-specific parameters',
      guard=CommonGuards.standbyGuard )

linkKwMatcher = CliMatcher.KeywordMatcher( 'link',
                                           helpdesc='Set link properties' )

def isHwPhyStatusSubType( p ):
   '''A phy's status object not being a subclass of Hardware::Phy::PhyStatus, is
   an indication that many fields of show int phy detail should not be printed.
   This method is the mechanism to determine when a field should be printed.'''
   return isinstance( p, Tac.Type( "Hardware::Phy::PhyStatus" ) )

def isHwPhyStatusGen3( p ):
   '''If a phy's status object is of type Hardware::PhyStatus::PhyStatusGen3 the
   the entire output of show int phy de is CAPI-fied'''
   return isinstance( p, Tac.Type( "Hardware::PhyStatus::PhyStatusGen3" ) )

skewShowKw = CliCommand.guardedKeyword( 'skew',
                                        'Display coherent skew information',
                                        coherentStatusGuard )

persistentShowKw = CliCommand.guardedKeyword( 'persistent',
                                              'Display coherent skew information ' \
                                              'stored in persistent storage',
                                              coherentStatusGuard )

class PhyConfigModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( 'Ethernet' )

IntfCli.IntfConfigMode.addModelet( PhyConfigModelet )

#----------------------------------------------------------------------------
#
# Summary / brief results, i.e.,
#
#    show interface [ <interface> ] phy
#
#----------------------------------------------------------------------------
def _clause45PmaPmdBitsFromPhyStatus( p ):
   if not p.pmaPmdDebugStateValid:
      return ''

   pmaPmdBits = { 'linkDown' : 'D', 'linkUp' : 'U' }[ p.pmaPmdRxLinkStatus ]
   pmaPmdBits += 'R' if p.pmaPmdRxFault else '.'
   pmaPmdBits += 'T' if p.pmaPmdTxFault else '.'

   return pmaPmdBits


def _clause45PcsBitsFromPhyStatus( p ):
   if not p.pcsDebugStateValid:
      return ''

   pcsBits = { 'linkDown' : 'D', 'linkUp' : 'U' }[ p.pcsRxLinkStatus ]
   pcsBits += 'R' if p.pcsRxFault else '.'
   pcsBits += 'T' if p.pcsTxFault else '.'
   pcsBits += 'B' if p.pcsHighBer else '.'
   pcsBits += '.' if p.pcsBlockLock else 'L'

   return pcsBits


def _clause45PhyXsBitsFromPhyStatus( p ):
   if not p.phyXsDebugStateValid:
      return ''

   phyXsBits = { 'linkDown' : 'D', 'linkUp' : 'U' }[ p.phyXsTxLinkStatus ]
   phyXsBits += 'R' if p.phyXsRxFault else '.'
   phyXsBits += 'T' if p.phyXsTxFault else '.'
   phyXsBits += '.' if p.phyXsLaneAlignmentStatus else 'A'
   phyXsBits += '.' if ( p.phyXsLane0To3Sync & 0x1 ) else '0'
   phyXsBits += '.' if ( p.phyXsLane0To3Sync & 0x2 ) else '1'
   phyXsBits += '.' if ( p.phyXsLane0To3Sync & 0x4 ) else '2'
   phyXsBits += '.' if ( p.phyXsLane0To3Sync & 0x8 ) else '3'

   return phyXsBits

def generateClause45PhySummary( p ):
   phyState = PhyStatusModel.phyStateToCliStr[ p.phyState ]
   phyStateChanges = p.phyStateChanges

   pmaPmdBits = _clause45PmaPmdBitsFromPhyStatus( p )
   pcsBits = _clause45PcsBitsFromPhyStatus( p )
   phyXsBits = _clause45PhyXsBitsFromPhyStatus( p )

   fmtStr = '%-14.14s %-15.15s %8.8s %8.8s %-7.7s %-5.5s %-8.8s'
   if p.showIntfPhySupported:
      statusLine = fmtStr % ( p.intfId,
                              phyState,
                              phyStateChanges,
                              p.resetCount,
                              pmaPmdBits,
                              pcsBits,
                              phyXsBits )
   else:
      statusLine = fmtStr % tuple( ( p.intfId, ) + ( "-", ) * 6 )

   return PhyModel.InterfacePhyStatusSummary(
         text=statusLine, clause="clause45" )

def generateClause22PhySummary( p ):
   phyState = PhyStatusModel.phyStateToCliStr[ p.phyState ]
   phyStateChanges = p.phyStateChanges

   if p.tacType.fullTypeName == "Hardware::Phy::PhyBcm54980Status":
      if p.bcmStats.sgmiiLinkState.current:
         sgmiiLinkState = 'linkUp'
      else:
         sgmiiLinkState = 'linkDown'

      if p.bcmStats.copperEnergy.current:
         copperEnergy = 'linkUp'
      else:
         copperEnergy = 'linkDown'
   else:
      copperEnergy = 'unknown'
      sgmiiLinkState = 'unknown'

   fmtStr = "%-14.14s %-15.15s %8.8s %8.8s %-8.8s %-8.8s"
   statusLine = fmtStr % ( p.intfId,
                           phyState,
                           str( phyStateChanges ),
                           str( p.resetCount ),
                           copperEnergy,
                           sgmiiLinkState )
   return PhyModel.InterfacePhyStatusSummary(
         text=statusLine, clause="clause22" )

def showInterfacesPhySummary( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   exposeInternal = 'all' in args
   phyStatuses, _, _, _, _ = getSysdbPhyStatuses( mode, intf, mod, detail=False,
                                                  exposeInternal=exposeInternal )
   if not phyStatuses:
      return PhyModel.InterfacesPhyStatusesSummary( interfacePhyStatuses={} )

   summaries = {}

   for p in phyStatuses:
      if p.tacType.fullTypeName in phySummaryModelsCol:
         modelType = phySummaryModelsCol[ p.tacType.fullTypeName ]
         summaries[ p.intfId ] = modelType().toPhySummaryModel( p )
      elif not isHwPhyStatusSubType( p ):
         # Just print a line for the interface to indicate that the phy exists.
         # Non-Hardware::Phy::PhyStatus-typed objects don't print information here.
         summaries[ p.intfId ] = PhyModel.InterfacePhyStatusSummary(
            text=str( p.intfId ), clause="clause45" )
      elif p.mdioClause == 'clause22':
         summaries[ p.intfId ] = generateClause22PhySummary( p )
      elif p.mdioClause == 'clause45':
         summaries[ p.intfId ] = generateClause45PhySummary( p )
      else:
         assert 0, "I only know about Clause22 and Clause45 PHYs, sorry!"

   return PhyModel.InterfacesPhyStatusesSummary(
         interfacePhyStatuses=summaries )

#----------------------------------------------------------------------------
#
# Detailed results, i.e.,
#
#    show interface [ <interface> ] phy detail
#
#----------------------------------------------------------------------------
xcvrPresenceToStr = {
   'xcvrPresenceUnknown' : 'unknown',
   'xcvrNotPresent' : 'not present',
   'xcvrPresent' : 'present'
   }

def _maybePrintPluggableXcvrSn( xcvrStatus, fmt2 ):
   if xcvrStatus.swappability == 'fixed':
      return

   if xcvrStatus.presence != 'xcvrPresent':
      return

   print( fmt2 % ( 'Transceiver SN', xcvrStatus.vendorInfo.vendorSn ) )

def _printPhyModel( p, fmt ):
   hwRev = p.hwRev if hasattr( p, 'hwRev' ) else ''
   modelStr = p.model if hasattr( p, 'model' ) else ''
   if hwRev:
      modelStr = modelStr + " (" + hwRev + ")"
   # OUI, device model and revision as read from HW
   if hasattr( p, 'phyOui' ) and p.phyOui:
      modelStr = modelStr + " (0x%06x,0x%x,0x%x)" % ( p.phyOui, p.phyModel,
                                                      p.phyRev )
   # Use a generic phy model name if the model string ends up being empty.
   print( fmt % ( 'Model', modelStr if modelStr else "PHY" ) )

def _getInterfaceState( p, phyIntfs ):
   phyIntf = phyIntfs.get( p.intfId )
   if phyIntf is None:
      return None
   intfStatusStr = phyIntf.prettyIntfStateStrings()[ 0 ]
   if intfStatusStr == "administratively down":
      intfStatusStr = 'adminDown'
   ethPhyIntfStatus = phyIntf.status()
   return IntfState( state=intfStatusStr,
                     changes=ethPhyIntfStatus.operStatusChange.count,
                     lastChange=ethPhyIntfStatus.operStatusChange.time )

def _getTransceiverMediaTypeString( intfId ):
   ethIntfXcvrStatus = intfXcvrStatusDir.get( intfId )
   xcvrTypeStr = ethIntfXcvrStatus.xcvrType if ethIntfXcvrStatus else ''
   return xcvrTypeStr

def _printGeneralDetail( p, xcvrStatus, fmt, fmt2, phyIntfs, tpStatus,
                         phyFeatureStatusSubSliceDir ):

   # NOTE: when changing the output of this print, also update the
   #       PhyCliTest regression test!!

   # XXX: PhyCliTest is a ptest, a breadth test would be really really useful
   if isHwPhyStatusGen3( p ):
      phyState = p.phyState.current
      phyStateChanges = p.phyState.changes
      lastPhyStateChange = p.phyState.lastChange
   else:
      phyState = getattr( p, 'phyState', PhyState.unknown )
      phyStateChanges = getattr( p, 'phyStateChanges', 0 )
      lastPhyStateChange = getattr( p, 'lastPhyStateChange', 0.0 )
   print( fmt % ( 'PHY state',
                 PhyStatusModel.phyStateToCliStr[ phyState ], phyStateChanges,
                 Ark.timestampToStr( lastPhyStateChange ) ) )

   intfState = _getInterfaceState( p, phyIntfs )
   if intfState:
      # XXX: this print has no coverage in PhyCliTest.py
      print( fmt % ( 'Interface state',
                    PhyStatusModel.interfaceStateToCliStr[ intfState.state ],
                    intfState.changes,
                    Ark.timestampToStr( intfState.lastChange ) ) )

   # NOTE: when changing the output of this print, also update the
   #       PhyCliTest regression test!!
   lastResetTime = p.lastResetTime if hasattr( p, 'lastResetTime' ) else 0.0
   print( fmt % ( 'HW resets',
                 '',
                 p.resetCount if hasattr( p, 'resetCount' ) else 0,
                 Ark.timestampToStr( lastResetTime ) ) )

   if xcvrStatus:
      print( fmt % ( 'Transceiver',
                    _getTransceiverMediaTypeString( p.intfId ),
                    xcvrStatus.presenceChanges,
                    Ark.timestampToStr( xcvrStatus.lastPresenceChange ) ) )

      _maybePrintPluggableXcvrSn( xcvrStatus, fmt2 )

   if hasattr( p, 'operSpeed' ):
      print( fmt % ( 'Oper speed', PhyModel.ethSpeedToStr[ p.operSpeed ], '', '' ) )

   # Display Modulation only when modulation is available and not Unknown
   if hasattr( p, 'modulation' ) and p.modulation != "modulationUnknown":
      print( fmt % ( 'Modulation', PhyModel.ethModulationToStr[ p.modulation ],
                    '', '' ) )

   # Display Lane Count only when laneCount is available and not Unknown
   if hasattr( p, 'laneCount' ) and p.laneCount != 'laneCountUnknown':
      print( fmt % ( 'Lane Count', PhyModel.ethLaneCountToStr[ p.laneCount ],
      '', '' ) )

   if hasattr( p, 'interruptCount' ):
      print( fmt % ( 'Interrupt Count', '', p.interruptCount, '' ) )

   if xcvrStatus:
      if xcvrStatus.resetCountValid:
         print( fmt % ( 'Transceiver Reset Count',
                       '',
                       xcvrStatus.xcvrStats.resetCount,
                       Ark.timestampToStr( xcvrStatus.xcvrStats.lastReset ) ) )

      if xcvrStatus.interruptCountValid:
         print( fmt % ( 'Transceiver Interrupt Count',
                       '',
                       xcvrStatus.xcvrStats.interruptCount,
                       Ark.timestampToStr( xcvrStatus.xcvrStats.lastInterrupt ) ) )

      if xcvrStatus.smbusFailCountValid:
         print( fmt % ( 'Transceiver Smbus Failures',
                       '',
                       xcvrStatus.xcvrStats.smbusFailCount,
                       Ark.timestampToStr( xcvrStatus.xcvrStats.lastSmbusFail ) ) )

   if tpStatus:
      print( fmt2 % ( 'Test pattern', "enabled" ) )

   if hasattr( p, 'diagsMode' ):
      print( fmt2 % ( 'Diags mode', p.diagsMode ) )

   _printPhyModel( p, fmt2 )

   # BUG387101: Remove the check for '0000' firmwareVersion when fixing this bug
   if( isHwPhyStatusSubType( p ) and p.firmwareVersion and
       p.firmwareVersion != "0000" ):
      versionOutput = p.firmwareVersion
      if p.model and p.model.lower() in allPhyModels.phyModel:
         fwVerAvailable = allPhyModels.phyModel[ p.model.lower() ].firmwareVersion
         assert fwVerAvailable, "Firmware version information of model %s is " \
               "empty!" % p.model.lower()
         if fwVerAvailable != p.firmwareVersion:
            versionOutput += " (%s available)" % fwVerAvailable

      print( fmt2 % ( 'PHY firmware version',  versionOutput ) )

   if hasattr( p, 'activeBlob' ):
      print( fmt2 % ( 'Active uC image', p.activeBlob ) )

   if hasattr( p, 'loopback' ):
      print( fmt2 % ( 'Loopback', p.loopback ) )

   txClockShift = PhyFeatureCli.getTxClockShift( phyFeatureStatusSubSliceDir,
                                                 p.intfId )
   if ( txClockShift !=
          Tac.Type( "Hardware::Phy::PhyFeatureConfig" ).defaultTxClockShift ):
      print( fmt2 % ( 'Configured TX clock shift', txClockShift ) )

def _maybePrintPmaPmdDetail( p, fmt ):

   # PMA/PMD -- 1.xxxx
   # Applies only to subtypes of Hardware::Phy::PhyStatus and clause 45.
   if not isHwPhyStatusSubType( p ) or p.mdioClause != 'clause45':
      return

   if p.pmaPmdDebugSignalDetectValid:
      print( fmt % ( 'PMA/PMD RX signal detect',
                    'ok' if p.pmaPmdRxSignalDetect else 'no signal',
                    p.pmaPmdRxSignalDetectChanges,
                    Ark.timestampToStr( p.lastPmaPmdRxSignalDetectChange ) ) )

   if not p.pmaPmdDebugStateValid:
      return

   if p.pmaPmdRxLinkStatusValid:
      print( fmt % ( 'PMA/PMD RX link status',
                    { 'linkDown' : 'down', 'linkUp' : 'up' }[ p.pmaPmdRxLinkStatus ],
                    p.pmaPmdRxLinkStatusChanges,
                    Ark.timestampToStr( p.lastPmaPmdRxLinkStatusChange ) ) )

   if p.pmaPmdRxFaultValid:
      print( fmt % ( 'PMA/PMD RX fault',
                    'fault' if p.pmaPmdRxFault else 'ok',
                    p.pmaPmdRxFaultChanges,
                    Ark.timestampToStr( p.lastPmaPmdRxFaultChange ) ) )

   if p.pmaPmdTxFaultValid:
      print( fmt % ( 'PMA/PMD TX fault',
                    'fault' if p.pmaPmdTxFault else 'ok',
                    p.pmaPmdTxFaultChanges,
                    Ark.timestampToStr( p.lastPmaPmdTxFaultChange ) ) )


def _maybePrintPcsDetail( p, fmt ):

   # PCS -- 3.xxxx
   if not isHwPhyStatusSubType( p ) or not p.pcsDebugStateValid:
      return

   if p.pcsRxLinkStatusValid:
      print( fmt % ( 'PCS RX link status',
                    { 'linkDown' : 'down', 'linkUp' : 'up' }[ p.pcsRxLinkStatus ],
                    p.pcsRxLinkStatusChanges,
                    Ark.timestampToStr( p.lastPcsRxLinkStatusChange ) ) )

   if p.pcsRxFaultValid:
      print( fmt % ( 'PCS RX fault',
                    'fault' if p.pcsRxFault else 'ok',
                    p.pcsRxFaultChanges,
                    Ark.timestampToStr( p.lastPcsRxFaultChange ) ) )

   if p.pcsTxFaultValid:
      print( fmt % ( 'PCS TX fault',
                    'fault' if p.pcsTxFault else 'ok',
                    p.pcsTxFaultChanges,
                    Ark.timestampToStr( p.lastPcsTxFaultChange ) ) )

   if p.pcsBlockLockValid:
      print( fmt % ( 'PCS block lock',
                    'ok' if p.pcsBlockLock else 'no block lock',
                    p.pcsBlockLockChanges,
                    Ark.timestampToStr( p.lastPcsBlockLockChange ) ) )

   if p.pcsHighBerValid:
      print( fmt % ( 'PCS high BER',
                    'high BER' if p.pcsHighBer else 'ok',
                    p.pcsHighBerChanges,
                    Ark.timestampToStr( p.lastPcsHighBerChange ) ) )

   if p.pcsErrBlocksValid:
      print( fmt % ( 'PCS err blocks',
                    p.pcsErrBlocks,
                    '',
                    Ark.timestampToStr( p.lastPcsNonzeroErrBlock ) ) )

   if p.pcsBerValid:
      print( fmt % ( 'PCS BER',
                    p.pcsBer,
                    p.pcsNonzeroBer,
                    Ark.timestampToStr( p.lastPcsNonzeroBer ) ) )

   if p.pcsRxLpiValid:
      print( fmt % ( 'PCS RX LPI received',
                   'true' if p.pcsRxLpiReceived else 'false',
                   p.pcsRxLpiChanges,
                   Ark.timestampToStr( p.lastPcsRxLpiChange ) ) )

   if p.pcsAmLockValid:
      print( fmt % ( 'PCS AM lock',
                    'ok' if p.pcsAmLock else 'no AM lock',
                    p.pcsAmLockChanges,
                    Ark.timestampToStr( p.lastPcsAmLockChange ) ) )

   if p.pcsDeskewStatusValid:
      print( fmt % ( 'PCS deskew status',
                    'true' if p.pcsDeskewStatus else 'false',
                    p.pcsDeskewStatusChanges,
                    Ark.timestampToStr( p.lastPcsDeskewStatusChange ) ) )

   if p.pcsBipErrorsValid:
      print( fmt % ( 'PCS BIP Errors',
                    str( p.pcsBipErrors ),
                    p.pcsNonzeroBip,
                    Ark.timestampToStr( p.lastPcsNonzeroBip ) ) )

   if p.pcsSyncStatusValid:
      print( fmt % ( 'PCS sync status',
                    'true' if p.pcsSyncStatus else 'false',
                    p.pcsSyncStatusChanges,
                    Ark.timestampToStr( p.lastPcsSyncStatusChange ) ) )

def _maybePrintPhyXsDetail( p, fmt ):

   # PHY XS -- 4.xxxx
   if not isHwPhyStatusSubType( p ) or not p.phyXsDebugStateValid:
      return

   # 4.0001.2 -- PHY XS TX link status (LL)
   print( fmt % ( 'XFI/XAUI TX link status',
                 { 'linkDown' : 'down', 'linkUp' : 'up' }[ p.phyXsTxLinkStatus ],
                 p.phyXsTxLinkStatusChanges,
                 Ark.timestampToStr( p.lastPhyXsTxLinkStatusChange ) ) )

   # 4.0008.10 -- PHY XS RX Fault (latched high)
   print( fmt % ( 'XFI/XAUI RX fault',
                 'fault' if p.phyXsRxFault else 'ok',
                 p.phyXsRxFaultChanges,
                 Ark.timestampToStr( p.lastPhyXsRxFaultChange ) ) )

   # 4.0008.11 -- PHY XS TX Fault (latched high)
   print( fmt % ( 'XFI/XAUI TX fault',
                 'fault' if p.phyXsTxFault else 'ok',
                 p.phyXsTxFaultChanges,
                 Ark.timestampToStr( p.lastPhyXsTxFaultChange ) ) )

   # 4.0018.12 -- PHY XS lane alignment status
   print( fmt % ( 'XFI/XAUI alignment status',
                 'ok' if p.phyXsLaneAlignmentStatus else 'unaligned',
                 p.phyXsLaneAlignmentStatusChanges,
                 Ark.timestampToStr( p.lastPhyXsLaneAlignmentStatusChange ) ) )

   # 4.0018.[3:0] -- PHY XS Lane 0 - 3 sync
   xls = "(0123) = "
   xls += '1' if ( p.phyXsLane0To3Sync & 0x1 ) else '0'
   xls += '1' if ( p.phyXsLane0To3Sync & 0x2 ) else '0'
   xls += '1' if ( p.phyXsLane0To3Sync & 0x4 ) else '0'
   xls += '1' if ( p.phyXsLane0To3Sync & 0x8 ) else '0'
   print( fmt % ( 'XAUI lane 0-3 sync',
                 xls,
                 p.phyXsLane0To3SyncChanges,
                 Ark.timestampToStr( p.lastPhyXsLane0To3SyncChange ) ) )

   print( fmt % ( 'XAUI sync w/o align HWM',
                 p.xauiSyncsUntilAlignHwm,
                 '',
                 Ark.timestampToStr( p.lastXauiSyncsUntilAlignHwmIncrement ) ) )

   print( fmt % ( 'XAUI excess sync w/o align',
                 p.excessiveXauiSyncUntilAlign,
                 '',
                 Ark.timestampToStr( p.lastExcessiveXauiSyncUntilAlign ) ) )

# The below Xcvr details are not used anymore but we still need to print these lines
# (with a hardcoded 0) to avoid a change in the CLI output
def printCommonXcvrDetails( p, fmt ):

   if not isHwPhyStatusSubType( p ):
      return

   # Details common to all Xcvr types
   print( fmt % ( 'Xcvr EEPROM read timeout',
                 '',
                 0,
                 'never' ) )

   print( fmt % ( 'Spurious xcvr detection',
                 '',
                 0,
                 'never' ) )

   print( fmt % ( 'DOM control/status fail',
                 '',
                 0,
                 '' ) )


def printXcvrDetails( xcvrStatus, fmt, lane ):

   # Xcvr-global state (i.e., applicable to the *Xcvr*, not a specific
   # lane)
   xcvrStats = xcvrStatus.xcvrStats

   print( fmt % ( 'Presence indication',
                 xcvrStats.presence,
                 xcvrStats.presenceChanges,
                 Ark.timestampToStr( xcvrStats.lastPresenceChange ) ) )

   print( fmt % ( 'Bad EEPROM checksums',
                 '',
                 xcvrStats.badEepromChecksumEvents,
                 Ark.timestampToStr( xcvrStats.lastBadEepromChecksumEvent ) ) )

   # Lane-specific state
   laneStats = xcvrStatus.laneStats[ lane ]

   print( fmt % ( 'RX_LOS since system boot',
                 laneStats.currentRxLosState,
                 laneStats.rxLosEvents,
                 Ark.timestampToStr( laneStats.lastRxLosEvent ) ) )

   print( fmt % ( 'RX_LOS since insertion',
                 '',
                 laneStats.rxLosEventsSinceInsertion,
                 '' ) )

   print( fmt % ( 'TX_FAULT since system boot',
                 laneStats.currentTxFaultState,
                 laneStats.txFaultEvents,
                 Ark.timestampToStr( laneStats.lastTxFaultEvent ) ) )

   print( fmt % ( 'TX_FAULT since insertion',
                 '',
                 laneStats.txFaultEventsSinceInsertion,
                 '' ) )


def _maybePrintXcvrDetail( p, xcvrStatus, fmt ):
   if not xcvrStatus:
      return

   if xcvrStatus.swappability == 'fixed':
      return

   printCommonXcvrDetails( p, fmt )

   # Not all xcvr drivers fill these in.
   if not xcvrStatus.eventStatsValid:
      return


   # p.lane is currently defined only in the Bcm84740Status, not in
   # the base PhyStatus in PhyEee.
   #
   # We ought to move the "lane" attribute from the Bcm84740Status
   # into the base PhyStatus, defined in the PhyEee package, and make
   # sure it is set properly for all transceiver types (SFP, QSFP,
   # ...). Then, we can just get rid of this switch-on-xcvrType, and
   # just call "printXcvrDetails" regardless of transceiver type:
   #
   #    printXcvrDetails( p.xcvrType, fmt, p.lane )
   #

   # if the phy status does not have a lane attribute, assume lane 0 for now.
   phyLane = p.lane if hasattr( p, "lane" ) else 0

   if xcvrStatus.xcvrType == 'cfp2':
      printXcvrDetails( xcvrStatus, fmt, phyLane )
      return

   if xcvrStatus.xcvrType == 'qsfpPlus':
      if xcvrStatus.swizzled:
         # this is an adapted SFP, same phyLane adjustment
         # as xcvrType == 'sfpPlus'
         printXcvrDetails( xcvrStatus.sfpStatus, fmt, 0 )
      else:
         printXcvrDetails( xcvrStatus, fmt, phyLane )
      return

   if xcvrStatus.xcvrType == 'sfpPlus':
      # Hack -- until we have the "lane" attr in the base PhyStatus
      # type, hard-code lane 0 for SFP.
      printXcvrDetails( xcvrStatus, fmt, 0 )
      return


def _debugStateStale( ps, xcvrStatus, snapshotRequest ):
   if ps.diagsMode != 'normalOperation':
      return (True, "port is in diags mode %s" % ps.diagsMode)

   if ps.phyState == 'errDisabled':
      return (True, "port is errDisabled")

   if not xcvrStatus:
      return (True, "Xcvr state unavailable" )

   # If a transceiver is present, the driver should be updating the
   # debug snapshot on request. If the timestamp is not up-to-date,
   # then the snapshot is stale.
   #
   # I don't think I've ever seen this case.
   if ( ps.debugStateSnapshotTimestamp < snapshotRequest ) and \
      ( xcvrStatus.presence == 'xcvrPresent' ):
      return (True, "failed to update debug state quickly enough")

   return (False,"")

def printOnePhyStatusPcsDetail( p, fmt ):
   _maybePrintPcsDetail( p, fmt )

def _determineSupportL1TopoOrNot( phyIntf, intfId ):
   # Currently, only support Ethernet interface. Fabric interfaces have no
   # resourceManagerStatus and L1 Topology APIs currently can't support Switch
   # interfaces
   supportedIntfType = intfId.startswith( "Ethernet" )
   if supportedIntfType:
      # We grab the activity lock because We need to make sure that L1 Topology
      # entities are not changed in between 1. reading L1 Topology generation
      # valid and 2. accessing other L1 Topology entities
      phyTypes = None
      with Tac.ActivityLockHolder():
         genReader = Tac.newInstance( "Hardware::L1Topology::GenerationReader",
                                      gv.l1TopoDir )
         l1TopoGenValid = genReader.topoGen().valid
         if l1TopoGenValid:
            # In some scenarios, linecard may be removed during the execution
            # of "sh int phy det" cmds. In this case, Hardware::L1Topology::
            # TraversalHelper::getPhyTypesForIntfId() will crash. Therefore,
            # We have to check whether the interface is active or not before
            # calling getPhyTypesForIntfId().
            slotId = phyIntf.slotId()
            if slotId:
               intfSet = gv.topoHelper.getIntfsInSlot( slotId )
               if intfId in intfSet.intf:
                  phyTypes = gv.topoHelper.getPhyTypesForIntfId( intfId )
      if phyTypes:
         return all( phyType in _phyDetailL1TopologySupportedPhys
                     for phyType in phyTypes.phyType )
   return False

def _getChipDesc( curCompGroup ):
   firstSerdesTopo = list( curCompGroup.component.values() )[ 0 ]
   chipTopo = firstSerdesTopo.phyChipTopology()
   chipDesc = chipTopo.descriptor
   return chipDesc

def _getDescs( phyIntf, intfId ):
   # Now traverse from the component closest to interface back to
   # the last component we can

   ingressDescsByHop = []
   egressDescsByHop = []
   rms = phyIntf.resourceManagerStatus()
   if not rms:
      return ( ingressDescsByHop, egressDescsByHop )
   ethIntfMode = rms.ethIntfModeDir.ethIntfMode.get( intfId )
   if not ethIntfMode:
      return ( ingressDescsByHop, egressDescsByHop )

   # We grab the activity lock because We need to make sure that L1 Topology
   # entities are not changed in between 1. reading L1 Topology generation
   # valid and 2. accessing other L1 Topology entities
   with Tac.ActivityLockHolder():
      genReader = Tac.newInstance( "Hardware::L1Topology::GenerationReader",
                                   gv.l1TopoDir )
      l1TopoGenValid = genReader.topoGen().valid
      if not l1TopoGenValid:
         return ( ingressDescsByHop, egressDescsByHop )

      travDirection = gv.topoHelper.getDirectionForIntfId( intfId, True )
      curCompGroupIngress = gv.topoHelper.getComponentGroupForIntfId( intfId,
                                                                      ethIntfMode,
                                                                      False )
      while curCompGroupIngress and curCompGroupIngress.component:
         descsIngress = [ comp.descriptor
                          for comp in curCompGroupIngress.component.values() ]
         if curCompGroupIngress.componentType == ComponentType.componentSerdes:
            ingressDescsByHop.append( ( descsIngress,
                                        _getChipDesc( curCompGroupIngress ) ) )
         curCompGroupIngress = gv.topoHelper.nextComponentGroup( curCompGroupIngress,
                                                                 travDirection,
                                                                 True )
         travDirection = IdOps.invertDirection( travDirection )

      travDirection = gv.topoHelper.getDirectionForIntfId( intfId, True )
      curCompGroupEgress = gv.topoHelper.getComponentGroupForIntfId( intfId,
                                                                     ethIntfMode,
                                                                     True )
      while curCompGroupEgress and curCompGroupEgress.component:
         descsEgress = [ comp.descriptor
                         for comp in curCompGroupEgress.component.values() ]
         if curCompGroupEgress.componentType == ComponentType.componentSerdes:
            egressDescsByHop.append( ( descsEgress,
                                       _getChipDesc( curCompGroupEgress ) ) )
         curCompGroupEgress = gv.topoHelper.nextComponentGroup( curCompGroupEgress,
                                                                travDirection,
                                                                True )
         travDirection = IdOps.invertDirection( travDirection )

   # Currently, L1Topology doesn't support egress and ingress have different hop
   # counts.
   assert len( ingressDescsByHop ) == len( egressDescsByHop ), \
      "Hop count of ingress path and egress path should be the same since " + \
      "currently L1 Topology doesn't support egress and ingress have different " + \
      "hop counts."

   return ( ingressDescsByHop, egressDescsByHop )

# Use serdes descriptors as key to lookup the phy HW TAC model mapping table
def _getPhyHwTacModels( serdesDescs, intfId ):
   phyHwTacModel = None
   for serdesDesc in serdesDescs:
      if serdesDesc.tx:
         phyHwTacModel = gv.aggregatedTable.serdesTx.get( serdesDesc )
      else:
         phyHwTacModel = gv.aggregatedTable.serdesRx.get( serdesDesc )
      if phyHwTacModel:
         break

   return phyHwTacModel

# Use chip descriptors as key to lookup the chip TAC model mapping table
def _getChipHwTacModels( chipDesc, intfId ):
   chipHwTacModel = None
   if chipDesc:
      chipHwTacModel = gv.aggregatedTable.chip.get( chipDesc )

   return chipHwTacModel

def _getPhyCliModel( serdesDescs, intfId ):
   phyCliModel = None
   if serdesDescs:
      phyType = serdesDescs[ 0 ].phyType
      phyScope = serdesDescs[ 0 ].phyScope
      typeScopeTuple = ( phyType, phyScope )
      phyCliModel = _getPhyCliModelFromCol( typeScopeTuple )

   return phyCliModel

def _getChipCliModel( chipDesc, intfId ):
   chipCliModel = None
   if chipDesc:
      chipType = chipDesc.chipType
      chipCliModel = _getChipCliModelFromCol( chipType )

   return chipCliModel

def printOnePhyStatusDetail( p, xcvrStatus, snapshotRequest, f,
                             phyTopology, phyIntfs, tpStatus, precoding,
                             phyFeatureStatusSubSliceDir, mode ):
   phyStatuses = []
   operSpeed = None
   interfaceState = None
   transceiver = None
   rs = None

   if isHwPhyStatusGen3( p ):
      intfState = _getInterfaceState( p, phyIntfs )
      if intfState:
         interfaceState = PhyStatusModel.InterfaceState().toModel( intfState.state,
               intfState.changes, intfState.lastChange )
      if xcvrStatus:
         mediaType = _getTransceiverMediaTypeString( p.intfId )
         transceiver = PhyStatusModel.Transceiver().toModel( mediaType, xcvrStatus )
      if p.rs:
         rs = PhyStatusModel.RsBaseR().toModel( p.rs )

   if isHwPhyStatusSubType( p ):
      ( statusStale, _unused ) = _debugStateStale( p, xcvrStatus, snapshotRequest )
   else:
      statusStale = False

   fmt = PhyStatusModel.phyDetailStatFmt
   fmt2 = PhyStatusModel.phyDetailInfoFmt

   def stdoutModel( phyStatusEntity ):
      nextPhyDisplayFn = phySpecificDisplayFn.get(
         phyStatusEntity.tacType.fullTypeName )
      if nextPhyDisplayFn:
         with ArPyUtils.FileHandleInterceptor( [ sys.stdout.fileno(),
            sys.stderr.fileno() ] ) as dataDumpCapture:
            nextPhyDisplayFn( phyStatusEntity, fmt, fmt2 )
            sys.stdout.flush()
            sys.stderr.flush()
         # output can have non-printable characters so properly escape.
         return PhyModel.RawPhyStatus(
            text=escapeNonPrintable( dataDumpCapture.contents() ),
            description=PhyModel.PhyDescription( phyChipName="" ) )
      return None

   if p.tacType.fullTypeName not in phyDetailModelsCol or \
      p.tacType.fullTypeName in includeRawPhyStatusForPhyDetailCol:
      with ArPyUtils.FileHandleInterceptor(
            [ sys.stdout.fileno(), sys.stderr.fileno() ] ) as dataDumpCapture:
         # If not a subclass of Hardware::Phy::PhyStatus, still assume much of this
         # information needs to be dumped.
         if not isHwPhyStatusSubType( p ) or p.showIntfPhySupported:
            _printGeneralDetail( p, xcvrStatus, fmt, fmt2, phyIntfs, tpStatus,
                                 phyFeatureStatusSubSliceDir )
            _maybePrintXcvrDetail( p, xcvrStatus, fmt )
            _maybePrintPmaPmdDetail( p, fmt )
            _maybePrintPcsDetail( p, fmt )
            _maybePrintPhyXsDetail( p, fmt )
         else:
            _printPhyModel( p, fmt2 )

         if precoding:
            precoding.render()

         if f:
            f.render()

         psdf = phySpecificDisplayFn.get( p.tacType.fullTypeName )
         if psdf:
            psdf( p, fmt, fmt2 )

         sys.stdout.flush()
         sys.stderr.flush()

      phyStatusArgs = {}
      phyStatusArgs[ 'text' ] = escapeNonPrintable( dataDumpCapture.contents() )
      phyStatusArgs[ 'description' ] = PhyModel.PhyDescription( phyChipName="" )
      if statusStale:
         phyStatusArgs[ 'stale' ] = True
      rawPhyStatusModel = PhyModel.RawPhyStatus( **phyStatusArgs )
      phyStatuses.append( rawPhyStatusModel )

   # check the phy topology for any phy status updates.
   phyIntf = phyIntfs[ p.intfId ]
   phyIntfStatuses = phyTopology.phyIntfStatuses.get( p.intfId )
   supportL1Topo = _determineSupportL1TopoOrNot( phyIntf, p.intfId )
   if supportL1Topo:
      # traversal from XCVR side to chip side and get all serdes descriptors
      ingressDescsByHop, egressDescsByHop = _getDescs( phyIntf, p.intfId )

      # Now use the descriptors to lookup the HW TAC model and CLI model
      # (from chip side to XCVR side)
      # Ingress way
      visitedChipDescsIngress = set()
      for serdesDescsIngress, chipDesc in reversed( ingressDescsByHop ):
         if chipDesc.stringValue() in visitedChipDescsIngress:
            # Found a duplicated chip model ==> return None to avoiding printing it
            chipDesc = None
         else:
            visitedChipDescsIngress.add( chipDesc.stringValue() )
         # Use descriptors as key to lookup the HW TAC model mapping table
         phyHwTacModelIng = _getPhyHwTacModels( serdesDescsIngress, p.intfId )
         chipHwTacModelIng = _getChipHwTacModels( chipDesc, p.intfId )
         # Use descriptors as key to lookup the CLI model mapping table
         phyCliModelIng = _getPhyCliModel( serdesDescsIngress, p.intfId )
         chipCliModelIng = _getChipCliModel( chipDesc, p.intfId )
         # populate the CLI model by HW TAC model
         if chipHwTacModelIng and chipCliModelIng:
            populatedChipCliModelIng = chipCliModelIng().toModel( chipHwTacModelIng )
            phyStatuses.append( populatedChipCliModelIng )
         if phyHwTacModelIng and phyCliModelIng:
            populatedPhyCliModelIng = phyCliModelIng().toModel( phyHwTacModelIng )
            phyStatuses.append( populatedPhyCliModelIng )
      # Egress way
      visitedChipDescsEgress = set()
      for serdesDescsEgress, chipDesc in reversed( egressDescsByHop ):
         if chipDesc.stringValue() in visitedChipDescsEgress:
            # Found a duplicated chip model ==> return None to avoiding printing it
            chipDesc = None
         else:
            visitedChipDescsEgress.add( chipDesc.stringValue() )
         # Use descriptors as key to lookup the HW TAC model mapping table
         phyHwTacModelEg = _getPhyHwTacModels( serdesDescsEgress, p.intfId )
         chipHwTacModelEg = _getChipHwTacModels( chipDesc, p.intfId )
         # Use descriptors as key to lookup the CLI model mapping table
         phyCliModelEg = _getPhyCliModel( serdesDescsEgress, p.intfId )
         chipCliModelEg = _getChipCliModel( chipDesc, p.intfId )
         # populate the CLI model by HW TAC model
         if chipHwTacModelEg and chipCliModelEg:
            populatedChipCliModelEg = chipCliModelEg().toModel( chipHwTacModelEg )
            phyStatuses.append( populatedChipCliModelEg )
         if phyHwTacModelEg and phyCliModelEg:
            populatedPhyCliModelEg = phyCliModelEg().toModel( phyHwTacModelEg )
            phyStatuses.append( populatedPhyCliModelEg )
   elif phyIntfStatuses:
      ethPhyIntfStatus = phyIntf.status()
      for phyPos, intfTopoData in phyIntfStatuses.phyIntfData.items():
         _, phyIntfStatus = PhyStatusLib.phyTopologyConfigStatus( em.root(),
               intfTopoData )
         if phyIntfStatus:
            modelTypes = phyDetailModelsCol.get( phyIntfStatus.
                                                 tacType.fullTypeName )
            staleModel = None
            if modelTypes:
               # Some of the external phys like Enigma support debugStateSnapshot.
               # These phys register their phy status type in
               # debugStateSupportedPhyTypes.
               # The external phys typically don't update the debugState when an
               # interface is inactive. So skip waiting for debugStateSnapshot to be
               # updated in such cases
               if phyIntfStatus.tacType.fullTypeName in \
                     debugStateSupportedPhyTypes and ethPhyIntfStatus.active:
                  try:
                     # pylint: disable=cell-var-from-loop
                     Tac.waitFor( lambda:
                           phyIntfStatus.debugStateSnapshotTimestamp >=
                           snapshotRequest, sleep=True, timeout=15,
                           description='external phy to update debug state',
                           maxDelay=1.0 )
                  except Tac.Timeout:
                     # Phy status was not updated in time. Do the best it can and
                     # print whatever is possible
                     staleModel = True

               for modelType in modelTypes:
                  model = modelType().toModel( phyIntfStatus )
                  # There can be many reasons why a model couldn't be created.
                  if model is not None:
                     if staleModel is not None:
                        model.stale = staleModel
                     phyStatusTypeName = phyIntfStatus.tacType.fullTypeName
                     if phyStatusTypeName in phySpecificSaveCheckpointFn:
                        getFecStatusGenIdFn = phySpecificGetFecStatusGenIdFn[
                              phyStatusTypeName ]
                        orgGenerationId = getFecStatusGenIdFn( fecConfigDir,
                              fecStatusDir, phyIntfStatus.intfId )
                        cktGenerationId = fecCounterCheckpointDir.generationId.get(
                              phyIntfStatus.intfId )
                        # If 'clear phy counters' was applied to the interface
                        # we will update the counters
                        if ( orgGenerationId is not None and
                             cktGenerationId is not None and
                             orgGenerationId.id == cktGenerationId.id ):
                           phySpecificClearCountersFn[ phyStatusTypeName ](
                              fecCounterCheckpointDir,
                              phyIntfStatus,
                              model )
                     phyStatuses.append( model )
            # The forwarding agent phy status has already been dumped above.
            elif phyPos != PhyPosition.asicPhy:
               # Look in the phySpecificDisplayFn infrastructure to dump additional
               # phy information.
               rawStdoutModel = stdoutModel( phyIntfStatus )
               if rawStdoutModel:
                  phyStatuses.append( rawStdoutModel )

   return PhyModel.InterfacePhyStatusesDetailed( phyStatuses=phyStatuses,
                                                 interfaceState=interfaceState,
                                                 operSpeed=operSpeed,
                                                 transceiver=transceiver,
                                                 macFaults=rs )


def _debugStatePrintable( ps, xcvrStatus, snapshotRequest ):
   # XXX_LWR: I wonder if we shouldn't just implement a
   #          "debugStateValid" externally-implemented attr in the
   #          various PhyStatus types, instead of embedding the
   #          knowledge of what might cause the debug state to be
   #          stale here in the CLI...?

   # If the latest state is newer than our request, the debug state is
   # up-to-date. This is hopefully the common case.
   if ps.debugStateSnapshotTimestamp >= snapshotRequest:
      return True

   # If we do not have a xcvr in the port, the PHY driver is busy
   # trying to detect one. It will not update the debug state until
   # after a xcvr has been detected, so don't wait for that event.
   if not xcvrStatus or xcvrStatus.presence != 'xcvrPresent':
      return True

   if not isinstance( ps, Tac.Type( "Hardware::Phy::PhyStatus" ) ):
      # Subsequent checks are only relevant for Hardware::Phy::PhyStatus objects.
      return False

   # If the PHY is in diags mode, the debug state (and, hence, the
   # snapshot timestamp) will not be updated, meaning this command
   # would hang forever.
   #
   # Instead, just say it's as updated as it will ever be, and print a
   # warning.
   if ps.diagsMode != 'normalOperation':
      return True

   # The driver does not touch/manage/... errDisabled ports, so don't
   # wait for those.
   if ps.phyState == 'errDisabled':
      return True

   return False

def getDebugStateDirName( intf ):
   '''Returns the slot directory for a given interface.

   This method returns the "key" used by many different root directories to
   represent the slot which manages the interface. These directories include:
      - Interface Status
      - Interface Debug Snapshot Request
      - etc.

   The translation has to happen regardless of the type of underlying interface (
   EthIntf, FabricIntf, SubIntf, etc. )

   Args:
      intf ( IntfCli.Intf ): The CLI's interface abstraction object
   '''

   return intf.slotId()

def _requestPhyDebugSnapshot( intf, nowish ):
   intfId = intf.name
   debugStateDirName = getDebugStateDirName( intf )
   assert debugStateDirName, 'Invalid intfId'
   phyDebugState = phyDebugStateDir[ debugStateDirName ].state.newMember( intfId )
   phyDebugState.snapshotRequest = nowish

def getPhyPrecodingDetail( intfs ):
   interfacesStat = {}
   interfaceStat = PhyPrecodingModel.PrecodingTxRxStatuses()
   precodingPhyStatuses = getPhyPrecodingStatuses( intfs )
   for intf, status in precodingPhyStatuses.items():
      # precoding status on each lane
      for laneStatus in status.precodingLaneStatus.values():
         laneStat = PhyPrecodingModel.PrecodingTxRxStatus().toModel( laneStatus )
         interfaceStat.txRxStatus[ laneStatus.lane ] = laneStat
      interfacesStat[ intf ] = interfaceStat
   return interfacesStat

def showInterfacesPhyDetail( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   exposeInternal = 'all' in args
   phyStatuses, foundIntfNames, phyTopology, phyIntfs, phyConfigDict = \
         getSysdbPhyStatuses( mode, intf, mod, detail=True,
                              exposeInternal=exposeInternal )

   if not phyStatuses:
      return PhyModel.InterfacesPhyStatusesDetailed( interfacePhyStatuses={} )

   testPatternIntfs = IntfCli.Intf.getAll( mode, intf, mod )
   allTestPatternStatus = getAllTestPatternStatus( testPatternIntfs )
   # returned ( status, caps ) : retrieve status
   tpStat = allTestPatternStatus[ 0 ]
   fecStatuses = PhyFecCli.showInterfacesPhyFecDetail( em, set( foundIntfNames ) )
   precodingStatuses = getPhyPrecodingDetail( set( foundIntfNames ) )

   nowish = Tac.now()

   # Keep track of the xcvr status associated with the phy status.
   phyStatusXcvrName = {}

   # Ask the PHY driver to snapshot the PHY's current state in great
   # detail by writing a snapshot request. When the snapshot timestamp
   # is later than the request, we know the debug state is current.
   for p in phyStatuses:
      # The phy config is assumed to be associated with a phy status object.
      phyConfig = phyConfigDict[ p.fullName ]
      if isHwPhyStatusGen3( p ) and hasattr( p, "xcvrName" ) and p.xcvrName:
         phyStatusXcvrName[ p.name ] = p.xcvrName
      else:
         phyStatusXcvrName[ p.name ] = phyConfig.xcvrName
      _requestPhyDebugSnapshot( phyIntfs[ p.intfId ], nowish )

   def allSnapshotsPrintable():
      stillUpdating = set( phyStatuses )
      for p in phyStatuses:
         xcvrName = phyStatusXcvrName[ p.name ]
         xcvrStatus = xcvrStatusDir.xcvrStatus.get( xcvrName, None )
         if _debugStatePrintable( p, xcvrStatus, nowish ):
            stillUpdating.remove( p )

      emptySet = set()
      return stillUpdating == emptySet

   # If for some reason, some of the state ends up being stale,
   # display that (with a warning), rather than hanging in the CLI for
   # 10 minutes before croaking with an exception.
   try:
      ds = "debug state to be read -- this may take up to 15 seconds"
      Tac.waitFor( allSnapshotsPrintable, sleep=True, timeout=15,
                   description=ds, maxDelay=1.0 )
   except Tac.Timeout:
      # We explicitly deal with the timeout case later, but print a
      # blank line after all the dots printed by Tac.waitFor.
      print()

   interfacePhyStatuses = {}
   pattern = Tac.Type( "Hardware::Phy::TestPattern" )
   for p in phyStatuses:
      xcvrName = phyStatusXcvrName[ p.name ]
      xcvrStatus = xcvrStatusDir.xcvrStatus.get( xcvrName, None )
      if p.intfId in tpStat:
         tpStatus = ( tpStat[ p.intfId ].testPatternTx != pattern.patternNone ) or \
                    ( tpStat[ p.intfId ].testPatternRx != pattern.patternNone )
      else:
         tpStatus = False
      debugStateDirName = getDebugStateDirName( phyIntfs[ p.intfId ] )
      assert debugStateDirName, 'Invalid intfId'
      phyFeatureStatusSubSliceDir = \
            PhyFeatureCli.phyFeatureStatusSliceDir.get( debugStateDirName )
      phyStatuses = printOnePhyStatusDetail( p, xcvrStatus, nowish,
            fecStatuses.get( p.intfId ), phyTopology, phyIntfs, tpStatus,
            precodingStatuses.get( p.intfId ), phyFeatureStatusSubSliceDir, mode )
      if p.intfId in interfacePhyStatuses:
         # 40G-only ports handled lane information by creating one phy status
         # object per lane giving all phy status objects for a port the same intfId.
         # This will change to the current model of one phy status per interface
         # eventually. In the meantime, assume that phy detail handling of these
         # 40G-only ports involves screen-scraping. Only combine the output from the
         # first phy status as it is assumed the output from the next phy statuses
         # follow the one phy status per interface model. Otherwise combining those
         # would duplicate that phy's data in the model output.
         assert len( phyStatuses.phyStatuses ) >= 1
         stored = interfacePhyStatuses[ p.intfId ]
         assert len( stored.phyStatuses ) >= 1
         stored.phyStatuses[ 0 ].text += phyStatuses.phyStatuses[ 0 ].text
         continue
      interfacePhyStatuses[ p.intfId ] = phyStatuses

   ret = PhyModel.InterfacesPhyStatusesDetailed(
         interfacePhyStatuses=interfacePhyStatuses )
   return ret

phyTopologyGlobal = None
def getSysdbPhyStatuses( mode, intf, mod, detail=True, exposeInternal=False ):
   global phyTopologyGlobal

   intfs = IntfCli.Intf.getAll( mode, intf, mod, exposeInactive=detail,
                                exposeUnconnected=detail, sort=False,
                                exposeInternal=exposeInternal )

   if not intfs:
      return ( [], set(), None, {}, {} )

   if phyTopologyGlobal is None:
      mg = em.mountGroup()
      # BUG19898: allow specific PHY CliPlugins to register their own
      #           mount hooks, so they can mount specific things
      #           themselves. This is a hack -- we need to understand the
      #           underlying problem.
      for mountHook in phySpecificMountHook_.values():
         mountHook( mode, em )

      mg.close( blocking=True )
      phyTopologyGlobal = PhyStatusLib.mountPhyTopology( em )
   phyTopology = phyTopologyGlobal

   ps = []
   # Maps phy topology phy interface status full name to the associated phy config
   # object, needed by show int phy detail code.
   pc = {}
   phyIntfs = {}
   notFoundIntfNames = set()
   # Iterate over the interfaces looking for them in the phy topology.
   for phyIntf in intfs:
      intfName = phyIntf.name
      phyData = PhyStatusLib.phyConfigStatusAtPos( em.root(), phyTopology, intfName,
                                                   PhyPosition.asicPhy )
      if not phyData:
         notFoundIntfNames.add( intfName )
         continue

      phyIntfs[ intfName ] = phyIntf
      for phyIntfConfig, phyIntfStatus in phyData:
         ps.append( phyIntfStatus )
         pc[ phyIntfStatus.fullName ] = phyIntfConfig

   if intf:
      # not all interfaces specified by user are available
      for i in sorted( notFoundIntfNames, key=Arnet.intfNameKey ):
         print( "%s: displaying PHY details not supported" % i )

   # We have a bunch of PhyStatuses, and want to print out info
   # based on the name of the corresponding Ethernet interface, not
   # "Phy17".
   ps = sorted( ps, key=lambda x: Arnet.intfNameKey( x.intfId ) )

   return ( ps, set( phyIntfs ), phyTopology, phyIntfs, pc )


class ShowIntfPhy( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces phy'
   data = dict( phy=phyShowKwNode )
   handler = showInterfacesPhySummary
   allowAllSyntax = True
   moduleAtEnd = True
   cliModel = PhyModel.InterfacesPhyStatusesSummary

BasicCli.addShowCommandClass( ShowIntfPhy )

class ShowIntfPhyDetail( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces phy detail'
   data = dict( phy=phyShowKwNode,
                detail='More comprehensive output' )
   handler = showInterfacesPhyDetail
   allowAllSyntax = True
   moduleAtEnd = True
   cliModel = PhyModel.InterfacesPhyStatusesDetailed

BasicCli.addShowCommandClass( ShowIntfPhyDetail )

phySpecificDisplayFn = {}
def registerPhySpecificDisplayFn( typeName, fn ):
   registeredFn = phySpecificDisplayFn.get( typeName )
   assert not registeredFn or registeredFn == fn
   phySpecificDisplayFn[ typeName ] = fn

# A collection of CLI models that support `toPhySummaryModel` function, which
# returns an InterfacePhyStatusSummary model used by `show int phy` command. Each
# CLI model is indexed by a phyStatusTypeStr.
phySummaryModelsCol = {}
def registerPhySummaryModels( phyStatusTypeStr, cliModelClass ):
   registeredModel = phySummaryModelsCol.get( phyStatusTypeStr )
   assert not registeredModel or registeredModel == cliModelClass
   phySummaryModelsCol[ phyStatusTypeStr ] = cliModelClass

# A collection of lists of CLI models that support `renderPhyDetail` function, which
# prints the output of `show int phy det` command. Each list of CLI models is indexed
# by a phyStatusTypeStr. (aid/2574)
phyDetailModelsCol = {}
def registerPhyDetailModels( phyStatusTypeStr, cliModelClasses ):
   phyDetailModelsCol[ phyStatusTypeStr ] = cliModelClasses

# A collection of phyTypes with its managingAgentName that support phy detail
# via L1Topology infrastructure. The collection is indexed by a phyTypeStr
_phyDetailL1TopologySupportedPhys = defaultdict( list )
def registerPhyDetailL1TopologySupport( phyTypeStr, managingAgentName ):
   _phyDetailL1TopologySupportedPhys[ phyTypeStr ].append( managingAgentName )

# A collection of phy CLI models that support `renderPhyDetail` function, which
# prints the output of `show int phy det` command. Each of the CLI model is indexed
# by (phyType, phyScope).
_phyCliModelCol = {}

registerPhyCliModel = _phyCliModelCol.__setitem__

_getPhyCliModelFromCol = _phyCliModelCol.get

# A collection of chip CLI models that support `renderPhyDetail` function, which
# prints the output of `show int phy det` command. Each of the CLI model is indexed
# by chipType.
_chipCliModelCol = {}

registerChipCliModel = _chipCliModelCol.__setitem__

_getChipCliModelFromCol = _chipCliModelCol.get

# A collection of lists of CLI models that support `render` function, which prints
# the output of `show int phy diag err histo` command. Each list of CLI models is
# indexed by a phyStatusTypeStr.
fecHistogramModelsCol = {}
def registerFecHistogramModels( phyStatusTypeStr, cliModelClasses ):
   fecHistogramModelsCol[ phyStatusTypeStr ] = cliModelClasses

# Set of phy status tac typenames for which the old method of generating the initial
# raw phy status should be included.
includeRawPhyStatusForPhyDetailCol = set()
def includeRawPhyStatusForPhyDetail( phyStatusTypeStr ):
   """
   Add the old method of generating the initial raw phy status when information from
   the phy tac object with the specified phyStatusTypeStr tac typename is used.
   """
   includeRawPhyStatusForPhyDetailCol.add( phyStatusTypeStr )

phySpecificMountHook_ = {}
def registerPhySpecificMountHook( typeName, fn ):
   registeredFn = phySpecificMountHook_.get( typeName )
   assert not registeredFn or registeredFn == fn
   phySpecificMountHook_[ typeName ] = fn

# Only the external phys that want to support debugStateSnapshot should register
# their phy status types here.
debugStateSupportedPhyTypes = []
def registerDebugStateSupportedPhyStatusType( typeName ):
   debugStateSupportedPhyTypes.append( typeName )

# Returns all phyCoherentStatus
coherentConfigDirGlobal = None
def getAllPhyCoherentStatus():
   global coherentConfigDirGlobal
   if coherentConfigDirGlobal is None:
      mg = em.mountGroup()
      coherentConfigDir = mg.mount(
         'hardware/archer/phy/config/data/coherent/slice', 'Tac::Dir', 'ri' )
      mg.close( blocking=True )
      coherentConfigDirGlobal = coherentConfigDir
   else:
      coherentConfigDir = coherentConfigDirGlobal

   allPhyCoherentStatus = {}
   if coherentStatusDir.get( 'slice' ) is not None:
      for sliceId, linecard in coherentStatusDir[ 'slice' ].items():
         if sliceId not in coherentConfigDir:
            continue
         phyCoherentConfig = coherentConfigDir[ sliceId ]
         for intf, phyCoherentStatus in linecard.phyCoherentStatus.items():
            if( phyCoherentConfig.generation.valid and
                  phyCoherentStatus.generation == phyCoherentConfig.generation ):
               allPhyCoherentStatus[ intf ] = phyCoherentStatus
   return allPhyCoherentStatus

# Returns a string for all the given available modulation capabilities
# Input : Phy::Coherent::ModulationCapabilities, sysCap
# When sysCap is True (for show int hardware default) do not print (default)
# Output : String
def getModulationCapabilitiesStr( modCaps, sysCap ):
   modEnum = Tac.Type( 'Phy::Coherent::Modulation' )
   modCapsStr = []
   if sysCap:
      defaultStr = ''
   else:
      defaultStr = '(default)'
   if modCaps.modulationDpQpsk:
      caps = 'DP-QPSK'
      if modCaps.defaultModulation == modEnum.modulationDpQpsk:
         caps += defaultStr
      modCapsStr.append( caps )
   if modCaps.modulation8Qam:
      caps = '8QAM'
      if modCaps.defaultModulation == modEnum.modulation8Qam:
         caps += defaultStr
      modCapsStr.append( caps )
   if modCaps.modulation16Qam:
      caps = '16QAM'
      if modCaps.defaultModulation == modEnum.modulation16Qam:
         caps += defaultStr
      modCapsStr.append( caps )

   if modCapsStr:
      return ','.join( modCapsStr )
   return 'none'

def printInterfaceModulationCapabilities( allPhyCoherentStatus, intfName,
      fmt, label, sysCap=False ):
   fmtStr = getInterfaceModulationCapabilities( allPhyCoherentStatus,
                                                intfName, label,
                                                sysCap )
   if fmtStr:
      print( fmt % ( label, fmtStr ) )

def getInterfaceModulationCapabilities( allPhyCoherentStatus, intfName,
      label, sysCap=False ):
   isCoherentIntf = intfType( *checkCoherentIntf( intfName ) )
   modCaps = ''
   # Do not print "Modulation" for DCO slots, as the modulation capabilities
   # are really the capabilities of the inserted transciver.
   if ( ( not( isCoherentIntf.coherentCapable and sysCap ) ) and
        intfName in allPhyCoherentStatus ):
      modCaps = getModulationCapabilitiesStr(
         allPhyCoherentStatus[ intfName ].modulationCapabilities, sysCap )
   return modCaps

EthIntfCli.registerInterfaceCapabilities( 'Modulation', getAllPhyCoherentStatus,
   printInterfaceModulationCapabilities, getInterfaceModulationCapabilities )

# Return a dictionary indexed by modulation of the line rates which
# that modulation supports
def getLineRateCapabilitiesDict( lineRateCaps, sysCap ):
   lineRateLabelMap = {
      LineRateEnum.lineRate100g: '100G',
      LineRateEnum.lineRate200g: '200G',
      LineRateEnum.lineRate300g: '300G',
      LineRateEnum.lineRate400g: '400G'}
   modLabelMap = {
      ModEnum.modulation16Qam : "16QAM",
      ModEnum.modulation8Qam : "8QAM",
      ModEnum.modulationDpQpsk : "DP-QPSK" }
   if sysCap:
      defaultStr = ''
   else:
      defaultStr = '(default)'
   modToLineRate = {}
   for mod in lineRateCaps:
      lineRateCapsForMod = lineRateCaps[ mod ]
      for cap in lineRateCapsForMod.attributes:
         if cap == 'defaultLineRate':
            continue
         if getattr( lineRateCapsForMod, cap ):
            modStr = modLabelMap[ mod ]
            lineRateStr = lineRateLabelMap[ cap ]
            if lineRateCapsForMod.defaultLineRate == cap:
               lineRateStr += defaultStr
            if modStr not in modToLineRate:
               modToLineRate[ modStr ] = []
            modToLineRate[ modStr ].append( lineRateStr )
   return modToLineRate

def getInterfaceLineRateCapabilities( allPhyCoherentStatus, intfName,
                                      label, sysCap=False ):
   isCoherentIntf = intfType( *checkCoherentIntf( intfName ) )
   lrCaps = {}
   # Do not print "LineRate" for DCO slots, as the line rate capabilities
   # are really the capabilities of the inserted transceiver.
   if ( not( isCoherentIntf.coherentCapable and sysCap ) and
        intfName in allPhyCoherentStatus ):
      lrCaps = getLineRateCapabilitiesDict(
         allPhyCoherentStatus[ intfName ].lineRateCapabilities, sysCap )
   return lrCaps

EthIntfCli.registerInterfaceCapabilities( 'LineRate', getAllPhyCoherentStatus,
   None, getInterfaceLineRateCapabilities )

# Check to see if an interface is coherent or coherentCapable
# It is a coherent when the slot is cfpxSupported
# It is a coherentCapable interface if it is a DCO slot
# with a coherent trasceiver [ Cfpx, Cfp2Dco ] or without a transceiver
# To check for the transceiver type we look at the narrowBand, it is
# set to true for cfpx and cfp2Dco transceievers only
# These 2 attributes will also help in rejecting coherent FEC commands on a
# cfp2 slot with a non-coherent transceiver.
def checkCoherentSlotCaps( xcvrStatus ):
   coherent = False
   coherentCapable = False
   if hasattr( xcvrStatus, 'cfp2SlotCapabilities' ):
      slotCaps = xcvrStatus.cfp2SlotCapabilities
      # Today, when a CFP2 slot supports CFPX modules, those are the only modules it
      # supports. If we ever build a switch that supports a non-coherent and a CFPX
      # transceiver, this code will have to change.
      if slotCaps.cfpxSupported:
         coherent = True
      if xcvrStatus.presence == 'xcvrPresent':
         if slotCaps.cfp2DcoSupported and xcvrStatus.narrowBand:
            coherentCapable = True
      elif slotCaps.cfp2DcoSupported:
         coherentCapable = True
   return ( coherent, coherentCapable )

# Check to see if an interface is supporting CMIS coherent FECs.
# It's coherent if the slot is osfp/qsfpDd and the module is
# coherentDetect (ZR).
def checkCmisCoherentSlotCaps( xcvrStatus, phyCoherentStatusValid ):
   coherent = False
   coherentCapable = False
   if xcvrStatus.xcvrType in [ 'osfp', 'qsfpDd' ]:
      if xcvrStatus.presence == 'xcvrPresent':
         coherent = coherentCapable = ( phyCoherentStatusValid and
                                        ( xcvrStatus.opticalDetect ==
                                          OpticalDetect.coherentDetect ) )
      else:
         coherentCapable = True
   return ( coherent, coherentCapable )

# This function is used to check whether we have a coherent module or coherent
# capable interface.
# Note: we have 2 groups of coherent modules: CFP2 form factor, and CMIS compliant
# modules (OSFP/QSFP-DD). CFP2 coherent capability is based on inventory (see
# checkCoherentSlotCaps above), while OSFP/QSFP-DD slots are always capable to host
# coherent modules and only these modules should have coherent capability.
# Non-coherent CMIS modules will get rejected.
def checkCoherentIntf( intf=None, checkCmisSlots=False ):
   allCoherentStatus = getAllPhyCoherentStatus()
   # pylint: disable=len-as-condition
   if len( archerXcvrStatusDir[ "all" ] ):
      for intfId, xcvrStatus in archerXcvrStatusDir.get( 'all' ).items():
         # check if the primary interface of intf is present in the directory
         # Don't check if the phyCoherentStatus object exists, since it may not
         # be present on a interface with no transceiver ( DCO slot )
         if intf[ :-2 ] == intfId:
            if checkCmisSlots:
               return checkCmisCoherentSlotCaps( xcvrStatus,
                                                 intf in allCoherentStatus )
            else:
               return checkCoherentSlotCaps( xcvrStatus )
   # Note that on modular systems there might be a management interface that shows
   # up under "xcvr/status/all". This means that it is possible that both the "all"
   # and "slice" collections of statuses can be populated at the same time. To
   # address this we make it so the modular check lives in an independent "if"
   # statement (as opposed to using an "elif" statement).
   if len( archerXcvrStatusDir[ "slice" ] ):
      for linecard in archerXcvrStatusDir.get( 'slice' ).values():
         for intfId, xcvrStatus in linecard.items():
            # check if the primary interface of intf is present in the directory
            # and check if the phyCoherentStatus object exists for that interface
            if intf.count( '/' ) > 1:
               # Ex : for intf Et4/4/1 return Et4/4
               # for intf Et5/3 return Et5/3
               intfToCheck = intf[ :-2 ]
            else:
               intfToCheck = intf
            if intfToCheck == intfId:
               if checkCmisSlots:
                  return checkCmisCoherentSlotCaps( xcvrStatus,
                                                    intf in allCoherentStatus )
               elif intf in allCoherentStatus:
                  return checkCoherentSlotCaps( xcvrStatus )
   return ( False, False )

# When default/no is specified, we configure the default FEC only when the config
# object already exists in Sysdb else this command has no affect
def configCoherentFec( intf=None, encoding=None, noOrDefBool=False ):
   from CliPlugin.PhyConfigCli import phyCliConfigDir
   cliCoherentDir = phyCliConfigDir.phyCliCoherentConfig
   cliConfig = cliCoherentDir.get( intf )
   if cliConfig is None and noOrDefBool:
      return False
   elif noOrDefBool:
      cliConfig.fecConfig = 'coherentFecEncodingCfgDefault'
      return False
   else:
      if cliConfig is None:
         cliConfig = cliCoherentDir.newMember( intf )
      cliConfig.fecConfig = encoding
      return True

# Checks if the interface is a coherent type, if so, it makes relevant call
# to the config function
# During start-up config parse time we always accept the commands i.e., when
# lookupPhysical() is False
def coherentFecDecisionMaker( intf=None, encoding=None, noOrDefault=False ):
   noOrDefBool = noOrDefault is True or str( noOrDefault ).lower() == 'default'
   isCoherentFecConfigured = False
   isCoherentIntf = intfType( False, False )
   cmisCoherentFecs = { 'coherentFecEncodingCfgCfec', 'coherentFecEncodingCfgOfec' }
   if intf.lookupPhysical():
      if encoding is None:
         # If no encoding was specified, we should accept the interface regardless
         # of xcvrType
         isCfp2CoherentIntf = intfType( *checkCoherentIntf( intf.name ) )
         isCmisCoherentIntf = intfType( *checkCoherentIntf( intf.name,
                                                            checkCmisSlots=True ) )
         # Since these tuples are mutually exclusive, we can OR them together
         isCoherentIntf = intfType( ( isCfp2CoherentIntf.coherent or
                                         isCmisCoherentIntf.coherent ),
                                    ( isCfp2CoherentIntf.coherentCapable or
                                         isCmisCoherentIntf.coherentCapable ) )
      else:
         isCoherentIntf = intfType( *checkCoherentIntf(
            intf.name,
            checkCmisSlots=( encoding in cmisCoherentFecs ) ) )
   else:
      isCoherentIntf = intfType( False, True )
   if isCoherentIntf.coherent or isCoherentIntf.coherentCapable:
      isCoherentFecConfigured = configCoherentFec( intf.name, encoding, noOrDefBool )
   elif not noOrDefBool:
      errorMsg = (
         'Error-correction encoding %s is unsupported for interface %s.' %
         ( PhyCliSave.coherentFecEnumToCmd( encoding ), intf.name ) )
      if ethPhyIntfDefaultConfigDir.xcvrCapVerificationBypass.errorCorrection:
         intf.mode_.addWarning( errorMsg )
         prompt = "Do you wish to proceed with this command? [y/N]"
         if BasicCliUtil.confirm( intf.mode_, prompt, answerForReturn=False ):
            isCoherentFecConfigured = configCoherentFec( intf.name,
                                                         encoding, noOrDefBool )
      else:
         intf.mode_.addError( errorMsg )
   return isCoherentFecConfigured

# Register the coherent function
EthIntfCli.registerErrorCorrectionCfgFn( 'coherent', coherentFecDecisionMaker )

# Preserve the order of the dictionary
coherentFecToLbls = OrderedDict( [
   ( 'fecEncodingG709', '  RS G.709:    ' ),
   ( 'fecEncodingHd7', '  SC 7%:       ' ),
   ( 'fecEncodingSd15', '  SD 15%:      ' ),
   ( 'fecEncodingSd20', '  SD 20%:      ' ),
   ( 'fecEncodingSd25', '  SD 25%:      ' ),
   ( 'fecEncodingSd25Bch', '  SD 25% BCH:  ' ),
   ( 'fecEncodingCfec', '  C-FEC        ' ),
   ( 'fecEncodingOfec', '  O-FEC        ' ),
   ] )

def isAnyCoherentFecSupported( intfName, checkCmisSlots=False ):
   coherent, coherentCapable = checkCoherentIntf( intfName,
                                                  checkCmisSlots=checkCmisSlots )
   return coherent or coherentCapable

def isGivenCoherentFecSupp( fecCaps, encoding ):
   for fecSet in fecCaps.values():
      if getattr( fecSet, encoding ):
         return True
   return False

def getCoherentFecCapabilitiesStr( intf, fecCaps, encoding, sysCap ):
   modulations = [ ]
   for modulationType, fecSet in fecCaps.items():
      if encoding in coherentFecToLbls:
         if getattr( fecSet, encoding ):
            modulation = PhyCliSave.modulationEnumToCmd( modulationType ).upper()
            # Prepend "coherent" to encoding and capitalize
            # F of "fec" for right comparison. See: PhyEee/PhyCoherentTypes.tac
            # Ex: if encoding is "fecEncodingSd25" make it "coherentFecEncodingSd25"
            if fecSet.defaultCoherentFec == "coherent" + \
                                            encoding[ 0 ].upper() + encoding[ 1: ]:
               modulations.append( ( modulation, True ) )
            else:
               modulations.append( ( modulation, False ) )
   modulations.sort( reverse=True )
   # Do not add (default) if this method is called by show int caps default
   capStr = ','.join( '%s%s' % (
                      modulation, '(default)' if not sysCap and default else '' )
                      for modulation, default in modulations )
   return capStr

def getInterfaceCoherentFecCapabilities( allPhyCoherentStatus, intf, sysCap ):
   fecDict = {}
   if intf.name in allPhyCoherentStatus:
      isCoherentIntf = intfType( *checkCoherentIntf( intf.name ) )
      # Do not show coherent default capabilities for DCO slots
      # as they are the capabilities of the inserted transceiver
      if not( isCoherentIntf.coherentCapable and sysCap ):
         for encoding, lbl in coherentFecToLbls.items():
            if isGivenCoherentFecSupp(
                  allPhyCoherentStatus [ intf.name ].fecCapabilities, encoding ):
               tmpStr = getCoherentFecCapabilitiesStr( intf,
                                 allPhyCoherentStatus[ intf.name ].fecCapabilities,
                                 encoding, sysCap )
               fecDict.update( { lbl: tmpStr } )
   return fecDict

def printInterfaceCoherentFecCapabilities( allPhyCoherentStatus, intf, fmt, sysCap ):
   coherentFecPrinted = False
   if intf.name in allPhyCoherentStatus:
      isCoherentIntf = intfType( *checkCoherentIntf( intf.name ) )
      # Do not show coherent default capabilities for DCO slots
      # as they are the capabilities of the inserted transceiver
      if not( isCoherentIntf.coherentCapable and sysCap ):
         for encoding, lbl in coherentFecToLbls.items():
            if isGivenCoherentFecSupp(
                  allPhyCoherentStatus [ intf.name ].fecCapabilities, encoding ):
               coherentFecPrinted = True
               print( fmt % ( lbl, getCoherentFecCapabilitiesStr( intf,
                  allPhyCoherentStatus[ intf.name ].fecCapabilities, encoding,
                  sysCap ) ) )
   return coherentFecPrinted

EthIntfCli.registerErrorCorrectionShCapFn( 'coherent', isAnyCoherentFecSupported,
      isGivenCoherentFecSupp, printInterfaceCoherentFecCapabilities,
      getInterfaceCoherentFecCapabilities )

def showCoherentErrorCorrectionEncoding( intf, errCorrModel ):
   if intf and errCorrModel:
      errCorrModel._coherentIntf = True # pylint:disable-msg=W0212
      enumToFecStr = {
            "coherentFecEncodingG709" : "reedSolomonG709",
            "coherentFecEncodingHd7" : "staircaseOverhead7Percent",
            "coherentFecEncodingSd15" : "softDecisionOverhead15Percent",
            "coherentFecEncodingSd20" : "softDecisionOverhead20Percent",
            "coherentFecEncodingSd25" : "softDecisionOverhead25Percent",
            "coherentFecEncodingSd25Bch" : "softDecisionOverhead25PercentBch",
            "coherentFecEncodingOfec" : "openFec",
            "coherentFecEncodingCfec" : "concatenatedFec",
            "coherentFecEncodingNone" : "none",
            "coherentFecEncodingCfgDefault" : "default",
            "coherentFecEncodingCfgG709" : "reedSolomonG709",
            "coherentFecEncodingCfgHd7" : "staircaseOverhead7Percent",
            "coherentFecEncodingCfgSd15" : "softDecisionOverhead15Percent",
            "coherentFecEncodingCfgSd20" : "softDecisionOverhead20Percent",
            "coherentFecEncodingCfgSd25" : "softDecisionOverhead25Percent",
            "coherentFecEncodingCfgSd25Bch" : "softDecisionOverhead25PercentBch",
            "coherentFecEncodingCfgCfec" : "concatenatedFec",
            "coherentFecEncodingCfgOfec" : "openFec",
            }
      from CliPlugin.PhyConfigCli import phyCliConfigDir
      cliCoherentDir = phyCliConfigDir.phyCliCoherentConfig
      allCoherentStatus = getAllPhyCoherentStatus()
      if intf.name in cliCoherentDir and cliCoherentDir[ intf.name ].fecConfig:
         errCorrModel.coherentEncodingConfigured = enumToFecStr[
               cliCoherentDir[ intf.name ].fecConfig ]
         errCorrModel._noCoherentConfiguration = False # pylint:disable-msg=W0212
      else:
         errCorrModel.coherentEncodingConfigured = "default"
         errCorrModel._noCoherentConfiguration = True # pylint:disable-msg=W0212

      if intf.name in allCoherentStatus:
         cs = allCoherentStatus[ intf.name ]

         # Fill in the coherentFec Status
         errCorrModel.errCorrEncodingStatus = enumToFecStr[ cs.fecStatus ]

         # Fill in the coherentFec capabilities
         fecCaps = cs.fecCapabilities
         operModulation = cs.modulationStatus
         if operModulation != "modulationNone":
            cFecSet = fecCaps[ operModulation ]
            errCorrModel.reedSolomonG709Available = cFecSet.fecEncodingG709
            errCorrModel.staircaseOverhead7PercentAvailable = cFecSet.fecEncodingHd7
            errCorrModel.softDecisionOverhead15PercentAvailable = \
                                                cFecSet.fecEncodingSd15
            errCorrModel.softDecisionOverhead20PercentAvailable = \
                                                cFecSet.fecEncodingSd20
            errCorrModel.softDecisionOverhead25PercentAvailable = \
                                                cFecSet.fecEncodingSd25
            errCorrModel.softDecisionOverhead25PercentBchAvailable = \
                                                cFecSet.fecEncodingSd25Bch
            errCorrModel.openFecAvailable = cFecSet.fecEncodingOfec
            errCorrModel.concatenatedFecAvailable = cFecSet.fecEncodingCfec
      else:
         # For a DCO slot with no transceiver, populate these to None and False
         errCorrModel.errCorrEncodingStatus = "none"
         errCorrModel.reedSolomonG709Available = False
         errCorrModel.staircaseOverhead7PercentAvailable = False
         errCorrModel.softDecisionOverhead15PercentAvailable = False
         errCorrModel.softDecisionOverhead20PercentAvailable = False
         errCorrModel.softDecisionOverhead25PercentAvailable = False
         errCorrModel.softDecisionOverhead25PercentBchAvailable = False
         errCorrModel.openFecAvailable = False
         errCorrModel.concatenatedFecAvailable = False

      # Initialize clientFec attr to False to work around adding a degrade func
      errCorrModel.fecEncodingConfigDisabled = False
      errCorrModel.fecEncodingConfigReedSolomon544Enabled = False
      errCorrModel.fecEncodingConfigReedSolomonEnabled = False
      errCorrModel.fecEncodingConfigFireCodeEnabled = False
      errCorrModel.fecEncodingReedSolomon544Available = False
      errCorrModel.fecEncodingReedSolomonAvailable = False
      errCorrModel.fecEncodingFireCodeAvailable = False

   return errCorrModel

EthIntfCli.registerErrorCorrectionShErrFn( 'coherent',
   showCoherentErrorCorrectionEncoding )

#----------------------------------------------------------------------------
#
# "show interfaces [ <interface> ] phy modulation"
#
#----------------------------------------------------------------------------

phyCliConfigDirGlobal = None
# Convert Phy::Coherent::Modulation to an enum value that the CAPI
# model understands
modToEnum = {
   "modulation16Qam" : "qam16",
   "modulation8Qam" : "qam8",
   "modulationDpQpsk" : "dpQpsk",
   "modulationNone" : "none",
   "modulationDefault" : "default" }

def intfModulationModel( cs, intfName, phyCliConfigDir ):
   modulation = PhyModulationModel.InterfacePhyModulation()

   # If the config exists, use that otherwise use the modulationStatus.
   # Note that if no config exists, modulationConfigured will be
   # modulationStatus in json but the CLI will print 'default'
   if intfName in phyCliConfigDir.phyCliCoherentConfig and \
      phyCliConfigDir.phyCliCoherentConfig[ intfName ].modulation \
      != 'modulationDefault':
      modulation.modulationConfigured = modToEnum[
         phyCliConfigDir.phyCliCoherentConfig[ intfName ].modulation ]
      modulation._noConfiguration = False      # pylint: disable-msg=W0212
   else:
      modulation._noConfiguration = True       # pylint: disable-msg=W0212
      modulation.modulationConfigured = modToEnum[ cs.modulationStatus ]

   # Fill in the modulationStatus
   modulation.modulationStatus = modToEnum[ cs.modulationStatus ]

   # Fill in the operational modulation capabilities
   modCaps = cs.operModulationCapabilities
   _availableModulations = \
                           PhyModulationModel.InterfaceAvailableModulations()
   # Only fill in the capabilities if they are True. Don't unnecessarily
   # fill the capabilities that are False
   if modCaps.modulation16Qam:
      _availableModulations.qam16 = True
   if modCaps.modulation8Qam:
      _availableModulations.qam8 = True
   if modCaps.modulationDpQpsk:
      _availableModulations.dpQpsk = True
   modulation.availableModulations = _availableModulations
   return modulation

def intfLineRateModel( cs, intfName, phyCliConfigDir ):
   # Convert Phy::Coherent::LineRate to an enum value that the CAPI
   # model understands
   lrToEnum = {
         "lineRate100g" : "100",
         "lineRate200g" : "200",
         "lineRate300g" : "300",
         "lineRate400g" : "400",
         "lineRateNone" : "none",
         "lineRateDefault" : "default" }

   lineRate = PhyLineRateModel.InterfacePhyLineRate()
   # For platforms which do not support setting line rate, the lineRateStatus
   # will be lineRateDefault. The CLI needs a valid lineRateStatus, so set it
   # to none
   lrStatus = cs.lineRateStatus
   if lrStatus == 'lineRateDefault':
      lrStatus = 'lineRateNone'

   # If the config exists, use that otherwise use the lineRateStatus.
   # Note that if no config exists, lineRateConfigured will be
   # lineRateStatus in json but the CLI will print 'default'
   if intfName in phyCliConfigDir.phyCliCoherentConfig and \
      phyCliConfigDir.phyCliCoherentConfig[ intfName ].lineRate \
      != 'lineRateDefault':
      lineRate.lineRateConfigured = lrToEnum[
         phyCliConfigDir.phyCliCoherentConfig[ intfName ].lineRate ]
      lineRate._noConfiguration = False      # pylint: disable-msg=W0212
   else:
      lineRate._noConfiguration = True       # pylint: disable-msg=W0212
      lineRate.lineRateConfigured = lrToEnum[ lrStatus ]

   # Fill in the modulationStatus
   lineRate.lineRateStatus = lrToEnum[ lrStatus ]

   lineRate.modulationStatus = modToEnum[ cs.modulationStatus ]

   # Fill in the operational modulation capabilities
   lineRateCaps = cs.lineRateCapabilities
   _availableLineRates = \
                         PhyLineRateModel.InterfaceAvailableLineRates()
   # Only fill in the capabilities if they are True. Don't unnecessarily
   # fill the capabilities that are False
   if cs.modulationStatus in lineRateCaps:
      if lineRateCaps[ cs.modulationStatus ].lineRate100g:
         _availableLineRates.lr100g = True
      if lineRateCaps[ cs.modulationStatus ].lineRate200g:
         _availableLineRates.lr200g = True
      if lineRateCaps[ cs.modulationStatus ].lineRate300g:
         _availableLineRates.lr300g = True
      if lineRateCaps[ cs.modulationStatus ].lineRate400g:
         _availableLineRates.lr400g = True
   lineRate.availableLineRates = _availableLineRates
   return lineRate

def showInterfacesPhyModulation( mode, args ):
   global phyCliConfigDirGlobal
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   rate = args.get( 'rate', '' )

   if rate:
      interfacesPhyModel = PhyLineRateModel.InterfacesPhyLineRate()
   else:
      interfacesPhyModel = PhyModulationModel.InterfacesPhyModulation()
   intfs = IntfCli.Intf.getAll( mode, intf, mod )

   # No intfs - return the default model
   if not intfs:
      return interfacesPhyModel

   if phyCliConfigDirGlobal is None:
      mg = em.mountGroup()
      # Mount the phyCliConfig and phyCoherentStatus
      phyCliConfigDir = mg.mount( 'hardware/phy/config/cli',
                                  'Hardware::Phy::PhyCliConfigDir', 'wi' )
      mg.close( blocking=True )
      phyCliConfigDirGlobal = phyCliConfigDir
   else:
      phyCliConfigDir = phyCliConfigDirGlobal

   # For now only modular systems mount the coherent status
   allCoherentStatus = getAllPhyCoherentStatus()

   for i in intfs:
      # Make sure we have information on the interface 'i'
      if i.name in allCoherentStatus:
         cs = allCoherentStatus[ i.name ]
         if rate:
            lineRate = intfLineRateModel( cs, i.name, phyCliConfigDir )
            # Finally, create an object for this interface
            interfacesPhyModel.interfaceLineRate[ i.name ] = lineRate
         else:
            modulation = intfModulationModel( cs, i.name, phyCliConfigDir )
            # Finally, create an object for this interface
            interfacesPhyModel.interfaceModulation[ i.name ] = modulation

   return interfacesPhyModel

class ShowIntfPhyModulation( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces phy modulation [rate]'
   data = dict( phy=phyShowKwNode,
                modulation='Display coherent modulation information',
                rate='Display coherent line rate information' )
   handler = showInterfacesPhyModulation
   moduleAtEnd = True
   cliModel = PhyModulationModel.InterfacesPhyCoherentBase

BasicCli.addShowCommandClass( ShowIntfPhyModulation )

#--------------------------------------------------------------------------------
#
# "show interfaces [ <interface> ] phy diag test-pattern [ counters ] [ detail ]"
#
#--------------------------------------------------------------------------------

def getCapabilities( capabilities ):
   pattern = Tac.Type( "Hardware::Phy::TestPattern" )
   capStr = []
   supportedPattern = ""
   testPatternEnumToStr = [
                           ( pattern.prbs7,  "7" ),
                           ( pattern.prbs9,  "9" ),
                           ( pattern.prbs11, "11" ),
                           ( pattern.prbs13, "13" ),
                           ( pattern.prbs15, "15" ),
                           ( pattern.prbs23, "23" ),
                           ( pattern.prbs31, "31" ),
                           ( pattern.prbs49, "49" ),
                           ( pattern.prbs58, "58" ),
                           ( pattern.prbs63, "63" ),
                          ]
   # iterating based on the order provided by testPatternEnumToStr dict,
   # so that the capStr would be in sorted order.
   for testPatternNum, outputStr in testPatternEnumToStr:
      if testPatternNum in capabilities and capabilities[ testPatternNum ]:
         capStr.append( outputStr )
   if capStr:
      supportedPattern = "PRBS " + ','.join( capStr )
   return supportedPattern

def showInterfacesTestPatternStatus( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )

   intfsTestPatData = PhyStatusModel.TestPatternCounters()
   intfs = IntfCli.Intf.getAll( mode, intf, mod )

   allData = getAllTestPatternStatus( intfs ) # returns ( status, caps )
   allTestPatStatus = allData[0]
   for ( intf, testPatSts ) in allTestPatStatus.items():
      testPatData = PhyStatusModel.IntfTestPatternCounters()
      laneSt = testPatSts.testPatternLaneStatus
      for st in laneSt.values():
         laneTestPatStat = PhyStatusModel.LaneTestPatternCounters(). \
                                                         toModel( st )
         testPatData.lanes[ st.phyLane ] = laneTestPatStat
      intfsTestPatData.interfaces[ intf ] = testPatData
   return intfsTestPatData

def showInterfacesTestPatternDetail( mode, args ):
   global phyCliConfigDirGlobal
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )

   testPattern = Tac.Type( "Hardware::Phy::TestPattern" )
   if phyCliConfigDirGlobal is None:
      mg = em.mountGroup()
      phyCliConfigDir = mg.mount( 'hardware/phy/config/cli',
                                  'Hardware::Phy::PhyCliConfigDir', 'ri' )
      mg.close( blocking=True )
      phyCliConfigDirGlobal = phyCliConfigDir
   else:
      phyCliConfigDir = phyCliConfigDirGlobal

   intfsTestPatternDetail = PhyStatusModel.TestPatternDetail()
   intfs = IntfCli.Intf.getAll( mode, intf, mod )
   allData = getAllTestPatternStatus( intfs ) # returns ( status, caps )
   allTestPatStatus = allData[0]
   for ( intf, testPatternStatus ) in allTestPatStatus.items():
      if not testPatternStatus.testPatternLaneStatus:
         continue
      testPatternDetail = PhyStatusModel.IntfTestPatternDetail()
      lastClear = testPatternStatus.lastClear
      testPatternDetail.lastClear = Ark.switchTimeToUtc( lastClear )
      if intf in phyCliConfigDir.phyCliConfig:
         testPatternDetail.operationalTestPattern = phyCliConfigDir. \
                                                    phyCliConfig[ intf ]. \
                                                    testPatternRx
      else:
         testPatternDetail.operationalTestPattern = testPattern.patternNone
      laneStatus = testPatternStatus.testPatternLaneStatus
      for status in laneStatus.values():
         laneTestPatternDetail = PhyStatusModel.LaneTestPatternDetail(). \
                                                   toModel( status, lastClear )
         testPatternDetail.lanes[ status.phyLane ] = laneTestPatternDetail
      intfsTestPatternDetail.interfaces[ intf ] = testPatternDetail
   return intfsTestPatternDetail

def showInterfacesTestPattern( mode, args ):
   global phyCliConfigDirGlobal
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )

   intfsTestPattern = PhyStatusModel.TestPatternStatus()
   intfs = IntfCli.Intf.getAll( mode, intf, mod )
   testPattern = Tac.Type( "Hardware::Phy::TestPattern" )

   if phyCliConfigDirGlobal is None:
      mg = em.mountGroup()
      phyCliConfigDir = mg.mount( 'hardware/phy/config/cli',
                                  'Hardware::Phy::PhyCliConfigDir', 'ri' )
      mg.close( blocking=True )
      phyCliConfigDirGlobal = phyCliConfigDir
   else:
      phyCliConfigDir = phyCliConfigDirGlobal
   allTestPatternStatus, allTestPatternCaps = getAllTestPatternStatus( intfs )
   for ( intf, capabilities ) in allTestPatternCaps.items():
      testPatternData = PhyStatusModel.IntfTestPatternStatus()
      tx = PhyStatusModel.TestPatternMode()
      rx = PhyStatusModel.TestPatternMode()
      if intf in allTestPatternStatus:
         tx.operational = allTestPatternStatus[ intf ].testPatternTx
         rx.operational = allTestPatternStatus[ intf ].testPatternRx
      else:
         tx.operational = testPattern.patternNone
         rx.operational = testPattern.patternNone
      if intf in phyCliConfigDir.phyCliConfig:
         tx.configured = phyCliConfigDir.phyCliConfig[ intf ].testPatternTx
         rx.configured = phyCliConfigDir.phyCliConfig[ intf ].testPatternRx
      else:
         tx.configured = testPattern.patternNone
         rx.configured = testPattern.patternNone
      testPatternData.tx = tx
      testPatternData.rx = rx
      testPatternData.capabilities = getCapabilities( capabilities.capability )
      intfsTestPattern.interfaces[ intf ] = testPatternData
   return intfsTestPattern


diagShowKw = CliMatcher.KeywordMatcher( 'diag',
                                        helpdesc='Diagnostic commands' )

errorCorrectionShowKw = CliMatcher.KeywordMatcher( 'error-correction',
                                   helpdesc='Forward error correction' )
histogramShowKw = CliMatcher.KeywordMatcher( 'histogram',
                             helpdesc='Histogram showing symbol errors' )
testShowKw = CliMatcher.KeywordMatcher( 'test',
                        helpdesc='Show interface in test mode' )

patternShowKw = CliMatcher.KeywordMatcher( 'pattern',
                                           helpdesc='Pattern used for testing' )

countersShowKw = CliMatcher.KeywordMatcher( 'counters',
                                            helpdesc='Test pattern error counters' )

class ShowIntfPhyDiagPatternCountersDetail( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces phy diag test pattern counters detail'
   data = dict( phy=phyShowKwNode,
                diag=diagShowKw,
                test=testShowKw,
                pattern=patternShowKw,
                counters=countersShowKw,
                detail='Detailed test pattern results' )
   handler = showInterfacesTestPatternDetail
   moduleAtEnd = True
   cliModel = PhyStatusModel.TestPatternDetail

BasicCli.addShowCommandClass( ShowIntfPhyDiagPatternCountersDetail )

class ShowIntfPhyDiagPatternCounters( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces phy diag test pattern counters'
   data = dict( phy=phyShowKwNode,
                diag=diagShowKw,
                test=testShowKw,
                pattern=patternShowKw,
                counters=countersShowKw )
   handler = showInterfacesTestPatternStatus
   moduleAtEnd = True
   cliModel = PhyStatusModel.TestPatternCounters

BasicCli.addShowCommandClass( ShowIntfPhyDiagPatternCounters )

class ShowIntfPhyDiagPattern( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces phy diag test pattern'
   data = dict( phy=phyShowKwNode,
                diag=diagShowKw,
                test=testShowKw,
                pattern=patternShowKw )
   handler = showInterfacesTestPattern
   moduleAtEnd = True
   cliModel = PhyStatusModel.TestPatternStatus

BasicCli.addShowCommandClass( ShowIntfPhyDiagPattern )

#--------------------------------------------------------------------------------
#
# "show interfaces [ <interface> ] phy diag error-correction histogram"
#
#--------------------------------------------------------------------------------

def getOneIntfFecHistogramStates( phyStatus, phyTopology, intfStatus ):
   phyIntfStatuses = phyTopology.phyIntfStatuses.get( phyStatus.intfId )
   if not phyIntfStatuses:
      return None

   phyStates = []

   for phyPos, intfTopoData in phyIntfStatuses.phyIntfData.items():
      _, phyIntfStatus = \
         PhyStatusLib.phyTopologyConfigStatus( em.root(), intfTopoData )
      if not phyIntfStatus:
         continue

      modelTypes = fecHistogramModelsCol.get( phyIntfStatus.tacType.fullTypeName )
      if not modelTypes:
         continue

      for modelType in modelTypes:
         instance = modelType()
         # We're grabbing the histogram here so that future use of it remains valid.
         # Same with phyName.
         histogram = instance.fecHistogram( phyIntfStatus )
         if not histogram:
            continue
         phyName = instance.phyName( histogram )
         # make sure the histogram is valid
         if ( phyName is None or
              not histogram.correctedCodewords or
              not histogram.generation.valid or
              # The intfStatus genId is the powerGenerationId of the device the
              # interface is on (eg the linecard, or the switch itself for
              # fixed systems). The populaters of fec histograms are consistent
              # in their use of powerGenerationId, wherever it may come from.
              #
              # So we expect these genIds to match, if they don't they're
              # stale, maybe the agent hasn't had the chance to update or the
              # linecard has been swapped with another type.
              histogram.generation.id != intfStatus.genId ):
            continue
         phyStates.append( FecHistogramState( instance, phyPos,
                                              histogram, phyName ) )
   return phyStates

def getOneIntfFecHistogramModel( phyStatus, phyTopology, phyIntfs ):
   intfHistogram = PhyModel.IntfFecHistogramPhy()
   intfStatus = phyIntfs[ phyStatus.intfId ].status()
   if not intfStatus:
      return None

   phyStates = getOneIntfFecHistogramStates( phyStatus, phyTopology, intfStatus )

   for instance, phyPos, histogram, phyName in phyStates:
      model = instance.toModel( histogram, phyName, phyPos )

      intfHistogram.phys.append( model )

   if not intfHistogram.phys:
      return None
   return intfHistogram

def showIntfPhyDiagErrCorrectHisto( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )

   intfsFecHistogram = PhyModel.FecHistogramPhy()

   phyStatuses, _, phyTopology, phyIntfs, _ = \
         getSysdbPhyStatuses( mode, intf, mod, detail=True,
                              exposeInternal=False )

   for phyStatus in phyStatuses:
      fecHistogram = \
         getOneIntfFecHistogramModel( phyStatus, phyTopology, phyIntfs )
      if fecHistogram:
         intfsFecHistogram.interfaces[ phyStatus.intfId ] = fecHistogram
   return intfsFecHistogram

class ShowIntfPhyDiagFecHistogram( IntfCli.ShowIntfCommand ):
   syntax = "show interfaces phy diag error-correction histogram"
   data = { 'phy' : phyShowKwNode,
            'diag' : diagShowKw,
            'error-correction' : errorCorrectionShowKw,
            'histogram' : histogramShowKw }
   handler = showIntfPhyDiagErrCorrectHisto
   moduleAtEnd = True
   cliModel = PhyModel.FecHistogramPhy

BasicCli.addShowCommandClass( ShowIntfPhyDiagFecHistogram )

#-------------------------------------------------------------------------------
# Register show tech command
#-------------------------------------------------------------------------------
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2020-09-25 11:11:11',
   cmds=[ 'show interfaces phy diag error-correction histogram | nz' ] )

#-------------------------------------------------------------------------------
# The "clear phy diag test pattern" command, in "exec" mode.
#-------------------------------------------------------------------------------

# counters callback list
clearTestPatternCountersHook = []

def registerClearTestPatternCountersHook( hook ):
   clearTestPatternCountersHook.append( hook )

def clearTestPatternCounters( mode ):
   for hook in clearTestPatternCountersHook:
      hook()

phyClearKw = CliMatcher.KeywordMatcher( 'phy',
                                        helpdesc='Clear PHY related information' )

class ClearPhyDiagTestPattern( CliCommand.CliCommandClass ):
   syntax = "clear phy diag test pattern"
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'phy' : phyClearKw,
      'diag' : 'Clear PHY diag information',
      'test' : 'Clear PHY diag test information',
      'pattern' : 'Clear PHY diag test pattern'
      }
   @staticmethod
   def handler( mode, args ):
      clearTestPatternCounters( mode )

BasicCli.EnableMode.addCommandClass( ClearPhyDiagTestPattern )

class PhyDebugStateDirCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      intfId = self.intf_.name
      debugStateDirName = getDebugStateDirName( self.intf_ )
      if debugStateDirName and intfId in \
            phyDebugStateDir[ debugStateDirName ].state:
         del phyDebugStateDir[ debugStateDirName ].state[ intfId ]

class PhyFeatureConfigSliceDirCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      intfId = self.intf_.name
      debugStateDirName = getDebugStateDirName( self.intf_ )
      if debugStateDirName:
         phyFeatureConfigDir = phyFeatureConfigSliceDir[ debugStateDirName ]
         phyFeatureConfig = phyFeatureConfigDir.config.get( intfId )
         if phyFeatureConfig:
            # Set them to default
            defPolarity = \
                  Tac.Type( "Hardware::Phy::BaseTPairPolarity" ). \
                     baseTPairPolarityDefault
            phyFeatureConfig.txClockShift = \
                  PhyFeatureConfigType.defaultTxClockShift
            phyFeatureConfig.fecHistogramBinSelect = None
            phyFeatureConfig.linkDetection = \
                  PhyFeatureConfigType.defaultLinkDetection
            phyFeatureConfig.phyLinkProfile = \
                  PhyFeatureConfigType.defaultPhyLinkProfile
            phyFeatureConfig.linkDetectionRF = \
                  Tac.Type( "Hardware::Phy::LinkDetectionRF" ).linkDetectionRfPassive
            phyFeatureConfig.standaloneLinkTraining = False
            phyFeatureConfig.linkStabilizationRfGeneration = \
                  PhyFeatureConfigType.defaultLinkStabilizationRfGeneration
            phyFeatureConfig.loopTimingPortType = \
                  PhyFeatureConfigType.defaultLoopTimingPortType
            phyFeatureConfig.loopTimingPhyRole = \
                  PhyFeatureConfigType.defaultLoopTimingPhyRole
            phyFeatureConfig.baseTRxPreDistortion = \
                  PhyFeatureConfigType.defaultBaseTRxPreDistortion
            phyFeatureConfig.baseTTxGain = PhyFeatureConfigType.defaultBaseTTxGain
            phyFeatureConfig.fastRetrain = PhyFeatureConfigType.defaultFastRetrain
            phyFeatureConfig.speedDownshifting = \
                  PhyFeatureConfigType.defaultSpeedDownshifting
            phyFeatureConfig.serdesMapping = \
                  Tac.Type( "Hardware::Phy::SerdesMapping" ).serdesMappingDefault
            phyFeatureConfig.baseTPolarityPairA = defPolarity
            phyFeatureConfig.baseTPolarityPairB = defPolarity
            phyFeatureConfig.baseTPolarityPairC = defPolarity
            phyFeatureConfig.baseTPolarityPairD = defPolarity
            phyFeatureConfig.mdiCrossoverMode = \
                  PhyFeatureConfigType.defaultMdiCrossoverMode
            phyFeatureConfig.parallelDetection = \
                  PhyFeatureConfigType.defaultParallelDetection
            phyFeatureConfig.cdrMode = \
                  Tac.Type( "Hardware::Phy::CdrMode" ).defaultCdr

#--------------------------------------------------------------------
#
# "show interfaces [ <interfaces> ] phy skew"
#
#--------------------------------------------------------------------

def showInterfacesPhySkew( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   model = PhyModel.InterfacesPhyCalibration()

   intfs = IntfCli.Intf.getAll( mode, intf, mod )
   intfNames = [ intf.name for intf in intfs ]

   # 'slice' should exist because of the guard
   if coherentStatusDir.get( 'slice' ) is None:
      return model

   for linecard in coherentStatusDir[ 'slice' ].values():
      for intf in linecard.phyCoherentSkewStatus:
         if intf not in intfNames:
            continue
         phyCoherentSkewStatus = linecard.phyCoherentSkewStatus[ intf ]
         model.interfaces[ intf ] = PhyModel.InterfacesPhySkew().toModel(
               phyCoherentSkewStatus )

   return model

class ShowIntfPhySkew( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces phy skew'
   data = dict( phy=phyShowKwNode,
                skew=skewShowKw )
   handler = showInterfacesPhySkew
   moduleAtEnd = True
   cliModel = PhyModel.InterfacesPhyCalibration

BasicCli.addShowCommandClass( ShowIntfPhySkew )

#--------------------------------------------------------------------
#
# "show interfaces [ <interfaces> ] phy skew persistent"
#
#--------------------------------------------------------------------

def showInterfacesPhySkewPersistent( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )

   model = PhyModel.InterfacesPhyCalPstore()

   intfs = IntfCli.Intf.getAll( mode, intf, mod )
   intfNames = [ intf.name for intf in intfs ]

   # 'slice' should exist because of the guard
   if coherentStatusDir.get( 'slice' ) is None:
      return model

   for linecard in coherentStatusDir[ 'slice' ].values():
      for intf in linecard.phyCoherentPstoreSkewStatus:
         if intf not in intfNames:
            continue
         phyCoherentPstoreSkewStatus = linecard.phyCoherentPstoreSkewStatus[ intf ]
         model.interfaces[ intf ] = PhyModel.InterfacesPhySkewPstore().toModel(
               phyCoherentPstoreSkewStatus )

   return model

class ShowIntfPhySkewPersistent( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces phy skew persistent'
   data = dict( phy=phyShowKwNode,
                skew=skewShowKw,
                persistent=persistentShowKw )
   handler = showInterfacesPhySkewPersistent
   moduleAtEnd = True
   cliModel = PhyModel.InterfacesPhyCalPstore

BasicCli.addShowCommandClass( ShowIntfPhySkewPersistent )

def getPhyPrecodingStatuses( intfs ):
   phyPrecodingStatuses = {}
   for linecard in precodingSliceDir.values():
      for precodingDir in linecard.values():
         for intf in intfs:
            status = precodingDir.precodingStatus.get( intf )
            caps = precodingDir.precodingCapabilities.get( intf )
            if status is None or caps is None:
               continue
            phyPrecodingStatuses[ intf ] = status
   return phyPrecodingStatuses

#--------------------------------------------------------------------------------
#
# "show interfaces [ <interface> ] phy precoding"
#
#--------------------------------------------------------------------------------
def getPhyPrecodingAll( intfs, phyCliConfigDir, ethPhyIntfStatusDir ):
   interfacesPrecoding = PhyPrecodingModel.InterfacesPrecoding()

   precodingStatuses = getPhyPrecodingStatuses( intfs )
   for intf, data in precodingStatuses.items():
      interfacePrecoding = PhyPrecodingModel.InterfacePrecoding()
      # precoding status on each lane
      for lnData in data.precodingLaneStatus.values():
         laneDetail = PhyPrecodingModel.PrecodingLaneDetail()
         txRxStatus = PhyPrecodingModel.PrecodingTxRxStatus().toModel( lnData )
         laneDetail.txRxStatus = txRxStatus
         requestStatus = PhyPrecodingModel.PrecodingRequestStatus().toModel( lnData )
         laneDetail.requestStatus = requestStatus
         interfacePrecoding.lanes[ lnData.lane ] = laneDetail
      # precoding configuration
      phyCliConfig = phyCliConfigDir.phyCliConfig.get( intf )
      if phyCliConfig:
         txConfig = phyCliConfig.precodingTx
         rxConfig = phyCliConfig.precodingRx
      else:
         precodingConfigEnum = Tac.Type( "Hardware::Phy::Precoding" )
         txConfig = precodingConfigEnum.precodingUnknown
         rxConfig = precodingConfigEnum.precodingUnknown
      enumToModelStr = { 'precodingUnknown': 'default',
                         'precodingEnabled': 'on',
                         'precodingDisabled': 'off' }
      interfacePrecoding.txConfig = enumToModelStr[ txConfig ]
      interfacePrecoding.rxConfig = enumToModelStr[ rxConfig ]
      # get CL72 link training state
      # BUG744552 TODO: should this instead be a part of the precoding status?
      # the phy knows when it is running link training. Why have this logic here?
      ethPhyIntfStatus = ethPhyIntfStatusDir.intfStatus[ intf ]
      phyFeatureStatus = PhyFeatureCli.getPhyFeatureStatusNested( intf )
      standaloneActive = ( phyFeatureStatus.standaloneLinkTraining
                           if phyFeatureStatus
                           else False )

      # pylint: disable-msg=W0212
      interfacePrecoding._cl72Enabled = ( ethPhyIntfStatus.autonegActive or
                                          standaloneActive )
      # pylint: enable-msg=W0212
      interfacesPrecoding.interfaces[ intf ] = interfacePrecoding
   return interfacesPrecoding

def showInterfacesPhyPrecoding( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   _, foundIntfNames, _, _, _ = getSysdbPhyStatuses( mode, intf, mod, detail=False )
   from CliPlugin.PhyConfigCli import phyCliConfigDir
   from CliPlugin.EthIntfCli import ethPhyIntfStatusDir
   return getPhyPrecodingAll( set( foundIntfNames ), phyCliConfigDir,
                              ethPhyIntfStatusDir )

class ShowIntfPhyPrecoding( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces phy precoding'
   data = dict( phy=phyShowKwNode,
                precoding='Display precoding information' )
   handler = showInterfacesPhyPrecoding
   moduleAtEnd = True
   cliModel = PhyPrecodingModel.InterfacesPrecoding

BasicCli.addShowCommandClass( ShowIntfPhyPrecoding )

def getPhyCliFeatureConfig( intf ):
   '''Gets the PhyFeatureConfig object corresponding to the interface name.'''
   debugStateDirName = getDebugStateDirName( intf )
   phyFeatureConfigDir = phyFeatureConfigSliceDir[ debugStateDirName ]
   # We do pre-create these, adding this for CliSave.
   return phyFeatureConfigDir.config.get( intf.name ) or \
      phyFeatureConfigDir.newConfig( intf.name )

# getPhyCliFeatureConfigForIntf takes intfName(string) as arg.
# getPhyCliFeatureConfig takes IntfCli.Intf object as arg.
def getPhyCliFeatureConfigForIntf( intfName ):
   if Cell.cellType() == "fixed":
      slotId = "FixedSystem"
   else:
      slotId = EthIntfLib.sliceName( intfName )
   phyFeatureConfigDir = phyFeatureConfigSliceDir[ slotId ]
   return phyFeatureConfigDir.config.get( intfName ) or \
      phyFeatureConfigDir.newConfig( intfName )

#--------------------------------------------------------------------
#
# "clear phy counters [ <interfaces> ]"
#
#--------------------------------------------------------------------

phySpecificGetFecStatusGenIdFn = {}
def registerPhySpecificGetFecStatusGenId( typeName, fn ):
   registeredFn = phySpecificGetFecStatusGenIdFn.get( typeName )
   assert not registeredFn or registeredFn == fn
   phySpecificGetFecStatusGenIdFn[ typeName ] = fn

phySpecificSaveCheckpointFn = {}
def registerPhySpecificSaveCheckpointFn( typeName, fn ):
   registeredFn = phySpecificSaveCheckpointFn.get( typeName )
   assert not registeredFn or registeredFn == fn
   phySpecificSaveCheckpointFn[ typeName ] = fn

phySpecificClearCountersFn = {}
def registerPhySpecificClearCountersFn( typeName, fn ):
   registeredFn = phySpecificClearCountersFn.get( typeName )
   assert not registeredFn or registeredFn == fn
   phySpecificClearCountersFn[ typeName ] = fn

def _copyHistogram( src, dst ):
   dst.uncorrectedCodewords = Tac.nonConst( src.uncorrectedCodewords )
   dst.correctedCodewords.clear()
   for binId, corr in src.correctedCodewords.items():
      dst.correctedCodewords[ binId ] = corr
   dst.generation = src.generation


def clearPhyCounters( mode, args ):
   intf = args.get( 'INTF' )
   mod = None
   phyStatuses, intfNames, _, phyIntfs, _ = getSysdbPhyStatuses( mode, intf, mod )
   EthFecEncoding = Tac.Type( 'Interface::EthFecEncoding' )

   # Because this is not a Tac::Dir and its parent is, we have to
   # mount it explicitly in order to correctly identify its type,
   # but we don't know where it is until we have completed the mount

   agents = []

   for name, sliceDir in fecStatusDir.items():
      cfg = fecConfigDir.get( name )
      if cfg and cfg.genId == sliceDir.genId:
         agents.append( sliceDir )

   for phyStatus in phyStatuses:
      if phyStatus.tacType.fullTypeName in phySpecificSaveCheckpointFn:
         saveCheckpointFn = phySpecificSaveCheckpointFn[
               phyStatus.tacType.fullTypeName ]
         saveCheckpointFn( fecConfigDir, fecStatusDir,
                           fecCounterCheckpointDir, phyStatus )
         continue
      # Note that these directories may not be present
      rsInfo = None
      rsGearboxInfo = None
      fireCodeInfo = None

      intfId = phyStatus.intfId

      # We don't know which agent has the interface, so we have to
      # iterate through all of them. Luckily there shouldn't be too
      # many (at least for now).

      for agent in agents:
         fecEncoding = agent.ethFecEncoding.get( intfId )
         if fecEncoding:
            if agent.generationId.get( intfId ) is None:
               continue
            fecCounterCheckpointDir.generationId[ intfId ] = \
                  Tac.nonConst( agent.generationId[ intfId ] )
            if fecEncoding in [ EthFecEncoding.fecEncodingReedSolomon544,
                                EthFecEncoding.fecEncodingReedSolomon ]:
               rsInfo = agent.rsFecStatus.get( intfId )
               rsGearboxInfo = agent.rsGearbox.get( intfId )
               if rsGearboxInfo:
                  assert fecEncoding == EthFecEncoding.fecEncodingReedSolomon
            elif fecEncoding == EthFecEncoding.fecEncodingFireCode:
               fireCodeInfo = agent.fireCodeFecStatus.get( intfId )
            break

      if rsInfo:
         if not fecCounterCheckpointDir.rsFecStatus.get( intfId ):
            fecCounterCheckpointDir.rsFecStatus.newMember( intfId )
         fecCounterCkpt = fecCounterCheckpointDir.rsFecStatus[ intfId ]
         fecCounterCkpt.fecBypassCorrectionAbility = \
               rsInfo.fecBypassCorrectionAbility
         fecCounterCkpt.fecBypassIndicationAbility = \
               rsInfo.fecBypassIndicationAbility
         fecCounterCkpt.fecLaneMapping = rsInfo.fecLaneMapping
         fecCounterCkpt.hiSer = Tac.nonConst( rsInfo.hiSer )
         fecCounterCkpt.fecAlignStatus = Tac.nonConst( rsInfo.fecAlignStatus )
         fecCounterCkpt.fecCorrectedCodewords = \
               Tac.nonConst( rsInfo.fecCorrectedCodewords )
         fecCounterCkpt.fecUncorrectedCodewords = \
               Tac.nonConst( rsInfo.fecUncorrectedCodewords )
         fecCounterCkpt.fecSymbolErrors.clear()
         for laneId in range( 0, len( rsInfo.fecSymbolErrors ) ):
            fecCounterCkpt.fecSymbolErrors[ laneId ] = \
                  Tac.nonConst( rsInfo.fecSymbolErrors[ laneId ] )

      if rsGearboxInfo:
         if not fecCounterCheckpointDir.rsGearbox.get( intfId ):
            fecCounterCheckpointDir.rsGearbox.newMember( intfId )
         fecCounterCkpt = fecCounterCheckpointDir.rsGearbox[ intfId ]
         fecCounterCkpt.pcsAlignStatus = \
               Tac.nonConst( rsGearboxInfo.pcsAlignStatus )
         fecCounterCkpt.pcsBlockLockStatus = \
               Tac.nonConst( rsGearboxInfo.pcsBlockLockStatus )
         fecCounterCkpt.pcsHostBipErrors.clear()
         fecCounterCkpt.pcsModuleBipErrors.clear()
         fecCounterCkpt.pcsLaneMapping.clear()
         for laneId in range( 0, len( rsGearboxInfo.pcsLaneMapping ) ):
            fecCounterCkpt.pcsLaneMapping[ laneId ] = \
                  rsGearboxInfo.pcsLaneMapping[ laneId ]
            if laneId in rsGearboxInfo.pcsHostBipErrors:
               fecCounterCkpt.pcsHostBipErrors[ laneId ] = \
                     Tac.nonConst( rsGearboxInfo.pcsHostBipErrors[ laneId ] )
            if laneId in rsGearboxInfo.pcsModuleBipErrors:
               fecCounterCkpt.pcsModuleBipErrors[ laneId ] = \
                     Tac.nonConst( rsGearboxInfo.pcsModuleBipErrors[ laneId ] )

      if fireCodeInfo:
         if not fecCounterCheckpointDir.fireCodeFecStatus.get( intfId ):
            fecCounterCheckpointDir.fireCodeFecStatus.newMember( intfId )
         fecCounterCkpt = fecCounterCheckpointDir.fireCodeFecStatus[ intfId ]
         fecCounterCkpt.laneFecCorrectedBlocks.clear()
         for laneId in range( 0, len( fireCodeInfo.laneFecCorrectedBlocks ) ):
            if laneId in fireCodeInfo.laneFecCorrectedBlocks:
               fecCounterCkpt.laneFecCorrectedBlocks[ laneId ] = \
                     Tac.nonConst( fireCodeInfo.laneFecCorrectedBlocks[ laneId ] )
         fecCounterCkpt.laneFecUncorrectedBlocks.clear()
         for laneId in range( 0, len( fireCodeInfo.laneFecUncorrectedBlocks ) ):
            if laneId in fireCodeInfo.laneFecUncorrectedBlocks:
               fecCounterCkpt.laneFecUncorrectedBlocks[ laneId ] = \
                     Tac.nonConst( fireCodeInfo.laneFecUncorrectedBlocks[ laneId ] )

   # Increment clearPhyCounters for handling by the phy agents.
   phyFeatureConfigs = {}
   for intfName in intfNames:
      config = getPhyCliFeatureConfig( phyIntfs[ intfName ] )
      if config and PhyFeatureCli.isPhyFeatureSupportedNested(
            intfName, 'clearPhyCountersSupported' ):
         phyFeatureConfigs[ intfName ] = config
         config.clearPhyCountersGenId += 1

   def clearPhyCountersHandled():
      for intfName, phyFeatureConfig in phyFeatureConfigs.items():
         sliceName = getDebugStateDirName( phyIntfs[ intfName ] )
         subsliceDir = PhyFeatureCli.phyFeatureStatusSliceDir.get( sliceName )
         if subsliceDir:
            for phyFeatureStatusDir in subsliceDir.values():
               # If the generation of the phy's PhyFeatureStatuses is good...
               if PhyFeatureCli.checkPhyFeatureStatusDirGeneration(
                     subsliceDir.name, phyFeatureStatusDir ):
                  phyFeatureStatus = phyFeatureStatusDir.status.get( intfName )
                  # If the phy feature status exists and clearPhyCounters are
                  # supported...
                  if ( phyFeatureStatus and
                       phyFeatureStatus.clearPhyCountersSupported and
                       # If status and config genIds don't match, the phy agent
                       # hasn't gotten to clearing phy counters.
                       phyFeatureStatus.clearPhyCountersGenId !=
                       phyFeatureConfig.clearPhyCountersGenId ):
                     return False
      return True

   try:
      ds = "phy counters to be cleared -- this may take up to 15 seconds"
      Tac.waitFor( clearPhyCountersHandled, sleep=True, timeout=15,
                   description=ds, maxDelay=1.0 )
   except Tac.Timeout:
      # Continue the best we can if phy agent doesn't respond within 15 seconds.
      # Print a blank line after all the dots printed by Tac.waitFor.
      print()

def strataGuard( mode, token ):
   if ( AgentDirectory.agent( mode.sysname, 'StrataCentral' ) or
        os.environ.get( 'SIMULATION_STRATA' ) ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def jericho2Guard( mode, token ):
   for sliceName in hwSliceDir:
      sandFapNiAgentName = 'SandFapNi-' + sliceName
      if AgentDirectory.agent( mode.sysname, sandFapNiAgentName ):
         return False
   return True

def clearPhyCounterGuard( mode, token ):
   if strataGuard( mode, token ) and jericho2Guard( mode, token ):
      return CliParser.guardNotThisPlatform
   else:
      return None

#--------------------------------------------------------------------------------
# clear phy counters [ INTF ]
#--------------------------------------------------------------------------------
class ClearPhyCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear phy counters [ INTF ]'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'phy' : phyClearKw,
      'counters' : CliCommand.guardedKeyword( 'counters', helpdesc='phy counters',
         guard=clearPhyCounterGuard ),
      'INTF' : IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( EthIntfCli.EthPhyAutoIntfType, ) ),
   }

   handler = clearPhyCounters

BasicCli.EnableMode.addCommandClass( ClearPhyCountersCmd )

#--------------------------------------------------------------------------------
# reset phy interface INTF chip PHY
#--------------------------------------------------------------------------------
def chipPhyAgentDir( intf ):
   if Cell.cellType() == "fixed":
      return chipSliceResetDir[ 'FixedSystem' ]
   else:
      linecard = EthIntfLib.sliceName( intf )
      return chipSliceResetDir[ linecard ]

def getPhyChips( mode, context ):
   intf = context.sharedResult[ 'INTF' ]
   agentDir = chipPhyAgentDir( intf )
   phyNames = {}
   for chipResetDir in agentDir.values():
      if intf in chipResetDir.intfIdToPhyTypes:
         for phy in chipResetDir.intfIdToPhyTypes[ intf ].phy:
            phyNames[ phy ] = 'phy name'
   return phyNames

def getIntfsToReset( intf, phy ):
   intfId = Tac.Value( "Arnet::IntfId", intf.name )
   agentDir = chipPhyAgentDir( intf.name )
   for chipResetDir in agentDir.values():
      if intfId in chipResetDir.intfIdToPhyTypes and \
         phy in chipResetDir.intfIdToPhyTypes[ intfId ].phy:
         for resetIntfs in chipResetDir.resetIdToIntfs.values():
            if intfId in resetIntfs.intfId:
               return resetIntfs.intfId
   return None

def resetIntfPhyChip( mode, args ):
   intf = args[ 'INTF' ]
   phy = args[ 'PHY' ]
   intfs = getIntfsToReset( intf, phy )
   if not intfs:
      return
   intfStr = IntfRange.intfListToCanonical( intfs )[ 0 ]
   warningStr = "Resetting phy chip will flap {} causing brief traffic disruption." \
      .format( intfStr )
   mode.addWarning( warningStr )
   prompt = "Continue? [y/N]"
   if not BasicCliUtil.confirm( mode, prompt, answerForReturn=False ):
      return
   cfg = getPhyCliFeatureConfigForIntf( intf.name )
   cfg.chipResetGenId[ phy ] = Tac.now()

nodePhy = CliCommand.guardedKeyword( 'phy',
                                     'Reset phy',
                                     CommonGuards.standbyGuard )
ethInterfaceNode = CliCommand.Node( EthIntfCli.EthPhyIntf.ethMatcher,
                                    storeSharedResult=True )
phyNameMatcher = CliMatcher.DynamicKeywordMatcher( getPhyChips, passContext=True, )

#--------------------------------------------------------------------------------
# reset phy interface INTF chip PHY
#--------------------------------------------------------------------------------
class ResetIntfPhyChipCmd( CliCommand.CliCommandClass ):
   syntax = 'reset phy interface INTF chip PHY'
   data = {
      'reset' : CliToken.Reset.resetMatcher,
      'phy' : nodePhy,
      'interface' : 'Reset phy component of interface',
      'INTF' : ethInterfaceNode,
      'chip' : 'Reset phy chip',
      'PHY' : phyNameMatcher,
   }
   handler = resetIntfPhyChip

BasicCli.EnableMode.addCommandClass( ResetIntfPhyChipCmd )

#--------------------------------------------------------------------
# Get the CLI agent to mount all the state it needs from Sysdb. Note
# that in the world of cells that (dis)appear dynamically, this is
# complicated...
#--------------------------------------------------------------------
def Plugin( entityManager ):
   global em
   global ethPhyIntfDefaultConfigDir
   global coherentStatusDir
   global xcvrStatusDir
   global intfXcvrStatusDir
   global phyDebugStateDir
   global phyFeatureConfigSliceDir
   global archerXcvrStatusDir
   global allPhyModels
   global precodingSliceDir
   global fecStatusDir
   global fecConfigDir
   global fecCounterCheckpointDir
   global pcspmaCounterCheckpointDir
   global xcvrConfigDir
   global fecHistogramDir
   global hwSliceDir
   global chipSliceResetDir

   em = entityManager
   ethPhyIntfDefaultConfigDir = ConfigMount.mount(
                         em, "interface/config/eth/phy/globalDefault",
                         "Interface::EthPhyIntfGlobalDefaultConfigDir", "w" )
   coherentStatusDir = LazyMount.mount( em, 'hardware/phy/status/data/coherent',
                                        'Tac::Dir', 'ri' )
   archerXcvrStatusDir = LazyMount.mount( em, 'hardware/archer/xcvr/status',
                                        'Tac::Dir', 'ri' )
   xcvrConfigDir = LazyMount.mount( em, "hardware/xcvr/config/all",
                                    "Xcvr::AllConfigDir", "ri" )
   xcvrStatusDir = XcvrAllStatusDir.xcvrAllStatusDir( em )
   intfXcvrStatusDir = XcvrEthIntfDir.ethIntfXcvrStatusDir( em )
   phyDebugStateDir = LazyMount.mount(
         em, 'hardware/archer/phy/config/cli/debug/slice', 'Tac::Dir', 'wi')
   phyFeatureConfigSliceDir = ConfigMount.mount(
         em, 'hardware/archer/phy/config/cli/feature/slice', 'Tac::Dir', 'wi' )
   allPhyModels = LazyMount.mount( em, 'hardware/phy/model',
                                   'Hardware::Phy::AllPhyModels', 'ri' )
   IntfCli.Intf.registerDependentClass( PhyDebugStateDirCleaner )
   IntfCli.Intf.registerDependentClass( PhyFeatureConfigSliceDirCleaner )
   precodingSliceDir = LazyMount.mount(
      em, 'hardware/phy/status/data/precoding/slice', 'Tac::Dir', 'ri' )
   fecStatusDir = LazyMount.mount( em, 'hardware/phy/status/errorCorrection/slice',
                                   'Tac::Dir', 'ri' )
   fecConfigDir = LazyMount.mount( em, 'hardware/phy/config/errorCorrection/slice',
                                   'Tac::Dir', 'ri' )
   fecCounterCheckpointDir = LazyMount.mount(
         em, 'hardware/phy/counter/checkpoint/fec',
         'Hardware::Phy::FecStatusDir', 'w' )
   pcspmaCounterCheckpointDir = LazyMount.mount(
         em, 'hardware/phy/counter/checkpoint/pcspma',
         'Hardware::Phy::StatusDir', 'w' )
   fecHistogramDir = LazyMount.mount(
         em, 'hardware/phy/status/data/fecHistogram', 'Tac::Dir', 'ri' )
   hwSliceDir = LazyMount.mount( em, 'hardware/slice', 'Tac::Dir', 'ri' )
   chipSliceResetDir = LazyMount.mount( em, 'hardware/phy/reset/slice',
                                        'Tac::Dir', "ri" )
   gv.aggregatedTable = Tac.newInstance( "Hardware::Phy::L1TopologyPhyStatusDir",
                                         "aggregatedTable" )

   mg = em.mountGroup()
   gv.l1PhyStatusDir = mg.mount( "hardware/l1/status/phy/slice",
                                  "Tac::Dir",
                                  "ri" )
   redundancyStatus = mg.mount( Cell.path( "redundancy/status" ),
                                "Redundancy::RedundancyStatus", "r" )
   gv.l1MappingDir = mg.mount( "hardware/l1/mapping",
                               "Tac::Dir",
                               "ri" )
   gv.l1TopoDir = mg.mount( "hardware/l1/topology",
                            "Tac::Dir",
                            "ri" )

   def _mountsComplete():
      global phyFeatureConfigPreCreatorStartUpSm
      phyFeatureConfigPreCreatorStartUpSm = Tac.newInstance(
         'Hardware::Phy::PhyFeatureConfigPreCreatorStartupSm',
         em.cEntityManager(), redundancyStatus )
      # Create the topoHelper
      gv.topoHelper = Tac.newInstance( 'Hardware::L1Topology::TraversalHelper',
                                       gv.l1TopoDir,
                                       gv.l1MappingDir )
      # Create the aggregation sm
      smControl = Tac.newInstance( "Arx::SmControl" )
      gv.serdesMappingAggrSm = Tac.newInstance(
                                    "Hardware::Phy::DescMappingAggrSm",
                                    gv.l1PhyStatusDir,
                                    gv.aggregatedTable,
                                    smControl )

   mg.close( _mountsComplete )
