# Copyright (c) 2010-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
PHY-layer-specific config commands
'''

# pylint: disable=consider-using-from-import
# pylint: disable=consider-using-f-string

import Arnet
import BasicCliUtil
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.FabricIntfCli as FabricIntfCli
import CliPlugin.IntfCli as IntfCli
from CliPlugin.IntfRangeCli import IntfRangeConfigMode
import CliPlugin.PhyCli as PhyCli
import CliPlugin.PhyFeatureCli as PhyFeatureCli
from CliPlugin.PhyTxEqLaneConfigCli import laneRangeMatcher
import CliToken.Hardware
import ConfigMount
import Intf.IntfRange as IntfRange
import PhyFecHistogramLib
import ProductAttributes
import Tac
from TypeFuture import TacLazyType
import LazyMount
from Toggles.L1TopologyToggleLib import toggleXcvrPolarityConfigCommandEnabled
from Toggles.PhyEeeToggleLib import (
   toggleParallelDetectionCliEnabled,
   togglePhyLinkProfileSupportEnabled,
   togglePhyPrecodingCmdEnabled,
   toggleSandFabricPrbsCliEnabled,
   toggleStrataFecHistogramsEnabled
)

phyCliConfigDir = None
phyAlaskaSliceDir = None
serdesMapping = TacLazyType( "Hardware::Phy::SerdesMapping" )
linkDetectionRf = TacLazyType( "Hardware::Phy::LinkDetectionRF" )
polarityTypeEnum = TacLazyType( "Hardware::Phy::BaseTPairPolarity" )
PhyLinkProfile = TacLazyType( "Hardware::Phy::PhyLinkProfile" )
pa = ProductAttributes.productAttributes().productAttributes

receiverKwMatcher = CliMatcher.KeywordMatcher( 'receiver',
      helpdesc='Configure the phy receiver' )
precodingKwMatcher = CliMatcher.KeywordMatcher( 'precoding',
      helpdesc='Set PHY-layer precoding' )

rollOffPrecision = 6 # arbitrarily specify 6 digits as the precision

#-------------------------------------------------------------------------------
# Utility functions shared by all commands defined in this file.
#-------------------------------------------------------------------------------
def _getOrCreatePhyCliConfig( name ):
   cfg = phyCliConfigDir.phyCliConfig.get( name )
   if not cfg:
      cfg = phyCliConfigDir.phyCliConfig.newMember( name )
   return cfg

#--------------------------------------------------------------------------------
# tx-preemphasis ATTENUATION
# The min/max are PHY specific. The range chosen was based on the
# assumption that only Aeluros/NetLogic PHYs would be using this.
#
# This version of the command, in which the user specifies an absolute
# amount of attenuation for which to compensate, is hidden because the
# method for choosing correct values requires deep understanding of
# what's being adjusted, and incorrect values can result in subtle
# failure modes that are difficult to debug.
#--------------------------------------------------------------------------------
matcherTxPreemphasis = CliMatcher.KeywordMatcher( 'tx-preemphasis',
      helpdesc='Set PHY-layer TX preemphasis' )

def _setTxPreemph( mode, overrideMethod, attenuation ):
   ao = Tac.Value( 'Hardware::Phy::AttenuationOverride' )
   ao.overrideMethod = overrideMethod
   ao.attn = attenuation
   cfg = _getOrCreatePhyCliConfig( mode.intf.name )
   cfg.txPreemphOverride = ao

class TxPreemphasisCmd( CliCommand.CliCommandClass ):
   syntax = 'tx-preemphasis ATTENUATION'
   data = {
      'tx-preemphasis' : matcherTxPreemphasis,
      'ATTENUATION' : CliMatcher.FloatMatcher( 0, 30,
         helpdesc='Attenuation of the TX path (in dB)', precisionString='%.25g',
         helpname='WORD' ),
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      attn = float( args[ 'ATTENUATION' ] )
      _setTxPreemph( mode, overrideMethod='absolute', attenuation=attn )

EthIntfCli.EthIntfModelet.addCommandClass( TxPreemphasisCmd )

#--------------------------------------------------------------------------------
# [ no | default ] tx-preemphasis standard
#--------------------------------------------------------------------------------
class TxPreemphasisStandardCmd( CliCommand.CliCommandClass ):
   syntax = 'tx-preemphasis standard'
   noOrDefaultSyntax = 'tx-preemphasis ...'
   data = {
      'tx-preemphasis' : matcherTxPreemphasis,
      # The 'standard' method adjusts the TX pre-emphasis settings so that
      # the eye is optimized at the transceiver's connector edge, i.e., to
      # compensate only for the attenuation internal to the system (due to
      # the printed-circuit board), as specified by the SFF standard.
      'standard' : 'Optimize PHY-layer TX settings per standard',
   }
   # This is not used on any platform. Rather than remove and risk
   # a customer possibly having this useless command in a config
   # resulting in some script failure, I'm making it hidden.
   # At some reasonable time in the future we can remove.
   # Today is 11/3/2023. Perhaps in 5 yrs we can remove?
   hidden = True

   @staticmethod
   def handler( mode, args ):
      _setTxPreemph( mode, overrideMethod='internalTraceOnly', attenuation=0 )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      _setTxPreemph( mode, overrideMethod='noOverride', attenuation=0 )

EthIntfCli.EthIntfModelet.addCommandClass( TxPreemphasisStandardCmd )

#--------------------------------------------------------------------------------
# [ no | default ] phy 1g-autoneg resilient
#--------------------------------------------------------------------------------
class DownspeedAutonegCmd( CliCommand.CliCommandClass ):
   syntax = 'phy 1g-autoneg resilient'
   noOrDefaultSyntax = 'phy 1g-autoneg ...'
   data = {
      'phy' : PhyCli.phyNode,
      '1g-autoneg' : 'Set PHY-layer 1G autonegotiation parameter',
      'resilient' : ( 'Force 1G autonegotiation to use longer interval '
                      'between link pulses' ),
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      cfg = _getOrCreatePhyCliConfig( mode.intf.name )
      cfg.tn8044BFwVersion = 'tn8044BFwVer111'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = _getOrCreatePhyCliConfig( mode.intf.name )
      cfg.tn8044BFwVersion = 'tn8044BFwVer123'

EthIntfCli.EthIntfModelet.addCommandClass( DownspeedAutonegCmd )

#--------------------------------------------------------------------------------
# [ no | default ] phy ll-tx-path PATH
#--------------------------------------------------------------------------------
pathMapping = { '5' : 'five', '10' : 'ten', '11' : 'eleven' }

class PhyLlTxPathCmd( CliCommand.CliCommandClass ):
   syntax = 'phy ll-tx-path PATH'
   noOrDefaultSyntax = 'phy ll-tx-path ...'
   data = {
      'phy' : PhyCli.phyNode,
      'll-tx-path' : 'Set PHY-layer TX path parameters',
      'PATH' : CliMatcher.EnumMatcher( {
         '5' : 'Set PHY LL TX path to 5',
         '10' : 'Set PHY LL TX path to 10',
         '11' : 'Set PHY LL TX path to 11'
      } )
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      cfg = _getOrCreatePhyCliConfig( mode.intf.name )
      cfg.llTxPath = pathMapping[ args[ 'PATH' ] ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = _getOrCreatePhyCliConfig( mode.intf.name )
      cfg.llTxPath = cfg.defaultLlTxPath

EthIntfCli.EthIntfModelet.addCommandClass( PhyLlTxPathCmd )

#--------------------------------------------------------------------------------
# [ no | default ] hardware phy rx-signal stabilization-delay DELAY
# Add CLI command to configure the delay before final receiver restart
#--------------------------------------------------------------------------------
class StabilizationDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware phy rx-signal stabilization-delay DELAY'
   noOrDefaultSyntax = 'hardware phy rx-signal stabilization-delay ...'
   data = {
      'hardware' : CliToken.Hardware.hardwareMatcherForConfig,
      'phy' : PhyCli.phyNode,
      'rx-signal' : 'Receive signal',
      'stabilization-delay' : 'Stabilization delay',
      'DELAY' : CliMatcher.IntegerMatcher( 0, 10000,
         helpdesc='Stabilization delay in milliseconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      cfg = _getOrCreatePhyCliConfig( mode.intf.name )
      cfg.stabilizationDelay = args[ 'DELAY' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = _getOrCreatePhyCliConfig( mode.intf.name )
      cfg.stabilizationDelay = cfg.defaultStabilizationDelay

EthIntfCli.EthIntfModelet.addCommandClass( StabilizationDelayCmd )

#--------------------------------------------------------------------------------
# [ no | default ] phy diag ( transmitter | receiver ) test pattern PATTERN
#--------------------------------------------------------------------------------
def patternOpCode( patternStr ):
   pattern = Tac.Type( "Hardware::Phy::TestPattern" )
   opCode = {
               "none"     : pattern.patternNone,
               "PRBS7"    : pattern.prbs7,
               "PRBS9"    : pattern.prbs9,
               "PRBS11"   : pattern.prbs11,
               "PRBS13"   : pattern.prbs13,
               "PRBS15"   : pattern.prbs15,
               "PRBS23"   : pattern.prbs23,
               "PRBS31"   : pattern.prbs31,
               "PRBS49"   : pattern.prbs49,
               "PRBS58"   : pattern.prbs58,
               "PRBS63"   : pattern.prbs63,
            }

   return opCode[ patternStr ]

class PhyDiagTestPatternCmd( CliCommand.CliCommandClass ):
   syntax = 'phy diag ( transmitter | receiver ) test pattern PATTERN'
   noOrDefaultSyntax = 'phy diag  ( transmitter | receiver ) test pattern ...'
   data = {
      'phy' : PhyCli.phyNode,
      'diag' : PhyCli.diagShowKw,
      'transmitter' : 'Configure the phy transmitter',
      'receiver' : receiverKwMatcher,
      'test' : 'Configure the interface in test mode',
      'pattern' : 'Configure the test pattern',
      'PATTERN' : CliMatcher.EnumMatcher( {
         'PRBS7' : 'Configure the PRBS7 test pattern',
         'PRBS9' : 'Configure the PRBS9 test pattern',
         'PRBS11' : 'Configure the PRBS11 test pattern',
         'PRBS13' : 'Configure the PRBS13 test pattern',
         'PRBS15' : 'Configure the PRBS15 test pattern',
         'PRBS23' : 'Configure the PRBS23 test pattern',
         'PRBS31' : 'Configure the PRBS31 test pattern',
         'PRBS49' : 'Configure the PRBS49 test pattern',
         'PRBS58' : 'Configure the PRBS58 test pattern',
         'PRBS63' : 'Configure the PRBS63 test pattern',
      } )
   }

   @staticmethod
   def handler( mode, args ):
      pattern = args.get( 'PATTERN', 'none' )
      cfg = _getOrCreatePhyCliConfig( mode.intf.name )
      if 'transmitter' in args:
         cfg.testPatternTx = patternOpCode( pattern )
      elif 'receiver' in args:
         cfg.testPatternRx = patternOpCode( pattern )
      else:
         assert 0, 'Invalid option for setTestPattern'

   noOrDefaultHandler = handler

EthIntfCli.EthIntfModelet.addCommandClass( PhyDiagTestPatternCmd )
# Fabric phy PRBS is currently only supported on Viper/Whistler
if toggleSandFabricPrbsCliEnabled() and pa.frontPanelFabric:
   FabricIntfCli.FabricModelet.addCommandClass( PhyDiagTestPatternCmd )

#--------------------------------------------------------------------------------
# phy precoding ( transmitter | receiver ) [ disabled ] -> old
# phy ( transmitter | receiver ) precoding [ disabled ] -> new
#--------------------------------------------------------------------------------
def setPrecoding( mode, txRxSel, noOrDefault=None, disabled=None ):
   cfg = _getOrCreatePhyCliConfig( mode.intf.name )
   if txRxSel == "transmitter":
      attr = 'precodingTx'
   else:
      assert txRxSel == "receiver"
      attr = 'precodingRx'

   precoding = Tac.Type( "Hardware::Phy::Precoding" )
   if noOrDefault:
      setattr( cfg, attr, precoding.precodingUnknown )
   elif disabled:
      setattr( cfg, attr, precoding.precodingDisabled )
   else:
      setattr( cfg, attr, precoding.precodingEnabled )

class PhyPrecodingCmd( CliCommand.CliCommandClass ):
   syntax = 'phy ( transmitter | receiver ) precoding [ disabled ]'
   noOrDefaultSyntax = 'phy ( transmitter | receiver ) precoding ...'
   data = {
      'phy' : PhyCli.phyNode,
      'transmitter' : 'Configure the phy transmitter',
      'receiver' : receiverKwMatcher,
      'precoding' : precodingKwMatcher,
      'disabled' : 'Disable precoding',
   }

   @staticmethod
   def handler( mode, args ):
      txRx = args.get( 'transmitter', 'receiver' )
      disabledArg = args.get( 'disabled' )
      setPrecoding( mode, txRx, disabled=disabledArg )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      txRx = args.get( 'transmitter', 'receiver' )
      setPrecoding( mode, txRx, noOrDefault=True )

if togglePhyPrecodingCmdEnabled():
   EthIntfCli.EthIntfModelet.addCommandClass( PhyPrecodingCmd )

class PhyPrecodingDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'phy precoding ( transmitter | receiver ) [ disabled ]'
   noOrDefaultSyntax = 'phy precoding ( transmitter | receiver ) ...'
   data = {
      'phy' : PhyCli.phyNode,
      'precoding' : precodingKwMatcher,
      'transmitter' : 'Configure the phy transmitter',
      'receiver' : receiverKwMatcher,
      'disabled' : 'Disable precoding',
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      """
      We included "phy precoding transmitter|receiver" before any forwarding agent
      used it, and wanted to change the command to
      phy transmitter|receiver precoding to be consistent with existing CLI commands.
      As the command shipped, update the command to print a warning with a pointer
      to the appropriate command.
      """
      mode.addWarning( 'The "phy precoding" command does nothing. Please use the '
                       '"phy transmitter precoding" and "phy receiver precoding" '
                       'commands instead.' )

   noOrDefaultHandler = handler

EthIntfCli.EthIntfModelet.addCommandClass( PhyPrecodingDisabledCmd )

class PhyIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del phyCliConfigDir.phyCliConfig[ self.intf_.name ]
      del phyCliConfigDir.phyCliCoherentConfig[ self.intf_.name ]

#--------------------------------------------------------------------------------
# [ no | default ] phy transmitter clock shift PPM_VALUE
#--------------------------------------------------------------------------------
def isTxClockShiftSupported( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name,
                                                 "txClockShiftSupported" ):
      return None
   return CliParser.guardNotThisPlatform

class PhyTransmitterClockShiftPpmvalueCmd( CliCommand.CliCommandClass ):
   syntax = 'phy transmitter clock shift PPM_VALUE'
   noOrDefaultSyntax = 'phy transmitter clock shift ...'
   data = {
      'phy' : PhyCli.phyNode,
      'transmitter' : 'Set transmitter parameters',
      'clock' : CliCommand.guardedKeyword( 'clock',
         helpdesc='Configure the transmit clock',
         guard=isTxClockShiftSupported ),
      'shift' : 'Shift the transmit clock frequency',
      'PPM_VALUE' : CliMatcher.IntegerMatcher( -200, 200,
         helpdesc='Amount in PPM to shift the transmit clock frequency' ),
   }

   @staticmethod
   def handler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.txClockShift = args[ 'PPM_VALUE' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.txClockShift = cfg.defaultTxClockShift

EthIntfCli.EthIntfModelet.addCommandClass( PhyTransmitterClockShiftPpmvalueCmd )

#-------------------------------------------------------------------------------
# The "[no|default] phy diag error-correction histogram phy 0 bins BIN_RANGE"
#-------------------------------------------------------------------------------

def th4FecHistoPhyNumberGuard( mode, token ):
   if PhyFecHistogramLib.getTh4FecHistoFeatureStatus(
         PhyFeatureCli.phyFeatureStatusSliceDir, mode.intf.name ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def th4FecHistoSampledHelp( fmt, ranges ):
   if not ranges:
      return None
   if len( ranges ) == 1:
      return fmt % ranges[ 0 ]
   return fmt % "%s and %s" % ( ", ".join( ranges[ :-1 ] ), ranges[ -1 ] )

# generates keyword dictionary for all possible ranges
def th4FecHistoRangeKeywords( mode ):
   helpFmt = 'Count bins %s when RS544 forward error-correction is enabled'
   helpSampledFmt = ( 'Periodically alternate counting bins %s when RS544 forward '
                      'error-correction is enabled' )

   caps = PhyFecHistogramLib.getTh4FecHistoFeatureStatus(
             PhyFeatureCli.phyFeatureStatusSliceDir, mode.intf.name )
   if not caps:
      return {}

   keywordDict = {}
   for cap in caps.rangesSupported:
      if cap == PhyFecHistogramLib.FecHistoRangeType.sampled:
         # dump all of the ranges into the help string
         nonsampled = \
            sorted( [ r for r in caps.rangesSupported
                      if r != PhyFecHistogramLib.FecHistoRangeType.sampled ] )
         nonsampled = [ r.stringValue for r in nonsampled ]
         sampledHelp = th4FecHistoSampledHelp( helpSampledFmt, nonsampled )
         if sampledHelp:
            keywordDict[ cap.stringValue ] = sampledHelp
      else:
         keywordDict[ cap.stringValue ] = helpFmt % cap.stringValue

   return keywordDict

# This command has this specific form, where TH4 phys are those selected by
# NUMBER, generated by existing phy models on an interface, and the subtokens
# are specific to it.
#
# To support another phy with different requirements, simply create a new class
# as below with its own number matcher whose result is strictly disjoint from this
# one and whatever subtokens it requires.
class Th4FecHistogramBinSelectCmd( CliCommand.CliCommandClass ):
   syntax = 'phy diag error-correction histogram PHYKW 0 bins BINS'
   noOrDefaultSyntax = 'phy diag error-correction histogram PHYKW 0 bins ...'
   data = {
         'phy' : PhyCli.phyNode,
         'diag' : PhyCli.diagShowKw,
         'error-correction' : PhyCli.errorCorrectionShowKw,
         'histogram' : PhyCli.histogramShowKw,
         # we need to workaround two of the same keyword
         'PHYKW' : PhyCli.phyNode,
         # elanini maybe TODO: populate help when possible with the chip modelName
         '0' : CliCommand.guardedKeyword( '0', 'Phy position 0',
                  guard=th4FecHistoPhyNumberGuard ),
         'bins' : 'FEC bins to count',
         # this accepts all valid ranges during boot
         'BINS' : CliMatcher.DynamicKeywordMatcher(
                     th4FecHistoRangeKeywords,
                     alwaysMatchInStartupConfig=True ),
   }

   @staticmethod
   def handler( mode, args ):
      # This command may exist and be accepted on a system which cannot
      # actually accept it. On Strata, the config will not do anything if this
      # is not supported at all.
      #
      # If config is copied from TH450G to TH4100G (different ranges) it will
      # also be accepted at boot. It will be accepted, saved, and applied.
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      # cli can accept invalid ranges at startup
      try:
         histoId = PhyFecHistogramLib.FecHistoRange( args[ 'BINS' ] )
      except IndexError:
         return
      cfg.fecHistogramBinSelect = histoId

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.fecHistogramBinSelect = None

if toggleStrataFecHistogramsEnabled():
   EthIntfCli.EthIntfModelet.addCommandClass( Th4FecHistogramBinSelectCmd )

#-------------------------------------------------------------------------------
# The "[no|default] phy link detection <aggressive>"
#-------------------------------------------------------------------------------
def isLinkDetectionSupported( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name,
                                                 "linkDetectionSupported" ):
      return None
   return CliParser.guardNotThisPlatform

class LinkDetectionCmd( CliCommand.CliCommandClass ):
   syntax = 'phy link detection aggressive'
   noOrDefaultSyntax = 'phy link detection aggressive ...'
   data = {
         'phy' : PhyCli.phyNode,
         'link' : PhyCli.linkKwMatcher,
         'detection' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                                        "detection",
                                        helpdesc="Set link detection properties" ),
                                        guard=isLinkDetectionSupported ),
         'aggressive' : 'Enable aggressive link detection',
   }

   @staticmethod
   def handler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.linkDetection = Tac.Type( "Hardware::Phy::LinkDetection" ).aggressive

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.linkDetection = cfg.defaultLinkDetection

EthIntfCli.EthIntfModelet.addCommandClass( LinkDetectionCmd )
FabricIntfCli.FabricModelet.addCommandClass( LinkDetectionCmd )

#-------------------------------------------------------------------------------
# The "[no|default] phy link profile <protection-switched>"
#-------------------------------------------------------------------------------
def configurePhyLinkProfileConfigs( intfModes, cliPhyLinkProfileConfig ):
   '''A method to set phyLinkProfile in phyFeatureConfig

   Params:
   -------
   intfModes : list
      a list of IntfConfigMode
   cliPhyLinkProfileConfig : PhyLinkProfile
      the CLI value
   '''
   for intfMode in intfModes:
      phyFeatureConfig = PhyCli.getPhyCliFeatureConfig( intfMode.intf )
      phyFeatureConfig.phyLinkProfile = cliPhyLinkProfileConfig

def getAffectedInterfacesForIntfMode( intfMode, intfModeNames ):
   '''A helper method to get all affected interfaces for the configured interface

   Params:
   -------
   intfMode : IntfConfigMode
       the CLI mode

   Return:
   -------
   affectedIntfs :  list
      all affected interfaces names because of the PhyLinkProfile changing on the
      configured interface
   '''
   # This is a platform dependent code to determine the affected interfaces.
   # For SandGen4 platforms we determine the affected interfaces based off of
   # speedGroup members of the configured interface.
   # This may not be the case for all platforms, we need to classify this and add
   # hooks to invoke the right methods in future if needed.
   affectedIntfs = []
   protectionSwitchedPhyLinkProfileLoaded = False

   group = intfMode.intf.speedGroupStatus()
   if group:
      # Check if any of the interfaces in the speedGroup have ProtectionSwitched
      # PhyLinkProfile already configured (apart from interfaces
      # under configuration). If it is, configuring/de-configuring phyLinkProfile
      # on intfMode.intf.name won't affect any other interfaces.
      for speedGroupIntfName in group.memberIntf:
         # We only care about interfaces that are not undergoing configuration
         if speedGroupIntfName in intfModeNames:
            continue

         phyFeatureConfig = PhyCli.getPhyCliFeatureConfigForIntf(
            speedGroupIntfName )
         if phyFeatureConfig.phyLinkProfile == PhyLinkProfile.protectionSwitched:
            protectionSwitchedPhyLinkProfileLoaded = True
            break

      if not protectionSwitchedPhyLinkProfileLoaded:
         affectedIntfs = list( group.memberIntf )

   return affectedIntfs

def getAffectedInterfacesForPhyLinkProfileIntfModes( intfModes, intfModeNames ):
   '''A helper method that gets the affected interfaces for all configured
   interfaces

   Params:
   -------
   intfModes : list
      a list of IntfConfigMode
   intfModeNames : list
      a list of interface names
   Return:
   -------
   affectedIntfNames : tuple
      a list of all affected interfaces
   '''
   affectedIntfNames = []
   for intfMode in intfModes:
      affectedIntfNames.extend(
         getAffectedInterfacesForIntfMode( intfMode, intfModeNames ) )
   # Remove duplicates
   return set( affectedIntfNames )

def acceptPhyLinkProfileWarningPrompt( mode, allIntfModes,
      phyLinkProfileConfigurableIntfModes, phyLinkProfileConfigurableIntfModeNames ):
   '''A helper method to display the warning prompt to the user.

   Params:
   -------
   mode : IntfConfigMode or IntfRangeConfigMode
      The CLI config mode
   allIntfModes : list
      a list of IntfConfigMode
   phyLinkProfileConfigurableIntfModes : list
      a list of IntfConfigMode
   phyLinkProfileConfigurableIntfModeNames : list
      a list of interfaces names

   Return:
   --------
   bool
   '''
   affectedIntfNames = getAffectedInterfacesForPhyLinkProfileIntfModes(
      phyLinkProfileConfigurableIntfModes, phyLinkProfileConfigurableIntfModeNames )

   allIntfModeIntfNames = []
   # Remove configured interfaces from affected interfaces list if they exist.
   # Also descern interface names of all interface modes.
   for intfMode in allIntfModes:
      affectedIntfNames.discard( intfMode.intf.name )
      allIntfModeIntfNames.append( intfMode.intf.name )

   if affectedIntfNames:
      # We only want IntfModeIntfNames to be in canonical form, whereas each of the
      # affected interface names will be printed individually in a list format.
      intfModeIntfNamesStr = IntfRange.intfListToCanonical(
         allIntfModeIntfNames )[ 0 ]
      affectedIntfNamesStr = '\n'.join( f'   {intf}'
         for intf in Arnet.sortIntf( affectedIntfNames ) )

      warningPrompt = (
         f'! Setting the PHY link profile on {intfModeIntfNamesStr} will cause'
         ' the following interfaces to flap:\n'
         '\n'
         f'{affectedIntfNamesStr}\n'
         '\n' )

      mode.addWarning( warningPrompt )
      promptText = "Do you wish to proceed with this command? [y/N]"
      ans = BasicCliUtil.confirm( mode, promptText, answerForReturn=False )
      # See if user cancelled the command.
      if not ans:
         abortMsg = "Command aborted by user"
         mode.addMessage( abortMsg )
         return False
   return True

def getPhyLinkProfileConfigurableIntfModes( intfModes, phyLinkProfileConfig,
      noOrDefault ):
   '''A helper method to filter interface modes that support PhyLinkProfile and
   that are not at the CLI value.

   Params:
   --------
   intfModes : list
      a list of IntfConfigMode
   phyLinkProfileConfig : PhyLinkProfile
      The CLI configured value
   noOrDefault : bool
      to indicate if noOrDefault handler called this method.

   Return:
   -------
   phyLinkProfileConfigurableIntfModes : list
      a list of IntfConfigMode
   '''

   def allowConfigurationOnIntf( intfMode ):
      # Allow configuration on interface only if the feature is supported or
      # when parsing startup-config or when we are reverting the command.
      # We should always take in no/default even if the feature is unsupported.
      return ( PhyFeatureCli.isPhyFeatureSupportedNested( intfMode.intf.name,
         "phyLinkProfileSupported" ) or intfMode.session_.startupConfig() or
         noOrDefault )

   phyLinkProfileConfigurableIntfModes = []
   phyLinkProfileConfigurableIntfNames = []

   for intfMode in intfModes:
      phyFeatureConfig = PhyCli.getPhyCliFeatureConfig( intfMode.intf )

      if ( allowConfigurationOnIntf( intfMode ) and
           phyFeatureConfig.phyLinkProfile != phyLinkProfileConfig ):
         phyLinkProfileConfigurableIntfModes.append( intfMode )
         phyLinkProfileConfigurableIntfNames.append( intfMode.intf.name )

   return phyLinkProfileConfigurableIntfModes, phyLinkProfileConfigurableIntfNames

def phyLinkProfileConfigHandler( mode, phyLinkProfileConfig, noOrDefault=False ):
   '''A helper method to set the config

   Params:
   -------
   mode : IntfConfigMode or IntfRangeConfigMode
      The CLI config mode
   phyLinkProfileConfig : PhyLinkProfile
      The CLI configured value
   noOrDefault : bool
      to indicate if noOrDefault handler called this method.
   '''
   intfModes = getPhyLinkProfileIntfModes( mode )

   phyLinkProfileConfigurableIntfModes, phyLinkProfileConfigurableIntfNames = \
      getPhyLinkProfileConfigurableIntfModes( intfModes, phyLinkProfileConfig,
            noOrDefault )

   if acceptPhyLinkProfileWarningPrompt( mode, intfModes,
         phyLinkProfileConfigurableIntfModes, phyLinkProfileConfigurableIntfNames ):
      configurePhyLinkProfileConfigs( phyLinkProfileConfigurableIntfModes,
         phyLinkProfileConfig )

def getPhyLinkProfileIntfModes( mode ):
   '''A helper method to get all the intfModes based on what mode we are in.

   Params:
   -------
   mode : IntfConfigMode or IntfRangeConfigMode
      The CLI config mode

   Return:
   -------
   intfModes : list
      a list of IntfConfigMode
   '''

   if isinstance( mode, IntfRangeConfigMode ):
      intfModes = list( mode.individualIntfModes_ )
   else:
      intfModes = [ mode ]
   return intfModes

def isPhyLinkProfileSupportedGuard( mode, token ):
   '''A helper method to determine the support of the command

   Params:
   -------
   mode : IntfConfigMode or IntfRangeConfigMode
      The CLI config mode
   token : str
      The guarded CLI token

   Return:
   -------
   CliParser.guardNotThisPlatform or None
   '''

   # We display unsupported message if all of the interfaces do not support the
   # command.
   for intfMode in getPhyLinkProfileIntfModes( mode ):
      if PhyFeatureCli.isPhyFeatureSupportedNested( intfMode.intf.name,
         "phyLinkProfileSupported" ):
         return None
   return CliParser.guardNotThisPlatform

profileMatcher = CliMatcher.KeywordMatcher( 'profile',
   helpdesc='Set a link profile' )
profileGuardedMatcher = CliCommand.Node( matcher=profileMatcher,
   guard=isPhyLinkProfileSupportedGuard )

class PhyLinkProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'phy link PROFILE_GUARDED protection-switched'
   noOrDefaultSyntax = 'phy link PROFILE_UNGUARDED protection-switched'

   data = {
      'phy': PhyCli.phyNode,
      'link': PhyCli.linkKwMatcher,
      'PROFILE_GUARDED': profileGuardedMatcher,
      'PROFILE_UNGUARDED': profileMatcher,
      'protection-switched': 'Set the protection-switched profile',
   }

   @staticmethod
   def handler( mode, args ):
      phyLinkProfileConfigHandler( mode, PhyLinkProfile.protectionSwitched )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      phyLinkProfileConfigHandler( mode, PhyLinkProfile.noProfile, noOrDefault=True )

if togglePhyLinkProfileSupportEnabled():
   EthIntfCli.EthIntfModelet.addCommandClass( PhyLinkProfileCmd )
   EthIntfCli.EthIntfRangeModelet.addCommandClass( PhyLinkProfileCmd )

#-------------------------------------------------------------------------------
# The "[no|default] phy link transmitter recovery"
#-------------------------------------------------------------------------------
def isLinkDetectionRfSupported( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name,
                                                 "linkDetectionRfSupported" ):
      return None
   return CliParser.guardNotThisPlatform

class LinkDetectionRFCmd( CliCommand.CliCommandClass ):
   syntax = 'phy link transmitter recovery'
   noOrDefaultSyntax = syntax
   data = {
         'phy' : PhyCli.phyNode,
         'link' : PhyCli.linkKwMatcher,
         'transmitter' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                                          "transmitter",
                                          helpdesc=\
                                                "Set link transmitter properties" ),
                                          guard=isLinkDetectionRfSupported ),
         'recovery' : 'Enable link transmitter recovery mechanism'
   }

   @staticmethod
   def handler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.linkDetectionRF = linkDetectionRf.linkDetectionRfReliable

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.linkDetectionRF = linkDetectionRf.linkDetectionRfPassive

EthIntfCli.EthIntfModelet.addCommandClass( LinkDetectionRFCmd )

#-------------------------------------------------------------------------------
# The "[no|default] phy link training"
#-------------------------------------------------------------------------------
def isStandaloneLinkTrainingSupported( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name,
                                                 "standaloneLinkTrainingSupported" ):
      return None
   return CliParser.guardNotThisPlatform

matcherTraining = CliMatcher.KeywordMatcher( "training",
                                             "Set standalone link training" )

class LinkTrainingCmd( CliCommand.CliCommandClass ):
   syntax = 'phy link training_GUARDED'
   noOrDefaultSyntax = 'phy link training ...'
   data = {
         'phy' : PhyCli.phyNode,
         'link' : PhyCli.linkKwMatcher,
         'training_GUARDED': CliCommand.Node(
            matcher=matcherTraining,
            guard=isStandaloneLinkTrainingSupported ),
         'training': matcherTraining,
   }

   @staticmethod
   def handler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.standaloneLinkTraining = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.standaloneLinkTraining = False

EthIntfCli.EthIntfModelet.addCommandClass( LinkTrainingCmd )

#-------------------------------------------------------------------------------
# The "[no|default] phy link stabilization remote-fault disabled"
#-------------------------------------------------------------------------------
def isLinkStabilizationRfSupported( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested(
         mode.intf.name, "linkStabilizationRfGenerationSupported" ):
      return None
   return CliParser.guardNotThisPlatform

stabilizationNode = CliCommand.guardedKeyword( 'stabilization',
      helpdesc='Set link stabilization properties',
      guard=isLinkStabilizationRfSupported )

remoteFaultKwMatcher = CliMatcher.KeywordMatcher( 'remote-fault',
      helpdesc = 'Set remote-fault stabilization parameters' )

class LinkStabilizationRfCmd( CliCommand.CliCommandClass ):
   syntax = 'phy link stabilization remote-fault disabled'
   noOrDefaultSyntax = syntax
   data = {
         'phy' : PhyCli.phyNode,
         'link' : PhyCli.linkKwMatcher,
         'stabilization' : stabilizationNode,
         'remote-fault' : remoteFaultKwMatcher,
         'disabled' : 'Disable forced remote-fault generation',
   }

   @staticmethod
   def handler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.linkStabilizationRfGeneration = \
         Tac.Type( "Hardware::Phy::LinkStabilizationRfGeneration" ).remoteFaultNever

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.linkStabilizationRfGeneration = \
         cfg.defaultLinkStabilizationRfGeneration

EthIntfCli.EthIntfModelet.addCommandClass( LinkStabilizationRfCmd )

#-------------------------------------------------------------------------------
# [no|default] phy media base-t negotiation port type PORTTYPE
#-------------------------------------------------------------------------------
def isLoopTimingPortTypeSupported( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name,
                                                 "loopTimingPortTypeSupported" ):
      return None
   return CliParser.guardNotThisPlatform

# Helper function that checks phyAlaska presence
def isPhyAlaskaSlicePresent( mode, token ):
   return phyAlaskaSliceDir is not None

# Helper function to guard base-t cli
def baseTSupportedGuard( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name, \
                                                 "loopTimingPortTypeSupported" ) or \
      isPhyAlaskaSlicePresent( mode, token ):
      return None
   return CliParser.guardNotThisPlatform

# Helper function to return an enum value based on CLI input
def portTypeStrToEnum( portTypeStr ):
   portTypeEnum = Tac.Type( "Hardware::Phy::LoopTimingPortType" )
   return { 'multi' : portTypeEnum.advertiseMultiPort,
            'single' : portTypeEnum.advertiseSinglePort }[ portTypeStr ]

mediaKwMatcher = CliMatcher.KeywordMatcher( 'media',
   helpdesc='Configure media specific options' )

baseTNode = CliCommand.guardedKeyword(
   'base-t',
   helpdesc='Options for BASE-T media',
   guard=baseTSupportedGuard )

baseTNodeNegotiation = CliCommand.guardedKeyword(
   'negotiation',
   helpdesc='Configure Clause 28 auto-negotiation parameters',
   guard=isLoopTimingPortTypeSupported )

class PhyPortTypeCommand( CliCommand.CliCommandClass ):
   syntax = 'phy media base-t negotiation port type PORTTYPE'
   noOrDefaultSyntax = 'phy media base-t negotiation port type ...'
   data = {
      'phy' : PhyCli.phyNode,
      'media' : mediaKwMatcher,
      'base-t' : baseTNode,
      'negotiation' : baseTNodeNegotiation,
      'port' : 'Configure port',
      'type' : 'Configure port type parameter',
      'PORTTYPE' : CliMatcher.EnumMatcher( {
         'multi' : 'Advertise as multi-port device',
         'single' : 'Advertise as single-port device'
      } )
   }

   @staticmethod
   def handler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.loopTimingPortType = portTypeStrToEnum( args[ 'PORTTYPE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.loopTimingPortType = cfg.defaultLoopTimingPortType

EthIntfCli.EthIntfModelet.addCommandClass( PhyPortTypeCommand )

#-------------------------------------------------------------------------------
# [no|default] phy media base-t negotiation role ( PHYROLE | ( auto [ non-spec ] ) )
#-------------------------------------------------------------------------------
def isLoopTimingPhyRoleSupported( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name,
                                                 "loopTimingPhyRoleSupported" ):
      return None
   return CliParser.guardNotThisPlatform

# Helper function to return an enum value based on CLI input
def phyRoleStrToEnum( phyRoleStr ):
   phyRoleEnum = Tac.Type( "Hardware::Phy::LoopTimingPhyRole" )
   return { 'primary' : phyRoleEnum.loopTimingRolePrimary,
            'secondary' : phyRoleEnum.loopTimingRoleSecondary,
            'auto' : phyRoleEnum.loopTimingRoleAuto,
            'non-spec' : phyRoleEnum.loopTimingRoleNonSpecAuto }[ phyRoleStr ]

baseTLoopTimingPhyRole = CliCommand.guardedKeyword(
   'role',
   helpdesc='Configure phy role',
   guard=isLoopTimingPhyRoleSupported )

class PhyPhyRoleCommand( CliCommand.CliCommandClass ):
   syntax = 'phy media base-t negotiation role ( PHYROLE | ( auto [ non-spec ] ) )'
   noOrDefaultSyntax = 'phy media base-t negotiation role ...'
   data = {
      'phy' : PhyCli.phyNode,
      'media' : mediaKwMatcher,
      'base-t' : baseTNode,
      'negotiation' : baseTNodeNegotiation,
      'role' : baseTLoopTimingPhyRole,
      'PHYROLE' : CliMatcher.EnumMatcher( {
         'primary' : 'Advertise as primary device',
         'secondary' : 'Advertise as secondary device'
      } ),
      'auto' : 'Advertise auto',
      'non-spec' : 'Advertise the manual field as primary (unspecified behavior)'
   }

   @staticmethod
   def handler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      advertise = args.get( 'PHYROLE') or args.get( 'non-spec', 'auto' )
      cfg.loopTimingPhyRole = phyRoleStrToEnum( advertise )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.loopTimingPhyRole = cfg.defaultLoopTimingPhyRole

EthIntfCli.EthIntfModelet.addCommandClass( PhyPhyRoleCommand )

# Helper functions to return an enum value based on CLI input
def polarityName( polarity ):
   return { "default" : polarityTypeEnum.baseTPairPolarityDefault,
            "standard" : polarityTypeEnum.baseTPairPolarityStandard,
            "reversed" : polarityTypeEnum.baseTPairPolarityReversed } [ polarity ]

class PhyPolarityTypeCommand( CliCommand.CliCommandClass ):
   syntax = 'phy media base-t polarity pair PAIR_NAME POLARITY'
   noOrDefaultSyntax = 'phy media base-t polarity pair PAIR_NAME ...'
   data = {
      'phy' : PhyCli.phyNode,
      'media' : mediaKwMatcher,
      'base-t' : baseTNode,
      'polarity' : 'Configure polarity parameters',
      'pair' : 'The wire pair to configure',
      'PAIR_NAME': CliMatcher.EnumMatcher( {
         'a' : 'Configure pair a',
         'b' : 'Configure pair b',
         'c' : 'Configure pair c',
         'd' : 'Configure pair d'
      } ),
      'POLARITY' : CliMatcher.EnumMatcher( {
         'standard' : 'Use IEEE standard polarity',
         'reversed' : 'Use reverse of IEEE standard polarity'
      } )
   }

   @staticmethod
   def handler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      pair = args[ 'PAIR_NAME' ]
      polarity = polarityName( args[ 'POLARITY' ] )
      setattr( cfg, 'baseTPolarityPair' + pair.upper(), polarity )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      pair = args[ 'PAIR_NAME' ]
      polarity = polarityName( 'default' )
      setattr( cfg, 'baseTPolarityPair' + pair.upper(), polarity )

EthIntfCli.EthIntfModelet.addCommandClass( PhyPolarityTypeCommand )

#-------------------------------------------------------------------------------
# The "[no|default] phy link serdes mapping direct"
#-------------------------------------------------------------------------------
def isSerdesMappingSupported( mode, token ):
   ''' Helper method to check if the serdesMapping is supported
   for this interface. This will act as a guard.
   '''
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name,
         "serdesMappingSupported" ):
      return None
   return CliParser.guardNotThisPlatform

def getSerdesMappingSubordinateIntfs( intfName ):
   ''' Helper method to get all the subordinate interfaces
   of this interface '''
   phyFeatureStatus = PhyFeatureCli.getPhyFeatureStatusNested( intfName )
   if phyFeatureStatus:
      return phyFeatureStatus.serdesMappingSubordinateIntfs
   return None

def acceptSerdesMappingWarningPrompt( mode, cfg ):
   ''' Helper method to display the warning prompt
   '''
   promptTemplate = \
""" Applying this command will cause the following interfaces to be disabled:

{}

"""
   warningPrompt = ""
   # All the subordinate interfaces of this interface.
   subordinateIntfs = getSerdesMappingSubordinateIntfs( mode.intf.name )
   # All the active subordinate interfaces of this interface.
   activeSubordinateIntfs = []
   ethPhyIntfStatusDir = EthIntfCli.ethPhyIntfStatusDir
   # We do not want the prompt to be displayed if we already have the
   # passed in serdesMapping configured.
   isCurrentSerdesMappingConfigDirect = \
      cfg.serdesMapping == serdesMapping.serdesMappingDirect
   if ( not isCurrentSerdesMappingConfigDirect and subordinateIntfs and
           mode.session.commandConfirmation() ):
      # We need to display only the active interfaces.
      for intfName in subordinateIntfs:
         ethPhyIntfStatus = ethPhyIntfStatusDir.intfStatus.get( intfName )
         if ethPhyIntfStatus and ethPhyIntfStatus.active:
            activeSubordinateIntfs.append( intfName )
      # Format the list string to fit the prompt syntax.
      # We also want the interfaces to be sorted.
      if activeSubordinateIntfs:
         warningPrompt = promptTemplate.format(
            '\n'.join( f'   {intf}'
            for intf in Arnet.sortIntf( activeSubordinateIntfs ) ) )

   # Check with user if they wish to proceed.
   if warningPrompt:
      mode.addWarning( warningPrompt )
      promptText = "Do you wish to proceed with this command? [y/N]"
      ans = BasicCliUtil.confirm( mode, promptText, answerForReturn=False )

      # See if user cancelled the command.
      if not ans:
         abortMsg = "Command aborted by user"
         mode.addMessage( abortMsg )
         return False
   return True

serdesKwMatcher = CliMatcher.KeywordMatcher( 'serdes',
      helpdesc='Set link serDes properties' )

mappingNode = CliCommand.guardedKeyword( 'mapping',
      helpdesc = 'Set link serDes mapping',
      guard=isSerdesMappingSupported )

class SerdesMappingCmd( CliCommand.CliCommandClass ):
   syntax = 'phy link serdes mapping direct'
   noOrDefaultSyntax = syntax
   data = {
         'phy' : PhyCli.phyNode,
         'link' : PhyCli.linkKwMatcher,
         'serdes' : serdesKwMatcher,
         'mapping' : mappingNode,
         'direct' : 'Set direct serDes mapping',
   }

   @staticmethod
   def handler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      if acceptSerdesMappingWarningPrompt( mode, cfg ):
         cfg.serdesMapping = serdesMapping.serdesMappingDirect

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.serdesMapping = serdesMapping.serdesMappingDefault

EthIntfCli.EthIntfModelet.addCommandClass( SerdesMappingCmd )

#-------------------------------------------------------------------------------
# [no|default] phy media base-t receiver filter pre-distortion FILTER
#-------------------------------------------------------------------------------
def isBaseTRxPreDistortionConfigSupported( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name,
                                                 'baseTRxPreDistortionSupported' ):
      return None
   return CliParser.guardNotThisPlatform

# Helper function to return an enum value based on CLI input
def rxPreDistortionFilterStrToEnum( rxPreDistortionFilterStr ):
   rxPreDistortionConfig = Tac.Type( 'Hardware::Phy::BaseTRxPreDistortionConfig' )
   # There is no enum value for the 'a' filter. 'a' token in CLI models the default
   # filter. Please see comments on BaseTRxPreDistortionConfig enum.
   return { 'a' : rxPreDistortionConfig.baseTRxPreDistortionFilterDefault,
            'b' : rxPreDistortionConfig.baseTRxPreDistortionFilterB,
            'disabled' : rxPreDistortionConfig.baseTRxPreDistortionFilterDisabled,
          }[ rxPreDistortionFilterStr ]

baseTNodePreDistortion = CliCommand.guardedKeyword(
   'pre-distortion',
   helpdesc='Configure the phy pre-distortion filter',
   guard=isBaseTRxPreDistortionConfigSupported )

class BaseTRxPreDistortionFilterConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'phy media base-t receiver filter pre-distortion FILTER'
   noOrDefaultSyntax = 'phy media base-t receiver filter pre-distortion ...'
   data = {
      'phy' : PhyCli.phyNode,
      'media' : mediaKwMatcher,
      'base-t' : baseTNode,
      'receiver' : 'Configure the phy receiver',
      'filter' : 'Configure the phy receiver filters',
      'pre-distortion' : baseTNodePreDistortion,
      'FILTER' : CliMatcher.EnumMatcher( {
         'a' : 'Choose pre-distortion filter A',
         'b' : 'Choose pre-distortion filter B',
         'disabled' : 'Disable pre-distortion filter',
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      filterArg = args.get( 'FILTER' )
      cfg.baseTRxPreDistortion = rxPreDistortionFilterStrToEnum( filterArg )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.baseTRxPreDistortion = cfg.defaultBaseTRxPreDistortion

EthIntfCli.EthIntfModelet.addCommandClass( BaseTRxPreDistortionFilterConfigCmd )

#-------------------------------------------------------------------------------
# [no|default] phy media base-t transmitter gain VALUE percent
#-------------------------------------------------------------------------------
def isBaseTTxGainConfigSupported( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name,
                                                 'baseTTxGainSupported' ):
      return None
   return CliParser.guardNotThisPlatform

# Helper function to return an enum value based on CLI input
def txGainStrToEnum( txGainStr ):
   baseTTxGainConfig = Tac.Type( 'Hardware::Phy::BaseTTxGainConfig' )
   return { '+1' : baseTTxGainConfig.baseTTxGainPlus1Percent,
            '+2' : baseTTxGainConfig.baseTTxGainPlus2Percent,
            '+3' : baseTTxGainConfig.baseTTxGainPlus3Percent,
            '+4' : baseTTxGainConfig.baseTTxGainPlus4Percent,
            '+5' : baseTTxGainConfig.baseTTxGainPlus5Percent,
            '+6' : baseTTxGainConfig.baseTTxGainPlus6Percent,
            '+7' : baseTTxGainConfig.baseTTxGainPlus7Percent,
            '+8' : baseTTxGainConfig.baseTTxGainPlus8Percent,
            '+9' : baseTTxGainConfig.baseTTxGainPlus9Percent,
            '+25' : baseTTxGainConfig.baseTTxGainPlus25Percent,
            '-10' : baseTTxGainConfig.baseTTxGainMinus10Percent,
            '-20' : baseTTxGainConfig.baseTTxGainMinus20Percent,
            '-30' : baseTTxGainConfig.baseTTxGainMinus30Percent,
            '-35' : baseTTxGainConfig.baseTTxGainMinus35Percent,
            '-45' : baseTTxGainConfig.baseTTxGainMinus45Percent,
            '-50' : baseTTxGainConfig.baseTTxGainMinus50Percent,
          }[ txGainStr ]

baseTNodeGain = CliCommand.guardedKeyword(
   'gain',
   helpdesc='Configure the phy transmitter gain',
   guard=isBaseTTxGainConfigSupported )

class BaseTTxGainConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'phy media base-t transmitter gain VALUE percent'
   noOrDefaultSyntax = 'phy media base-t transmitter gain ...'
   data = {
      'phy' : PhyCli.phyNode,
      'media' : mediaKwMatcher,
      'base-t' : baseTNode,
      'transmitter' : 'Configure the phy transmitter',
      'gain' : baseTNodeGain,
      'VALUE' : CliMatcher.EnumMatcher( {
         '+1' : 'Increase transmit power by 1 percent',
         '+2' : 'Increase transmit power by 2 percent',
         '+3' : 'Increase transmit power by 3 percent',
         '+4' : 'Increase transmit power by 4 percent',
         '+5' : 'Increase transmit power by 5 percent',
         '+6' : 'Increase transmit power by 6 percent',
         '+7' : 'Increase transmit power by 7 percent',
         '+8' : 'Increase transmit power by 8 percent',
         '+9' : 'Increase transmit power by 9 percent',
         '+25' : 'Increase transmit power by 25 percent',
         '-10' : 'Decrease transmit power by 10 percent',
         '-20' : 'Decrease transmit power by 20 percent',
         '-30' : 'Decrease transmit power by 30 percent',
         '-35' : 'Decrease transmit power by 35 percent',
         '-45' : 'Decrease transmit power by 45 percent',
         '-50' : 'Decrease transmit power by 50 percent',
      } ),
      'percent' : 'Configure transmitter gain in percentage points',
   }

   @staticmethod
   def handler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.baseTTxGain = txGainStrToEnum( args.get( 'VALUE' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.baseTTxGain = cfg.defaultBaseTTxGain

EthIntfCli.EthIntfModelet.addCommandClass( BaseTTxGainConfigCmd )

#-------------------------------------------------------------------------------
# [no|default] phy media base-t negotiation fast-retrain disabled
#-------------------------------------------------------------------------------
def isFastRetrainConfigSupported( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name,
                                                 'fastRetrainSupported' ):
      return None
   return CliParser.guardNotThisPlatform

baseTNodeFastRetrain = CliCommand.guardedKeyword(
   'fast-retrain',
   helpdesc='Configure fast retrain',
   guard=isFastRetrainConfigSupported )

class FastRetrainConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'phy media base-t negotiation fast-retrain disabled'
   noOrDefaultSyntax = syntax
   data = {
      'phy' : PhyCli.phyNode,
      'media' : mediaKwMatcher,
      'base-t' : baseTNode,
      'negotiation' : baseTNodeNegotiation,
      'fast-retrain' : baseTNodeFastRetrain,
      'disabled' : 'Disable fast retrain',
   }

   @staticmethod
   def handler( mode, args ):
      fastRetrainConfig = Tac.Type( 'Hardware::Phy::FastRetrainConfig' )
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.fastRetrain = fastRetrainConfig.fastRetrainDisabled

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.fastRetrain = cfg.defaultFastRetrain

EthIntfCli.EthIntfModelet.addCommandClass( FastRetrainConfigCmd )

#-------------------------------------------------------------------------------
# The "[no|default] phy media base-t negotiation downshifting disabled"
#-------------------------------------------------------------------------------
def isSpeedDownshiftingSupported( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name,
         "speedDownshiftingSupported" ):
      return None
   return CliParser.guardNotThisPlatform

baseTNodeDownshifting = CliCommand.guardedKeyword(
   'downshifting',
   helpdesc='Configure speed downshifting',
   guard=isSpeedDownshiftingSupported )

class SpeedDownshiftingCmd( CliCommand.CliCommandClass ):
   syntax = 'phy media base-t negotiation downshifting disabled'
   noOrDefaultSyntax = syntax
   data = {
      'phy' : PhyCli.phyNode,
      'media' : mediaKwMatcher,
      'base-t' : baseTNode,
      'negotiation' : baseTNodeNegotiation,
      'downshifting' : baseTNodeDownshifting,
      'disabled' : 'Disable speed downshifting',
   }

   @staticmethod
   def handler( mode, args ):
      speedDownshifting = Tac.Type( "Hardware::Phy::SpeedDownshifting" )
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.speedDownshifting = speedDownshifting.speedDownshiftingDisabled

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.speedDownshifting = cfg.defaultSpeedDownshifting

EthIntfCli.EthIntfModelet.addCommandClass( SpeedDownshiftingCmd )

#-------------------------------------------------------------------------------
# The "[no|default] phy media base-t pair layout MDICONFIG"
#-------------------------------------------------------------------------------

# Helper function to return an enum value based on CLI input
def mdiModeStrToEnum( mdiModeStr ):
   mdiModeEnum = TacLazyType( "Hardware::BaseT::MdiCrossoverMode" )
   return { 'auto' : mdiModeEnum.mdiAuto,
            'mdi' : mdiModeEnum.mdiStraight,
            'mdi-x' : mdiModeEnum.mdiCrossover }[ mdiModeStr ]

def isMdiCrossoverModeSupported( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name,
         "mdiCrossoverModeSupported" ):
      return None
   return CliParser.guardNotThisPlatform

baseTNodeLayout = CliCommand.guardedKeyword(
   'layout',
   helpdesc='Configure MDI/MDI-X pair layout',
   guard=isMdiCrossoverModeSupported )

class MdiCrossoverConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'phy media base-t pair layout MDICONFIG'
   noOrDefaultSyntax = 'phy media base-t pair layout ...'
   data = {
      'phy' : PhyCli.phyNode,
      'media' : mediaKwMatcher,
      'base-t' : baseTNode,
      'pair' : 'Configure BASE-T pair parameters',
      'layout' : baseTNodeLayout,
      'MDICONFIG' : CliMatcher.EnumMatcher( {
         'auto' : 'Automatically select MDI or MDI-X',
         'mdi' : 'MDI',
         'mdi-x' : 'MDI-X'
      } )
   }

   @staticmethod
   def handler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.mdiCrossoverMode = mdiModeStrToEnum( args[ 'MDICONFIG' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.mdiCrossoverMode = cfg.defaultMdiCrossoverMode

EthIntfCli.EthIntfModelet.addCommandClass( MdiCrossoverConfigCmd )

#-------------------------------------------------------------------------------
# [no|default] phy media base-x parallel-detection disabled
#-------------------------------------------------------------------------------

def isParallelDetectionConfigSupported( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name,
                                                 'parallelDetectionSupported' ):
      return None
   return CliParser.guardNotThisPlatform

baseXKwMatcher = CliMatcher.KeywordMatcher(
   'base-x',
   helpdesc='Options for BASE-X media' )

baseXNodeParallelDetection = CliCommand.guardedKeyword(
   'parallel-detection',
   helpdesc='Configure parallel detection',
   guard=isParallelDetectionConfigSupported )

class ParallelDetectionConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'phy media base-x parallel-detection disabled'
   noOrDefaultSyntax = syntax
   data = {
      'phy' : PhyCli.phyNode,
      'media' : mediaKwMatcher,
      'base-x' : baseXKwMatcher,
      'parallel-detection' : baseXNodeParallelDetection,
      'disabled' : 'Disable parallel detection',
   }

   @staticmethod
   def handler( mode, args ):
      parallelDetectionConfig = Tac.Type( 'Hardware::Phy::ParallelDetectionConfig' )
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.parallelDetection = parallelDetectionConfig.parallelDetectionDisabled

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = PhyCli.getPhyCliFeatureConfig( mode.intf )
      cfg.parallelDetection = cfg.defaultParallelDetection

if toggleParallelDetectionCliEnabled():
   EthIntfCli.EthIntfModelet.addCommandClass( ParallelDetectionConfigCmd )

#-------------------------------------------------------------------------------
# The "[no|default] phy receiver cdr mode MODE"
#-------------------------------------------------------------------------------

def isCdrModeConfigSupported( mode, token ):
   if PhyFeatureCli.isPhyFeatureSupportedNested( mode.intf.name,
         "cdrModeSupported" ):
      return None
   return CliParser.guardNotThisPlatform

cdrNode = CliCommand.guardedKeyword(
   'cdr',
   helpdesc='Configure Clock and Data Recovery',
   guard=isCdrModeConfigSupported )

class CdrModeConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'phy receiver cdr mode CDRMODE'
   noOrDefaultSyntax = 'phy receiver cdr mode ...'
   data = {
      'phy' : PhyCli.phyNode,
      'receiver' : receiverKwMatcher,
      'cdr' : cdrNode,
      'mode' : 'Configure CDR mode',
      'CDRMODE' : CliMatcher.EnumMatcher( {
         'auto' : 'Automatically select CDR mode',
         'baud-rate' : 'Baud-Rate CDR',
         'oversampling' : 'Oversampling CDR',
      } )
   }

   handler = "PhyConfigCdrModeCliHandler.handler"
   noOrDefaultHandler = "PhyConfigCdrModeCliHandler.noOrDefaultHandler"

EthIntfCli.EthIntfModelet.addCommandClass( CdrModeConfigCmd )

# -------------------------------------------------------------------------------
# [no|default] phy (transmitter|receiver) polarity lanes LANES (standard|reversed)
# -------------------------------------------------------------------------------

class XcvrPolarityConfigCmd( CliCommand.CliCommandClass ):
   syntax = \
      'phy ( transmitter | receiver ) polarity lanes LANES ( standard | reversed )'
   noOrDefaultSyntax = 'phy ( transmitter | receiver ) polarity lanes LANES ...'
   data = {
      'phy': PhyCli.phyNode,
      'transmitter': 'Configure transmitter parameters',
      'receiver': 'Configure receiver parameters',
      'polarity': 'Configure lane polarity',
      'lanes': 'Specify which lanes the polarity should be configured on',
      'LANES': laneRangeMatcher,
      'standard': 'Configure polarity to system defaults',
      'reversed': 'Configure polarity to swapped from system defaults'
   }

   handler = "XcvrPolarityConfigCliHandler.handler"
   noOrDefaultHandler = "XcvrPolarityConfigCliHandler.noOrDefaultHandler"

if toggleXcvrPolarityConfigCommandEnabled():
   EthIntfCli.EthIntfModelet.addCommandClass( XcvrPolarityConfigCmd )
   FabricIntfCli.FabricModelet.addCommandClass( XcvrPolarityConfigCmd )

#-------------------------------------------------------------------------------
# Mount stuff required by these commands
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global phyCliConfigDir
   global phyAlaskaSliceDir
   phyCliConfigDir = ConfigMount.mount( entityManager, 'hardware/phy/config/cli',
                                        'Hardware::Phy::PhyCliConfigDir', 'w' )
   phyAlaskaSliceDir = LazyMount.mount( entityManager,
                                        'hardware/phy/config/alaska/cell',
                                        'Tac::Dir', 'r')
   IntfCli.Intf.registerDependentClass( PhyIntfCleaner )
