# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tac

from CliModel import Dict
from CliModel import Enum
from CliModel import Int
from CliModel import Float
from CliModel import Model
from CliModel import Submodel
from CliModel import Str
from CliPlugin.EventsModel import CountAndChangedStat, ConditionAndChangedStat
from CliPlugin.PhyStatusModel import phyDetailLongHeaderFmt
from CliPlugin.PhyStatusModel import phyDetailStatFmt, phyDetailInfoFmt, Fec, \
      fecEncodings
from Ark import utcTimeRelativeToNowStr

fmt = '  %-27.27s %-16.16s %8.8s %22.22s'
fmt2 = '  %s'

laneStr =  '  %-27.27s' % '  PCS lane'
for n in range( 0, 20 ):
   laneStr = laneStr + ( ' %02d' % n )

def printChangingValues( counter, value, changes, lastChange ):
   # Ark converts from monotonic time to UTC time, so make sure
   # we give it a monotonic time to convert
   lastTimestamp = utcTimeRelativeToNowStr( lastChange )
   print( fmt % ( counter, value, changes, lastTimestamp ) )

def printCounter( counter, value, changes, lastChange ):
   printChangingValues( counter, str( value ), changes, lastChange )

def printCondition( conditionType, condition, changes, lastChange ):
   printChangingValues( conditionType, str( condition ),
         changes, lastChange )

def printHexCondition( counter, value, changes, lastChange ):
   printChangingValues( counter, hex( value ), changes, lastChange )

# This is its own function in case we ever split this into its own command,
# in which case we will want to print out labels for the other table
# columns here as well.
def printCountersHeader( header ):
   print( '  %-27.27s %s' % ( 'Forward Error Correction', header ) )

def printPerLane( laneDict ):
   for index, lane in sorted( laneDict.items() ):
      myKey = '  Lane ' + str( index )
      printCounter( myKey, lane.value, lane.changes, lane.lastChange )

class RsFecAlignmentLock( ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'FEC alignment lock', 'ok' if self.value else
                                 'unaligned', self.changes,
                                 utcTimeRelativeToNowStr( self.lastChange ) ) )

class RsFecHiSer( ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'FEC high symbol error rate',
                                 'high SER' if self.value else 'ok',
                                 self.changes,
                                 utcTimeRelativeToNowStr( self.lastChange ) ) )

def renderRsFecLaneAlignmentMarkerLockPhyDetail( modelDict ):
   print( phyDetailLongHeaderFmt % ( 'FEC lane alignment marker lock', ) )
   for lane, alignmentMarkerLock in sorted( modelDict.items() ):
      print( phyDetailStatFmt % ( '  Lane %d' % lane,
                                 'ok' if alignmentMarkerLock.value else 'unaligned',
                                 alignmentMarkerLock.changes,
                                 utcTimeRelativeToNowStr(
                                    alignmentMarkerLock.lastChange ) ) )

class RsFecAlignmentMarkerLock( ConditionAndChangedStat ):
   pass

class RsFecCorrectedCodewords( CountAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'FEC corrected codewords',
                                 self.value,
                                 self.changes,
                                 utcTimeRelativeToNowStr( self.lastChange ) ) )
class RsFecUnCorrectedCodewords( CountAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'FEC uncorrected codewords',
                                 self.value,
                                 self.changes,
                                 utcTimeRelativeToNowStr( self.lastChange ) ) )

def renderRsFecCorrectedSymbolsPhyDetail( modelDict ):
   print( phyDetailInfoFmt % ( 'FEC lane corrected symbols', '' ) )
   for lane, correctedSymbols in sorted( modelDict.items() ):
      print( phyDetailStatFmt % ( '  Lane %d' % lane,
                                 correctedSymbols.value, correctedSymbols.changes,
                                 utcTimeRelativeToNowStr(
                                    correctedSymbols.lastChange ) ) )

def renderRsFecCorrectedSymbolRatePhyDetail( corrSymRate, totalSymbols, 
      uncorrWordsLastChange, alignLock ):
   suffix = ""
   if Tac.utcNow() - uncorrWordsLastChange <= 5 or not alignLock:
      suffix = "*"
   prefix = ""
   correctedSymbolRate = 0.0
   if totalSymbols != 0:
      if corrSymRate == 0:
         prefix = "< "
         correctedSymbolRate = 1.0 / totalSymbols
      else:
         correctedSymbolRate = corrSymRate
   else:
      correctedSymbolRate = 0.0
   print( phyDetailInfoFmt % ( 'FEC corrected symbol rate', '%s%.2E%s'
         % ( prefix, correctedSymbolRate, suffix ) ) )

class RsFecCorrectedSymbols( CountAndChangedStat ):
   pass

def renderRsFecCodewordSize( size ):
   print( '  %-27.27s %s' % ( 'Reed-Solomon codeword size', size ) )

def renderRsFecLaneMapPhyDetail( modelDict ):
   fecLanes = []
   pmaLanes = []
   for fecLane, pmaLane in sorted( modelDict.items() ):
      fecLanes.append( '%02d' % fecLane )
      pmaLanes.append( '%02d' % pmaLane )
   print( phyDetailInfoFmt % ( 'FEC lane mapping', '' ) )
   print( phyDetailInfoFmt % ( '  FEC lane', ' '.join( fecLanes ) ) )
   print( phyDetailInfoFmt % ( '  PMA lane', ' '.join( pmaLanes ) ) )

class RsFec( Fec ):
   # class RsFec was initially written with '528' codeword size in mind. So
   # codeword size of class RsFec is set to '528' to make existing code work.
   # We can change this later if necessary. For now, override the codeword
   # size to '544' in derived class RsFec544.
   _totalSymbols = Float( optional=True, 
         help='Number of total symbols' )
   codewordSize = Enum( help='FEC codeword size', optional=True,
         values=( '544', '528' ) )
   hiSer = Submodel(
         valueType=RsFecHiSer, optional=True,
         help='FEC high symbol error rate indication' )
   alignmentLock = Submodel(
         valueType=RsFecAlignmentLock, optional=True,
         help='FEC alignment lock status' )
   laneAlignmentMarkerLock = Dict(
         keyType=int, valueType=RsFecAlignmentMarkerLock, optional=True,
         help='FEC lane alignment marker lock' )
   correctedCodewords = Submodel(
         valueType=RsFecCorrectedCodewords, optional=True,
         help='Number of corrected FEC codewords' )
   uncorrectedCodewords = Submodel(
         valueType=RsFecUnCorrectedCodewords, optional=True,
         help='Number of uncorrected FEC codewords' )
   # correctedSymbolRate will be a Float rather than a submodel because the customer
   # or user will not need to go to an extra level of indirection to obtain the 
   # corrected symbol rate value.
   correctedSymbolRate = Float( optional=True, help='FEC corrected symbol rate' )
   correctedSymbols = Dict(
         keyType=int, valueType=RsFecCorrectedSymbols, optional=True,
         help='Count of FEC corrected symbols per lane' )
   laneMap = Dict(
         keyType=int, valueType=int, optional=True,
         help='FEC lane to PMA lane map' )

   def render( self ):
      '''This is still used by the existing FEC CLI infrastructure.'''
      renderRsFecCodewordSize( self.codewordSize )
      if self.alignmentLock:
         self.alignmentLock.renderPhyDetail()
      if self.laneAlignmentMarkerLock:
         renderRsFecLaneAlignmentMarkerLockPhyDetail( self.laneAlignmentMarkerLock )
      if self.correctedCodewords:
         self.correctedCodewords.renderPhyDetail()
      if self.uncorrectedCodewords:
         self.uncorrectedCodewords.renderPhyDetail()
      # We should display hiSer only when it is changed since it changes
      # only when FecBypassIndicationCapbility is set.
      if self.hiSer:
         if self.hiSer.changes != 0:
            self.hiSer.renderPhyDetail()
      # Since correctedSymbolRate is a Float, we need to check that it is
      # not None rather than checking the value stored.
      # We also should ensure uncorrectedCodewords and alignmentLock exist
      # because correctedSymbolRate needs to display an asterisk following 
      # the symbol error rate value
      if self.correctedSymbolRate is not None:
         if self.uncorrectedCodewords and self.alignmentLock:
            renderRsFecCorrectedSymbolRatePhyDetail(
                  self.correctedSymbolRate,
                  self._totalSymbols,
                  self.uncorrectedCodewords.lastChange,
                  self.alignmentLock.value )
      if self.correctedSymbols:
         renderRsFecCorrectedSymbolsPhyDetail( self.correctedSymbols )
      if self.laneMap:
         renderRsFecLaneMapPhyDetail( self.laneMap )

   def renderPhyDetail( self ):
      printCountersHeader( fecEncodings[ self.encoding ] )
      self.render()

   def toModel( self, hwRsFec ):
      self.hiSer = RsFecHiSer().toModel( hwRsFec.hiSer )
      self.alignmentLock = RsFecAlignmentLock().toModel(
            hwRsFec.alignmentLock )

      laneAlignmentMarkerLock = {}
      laneAlignmentMarkerLockItems = hwRsFec.laneAlignmentMarkerLock.items()
      for rsFecLane, alignmentMarkerLock in laneAlignmentMarkerLockItems:
         laneAlignmentMarkerLock[rsFecLane] = RsFecAlignmentMarkerLock().toModel(
               alignmentMarkerLock )
      self.laneAlignmentMarkerLock = laneAlignmentMarkerLock

      self.correctedCodewords = RsFecCorrectedCodewords().toModel(
            hwRsFec.correctedCodewords )

      self.uncorrectedCodewords = RsFecUnCorrectedCodewords().toModel(
            hwRsFec.uncorrectedCodewords )

      fecSer = hwRsFec.fecSer # stash to avoid the tac model changing under us
      if fecSer:
         self.correctedSymbolRate = fecSer.ser
         self._totalSymbols = fecSer.totalSymbols

      laneCorrectedSymbols = {}
      for rsFecLane, correctedSymbols in hwRsFec.correctedSymbols.items():
         laneCorrectedSymbols[ rsFecLane ] = RsFecCorrectedSymbols().toModel(
               correctedSymbols )
      self.correctedSymbols = laneCorrectedSymbols

      self.laneMap = dict( hwRsFec.fecLaneMap )

      self.encoding = 'reedSolomon'
      self.codewordSize = '528'
      return self

class RsFec544( RsFec ):
   def toModel( self, hwRsFec ):
      super().toModel( hwRsFec )
      self.codewordSize = '544'
      return self

class RsGearboxModel( Model ):
   pcsAlignStatus = Int( help='Last PCS alignment status bitmap' )
   pcsAlignStatusChanges = Int( help='Number of times PCS alignment status bitmap'
      ' has changed' )
   pcsAlignStatusLastChange = Float( help='Last time PCS alignment status bitmap'
      ' was changed' )

   pcsLaneMapping = Dict( keyType=int, valueType=int,
         help='Mapping from FEC lanes to lanes of the PMA interface' )

   pcsBlockLockStatus = Int( help='State of the PCS blocks locks' )
   pcsBlockLockChanges = Int( help='Number of times state of the PCS block'
      ' locks has changed' )
   pcsBlockLockLastChange = Float( help='Last time state of PCS block locks'
      ' was changed' )

   pcsHostBipErrors = Dict( keyType=int, valueType=CountAndChangedStat,
      help='Count of PCS Bit Interleaved Parity (BIP) errors per FEC lane'
      ' on the host side' )
   pcsModuleBipErrors = Dict( keyType=int, valueType=CountAndChangedStat,
      help='Count of PCS Bit Interleaved Parity (BIP) errors per FEC lane'
      ' on the module side' )

   def render( self ):
      printHexCondition( 'FEC PCS lane alignment', self.pcsAlignStatus,
                    self.pcsAlignStatusChanges,
                    self.pcsAlignStatusLastChange )
      printHexCondition( 'FEC PCS block lock status', self.pcsBlockLockStatus,
                    self.pcsBlockLockChanges,
                    self.pcsBlockLockLastChange )

      # Don't record last time changed for lane mapping because we
      # do not expect it to change
      mappingStr =  '  %-27.27s' % '  PCS receive lane'
      # pylint: disable-next=consider-using-enumerate
      for lane in range( 0, len( self.pcsLaneMapping ) ):
         mappingStr = mappingStr + ( ' %02d' % self.pcsLaneMapping[ lane ] )
      print( fmt2 % 'FEC PCS lane mapping' )
      print( laneStr )
      print( mappingStr )

      # Put PCS BIP errors last because if we ever do a non-Avago gearbox
      # this may have to change
      print( '  FEC PCS host BIP errors' )
      printPerLane( self.pcsHostBipErrors )
      print( '  FEC PCS module BIP errors' )
      printPerLane( self.pcsModuleBipErrors )

class FireCodeModel( Model ):
   perLaneCorrectedFecBlocks = Dict( keyType=int, valueType=CountAndChangedStat,
      help='Count of corrected FEC blocks per FEC lane' )

   perLaneUncorrectedFecBlocks = Dict( keyType=int, valueType=CountAndChangedStat,
      help='Count of uncorrected FEC blocks per FEC lane' )

   def render( self ):
      print( fmt2 % 'FEC corrected blocks' )
      printPerLane( self.perLaneCorrectedFecBlocks )
      print( fmt2 % 'FEC uncorrected blocks' )
      printPerLane( self.perLaneUncorrectedFecBlocks )

class FcFec( Fec ):
   perLaneCorrectedFecBlocks = Dict( keyType=int, valueType=CountAndChangedStat,
      help='Count of corrected FEC blocks per FEC lane' )

   perLaneUncorrectedFecBlocks = Dict( keyType=int, valueType=CountAndChangedStat,
      help='Count of uncorrected FEC blocks per FEC lane' )

   def render( self ):
      for laneId in self.perLaneCorrectedFecBlocks:
         self.perLaneCorrectedFecBlocks[ laneId ].render()
         self.perLaneUncorrectedFecBlocks[ laneId ].render()

   def renderPhyDetail( self ):
      printCountersHeader( fecEncodings[ self.encoding ] )
      print( fmt2 % 'FEC corrected blocks' )
      printPerLane( self.perLaneCorrectedFecBlocks )
      print( fmt2 % 'FEC uncorrected blocks' )
      printPerLane( self.perLaneUncorrectedFecBlocks )

   def toModel( self, hwFcFec ):
      for laneId, codewords in hwFcFec.correctedCodewords.items():
         self.perLaneCorrectedFecBlocks[ laneId ] = CountAndChangedStat().toModel(
            codewords )
      for laneId, codewords in hwFcFec.uncorrectedCodewords.items():
         self.perLaneUncorrectedFecBlocks[ laneId ] = CountAndChangedStat().toModel(
            codewords )

      self.encoding = 'fireCode'
      return self

class PreFecBitError( Model ):
   _uncorrError = Submodel( valueType=RsFecUnCorrectedCodewords, optional=True,
         help='FEC uncorrected blocks' )
   _prefix = Str( optional=True, help='Prefix string to indicate no bit error' )
   value = Float( help='Pre-FEC bit error rate' )
   def toModel( self, hwPreFecBer, uncorr=None ):
      self._uncorrError = uncorr
      self._prefix = ''
      berType = Tac.Type( "Hardware::PhyStatus::PreFecBer" )
      if isinstance( hwPreFecBer, berType ):
         # Stash the values in case they get changed by a different thread
         totalBits = hwPreFecBer.totalBits
         errorBits = hwPreFecBer.errorBits
         if totalBits != 0:
            if errorBits == 0:
               self._prefix = "< "
               self.value = 1.0 / totalBits
            else:
               self.value = float( errorBits ) / totalBits
         else:
            self.value = 0.0
      else:
         self.value = hwPreFecBer
      return self

   def hasNewUncorrectedCodewords( self ):
      return ( self._uncorrError and self._uncorrError.value and
            Tac.utcNow() - self._uncorrError.lastChange <= 5.0 )
 
   def renderPhyDetail( self ):
      suffix = '*' if self.hasNewUncorrectedCodewords() else ''
      print( phyDetailInfoFmt % ( 'Pre-FEC bit error rate',
         self._prefix + "%.2E" % ( self.value ) + suffix ) )

# When the entire show int phy detail is converted to CAPI
# (or if we ever implement a fec specific show command)
# we will have a model consisting of a dictionary from
# Interfaces to InterfaceFecInfos
#
# Note that this change is okay because CliModels are not made public
# until a registered Cli command returns them.
class InterfaceFecInfo( Model ):
   fecType = Enum( help='The type of FEC encoding currently in use',
      values=( 'Reed-Solomon', 'Fire-Code', 'None', 'Unknown' ) )
   reedSolomon = Submodel( valueType=RsFec,
      optional=True, help='Reed-Solomon FEC debug information' )
   rsGearbox = Submodel( valueType=RsGearboxModel,
      optional=True, help='Reed-Solomon Gearbox debug information' )
   fireCode = Submodel( valueType=FireCodeModel,
      optional=True, help='Fire code FEC debug information' )

   def render( self ):
      printCountersHeader( self.fecType )
      if self.reedSolomon is not None:
         self.reedSolomon.render()
      if self.rsGearbox is not None:
         self.rsGearbox.render()
      if self.fireCode is not None:
         self.fireCode.render()
