# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import time

import CliModel
import IntfModels
import Arnet
from CliPlugin import EventsModel
from CliPlugin import PhyFecModel
from CliPlugin import PhyModelEnumLib
from CliPlugin import PhyStatusModel
from PhyFecHistogramLib import FecHistoRange
import Tac
import Ark

ethSpeedToStr = {
   'speedUnknown' : 'unknown',
   'speed10Mbps' : '10Mbps',
   'speed100Mbps' : '100Mbps',
   'speed1Gbps' : '1Gbps',
   'speed2p5Gbps' : '2.5Gbps',
   'speed5Gbps' : '5Gbps',
   'speed10Gbps' : '10Gbps',
   'speed25Gbps' : '25Gbps',
   'speed40Gbps' : '40Gbps',
   'speed50Gbps' : '50Gbps',
   'speed100Gbps' : '100Gbps',
   'speed200Gbps' : '200Gbps',
   'speed400Gbps' : '400Gbps',
   'speed800Gbps' : '800Gbps',
   # Proprietary / vendor specific speeds
   'speedHg11Gbps' : '11Gbps',
   'speedHg21Gbps' : '21Gbps',
   'speedHg27Gbps' : '27Gbps',
   'speedHg42Gbps' : '42Gbps',
   'speedHg53Gbps' : '53Gbps',
   'speedHg106Gbps' : '106Gbps',
   }

ethModulationToStr = {
   'modulationNrz' : 'NRZ',
   'modulationPam4' : 'PAM-4',
   }

ethLaneCountToStr = {
   'laneCount1' : '1',
   'laneCount2' : '2',
   'laneCount4' : '4',
   'laneCount8' : '8',
   'laneCount10' : '10',
   }

class PhyDescription( CliModel.Model ):
   phyChipName = CliModel.Str( help="Name of the phy chip" )
   location = CliModel.Enum( values=( "system", "line" ), optional=True,
         help="Location of phy within phy chip" )
   position = CliModel.Int( help="Position of the phy chip in topology",
                            optional=True )

   def renderPhyDetail( self ):
      phyName = self.phyChipName
      if self.location:
         phyName += " %s" % self.location
      if self.position is not None:
         phyName += " (phy %s)" % self.position
      print( PhyStatusModel.phyHeaderFmt % phyName )

class PhyStatus( CliModel.Model ):
   """Status of an individual phy"""
   description = CliModel.Submodel( valueType=PhyDescription,
                                    help='Phy description' )
   chip = CliModel.Submodel( valueType=PhyStatusModel.PhyChipStatus, optional=True,
                             help="Phy chip information" )
   rs = CliModel.Submodel( valueType=PhyStatusModel.Rs, optional=True,
                           help="Reconciliation sublayer information" )
   pcs = CliModel.Submodel( valueType=PhyStatusModel.Pcs, optional=True,
                            help="PCS information" )
   fec = CliModel.Submodel( valueType=PhyStatusModel.Fec, optional=True,
                            help="FEC information" )
   preFecBer = CliModel.Submodel( valueType=PhyFecModel.PreFecBitError,
                                  help="Pre FEC bit error rate of the link",
                                  optional=True )
   pma = CliModel.Submodel( valueType=PhyStatusModel.Pma, optional=True,
                            help="PMA information" )
   # Only set this if the status is actually stale
   stale = CliModel.Bool(
         help="Status of this phy might be stale",
         optional=True )

   def renderPhyDetail( self ):
      self.description.renderPhyDetail()
      if self.chip:
         self.chip.renderPhyDetail()
      if self.rs:
         self.rs.renderPhyDetail()
      if self.pcs:
         self.pcs.renderPhyDetail()
      if self.fec:
         self.fec.renderPhyDetail()
      if self.preFecBer:
         self.preFecBer.renderPhyDetail()
      if self.pma:
         self.pma.renderPhyDetail()

class RawPhyStatus( PhyStatus ):
   """Unstructured text dump representing the status of a single phy"""
   __public__ = False
   text = CliModel.Str( help='The text from CLI output' )

   def renderPhyDetail( self ):
      if self.stale:
         print( "*** WARNING: some information is stale." )

      print( self.text, end='' )

class InterfacePhyStatusesDetailed( CliModel.Model ):
   """Stores all the phys attached to a single interface.
   A single interface can have any number of phys"""
   phyStatuses = CliModel.List( valueType=PhyStatus,
         help='List of phy information for this interface' )
   interfaceState = CliModel.Submodel( valueType=PhyStatusModel.InterfaceState,
                                       help="State of the interface",
                                       optional=True )
   operSpeed = CliModel.Enum( values=list( ethSpeedToStr.values() ),
                              help="Operational speed of the interface",
                              optional=True )
   transceiver = CliModel.Submodel( valueType=PhyStatusModel.Transceiver,
                                    help="Transceiver information",
                                    optional=True )
   macFaults = CliModel.Submodel( valueType=PhyStatusModel.RsBaseR,
                           help="MAC local/remote fault information",
                           optional=True )

   def render( self ):
      upperHeadings = ( '', 'Current State', 'Changes', 'Last Change' )
      lowerHeadings = tuple ( '-' * len ( h ) for h in upperHeadings )
      print( PhyStatusModel.phyDetailStatFmt % upperHeadings )
      print( PhyStatusModel.phyDetailStatFmt % lowerHeadings )
      if self.interfaceState:
         self.interfaceState.renderPhyDetail()
      if self.operSpeed:
         print( PhyStatusModel.phyDetailInfoFmt % ( "Oper speed", self.operSpeed ) )
      if self.transceiver:
         self.transceiver.renderPhyDetail()
      if self.macFaults:
         self.macFaults.renderPhyDetail()
      for phy in self.phyStatuses:
         phy.renderPhyDetail()

class InterfacePhyStatusSummary( CliModel.Model ):
   """Summary model representing the status of all phys on one interface"""
   __public__ = False
   text = CliModel.Str( help='The text from CLI output' )
   clause = CliModel.Enum( values=( "clause22", "clause45" ),
         help="Specifies whether the phy is clause 22 or clause 45" )

   def render( self ):
      print( self.text )

class InterfacesPhyStatusesDetailed( CliModel.Model ):
   """Show a detailed status of the phy interfaces"""
   __revision__ = 2
   interfacePhyStatuses = CliModel.Dict( keyType=IntfModels.Interface,
         valueType=InterfacePhyStatusesDetailed,
         help='Collection of phy information, keyed by interface name' )

   def render( self ):
      if self.interfacePhyStatuses:
         print( 'Current System Time:', time.ctime( Tac.utcNow() ) )
      for interfaceName in Arnet.sortIntf( self.interfacePhyStatuses ):
         print( interfaceName )
         self.interfacePhyStatuses[ interfaceName ].render()

class InterfacesPhyStatusesSummary( CliModel.Model ):
   """Show a summary status of the phy interfaces (one line per intf)"""
   __public__ = False
   interfacePhyStatuses = CliModel.Dict( keyType=IntfModels.Interface,
         valueType=InterfacePhyStatusSummary,
         help='Collection of phy information, keyed by interface name' )

   def renderClause45PhySummary( self ):
      print( '''Key:
      U    = Link up
      D    = Link down
      R    = RX Fault
      T    = TX Fault
      B    = High BER
      L    = No Block Lock
      A    = No XAUI Lane Alignment
      0123 = No XAUI lane sync in lane N
      ''' )

      fmtStr = '%-16.16s %-15.15s %8.8s %8.8s %-7.7s %-5.5s %-8.8s'

      print( fmtStr % ( '', '', 'State', 'Reset', '', '', '' ) )
      
      print( fmtStr % ( 'Port', 'PHY state', 'Changes', 'Count',
                       'PMA/PMD', 'PCS', 'XAUI' ) )

      print( fmtStr % ( '-' * 16, '-' * 15, '-' * 8, '-' * 8,
                       '-' * 7, '-' * 5, '-' * 8 ) )

      for intf in Arnet.sortIntf( self.interfacePhyStatuses ):
         phy = self.interfacePhyStatuses[ intf ]
         if phy.clause == 'clause45':
            phy.render()

   def renderClause22PhySummary( self ):
      fmtStr = "%-16.16s %-15.15s %8.8s %8.8s %-8.8s %-8.8s"

      l1 = ( '' , '' , 'Reset' , 'State' , '' , '' )
      l2 = ( 'Port', 'PHY state', 'Changes', 'Count', 'Copper', 'SGMII' )
      l3 = ( '-' * 16, '-' * 15, '-' * 8, '-' * 8, '-' * 8, '-' * 8 )

      print( fmtStr % tuple( l1 ) )
      print( fmtStr % tuple( l2 ) )
      print( fmtStr % tuple( l3 ) )

      for intf in Arnet.sortIntf( self.interfacePhyStatuses ):
         phy = self.interfacePhyStatuses[ intf ]
         if phy.clause == 'clause22':
            phy.render()

   def render( self ):
      foundClause22 = False
      foundClause45 = False

      for phyStatus in self.interfacePhyStatuses.values():
         if phyStatus.clause == 'clause22':
            foundClause22 = True
         if phyStatus.clause == 'clause45':
            foundClause45 = True
         if foundClause22 and foundClause45:
            break

      if foundClause22:
         self.renderClause22PhySummary()
      if foundClause45:
         if foundClause22:
            print( '' )
         self.renderClause45PhySummary()

#--------------------------------------------------------------------------------
#
# Models for "show interfaces [ <interface> ] phy diag error-correction histogram"
#
#--------------------------------------------------------------------------------

class FecHistoBin( EventsModel.CountAndChangedStat ):
   def populateModel( self, event ):
      self.value = event.count
      self.changes = event.changes
      self.lastChange = Ark.switchTimeToUtc( event.lastChange )
      return self

   def renderPhy( self, index, _postfix=''):
      print( PhyStatusModel.phyDetailStatFmt % (
            'Bin' + str( index ) + _postfix, self.value, self.changes,
            Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class FecHistogram( CliModel.Model ):
   description = CliModel.Submodel( valueType=PhyDescription, optional=True,
         help="Phy description" )
   correctedCodewords = CliModel.Dict(
         keyType=str, valueType=FecHistoBin, optional=True,
         help="The number of codewords indexed by the number or range of symbol "
              "errors in the codeword" )
   uncorrectedCodewords = CliModel.Submodel( valueType=FecHistoBin, optional=True,
         help="The number of codewords with uncorrected errors" )
   sampled = CliModel.Bool( help="Corrected codewords counts are sampled",
                            optional=True )

   def phyName( self, fecHistogram ):
      if fecHistogram.phyDisplayName:
         return fecHistogram.phyDisplayName
      parent = fecHistogram.parent
      # could race the phy agent deleting the histogram. if it's deleted we're
      # in the orphanage and parent is None. so return None.
      if not parent:
         return None
      return parent.phyDisplayName

   def fecHistogram( self, phyStatus ):
      raise NotImplementedError

   def direction( self ):
      raise NotImplementedError

   def toModel( self, fecHistogram, phyName, position ):
      location = self.direction()
      self.description = PhyDescription( phyChipName=phyName, location=location,
                                         position=position )
      self.sampled = fecHistogram.sampled
      for binIdx, binValue in fecHistogram.correctedCodewords.items():
         self.correctedCodewords[ binIdx.stringValue ] = \
            FecHistoBin().populateModel( binValue )
      self.uncorrectedCodewords = FecHistoBin().populateModel(
            fecHistogram.uncorrectedCodewords )
      return self

   def render( self ):
      descFmt = "%s %s (phy %s)"
      if self.sampled:
         descFmt += " sampled"
      phyDesc = descFmt % ( self.description.phyChipName,
                            self.description.location,
                            self.description.position )
      print( PhyStatusModel.phyHeaderFmt % phyDesc )
      maxBin = 0
      if self.correctedCodewords:
         # convert all keys to FecHistogramRange for ease of use
         for binIndex in sorted( map( FecHistoRange, self.correctedCodewords ) ):
            maxBin = max( maxBin, binIndex.end )
            binIndexStr = binIndex.stringValue
            self.correctedCodewords[ binIndexStr ].renderPhy( binIndexStr )
      if self.uncorrectedCodewords:
         self.uncorrectedCodewords.renderPhy( maxBin + 1, '+' )

class IntfFecHistogramPhy( CliModel.Model ):
   phys = CliModel.List( valueType=FecHistogram,
                         help="Fec histogram for all phys in order of topology" )

class FecHistogramPhy( CliModel.Model ):
   interfaces = CliModel.Dict(
         keyType=IntfModels.Interface,
         valueType=IntfFecHistogramPhy,
         help="Map of interface to FEC histogram" )
   __revision__ = 2

   def render( self ):
      for intf in Arnet.sortIntf( self.interfaces ):
         print( PhyStatusModel.phyHeaderFmt % intf )
         print( PhyStatusModel.phyDetailStatFmt % ( 'Symbol Errors Per Codeword',
               'Codewords', 'Changes', 'Last Change' ) )
         print( PhyStatusModel.phyDetailStatFmt % (
               '-' * len( 'Symbol Errors Per Codeword' ),
               '-' * len( 'Codewords' ),
               '-' * len( 'Changes' ),
               '-' * len( 'Last Change' ) ) )
         intfFecHistogramPhy = self.interfaces[ intf ]
         for phy in intfFecHistogramPhy.phys:
            phy.render()

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         for intf in dictRepr[ 'interfaces' ].values():
            for phy in intf[ 'phys' ][ -2: ]:
               # PhyDescription is technically optional
               # If there is no description or location in description we
               # cannot determine if phy is system or line, so ignore it
               loc = phy.get( 'description', {} ).get( 'location' )
               if loc is None:
                  continue
               intf[ loc ] = phy
               if 'correctedCodewords' not in phy:
                  continue
               newCorrectedCodewords = {}
               for binKey, binValue in phy[ 'correctedCodewords' ].items():
                  # if we can't convert the key into a range, ignore it
                  try:
                     histoRange = FecHistoRange( binKey )
                  except IndexError:
                     continue
                  # the key was turned into a string but must degrade into an int
                  # additionally, the key can denote a range eg 0-7
                  # we will turn string "0-7" into int 0
                  # the ranges are supposed to be non-overlapping
                  newCorrectedCodewords[ histoRange.start ] = binValue
               phy[ 'correctedCodewords' ] = newCorrectedCodewords
            del intf[ 'phys' ]
      return dictRepr
  
#--------------------------------------------------------------------------------
#
# Models for "show interfaces [ <interface> ] phy skew"
#
#--------------------------------------------------------------------------------

class SkewValues( CliModel.Model ):
   skewH = CliModel.Float( help="Horizontal Polarization Skew (picoseconds)" )
   skewV = CliModel.Float( help="Vertical Polarization Skew (picoseconds)" )
   source = CliModel.Enum( values=PhyModelEnumLib.getModel(
                              PhyModelEnumLib.skewSource ),
                           help="Source of the active skew values" )
   calibrationStatus = CliModel.Enum( values=[ "ok", "failed" ],
                                      help="Status of skew calibration",
                                      optional=True )
   calibrationFailureReason = CliModel.Enum( values=PhyModelEnumLib.getModel(
                                                PhyModelEnumLib.skewCalStatus ),
                                             help="Reason for failed calibration",
                                             optional=True )

   def toModel( self, skewStatus ):
      calStatusEnum = Tac.Type( 'Phy::Coherent::SkewCalibrationStatus' )

      self.skewH = skewStatus.skewH
      self.skewV = skewStatus.skewV

      self.source = PhyModelEnumLib.skewSource[ skewStatus.source ].capi
      reason = PhyModelEnumLib.skewCalStatus[ skewStatus.calibrationStatus ].capi

      if skewStatus.calibrationStatus == calStatusEnum.skewNotApplicable:
         self.calibrationStatus = None
         self.calibrationFailureReason = None
      elif skewStatus.calibrationStatus == calStatusEnum.skewOk:
         self.calibrationStatus = 'ok'
         self.calibrationFailureReason = None
      else:
         self.calibrationStatus = 'failed'
         self.calibrationFailureReason = reason
      return self

   def getRenderOutput( self, direction ):

      skewValues = ""
      skewSource = ""
      skewStatus = ""

      # When the skew is values are being computed we display 'unknown' because
      # we do not want to show inacurate data
      if self.source == "computing":
         skewValues = "%-12.12s" % "   unknown"
      else:
         skewValues = " %4.1f, %4.1f " % ( self.skewH, self.skewV )

      skewSource = "%-10.10s" % self.source

      # When the skew is in the middle of computing, or if we did not actually
      # compute a value at all (using defaults by choice), then we do not need
      # to display a calibration status
      if self.calibrationStatus is None:
         skewStatus = ""
      else:
         skewStatus = "%s-%s" % ( direction, self.calibrationStatus )
         if self.calibrationStatus != "ok":
            skewStatus = skewStatus + "(%s)" % \
               PhyModelEnumLib.skewCalStatus[ self.calibrationFailureReason ].printed

      return ( skewValues, skewSource, skewStatus )

class InterfacesPhySkew( CliModel.Model ):
   transmitter = CliModel.Submodel( valueType=SkewValues,
                                    help="Transmitter skew values" )
   receiver = CliModel.Submodel( valueType=SkewValues,
                                 help="Receiver skew values" )

   def toModel( self, phyCoherentSkewStatus ):
      self.receiver = SkewValues().toModel( phyCoherentSkewStatus.rxSkewStatus )
      self.transmitter = SkewValues().toModel( phyCoherentSkewStatus.txSkewStatus )
      return self

   def renderModel( self, intfName ):
      fmt = "%-16.16s   %-12.12s   %-10.10s   %-12.12s   %-10.10s   %-24s"

      ( txSkew, txSource, txStatus ) = self.transmitter.getRenderOutput( "Tx" )
      ( rxSkew, rxSource, rxStatus ) = self.receiver.getRenderOutput( "Rx" )

      status = ""
      delim = ""
      if txStatus:
         status = txStatus
         delim = ","
      if rxStatus:
         status = status + delim + rxStatus

      print( fmt % ( intfName, txSkew, txSource, rxSkew, rxSource, status ) )

class InterfacesPhyCalibration( CliModel.Model ):
   interfaces = CliModel.Dict( keyType=IntfModels.Interface,
                               valueType=InterfacesPhySkew,
                               help="Active interface skew values" )

   def render( self ):
      fmtHeader = "%-16.16s   %-12.12s   %-10.10s   %-12.12s   %-10.10s   %-24.24s"

      l1 = []
      l2 = []
      l3 = []

      l1.append( '' )
      l2.append( 'Interface' )
      l3.append( '-' * 16 )

      l1.append( 'Tx Skew (ps)' )
      l2.append( '   H     V' )
      l3.append( '-' * 12 )

      l1.append( 'Tx Source' )
      l2.append( '' )
      l3.append( '-' * 10 )

      l1.append( 'Rx Skew (ps)' )
      l2.append( '   H     V' )
      l3.append( '-' * 12 )

      l1.append( 'Rx Source' )
      l2.append( '' )
      l3.append( '-' * 10 )

      l1.append( 'Calibration Status' )
      l2.append( '' )
      l3.append( '-' * 24 )

      print( fmtHeader % tuple( l1 ) )
      print( fmtHeader % tuple( l2 ) )
      print( fmtHeader % tuple( l3 ) )

      for intf in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ intf ].renderModel( intf )

#--------------------------------------------------------------------------------
#
# Models for "show interfaces [ <interface> ] phy skew persistent"
#
#--------------------------------------------------------------------------------

class PstoreSkewValues( CliModel.Model ):
   skewH = CliModel.Float( help="Horizontal Polarization Skew (picoseconds)" )
   skewV = CliModel.Float( help="Vertical Polarization Skew (picoseconds)" )
   state = CliModel.Enum( values=[ "active", "inactive" ],
                          help="Indicates if the values read from non-volatile "
                               "memory are actively used or not." )

   def toModel( self, pstoreStatus ):
      if pstoreStatus.valid:
         if pstoreStatus.active:
            self.state = "active"
         else:
            self.state = "inactive"
         self.skewH = pstoreStatus.skewH
         self.skewV = pstoreStatus.skewV
         return self
      return None

class InterfacesPhySkewPstore( CliModel.Model ):
   transmitter = CliModel.Submodel( valueType=PstoreSkewValues,
                                    help="Transmitter skew values",
                                    optional=True )
   receiver = CliModel.Submodel( valueType=PstoreSkewValues,
                                 help="Receiver skew values",
                                 optional=True )
   xcvrSn = CliModel.Str( help="Transceiver serial number" )
   version = CliModel.Int( help="Skew persistent store version" )

   def toModel( self, pstoreSkewStatus ):
      self.receiver = PstoreSkewValues().toModel( pstoreSkewStatus.rxSkewStatus )
      self.transmitter = PstoreSkewValues().toModel( pstoreSkewStatus.txSkewStatus )
      self.xcvrSn = pstoreSkewStatus.xcvrSn
      self.version = pstoreSkewStatus.version
      return self

   def renderModel( self, intfName ):
      fmt = "%-16.16s   %-16.16s   %-3.3s   " \
            " %-4.4s %-4.4s  %-8.8s    %-4.4s %-4.4s  %-8.8s"

      dataToFmt = [ intfName, self.xcvrSn, self.version ]
      for direction in [ self.transmitter, self.receiver ]:
         if not direction:
            dataToFmt.extend( [ '  - ', '  - ', "invalid" ] )
         else:
            dataToFmt.extend( [ "%4.1f" % direction.skewH, "%4.1f" % direction.skewV,
                                direction.state ] )

      print( fmt % tuple( dataToFmt ) )

class InterfacesPhyCalPstore( CliModel.Model ):
   interfaces = CliModel.Dict( keyType=IntfModels.Interface,
                               valueType=InterfacesPhySkewPstore,
                               help="Active interface skew values" )

   def render( self ):
      fmtHeader = "%-16.16s   %-16.16s   %-3.3s   %-20.20s   %-20.20s"

      l1 = []
      l2 = []
      l3 = []

      l1.append( '' )
      l2.append( 'Interface' )
      l3.append( '-' * 16 )

      l1.append( '' )
      l2.append( 'Transceiver SN' )
      l3.append( '-' * 16 )

      l1.append( '' )
      l2.append( 'Ver' )
      l3.append( '-' * 3 )

      l1.append( '      Tx Skew' )
      l2.append( '   H    V   State' )
      l3.append( '-' * 20 )

      l1.append( '      Rx Skew' )
      l2.append( '   H    V   State' )
      l3.append( '-' * 20 )

      print( fmtHeader % tuple( l1 ) )
      print( fmtHeader % tuple( l2 ) )
      print( fmtHeader % tuple( l3 ) )

      for intf in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ intf ].renderModel( intf )

# Internal PHY generation conventions:
#    Gen1 : Generation 1 (10G)
#    Gen2 : Generation 2 (40G)
#    Gen3 : Generation 3 (100G)
#    Gen4 : Generation 4 (400G)

class InternalPhyIntfStatus( PhyStatus ):
   """Common internal PHYs status associated with an interface."""
   phyState = CliModel.Submodel( valueType=PhyStatusModel.PhyState,
                                 help="PHY state information" )
   operSpeed = CliModel.Enum( values=list( ethSpeedToStr.values() ),
                              help="Operational speed of the interface" )
   interruptCount = CliModel.Int( help="PHY interrupt counter", optional=True )
   modulation = CliModel.Enum( values=list( ethModulationToStr.values() ),
                               help="PHY modulation configured on the interface",
                               optional=True )
   laneCount = CliModel.Enum( values=list( ethLaneCountToStr.values() ),
                              help="PHY lane count to support operational speed",
                              optional=True )

   def toChipModel( self, chip ):
      return PhyStatusModel.PhyChipStatus().toModel( chip )

   def toModel( self, phy ):
      if not phy.generation.valid:
         return None

      self.phyState = PhyStatusModel.PhyState().toModel( phy.phyState )
      self.interruptCount = phy.interruptCount
      chip = phy.chip # stash to avoid the tac model changing under us
      if chip:
         self.chip = self.toChipModel( chip )
      self.operSpeed = ethSpeedToStr[ phy.operSpeed ]
      return self

   def renderPhyDetail( self ):
      self.description.renderPhyDetail()
      for attr in ( 'chip', 'phyState' ):
         attr = getattr( self, attr, None )
         if attr is not None:
            # pylint: disable-msg=E1103
            attr.renderPhyDetail()
      if self.interruptCount is not None:
         print( PhyStatusModel.phyDetailStatFmt % ( 'Interrupt count', '',
                                                    self.interruptCount, '' ) )
      if self.operSpeed:
         print( PhyStatusModel.phyDetailInfoFmt % ( 'Oper speed', self.operSpeed  ) )
      if self.modulation:
         print( PhyStatusModel.phyDetailInfoFmt % ( 'Modulation', self.modulation ) )
      if self.laneCount:
         print( PhyStatusModel.phyDetailInfoFmt % ( "Lane count", self.laneCount ) )
      for attr in ( 'pcs', 'fec', 'preFecBer', 'pma' ):
         attr = getattr( self, attr, None )
         if attr is not None:
            # pylint: disable-msg=E1103
            attr.renderPhyDetail()

   def toPhySummaryModel( self, phy ):
      self.toModel( phy )

      fmtStr = '%-16.16s %-15.15s %8.8s %8.8s %-7.7s %-5.5s %-8.8s'
      if not self:
         statusLine = fmtStr % tuple( ( phy.intfId, ) + ( "-", ) * 6 )
         return InterfacePhyStatusSummary( text=statusLine, clause="clause45" )

      if self.phyState:
         phyState = PhyStatusModel.phyStateToCliStr[ self.phyState.value ]
         phyStateChanges = self.phyState.changes
      else:
         phyState = "-"
         phyStateChanges = "-"

      # New PHY status doesn't support HW reset count anymore.
      resetCount = "-"

      if self.pma:
         pmaPmdBits = self.pma.toPhySummary()
      else:
         pmaPmdBits = "-"

      if self.pcs:
         pcsBits = self.pcs.toPhySummary()
      else:
         pcsBits = "-"

      # Currently we don't have any PHY XAUI data block.
      phyXsBits = "-"

      statusLine = fmtStr % ( phy.intfId,
                              phyState,
                              phyStateChanges,
                              resetCount,
                              pmaPmdBits,
                              pcsBits,
                              phyXsBits )

      return InterfacePhyStatusSummary( text=statusLine, clause="clause45" )

class InternalGen2PhyIntfStatus( InternalPhyIntfStatus ):
   """
   Generation 2 internal PHYs interface status. FEC block is not used in
   Generation 2 internal PHYs.
   """
   def toModel( self, phy ):
      super().toModel( phy )
      pcs = phy.pcs # stash to avoid the tac model changing under us
      if pcs:
         self.pcs = PhyStatusModel.MultiLanePcsBaseR().toModel(
            pcs, numLanes=phy.pcsLanes )
      pma = phy.pma # stash to avoid the tac model changing under us
      if pma:
         self.pma = PhyStatusModel.PmaGen2().toModel( pma, phy.pmaLanes )
      return self

class InternalGen3PhyIntfStatus( InternalGen2PhyIntfStatus ):
   """3rd generation internal PHYs (100G)"""
   def toModel( self, phy ):
      super().toModel( phy )
      if not self:
         return self

      rsFec = phy.rsFec # stash to avoid the tac model changing under us
      fcFec = phy.fcFec # stash to avoid the tac model changing under us
      if rsFec:
         if self.pcs:
            # When RS-FEC is enabled, the following PCS attributes are not used. Any
            # PCS data in the HW tac model is assumed to be stale when RS-FEC is
            # enabled.
            self.pcs.blockLock = None
            self.pcs.laneBlockLock = None
            self.pcs.laneAlignmentMarkerLock = None
            self.pcs.bipErrors = None
            self.pcs.laneMap = None
         self.fec = PhyFecModel.RsFec().toModel( rsFec )
      elif fcFec:
         self.fec = PhyFecModel.FcFec().toModel( fcFec )

      # PCS block lock registers are not valid for 1G speeds, omit
      # this field in that case
      EthSpeedApi = Tac.Type( "Interface::EthSpeedApi" )
      speedMbps = EthSpeedApi.speedMbps( phy.operSpeed )
      if speedMbps > 0 and speedMbps <= 1000: # pylint: disable=chained-comparison
         if self.pcs:
            self.pcs.blockLock = None

      preFecBer = phy.preFecBer # stash to avoid the tac model changing under us
      if preFecBer:
         self.preFecBer = PhyFecModel.PreFecBitError().toModel(
            preFecBer, uncorr=self.fec.uncorrectedCodewords if self.fec else None )
      return self

class SwitchAsic400gPhyIntfStatus( InternalGen3PhyIntfStatus ):
   """4th generation internal PHYs (400G)"""
   def toModel( self, phy ):
      self.modulation = None
      self.laneCount = None
      super().toModel( phy )
      if phy.isRsFec544:
         rsFec = phy.rsFec # stash
         if rsFec:
            self.fec = PhyFecModel.RsFec544().toModel( rsFec )
      modulation = phy.modulation # stash
      if modulation != Tac.Type( "Interface::EthModulation" ).modulationUnknown:
         self.modulation = ethModulationToStr[ modulation ]
      laneCount = phy.laneCount # stash
      if laneCount != Tac.Type( "Interface::EthLaneCount" ).laneCountUnknown:
         self.laneCount = ethLaneCountToStr[ laneCount ]
      return self

class SwitchAsic800gPhyIntfStatus( SwitchAsic400gPhyIntfStatus ):
   """5th generation internal PHYs (800G)"""
