# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from IntfModels import Interface
from CliModel import Enum
from CliModel import Bool
from CliModel import Dict 
from CliModel import Model 
from CliModel import Submodel 
from Arnet import sortIntf

class InterfaceAvailableModulations( Model ):
   qam8 = Bool( help="8 state quadrature amplitude modulation", optional=True )
   qam16 = Bool( help="16 state quadrature amplitude modulation", optional=True )
   dpQpsk = Bool( help="dual-polarization quadrature phase-shift keying", 
                  optional=True )

class InterfacePhyCoherentBase( Model ):
   # The modulation mode in operation
   modulationStatus = Enum(
         values=( 'none', 'qam16', 'qam8', 'dpQpsk' ),
         help="Modulation in operation. 'none' implies that this interface " + \
               "is inactive" )

   def modulationStatusString( self ):
      """Returns a string representing the modulation status"""
      valueToStr = { "none" : "none",
                     "qam16" : "16QAM",
                     "qam8" : "8QAM",
                     "dpQpsk" : "DP-QPSK" }
      return valueToStr.get( self.modulationStatus, 'none' )

class InterfacePhyModulation( InterfacePhyCoherentBase ):
   # The user applied configuration.
   _noConfiguration = Bool( help="True if user hasn't applied any configuration" )
   modulationConfigured = Enum(
         values=( 'none', 'qam16', 'qam8', 'dpQpsk' ),
         help="Configured modulation. 'none' implies that no configuration has " + \
               "been applied on this interface." )
   availableModulations = \
         Submodel( valueType=InterfaceAvailableModulations,
                   help="Modulation choices available for " + \
                        "configuration on this interface" ) 

   def modulationConfiguredString( self ):
      """Returns a string representing the modulation configuration"""
      if self._noConfiguration:
         return 'default'
      valueToStr = { 'qam16' : '16QAM',
                     'qam8' : '8QAM',
                     'dpQpsk' : 'DP-QPSK' }
      return valueToStr.get( self.modulationConfigured, 'default' )
   
   def modulationAvailableString( self ):
      """Returns a string representing the available modulations"""
      available = []
      if self.availableModulations.qam16:
         available.append( '16QAM' )
      if self.availableModulations.qam8:
         available.append( '8QAM' )
      if self.availableModulations.dpQpsk:
         available.append( 'DP-QPSK' )
     
      # Add ', '
      if available:
         return ', '.join( available )
      return 'none'

class InterfacesPhyCoherentBase( Model ):
   pass

class InterfacesPhyModulation( InterfacesPhyCoherentBase ):
   interfaceModulation = Dict(
         keyType=Interface,
         valueType=InterfacePhyModulation,
         help="Mapping between an interface and phy modulation information" )

   def render( self ):
      if not self.interfaceModulation:
         return

      hdrFmt = '%-18s  %-23s %-23s %-15s'
      print( hdrFmt % ( 'Interface', 'Configured', 'Available', 'Operational' ) )
      print( hdrFmt % ( '-' * 16, '-' * 20, '-' * 20, '-' * 11 ) )

      for intf in sortIntf( self.interfaceModulation ):
         status = self.interfaceModulation[ intf ]
         print( hdrFmt % ( intf,
                          status.modulationConfiguredString(),
                          status.modulationAvailableString(),
                          status.modulationStatusString() ) )

