# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
The CAPI model for the "show int negotiation detail" command.
'''

import time

import Ark
import Arnet
import CliModel
import IntfModels
import Tac

from CliPlugin import PhyModel
from CliPlugin.EventsModel import ConditionAndChangedStat, EventStat
from TypeFuture import TacLazyType

AutonegFecAdvertisement = TacLazyType( 'Interface::AutonegFecAdvertisement' )

anModeToStr = {
      'anegModeUnknown': 'unknown',
      'anegModeUnsupported': 'unsupported',
      'anegModeDisabled': 'disabled',
      'anegModeClause28': 'clause 28',
      'anegModeClause37': 'clause 37',
      'anegModeClause73': 'clause 73',
      'anegModeSgmii': 'sgmii',
   }

anStatusToStr = {
      'anegStateUnknown': 'unknown',
      'anegStateOff': 'off',
      'anegStateNegotiating': 'negotiating',
      'anegStateSuccessful': 'success',
      'anegStateFailed': 'failed',
   }

pauseToStr = {
      'pauseUnknown': 'unknown',
      'pauseDisabled': 'disabled',
      'pauseTxOnly': 'tx only',
      'pauseSymmetric': 'symmetric',
      'pauseRxOrSymmetric': 'rx or symmetric'
   }

fecEncodingToStr = {
      'fecEncodingReedSolomon': 'reed-solomon',
      'fecEncodingReedSolomon544': 'reed-solomon',
      'fecEncodingFireCode': 'fire-code',
      'fecEncodingDisabled': 'disabled',
      'fecEncodingUnknown': 'unsupported'
   }

flowcontrolStatusToStr = {
      'flowControlStatusOn': 'on',
      'flowControlStatusOff': 'off',
      'flowControlStatusDisagree': 'disagree',
      'flowControlStatusUnknown': 'unknown'
   }

linkModes = {
      'unknown',

      # BASE-T modes
      '10M/full',
      '10M/half',
      '100M/full',
      '100M/half',
      '1G/full',
      '1G/half',
      '2.5G/full',
      '5G/full',
      '10G/full',

      # CR/optic modes
      '1G-1',
      '10G-1',
      '25G-1',
      '40G-4',
      '50G-1',
      '50G-2',
      '100G-1',
      '100G-2',
      '100G-4',
      '200G-2',
      '200G-4',
      '400G-4',
      '400G-8',
      '800G-8',
   }

techAbilityToStr = {
      'techAbilityUnknown': 'unknown',
      'techAbility10gKr': '10G-KR',
      'techAbility25gCr': '25G-CR',
      'techAbility25gCrS': '25G-CR-S',
      'techAbilityEthCon25gCr': '25G-CR(consortium)',
      'techAbilityEthCon25gKr': '25G-KR(consortium)',
      'techAbility40gCr4': '40G-CR4',
      'techAbility50gCr1': '50G-CR1',
      'techAbilityEthCon50gCr2': '50G-CR2(consortium)',
      'techAbilityEthCon50gKr2': '50G-KR2(consortium)',
      'techAbility100gCr1': '100G-CR1',
      'techAbility100gCr2': '100G-CR2',
      'techAbility100gCr4': '100G-CR4',
      'techAbility200gCr2': '200G-CR2',
      'techAbility200gCr4': '200G-CR4',
      'techAbility400gCr4': '400G-CR4',
      'techAbilityEthCon400gCr8': '400G-CR8(consortium)',
   }

ConditionTac = Tac.Type("Ark::ConditionOrChangedStat")
EventStatTac = Tac.Type("Ark::EventStat")

def _printLine( label, value='', changes='', timestamp='', nest=0 ):
   if not isinstance( timestamp, str ):
      timestamp = Ark.utcTimeRelativeToNowStr( timestamp )
   if not changes and not timestamp:
      print( f"  {nest * '  ' + label!s:30.30} {value!s:48.48}" )
   else:
      print( f"  {nest * '  ' + label!s:30.30} {value!s:16.16} {changes!s:>8.8}"
             f" {timestamp!s:>22.22}" )

class NegotiationMode( CliModel.Model ):
   value = CliModel.Enum( values=anModeToStr.keys(),
                          help="Auto-negotiation mode" )
   changes = CliModel.Int( help="The number of times the condition changed" )
   lastChange = CliModel.Float( help="The last time the condition changed" )

   def render( self, nest=0 ):
      _printLine( 'Auto-negotiation mode', anModeToStr[ self.value ],
                  self.changes, self.lastChange, nest=nest )

class NegotiationStatus( CliModel.Model ):
   value = CliModel.Enum( values=anStatusToStr.keys(),
                          help="Auto-negotiation status" )
   changes = CliModel.Int( help="The number of times the condition changed" )
   lastChange = CliModel.Float( help="The last time the condition changed" )

   def render( self, nest=0 ):
      _printLine( 'Auto-negotiation status', anStatusToStr[ self.value ],
                  self.changes, self.lastChange, nest=nest )

class FecAbility( CliModel.Model ):
   value = CliModel.Enum( values=fecEncodingToStr.keys(),
                          help='FEC ability' )

def _getTechAbilities( abilities ) -> list[str]:
   result = []
   for k, v in techAbilityToStr.items():
      if getattr( abilities, k ):
         result.append( v )
   return result

class NegotiationAdvert( CliModel.Model ):
   changes = CliModel.Int( help="The number of times the value changed" )
   lastChange = CliModel.Float( help="The last time the value changed" )
   techAbility = CliModel.List( valueType=str,
                                help="Requested technology abilities" )
   pause = CliModel.Enum( values=pauseToStr.keys(),
                          help="Requested pause mode" )
   fecAbility = CliModel.List( valueType=FecAbility,
                               help="FEC ability advertisement" )
   fecRequested = CliModel.List( valueType=FecAbility,
                                 help="Requested FEC mode" )

   def toModel( self, autonegAdvert, advertChanges, advertLastChange,
                ieeeFecAdvert, consFecAdvert ):
      self.changes = advertChanges
      self.lastChange = advertLastChange
      self.techAbility = _getTechAbilities( autonegAdvert.technologyAbilities )
      self.pause = autonegAdvert.pause

      self.fecAbility = []
      if ieeeFecAdvert.cl74Fec.ability or consFecAdvert.cl74Fec.ability:
         self.fecAbility.append( FecAbility( value="fecEncodingFireCode" ) )
      if ieeeFecAdvert.rs528Fec.ability or consFecAdvert.rs528Fec.ability:
         self.fecAbility.append( FecAbility( value="fecEncodingReedSolomon" ) )
      self.fecRequested = []
      if ieeeFecAdvert.cl74Fec.request or consFecAdvert.cl74Fec.request:
         self.fecRequested.append( FecAbility( value="fecEncodingFireCode" ) )
      if ieeeFecAdvert.rs528Fec.request or consFecAdvert.rs528Fec.request:
         self.fecRequested.append( FecAbility( value="fecEncodingReedSolomon" ) )
      return self

   def renderWithSide( self, side, nest ):
      _printLine( side, '', self.changes, self.lastChange, nest=nest )
      _printLine( 'Technology ability', ', '.join( self.techAbility ),
                  nest=nest + 1 )
      _printLine( 'Pause', pauseToStr[ self.pause ], nest=nest + 1 )
      _printLine( 'FEC ability',
                  ', '.join( fecEncodingToStr[ f.value ] for f in self.fecAbility ),
                  nest=nest + 1 )
      _printLine( 'FEC requested', ', '.join( fecEncodingToStr[ f.value ]
                                              for f in self.fecRequested ),
                  nest=nest + 1 )

class LinkMode( CliModel.Model ):
   value = CliModel.Enum( values=linkModes, help="Link speed, lanes and duplex" )
   changes = CliModel.Int( help="The number of times the value changed" )
   lastChange = CliModel.Float( help="The last time the value changed" )

   def toModel( self, autonegStatus ):
      self.changes = autonegStatus.speedChanges.count
      self.lastChange = Ark.switchTimeToUtc( autonegStatus.speedChanges.last )
      EthSpeedApi = Tac.Value( 'Interface::EthSpeedApi' )
      if autonegStatus.mode == 'anegModeClause28':
         if autonegStatus.speed == 'speedUnknown' or \
               autonegStatus.duplex == 'duplexUnknown':
            self.value = 'unknown'
            return self
         speedMbps = EthSpeedApi.speedMbps( autonegStatus.speed )
         duplexStr = 'half' if autonegStatus.duplex == 'duplexHalf' else 'full'
         if speedMbps < 1000:
            self.value = f'{speedMbps}M/{duplexStr}'
         else:
            self.value = f'{( speedMbps/1000 ):g}G/{duplexStr}'
      else:
         if autonegStatus.speed == 'speedUnknown' or \
               autonegStatus.laneCount == 'laneCountUnknown':
            self.value = 'unknown'
            return self
         speedGbps = EthSpeedApi.speedMbps( autonegStatus.speed ) // 1000
         lanes = PhyModel.ethLaneCountToStr[ autonegStatus.laneCount ]
         self.value = f'{speedGbps}G-{lanes}'
      return self

   def render( self, nest=0 ):
      _printLine( 'Speed', self.value, self.changes, self.lastChange, nest=nest )

class FecMode( CliModel.Model ):
   value = CliModel.Enum( values=fecEncodingToStr.keys(),
                          help="Resolved FEC mode" )
   changes = CliModel.Int( help="The number of times the value changed" )
   lastChange = CliModel.Float( help="The last time the value changed" )

   def render( self, nest=0 ):
      _printLine( 'FEC', fecEncodingToStr[ self.value ], self.changes,
                  self.lastChange, nest=nest )

class FlowcontrolStatus( CliModel.Model ):
   value = CliModel.Enum( values=flowcontrolStatusToStr.values(),
                          help="Flowcontrol status" )
   changes = CliModel.Int( help="The number of times the value changed" )
   lastChange = CliModel.Float( help="The last time the value changed" )

   def renderWithLabel( self, label, nest=0 ):
      _printLine( label, self.value, self.changes, self.lastChange, nest=nest )

class NegotiationResolution( CliModel.Model ):
   linkMode = CliModel.Submodel( valueType=LinkMode,
                                 help="Resolved link mode" )
   flowControlRx = CliModel.Submodel( valueType=FlowcontrolStatus,
                                      help="Rx flow control mode" )
   flowControlTx = CliModel.Submodel( valueType=FlowcontrolStatus,
                                      help="Tx flow control mode" )
   fec = CliModel.Submodel( valueType=FecMode,
                            help="Resolved FEC mode" )

   def toModel( self, autonegStatus ):
      self.linkMode = LinkMode().toModel( autonegStatus )
      self.flowControlRx = FlowcontrolStatus(
            value=flowcontrolStatusToStr[ autonegStatus.rxFlowcontrol ],
            changes=autonegStatus.rxFlowcontrolChanges.count,
            lastChange=Ark.switchTimeToUtc(
               autonegStatus.rxFlowcontrolChanges.last ) )
      self.flowControlTx = FlowcontrolStatus(
            value=flowcontrolStatusToStr[ autonegStatus.txFlowcontrol ],
            changes=autonegStatus.txFlowcontrolChanges.count,
            lastChange=Ark.switchTimeToUtc(
               autonegStatus.txFlowcontrolChanges.last ) )
      self.fec = FecMode(
         value=autonegStatus.fecEncoding,
         changes=autonegStatus.fecEncodingChanges.count,
         lastChange=Ark.switchTimeToUtc( autonegStatus.fecEncodingChanges.last ) )
      return self

   def render( self, nest=0 ):
      self.linkMode.render( nest=nest )
      self.flowControlRx.renderWithLabel( 'Flow control Rx', nest=nest )
      self.flowControlTx.renderWithLabel( 'Flow control Tx', nest=nest )
      self.fec.render( nest=nest )


class AutonegStatesBase( CliModel.Model ):
   def toModel( self, anDebug ):
      for attr in self:
         anDebugAttr = getattr( anDebug, attr, None )
         if anDebugAttr is None:
            raise Exception( "All attrs in Model should exist in the Tac type." )

         if isinstance(anDebugAttr, ConditionTac ):
            setattr( self, attr, ConditionAndChangedStat().toModel( anDebugAttr ) )
         elif isinstance(anDebugAttr, EventStatTac ):
            setattr( self, attr, EventStat().toModel( anDebugAttr ) )
         else:
            raise Exception( f"Field type { type( anDebugAttr ) } not handled." )
      return self

class TrainingBase( CliModel.Model ):
   def toModel( self, laneInfo ):
      for attr in self:
         anDebugAttr = getattr( laneInfo, attr, None )
         if anDebugAttr is None:
            raise Exception( "All attrs in Model should exist in the Tac type." )

         if isinstance(anDebugAttr, ConditionTac ):
            setattr( self, attr, ConditionAndChangedStat().toModel( anDebugAttr ) )
         elif isinstance(anDebugAttr, EventStatTac ):
            setattr( self, attr, EventStat().toModel( anDebugAttr ) )
         else:
            raise Exception( f"Field type { type( anDebugAttr ) } not handled." )
      return self

class PhyNegotiationStatusDetailed( CliModel.Model ):
   componentName = CliModel.Str( help="Name of the component(phy or transceiver)" )
   anMode = CliModel.Submodel( valueType=NegotiationMode,
                               help="Auto-negotiation mode" )

   anStatus = CliModel.Submodel( valueType=NegotiationStatus,
                                 help="Auto-negotiation status" )

   remoteAdvertisements = \
         CliModel.Submodel( valueType=NegotiationAdvert,
                            help="Remote advertisement information" )

   localAdvertisements = CliModel.Submodel( valueType=NegotiationAdvert,
                                            help="Local advertisement information" )

   resolution = CliModel.Submodel( valueType=NegotiationResolution,
                                   help="Negotiation resolution information" )

   autonegStates = CliModel.Submodel( valueType=AutonegStatesBase,
                                      optional=True,
                                      help="Negotiation state information" )

   training = CliModel.Dict( keyType=int,
                             valueType=TrainingBase,
                             optional=True,
                             help="Training state information, keyed by lane" )

   def toModel( self, compName, autonegStatusPi, autonegPd, autonegStatesModel,
                trainingModel ):
      self.componentName = compName
      self.anMode = NegotiationMode(
         value=autonegStatusPi.mode,
         changes=autonegStatusPi.modeChanges.count,
         lastChange=Ark.switchTimeToUtc( autonegStatusPi.modeChanges.last ) )
      self.anStatus = NegotiationStatus(
         value=autonegStatusPi.state,
         changes=autonegStatusPi.stateChanges.count,
         lastChange=Ark.switchTimeToUtc( autonegStatusPi.stateChanges.last ) )
      self.remoteAdvertisements = NegotiationAdvert().toModel(
            autonegStatusPi.partnerAdvertisement,
            autonegStatusPi.partnerAdvertisementChanges.count,
            Ark.switchTimeToUtc( autonegStatusPi.partnerAdvertisementChanges.last ),
            AutonegFecAdvertisement(),
            AutonegFecAdvertisement() )
      self.localAdvertisements = NegotiationAdvert().toModel(
            autonegStatusPi.advertisement,
            autonegStatusPi.advertisementChanges.count,
            Ark.switchTimeToUtc( autonegStatusPi.advertisementChanges.last ),
            autonegStatusPi.ieeeFecLocalAdv, autonegStatusPi.consortiumFecLocalAdv )
      self.resolution = NegotiationResolution().toModel( autonegStatusPi )
      
      if autonegPd:
         if autonegStatesModel:
            self.autonegStates = autonegStatesModel().toModel( autonegPd )
         if trainingModel:
            for lane, laneInfo in autonegPd.trainingStatusInfo.items():
               self.training[ lane ] = trainingModel().toModel( laneInfo )
      return self

   def render( self, nest=0 ):
      upperHeadings = ( '', 'Current State', 'Changes', 'Last Change' )
      lowerHeadings = tuple( '-' * len ( h ) for h in upperHeadings )
      _printLine( *upperHeadings )
      _printLine( *lowerHeadings )
      self.anMode.render( nest=nest )
      self.anStatus.render( nest=nest )
      _printLine( 'Advertisements', nest=nest )
      self.localAdvertisements.renderWithSide( side='Local', nest=nest + 1 )
      self.remoteAdvertisements.renderWithSide( side='Partner', nest=nest + 1 )
      _printLine( 'Resolution', nest=nest )
      self.resolution.render( nest=nest + 1 )
      if self.autonegStates:
         self.autonegStates.render( nest=nest )
      if self.training:
         _printLine( 'Link training progress', nest=nest )
         for lane, info in sorted( self.training.items() ):
            _printLine( f"Lane {lane}", nest=nest + 1 )
            info.render( nest=nest + 2 )

class IntfNegotiationStatusDetailed( CliModel.Model ):
   phyNegotiationStatuses = CliModel.List(
      valueType=PhyNegotiationStatusDetailed,
      help='Collection of phy/xcvr specific negotiation information' )

   def render( self, nest=0 ):
      for phyNegoSts in self.phyNegotiationStatuses:
         if phyNegoSts.anStatus.value == 'anegStateOff':
            print( f"  {nest * '  ' + phyNegoSts.componentName}: "
                   f"Auto-Negotiation Disabled" )
            continue
         _printLine( phyNegoSts.componentName, nest=nest )
         phyNegoSts.render( nest=nest + 1 )

class NegotiationStatusesDetailed( CliModel.Model ):
   """Show a detailed negotiation status of the phy interfaces"""
   negotiationStatuses = CliModel.Dict(
      keyType=IntfModels.Interface,
      valueType=IntfNegotiationStatusDetailed,
      help='Collection of interface negotiation information, '
           'keyed by interface name' )

   def render( self ):
      if self.negotiationStatuses:
         print( 'Current System Time:', time.ctime( Tac.utcNow() ) )
      for interfaceName in Arnet.sortIntf( self.negotiationStatuses ):
         if all( phyNegoSts.anStatus.value == 'anegStateOff'
                 for phyNegoSts in
                 self.negotiationStatuses[ interfaceName ].phyNegotiationStatuses ):
            print( f"{ interfaceName }: Auto-Negotiation Disabled" )
            continue
         print( interfaceName )
         self.negotiationStatuses[ interfaceName ].render( nest=0 )
