# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from IntfModels import Interface
from CliPlugin.PhyStatusModel import phyDetailInfoFmt
from CliModel import Enum, Bool, Dict, Model, Submodel
from Arnet import sortIntf

class PrecodingTxRxStatus( Model ):
   txEnabled = Bool( help="Precoding is enabled on transmitter" )
   rxEnabled = Bool( help="Precoding is enabled on receiver" )

   def toModel( self, hwStat ):
      self.txEnabled = hwStat.precodingTxEnabled
      self.rxEnabled = hwStat.precodingRxEnabled
      return self

   def statusString( self ):
      status = []
      if self.txEnabled:
         status.append( 'tx' )
      if self.rxEnabled:
         status.append( 'rx' )
      return ','.join( status ) or 'none'

   def boolToOnOrOff( self, attribute ):
      if attribute:
         return 'on'
      return 'off'

class PrecodingRequestStatus( Model ):
   requestedOfPeer = Bool( help="Local device requests precoding" )
   requestedByPeer = Bool( help="Link partner requests precoding" )

   def toModel( self, hwStat ):
      self.requestedByPeer = hwStat.requestedByPeer
      self.requestedOfPeer = hwStat.requestedOfPeer
      return self

   def requestString( self, cl72Enabled ):
      if not cl72Enabled:
         return 'n/a'

      requested = []
      if self.requestedOfPeer:
         requested.append( 'local' )
      if self.requestedByPeer:
         requested.append( 'partner' )
      return ','.join( requested ) or 'none'

class PrecodingLaneDetail( Model ):
   txRxStatus = Submodel( valueType=PrecodingTxRxStatus,
                          help="Precoding status of Tx and Rx on per-lane basis" )
   requestStatus = Submodel( valueType=PrecodingRequestStatus,
                             help="Precoding request status on per-lane basis" )

class InterfacePrecoding( Model ):
   lanes = Dict( keyType=int,
                 valueType=PrecodingLaneDetail,
                 help="Detailed precoding information on per-lane basis" )
   txConfig = Enum( values=( 'default', 'on', 'off' ),
                    help="Configured precoding on transmitter" )
   rxConfig = Enum( values=( 'default', 'on', 'off' ),
                    help="Configured precoding on receiver" )
   _cl72Enabled = Bool( 'CL72 link training is enabled' )

class InterfacesPrecoding( Model ):
   interfaces = Dict( keyType=Interface, valueType=InterfacePrecoding,
         help="Mapping between an interface and phy precoding information" )

   def render( self ):
      if not self.interfaces:
         return
      hdrFmt = '%-16s %-4s %-8s %-7s %-8s %-7s %-19s'
      # pylint: disable-next=consider-using-f-string
      print( "%35s %16s" % ( "Configured", "Operational" ) )
      print( hdrFmt % ( 'Interface', 'Lane', 'Transmit', 'Receive',
                       'Transmit', 'Receive', 'Requested by device' ) )
      print( hdrFmt % ( '-' * 16, '-' * 4, '-' * 8, '-' * 7,
                       '-' * 8, '-' * 7, '-' * 19 ) )
      for intf in sortIntf( self.interfaces ):
         data = self.interfaces[ intf ]
         for lane, laneStatus in sorted( data.lanes.items() ):
            txRxStatus = laneStatus.txRxStatus
            print( hdrFmt % ( intf,
                             lane,
                             data.txConfig,
                             data.rxConfig,
                             txRxStatus.boolToOnOrOff( txRxStatus.txEnabled ),
                             txRxStatus.boolToOnOrOff( txRxStatus.rxEnabled ),
                             laneStatus.requestStatus.requestString(
                                data._cl72Enabled ) ) ) # pylint: disable-msg=W0212

class PrecodingTxRxStatuses( Model ):
   txRxStatus = Dict( keyType=int,
                      valueType=PrecodingTxRxStatus,
                      help="Precoding information on per-lane basis" )

   def render( self ):
      print( phyDetailInfoFmt % ( "Active Precoding", "" ) )
      for lane, status in sorted( self.txRxStatus.items() ):
         # pylint: disable-next=consider-using-f-string
         print( phyDetailInfoFmt % ( '  Lane %d' % lane, status.statusString() ) )
