#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Ark, CliModel, Tac
from CliPlugin import EventsModel
from IntfModels import Interface
from Arnet import sortIntf
from time import ctime
from decimal import Decimal
import string

phyDetailStatFmt = '  %-27.27s %-16.16s %8.8s %22.22s'
phyDetailInfoFmt = '  %-27.27s %s'
phyDetailLongHeaderFmt = '  %-40.40s'
phyHeaderFmt = ' %s'

testPatternStrings = ( "patternNone", "prbs7", "prbs9", "prbs11", "prbs13", "prbs15",
                       "prbs23", "prbs31", "prbs49", "prbs58", "prbs63" )

phyStateToCliStr = {
   'unknown' : 'unknown',
   'init' : 'init',
   'detectingXcvr' : 'detectingXcvr',
   'autonegotiating' : 'autonegotiating',
   'linkDown' : 'linkDown',
   'linkUp' : 'linkUp',
   'errDisabled' : 'errDisabled'
   }

lockLostToStr = {
   'unknown' : 'unknown',
   'waitingForLock' : 'waiting for lock',
   'notLost' : 'not lost',
   'lost' : 'lost',
   }

# The set of printable characters to be considered. Avoid '\r' as it can overwrite
# screen output.
printableChars = set( string.printable ) - set( '\r\x0b\x0c' )

# output can have non-printable characters so properly escape.
def escapeNonPrintable( strData ):
   # Format string to be printable-friendly. Characters that don't print
   # well are replaced by a backslash plus 'x' plus their two-byte hex value.
   return "".join( ch if ch in printableChars else
                   '\\x%02x' % ord( ch ) for ch in strData )

class LaneTestPatternCounters( CliModel.Model ):
   linkState = CliModel.Enum(
         values=( "notApplicable", "notLocked", "locked" ),
         help="Lock status of received diagnostic test pattern" )
   bitErrors = CliModel.Int( help="Total bit errors" )
   largestBurst = CliModel.Int( help="Largest burst of bit errors" )
   burstCount = CliModel.Int( help="Number of bursts of bit errors" )
   lastErrorTime = CliModel.Float( help="Last time bit errors were observed" )

   def lockStToStr( self, locked="notApplicable" ):
      valtoStr =  {
                     "notApplicable"    : "N/A",
                     "notLocked"        : "not locked",
                     "locked"           : "locked",
                  }
      return valtoStr[ locked ]

   def toModel( self, event ):
      self.linkState = event.locked
      self.bitErrors = event.errors.total
      self.largestBurst = event.errors.largestBurst
      self.burstCount = event.errors.changes
      self.lastErrorTime = Ark.switchTimeToUtc( event.errors.lastChange )
      return self

   def render( self, intf, lane ): # pylint: disable-msg=W0221
      hdrFmt = '%-16s %-5s %-11s %-12s %-10s %-8s %-17s'
      print( hdrFmt % ( intf, lane,
                       self.lockStToStr( self.linkState ),
                       self.bitErrors,
                       self.largestBurst,
                       self.burstCount,
                       Ark.utcTimeRelativeToNowStr(
                          self.lastErrorTime ) ) )

class IntfTestPatternCounters( CliModel.Model ):
   lanes = CliModel.Dict(
         keyType=int,
         valueType=LaneTestPatternCounters,
         help="Mapping of lanes that are a part of the interface to test pattern "
              "counter data." )

class TestPatternCounters( CliModel.Model ):
   interfaces = CliModel.Dict(
         keyType=Interface,
         valueType=IntfTestPatternCounters,
         help="Mapping of interface to test pattern counter data." )

   def render( self ):
      if not self.interfaces:
         return

      print( ( 'Current System Time: ' ) + str( ctime ( Tac.utcNow() ) ) )
      hdrFmt = '%-16s %-5s %-11s %-12s %-10s %-8s %-17s'
      print( " %54s %8s " % ( 'Largest', 'Burst' ) )
      print( hdrFmt % ( 'Interface', 'Lane', 'Link State', 'Bit Errors', 'Burst',
                       'Count', 'Last Error Time' ) )
      print( hdrFmt % ( '-' * 16, '-' * 5, '-' * 11, '-' * 12, '-' * 10, '-' * 8,
                       '-' * 17 ) )
      for intf in sortIntf( self.interfaces ):
         status = self.interfaces[ intf ]
         for lane, laneSt in sorted( status.lanes.items() ):
            laneSt.render( intf, lane )

class LaneTestPatternDetail( LaneTestPatternCounters ):
   linkStateChanges = CliModel.Int( help="Number of link state changes since "
                                         "last clear" )
   lastLinkStateChange = CliModel.Float( help="Time of last link state change" )
   lockLostLatchState = CliModel.Enum(
         values=lockLostToStr,
         help=( "Status of the lock latch since RX became operational. "
                "Reset when a new RX pattern becomes operational." ) )
   bitRate = CliModel.Float( help="Bit rate of the lane in Gbps" )
   totalBits = CliModel.Float( help="Total bits received since last clear in Gb" )
   bitErrorRate = CliModel.Float( help="Bit error rate since last clear" )
   bitErrorsSinceLastLock =  CliModel.Int( help="Total bit errors since last lock" )
   totalBitsSinceLastLock = CliModel.Float(
                           help="Total bits received since last lock in Gb" )
   berSinceLastLock = CliModel.Float(
                           help="Bit error rate since last lock" )

   def calculateBer( self, bitRate, bitErrors, startTime ):
      testDuration = Tac.now() - startTime
      # bitRate is in units of Gbps. Convert it  before calculating BER.
      totalBits = bitRate * 1e9 * testDuration
      if not bitErrors:
         bitErrors = 1
      return float( bitErrors / totalBits )

   def calculateTotalBits( self, bitRate, startTime ):
      testDuration = Tac.now() - startTime
      totalBits = bitRate * testDuration
      return totalBits

   def getPrefixSuffix( self, linkState, bitErrors, linkStateChanges=0 ):
      # NOTE: on the prefixes and suffixes used.
      # - suffix '*', when the counter values are approximated due to loss of lock.
      # - when we detect 0 bit errors, prefix '<' to denote the projected 
      #   BER as " < displayed_error "
      # - when the checker loses the lock, counters stop capturing the errors.
      #   Prefix '>' to denote the projected BER as " > found_value"
      prefix = ""
      suffix = ""
      if linkState == "notLocked" or linkStateChanges > 1:
         suffix = "*"
      if linkState == "notLocked":
         prefix = "> "
      elif not bitErrors:
         prefix = "< "
      return prefix, suffix

   # pylint: disable-msg=W0221
   def toModel( self, event, startTime ):
      super().toModel( event )
      self.linkStateChanges = event.linkStateChanges
      self.lastLinkStateChange = Ark.switchTimeToUtc( event.lastLinkStateChange )
      self.lockLostLatchState = event.lockLost
      self.bitRate = event.bitRate
      self.totalBits = self.calculateTotalBits( self.bitRate, startTime )
      self.bitErrorRate = self.calculateBer( self.bitRate,
                                             self.bitErrors, startTime )
      self.bitErrorsSinceLastLock = event.errorsSinceLastLock
      self.totalBitsSinceLastLock = self.calculateTotalBits(
                                       self.bitRate, event.lastLockTime )
      self.berSinceLastLock = self.calculateBer( self.bitRate,
                                                 self.bitErrorsSinceLastLock,
                                                 event.lastLockTime )
      return self

   # pylint: disable-msg=W0221
   def render( self ):
      infoFmt = "     %-27.27s %s"
      statFmt = "     %-27.27s %-16.16s %8.8s %22.22s"
      prefix, suffix = self.getPrefixSuffix( self.linkState,
                                             self.bitErrors,
                                             self.linkStateChanges )
      print( infoFmt % ( 'Bit rate',
                        f"{self.bitRate:,.3f} Gbps" ) )
      print( statFmt % ( 'Lock state',
                        self.lockStToStr( self.linkState ),
                        self.linkStateChanges,
                        Ark.utcTimeRelativeToNowStr( self.lastLinkStateChange ) ) )
      print( infoFmt % ( 'Lock lost latch',
                        lockLostToStr[ self.lockLostLatchState ] ) )
      print( infoFmt % ( 'Largest burst', self.largestBurst ) )
      print( statFmt % ( 'Bit errors', str( self.bitErrors ) + str( suffix ),
                        self.burstCount,
                        Ark.utcTimeRelativeToNowStr( self.lastErrorTime ) ) )
      print( infoFmt % ( 'Total bits',
                        f"{self.totalBits:,.3f} Gb" ) )
      expVal = "%.2E" % Decimal ( self.bitErrorRate )
      print( "     %-27.27s %s%s%s" % ( 'Bit error rate',
                                       prefix, expVal, suffix ) )
      prefix, suffix = self.getPrefixSuffix( self.linkState,
                                             self.bitErrors )
      print( "     %-27.27s %s%s" % ( 'Bit errors since last lock',
                                     self.bitErrorsSinceLastLock , suffix ) )
      print( infoFmt % ( 'Total bits since last lock',
                        f"{self.totalBitsSinceLastLock:,.3f} Gb" ) )
      expVal = "%.2E" % Decimal ( self.berSinceLastLock )
      print( "     %-27.27s %s%s%s" % ( 'BER since last lock',
                                       prefix, expVal, suffix ) )

class IntfTestPatternDetail( CliModel.Model ):
   lanes = CliModel.Dict( keyType=int, valueType=LaneTestPatternDetail,
                          help="Mapping of lanes to detailed test pattern status" )
   lastClear = CliModel.Float( help="The last time the test pattern counters"
                                    "were cleared" )
   operationalTestPattern = CliModel.Enum( values=testPatternStrings,
                                      optional=True,
                                      help="Operational test pattern" )

class TestPatternDetail( CliModel.Model ):
   interfaces = CliModel.Dict(
         keyType=Interface,
         valueType=IntfTestPatternDetail,
         help="Mapping of interface to detailed test pattern data" )

   def render( self ):
      if not self.interfaces:
         return

      statFmt = '%-32s %-16s %8s %22s'
      infoFmt = "  %-30.27s %s"
      print( "*: Data may not be accurate due to loss of lock.\n" )
      print( 'Current System Time: ', ctime( Tac.utcNow() ) )
      for intf in sortIntf( self.interfaces ):
         status = self.interfaces[ intf ]
         print( intf )
         print( infoFmt % ( 'Last clear',
                           Ark.utcTimeRelativeToNowStr( status.lastClear ) ) )
         print( infoFmt % ( 'Operational test pattern',
                           status.operationalTestPattern.upper()
                              if ( status.operationalTestPattern != "patternNone" )
                              else "None" ) )
         print( statFmt % ( '', 'Current State', 'Changes', 'Last Change' ) )
         print( statFmt % ( '', '-' * 13, '-' * 7, '-' * 11 ) )
         for lane, laneSt in sorted( status.lanes.items() ):
            print( "  Lane %d" % lane )
            laneSt.render()

class TestPatternMode( CliModel.Model ):
   configured = CliModel.Enum(
                  values=testPatternStrings,
                  optional=True,
                  help="Configured test pattern mode" )
   operational = CliModel.Enum(
                  values=testPatternStrings,
                  optional=True,
                  help="Operational test pattern mode" )

class IntfTestPatternStatus( CliModel.Model ):
   tx = CliModel.Submodel( valueType= TestPatternMode, optional=True,
                           help="Transmit test pattern mode" )
   rx = CliModel.Submodel( valueType= TestPatternMode, optional=True,
                           help="Receive test pattern mode" )
   capabilities = CliModel.Str( help="Supported test-patterns" )

   def patternToString( self, pattern ):
      valtoStr = {
                     "patternNone"  : "None",
                     "prbs7"        : "PRBS7",
                     "prbs9"        : "PRBS9",
                     "prbs11"       : "PRBS11",
                     "prbs13"       : "PRBS13",
                     "prbs15"       : "PRBS15",
                     "prbs23"       : "PRBS23",
                     "prbs31"       : "PRBS31",
                     "prbs49"       : "PRBS49",
                     "prbs58"       : "PRBS58",
                     "prbs63"       : "PRBS63",
                 }
      return valtoStr[ pattern ]

   def render( self, intf ): # pylint: disable-msg=W0221
      hdrFmt = '%-16s %-8s %-7s %-8s %-7s %-28s'
      print( hdrFmt % ( intf, self.patternToString( self.tx.configured ),
                        self.patternToString( self.rx.configured ),
                        self.patternToString( self.tx.operational ),
                        self.patternToString( self.rx.operational ),
                        self.capabilities ) )

class TestPatternStatus( CliModel.Model ):
   interfaces = CliModel.Dict(
         keyType=Interface,
         valueType=IntfTestPatternStatus,
         help="Mapping between an interface and test pattern config" )

   def render( self ):
      if not self.interfaces:
         return
      hdrFmt = '%-16s %-8s %-7s %-8s %-7s %-28s'
      print( "%30s %17s" % ( "Configured", "Operational" ) )
      print( hdrFmt % ( 'Interface', 'Transmit', 'Receive', 'Transmit',
            'Receive', "Available" ) )
      print( hdrFmt % ( '-' * 16, '-' * 8, '-' * 7, '-' * 8, '-' * 7, '-' * 28 ) )
      for intf in sortIntf( self.interfaces ):
         self.interfaces[ intf ].render( intf )

class PhyChipStatus( CliModel.Model ):
   oui = CliModel.Int( help="Organizationally Unique Identifier (OUI)" )
   model = CliModel.Int( help="Model number" )
   rev = CliModel.Int( help="Revision number" )
   hwRev = CliModel.Str( help="PHY hardware revision", optional=True )
   modelName = CliModel.Str( help="PHY model name", optional=True )

   def toModel( self, phyChip ):
      self.oui = phyChip.oui
      self.model = phyChip.model
      self.rev = phyChip.rev
      if phyChip.hwRev:
         self.hwRev = phyChip.hwRev
      if phyChip.modelName:
         self.modelName = phyChip.modelName
      return self

   def renderPhyDetail( self ):
      modelStr = self.modelName if self.modelName else ''
      if self.hwRev:
         modelStr += " (" + self.hwRev + ")"
      if self.oui or self.model or self.rev:
         # if any of the three register values is not equal to  0 we assume they all
         # have valid entries
         modelStr += " (0x%06x,0x%x,0x%x)" % ( self.oui, self.model, self.rev )
      if modelStr:
         print( phyDetailInfoFmt % ( "Model", modelStr ) )

phyFirmwareStatusToCliStr = {
   "unknown" : "unknown",
   "error" : "error",
   "ok" : "ok",
}

class PhyFirmwareStatus( CliModel.Model ):
   value = CliModel.Enum( values=list( phyFirmwareStatusToCliStr.values() ),
                          help="Current PHY firmware status" )
   changes = CliModel.Int( help="Number of PHY firmware status changes" )
   lastChange = CliModel.Float( help="Last time PHY firmware status changed" )

   def toModel( self, state ):
      self.value = phyFirmwareStatusToCliStr[ state.current ]
      self.changes = state.changes
      self.lastChange = Ark.switchTimeToUtc( state.lastChange )
      return self

   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( "Firmware status",
                                  self.value,
                                  self.changes,
                                  Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class HardwareReset( CliModel.Model ):
   count = CliModel.Int( help="PHY reset counter" )
   lastResetTime = CliModel.Float( help="Last time PHY was reset" )

   def toModel( self, reset ):
      self.count = reset.resetCount
      self.lastResetTime = Ark.switchTimeToUtc( reset.lastResetTime )
      return self

   def renderPhyDetail( self ):
      print(
         phyDetailStatFmt % ( "HW resets", "", self.count,
                              Ark.utcTimeRelativeToNowStr( self.lastResetTime ) ) )

class PhyState( CliModel.Model ):
   value = CliModel.Enum( values=list( phyStateToCliStr.values() ),
                          help="Current PHY state" )
   changes = CliModel.Int( help="Number of PHY state changes" )
   lastChange = CliModel.Float( help="Last time PHY state changed" )

   def toModel( self, state ):
      self.value = phyStateToCliStr[ state.current ]
      self.changes = state.changes
      self.lastChange = Ark.switchTimeToUtc( state.lastChange )
      return self

   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( "PHY state", self.value, self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

interfaceStateToCliStr = {
   "down"      : "down",
   "up"        : "up",
   "adminDown" : "admin down",
}

class InterfaceState( CliModel.Model ):
   current = CliModel.Enum( values=list( interfaceStateToCliStr ),
                            help="Current interface state" )
   changes = CliModel.Int( help="Number of interface state changes" )
   lastChange = CliModel.Float( help="Last time interface state changed" )

   def toModel( self, current, changes, lastChange ):
      self.current = current
      self.changes = changes
      self.lastChange = Ark.switchTimeToUtc( lastChange )
      return self

   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( "Interface state",
                                 interfaceStateToCliStr[ self.current ],
                                 self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class TransceiverMediaType( CliModel.Model ):
   value = CliModel.Str( help="Transceiver media type" )
   changes = CliModel.Int( help="Number of changes in transceiver presence" )
   lastChange = CliModel.Float( help="Last time transceiver presence changed" )

   def toModel( self, mediaType, xcvrStatus ):
      self.value = escapeNonPrintable( mediaType )
      self.changes = xcvrStatus.presenceChanges
      self.lastChange = Ark.switchTimeToUtc( xcvrStatus.lastPresenceChange )
      return self

   def renderPhyDetail( self ):
      if self.value:
         print(
            phyDetailStatFmt % ( "Transceiver", self.value, self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class Transceiver( CliModel.Model ):
   mediaType = CliModel.Submodel( valueType=TransceiverMediaType,
                                  help="Transceiver media type" )
   vendorSn = CliModel.Str( help="Vendor serial number of transceiver",
                            optional=True )

   def toModel( self, mediaType, xcvrStatus ):
      self.mediaType = TransceiverMediaType().toModel( mediaType, xcvrStatus )
      if xcvrStatus.swappability != 'fixed' and xcvrStatus.presence == 'xcvrPresent':
         if xcvrStatus.vendorInfo.vendorSn:
            self.vendorSn = escapeNonPrintable( xcvrStatus.vendorInfo.vendorSn )
      return self

   def renderPhyDetail( self ):
      self.mediaType.renderPhyDetail()
      if self.vendorSn:
         print( phyDetailInfoFmt % ( "Transceiver SN", self.vendorSn ) )

class LinkStatus( CliModel.Model ):
   value = CliModel.Enum( values=( "up", "down" ), help="The status of the link" )
   changes = CliModel.Int( help="The number of times the link status changed" )
   lastChange = CliModel.Float( help="The last time the link status changed" )

   def toModel( self, event ):
      self.value = "up" if event.current else "down"
      self.changes = event.changes
      self.lastChange = Ark.switchTimeToUtc( event.lastChange )
      return self

class TxStatus( CliModel.Model ):
   value = CliModel.Enum( values=( "on", "off" ), help="The transmit status" )
   changes = CliModel.Int( help="The number of times the transmit status changed" )
   lastChange = CliModel.Float( help="The last time the transmit status changed" )

   def toModel( self, event ):
      self.value = "on" if event.current else "off"
      self.changes = event.changes
      self.lastChange = Ark.switchTimeToUtc( event.lastChange )
      return self

class Pma( CliModel.Model ):
   pass

class PmaLinkStatus( LinkStatus ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'PMA/PMD RX link status',
      self.value, self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class PmaTxStatus( TxStatus ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'PMA/PMD TX status', self.value, self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class PmaTxFault( EventsModel.ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'PMA/PMD TX fault',
      'fault' if self.value else 'ok',
                                 self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class PmaRxFault( EventsModel.ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'PMA/PMD RX fault',
      'fault' if self.value else 'ok',
                                 self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

def renderPmaLaneSignalDetect( modelDict ):
   print( phyDetailLongHeaderFmt % ( "PMA/PMD lane RX signal detect" ) )
   for lane, signalDetect in sorted( modelDict.items() ):
      print( phyDetailStatFmt % ( '  Lane %d' % lane,
      'ok' if signalDetect.value else
                                 'no signal', signalDetect.changes,
                                 Ark.utcTimeRelativeToNowStr(
                                    signalDetect.lastChange ) ) )

class PmaSignalDetect( EventsModel.ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'PMA/PMD RX signal detect',
      'ok' if self.value else
                                 'no signal', self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class LpInformationValid( EventsModel.ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'Link partner information',
                                 'valid' if self.value else 'invalid', self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class PmaClause45( Pma ):
   linkStatus = CliModel.Submodel( valueType=PmaLinkStatus, optional=True,
                                   help="PMA link status" )
   txFault = CliModel.Submodel( valueType=PmaTxFault, optional=True,
                                help="PMA transmit fault" )
   rxFault = CliModel.Submodel( valueType=PmaRxFault, optional=True,
                                help="PMA receive fault" )
   signalDetect = CliModel.Submodel( valueType=PmaSignalDetect, optional=True,
                                     help="PMA signal detect" )
   laneSignalDetect = CliModel.Dict( keyType=int, valueType=PmaSignalDetect,
                                     optional=True, help="PMA lane signal detect" )

   def renderPhyDetail( self ):
      if self.linkStatus:
         self.linkStatus.renderPhyDetail()
      if self.txFault:
         self.txFault.renderPhyDetail()
      if self.rxFault:
         self.rxFault.renderPhyDetail()
      if self.signalDetect:
         self.signalDetect.renderPhyDetail()
      if self.laneSignalDetect:
         renderPmaLaneSignalDetect( self.laneSignalDetect )

   def toPhySummary( self ):
      if self.linkStatus:
         pmaPmdBits = { "down" : "D", "up" : "U" }[ self.linkStatus.value ]
      else:
         pmaPmdBits = "."
      pmaPmdBits += 'R' if self.rxFault and self.rxFault.value else '.'
      pmaPmdBits += 'T' if self.txFault and self.txFault.value else '.'

      return pmaPmdBits

def renderPmaLaneLinkStatus( modelDict ):
   print( phyDetailInfoFmt % ( "PMA/PMD lane RX link status", "" ) )
   for lane, linkStatus in sorted( modelDict.items() ):
      print( phyDetailStatFmt % (
         "  Lane %d" % lane, linkStatus.value, linkStatus.changes,
         Ark.utcTimeRelativeToNowStr( linkStatus.lastChange ) ) )

def renderPmaLaneTxStatus( modelDict ):
   print( phyDetailInfoFmt % ( "PMA/PMD lane TX status", "" ) )
   for lane, txStatus in sorted( modelDict.items() ):
      print( phyDetailStatFmt % (
         "  Lane %d" % lane, txStatus.value, txStatus.changes,
         Ark.utcTimeRelativeToNowStr( txStatus.lastChange ) ) )

class PmaBaseR( PmaClause45 ):
   # laneLinkStatus isn't in IEEE but many phys implement it.
   laneLinkStatus = CliModel.Dict( keyType=int,
                                   valueType=PmaLinkStatus,
                                   optional=True, help="PMA lane link status" )

class PmaBaseT( PmaClause45 ):
   linkPartnerInfoValid = CliModel.Submodel( valueType=LpInformationValid,
                                             optional=True,
                                             help="Whether or not link partner \
                                                   information is valid" )

fecEncodings = { 'reedSolomon' : 'Reed-Solomon', 'fireCode' : 'Fire-Code' }
class Fec( CliModel.Model ):
   encoding = CliModel.Enum( help='The FEC encoding', values=list( fecEncodings ) )

class Pcs( CliModel.Model ):
   pass

class PcsLinkStatus( LinkStatus ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'PCS RX link status', self.value, self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class PcsTxFault( EventsModel.ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'PCS TX fault', 'fault' if self.value else 'ok',
                                 self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class PcsRxFault( EventsModel.ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'PCS RX fault', 'fault' if self.value else 'ok',
                                 self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class PcsHighBer( EventsModel.ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'PCS high BER',
      'high BER' if self.value else 'ok',
                                 self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class PcsBlockLock( EventsModel.ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'PCS block lock', 'ok' if self.value else
                                 'no block lock', self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class PcsLastHighBerCount( EventsModel.CountAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'PCS BER', self.value, self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class PcsLastErroredBlockCount( EventsModel.CountAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'PCS err blocks', self.value, self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class PcsRxLpi( EventsModel.ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( "PCS RX LPI received",
                                 "true" if self.value else "false",
                                 self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class PcsClause45( Pcs ):
   linkStatus = CliModel.Submodel( valueType=PcsLinkStatus, optional=True,
                                   help="PCS link status" )
   rxLpi = CliModel.Submodel(
      valueType=PcsRxLpi, optional=True, help="PCS has received LPI signaling" )
   txFault = CliModel.Submodel( valueType=PcsTxFault, optional=True,
                                help="PCS transmit fault" )
   rxFault = CliModel.Submodel( valueType=PcsRxFault, optional=True,
                                help="PCS receive fault")
   blockLock = CliModel.Submodel( valueType=PcsBlockLock, optional=True,
                                  help="PCS block lock" )
   highBer = CliModel.Submodel( valueType=PcsHighBer, optional=True,
                                help="PCS high BER status" )
   lastHighBerCount = CliModel.Submodel(
      valueType=PcsLastHighBerCount, optional=True, help="PCS last high BER count" )
   lastErroredBlockCount = CliModel.Submodel( valueType=PcsLastErroredBlockCount,
                                              optional=True,
                                              help="PCS last errored block count" )

   def toModel( self, pcs ):
      self.linkStatus = PcsLinkStatus().toModel( pcs.linkStatus )
      self.txFault = PcsTxFault().toModel( pcs.txFault )
      self.rxFault = PcsRxFault().toModel( pcs.rxFault )
      self.blockLock = PcsBlockLock().toModel( pcs.blockLock )
      self.highBer = PcsHighBer().toModel( pcs.highBer )
      self.lastHighBerCount = PcsLastHighBerCount().toModel( pcs.lastHighBerCount )
      self.lastErroredBlockCount = PcsLastErroredBlockCount().toModel(
         pcs.lastErroredBlockCount )
      return self

   def renderPhyDetail( self ):
      if self.linkStatus:
         self.linkStatus.renderPhyDetail()
      if self.rxLpi:
         self.rxLpi.renderPhyDetail()
      if self.rxFault:
         self.rxFault.renderPhyDetail()
      if self.txFault:
         self.txFault.renderPhyDetail()
      if self.blockLock:
         self.blockLock.renderPhyDetail()
      if self.highBer:
         self.highBer.renderPhyDetail()
      if self.lastErroredBlockCount:
         self.lastErroredBlockCount.renderPhyDetail()
      if self.lastHighBerCount:
         self.lastHighBerCount.renderPhyDetail()

   def toPhySummary( self ):
      if self.linkStatus:
         pcsBits = { "down" : "D", "up" : "U" }[ self.linkStatus.value ]
      else:
         pcsBits = "."
      pcsBits += "R" if self.rxFault and self.rxFault.value else "."
      pcsBits += "T" if self.txFault and self.txFault.value else "."
      pcsBits += "B" if self.highBer and self.highBer.value else "."
      pcsBits += "." if self.blockLock and self.blockLock.value else "L"

      return pcsBits

class PcsDeskew( EventsModel.ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( 'PCS deskew',
                                 "ok" if self.value else "not deskewed",
                                 self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class PcsBaseR( PcsClause45 ):
   deskew = CliModel.Submodel( valueType=PcsDeskew, optional=True,
                                help="PCS deskew status" )

   def renderPhyDetail( self ):
      if self.linkStatus:
         self.linkStatus.renderPhyDetail()
      if self.rxLpi:
         self.rxLpi.renderPhyDetail()
      if self.rxFault:
         self.rxFault.renderPhyDetail()
      if self.txFault:
         self.txFault.renderPhyDetail()
      if self.deskew:
         self.deskew.renderPhyDetail()
      if self.blockLock:
         self.blockLock.renderPhyDetail()
      if self.highBer:
         self.highBer.renderPhyDetail()
      if self.lastErroredBlockCount:
         self.lastErroredBlockCount.renderPhyDetail()
      if self.lastHighBerCount:
         self.lastHighBerCount.renderPhyDetail()

def renderPcsLaneBlockLockPhyDetail( modelDict ):
   print( phyDetailInfoFmt % ( 'PCS lane block lock', '' ) )
   for lane, blockLock in sorted( modelDict.items() ):
      print( phyDetailStatFmt % ( '  Lane %d' % lane, 'ok' if blockLock.value else
                                 'no block lock', blockLock.changes,
                                 Ark.utcTimeRelativeToNowStr(
                                    blockLock.lastChange ) ) )

def renderPcsLaneAlignmentMarkerLockPhyDetail( modelDict ):
   print( phyDetailLongHeaderFmt % 'PCS lane alignment marker lock' )
   for lane, amLock in sorted( modelDict.items() ):
      print( phyDetailStatFmt % ( '  Lane %d' % lane, 'ok' if amLock.value else
                                 'no AM lock', amLock.changes,
                                 Ark.utcTimeRelativeToNowStr( amLock.lastChange ) ) )

class PcsAlignmentMarkerLock( EventsModel.ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( "PCS alignment marker lock", "ok" if self.value
                                 else "no AM lock", self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

def renderPcsBipErrorsPhyDetail( modelDict ):
   print( phyDetailInfoFmt % ( 'PCS lane BIP errors', '' ) )
   for lane, errors in sorted( modelDict.items() ):
      print( phyDetailStatFmt % ( '  Lane %d' % lane, errors.value, errors.changes,
                                 Ark.utcTimeRelativeToNowStr( errors.lastChange ) ) )

class PcsBipErrors( EventsModel.CountAndChangedStat ):
   pass

def renderPcsLaneMapPhyDetail( modelDict ):
   localLanes = []
   peerLanes = []
   print( phyDetailInfoFmt % ( "PCS lane mapping", "" ) )
   for localLane, peerLane in sorted( modelDict.items() ):
      localLanes.append( '%02d' % localLane )
      peerLanes.append( '%02d' % peerLane )
   print( phyDetailInfoFmt % ( 'PCS lane', " ".join( localLanes ) ) )
   print( phyDetailInfoFmt % ( 'PCS received lane', " ".join( peerLanes ) ) )

def renderPcsLaneDeskewPhyDetail( modelDict ):
   print( phyDetailInfoFmt % ( "PCS lane deskew", "" ) )
   for lane, status in sorted( modelDict.items() ):
      print( phyDetailStatFmt % ( "  Lane %d" % lane,
                                 "ok" if status.value else "not deskewed",
                                 status.changes,
                                 Ark.utcTimeRelativeToNowStr( status.lastChange ) ) )

class MultiLanePcsBaseR( PcsBaseR ):
   laneBlockLock = CliModel.Dict( keyType=int, valueType=PcsBlockLock,
                                  optional=True, help="PCS lane block lock" )
   laneDeskew = CliModel.Dict( keyType=int, valueType=PcsDeskew, optional=True,
                               help="PCS lane deskew status" )
   laneAlignmentMarkerLock = CliModel.Dict(
      keyType=int, valueType=PcsAlignmentMarkerLock, optional=True,
      help="PCS lane alignment marker lock" )
   bipErrors = CliModel.Dict( keyType=int, valueType=PcsBipErrors,
                              optional=True, help="PCS BIP errors" )
   laneMap = CliModel.Dict( keyType=int, valueType=int, optional=True,
                            help="Local PCS lane to peer PCS lane map" )

   def toModel( self, pcs, **kwargs ): # pylint: disable-msg=W0221
      """
      This toModel method requires the following keyword arguments.

      - numLanes : int
           Number of pcs lanes active.

      This method does NOT populate laneDeskew; so far, not many phys use it.
      """
      super().toModel( pcs )
      assert "numLanes" in kwargs
      if kwargs[ "numLanes" ] > 1:
         attrModelPairs = ( ( 'laneBlockLock', PcsBlockLock ),
                            ( 'laneAlignmentMarkerLock', PcsAlignmentMarkerLock ),
                            ( 'bipErrors', PcsBipErrors ) )
         for attr, attrModel in attrModelPairs:
            for lane, attrVal in getattr( pcs, attr ).items():
               model = attrModel().toModel( attrVal )
               # pylint: disable=unsupported-assignment-operation
               getattr( self, attr )[ lane ] = model

         for lane, laneMap in pcs.laneMap.items():
            # pylint: disable=unsupported-assignment-operation
            self.laneMap[ lane ] = laneMap
      return self

   def renderPhyDetail( self ):
      if self.linkStatus:
         self.linkStatus.renderPhyDetail()
      if self.rxLpi:
         self.rxLpi.renderPhyDetail()
      if self.rxFault:
         self.rxFault.renderPhyDetail()
      if self.txFault:
         self.txFault.renderPhyDetail()
      if self.deskew:
         self.deskew.renderPhyDetail()
      if self.laneDeskew:
         renderPcsLaneDeskewPhyDetail( self.laneDeskew )
      if self.blockLock:
         self.blockLock.renderPhyDetail()
      if self.highBer:
         self.highBer.renderPhyDetail()
      if self.lastErroredBlockCount:
         self.lastErroredBlockCount.renderPhyDetail()
      if self.lastHighBerCount:
         self.lastHighBerCount.renderPhyDetail()
      if self.laneBlockLock:
         renderPcsLaneBlockLockPhyDetail( self.laneBlockLock )
      if self.laneAlignmentMarkerLock:
         renderPcsLaneAlignmentMarkerLockPhyDetail( self.laneAlignmentMarkerLock )
      if self.bipErrors:
         renderPcsBipErrorsPhyDetail( self.bipErrors )
      if self.laneMap:
         renderPcsLaneMapPhyDetail( self.laneMap )

class Rs( CliModel.Model ):
   pass

class RsLocalFault( EventsModel.ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( "MAC Rx Local Fault", "true" if self.value else
                                 "false", self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class RsRemoteFault( EventsModel.ConditionAndChangedStat ):
   def renderPhyDetail( self ):
      print( phyDetailStatFmt % ( "MAC Rx Remote Fault", "true" if self.value else
                                 "false", self.changes,
                                 Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )

class RsBaseR( Rs ):
   localFault = CliModel.Submodel( valueType=RsLocalFault, optional=True,
                                   help="MAC Local fault" )
   remoteFault = CliModel.Submodel( valueType=RsRemoteFault, optional=True,
                                    help="MAC remote fault" )
   def renderPhyDetail( self ):
      self.localFault.renderPhyDetail()
      self.remoteFault.renderPhyDetail()

   def toModel( self, hwRs ):
      self.localFault = RsLocalFault().toModel( hwRs.localFault )
      self.remoteFault = RsRemoteFault().toModel( hwRs.remoteFault )
      return self

class PmaGen2( PmaBaseR ):
   """Generation 2 PMA block for internal PHYs."""
   
   txStatus = CliModel.Submodel( valueType=PmaTxStatus, optional=True,
                                 help="PMA lane aggregate transmit status" )
   laneTxStatus = CliModel.Dict( keyType=int, valueType=PmaTxStatus,
                                 optional=True, help="PMA per lane transmit status" )

   def toModel( self, pma, numLanes ):
      self.signalDetect = PmaSignalDetect().toModel( pma.signalDetect )
      self.linkStatus = PmaLinkStatus().toModel( pma.linkStatus )
      for lane, signalDetect in pma.laneSignalDetect.items():
         model = PmaSignalDetect().toModel( signalDetect )
         self.laneSignalDetect[ lane ] = model
      for lane, linkStatus in pma.laneLinkStatus.items():
         model = PmaLinkStatus().toModel( linkStatus )
         self.laneLinkStatus[ lane ] = model
      if pma.laneTxStatus:
         self.txStatus = PmaTxStatus().toModel( pma.txStatus )
      for lane, txStatus in pma.laneTxStatus.items():
         model = PmaTxStatus().toModel( txStatus )
         self.laneTxStatus[ lane ] = model
      return self

   def renderPhyDetail( self ):
      if self.linkStatus:
         self.linkStatus.renderPhyDetail()
      if self.laneLinkStatus:
         renderPmaLaneLinkStatus( self.laneLinkStatus )
      if self.txFault:
         self.txFault.renderPhyDetail()
      if self.rxFault:
         self.rxFault.renderPhyDetail()
      if self.signalDetect:
         self.signalDetect.renderPhyDetail()
      if self.laneSignalDetect:
         renderPmaLaneSignalDetect( self.laneSignalDetect )
      if self.txStatus:
         self.txStatus.renderPhyDetail()
      if self.laneTxStatus:
         renderPmaLaneTxStatus( self.laneTxStatus )

class PhyFeatureStatus( CliModel.Model ):
   txClockShift = CliModel.Int(
      help="Configured TX clock shift in parts per million (PPM)",
      optional=True )

   def renderPhyDetail( self ):
      if self.txClockShift:
         print( phyDetailInfoFmt % ( 'Configured TX clock shift',
                                    '%d PPM' % self.txClockShift ) )

class MdiCrossoverModel( CliModel.Model ):
   mdiState = CliModel.Enum( values=( "mdi", "mdix" ), help="MDI/MDI-X layout" )
   changes = CliModel.Int( help="Number of MDI state changes" )
   lastChange = CliModel.Float( help="Last time MDI state changed" )

   def toModel( self, state ):
      self.mdiState = "mdix" if state.current else "mdi"
      self.changes = state.changes
      self.lastChange = Ark.switchTimeToUtc( state.lastChange )
      return self

   def renderPhyDetail( self ):
      print( phyDetailStatFmt % (
             "MDI status",
             "MDI" if self.mdiState == "mdi" else "MDI-X",
             self.changes,
             Ark.utcTimeRelativeToNowStr( self.lastChange ) ) )


