#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import CliCommand
import CliGlobal
from CliPlugin import XcvrAllStatusDir, IntfCli
import CliMatcher
import LazyMount
import MultiRangeRule
import re
from Toggles.PhyEeeToggleLib import togglePhyTxEqProfilesConfigCliCmdEnabled
import Tracing
from XcvrLib import getXcvrStatus

gv = CliGlobal.CliGlobal( phyTxEqProfiles=None, phyFeatureConfigSliceDir=None,
                          xcvrStatusDir=None, xcvrConfigDir=None )
traceHandle = Tracing.Handle( 'PhyTxEqLaneConfigCli' )
t0 = traceHandle.trace0

def intfHelper( intfName, isFixedSystem, intfVar ):
   if isFixedSystem:
      slotMatch = re.match( r'(?:Ethernet|Fabric)(\d+)(/\d+)?', intfName )
   else:
      slotMatch = re.match( r'(?:Ethernet|Fabric)(\d+/\d+)(/\d+)?', intfName )
   if not slotMatch:
      return None
   return { "slotName": slotMatch.group( 1 ),
            "primaryIntf": not slotMatch.group( 2 )
               or slotMatch.group( 2 ) == "/1" }.get( intfVar )

def isPrimaryIntf( intfName ):
   isFixedSystem = Cell.cellType() == "fixed"
   return intfHelper( intfName, isFixedSystem, "primaryIntf" )

def laneRangeFn( mode ):
   intfName = mode.intf.name
   xcvrName = gv.xcvrConfigDir.intfToXcvrName.get( intfName )
   if not xcvrName: # If Fru hasn't started, cli will accept all lanes.
      return ( 0, 7 )
   xcvrStatus = getXcvrStatus( gv.xcvrStatusDir.xcvrStatus.get( xcvrName ) )
   if not xcvrStatus:
      return ( 0, 7 )
   return ( 0, xcvrStatus.capabilities.maxChannels - 1 )

laneRangeMatcher = MultiRangeRule.MultiRangeMatcher(
   rangeFn=laneRangeFn, noSingletons=False, helpdesc='Lane number(s)',
   dynamicRange=True )

profileNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: gv.phyTxEqProfiles.customProfiles, helpname='WORD',
   helpdesc='Profile name' )

# --------------------------------------------------------------------
# "phy transmitter equalization lane LANENUM profile PROFILENAME"
# --------------------------------------------------------------------

class PhyTxEqLaneConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'phy transmitter equalization lane LANENUM profile PROFILENAME'
   noOrDefaultSyntax = 'phy transmitter equalization lane LANENUM ...'
   data = {
      'phy': 'Configure phy-specific parameters',
      'transmitter': 'Configure transmitter parameters',
      'equalization': 'Configure equalization parameters',
      'lane': 'Port Lane(s)',
      'LANENUM': laneRangeMatcher,
      'profile': 'Assign a profile',
      'PROFILENAME': profileNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      if not intfHelper( mode.intf.name, mode.intf.slotId() == "FixedSystem",
                         "primaryIntf" ):
         mode.addWarning( f'Intf: { mode.intf.name } is not a primary interface' )
         return
      phyFeatureConfigDir = gv.phyFeatureConfigSliceDir.get( mode.intf.slotId() )
      cfg = phyFeatureConfigDir.newConfig( mode.intf.name )
      profName = args[ 'PROFILENAME' ]
      if profName in gv.phyTxEqProfiles.customProfiles:
         laneNums = args[ 'LANENUM' ].values()
         for laneNum in laneNums:
            cfg.phyTxEqProfilesConfig[ laneNum + 1 ] = profName
      else:
         mode.addWarning( f'Profile { profName } is not a valid profile' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if not intfHelper( mode.intf.name, mode.intf.slotId() == "FixedSystem",
                         "primaryIntf" ):
         mode.addWarning( f'Intf: { mode.intf.name } is not a primary interface' )
         return
      phyFeatureConfigDir = gv.phyFeatureConfigSliceDir.get( mode.intf.slotId() )
      cfg = phyFeatureConfigDir.newConfig( mode.intf.name )
      laneNums = args[ 'LANENUM' ].values()
      for laneNum in laneNums:
         del cfg.phyTxEqProfilesConfig[ laneNum + 1 ]

if togglePhyTxEqProfilesConfigCliCmdEnabled():
   IntfCli.IntfConfigMode.addCommandClass( PhyTxEqLaneConfigCmd )

# ------------------------------------------------------
# Plugin method
# ------------------------------------------------------

def Plugin( em ):
   gv.phyTxEqProfiles = LazyMount.mount( em,
      'hardware/archer/phy/config/cli/phyTxEqProfiles',
      'Hardware::Phy::PhyTxEqProfiles', 'r' )
   gv.phyFeatureConfigSliceDir = LazyMount.mount(
      em, 'hardware/archer/phy/config/cli/feature/slice', 'Tac::Dir', 'wi' )
   gv.xcvrStatusDir = XcvrAllStatusDir.xcvrAllStatusDir( em )
   gv.xcvrConfigDir = LazyMount.mount( em, "hardware/xcvr/config/all",
                                       "Xcvr::AllConfigDir", "r" )
