#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliModel
import TableOutput

class PhyTxEqSlotLane( CliModel.Model ):
   operational = CliModel.Str( help="Profile that is currently operational" )
   configured = CliModel.List( valueType=str,
      help="Profiles that are currently configured" )

class PhyTxEqSlot( CliModel.Model ):
   lanes = CliModel.Dict(
      keyType=int,
      help="A mapping of transceiver lane numbers to their tuning profile configs",
      valueType=PhyTxEqSlotLane )

class PhyTxEqProfilesModel( CliModel.Model ):
   """Summary model of configured and operational PhyTxEqProfiles"""
   slot = CliModel.Dict(
      keyType=str,
      help="A mapping of transceiver slots to their lane tuning data",
      valueType=PhyTxEqSlot )

   def render( self ):
      headings = ( "Slot", "Lane", "Configured", "Operational" )
      table = TableOutput.createTable( headings )

      leftForm = TableOutput.Format( justify="left", maxWidth=15, wrap=True )
      table.formatColumns( leftForm, leftForm, leftForm, leftForm )
      for fmt in table.columnFormats_:
         fmt.noPadLeftIs( True )
         fmt.padLimitIs( True )

      for slotName, laneData in self.slot.items():
         for laneId, tuningData in laneData.lanes.items():
            configured = ", ".join( tuningData.configured )
            operational = tuningData.operational
            table.newRow( slotName, laneId, configured, operational )
      print( table.output() )
