# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliToken.Clear, CliToken.Ip, CliToken.Pim
import Tac, LazyMount
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliPlugin import PimBidirCliToken
from CliPlugin import PimCliLib
from IpLibConsts import DEFAULT_VRF as vrfDefault

pimBidirClearConfigColl = None
_mfibVrfConfig = None

def validVrfName( mode, name ):
   if name in _mfibVrfConfig.config or name == vrfDefault:
      return True
   else:
      mode.addError("Invalid vrf name")
      return False

pimBidirClearConfigColl = None

def _pimBidirClearConfigCreation( vrfName ):
   pimBidirClearConfigColl.vrfClearConfig.newMember( vrfName )

def _pimBidirClearConfigDeletion( vrfName ):
   if vrfName in pimBidirClearConfigColl.vrfClearConfig and \
         vrfName != vrfDefault:
      del pimBidirClearConfigColl.vrfClearConfig[ vrfName ]

def clearPimMessageCounters( vrfName, af='ipv4' ):
   if vrfName in pimBidirClearConfigColl.vrfClearConfig:
      pimBidirClearConfigColl.vrfClearConfig[ vrfName ].countersCount += 1

#--------------------------------------------------------
# clear ip mroute bidirectional :
#   clear ip mroute bidirectional *
#   clear ip mroute bidirectional G
#----------------------------------------------------
def doClearIpMrouteBidirMode( mode, vrfName=vrfDefault, clearAll=False,
      groupAddr=None, secondAddr=None ):
   args = { 'VRF_NAME': vrfName, 'GROUP_ADDR': groupAddr }
   if clearAll:
      doClearIpMrouteBidirModeAll( mode, args )
      return
   doClearIpMrouteBidirModeGroup( mode, args )

def doClearIpMrouteBidirModeAll( mode, args ):
   vrfName = args.get( 'VRFNAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   if not validVrfName( mode, vrfName ):
      return

   if vrfName in pimBidirClearConfigColl.vrfClearConfig:
      pimBidirClearConfig = pimBidirClearConfigColl.vrfClearConfig[ vrfName ]
   else:
      return

   info = Tac.Value("Routing::Pim::Bidir::ClearConfig::GroupKey")
   flags = Tac.Value("Routing::Pim::Bidir::MrouteClearFlags")

   flags.clearAll = True
   info.flags = flags
   pimBidirClearConfig.groupKey = info

   if pimBidirClearConfig.groupsCount == 0xff:
      pimBidirClearConfig.groupsCount = 0
   else:
      pimBidirClearConfig.groupsCount += 1

   pimBidirClearConfig.groupKey = \
         Tac.Value("Routing::Pim::Bidir::ClearConfig::GroupKey")

def doClearIpMrouteBidirModeGroup( mode, args ):
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   if not validVrfName( mode, vrfName ):
      return

   groupAddr = args[ 'GROUP_ADDR' ]

   if vrfName in pimBidirClearConfigColl.vrfClearConfig:
      pimBidirClearConfig = pimBidirClearConfigColl.vrfClearConfig[ vrfName ]
   else:
      return

   info = Tac.Value("Routing::Pim::Bidir::ClearConfig::GroupKey")
   flags = Tac.Value("Routing::Pim::Bidir::MrouteClearFlags")
   group = None

   try:
      group = PimCliLib.ipPimParseGroup(groupAddr)
   except ValueError:
      mode.addError("Must enter a multicast group")
      return

   if group:
      info.gAddr = group
      flags.clearGroup = True

   info.flags = flags

   pimBidirClearConfig.groupKey = info

   if pimBidirClearConfig.groupsCount == 0xff:
      pimBidirClearConfig.groupsCount = 0
   else:
      pimBidirClearConfig.groupsCount += 1
   pimBidirClearConfig.groupKey = \
         Tac.Value("Routing::Pim::Bidir::ClearConfig::GroupKey")

class ClearIpMrouteBidirModeAllCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip mroute [ vrf VRF_NAME ] bidirectional *'
   data = {
     'clear' : CliToken.Clear.clearKwNode,
     'ip' : CliToken.Ip.ipMatcherForClear,
     'mroute' : CliToken.Pim.mrouteMatcher,
     'vrf' : CliToken.Pim.vrfMatcher,
     'VRF_NAME' : CliToken.Pim.vrfNameMatcher,
     'bidirectional' : CliToken.Pim.bidirMatcher,
     '*' : PimBidirCliToken.allMroutesBidirMatcher
   }

   handler = doClearIpMrouteBidirModeAll

class ClearIpMrouteBidirModeGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip mroute [ vrf VRF_NAME ] bidirectional GROUP_ADDR'
   data = {
     'clear' : CliToken.Clear.clearKwNode,
     'ip' : CliToken.Ip.ipMatcherForClear,
     'mroute' : CliToken.Pim.mrouteMatcher,
     'vrf' : CliToken.Pim.vrfMatcher,
     'VRF_NAME' : CliToken.Pim.vrfNameMatcher,
     'bidirectional' : CliToken.Pim.bidirMatcher,
     'GROUP_ADDR' : PimBidirCliToken.groupBidirMatcher
   }
   handler = doClearIpMrouteBidirModeGroup

BasicCli.EnableMode.addCommandClass( ClearIpMrouteBidirModeAllCmd )
BasicCli.EnableMode.addCommandClass( ClearIpMrouteBidirModeGroupCmd )

def Plugin( entityManager ):
   global pimBidirClearConfigColl
   global _mfibVrfConfig
   pimBidirClearConfigColl = LazyMount.mount(
         entityManager, "routing/pim/bidir/clear/config",
         "Routing::Pim::Bidir::ClearConfigColl", "w")
   _mfibVrfConfig = LazyMount.mount( entityManager,
         'routing/multicast/vrf/config',
         'Routing::Multicast::Fib::VrfConfig', 'r' )
   PimCliLib.pimClearMessageCountersHook.addExtension( clearPimMessageCounters )
   PimCliLib.clearIpMrouteHook.addExtension(
         ( 'bidirMode', doClearIpMrouteBidirMode ) )
   PimCliLib.pimBidirVrfConfiguredHook.addExtension( _pimBidirClearConfigCreation )
   PimCliLib.pimBidirVrfDeletedHook.addExtension( _pimBidirClearConfigDeletion )
