#!/usr/bin/env python3
# Copyright (c) 2010, 2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliPlugin import AclCli
from CliPlugin import IpAddrMatcher
from CliPlugin import MrouteCli
from CliPlugin import PimCliLib
from CliPlugin import RouterMulticastCliLib
from CliPlugin.RouterMulticastCliLib import lazyGetters, AddressFamily
import BasicCli, LazyMount
import Tac
import CliToken.Ip, CliToken.Pim
from McastCommonCliLib import isMulticastRoutingEnabled
from McastCommonCliLib import ipv4NodeForShow, ipv6NodeForShow
import CliCommand

# Globals
_routingHwStatus = None
_pimGlobalStatus  = None

aclStatus = None
aclCheckpoint = None

# Af Independent Config Types
MfibVrfConfig = "Routing::Multicast::Fib::VrfConfig"
PimClearConfig = "Routing::Pim::ClearConfigColl"

( mfibConfigRoot,
  mfibConfigRootFromMode,
  mfibConfig,
  mfibConfigFromMode ) = lazyGetters( MfibVrfConfig, collectionName='config' )

( pimClearConfigRoot,
  pimClearConfigRootFromMode,
  pimClearConfig,
  pimClearConfigFromMode ) = lazyGetters( PimClearConfig,
                                          collectionName='vrfClearConfig',
                                          writeMount=True )

def _vrfDefinitionHook( vrfName ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      pimClearConfigColl = pimClearConfigRoot( af=af )
      pimClearConfig( af=af, vrfName=vrfName )
      assert vrfName in pimClearConfigColl.vrfClearConfig

def _vrfDeletionHook( vrfName ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      pimClearConfigColl = pimClearConfigRoot( af=af )
      if vrfName in pimClearConfigColl.vrfClearConfig:
         del pimClearConfigColl.vrfClearConfig[ vrfName ]

def clearPimMessageCounters( vrfName, af=AddressFamily.ipv4 ):
   pimClearConfig( af, vrfName ).countersCount += 1

def _pimBidirectionalSupported():
   return _routingHwStatus.pimBidirectionalSupported

#------------------------------------------------------------------------------------
# Clear tokens and rules
#------------------------------------------------------------------------------------
groupOrSourceMatcher = IpAddrMatcher.IpAddrMatcher( 'Source or Group address' )

#------------------------------------------------------------------------------------
def doClearPimProtocolCounters( mode, args ):
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   if args.get( 'ip' ):
      af = 'ipv4'
   else:
      af = args.get( 'ipv4' ) or args.get( 'ipv6' )

   if not isMulticastRoutingEnabled( mode, vrfName, af ):
      return

   if af:
      assert af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]
      afList = [ af ]
   else:
      afList = [ AddressFamily.ipv4, AddressFamily.ipv6 ]

   for clearCountersFunc in PimCliLib.pimClearMessageCountersHook.extensions():
      for af in afList:
         clearCountersFunc( vrfName, af=af )

#------------------------------------------------------------------------------------
# Legacy:
# switch# clear ip pim [ vrf <vrfName> ] protocol counters
#------------------------------------------------------------------------------------
class ClearPimProtocolCountersCmd( CliCommand.CliCommandClass ):
   syntax = '''clear ( ( ip pim ) | ( pim [ ipv4 | ipv6 ] ) )
               [ vrf VRF_NAME ] protocol counters'''
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'ip' : CliToken.Ip.ipMatcherForClear,
      'pim' : CliToken.Pim.pimNodeAfterClear,
      'ipv4' : ipv4NodeForShow,
      'ipv6' : ipv6NodeForShow,
      'vrf' : CliToken.Pim.vrfMatcher,
      'VRF_NAME' : CliToken.Pim.vrfNameMatcher,
      'protocol' : CliToken.Pim.protocolMatcher,
      'counters' : CliToken.Pim.countersMatcher
   }
   handler = doClearPimProtocolCounters

BasicCli.EnableMode.addCommandClass( ClearPimProtocolCountersCmd )

#------------------------------------------------------------------------------------
# switch# clear ip mroute * | [ groupOrSource | groupOrSource ]
#------------------------------------------------------------------------------------
def doClearIpMrouteAll( mode, args ):
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   if not isMulticastRoutingEnabled( mode, vrfName, AddressFamily.ipv4 ):
      return
   for pimMode, hook in PimCliLib.clearIpMrouteHook.extensions():
      if pimMode == 'sparseMode':
         hook( mode, vrfName, clearAll=True )
      else:
         if _pimBidirectionalSupported():
            hook( mode, vrfName, clearAll=True )

def doClearIpMrouteVrf( mode, args ):
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   groupOrSource = args[ 'GROUP_OR_SOURCE' ]
   secondAddr = args.get( 'SECOND_ADDR' )

   if not isMulticastRoutingEnabled( mode, vrfName, AddressFamily.ipv4 ):
      return
   try:
      ( source, group ) = PimCliLib.ipPimParseSg( groupOrSource, secondAddr )
   except ValueError:
      mode.addError( "Must enter a multicast group or unicast source" )
      return

   for pimMode, hook in PimCliLib.clearIpMrouteHook.extensions():
      if pimMode == 'sparseMode':
         hook( mode, vrfName, False, groupOrSource, secondAddr )
      else:
         if _pimBidirectionalSupported():
            if source and not group:
               continue
            hook( mode, vrfName, groupOrSource )

class ClearIpMrouteAllCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip mroute [ vrf VRF_NAME ] *'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'ip' : CliToken.Ip.ipMatcherForClear,
      'mroute' : CliToken.Pim.mrouteMatcher,
      'vrf' : CliToken.Pim.vrfMatcher,
      'VRF_NAME' : CliToken.Pim.vrfNameMatcher,
      '*' : 'All routes'
   }
   handler = doClearIpMrouteAll

BasicCli.EnableMode.addCommandClass( ClearIpMrouteAllCmd )

class ClearIpMrouteVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip mroute [ vrf VRF_NAME ] GROUP_OR_SOURCE [ SECOND_ADDR ]'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'ip' : CliToken.Ip.ipMatcherForClear,
      'mroute' : CliToken.Pim.mrouteMatcher,
      'vrf' : CliToken.Pim.vrfMatcher,
      'VRF_NAME' : CliToken.Pim.vrfNameMatcher,
      'GROUP_OR_SOURCE' : groupOrSourceMatcher,
      'SECOND_ADDR' : groupOrSourceMatcher
   }
   handler = doClearIpMrouteVrf

BasicCli.EnableMode.addCommandClass( ClearIpMrouteVrfCmd )

#------------------------------------------------------------------------------------
# clear ip pim access-list counters
#------------------------------------------------------------------------------------
def clearIpAclCounters( mode, args ):
   AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, 'ip' )

class ClearIpAclCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip pim access-list counters'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'ip' : CliToken.Ip.ipMatcherForClear,
      'pim' : CliToken.Pim.pimNodeAfterClear,
      'access-list' : AclCli.accessListMatcher,
      'counters' : 'Clear IP access list counters'
   }
   handler = clearIpAclCounters

BasicCli.EnableMode.addCommandClass( ClearIpAclCountersCmd )

#------------------------------------------------------------------------------------

def Plugin( entityManager ):
   global _routingHwStatus, _pimGlobalStatus
   global aclStatus, aclCheckpoint

   #Af independent Config mounts
   configTypes = [ MfibVrfConfig, ]
   configTypesWriteMount = [ PimClearConfig, ]
   RouterMulticastCliLib.doLazyMounts( entityManager, configTypes )
   RouterMulticastCliLib.doLazyMounts( entityManager, configTypesWriteMount,
                                       useWriteMount=True )

   _routingHwStatus = LazyMount.mount( entityManager, 'routing/hardware/status',
         'Routing::Hardware::Status', 'r' )
   _pimGlobalStatus = LazyMount.mount( entityManager,
         Tac.Type( 'Routing::Pim::GlobalStatus' ).mountPath( 'ipv4' ),
         'Routing::Pim::GlobalStatus', 'r' )

   aclStatus = LazyMount.mount( entityManager, "acl/status/all", "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                    "Acl::CheckpointStatus", "w" )

   MrouteCli.routerMcastVrfDefinitionHook.addExtension( _vrfDefinitionHook )
   MrouteCli.routerMcastVrfDeletionHook.addExtension( _vrfDeletionHook )
   PimCliLib.pimClearMessageCountersHook.addExtension( clearPimMessageCounters )
