# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMatcher, ShowCommand
from CliToken.Ip import ipMatcherForShow
from CliToken.Pim import pimNodeAfterShow, pimNodeAfterShowIp
from CliPlugin import PimCliLib
from CliPlugin.IraCommonCli import AddressFamily
from CliPlugin.PimRegConfigCli import anycastRpKwMatcher
from CliPlugin.RouterMulticastCliLib import doLazyMounts, lazyGetters
from CliPlugin.VrfCli import VrfExprFactory, DEFAULT_VRF
import SmashLazyMount
import PimRegModel
from McastCommonCliLib import ShowAddressFamilyExpr
import Tac

_allVrfConfig = None
_entityManager = None

PimRegConfigColl = "Routing::PimReg::ConfigColl"

( pimRegConfigColl,
  pimRegConfigCollFromMode,
  pimRegConfigVrf,
  pimRegConfigVrfFromMode ) = lazyGetters( PimRegConfigColl )

PimRegAnycastRpLocalStatusColl = "Routing::PimReg::AnycastRpLocalStatusColl"

( pimRegAnycastRpLocalStatusColl,
  pimRegAnycastRpLocalStatusCollFromMode,
  pimRegAnycastRpLocalStatusVrf,
  pimRegAnycastRpLocalStatusVrfFromMode ) = \
        lazyGetters( PimRegAnycastRpLocalStatusColl,
                     collectionName='vrfAnycastRpLocalStatus' )

PimGlobalStatus = "Routing::Pim::GlobalStatus"

( pimGlobalStatus,
  pimGlobalStatusFromMode,
  pimGlobalStatusVrf,
  pimGlobalStatusVrfFromMode ) = lazyGetters( PimGlobalStatus,
                                          collectionName='pimEnabledSparseModeVrf' )

MfibVrfConfig = "Routing::Multicast::Fib::VrfConfig"

( mfibVrfConfig,
  mfibVrfConfigFromMode,
  mfibVrfConfigVrf,
  mfibVrfConfigVrfFromMode ) = lazyGetters( MfibVrfConfig,
                                            collectionName='config' )

# Add to message counters extension
def getPimMessageCounters( vrfName, af, **kwargs ):
   if vrfName in pimGlobalStatus( af ).pimEnabledSparseModeVrf:
      path = PimCliLib.getPath( 'Routing::Pim::Smash::MessageCounters', 
                                af, vrfName, "pimreg" )
      return SmashLazyMount.mount(
            _entityManager,
            path,
            'Routing::Pim::Smash::MessageCounters',
            SmashLazyMount.mountInfo( 'reader' ) )
   return None

def allowed( af, vrfName ):
   return vrfName in pimGlobalStatus( af ).pimEnabledSparseModeVrf

vrfExprFactory = VrfExprFactory(
   helpdesc='VRF name' )

registerSourceKwMatcher = CliMatcher.KeywordMatcher(
   'register-source',
   helpdesc='Source interface used for PIM register messages' )

#------------------------------------------------------------------------------------
# Legacy:
# show ip pim [vrf <vrfName>] register-source
#------------------------------------------------------------------------------------
# show pim {ipv4|ipv6} [vrf <vrfName>] register-source
#------------------------------------------------------------------------------------

def doShowIpPimRegisterSource( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   af = args.get( 'addressFamily', AddressFamily.ipv4 )

   if not PimCliLib.validVrfName( mode, vrfName, mfibVrfConfig( af ), af ):
      return None

   if vrfName not in pimRegConfigColl( af ).vrfConfig:
      # pylint: disable-next=consider-using-f-string
      mode.addError( "Vrf %s has not been configured" % vrfName )
      return None

   if not allowed( af, vrfName ):
      mode.addWarning( "WARNING: PIM is not enabled." )

   registerSourceModel = PimRegModel.RegisterSource()
   registerSourceModel.fromTacc( pimRegConfigColl( af ).vrfConfig[ vrfName ]  )
   return registerSourceModel

class ShowPimRegisterSourceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim AF [ VRF ] register-source'
   data = {
      'pim': pimNodeAfterShow,
      'AF': ShowAddressFamilyExpr,
      'VRF': vrfExprFactory,
      'register-source': registerSourceKwMatcher
   }
   handler = doShowIpPimRegisterSource
   cliModel = PimRegModel.RegisterSource
   
BasicCli.addShowCommandClass( ShowPimRegisterSourceCmd )

class ShowPimRegisterSourceLegacyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ VRF ] register-source'
   data = {
      'ip': ipMatcherForShow,
      'pim': pimNodeAfterShowIp,
      'VRF': vrfExprFactory,
      'register-source': registerSourceKwMatcher
   }
   handler = doShowIpPimRegisterSource
   cliModel = PimRegModel.RegisterSource
   
BasicCli.addShowCommandClass( ShowPimRegisterSourceLegacyCmd )

#-----------------------------------------------------------------------------------
# Legacy:
# show ip pim [vrf <vrfName>] anycast-rp 
#------------------------------------------------------------------------------------
# show pim {ipv4|ipv6} [vrf <vrfName>] anycast-rp
#------------------------------------------------------------------------------------

def doShowIpPimAnycastRp( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   af = args.get( 'addressFamily', AddressFamily.ipv4 )

   if not PimCliLib.validVrfName( mode, vrfName, mfibVrfConfig( af ), af ):
      return None

   if not allowed( af, vrfName ):
      mode.addWarning( "WARNING: PIM is not enabled." )

   anycastRpLocalStatusModel = PimRegModel.AnycastRpLocalStatus()
   anyRpLocStatus = pimRegAnycastRpLocalStatusColl( af ).\
         vrfAnycastRpLocalStatus.get( vrfName )
   if anyRpLocStatus:
      anycastRpLocalStatusModel.fromTacc( anyRpLocStatus )
   return anycastRpLocalStatusModel

class ShowPimAnycastRpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim AF [ VRF ] anycast-rp'
   data = {
      'pim': pimNodeAfterShow,
      'AF': ShowAddressFamilyExpr,
      'VRF': vrfExprFactory,
      'anycast-rp': anycastRpKwMatcher
   }
   handler = doShowIpPimAnycastRp
   cliModel = PimRegModel.AnycastRpLocalStatus
   
BasicCli.addShowCommandClass( ShowPimAnycastRpCmd )

class ShowPimAnycastRpLegacyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ VRF ] anycast-rp'
   data = {
      'ip': ipMatcherForShow,
      'pim': pimNodeAfterShowIp,
      'VRF': vrfExprFactory,
      'anycast-rp': anycastRpKwMatcher
   }
   handler = doShowIpPimAnycastRp
   cliModel = PimRegModel.AnycastRpLocalStatus
   
BasicCli.addShowCommandClass( ShowPimAnycastRpLegacyCmd )


def Plugin( entityManager ):
   global _entityManager

   _entityManager = entityManager
   lazyMountTypes = [ PimRegConfigColl, PimRegAnycastRpLocalStatusColl,
         PimGlobalStatus, MfibVrfConfig ]
   doLazyMounts( entityManager, lazyMountTypes )

   PimCliLib.pimMessageCountersHook.addExtension( getPimMessageCounters )
