#!/usr/bin/env python3
# Copyright (c) 2010, 2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=inconsistent-return-statements

from collections import defaultdict
import BasicCli
import CliParser, CliToken.Ip, CliToken.Pim
import Arnet
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from CliPlugin import AclCli
from CliPlugin import AclCliModel
from CliPlugin import BridgingCli
from CliPlugin import EvpnGatewayShowModel
from CliPlugin import IntfCli
from CliPlugin import IpAddrMatcher
from CliPlugin import MembershipJoinStatusModel
from CliPlugin import PimCliLib
from CliPlugin import PimConfigCheckModel
from CliPlugin.IpGenAddrMatcher import IpGenPrefixMatcher
from CliPlugin.IraIp6RouteCliLib import isValidIpv6PrefixWithError
from CliPlugin.IraIpRouteCliLib import isValidPrefixWithError
from CliPlugin.PimCliLib import interfaceConfigCheck, pimNeighborConfigCheck
from CliPlugin.PimCliLib import validVrfName, getPath, isMulticastRoutingEnabled
from CliPlugin.VlanCli import vlanIdMatcher
from CliPlugin.McastCommonCli import mcastv6RoutingSupportedGuard
import LazyMount, Tac, sys
import TacSigint
import PimModel
import PimModelLib
import PimLib
import SmashLazyMount
import SharedMem
import Shark
import SharkLazyMount
import PimSmashHelper
from IpLibConsts import DEFAULT_VRF as vrfDefault
from CliPrint import CliPrint
import Cell
from Arnet.NsLib import DEFAULT_NS
from McastCommonCliLib import AddressFamily
from McastCommonCliLib import validateMulticastRouting, ShowAddressFamilyExpr
from McastCommonCliLib import mcastRoutingSupported
import EosCloudInitLib
from CliToken.Pim import ( pimNodeAfterShow, pimNodeAfterShowIp, sparseModeMatcher,
                           mrouteInterfaceMatcherForShow, mrouteCountMatcher,
                           configSanityMatcher, vrfMatcher, vrfNameMatcher,
                           mrouteMatcher, pimMatcherForShow )
import CliMatcher
import CliCommand
import ShowCommand
# pylint: disable-next=ungrouped-imports
from McastCommonCliLib import ipv4NodeForShow, ipv6NodeForShow

from TypeFuture import TacLazyType
import Tracing
traceHandle = Tracing.Handle( "PimShowCli" )
t0 = traceHandle.trace0

MembershipClient = TacLazyType( "Irb::Multicast::App::Client" )

# pkgdeps: rpm SfCommon-lib

# Globals
_entityManager = None

_pimBidirStatusColl = {}
_pimBidirGroupSmashReaderSmColl = {}
_pimBidirGroupSmashRestartSmColl = {}

_pimConfigColl = None
_pimStatusColl = None
_pimGlobalStatus = None

_routingVrfInfoDir = None
_routing6VrfInfoDir = None
_routingHwStatus = None
_mfibVrfConfig = None
_ipConfig = None
_ipStatus = None
_vrfStatusLocal = None

_pim6StatusColl = None
_pim6GlobalStatus = None
_mfib6VrfConfig = None

_bessAgentStatusCli = None
_bessAgentStatusSfe = None

_pegVlanStatus4 = None
_pegVlanStatus6 = None
_enabledStatus = None
_pegDrStatus4 = None
_pegDrStatus6 = None
_membershipJoinStatus = {}
_membershipJoinStatusEvpn = {}
_membershipJoinStatusCli = {}
_l3IntfStatusDir = None

_shmemEm = None

aclCpConfig = None
aclStatus = None
aclCheckpoint = None

mlib = CliPrint().lib

def ipPimSupportedGuard( mode=None, token=None ):
   if mcastRoutingSupported( _entityManager.root(), routingHardwareStatus=None ):
      return None
   return CliParser.guardNotThisPlatform

def ipPimSupportedTechGuard( mode=None, token=None ):
   if ( ( ipPimSupportedGuard( mode, token ) is None ) or
        ( mcastv6RoutingSupportedGuard( mode, token ) is None ) ):
      return None
   return CliParser.guardNotThisPlatform

class Printer:
   def __init__( self, mode ):
      self.mode = mode
   # pylint: disable-msg=W0201
   def __enter__( self ):
      fmt = self.mode.session_.outputFormat()
      fd = sys.stdout.fileno()
      self.printer = mlib.initPrinter( fd, fmt )
      mlib.startRender( self.printer )
      return self
   # pylint: disable-msg=W0622
   def __exit__(self, type, value, traceback):
      mlib.endRender( self.printer )
      mlib.deinitPrinter( self.printer )

   def startDict( self, name ):
      mlib.startDict( self.printer, name )

   def endDict( self, name ):
      mlib.endDict( self.printer, name )

# Add to message counters extension
def getPimMessageCounters( vrfName, af=AddressFamily.ipv4, **kwargs ):
   # pylint: disable-msg=W0212
   pimGlobalStatus = getPimGlobalStatus( af )
   if vrfName in pimGlobalStatus.pimEnabledSparseModeVrf or \
         vrfName in pimGlobalStatus.pimEnabledBidirVrf:

      path = PimLib.getPath( 'Routing::Pim::Smash::MessageCounters',
                                 af, vrfName, "pim" )
      return SmashLazyMount.mount(
            _entityManager,
            path,
            'Routing::Pim::Smash::MessageCounters',
            SmashLazyMount.mountInfo( 'reader' ) )
   return None

def getPimStatus( vrfName, af=AddressFamily.ipv4 ):
   # pylint: disable-msg=W0212
   if af == AddressFamily.ipv4:
      statusColl = _pimStatusColl
   else:
      statusColl = _pim6StatusColl

   globalStatus = getPimGlobalStatus( af )
   if ( vrfName in globalStatus.pimEnabledSparseModeVrf or \
        vrfName in globalStatus.pimEnabledBidirVrf ) and \
        vrfName in statusColl.vrfStatus:
      return statusColl.vrfStatus[ vrfName ]
   return None

def getPimSmashStatus( vrfName, af=AddressFamily.ipv4 ):
   # pylint: disable-msg=W0212
   pimGlobalStatus = getPimGlobalStatus( af )
   if vrfName in pimGlobalStatus.pimEnabledSparseModeVrf or \
         vrfName in pimGlobalStatus.pimEnabledBidirVrf:
      return PimSmashHelper.mountInDependencyOrder(
         _entityManager, "routing/pim/status/%s" % vrfName, 'keyshadow',
         Tac.Type( "Routing::Pim::Smash::Status" ) )
   return None

def getNsName( vrfName ):
   if vrfName in _vrfStatusLocal.vrf:
      nsName = _vrfStatusLocal.vrf[ vrfName ].networkNamespace
   else:
      nsName = DEFAULT_NS
   return nsName

def getPimGlobalStatus( af ):
   return _pimGlobalStatus if af == AddressFamily.ipv4 else _pim6GlobalStatus

def waitForPimStatusSms( mode, af=AddressFamily.ipv4 ):
   def checkStatusReaderSm():
      if af == AddressFamily.ipv4:
         pimGlobalStatusReaderSm = PimLib.getPimGlobalStatusReaderSm(
                                                _entityManager, AddressFamily.ipv4 )
         return pimGlobalStatusReaderSm.initialized
      elif af == AddressFamily.ipv6:
         pim6GlobalStatusReaderSm = PimLib.getPimGlobalStatusReaderSm(
                                                _entityManager, AddressFamily.ipv6 )
         return pim6GlobalStatusReaderSm.initialized
      else:
         pimGlobalStatusReaderSm = PimLib.getPimGlobalStatusReaderSm(
                                                _entityManager, AddressFamily.ipv4 )
         pim6GlobalStatusReaderSm = PimLib.getPimGlobalStatusReaderSm(
                                                _entityManager, AddressFamily.ipv6 )
         return ( pimGlobalStatusReaderSm.initialized and
                  pim6GlobalStatusReaderSm.initialized )

   try:
      description = ' Pim table to be populated'
      Tac.waitFor( checkStatusReaderSm, timeout=1,
                   warnAfter=None, sleep=not Tac.activityManager.inExecTime.isZero,
                   maxDelay=0.1, description=description )
   except Tac.Timeout:
      mode.addError( "Command timed out" )
      return False

   return True

def allowed( af, vrfName ):
   if af == AddressFamily.ipv4:
      vrfConfig = _mfibVrfConfig
      routingInfo = _routingVrfInfoDir
   else:
      vrfConfig = _mfib6VrfConfig
      routingInfo = _routing6VrfInfoDir

   return PimCliLib.allowed( vrfName, routingInfo, vrfConfig )

def vrfPimNeighbors( vrfName=vrfDefault, af=AddressFamily.ipv4,
                     pimMode=None, intfId=None, bfd=False, detail=False ):
   if bfd:
      vrfNeighborsModel = PimModel.VrfPimBfdNeighbors()
   else:
      vrfNeighborsModel = PimModel.VrfPimNeighbors()

   def processNeighbor( neighbor ):
      for nb in neighbor.values():
         nbModel = PimModel.PimNeighbor()
         nbModel.fromTacc( intf, nb, detail )
         intfModel = PimModel.IntfPimNeighbors()
         if nb.intfId in vrfNeighborsModel.interfaces:
            intfModel = vrfNeighborsModel.interfaces[ nb.intfId ]
         intfModel.neighbors[ nb.address ] = nbModel
         vrfNeighborsModel.interfaces[ nb.intfId ] = intfModel
      TacSigint.check()

   _pimStatus = getPimStatus( vrfName, af )

   if _pimStatus:
      if intfId:
         intf = _pimStatus.pimIntf.get( intfId )
         if intf and displayIntf( intf, pimMode ):
            if bfd :
               processNeighbor( intf.neighbor )
            else :
               processNeighbor( intf.activeNeighbor )
      else:
         for intf in _pimStatus.pimIntf.values():
            if displayIntf( intf, pimMode ):
               if bfd :
                  processNeighbor( intf.neighbor )
               else :
                  processNeighbor( intf.activeNeighbor )
   return vrfNeighborsModel

def neighborsTable ( cliVrfName=vrfDefault, vrfConfig=None, af=AddressFamily.ipv4,
                     pimMode=None, intfId=None, bfd=False, detail=False ):
   if bfd:
      neighborsModel = PimModel.PimBfdNeighbors()
   else:
      neighborsModel = PimModel.PimNeighbors()
   if cliVrfName == "all":
      # pylint: disable-msg=W0212
      neighborsModel._isAll = True
      for vrfName in vrfConfig.config:
         nbsModel = vrfPimNeighbors( vrfName, af, pimMode, intfId, bfd, detail )
         neighborsModel.vrfs[ vrfName ] = nbsModel
   else:
      # pylint: disable-msg=W0212
      neighborsModel._isAll = False # pylint: disable=protected-access
      nbsModel = vrfPimNeighbors( cliVrfName, af, pimMode, intfId, bfd, detail )
      neighborsModel.vrfs[ cliVrfName ] = nbsModel
   return neighborsModel

def displayIntf( intf, pimMode=None ):
   if pimMode is None:
      return True

   if pimMode == "sparse-mode":
      pimMode = "modePimSm"
   elif pimMode == "bidirectional":
      pimMode = "modePimBidir"
   else:
      return False

   if intf.mode in [ pimMode, "modePimSmAndBidir" ]:
      return True

   if pimMode == "modePimSm" and intf.borderRouter:
      return True

   return False

def extractBessIntfCtrs( af, vrfName ):

   class BessInterfaceCounters:

      def __init__( self ):
         self.bytesIn = 0
         self.pktsIn = 0
         self.bytesOut = 0
         self.pktsOut = 0
         self.pktsQueued = 0
         self.pktsDropped = 0

   intfCtrInfoDict = defaultdict( BessInterfaceCounters )

   path = PimLib.getPath( 'Sfe::Multicast::IntfCounterTable',
                                 af, vrfName )

   bessIntfCounterTable = SmashLazyMount.mount(
            _entityManager,
            path,
            'Sfe::Multicast::IntfCounterTable',
            SmashLazyMount.mountInfo( 'reader' ) )

   for intf, intfCounters in bessIntfCounterTable.intf.items():
      intfCtrInfo = intfCtrInfoDict[ intf ]
      intfCtrInfo.bytesIn = intfCounters.bytesIn
      intfCtrInfo.pktsIn = intfCounters.pktsIn
      intfCtrInfo.bytesOut = intfCounters.bytesOut
      intfCtrInfo.pktsOut = intfCounters.pktsOut
      intfCtrInfo.pktsQueued = intfCounters.pktsQueued
      intfCtrInfo.pktsDropped = intfCounters.pktsDropped

   return intfCtrInfoDict

def interfaces( vrfName=vrfDefault, af=AddressFamily.ipv4,
                pimMode=None, intfId=None, detail=False ):
   ''' AF independent function for interfaces. '''
   intfsModel = PimModel.PimInterfaces()

   bessIntfCtrs = {}
   v4Enabled = _bessAgentStatusCli.v4Enabled or _bessAgentStatusSfe.v4Enabled
   v6Enabled = _bessAgentStatusCli.v6Enabled or _bessAgentStatusSfe.v6Enabled
   if( ( af == AddressFamily.ipv4 and v4Enabled ) or
       ( af == AddressFamily.ipv6 and v6Enabled ) or
       EosCloudInitLib.isModeSfe() ):
      bessIntfCtrs = extractBessIntfCtrs( af, vrfName )

      if not bessIntfCtrs:
         return intfsModel

   _pimStatus = getPimStatus( vrfName, af )
   if _pimStatus:
      nsName = getNsName( vrfName )
      if intfId:
         intf = _pimStatus.pimIntf.get( intfId )
         if intf and displayIntf( intf, pimMode ):
            intfModel = PimModel.PimInterface()
            intfModel.fromTacc( intf, detail, bessIntfCtrs, netnsName=nsName )
            intfsModel.interfaces[ intfId ] = intfModel
      else:
         # pylint: disable-next=redefined-argument-from-local
         for intfId, intf in _pimStatus.pimIntf.items():
            if displayIntf( intf, pimMode ):
               intfModel = PimModel.PimInterface()
               intfModel.fromTacc( intf, detail, bessIntfCtrs, netnsName=nsName )
               intfsModel.interfaces[ intfId ] = intfModel
               TacSigint.check()
   return intfsModel

def protocolCounters( vrfName=vrfDefault, af=None, intfId=None,
                      detail=False ):
   ''' AF independent function for counters. '''

   allCountersModel = PimModel.PimMessageCounters()

   # Collect all the counters from all the agents.
   allCounterList = []
   if af:
      afList = [ af ]
   else:
      afList = [ AddressFamily.ipv4, AddressFamily.ipv6 ]

   for af in afList: # pylint: disable=redefined-argument-from-local
      for getCountersFunc in PimCliLib.pimMessageCountersHook.extensions():
         allCounterList.append( getCountersFunc( vrfName, af ) )

   allCountersModel.fromTacc( allCounterList, intfId, detail )

   return allCountersModel

def pimCliShowCommand( func ):
   ''' Decorator for show commands.  If either routing or multicast-routing
       is disabled, prepend output with warning. '''
   def newFunc( mode, *args, **kwargs):
      vrfName = kwargs.get( 'vrfName' )
      af = kwargs.get( 'ipFamily' )
      if vrfName is None:
         vrfName = vrfDefault
      if not allowed( af, vrfName ):
         mode.addWarning( "unicast routing or multicast routing not configured " \
                          "for vrf %s in %s" % ( vrfName, af ) )
      return func( mode, *args, **kwargs )
   return newFunc

def _pimBidirectionalGuard( mode, token ):
   if token != 'bidirectional' or _routingHwStatus.pimBidirectionalSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def getAddrFamily( args ):
   if args.get('ip'):
      return 'ipv4'
   else:
      return args.get( 'ipv4' ) or args.get( 'ipv6' )

#------------------------------------------------------------------------------------
# Show tokens for new parser
#------------------------------------------------------------------------------------
ipAfterShowMatcher = CliToken.Ip.ipMatcherForShow
vrfNameExpr = VrfCli.VrfExprFactory( helpdesc='VRF name' )
modeNode = CliCommand.Node( matcher=CliMatcher.EnumMatcher( {
      'sparse-mode' : 'Sparse mode PIM',
      'bidirectional' : 'PIM Bidirectional mode information'
   } ),
   guard=_pimBidirectionalGuard )
neighborMatcher = CliMatcher.KeywordMatcher( 'neighbor',
   helpdesc='Show PIM neighbors' )
bfdMatcher = CliMatcher.KeywordMatcher( 'bfd',
   helpdesc='BFD state information' )
interfaceKw = CliMatcher.KeywordMatcher( 'interface',
   helpdesc='Show PIM interface information' )
detailKw = CliMatcher.KeywordMatcher( 'detail',
   helpdesc='Display information in detail' )
counterDetailKw = CliMatcher.KeywordMatcher( 'detail',
   helpdesc='Detailed counter information' )
groupOrSourceMatcher = IpAddrMatcher.IpAddrMatcher(
   'A multicast group address or unicast source address' )
ipGenPrefixMatcher = IpGenPrefixMatcher(
      'group address range with prefix', addrWithMask=False )
mrouteDetailMatcher = CliMatcher.KeywordMatcher( 'detail',
   helpdesc = 'Detailed route information' )
evpnMatcher = CliMatcher.KeywordMatcher( 'evpn',
   helpdesc='Show PIM EVPN data' )
gatewayMatcher = CliMatcher.KeywordMatcher( 'gateway',
   helpdesc='Show PIM gateway data' )
drMatcher = CliMatcher.KeywordMatcher( 'dr',
   helpdesc='Show Designated Router' )
membershipMatcher = CliMatcher.KeywordMatcher( 'membership',
   helpdesc='Group membership learned via PIM' )
evpnCliMatcher = CliMatcher.KeywordMatcher( 'cli',
   helpdesc='Show PIM EVPN data added from CLI' )

class ShowMrouteCommonArguments( CliCommand.CliExpression ):
   expression = '[ { GROUP_OR_SOURCE } ] [ detail ]'
   data = {
      'GROUP_OR_SOURCE': CliCommand.Node( groupOrSourceMatcher, maxMatches=2 ),
      'detail': mrouteDetailMatcher,
   }

#------------------------------------------------------------------------------------
# Legacy:
# show ip pim [ sparse-mode | bidirectional ] neighbor [ <intf-name> ] [ bfd|detail ]
#------------------------------------------------------------------------------------
# show pim {ipv4|ipv6} [sparse-mode|bidirectional] neighbor [<intf-name> [bfd|detail]
#------------------------------------------------------------------------------------
def doShowIpPimNeighbor( mode, args ):
   af = args.get( 'addressFamily', 'ipv4' )
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   intf = args.get( 'INTF' )
   pimMode = args.get( 'MODE' )

   if af == AddressFamily.ipv4:
      vrfConfig = _mfibVrfConfig
   else:
      vrfConfig = _mfib6VrfConfig
   if not validVrfName( mode, vrfName, vrfConfig, af ):
      return

   if not waitForPimStatusSms( mode, af ):
      return

   intfId = None
   if intf:
      intfId = intf.name
      if not intf.lookup() or not intf.status().deviceName:
         mode.addError( "%s is not operational" % intfId )
         return None
   return neighborsTable( vrfName, vrfConfig, af, pimMode, intfId, False,
                          'detail' in args )

def doShowIpPimBfdNeighbor( mode, args ):
   af = args.get( 'addressFamily', 'ipv4' )
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   intf = args.get( 'INTF' )
   pimMode = args.get( 'MODE' )

   if af == AddressFamily.ipv4:
      vrfConfig = _mfibVrfConfig
   else:
      vrfConfig = _mfib6VrfConfig
   if not validVrfName( mode, vrfName, vrfConfig, af ):
      return

   if not waitForPimStatusSms( mode, af ):
      return

   intfId = None
   if intf:
      if not intf.lookup() or not intf.status().deviceName:
         mode.addError( "%s is not operational" % intf.name )
         return None
      intfId = Tac.newInstance( 'Arnet::IntfId', intf.name )

   #Bfd is supported on VRFs
   return neighborsTable( vrfName, vrfConfig, af, pimMode, intfId, bfd=True )

#--------------------------------------------------------------------------------
# show ip pim [ vrf VRFNAME ] [ ( sparse-mode | bidirectional ) ] neighbor [ INTF ]
# [ detail ]
#--------------------------------------------------------------------------------
class IpPimNeighborCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ vrf VRF_NAME ] [ MODE ] neighbor [ INTF ] [ detail ]'
   data = {
      'ip' : ipAfterShowMatcher,
      'pim' : pimNodeAfterShowIp,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'MODE' : modeNode,
      'neighbor' : neighborMatcher,
      'INTF' : IntfCli.Intf.matcher,
      'detail' : detailKw
   }
   handler = doShowIpPimNeighbor
   cliModel = PimModel.PimNeighbors

BasicCli.addShowCommandClass( IpPimNeighborCmd )

#--------------------------------------------------------------------------------
# show pim ( ipv4 | ipv6 ) [ vrf VRFNAME ] [ ( sparse-mode | bidirectional ) ]
# neighbor [ INTF ] [ detail ]
#--------------------------------------------------------------------------------
class PimNeighborCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim AF [ vrf VRF_NAME ] [ MODE] neighbor [ INTF ] [ detail ]'
   data = {
      'pim' : pimNodeAfterShow,
      'AF' : ShowAddressFamilyExpr,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'MODE' : modeNode,
      'neighbor' : neighborMatcher,
      'INTF' : IntfCli.Intf.matcher,
      'detail' : detailKw
   }
   handler = doShowIpPimNeighbor
   cliModel = PimModel.PimNeighbors

BasicCli.addShowCommandClass( PimNeighborCmd )

#--------------------------------------------------------------------------------
# show ip pim [ vrf VRFNAME ] [ ( sparse-mode | bidirectional ) ]
# neighbor [ INTF ] bfd
#--------------------------------------------------------------------------------
class IpPimNeighborBfdCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ vrf VRF_NAME ] [ MODE ] neighbor [ INTF ] bfd'
   data = {
      'ip' : ipAfterShowMatcher,
      'pim' : pimNodeAfterShowIp,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'MODE' : modeNode,
      'neighbor' : neighborMatcher,
      'INTF' : IntfCli.Intf.matcher,
      'bfd' : bfdMatcher
   }
   handler = doShowIpPimBfdNeighbor
   cliModel = PimModel.PimBfdNeighbors

BasicCli.addShowCommandClass( IpPimNeighborBfdCmd )

#--------------------------------------------------------------------------------
# show pim ( ipv4 | ipv6 ) [ vrf VRFNAME ] [ ( sparse-mode | bidirectional ) ]
# neighbor [ INTF ] bfd
#--------------------------------------------------------------------------------
class PimNeighborBfdCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim AF [ vrf VRF_NAME ] [ MODE ] neighbor [ INTF ] bfd'
   data = {
      'pim' : pimNodeAfterShow,
      'AF' : ShowAddressFamilyExpr,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'MODE' : modeNode,
      'neighbor' : neighborMatcher,
      'INTF' : IntfCli.Intf.matcher,
      'bfd' : bfdMatcher
   }
   handler = doShowIpPimBfdNeighbor
   cliModel = PimModel.PimBfdNeighbors

BasicCli.addShowCommandClass( PimNeighborBfdCmd )

#-----------------------------------------------------------------------------------
# Legacy:
# show ip pim [ sparse-mode | bidirectional ] interface [ <intf-name> ] [ detail ]
#-----------------------------------------------------------------------------------
# show pim {ipv4|ipv6} [vrf <vrfName> ][ sparse-mode | bidirectional ] \
#      interface [ <intf-name> ] [ detail ]
#-----------------------------------------------------------------------------------
def doShowIpPimInterface( mode, args ):
   af = args.get( 'addressFamily', 'ipv4' )
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   intf = args.get( 'INTF' )
   pimMode = args.get( 'MODE' )

   if af == AddressFamily.ipv4:
      vrfConfig = _mfibVrfConfig
   else:
      vrfConfig = _mfib6VrfConfig
   if not validVrfName( mode, vrfName, vrfConfig, af ):
      return

   if not waitForPimStatusSms( mode, af ):
      return

   intfId = None
   if intf:
      if not intf.lookup() or not intf.status().deviceName:
         mode.addError( "%s is not operational" % intf.name )
         return None
      intfId = Tac.newInstance( 'Arnet::IntfId', intf.name )

   revision = mode.session_.requestedModelRevision()
   fmt = mode.session_.outputFormat()

   # If requesting JSON format and version 1 which supports only sparse-mode,
   # return only the sparse-mode entries.  Ignore all the bidirectional
   # interface entries for version 1.
   if fmt == 2 and revision == 1:
      pimMode = "sparse-mode"
   return interfaces( vrfName, af, pimMode, intfId, 'detail' in args )

#--------------------------------------------------------------------------------
# show ip pim [ vrf VRFNAME ] [ sparse-mode | bidirectional ]
# interface [ INTF ] [ detail ]
#--------------------------------------------------------------------------------
class IpPimInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ vrf VRF_NAME ] [ MODE ] interface [ INTF ] [ detail ]'
   data = {
      'ip' : ipAfterShowMatcher,
      'pim' : pimNodeAfterShowIp,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'MODE' : modeNode,
      'interface' : interfaceKw,
      'INTF' : IntfCli.Intf.matcher,
      'detail' : detailKw
   }
   handler = doShowIpPimInterface
   cliModel = PimModel.PimInterfaces

BasicCli.addShowCommandClass( IpPimInterfaceCmd )

#--------------------------------------------------------------------------------
# show pim ( ipv4 | ipv6 ) [ vrf VRFNAME ] [ sparse-mode | bidirectional ]
# interface [ INTF ] [ detail ]
#--------------------------------------------------------------------------------
class PimInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim AF [ vrf VRF_NAME ] [ MODE ] interface [ INTF ] [ detail ]'
   data = {
      'pim' : pimNodeAfterShow,
      'AF' : ShowAddressFamilyExpr,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'MODE' : modeNode,
      'interface' : interfaceKw,
      'INTF' : IntfCli.Intf.matcher,
      'detail' : detailKw
   }
   handler = doShowIpPimInterface
   cliModel = PimModel.PimInterfaces

BasicCli.addShowCommandClass( PimInterfaceCmd )

#------------------------------------------------------------------------------------
# Legacy:
# show ip pim protocol counters [ <intf-name> ] [ detail ] ( legacy )
#------------------------------------------------------------------------------------
# show pim [ipv4|ipv6] [vrf <vrfName>] protocol counters [ <intf-name> ] [ detail ]
#------------------------------------------------------------------------------------
def doShowIpPimProtocol( mode, args ):
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   intf = args.get( 'INTF' )
   af = getAddrFamily( args )

   if not validateMulticastRouting( mode, vrfName, af ):
      return None

   if not waitForPimStatusSms( mode, af ):
      return None

   intfId = None
   if intf:
      if not intf.lookup() or not intf.status().deviceName:
         mode.addError( "%s is not operational" % intf.name )
         return None
      intfId = Tac.newInstance( 'Arnet::IntfId', intf.name )
   return protocolCounters( vrfName, af, intfId, 'detail' in args )

class PimProtocolLegacyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ vrf VRF_NAME ] protocol counters [ INTF ] [ detail ]'
   data = {
      'ip' : ipAfterShowMatcher,
      'pim' : pimNodeAfterShowIp,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'protocol' : CliToken.Pim.protocolMatcher,
      'counters' : CliToken.Pim.countersMatcher,
      'INTF' : IntfCli.Intf.matcher,
      'detail' : counterDetailKw
   }
   handler = doShowIpPimProtocol
   cliModel = PimModel.PimMessageCounters

BasicCli.addShowCommandClass( PimProtocolLegacyCmd )

class PimProtocolCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show pim [ ipv4 | ipv6 ] [ vrf VRF_NAME ]
               protocol counters [ INTF ] [ detail ]'''
   data = {
      'pim' : pimNodeAfterShow,
      'ipv4' : ipv4NodeForShow,
      'ipv6' : ipv6NodeForShow,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'protocol' : CliToken.Pim.protocolMatcher,
      'counters' : CliToken.Pim.countersMatcher,
      'INTF' : IntfCli.Intf.matcher,
      'detail' : counterDetailKw
   }
   handler = doShowIpPimProtocol
   cliModel = PimModel.PimMessageCounters

BasicCli.addShowCommandClass( PimProtocolCmd )

#------------------------------------------------------------------------------------
# Legacy:
# show ip pim upstream joins [ <sourceOrGroup1> ] [ <sourceOrGroup2> ]
#                            [ neighbor <address> ]
#------------------------------------------------------------------------------------
def doShowIpPimUpstreamJoins( mode, args ):
   model = PimModelLib.PimUpstreamJoins()
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   af = 'ipv4'
   groupOrSource = args.get( 'GROUP_OR_SOURCE', [] ) + [ None, None ]
   neighborAddr = args.get( 'NEIGHBOR_ADDR' )

   vrfConfig = _mfibVrfConfig
   if not validVrfName( mode, vrfName, vrfConfig, af ):
      return

   try:
      ( source, group ) = PimCliLib.ipPimParseSg( groupOrSource[ 0 ],
                             groupOrSource[ 1 ] )
   except ValueError:
      mode.addError("Must enter a multicast group and/or unicast source")
      return model

   if not waitForPimStatusSms( mode, af ):
      return

   if neighborAddr is not None:
      neighborAddr = Arnet.IpGenAddr( neighborAddr )

   pimStatus = getPimStatus( vrfName, af )
   joinPruneStatus = []
   for status in PimCliLib.pimUpstreamJoinsHook.extensions():
      s = status( vrfName )
      if s is not None:
         joinPruneStatus.append( s )

   model.initialize( pimStatus, joinPruneStatus, source, group, neighborAddr )
   return model

class IpPimUpstreamJoinsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip pim [ vrf VRF_NAME ] upstream joins
               [ { GROUP_OR_SOURCE } ] [ neighbor NEIGHBOR_ADDR ]'''
   data = {
      'ip' : ipAfterShowMatcher,
      'pim' : pimNodeAfterShowIp,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'upstream' : CliToken.Pim.upstreamMatcher,
      'joins' : CliToken.Pim.joinsMatcher,
      'GROUP_OR_SOURCE' : CliCommand.Node( groupOrSourceMatcher, maxMatches=2 ),
      'neighbor' : CliToken.Pim.neighborsMatcher,
      'NEIGHBOR_ADDR' : CliToken.Pim.joinsNeighborAddrMatcher
   }
   handler = doShowIpPimUpstreamJoins
   cliModel = PimModelLib.PimUpstreamJoins

BasicCli.addShowCommandClass( IpPimUpstreamJoinsCmd )

#------------------------------------------------------------------------------------
# Legacy:
# show ip pim config-sanity
#------------------------------------------------------------------------------------
def doShowIpPimConfigSanity( mode, args ):
   print( "DISCLAIMER: Below are only hints of potential"
          " PIM misconfiguration.\n"
          "They do not necessarily imply that there is a real problem.\n" )

   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )

   if not allowed( AddressFamily.ipv4, vrfName ):
      mode.addWarning( "'ip multicast-routing' is not configured for vrf %s. "
                       % vrfName )

   for cmd in sorted( PimCliLib.configSanityCallbacks ):
      sys.stdout.flush()
      try:
         if vrfName is not vrfDefault:
            cmd = cmd.replace( "pim", "pim vrf %s" % vrfName )
         mode.session_.runCmd( cmd, aaa=False )
      except CliParser.GuardError as e:
         print( f"(unavailable: {cmd}, guardError: {e.guardCode})" )
      except CliParser.AlreadyHandledError:
         pass
      except KeyboardInterrupt as e:
         raise RuntimeError( "Program interrupted by the user" ) from e
      except Exception as e: # pylint: disable=broad-except
         # Catch all Exceptions and log in sanity
         print( f"(unavailable: {cmd}, error: {e})" )
      except: # pylint: disable=bare-except
         # Catch all errors, so that one command failure doesn't cause
         # output from others to be skipped.
         print( "(unavailable command %s)" % cmd )
      TacSigint.check()

class PimConfigSanityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ vrf VRF_NAME ] config-sanity'
   data = {
      'ip' : ipAfterShowMatcher,
      'pim' : pimNodeAfterShowIp,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'config-sanity' : configSanityMatcher
   }
   handler = doShowIpPimConfigSanity

BasicCli.addShowCommandClass( PimConfigSanityCmd )

def pimIntfConfigSanity( mode, args ):
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   return interfaceConfigCheck( mode, vrfName, _ipConfig, _pimConfigColl )

class PimIntfConfigSanityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ vrf VRF_NAME ] config-sanity interface'
   data = {
      'ip' : ipAfterShowMatcher,
      'pim' : pimNodeAfterShowIp,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'config-sanity' : configSanityMatcher,
      'interface' : 'Show hints of potential PIM problems regarding interface config'
   }
   handler = pimIntfConfigSanity
   cliModel = PimConfigCheckModel.IntfConfigCheck

BasicCli.addShowCommandClass( PimIntfConfigSanityCmd )

def pimNeighborConfigSanity( mode, args ):
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   if not waitForPimStatusSms( mode ):
      return

   _pimVrfStatus = getPimStatus( vrfName )

   return pimNeighborConfigCheck( mode,  _ipConfig, _pimConfigColl, _pimVrfStatus,
                                  getPimMessageCounters( vrfName ) )

class PimNeighborConfigSanityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ vrf VRF_NAME ] config-sanity neighbor'
   data = {
      'ip' : ipAfterShowMatcher,
      'pim' : pimNodeAfterShowIp,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'config-sanity' : configSanityMatcher,
      'neighbor' : 'Show hints of potential PIM problems regarding neighbor config'
   }
   handler = pimNeighborConfigSanity
   cliModel = PimConfigCheckModel.NbrConfigCheck

BasicCli.addShowCommandClass( PimNeighborConfigSanityCmd )

PimCliLib.registerShowPimConfigSanityCmdCallback(
      [ 'show ip pim config-sanity interface',
        'show ip pim config-sanity neighbor' ] )

#------------------------------------------------------------------------------------
# Legacy:
# show ip pim rp [ groupPrefix ] [ detail ]
#------------------------------------------------------------------------------------
# show pim {ipv4|ipv6} rp [ groupPrefix ] [ detail ]
#------------------------------------------------------------------------------------
def doShowIpPimRp( mode, args ):
   model = PimModelLib.PimRpCandidatesAll()
   vrfName = args.get( 'VRF', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   af = AddressFamily.ipv6 if 'ipv6' in args else AddressFamily.ipv4
   groupRange = args.get( 'PREFIX' )
   detail = 'detail' in args

   if af == AddressFamily.ipv4:
      vrfConfig = _mfibVrfConfig
      if groupRange and not isValidPrefixWithError( mode, str( groupRange ) ):
         return
   else:
      vrfConfig = _mfib6VrfConfig
      if groupRange and not isValidIpv6PrefixWithError(
            mode, groupRange.v6Prefix ):
         return
   if not validVrfName( mode, vrfName, vrfConfig, af ):
      return

   crpSets = {}
   revision = mode.session_.requestedModelRevision()
   fmt = mode.session_.outputFormat()
   groupRange = str( groupRange ) if groupRange else None

   if fmt == 2 and revision == 1:
      for (pimMode, hook) in PimCliLib.pimShowIpPimRpHook.extensions():
         if pimMode == 'sparseMode':
            modeStr, crpSet = hook( mode, vrfName, groupRange, detail, af )
            crpSets[ modeStr ] = crpSet
   else:
      for ( pimMode, hook ) in PimCliLib.pimShowIpPimRpHook.extensions():
         if pimMode == 'bidirectional':
            #Check if bidirectional supported
            if _routingHwStatus.pimBidirectionalSupported:
               modeStr, crpSet = hook( mode, vrfName, groupRange, detail, af )
               crpSets[ modeStr ] = crpSet
            else:
               continue
         if pimMode == 'sparseMode':
            modeStr, crpSet = hook( mode, vrfName, groupRange, detail, af )
            crpSets[ modeStr ] = crpSet

   model.initialize( crpSets )
   return model

class ShowIpPimRp( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ VRF ] rp [ PREFIX ] [ detail ]'
   data = {
      'ip': ipAfterShowMatcher,
      'pim': pimNodeAfterShowIp,
      'VRF': vrfNameExpr,
      'rp': CliToken.Pim.rpMatcher,
      'PREFIX': ipGenPrefixMatcher,
      'detail': CliToken.Pim.detailMatcher
   }
   handler = doShowIpPimRp
   cliModel = PimModelLib.PimRpCandidatesAll

BasicCli.addShowCommandClass( ShowIpPimRp )

class ShowPimAfRp( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim ( ipv4 | ipv6 ) [ VRF ] rp [ PREFIX ] [ detail ]'
   data = {
      'pim': pimNodeAfterShow,
      'ipv4': ipv4NodeForShow,
      'ipv6': ipv6NodeForShow,
      'VRF': vrfNameExpr,
      'rp': CliToken.Pim.rpMatcher,
      'PREFIX': ipGenPrefixMatcher,
      'detail': CliToken.Pim.detailMatcher
   }
   handler = doShowIpPimRp
   cliModel = PimModelLib.PimRpCandidatesAll

BasicCli.addShowCommandClass( ShowPimAfRp )

#-----------------------------------------------------------------------------------
# Legacy:
# show ip mroute count
#-----------------------------------------------------------------------------------
def doShowIpMrouteCountAll( mode, args ):
   model = PimModelLib.MrouteCountAll
   af = AddressFamily.ipv4
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )

   vrfConfig = _mfibVrfConfig
   if not validVrfName( mode, vrfName, vrfConfig, af ):
      return

   if not waitForPimStatusSms( mode, af ):
      return

   submodel = {}

   revision = mode.session_.requestedModelRevision()
   fmt = mode.session_.outputFormat()

   if fmt == 2 and revision == 1:
      for (pimMode, hook) in PimCliLib.pimShowIpMrouteCountHook.extensions():
         if pimMode == 'sparseMode':
            return  hook(mode, vrfName,  submodel=False)[ 1 ]

   #if the revision is not the latest revision, Cli didn't check model

   else:
      with Printer( mode ) as p:
         for ( pimMode, hook ) in PimCliLib.pimShowIpMrouteCountHook.extensions():
            p.startDict( pimMode )
            submodel[ pimMode ] = hook( mode, vrfName, submodel=True )[ 1 ]
            p.endDict( pimMode )

      for pimMode in submodel: # pylint: disable=consider-using-dict-items
         setattr( model, pimMode, submodel[ pimMode ] )
      return model

class IpMrouteCountAllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip mroute [ vrf VRF_NAME ] count'
   data = {
      'ip' : ipAfterShowMatcher,
      'mroute' : mrouteMatcher,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'count' : mrouteCountMatcher
   }
   handler = doShowIpMrouteCountAll
   cliModel = PimModelLib.MrouteCountAll

BasicCli.addShowCommandClass( IpMrouteCountAllCmd )

#-----------------------------------------------------------------------------------
# Legacy:
#show ip mroute [ <sourceOrGroup> ][ <sourceOrGroup> ][ detail ]
#-----------------------------------------------------------------------------------
def processShowIpMrouteArgs( mode, args ):
   af = AddressFamily.ipv4
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   groupOrSource = args.get( 'GROUP_OR_SOURCE', [] ) + [ None, None ]
   groupOrSource1 = groupOrSource[ 0 ]
   groupOrSource2 = groupOrSource[ 1 ]
   detail = 'detail' in args
   if not vrfName:
      vrfName = vrfDefault

   if not waitForPimStatusSms( mode, af ):
      return {}

   # show ip mroute None None => SM + BIDIR
   # show ip mroute vrf all None None => SM + BIDR
   if groupOrSource1 is None and groupOrSource2 is None:
      source = None
      group = None

   # show ip mroute s, g => SM
   # show ip mroute vrf all s, g => SM
   if groupOrSource1 and groupOrSource2:
      try:
         ( source, group ) = PimCliLib.ipPimParseSg( groupOrSource1, groupOrSource2 )
      except ValueError:
         mode.addError("Must enter a multicast group and/or unicast source")
         return {}

   # show ip mroute s => SM and show ip mroute g => SM + BIDIR
   # show ip mroute vrf all s => SM and show ip mroute vrf all g => SM + BIDIR
   if groupOrSource1 and groupOrSource2 is None:
      err = IpAddrMatcher.validateMulticastIpAddr( groupOrSource1 )
      if err:
         source = Arnet.IpGenAddr( groupOrSource1 )
         group = None
      else:
         group = Arnet.IpGenAddr( groupOrSource1 )
         source = None

   commonArgDict = {}
   commonArgDict[ 'af' ] = af
   commonArgDict[ 'vrfName' ] = vrfName
   commonArgDict[ 'group' ] = group
   commonArgDict[ 'source' ] = source
   commonArgDict[ 'detail' ] = detail
   return commonArgDict

def showIpMroutePerVrf( mode, p, vrfName, args ):
   '''
   This is a DeferredModel type and hence no need to populate the model, the model
   will be dumped to screen by the handler function. Returning just the model class
   back.
   '''
   model = PimModelLib.IpMroutesAll
   submodel = {}
   for ( pimMode, hook ) in PimCliLib.pimShowIpMrouteHook.extensions():
      p.startDict( pimMode )
      submodel[ pimMode ] = hook( mode, vrfName, args[ 'source' ],
                                  args[ 'group' ], args[ 'detail' ],
                                  submodel=True )[ 1 ]
      p.endDict( pimMode )
   return model

def doShowIpMroute( mode, args ):
   model = PimModelLib.IpMroutesAll
   args = processShowIpMrouteArgs( mode, args )
   if not args:
      return
   revision = mode.session_.requestedModelRevision()
   fmt = mode.session_.outputFormat()
   vrfConfig = _mfibVrfConfig

   if not validVrfName( mode, args[ 'vrfName' ], vrfConfig, args[ 'af' ] ):
      return

   if fmt == 2 and revision == 1:
      for (pimMode, hook) in PimCliLib.pimShowIpMrouteHook.extensions():
         if pimMode == 'sparseMode':
            return hook( mode, args[ 'vrfName' ], args[ 'source' ],
                         args[ 'group' ], args[ 'detail' ],
                         submodel=False )[ 1 ]
   else:
      with Printer( mode ) as p:
         model = showIpMroutePerVrf( mode, p, args[ 'vrfName' ], args )
      return model

class IpMrouteAllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip mroute [ vrf VRF_NAME ] [ SHOW_MROUTE_COMMON_ARGS ]'
   data = {
      'ip' : ipAfterShowMatcher,
      'mroute' : mrouteMatcher,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'SHOW_MROUTE_COMMON_ARGS': ShowMrouteCommonArguments,
   }
   handler = doShowIpMroute
   cliModel = PimModelLib.IpMroutesAll

BasicCli.addShowCommandClass( IpMrouteAllCmd )

# -------------------------------------------------------------------------------
# "show ip mroute vrf all ..." is a new command because
# "show ip mroute vrf VRF_NAME ..." already has a published model which was hard
# to extend for all vrfs and maintain backward compatability
#
# show ip mroute vrf all [ <sourceOrGroup> ][ detail ]
# -------------------------------------------------------------------------------

def doShowIpMrouteAllVrfs( mode, args ):
   '''
   This is a DeferredModel type and hence no need to populate the model, the model
   will be dumped to screen by the handler function. Returning just the model class
   back.
   '''
   model = PimModelLib.IpMroutesAllVrfs
   args = processShowIpMrouteArgs( mode, args )
   if not args:
      return
   fmt = mode.session_.outputFormat()
   vrfConfig = _mfibVrfConfig
   submodel = {}

   vrfList = []
   for vrf in vrfConfig.config:
      if isMulticastRoutingEnabled( vrf, vrfConfig ):
         if vrf != 'default':
            vrfList.append( vrf )
   if fmt == 1:
      vrfList.sort()
   if isMulticastRoutingEnabled( vrfDefault, vrfConfig ):
      vrfList.insert( 0, vrfDefault )

   with Printer( mode ) as p:
      p.startDict( "vrfs" )
      for vrf in vrfList:
         p.startDict( vrf )
         if fmt == 1:
            print( 'VRF:%s' % vrf )
            sys.stdout.flush()
         modelPerVrf = showIpMroutePerVrf( mode, p, vrf, args )
         p.endDict( vrf )
         submodel[ vrf ] = modelPerVrf
      p.endDict( "vrfs" )
   return model

class IpMrouteAllVrfsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip mroute vrf all [ SHOW_MROUTE_COMMON_ARGS ]'
   data = {
      'ip': ipAfterShowMatcher,
      'mroute': mrouteMatcher,
      'vrf': vrfMatcher,
      'all': 'Multicast routes from all VRFs',
      'SHOW_MROUTE_COMMON_ARGS': ShowMrouteCommonArguments,
   }
   handler = doShowIpMrouteAllVrfs
   cliModel = PimModelLib.IpMroutesAllVrfs

BasicCli.addShowCommandClass( IpMrouteAllVrfsCmd )

#----------------------------------------------------------------------------------
# Legacy:
# show ip mroute [ <source> ] <group> interface [ <interface> ][ detail ]
#-----------------------------------------------------------------------------------
def doShowIpMrouteInterfaceAll( mode, args ):
   model = PimModelLib.MrouteInterfaceAll
   af = AddressFamily.ipv4
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   groupOrSource1 = args.get( 'GROUP_OR_SOURCE1' )
   groupOrSource2 = args.get( 'GROUP_OR_SOURCE2' )
   intf = args.get( 'INTF' )
   detail = 'detail' in args

   vrfConfig = _mfibVrfConfig
   if not validVrfName( mode, vrfName, vrfConfig, af ):
      return

   if not waitForPimStatusSms( mode, af ):
      return

   submodel = {}

   #show ip mroute s, g interface => SM
   if groupOrSource2:
      try:
         ( source, group ) = PimCliLib.ipPimParseSg( groupOrSource1, groupOrSource2 )
      except ValueError:
         mode.addError("Must enter a multicast group and/or unicast source")
         return

   #show ip mroute g interface => Bidir + SM ( use 0.0.0.0 as source)
   if groupOrSource2 is None:
      try:
         group = PimCliLib.ipPimParseGroup( groupOrSource1 )
      except ValueError:
         mode.addError("Must enter a multicast group")
         return

      source = Arnet.IpGenAddr("0.0.0.0")

   revision = mode.session_.requestedModelRevision()
   fmt = mode.session_.outputFormat()

   if fmt == 2 and revision == 1:
      for ( pimMode, hook ) in PimCliLib.pimShowIpMrouteIntfHook.extensions():
         if pimMode == 'sparseMode':
            return  hook(mode, vrfName, source, group, intf,
                                 detail, submodel=False)[ 1 ]
   else:
      with Printer( mode ) as p:
         for ( pimMode, hook ) in PimCliLib.pimShowIpMrouteIntfHook.extensions():
            p.startDict( pimMode )
            submodel[ pimMode ] = hook(mode, vrfName, source, group, intf,
               detail, submodel=True)[ 1 ]
            p.endDict( pimMode )

      for pimMode in submodel: # pylint: disable=consider-using-dict-items
         setattr( model, pimMode , submodel[ pimMode ] )

      return model

class IpMrouteInterfaceAllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip mroute [ vrf VRF_NAME ] GROUP_OR_SOURCE1 [ GROUP_OR_SOURCE2 ]
               interface [ INTF ] [ detail ]'''
   data = {
      'ip' : ipAfterShowMatcher,
      'mroute' : mrouteMatcher,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'GROUP_OR_SOURCE1' : groupOrSourceMatcher,
      'GROUP_OR_SOURCE2' : groupOrSourceMatcher,
      'interface' : mrouteInterfaceMatcherForShow,
      'INTF' : IntfCli.Intf.matcher,
      'detail' : mrouteDetailMatcher
   }
   handler = doShowIpMrouteInterfaceAll
   cliModel = PimModelLib.MrouteInterfaceAll

BasicCli.addShowCommandClass( IpMrouteInterfaceAllCmd )

#------------------------------------------------------------------------------------
# Legacy:
# show ip pim [ vrf VRF_NAME ] rp-hash <group> [ detail ]
#------------------------------------------------------------------------------------
def doShowIpPimRpHash( mode, args ):
   model = PimModelLib.PimRpHashAll()
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   group = args[ 'GROUP_ADDR' ]

   if not validVrfName( mode, vrfName, _mfibVrfConfig ):
      return

   rpHashs = {}
   for hook in PimCliLib.pimShowIpPimRpHashHook.extensions():
      modeStr, rpHash = hook( mode, vrfName, group, 'detail' in args )
      rpHashs[ modeStr ] = rpHash
   model.initialize( rpHashs )
   return model

class IpPimRpHashCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ vrf VRF_NAME ] rp-hash GROUP_ADDR [ detail ]'
   data = {
      'ip' : ipAfterShowMatcher,
      'pim' : pimNodeAfterShowIp,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'rp-hash' : CliToken.Pim.rpHashMatcher,
      'GROUP_ADDR' : CliToken.Pim.groupMatcher,
      'detail' : CliToken.Pim.detailMatcher
   }
   handler = doShowIpPimRpHash
   cliModel = PimModelLib.PimRpHashAll

BasicCli.addShowCommandClass( IpPimRpHashCmd )

#------------------------------------------------------------------------------------
# Legacy:
# show ip pim access-list [<acl>]
#------------------------------------------------------------------------------------
def showIpAcl( mode, args ):
   return AclCli.showServiceAcl( mode,
                                 aclCpConfig,
                                 aclStatus,
                                 aclCheckpoint,
                                 'ip',
                                 args.get( '<aclNameExpr>' ),
                                 serviceName='pim' )

class IpPimAclCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim access-list [ ACL_NAME ]'
   data = {
      'ip' : ipAfterShowMatcher,
      'pim' : pimNodeAfterShowIp,
      'access-list' : AclCli.accessListMatcher,
      'ACL_NAME' : AclCli.ipAclNameExpression
   }
   handler = showIpAcl
   cliModel = AclCliModel.AllAclList

BasicCli.addShowCommandClass( IpPimAclCmd )

#------------------------------------------------------------------------------------
# show pim [ ipv4 | ipv6 ] sparse-mode evpn gateway [ detail ]
#------------------------------------------------------------------------------------
def showEvpnGateway( mode, args ):
   model = EvpnGatewayShowModel.EvpnGatewayVrfs()
   model.detailIs( 'detail' in args )
   assert args.get( 'MODE' ) == 'sparse-mode'
   af = args.get( 'addressFamily' )
   if af == 'ipv6':
      pegVlanStatus = _pegVlanStatus6
   else:
      pegVlanStatus = _pegVlanStatus4

   for vlan in pegVlanStatus.pegVlan:
      if vlan not in _enabledStatus.sbdVlanToVrf:
         continue
      vrf = _enabledStatus.sbdVlanToVrf[ vlan ]
      vrfModel = EvpnGatewayShowModel.EvpnGatewayVrf()
      vrfModel.vlan = vlan
      model.vrfs[ vrf ] = vrfModel
   return model

class EvpnGatewayCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim [ AF ] MODE evpn gateway [ detail ]'
   data = {
      'pim' : pimNodeAfterShow,
      'AF' : ShowAddressFamilyExpr,
      'MODE' : sparseModeMatcher,
      'evpn' : evpnMatcher,
      'gateway' : gatewayMatcher,
      'detail' : detailKw
   }
   handler = showEvpnGateway
   cliModel = EvpnGatewayShowModel.EvpnGatewayVrfs

BasicCli.addShowCommandClass( EvpnGatewayCmd )

#------------------------------------------------------------------------------------
# show pim [ ipv4 | ipv6 ] evpn gateway dr
#------------------------------------------------------------------------------------
def showEvpnGatewayDr( mode, args ):
   af = args.get( 'addressFamily' )
   if af == 'ipv6':
      pegDrStatus = _pegDrStatus6
   else:
      pegDrStatus = _pegDrStatus4

   model = EvpnGatewayShowModel.EvpnGatewayDrs()
   for vlanId, drVlan in pegDrStatus.pegDrVlan.items():
      dr = EvpnGatewayShowModel.EvpnGatewayDr()
      dr.role = drVlan.pegDrRole
      dr.addr = drVlan.drAddr
      dr.algorithm = EvpnGatewayShowModel.drAlgorithmForModel( drVlan.drAlgorithm )
      model.vlans[ vlanId ] = dr

   return model

class EvpnGatewayDrCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim [ AF ] evpn gateway dr'
   data = {
      'pim' : pimNodeAfterShow,
      'AF' : ShowAddressFamilyExpr,
      'evpn' : evpnMatcher,
      'gateway' : gatewayMatcher,
      'dr' : drMatcher
   }
   handler = showEvpnGatewayDr
   cliModel = EvpnGatewayShowModel.EvpnGatewayDrs

BasicCli.addShowCommandClass( EvpnGatewayDrCmd )

#------------------------------------------------------------------------------------
# show pim [ ipv4 | ipv6 ] vrf <vrfName> sparse-mode pim evpn membership
#  - display membershipJoinStatus for sbdVlanId produced by Pimsm
#  - when vrf is not specified, display membership for all vrfs
#------------------------------------------------------------------------------------
def showPimsmEvpnMembership( mode, args ):

   assert args.get( 'MODE' ) == 'sparse-mode'
   vrfName = args.get( 'VRF_NAME' )
   af = args.get( 'addressFamily' )
   if af != 'ipv6':
      af = 'ipv4'
   model = MembershipJoinStatusModel.MembershipJoinStatusPimsm()
   shmemMg = _shmemEm.getMountGroup()

   path = getPath( 'Irb::Multicast::Gmp::MembershipJoinStatus', af,
                   MembershipClient.pimsm )
   _membershipJoinStatus[ af ] = shmemMg.doMount(
                     path,
                     'Irb::Multicast::Gmp::MembershipJoinStatus',
                     Shark.mountInfo( 'shadow' ) )
   shmemMg.doClose()
   model.initFromTacModel( _membershipJoinStatus[ af ],
                           _enabledStatus, vrfName )
   return model

class PimSparseModePimEvpnMembershipCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim [ AF ] [ vrf VRF_NAME ] MODE PIM evpn membership'
   data = {
      'pim' : pimNodeAfterShow,
      'AF' : ShowAddressFamilyExpr,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'MODE': sparseModeMatcher,
      'PIM' : pimMatcherForShow,
      'evpn' : evpnMatcher,
      'membership': membershipMatcher
   }
   cliModel = MembershipJoinStatusModel.MembershipJoinStatusPimsm
   handler = showPimsmEvpnMembership

BasicCli.addShowCommandClass( PimSparseModePimEvpnMembershipCmd )

def showEvpnMembership( mode, args ):
   assert args.get( 'MODE' ) == 'sparse-mode'
   vrfName = args.get( 'VRF_NAME' )
   vlan = args.get( 'VLANID' )
   model = MembershipJoinStatusModel.MembershipJoinStatusEvpn()

   if vrfName and not _enabledStatus.vrfToSBDVlan.get( vrfName ):
      return model
   # vrfName is either valid or None
   vlanId = vlan.id if vlan else None
   af = args.get( 'addressFamily' )

   if af != 'ipv6':
      af = 'ipv4'

   if 'cli' in args:
      coll = _membershipJoinStatusCli[ af ]
   else:
      coll = _membershipJoinStatusEvpn[ af ]
   model.initFromTacModel( coll, _l3IntfStatusDir, vrfName, vlanId )
   return model

#------------------------------------------------------------------------------------
# show pim [ ipv4 | ipv6 ] vrf <vrfName> sparse-mode evpn pim membership
#  - display membershipJoinStatus for vlans under each vrf
#  - when vrf is not specified, display membership from EVPN for all vrfs
#------------------------------------------------------------------------------------
class PimSparseModeEvpnMembershipCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show pim [ AF ] [ vrf VRF_NAME ] MODE evpn PIM
               membership [ vlan VLANID ]'''
   data = {
      'pim' : pimNodeAfterShow,
      'AF' : ShowAddressFamilyExpr,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'MODE' : sparseModeMatcher,
      'evpn' : evpnMatcher,
      'PIM' : pimMatcherForShow,
      'membership' : membershipMatcher,
      'vlan' : BridgingCli.matcherVlan,
      'VLANID' : vlanIdMatcher,
   }
   cliModel = MembershipJoinStatusModel.MembershipJoinStatusEvpn
   handler = showEvpnMembership

BasicCli.addShowCommandClass( PimSparseModeEvpnMembershipCmd )

#------------------------------------------------------------------------------------
# show pim [ AF ] vrf <vrfName> sparse-mode evpn cli membership (hidden command)
#  - display membershipJoinStatus configured from CLI for vlans under each vrf
#  - when vrf is not specified, display membership for all vrfs
#------------------------------------------------------------------------------------
class PimSparseModeEvpnCliMembershipCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show pim [ AF ] [ vrf VRF_NAME ] MODE evpn cli membership
               [ vlan VLANID ]'''
   data = {
      'pim' : pimNodeAfterShow,
      'AF' : ShowAddressFamilyExpr,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'MODE' : sparseModeMatcher,
      'evpn' : evpnMatcher,
      'cli' : evpnCliMatcher,
      'membership' : membershipMatcher,
      'vlan' : BridgingCli.matcherVlan,
      'VLANID' : vlanIdMatcher,
   }
   cliModel = MembershipJoinStatusModel.MembershipJoinStatusEvpn
   handler = showEvpnMembership
   hidden = True

BasicCli.addShowCommandClass( PimSparseModeEvpnCliMembershipCmd )

#------------------------------------------------------------------------------------

def Plugin( entityManager ):
   global _mfibVrfConfig
   global _routingVrfInfoDir
   global _routing6VrfInfoDir
   global _routingHwStatus
   global _ipConfig
   global _ipStatus
   global _entityManager
   global _pimGlobalStatus
   global _pimConfigColl
   global _pimStatusColl
   global _vrfStatusLocal
   global _pim6StatusColl
   global _pim6GlobalStatus
   global _mfib6VrfConfig
   global _bessAgentStatusCli
   global _bessAgentStatusSfe
   global aclCpConfig
   global aclStatus
   global aclCheckpoint
   global _pegVlanStatus4
   global _pegVlanStatus6
   global _enabledStatus
   global _pegDrStatus4
   global _pegDrStatus6
   global _l3IntfStatusDir
   global _shmemEm

   _entityManager = entityManager

   _ipConfig = LazyMount.mount( entityManager, 'ip/config', 'Ip::Config', 'r' )

   _ipStatus = LazyMount.mount( entityManager, 'ip/status', 'Ip::Status', 'r' )

   _mfibVrfConfig = LazyMount.mount( entityManager, 'routing/multicast/vrf/config',
                                     'Routing::Multicast::Fib::VrfConfig', 'r' )

   _pimConfigColl = LazyMount.mount( entityManager, 'routing/pim/config',
                                     'Routing::Pim::ConfigColl', 'r' )

   _pimStatusColl = PimLib.getPimStatusColl( AddressFamily.ipv4,
                                             "modePimSmAndBidir" )

   _pim6StatusColl = PimLib.getPimStatusColl( AddressFamily.ipv6,
                                             "modePimSmAndBidir" )

   _routingVrfInfoDir =  LazyMount.mount( entityManager,
                                          "routing/vrf/routingInfo/status",
                                          "Tac::Dir", "r" )
   _routing6VrfInfoDir =  LazyMount.mount( entityManager,
                                           "routing6/vrf/routingInfo/status",
                                           "Tac::Dir", "r" )

   _routingHwStatus = LazyMount.mount( entityManager,
         'routing/hardware/status', 'Routing::Hardware::Status', 'r' )

   _vrfStatusLocal = LazyMount.mount( entityManager,
                                    Cell.path( "ip/vrf/status/local" ),
                                    "Ip::AllVrfStatusLocal", "r" )

   aclCpConfig = LazyMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "r" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                   "Acl::CheckpointStatus", "w" )

   _mfib6VrfConfig = LazyMount.mount( entityManager, 'routing6/multicast/vrf/config',
                                     'Routing::Multicast::Fib::VrfConfig', 'r' )

   _pimGlobalStatus = PimLib.getPimGlobalStatus( entityManager, AddressFamily.ipv4 )

   _pim6GlobalStatus = PimLib.getPimGlobalStatus( entityManager, AddressFamily.ipv6 )

   _bessAgentStatusCli = LazyMount.mount( entityManager, 'bess/agentstatus/cli',
                                       "McastCommon::BessAgentStatus", "r" )
   _bessAgentStatusSfe = LazyMount.mount( entityManager, 'bess/agentstatus/sfe',
                                       "McastCommon::BessAgentStatus", "r" )

   pegVlanStatusPath = \
         Tac.Type( "Routing::Multicast::PegVlanStatus" ).mountPath( "ipv4" )
   _pegVlanStatus4 = SmashLazyMount.mount( entityManager, pegVlanStatusPath,
         "Routing::Multicast::PegVlanStatus", SmashLazyMount.mountInfo( 'reader' ) )
   pegVlanStatusPath = \
         Tac.Type( "Routing::Multicast::PegVlanStatus" ).mountPath( "ipv6" )
   _pegVlanStatus6 = SmashLazyMount.mount( entityManager, pegVlanStatusPath,
         "Routing::Multicast::PegVlanStatus", SmashLazyMount.mountInfo( 'reader' ) )
   _enabledStatus = LazyMount.mount( entityManager, "vxlan/enabledStatus",
         "Vxlan::EnabledStatus", "r" )
   pegDrStatusPath = \
         Tac.Type( "Routing::Multicast::PegDrStatus" ).mountPath( "ipv4" )
   _pegDrStatus4 = SmashLazyMount.mount( entityManager, pegDrStatusPath,
         "Routing::Multicast::PegDrStatus", SmashLazyMount.mountInfo( 'reader' ) )
   pegDrStatusPath = \
         Tac.Type( "Routing::Multicast::PegDrStatus" ).mountPath( "ipv6" )
   _pegDrStatus6 = SmashLazyMount.mount( entityManager, pegDrStatusPath,
         "Routing::Multicast::PegDrStatus", SmashLazyMount.mountInfo( 'reader' ) )

   _shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   shmemMg = _shmemEm.getMountGroup()
   for af in [ 'ipv4', 'ipv6' ]:
      path = getPath( 'Irb::Multicast::Gmp::MembershipJoinStatusCli', af,
                      MembershipClient.cli )
      _membershipJoinStatusCli[ af ] = LazyMount.mount( entityManager, path,
                       'Irb::Multicast::Gmp::MembershipJoinStatusCli', 'r' )
      path = getPath( 'Irb::Multicast::Gmp::MembershipJoinStatus', af,
                      MembershipClient.evpn )
      _membershipJoinStatusEvpn[ af ] = SharkLazyMount.mount(
         entityManager, path, 'Irb::Multicast::Gmp::MembershipJoinStatus',
         SharkLazyMount.mountInfo( 'shadow' ), autoUnmount=True )

   shmemMg.doClose()

   _l3IntfStatusDir = LazyMount.mount( entityManager, 'l3/intf/status',
                                       "L3::Intf::StatusDir", "r" )

   _ = PimLib.getPimStatusModeFilterSm( AddressFamily.ipv4, "modePimSmAndBidir" )

   _ = PimLib.getPimStatusModeFilterSm( AddressFamily.ipv6, "modePimSmAndBidir" )

   PimCliLib.pimMessageCountersHook.addExtension( getPimMessageCounters )
