#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import ShowCommand
# pylint: disable-next=consider-using-from-import
import CliPlugin.PimsmCliShow as PimsmCliShow
from CliPlugin.IntfCli import Intf
from CliPlugin.Ip6AddrMatcher import (
      Ip6AddrMatcher,
      Ip6PrefixMatcher,
)
from CliPlugin.IpAddrMatcher import (
      IpAddrMatcher,
      ipAddrWithMaskExpr,
)
from CliPlugin.PimConfigCheckModel import (
      IgmpConfigCheck,
      PimConfigCheck,
)
from CliPlugin.PimsmEnableMode import vrfExprFactory
from CliToken.Ip import ipMatcherForShow
from CliToken.Pim import (
      rpMatcher,
      detailMatcher,
      pimNodeAfterShowIp,
      rpHashMatcher,
      neighborsMatcher,
      joinsMatcher,
      upstreamMatcher,
      sparseModeMatcher,
      joinMatcher,
      routeMatcher,
)
from McastCommonCliLib import (
      mcastRoutingSupportedGuard,
      mcastGenRoutingSupportedGuard,
      ipv4NodeForShow,
      ipv6NodeForShow,
)
from PimModel import (
      MrouteCount,
      GroupSms,
      MrouteSmInterfaces,
)
from PimModelLib import (
      PimRpCandidates,
      PimUpstreamJoins,
      PimRpHash,
)
from PimsmModel import (
      SourcesGroups,
      NonDrDropRuleInterfaces,
)

matcherConfigSanity = CliMatcher.KeywordMatcher( 'config-sanity',
      helpdesc='Show hints of potential PIMSM config problems' )
matcherCount = CliMatcher.KeywordMatcher( 'count',
      helpdesc='Count of multicast routes' )
matcherDetail = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='Detailed route information' )
matcherGroupOrSource = IpAddrMatcher(
      helpdesc='A multicast group address or unicast source address' )
matcherGroupOrSource6 = Ip6AddrMatcher(
      helpdesc='A multicast group address or unicast source address' )
matcherHash = CliMatcher.KeywordMatcher( 'hash',
      helpdesc='Rendezvous Point hashing' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Route interface specific state' )
nodeMroute = CliCommand.guardedKeyword( 'mroute',
      helpdesc='PIM routing information',
      guard=mcastRoutingSupportedGuard )
nodePim = CliCommand.guardedKeyword( 'pim',
      helpdesc='PIM information',
      guard=mcastGenRoutingSupportedGuard )
nodePimAfterIp = CliCommand.guardedKeyword( 'pim',
      helpdesc='PIM information',
      guard=mcastRoutingSupportedGuard )

#--------------------------------------------------------------------------------
# show ip mroute sparse-mode [ vrf VRF ] IP1 IP2 interface [ INTF ] [ detail ]
# show pim ipv4 [ vrf VRF ] sparse-mode route IP1 IP2 interface [ INTF ] [ detail ]
#--------------------------------------------------------------------------------
class IpMrouteSparseModeGrouporsourceInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show
               ( ( ip mroute sparse-mode [ VRF ] )
               | ( pim ipv4 [ VRF ] sparse-mode route ) )
               IP1 IP2 interface [ INTF ] [ detail ]'''
   data = {
      'ip': ipMatcherForShow,
      'mroute': nodeMroute,
      'sparse-mode': sparseModeMatcher,
      'VRF': vrfExprFactory,
      'pim': nodePim,
      'ipv4': ipv4NodeForShow,
      'route': routeMatcher,
      'IP1': matcherGroupOrSource,
      'IP2': matcherGroupOrSource,
      'interface': matcherInterface,
      'INTF': Intf.matcher,
      'detail': 'Display information in detail',
   }

   handler = PimsmCliShow.showIpMrouteInterface
   cliModel = MrouteSmInterfaces

BasicCli.addShowCommandClass( IpMrouteSparseModeGrouporsourceInterfaceCmd )

#--------------------------------------------------------------------------------
# show pim ipv6 [ vrf VRF ] sparse-mode route IP1 IP2 interface [ INTF ] [ detail ]
#--------------------------------------------------------------------------------
class PimIpv6SparseModeRouteIpv6InterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show pim ipv6 [ VRF ] sparse-mode route
               IP1 IP2 interface [ INTF ] [ detail ]'''
   data = {
      'pim': nodePim,
      'ipv6': ipv6NodeForShow,
      'VRF': vrfExprFactory,
      'sparse-mode': sparseModeMatcher,
      'route': routeMatcher,
      'IP1': matcherGroupOrSource6,
      'IP2': matcherGroupOrSource6,
      'interface': matcherInterface,
      'INTF': Intf.matcher,
      'detail': 'Display information in detail',
   }

   handler = PimsmCliShow.showIpMrouteInterface
   cliModel = MrouteSmInterfaces

BasicCli.addShowCommandClass( PimIpv6SparseModeRouteIpv6InterfaceCmd )

#--------------------------------------------------------------------------------
# show ip mroute sparse-mode [ vrf VRF ] ( sources | groups )
#--------------------------------------------------------------------------------
class IpMrouteSparseModeGroupsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip mroute sparse-mode [ VRF ] ( sources | groups )'
   data = {
      'ip': ipMatcherForShow,
      'mroute': nodeMroute,
      'sparse-mode': sparseModeMatcher,
      'VRF': vrfExprFactory,
      'groups': 'Show number of sources per group',
      'sources': 'Show number of groups per source',
   }

   handler = PimsmCliShow.showIpMrouteSourcesGroups
   cliModel = SourcesGroups

BasicCli.addShowCommandClass( IpMrouteSparseModeGroupsCmd )

#--------------------------------------------------------------------------------
# show ip pim [ vrf VRF ] config-sanity sparse-mode
#--------------------------------------------------------------------------------
class IpPimConfigSanitySparseModeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ VRF ] config-sanity sparse-mode'
   data = {
      'ip': ipMatcherForShow,
      'pim': nodePimAfterIp,
      'VRF': vrfExprFactory,
      'config-sanity': matcherConfigSanity,
      'sparse-mode': 'Show hints of potential PIMSM config problems',
   }

   handler = PimsmCliShow.pimsmConfigSanity
   cliModel = PimConfigCheck

BasicCli.addShowCommandClass( IpPimConfigSanitySparseModeCmd )

#--------------------------------------------------------------------------------
# show ip pim [ vrf VRF ] config-sanity igmp
#--------------------------------------------------------------------------------
class IpPimConfigSanityIgmpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ VRF ] config-sanity igmp'
   data = {
      'ip': ipMatcherForShow,
      'pim': nodePimAfterIp,
      'VRF': vrfExprFactory,
      'config-sanity': matcherConfigSanity,
      'igmp': 'Show hints of potential problems related to IGMP',
   }

   handler = PimsmCliShow.igmpConfigSanity
   cliModel = IgmpConfigCheck

BasicCli.addShowCommandClass( IpPimConfigSanityIgmpCmd )

#--------------------------------------------------------------------------------
# show ip pim [ vrf VRF ] sparse-mode rp [ PREFIX ] [ detail ]
# show pim ipv4 [ vrf VRF ] sparse-mode rp [ PREFIX ] [ detail ]
#--------------------------------------------------------------------------------
class IpPimSparseModeRpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ( ( ip IP_PIM ) | ( pim ipv4 ) )
               [ VRF ] sparse-mode rp [ PREFIX ] [ detail ]'''
   data = {
      'ip': ipMatcherForShow,
      'IP_PIM': nodePimAfterIp, # Keyword 'pim'.
      'pim': nodePim,
      'ipv4': ipv4NodeForShow,
      'VRF': vrfExprFactory,
      'sparse-mode': sparseModeMatcher,
      'rp': rpMatcher,
      'PREFIX': ipAddrWithMaskExpr( 'Group range prefix',
                                     'Group range prefix mask',
                                     'Group address range with prefix',
                                     partial=True,
                                     returnError='PimCli::Error' ),
      'detail': detailMatcher,
   }

   handler = PimsmCliShow.showIpPimRpSparseMode
   cliModel = PimRpCandidates

BasicCli.addShowCommandClass( IpPimSparseModeRpCmd )

#--------------------------------------------------------------------------------
# show pim ipv6 [ vrf VRF ] sparse-mode rp [ PREFIX ] [ detail ]
#--------------------------------------------------------------------------------
class PimIpv6SparseModeRpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim ipv6 [ VRF ] sparse-mode rp [ PREFIX ] [ detail ]'
   data = {
      'pim': nodePim,
      'ipv6': ipv6NodeForShow,
      'VRF': vrfExprFactory,
      'sparse-mode': sparseModeMatcher,
      'rp': rpMatcher,
      'PREFIX': Ip6PrefixMatcher( 'IPv6 address prefix' ),
      'detail': detailMatcher,
   }

   handler = PimsmCliShow.showIpPimRpSparseMode
   cliModel = PimRpCandidates

BasicCli.addShowCommandClass( PimIpv6SparseModeRpCmd )

#--------------------------------------------------------------------------------
# show ip pim [ vrf VRF ] sparse-mode rp-hash GROUP [ detail ]
# show pim ipv4 [ vrf VRF ] sparse-mode rp hash GROUP [ detail ]
#--------------------------------------------------------------------------------
class IpPimSparseModeRpHashGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show
               ( ( ip IP_PIM [ VRF ] sparse-mode rp-hash )
               | ( pim ipv4 [ VRF ] sparse-mode rp hash ) )
               GROUP [ detail ]'''
   data = {
      'ip': ipMatcherForShow,
      'IP_PIM': nodePimAfterIp, # Keyword 'pim'.
      'VRF': vrfExprFactory,
      'sparse-mode': sparseModeMatcher,
      'rp-hash': rpHashMatcher,
      'pim': nodePim,
      'ipv4': ipv4NodeForShow,
      'rp': rpMatcher,
      'hash': matcherHash,
      'GROUP': matcherGroupOrSource,
      'detail': detailMatcher,
   }

   handler = PimsmCliShow.showIpPimRpHash
   cliModel = PimRpHash

BasicCli.addShowCommandClass( IpPimSparseModeRpHashGroupCmd )

#--------------------------------------------------------------------------------
# show pim ipv6 [ vrf VRF ] sparse-mode rp hash GROUP [ detail ]
#--------------------------------------------------------------------------------
class PimIpv6SparseModeRpHashGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim ipv6 [ VRF ] sparse-mode rp hash GROUP [ detail ]'
   data = {
      'pim': nodePim,
      'ipv6': ipv6NodeForShow,
      'VRF': vrfExprFactory,
      'sparse-mode': sparseModeMatcher,
      'rp': rpMatcher,
      'hash': matcherHash,
      'GROUP': Ip6AddrMatcher( helpdesc='Multicast group Address' ),
      'detail': detailMatcher,
   }

   handler = PimsmCliShow.showIpPimRpHash
   cliModel = PimRpHash

BasicCli.addShowCommandClass( PimIpv6SparseModeRpHashGroupCmd )

#--------------------------------------------------------------------------------
# show ip pim [ vrf VRF ] sparse-mode upstream joins
#      [ IP1 [ IP2 ] ] neighbor NEIGHBOR ]
# show pim ipv4 [ vrf VRF ] sparse-mode join [ IP1 [ IP2 ] ] [ neighbor NEIGHBOR ]
#--------------------------------------------------------------------------------
class IpPimSparseModeUpstreamJoinsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show
               ( ( ip IP_PIM [ VRF ] sparse-mode upstream joins )
               | ( pim ipv4 [ VRF ] sparse-mode join ) )
               [ IP1 [ IP2 ] ] [ neighbor NEIGHBOR ]'''
   data = {
      'ip': ipMatcherForShow,
      'IP_PIM': nodePimAfterIp, # Keyword 'pim'.
      'VRF': vrfExprFactory,
      'sparse-mode': sparseModeMatcher,
      'upstream': upstreamMatcher,
      'joins': joinsMatcher,
      'pim': nodePim,
      'ipv4': ipv4NodeForShow,
      'join': joinMatcher,
      'IP1': matcherGroupOrSource,
      'IP2': matcherGroupOrSource,
      'neighbor': neighborsMatcher,
      'NEIGHBOR': IpAddrMatcher( helpdesc='IP address of neighbor' ),
   }

   handler = PimsmCliShow.showIpPimUpstreamJoinsSm
   cliModel = PimUpstreamJoins

BasicCli.addShowCommandClass( IpPimSparseModeUpstreamJoinsCmd )

#--------------------------------------------------------------------------------
# show pim ipv6 [ vrf VRF ] sparse-mode join [ IP1 [ IP2 ] ] [ neighbor NEIGHBOR ]
#--------------------------------------------------------------------------------
class PimIpv6SparseModeJoinCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show pim ipv6 [ VRF ] sparse-mode join
               [ IP1 [ IP2 ] ] [ neighbor NEIGHBOR ]'''
   data = {
      'pim': nodePim,
      'ipv6': ipv6NodeForShow,
      'VRF': vrfExprFactory,
      'sparse-mode': sparseModeMatcher,
      'join': joinMatcher,
      'IP1': matcherGroupOrSource6,
      'IP2': matcherGroupOrSource6,
      'neighbor': neighborsMatcher,
      'NEIGHBOR': Ip6AddrMatcher( helpdesc='IPv6 Address of neighbor' ),
   }

   handler = PimsmCliShow.showIpPimUpstreamJoinsSm
   cliModel = PimUpstreamJoins

BasicCli.addShowCommandClass( PimIpv6SparseModeJoinCmd )

#--------------------------------------------------------------------------------
# show pim ipv4 [ vrf VRF ] sparse-mode non-dr drop-rules
#--------------------------------------------------------------------------------
class PimIpv4SparseModeNonDrDropRulesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim ipv4 [ VRF ] sparse-mode non-dr drop-rules'
   data = {
      'pim': nodePim,
      'ipv4': ipv4NodeForShow,
      'VRF': vrfExprFactory,
      'sparse-mode': sparseModeMatcher,
      'non-dr': 'Show information related to non designated router',
      'drop-rules': 'Show state information for rules for dropping packets',
   }

   handler = PimsmCliShow.showPimSparseModeNonDrDropRules
   cliModel = NonDrDropRuleInterfaces

BasicCli.addShowCommandClass( PimIpv4SparseModeNonDrDropRulesCmd )

#--------------------------------------------------------------------------------
# show ip mroute sparse-mode [ vrf VRF ] [ IP1 [ IP2 ] ] [ detail ]
# show pim ipv4 [ vrf VRF ] sparse-mode route [ IP1 [ IP2 ] ] [ detail ]
#--------------------------------------------------------------------------------
class IpMrouteSparseModeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show
               ( ( ip mroute sparse-mode [ VRF ] )
               | ( pim ipv4 [ VRF ] sparse-mode route ) )
               [ IP1 [ IP2 ] ] [ detail ]'''
   data = {
      'ip': ipMatcherForShow,
      'mroute': nodeMroute,
      'sparse-mode': sparseModeMatcher,
      'VRF': vrfExprFactory,
      'pim': pimNodeAfterShowIp,
      'ipv4': ipv4NodeForShow,
      'route': routeMatcher,
      'IP1': matcherGroupOrSource,
      'IP2': matcherGroupOrSource,
      'detail': matcherDetail,
   }

   handler = PimsmCliShow.showIpMroute
   cliModel = GroupSms

BasicCli.addShowCommandClass( IpMrouteSparseModeCmd )

#--------------------------------------------------------------------------------
# show pim ipv6 [ vrf VRF ] sparse-mode route [ IP1 [ IP2 ] ] [ detail ]
#--------------------------------------------------------------------------------
class PimIpv6SparseModeRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim ipv6 [ VRF ] sparse-mode route [ IP1 [ IP2 ] ] [ detail ]'
   data = {
      'pim': pimNodeAfterShowIp,
      'ipv6': ipv6NodeForShow,
      'VRF': vrfExprFactory,
      'sparse-mode': sparseModeMatcher,
      'route': routeMatcher,
      'IP1': matcherGroupOrSource6,
      'IP2': matcherGroupOrSource6,
      'detail': matcherDetail,
   }

   handler = PimsmCliShow.showIpMroute
   cliModel = GroupSms

BasicCli.addShowCommandClass( PimIpv6SparseModeRouteCmd )

#--------------------------------------------------------------------------------
# show ip mroute sparse-mode [ vrf VRF ] count
# show pim ipv4 [ vrf VRF ] sparse-mode route count
#--------------------------------------------------------------------------------
class PimIpv4SparseModeRouteCountCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show
               ( ( ip mroute sparse-mode [ VRF ] )
               | ( pim ipv4 [ VRF ] sparse-mode route ) )
               count'''
   data = {
      'ip': ipMatcherForShow,
      'mroute': nodeMroute,
      'sparse-mode': sparseModeMatcher,
      'VRF': vrfExprFactory,
      'pim': nodePim,
      'ipv4': ipv4NodeForShow,
      'route': routeMatcher,
      'count': matcherCount,
   }

   handler = PimsmCliShow.showPimRouteCount
   cliModel = MrouteCount

BasicCli.addShowCommandClass( PimIpv4SparseModeRouteCountCmd )

#--------------------------------------------------------------------------------
# show pim ipv6 [ vrf VRF ] sparse-mode route count
#--------------------------------------------------------------------------------
class PimIpv6SparseModeRouteCountCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim ipv6 [ VRF ] sparse-mode route count'
   data = {
      'pim': nodePim,
      'ipv6': ipv6NodeForShow,
      'VRF': vrfExprFactory,
      'sparse-mode': sparseModeMatcher,
      'route': routeMatcher,
      'count': matcherCount,
   }

   handler = PimsmCliShow.showPimRouteCount
   cliModel = MrouteCount

BasicCli.addShowCommandClass( PimIpv6SparseModeRouteCountCmd )
